/**
 * \ingroup pack
 *
 * \file wds.h
 */
#include "common.h"
#include <stdint.h>

#define IPV6_ADDRESS_ARRAY_SIZE 8
#define MAX_WDS_3GPP_CONF_LTE_ATTACH_PROFILE_LIST_SIZE 24
#define PACK_WDS_IPV4                          4
#define PACK_WDS_IPV6                          6


/**
 * structure contains 3GPP LTE QoS parameters
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  \param  QCI
 *          - QOS specified using the QOS Class Identifier (QOS) values
 *            QCI value 0    - Requests the network to assign the
 *                             appropriate QCI value
 *            QCI values 1-4 - Associated with guaranteed bit rates
 *            QCI values 5-9 - Associated with non-guaranteed bit rates
 *
 *  \param  gDlBitRate
 *          - Guaranteed DL bit rate
 *
 *  \param  maxDlBitRate
 *          - maxDlBitRate
 *
 *  \param  gUlBitRate
 *          - Guaranteed UL bit rate
 *
 *  \param  maxUlBitRate
 *          - Maximum UL bit rate
 *
 */
typedef struct 
{
    uint8_t QCI;
    uint8_t gDlBitRate;
    uint32_t maxDlBitRate;
    uint32_t gUlBitRate;
    uint32_t maxUlBitRate;
}LibPackQosClassID;

/**
 * structure contains traffic flow template parameters
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  \param  filterId
 *          - Filter identifier
 *
 *  \param  eValid
 *          - Evaluation precedence index
 *
 *  \param  pVersion
 *          - IP version number
 *            - 4 - IPv4
 *            - 6 - IPv6
 *
 *  \param  sourceIP
 *          - Source IP address
 *           - IPv4 - Fill the first 4 uint8_ts
 *           - IPv6 - Fill all the 16 uint8_ts
 *
 *  \param  sourceIPMask
 *          - Mask value for the source address
 *
 *  \param  nextHeader
 *          - Next header/protocol value
 *
 *  \param  destPortRangeStart
 *          - Start value of the destination port range
 *
 *  \param  destPortRangeEnd
 *          - End value of the destination port range
 *
 *  \param  srcPortRangeStart
 *          - Start value of the source port range
 *
 *  \param  srcPortRangeEnd
 *          - End value of the source port range
 *
 *  \param  IPSECSPI
 *          - IPSEC security parameter index
 *
 *  \param  tosMask
 *          - TOS mask (Traffic class for IPv6)
 *
 *  \param  flowLabel
 *          - Flow label
 *
 */
typedef struct 
{
    uint8_t  filterId;
    uint8_t  eValid;
    uint8_t  ipVersion;
    uint16_t  *pSourceIP;
    uint8_t  sourceIPMask;
    uint8_t  nextHeader;
    uint32_t  destPortRangeStart;
    uint16_t  destPortRangeEnd;
    uint16_t  srcPortRangeStart;
    uint16_t  srcPortRangeEnd;
    uint32_t IPSECSPI;
    uint16_t  tosMask;
    uint32_t flowLabel;
}LibPackTFTIDParams;



/**
 * This structure contains the GPRS Quality Of Service Information
 *
 *  @param  precedenceClass
 *          - Precedence class
 *
 *  @param  delayClass
 *          - Delay class
 *
 *  @param  reliabilityClass
 *          - Reliability class
 *
 *  @param  peakThroughputClass
 *          - Peak throughput class
 *
 *  @param  meanThroughputClass
 *          - Mean throughput class
 *
 */
typedef struct 
{
    uint32_t precedenceClass;
    uint32_t delayClass;
    uint32_t reliabilityClass;
    uint32_t peakThroughputClass;
    uint32_t meanThroughputClass;
}LibPackGPRSRequestedQoS;


/**
 * This structure contains the UMTS Quality Of Service Information
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  \param  trafficClass
 *          - 0x00 - Subscribed
 *          - 0x01 - Conversational
 *          - 0x02 - Streaming
 *          - 0x03 - Interactive
 *          - 0x04 - Background
 *
 *  \param  maxUplinkBitrate
 *          - Maximum uplink bit rate in bits/sec
 *
 *  \param  maxDownlinkBitrate
 *          - Maximum downlink bit rate in bits/sec
 *
 *  \param  grntUplinkBitrate
 *          - Guaranteed uplink bit rate in bits/sec
 *
 *  \param  grntDownlinkBitrate
 *          - Guranteed downlink bit rate in bits/sec
 *
 *  \param  qosDeliveryOrder - Qos delivery order
 *          - 0x00 - Subscribe
 *          - 0x01 - delivery order on
 *          - 0x02 - delivery order off
 *
 *  \param  maxSDUSize
 *          - Maximum SDU size
 *
 *  \param  sduErrorRatio - SDU error ratio
 *          - Target value for fraction of SDUs lost or
 *            detected as erroneous.
 *          - 0x00 - Subscribe
 *          - 0x01 - 1*10^(-2)
 *          - 0x02 - 7*10^(-3)
 *          - 0x03 - 1*10^(-3)
 *          - 0x04 - 1*10^(-4)
 *          - 0x05 - 1*10^(-5)
 *          - 0x06 - 1*10^(-6)
 *          - 0x07 - 1*10^(-1)
 *
 *  \param  resBerRatio - Residual bit error ratio
 *          - Target value for undetected bit error ratio in
 *            in the delivered SDUs.
 *          - 0x00 - Subscribe
 *          - 0x01 - 5*10^(-2)
 *          - 0x02 - 1*10^(-2)
 *          - 0x03 - 5*10^(-3)
 *          - 0x04 - 4*10^(-3)
 *          - 0x05 - 1*10^(-3)
 *          - 0x06 - 1*10^(-4)
 *          - 0x07 - 1*10^(-5)
 *          - 0x08 - 1*10^(-6)
 *          - 0x09 - 1*10^(-8)
 *
 *  \param  deliveryErrSDU - Delivery of erroneous SDUs
 *          - Indicates whether SDUs detected as erroneous shall be
 *            delivered or not.
 *          - 0x00 - Subscribe
 *          - 0x01 - 5*10^(-2)
 *          - 0x02 - 1*10^(-2)
 *          - 0x03 - 5*10^(-3)
 *          - 0x04 - 4*10^(-3)
 *          - 0x05 - 1*10^(-3)
 *          - 0x06 - 1*10^(-4)
 *          - 0x07 - 1*10^(-5)
 *          - 0x08 - 1*10^(-6)
 *          - 0x09 - 1*10^(-8)
 *
 *  \param  transferDelay - Transfer delay (ms)
 *          - Indicates the targeted time between a request to transfer an
 *            SDU at one SAP to its delivery at the other SAP in milliseconds.
 *
 *  \param  trafficPriority - Transfer handling priority
 *          - Specifies the relative importance for handling of SDUs that
 *            belong to the UMTS bearer, compared to the SDUs of other bearers.
 *
 */
typedef struct 
{
    uint8_t  trafficClass;
    uint32_t maxUplinkBitrate;
    uint32_t maxDownlinkBitrate;
    uint32_t grntUplinkBitrate;
    uint32_t grntDownlinkBitrate;
    uint8_t  qosDeliveryOrder;
    uint32_t maxSDUSize;
    uint8_t  sduErrorRatio;
    uint8_t  resBerRatio;
    uint8_t  deliveryErrSDU;
    uint32_t transferDelay;
    uint32_t trafficPriority;
}LibPackUMTSQoS;

/**
 * structure contains UMTS requested QoS with Signaling Indication flag
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  @param  UMTSReqQoS
 *          - Contains the UMTS Quality Of Service Information
 *
 *  @param  SigInd - Signaling Indication flag
 *          - TRUE  - Signaling indication ON
 *          - FALSE - Signaling indication OFF
 *
 */
typedef struct
{
    LibPackUMTSQoS UMTSReqQoS;
    uint8_t           SigInd;
}LibPackUMTSReqQoSSigInd;


/**
 * @param pTech
 *          - Indicates the technology preference
 *              - 1 - UMTS
 *              - 2 - CDMA
 *              - 3 - eMBMS
 *              - 4 - Modem Link Label.
 *                    Modem Link is an interface for transferring data between
 *                    entities on AP and modem.
 *          - optional
 * @param pprofileid3gpp
 *          - pointer to 3GPP profile id
 *          - optional
 * @param pprofileid3gpp2
 *          - pointer to 3GPPs profile id
 *          - optional
 * @param pAuth
 *          - Authentication type, it can be PAP or CHAP
 *          - optional
 * @param pUser
 *          - username for authentication process
 *          - optional
 * @param pPass
 *          - password for authentication process
 *          - optional
 */
typedef struct {
    uint8_t *pTech;
    uint32_t *pprofileid3gpp;
    uint32_t *pprofileid3gpp2;
    uint32_t *pAuth;
    char *pUser;
    char *pPass;
} pack_wds_SLQSStartDataSession_t;

/**
 * @param  psid
 *         - Assigned session ID when starting a data session
 *
 *  @param  pFailureReason
 *          - Reason data session failed to be established
 *          - See qaGobiApiTableCallEndReasons.h for Call End Reason
 *
 *  @param  pVerboseFailReasonType
 *          - Parameter describing type of verbose failure reason
 *          - See qaGobiApiTableCallEndReasons.h for Call End Reason Type
 *
 *  @param  pVerboseFailureReason
 *          - Verbose reason explaining why call failed. Depends on
 *            verbFailReasonType parameter
 *          - See qaGobiApiTableCallEndReasons.h for Call End Reason
 */
typedef struct {
    uint32_t *psid;
    uint32_t *pFailureReason;
    uint32_t *pVerboseFailReasonType;
    uint32_t *pVerboseFailureReason;
} unpack_wds_SLQSStartDataSession_t;

/**
 * Start data session
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSStartDataSession(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSStartDataSession_t *reqArg
        );

/**
 * start data session unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSStartDataSession(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSStartDataSession_t *pOutput
        );

/**
 * @param conn_status connection status
 * @param reconfigReqd Indicates whether the network interface
 *        on the host needs to be reconfigured.
 * @param sessionEndReason Call End Reason
 * @param verboseSessnEndReasonType Verbose call end reason type
 * @param verboseSessnEndReason Reason the call ended (verbose)
 * @param ipFamily IP family of the packet data connection.
 * @param techName Technology name of the packet data connection.
 * @param bearerID 
 *          - bearer ID (3GPP) or RLP ID (3GPP2) of the
 *            packet data connection.
 *          - Valid Values - 0 to 16
 *          - 0xFF - Invalid value.
 */
typedef struct{
    uint8_t   conn_status;
    uint8_t   reconfigReqd;
    uint16_t  sessionEndReason;
    uint16_t  verboseSessnEndReasonType;
    uint16_t  verboseSessnEndReason;
    uint8_t   ipFamily;
    uint16_t  techName;
    uint8_t   bearerID;
} unpack_wds_SLQSSetPacketSrvStatusCallback_t;

/**
 * set packet srv status callback unpack
 * @param[in] pResp qmi response
 * @param[in] respLen length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSSetPacketSrvStatusCallback(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSSetPacketSrvStatusCallback_t *pOutput
        );

/**
 * @param sid session id
 */
typedef struct {
    uint32_t *psid;
} pack_wds_SLQSStopDataSession_t;

/**
 * stop data session pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSStopDataSession(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSStopDataSession_t *reqArg
        );

/**
 * stop data session unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSStopDataSession(
        uint8_t *pResp,
        uint16_t respLen
        );
/**
 * This structure contains the Profile Identifier Information
 *
 *  @param  profileType
 *          - Identifies the type of profile
 *            0x00 = 3GPP
 *
 *  @param  profileIndex
 *          - Index of profile whose settings were loaded prior to
 *            session parameter negotiation for the current call.
 *            If this TLV is not present, data call parameters are
 *            based on device default settings for each parameter
 *
 */
struct wds_ProfileIdentifier
{
    uint8_t profileType;
    uint8_t profileIndex;
};


/**
 * This structure contains the GPRS Quality Of Service Information
 *
 *  @param  precedenceClass
 *          - Precedence class
 *
 *  @param  delayClass
 *          - Delay class
 *
 *  @param  reliabilityClass
 *          - Reliability class
 *
 *  @param  peakThroughputClass
 *          - Peak throughput class
 *
 *  @param  meanThroughputClass
 *          - Mean throughput class
 *
 */
struct wds_GPRSQoS
{
    uint32_t precedenceClass;
    uint32_t delayClass;
    uint32_t reliabilityClass;
    uint32_t peakThroughputClass;
    uint32_t meanThroughputClass;
};

/**
 * This structure contains the PCSCFIPv4ServerAddressList Information
 *
 *  @param  numInstances
 *          - number of address following
 *
 *  @param  pscsfIPv4Addr
 *          - P-CSCF IPv4 server addresses(Max 16 address, 4 bytes each)
 */
struct wds_PCSCFIPv4ServerAddressList
{
    uint8_t  numInstances;
    uint32_t pscsfIPv4Addr[64];
};

/**
 * This structure contains the PCSCFFQDNAddress Information
 *
 *  @param  fqdnLen
 *          - length of the received FQDN address
 *
 *  @param  fqdnAddr
 *          - FQDN address(Max 256 characters)
 */
struct wds_PCSCFFQDNAddress
{
    uint16_t fqdnLen;
    uint8_t fqdnAddr[256];
};

/**
 * This structure contains the PCSCFFQDNAddressList Information
 *
 *  @param  numInstances
 *          - Number of FQDN addresses received
 *
 *  @param  pcsfFQDNAddress
 *          - FQDN address information(Max 10 addresses)
 */
struct wds_PCSCFFQDNAddressList
{
    uint8_t                    numInstances;
    struct wds_PCSCFFQDNAddress pcsfFQDNAddress[10];
};

/**
 * This structure contains the DomainName Information
 *
 *  @param  domainLen
 *          - length of the recieved Domain name
 *
 *  @param  domainName
 *          - Domain name(Max 256 characters)
 */
struct wds_Domain
{
    uint16_t domainLen;
    uint8_t domainName[256];
};

/**
 * This structure contains the DomainNameList Information
 *
 *  @param  numInstances
 *          - Number of Domain name receieved
 *
 *  @param  domain
 *          - Domain name information(Max 10 Domain names)
 */
struct wds_DomainNameList
{
    uint8_t          numInstances;
    struct wds_Domain domain[10];
};

/**
 * This structure contains the IPV6 Address Information
 *
 *  @param  IPV6PrefixLen
 *          - Length of the received IPv6 address in no. of bits;
 *            can take value between 0 and 128
 *              - 0xFF - Not Available
 *
 *  @param  IPAddressV6
 *          - IPv6 address(in network byte order);
 *            This is an 8-element array of 16 bit numbers,
 *            each of which is in big endian format.
 */
struct wds_IPV6AddressInfo
{
    uint8_t   IPV6PrefixLen;
    uint16_t IPAddressV6[8];
};

/**
 * This structure contains the IPV6 Gateway Address Information
 *
 *  @param  gwV6PrefixLen
 *          - Length of the received IPv6 Gateway address in no. of bits;
 *            can take value between 0 and 128
 *
 *  @param  IPAddressV6
 *          - IPv6 Gateway address(in network byte order);
 *            This is an 8-element array of 16 bit numbers,
 *            each of which is in big endian format.
 */
struct wds_IPV6GWAddressInfo
{
    uint8_t   gwV6PrefixLen;
    uint16_t gwAddressV6[8];
};

/**
 * @param IPv4 ipv4 address
 * @param ProfileName profile name
 * @param PDPType PDP type
 * @param APNName APN name
 * @param PrimaryDNSV4
 * @param SecondaryDNSV4
 * @param UMTSGrantedQoS UMTS Granted Qos
 * @param GPRSGrantedQoS GPRS Granted QoS
 * @param Username
 * @param Authentication
 * @param ProfielID
 * @param GWAddressV4 Gateway IPv4
 * @param SubnetMaskV4 Subnet mask IPV4
 * @param PCSCFAddrPCO
 * @param PrimaryDNSV6 Primary DNS IPV6
 * @param SecondryDNSV6 Secondry DNS IPV6
 * @param UMTSGrantedQoS UMTS Granted Qos
 * @param SecondaryDNSV4
 * @param Mtu Maximum Transfer Unit
 * @param DomainList
 * @param IPFamilyPreference
 */
typedef struct{
    uint32_t                        IPv4;
    uint8_t                         ProfileName[128];
    uint32_t                        PDPType;
    uint8_t                         APNName[128];
    uint32_t                        PrimaryDNSV4;
    uint32_t                        SecondaryDNSV4;
    LibPackUMTSQoS                  UMTSGrantedQoS;
    struct wds_GPRSQoS              GPRSGrantedQoS;
    uint8_t                         Username[128];
    uint32_t                        Authentication;
    struct wds_ProfileIdentifier    ProfileID;
    uint32_t                        GWAddressV4;
    uint32_t                        SubnetMaskV4;
    uint8_t                         PCSCFAddrPCO;
    struct wds_PCSCFIPv4ServerAddressList ServerAddrList;
    struct wds_PCSCFFQDNAddressList       PCSCFFQDNAddrList;
    uint16_t                        PrimaryDNSV6[8];
    uint16_t                        SecondaryDNSV6[8];
    uint32_t                        Mtu;
    struct wds_DomainNameList       DomainList;
    uint8_t                         IPFamilyPreference;
    uint8_t                         IMCNflag;
    uint16_t                        Technology;
    struct wds_IPV6AddressInfo      IPV6AddrInfo;
    struct wds_IPV6GWAddressInfo    IPV6GWAddrInfo;
} unpack_wds_SLQSGetRuntimeSettings_t;

/**
 * get runtime settings unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSGetRuntimeSettings(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSGetRuntimeSettings_t *pOutput
        );
/**
 * Network information structure
 *
 *  @param  NetworkType
 *          - Values:
 *              - 0 - 3GPP
 *              - 1 - 3GPP2
 *
 *  @param  RATMask
 *          - Radio Access Technology (RAT) mask to
 *          indicate the type of technology.
 *          - Values:
 *              - 0 - Don't Care
 *              - 0x8000 - NULL Bearer
 *          - CDMA RAT mask values:
 *              - 0x01 - CDMA_1x
 *              - 0x02 - EVDO_REV0
 *              - 0x04 - EVDO_REVA
 *              - 0x08 - EVDO_REVB
 *              - 0x10 - EHRPD
 *              - 0x20 - FMC
 *          - UMTS RAT mask values:
 *              - 0x01 - WCDMA
 *              - 0x02 - GPRS
 *              - 0x04 - HSDPA
 *              - 0x08 - HSUPA
 *              - 0x10 - EDGE
 *              - 0x20 - LTE
 *              - 0x40 - HSDPA+
 *              - 0x80 - DC_HSDPA+
 *              - 0x100 - 64_QAM
 *              - 0x200 - TDSCDMA
 *
 *  @param  SOMask
 *          - Service Option (SO) mask to indicate the
 *            service option or type of application.
 *          - Values:
 *              - 0 - Don't Care
 *          - CDMA 1x SO mask values:
 *              - 0x01 - CDMA_1X_IS95
 *              - 0x02 - CDMA_1X_IS2000
 *              - 0x04 - CDMA_1X_IS2000_REL_A
 *          - CDMA EV-DO Rev 0 SO mask values:
 *              - 0x01 - DPA
 *          - CDMA EV-DO Rev A SO mask values:
 *              - 0x01 - DPA
 *              - 0x02 - MFPA
 *              - 0x04 - EMPA
 *              - 0x08 - EMPA_EHRPD
 *          - CDMA EV-DO Rev B SO mask values:
 *              - 0x01 - DPA
 *              - 0x02 - MFPA
 *              - 0x04 - EMPA
 *              - 0x08 - EMPA_EHRPD
 *              - 0x10 - MMPA
 *              - 0x20 - MMPA_EHRPD
 */
typedef struct
{
    uint8_t  NetworkType;
    uint32_t RATMask;
    uint32_t SOMask;
}wds_currNetworkInfo;

/**
 * @param xferStatAvail transfer statistic available
 * @param tx_bytes transmit bytes
 * @param rx_bytes received bytes
 * @param tx_pkts transmit packets
 * @param rx_pkts received packets
 * @param mipstatAvail Mobile IP status available
 * @param mipStatus Mobile IP status
 * @param dBTechAvail Data Bearer technology available
 * @param dBTechnology Data Bearer technology
 * @param dormancyStatAvail Dormancy status available
 * @param dormancyStatus Dormancy status
 * @param currDBTechAvail Current Data Bearer technology available
 * @param ratMask RAT mask to indicate type of technology
 * @param soMask SO mask to indicate the service type
 * @param dataSysStatAvail Data System Status available
 * @param prefNetwork preferred network
 * @param currNWInfo Current Network Info
 */
typedef struct{
    uint8_t             xferStatAvail;
    uint64_t            tx_bytes;
    uint64_t            rx_bytes;
    uint64_t            tx_pkts;
    uint64_t            rx_pkts;
    uint8_t             mipstatAvail;
    uint32_t            mipStatus;
    uint8_t             dBTechAvail;
    uint32_t            dBTechnology;
    uint8_t             dormancyStatAvail;
    uint32_t            dormancyStatus;
    uint8_t             currDBTechAvail;
    uint32_t            ratMask;
    uint32_t            soMask;
    uint8_t             dataSysStatAvail;
    uint8_t             prefNetwork;
    uint8_t             netInfoLen;
    wds_currNetworkInfo currNWInfo[255];
} unpack_wds_SLQSSetWdsEventCallback_ind_t;

/**
 * set event callback unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSSetWdsEventCallback_ind(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSSetWdsEventCallback_ind_t *pOutput
        );

/**
 * set event callback unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSSetWdsEventCallback(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 * @param dataBearer data bearer
 * @param dormancyStatus dormancy status
 * @param mobileIP mobile IP
 * @param currentDataBearer current data bearer
 * @param dataSystemStatus data system status
 * @param interval interval
 */
typedef struct{
    uint8_t   dataBearer;
    uint8_t   dormancyStatus;
    uint8_t   mobileIP;
    uint8_t   transferStats;
    uint8_t   currentDataBearer;
    uint8_t   dataSystemStatus;
    uint8_t   interval;
} pack_wds_SLQSSetWdsEventCallback_t;


/**
 * set event callback pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: No
 */
int pack_wds_SLQSSetWdsEventCallback(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSSetWdsEventCallback_t     *reqArg
        );

/**
 * @param pReqSettings Requested Settings (Optional Parameter)
          - Set bits to 1, corresponding to requested information.
            All other bits must be set to 0.
          - If the values are not available, the corresponding TLVs are
            not returned in the response.
          - Absence of this mask TLV results in the device returning all
            of the available information corresponding to bits 0 through 12.
          - In cases where the information from bit 13 or greater is required,
            this TLV with all the necessary bits set must be present in the request.
          - Values
            - Bit 0 - Profile identifier
            - Bit 1 - Profile name
            - Bit 2 - PDP type
            - Bit 3 - APN name
            - Bit 4 - DNS address
            - Bit 5 - UMTS/GPRS granted QoS
            - Bit 6 - Username
            - Bit 7 - Authentication Protocol
            - Bit 8 - IP address
            - Bit 9 - Gateway info (address and subnet mask)
            - Bit 10 - PCSCF address using PCO flag
            - Bit 11 - PCSCF server address list
            - Bit 12 - PCSCF domain name list
            - Bit 13 - MTU
            - Bit 14 - domain name list
            - Bit 15 - IP family
            - Bit 16 - IM_CM flag
            - Bit 17 - Technology name
            - Bit 18 - Operator reserved PCO
 */
typedef struct{
    uint32_t         *pReqSettings;
} pack_wds_SLQSGetRuntimeSettings_t;

/**
 * get runtime settings pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSGetRuntimeSettings(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSGetRuntimeSettings_t     *reqArg
        );

/**
 * This structure contains the UMTS Quality Of Service Information
 *
 *  @param  trafficClass
 *          - 0x00 - Subscribed
 *          - 0x01 - Conversational
 *          - 0x02 - Streaming
 *          - 0x03 - Interactive
 *          - 0x04 - Background
 *
 *  @param  maxUplinkBitrate
 *          - Maximum uplink bit rate in bits/sec
 *
 *  @param  maxDownlinkBitrate
 *          - Maximum downlink bit rate in bits/sec
 *
 *  @param  grntUplinkBitrate
 *          - Guaranteed uplink bit rate in bits/sec
 *
 *  @param  grntDownlinkBitrate
 *          - Guaranteed downlink bit rate in bits/sec
 *
 *  @param  qosDeliveryOrder - Qos delivery order
 *          - 0x00 - Subscribe
 *          - 0x01 - Delivery order on
 *          - 0x02 - Delivery order off
 *
 *  @param  maxSDUSize
 *          - Maximum SDU size
 *
 *  @param  sduErrorRatio - SDU error ratio
 *          - Target value for fraction of SDUs lost or
 *            detected as erroneous.
 *          - 0x00 - Subscribe
 *          - 0x01 - 1*10^(-2)
 *          - 0x02 - 7*10^(-3)
 *          - 0x03 - 1*10^(-3)
 *          - 0x04 - 1*10^(-4)
 *          - 0x05 - 1*10^(-5)
 *          - 0x06 - 1*10^(-6)
 *          - 0x07 - 1*10^(-1)
 *
 *  @param  resBerRatio - Residual bit error ratio
 *          - Target value for undetected bit error ratio in
 *            in the delivered SDUs.
 *          - 0x00 - Subscribe
 *          - 0x01 - 5*10^(-2)
 *          - 0x02 - 1*10^(-2)
 *          - 0x03 - 5*10^(-3)
 *          - 0x04 - 4*10^(-3)
 *          - 0x05 - 1*10^(-3)
 *          - 0x06 - 1*10^(-4)
 *          - 0x07 - 1*10^(-5)
 *          - 0x08 - 1*10^(-6)
 *          - 0x09 - 1*10^(-8)
 *
 *  @param  deliveryErrSDU - delivery of erroneous SDUs
 *          - Indicates whether SDUs detected as erroneous shall be
 *            delivered or not.
 *          - 0x00 - Subscribe
 *          - 0x01 - 5*10^(-2)
 *          - 0x02 - 1*10^(-2)
 *          - 0x03 - 5*10^(-3)
 *          - 0x04 - 4*10^(-3)
 *          - 0x05 - 1*10^(-3)
 *          - 0x06 - 1*10^(-4)
 *          - 0x07 - 1*10^(-5)
 *          - 0x08 - 1*10^(-6)
 *          - 0x09 - 1*10^(-8)
 *
 *  @param  transferDelay - Transfer delay (ms)
 *          - Indicates the targeted time between a request to transfer an
 *            SDU at one SAP to its delivery at the other SAP in milliseconds.
 *
 *  @param  trafficPriority - Transfer handling priority
 *          - Specifies the relative importance for handling of SDUs that
 *            belong to the UMTS bearer, compared to the SDUs of other bearers.
 */
struct wds_UMTSMinQoS
{
    uint8_t  trafficClass;
    uint32_t maxUplinkBitrate;
    uint32_t maxDownlinkBitrate;
    uint32_t grntUplinkBitrate;
    uint32_t grntDownlinkBitrate;
    uint8_t  qosDeliveryOrder;
    uint32_t maxSDUSize;
    uint8_t  sduErrorRatio;
    uint8_t  resBerRatio;
    uint8_t  deliveryErrSDU;
    uint32_t transferDelay;
    uint32_t trafficPriority;
};


/**
 * This structure contains Input parameters of SLQSCreateProfile
 * and SLQSModifyProfile and output parameters of SLQSGetProfileSettings
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  @param  pProfileName
 *          - One or more bytes describing the profile
 *
 *  @param  pProfilenameSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pProfileName field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pPDPType
 *          - Packet Data Protocol (PDP) type specifies the
 *            type of data payload exchanged over the air link
 *            when the packet data session is established with
 *            this profile
 *            - 0x00 - PDP-IP (IPv4)
 *            - 0x01 - PDP-PPP
 *            - 0x02 - PDP-IPV6
 *            - 0x03 - PDP-IPV4V6
 *
 *  @param  pPdpHdrCompType
 *          - PDP header compression type
 *            - 0 - PDP header compression is OFF
 *            - 1 - Manufacturer preferred compression
 *            - 2 - PDP header compression based on RFC 1144
 *            - 3 - PDP header compression based on RFC 25074
 *                  PDP header compression based on RFC 3095
 *
 *  @param  pPdpDataCompType
 *          - PDP data compression type
 *            - 0 - PDP data compression is OFF
 *            - 1 - Manufacturer preferred compression
 *            - 2 - V.42BIS data compression
 *            - 3 - V.44 data compression
 *
 *  @param  pAPNName
 *          - Access point name
 *
 *  @param  pAPNnameSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pAPNName field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pPriDNSIPv4AddPref
 *          - Primary DNS IPv4 Address Preference
 *
 *  @param  pSecDNSIPv4AddPref
 *          - Secondary DNS IPv4 Address Preference
 *
 *  @param  pUMTSReqQoS
 *          - UMTS Requested QoS
 *
 *  @param  pUMTSMinQoS
 *          - UMTS Minimum QoS
 *
 *  @param  pGPRSRequestedQoS
 *          - GPRS Minimum QoS
 *
 *  @param  pUsername
 *          - User name
 *
 *  @param  pUsernameSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pUsername field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pPassword
 *          - Password
 *
 *  @param  pPasswordSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pPassword field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pAuthenticationPref
 *          - Authentication Preference
 *             - Bit map that indicates the authentication
 *               algorithm preference
 *               - Bit 0 - PAP preference
 *                 - 0 - PAP is never performed
 *                 - 1 - PAP may be performed
 *               - Bit 1 - CHAP preference
 *                 - 0 - CHAP is never performed
 *                 - 1 - CHAP may be performed
 *               - If more than one bit is set, then the device decides
 *                 which authentication procedure is performed while setting
 *                 up the data session. For example, the device may have a
 *                 policy to select the most secure authentication mechanism.
 *
 *  @param  pIPv4AddrPref
 *          - IPv4 Address Preference
 *
 *  @param  pPcscfAddrUsingPCO
 *          - P-CSCF Address using PCO Flag
 *            - 1 - (TRUE) implies request PCSCF address using PCO
 *            - 0 - (FALSE) implies do not request
 *                  By default, this value is 0
 *
 *  @param  pPdpAccessConFlag
 *          - PDP access control flag
 *            - 0 - PDP access control none
 *            - 1 - PDP access control reject
 *            - 2 - PDP access control permission
 *
 *  @param  pPcscfAddrUsingDhcp
 *          - P-CSCF address using DHCP
 *            - 1 - (TRUE) implies Request PCSCF address using DHCP
 *            - 0 - (FALSE) implies do not request
 *                  By default, value is 0
 *
 *  @param  pImCnFlag
 *          - IM CN flag
 *            - 1 - (TRUE) implies request IM CN flag for
 *                   this profile
 *            - 0 - (FALSE) implies do not request IM CN
 *                  flag for this profile
 *
 *  @param  pTFTID1Params
 *          - Traffic Flow Template
 *
 *  @param  pTFTID2Params
 *          - Traffic Flow Template
 *
 *  @param  pPdpContext
 *          - PDP context number
 *
 *  @param  pSecondaryFlag
 *          - PDP context secondary flag
 *            - 1 - (TRUE) implies this is secondary profile
 *            - 0 - (FALSE) implies this is not secondary profile
 *
 *  @param  pPrimaryID
 *          - PDP context primary ID
 *          - function SLQSGetProfileSettings() returns a default value
 *            0xFF if this parameter is not returned by the device
 *
 *  @param  pIPv6AddPref
 *          - IPv6 address preference
 *            Preferred IPv6 address to be assigned to the TE; actual
 *            assigned address is negotiated with the network and may
 *            differ from this value; if not specified, the IPv6
 *            address is obtained automatically from the network
 *
 *  @param  pUMTSReqQoSSigInd
 *          - UMTS requested QoS with Signalling Indication flag
 *
 *  @param  pUMTSMinQoSSigInd
 *          - UMTS minimum QoS with Signalling Indication flag
 *
 *  @param  pPrimaryDNSIPv6addpref
 *          - Primary DNS IPv6 address preference
 *            - The value may be used as a preference during
 *              negotiation with the network; if not specified, the
 *              wireless device will attempt to obtain the DNS
 *              address automatically from the network; the
 *              negotiated value is provided to the host via DHCP
 *
 *  @param  pSecondaryDNSIPv6addpref
 *          - Secondary DNS IPv6 address preference
 *
 *  @param  paddrAllocationPref
 *          - DHCP/NAS preference
 *            - This enumerated value may be used to indicate
 *              the address allocation preference
 *               - 0 - NAS signaling is used for address allocation
 *               - 1 - DHCP is used for address allocation
 *
 *  @param  pQosClassID
 *          - 3GPP LTE QoS parameters
 *
 *  @param  pAPNDisabledFlag
 *          - Optional 1 uint8_t Flag indicating if the APN is disabled/enabled
 *          - If set, the profile can not be used for making data calls
 *          - Any data call is failed locally
 *          - Values:
 *            - 0 - FALSE(default)
 *            - 1 - True
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  @param  pPDNInactivTimeout
 *          - Optional 4 Bytes indicating the duration of inactivity timer
 *            in seconds
 *          - If the PDP context/PDN connection is inactive for this duration
 *            i.e. No data Tx/Rx occurs, the PDP context/PDN connection is
 *            disconnected
 *          - Default value of zero indicates infinite value
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  @param  pAPNClass
 *          - Optional 1 uint8_t numeric identifier representing the APN in profile
 *          - Can be set and queried but is not used by the modem
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 */
typedef struct
{
    uint8_t                    *pProfilename;
    uint16_t                   *pProfilenameSize;
    uint8_t                    *pPDPtype;
    uint8_t                    *pPdpHdrCompType;
    uint8_t                    *pPdpDataCompType;
    uint8_t                    *pAPNName;
    uint16_t                   *pAPNnameSize;
    uint32_t                   *pPriDNSIPv4AddPref;
    uint32_t                   *pSecDNSIPv4AddPref;
    LibPackUMTSQoS             *pUMTSReqQoS;
    LibPackUMTSQoS             *pUMTSMinQoS;
    LibPackGPRSRequestedQoS    *pGPRSRequestedQos;
    LibPackGPRSRequestedQoS    *pGPRSMinimumQoS;
    uint8_t                    *pUsername;
    uint16_t                    *pUsernameSize;
    uint8_t                    *pPassword;
    uint16_t                   *pPasswordSize;
    uint8_t                    *pAuthenticationPref;
    uint32_t                   *pIPv4AddrPref;
    uint8_t                    *pPcscfAddrUsingPCO;
    uint8_t                    *pPdpAccessConFlag;
    uint8_t                    *pPcscfAddrUsingDhcp;
    uint8_t                    *pImCnFlag;
    LibPackTFTIDParams         *pTFTID1Params;
    LibPackTFTIDParams         *pTFTID2Params;
    uint8_t                    *pPdpContext;
    uint8_t                    *pSecondaryFlag;
    uint8_t                    *pPrimaryID;
    uint16_t                   *pIPv6AddPref;
    LibPackUMTSReqQoSSigInd    *pUMTSReqQoSSigInd;
    LibPackUMTSReqQoSSigInd    *pUMTSMinQosSigInd;
    uint16_t                   *pPriDNSIPv6addpref;
    uint16_t                   *pSecDNSIPv6addpref;
    uint8_t                    *pAddrAllocPref;
    LibPackQosClassID          *pQosClassID;
    uint8_t                    *pAPNDisabledFlag;
    uint32_t                   *pPDNInactivTimeout;
    uint8_t                    *pAPNClass;
} LibPackprofile_3GPP;
/**
 * This structure contains the 3GPP2 profile parameters
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  @param  pNegoDnsSrvrPref
 *          - Negotiate DNS Server Preference
 *            - 1 - (TRUE)implies request DNS addresses
 *             from the PDSN
 *            - 0 - (FALSE)implies do not request DNS
 *             addresses from the PDSN
 *            - Default value is 1 (TRUE)
 *
 *  @param  pPppSessCloseTimerDO
 *          - PPP Session Close Timer for DO
 *            - Timer value (in seconds) on DO indicating how
 *              long the PPP Session should linger before
 *              closing down
 *
 *  @param  pPppSessCloseTimer1x
 *          - PPP Session Close Timer for 1X
 *            - Timer value (in seconds) on 1X indicating how
 *              long the PPP session should linger before closing down
 *
 *  @param  pAllowLinger
 *          - Allow/disallow lingering of interface
 *            - 1 -(TRUE) implies allow lingering
 *            - 0 -(FALSE) implies do not allow lingering
 *
 *  @param  pLcpAckTimeout
 *          - LCP ACK Timeout
 *            - Value of LCP ACK Timeout in milliseconds
 *
 *  @param  pIpcpAckTimeout
 *          - IPCP ACK Timeout
 *            - Value of IPCP ACK Timeout in milliseconds
 *
 *  @param  pAuthTimeout
 *          - AUTH Timeout
 *            - Value of Authentication Timeout in milliseconds
 *
 *  @param  pLcpCreqRetryCount
 *          - LCP Configuration Request Retry Count
 *
 *  @param  pIpcpCreqRetryCount
 *          - IPCP Configuration Request Retry Count
 *
 *  @param  pAuthRetryCount
 *          - Authentication Retry Count value
 *
 *  @param  pAuthProtocol
 *          - Authentication Protocol
 *            - 1 - PAP
 *            - 2 - CHAP
 *            - 3 - PAP or CHAP
 *
 *  @param  pUserId
 *          - User ID to be used during data network authentication
 *          - maximum length allowed is 127 bytes;
 *          - QMI_ERR_ARG_TOO_LONG will be returned if the storage on the
 *            wireless device is insufficient in size to hold the value.
 *
 *  @param  pUserIdSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pUserId field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pAuthPassword
 *          - Password to be used during data network authentication;
 *          - maximum length allowed is 127 bytes
 *          - QMI_ERR_ARG_TOO_LONG will be returned if the storage on the
 *            wireless device is insufficient in size to hold the value.
 *
 *  @param  pAuthPasswordSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pAuthPassword field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pDataRate
 *          - Data Rate Requested
 *            - 0 - Low (Low speed Service Options (SO15) only)
 *            - 1 - Medium (SO33 + low R-SCH)
 *            - 2 - High (SO33 + high R-SCH)
 *            - Default is 2
 *
 *  @param  pAppType
 *          - Application Type:
 *            - 0x00000001 - Default Application Type
 *            - 0x00000020 - LBS Application Type
 *            - 0x00000040 - Tethered Application Type
 *            - This parameter is not used while creating/modifying a profile
 *
 *  @param  pDataMode
 *          - Data Mode to use:
 *            - 0 - CDMA or HDR (Hybrid 1X/1xEV-DO)
 *            - 1 - CDMA Only (1X only)
 *            - 2 - HDR Only (1xEV-DO only)
 *            - Default is 0
 *
 *  @param  pAppPriority
 *          - Application Priority
 *            - Numerical 1 uint8_t value defining the application
 *              priority; higher value implies higher priority
 *            - This parameter is not used while creating/modifying a profile
 *
 *  @param  pApnString
 *          - String representing the Access Point Name
 *          - maximum length allowed is 100 bytes
 *          - QMI_ERR_ARG_TOO_LONG will be returned if the APN
 *            name is too long.
 *
 *  @param  pApnStringSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pApnString field. Size of this parameter is 2
 *            bytes.
 *
 *  @param  pPdnType
 *          - Packed Data Network Type Requested:
 *            - 0 - IPv4 PDN Type
 *            - 1 - IPv6 PDN Type
 *            - 2 - IPv4 or IPv6 PDN Type
 *            - 3 - Unspecified PDN Type (implying no preference)
 *
 *  @param  pIsPcscfAddressNedded
 *          - This boolean value is used to control if PCSCF
 *            address is requested from PDSN
 *            - 1 -(TRUE) implies request for PCSCF value from the PDSN
 *            - 0 -(FALSE) implies do not request for PCSCF value from the PDSN
 *
 *  @param  pPrimaryV4DnsAddress
 *          - IPv4 Primary DNS address
 *            - The Primary IPv4 DNS address that can be statically assigned
 *              to the UE
 *
 *  @param  pSecondaryV4DnsAddress
 *          - IPv4 Secondary DNS address
 *            - The Secondary IPv4 DNS address that can be statically assigned
 *              to the UE
 *
 *  @param  pPriV6DnsAddress
 *          - Primary IPv6 DNS address
 *            - The Primary IPv6 DNS address that can be statically assigned
 *              to the UE
 *
 *  @param  pSecV6DnsAddress
 *          - Secondary IPv6 DNS address
 *            - The Secondary IPv6 DNS address that can be statically assigned
 *              to the UE
 *
 *  @param  pRATType
 *          - Optional 1 uint8_t Flag indicating RAT Type
 *          - Values:
 *            - 1 - HRPD
 *            - 2 - EHRPD
 *            - 3 - HRPD_EHRPD
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  @param  pAPNEnabled3GPP2
 *          - Optional 1 uint8_t Flag indicating if the APN is disabled/enabled
 *          - If disabled, the profile can not be used for making data calls
 *          - Values:
 *            - 0 - Disabled
 *            - 1 - Enabled(default value)
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  @param  pPDNInactivTimeout3GPP2
 *          - Optional 4 Bytes indicating the duration of inactivity timer
 *            in seconds
 *          - If the PDP context/PDN connection is inactive for this duration
 *            i.e. No data Tx/Rx occurs, the PDP context/PDN connection is
 *            disconnected
 *          - Default value of zero indicates infinite value
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  @param  pAPNClass3GPP2
 *          - Optional 1 uint8_t numeric identifier representing the APN in profile
 *          - Can be set and queried but is not used by the modem
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 */
typedef struct 
{
    uint8_t   *pNegoDnsSrvrPref;
    uint32_t  *pPppSessCloseTimerDO;
    uint32_t  *pPppSessCloseTimer1x;
    uint8_t   *pAllowLinger;
    uint16_t  *pLcpAckTimeout;
    uint16_t  *pIpcpAckTimeout;
    uint16_t  *pAuthTimeout;
    uint8_t   *pLcpCreqRetryCount;
    uint8_t   *pIpcpCreqRetryCount;
    uint8_t   *pAuthRetryCount;
    uint8_t   *pAuthProtocol;
    uint8_t   *pUserId;
    uint16_t  *pUserIdSize;
    uint8_t   *pAuthPassword;
    uint16_t  *pAuthPassword_tSize;
    uint8_t   *pDataRate;
    uint32_t  *pAppType;
    uint8_t   *pDataMode;
    uint8_t   *pAppPriority;
    uint8_t   *pApnString;
    uint16_t  *pApnStringSize;
    uint8_t   *pPdnType;
    uint8_t   *pIsPcscfAddressNedded;
    uint32_t  *pPrimaryV4DnsAddress;
    uint32_t  *pSecondaryV4DnsAddress;
    uint16_t  *pPriV6DnsAddress;
    uint16_t  *pSecV6DnsAddress;
    uint8_t   *pRATType;
    uint8_t   *pAPNEnabled3GPP2;
    uint32_t  *pPDNInactivTimeout3GPP2;
    uint8_t   *pAPNClass3GPP2;
}LibPackprofile_3GPP2;

/**
 * This union  consist of profile_3GPP and profile_3GPP2
 * out of which one will be used to create profile.
 *
 */
typedef union
{
    LibPackprofile_3GPP  SlqsProfile3GPP;
    LibPackprofile_3GPP2 SlqsProfile3GPP2;
}wds_profileInfo;

/**
 *  @param  ProfileID
 *          - 1 to 16 for 3GPP profile
 *          - 101 to 106 for 3GPP2 profile
 *
 *  @param  ProfileType
 *          - Identifies the technology type of the profile
 *            - 0x00 - 3GPP
 *            - 0x01 - 3GPP2
 *            - NULL is not allowed
 *
 *  @param  curProfile
 *          - union of 3GPP and 3GPP2 profile
 *
 *  @note   - If profileID is NULL, 3GPP profile will be created
 *            and index will be assigned based on availability in device.
 *  @note   - If profileID is not NULL depending on pProfileType 3GPP/3GPP2
 *            relevant profile will be created
 *
 */
typedef struct
{
    uint8_t   *pProfileId;
    uint8_t   *pProfileType;
    wds_profileInfo  *pCurProfile;
} pack_wds_SLQSCreateProfile_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSCreateProfile(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSCreateProfile_t *reqArg
        );

typedef struct 
{
    uint8_t   ProfileType;
    uint8_t   ProfileIndex;
    uint16_t  ExtErrorCode;
}PackCreateProfileOut;

/**
 * @param profile type
 * @param profile index
 * @param extended error
 */

typedef struct
{
    PackCreateProfileOut *pCreateProfileOut;
    uint8_t  *pProfileID;
    uint16_t   Tlvresult;
} unpack_wds_SLQSCreateProfile_t;

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput     response structure to fill
 * @param[in]   pProfileId  profile id pointer passed in req
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_wds_SLQSCreateProfile(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSCreateProfile_t *pOutput
        );

/**
 *  @param  ProfileID
 *          - 1 to 16 for 3GPP profile
 *          - 101 to 106 for 3GPP2 profile
 *
 *  @param  ProfileType
 *          - Identifies the technology type of the profile
 *            - 0x00 - 3GPP
 *            - 0x01 - 3GPP2
 *            - NULL is not allowed
 *
 *  @param  curProfile
 *          - union of 3GPP and 3GPP2 profile
 *
 *  @note   - If profileID is NULL, 3GPP profile will be created
 *            and index will be assigned based on availability in device.
 *  @note   - If profileID is not NULL depending on pProfileType 3GPP/3GPP2
 *            relevant profile will be created
 *
 */
typedef struct
{
    uint8_t   *pProfileId;
    uint8_t   *pProfileType;
    wds_profileInfo  curProfile;
} pack_wds_SLQSModifyProfile_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSModifyProfile(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSModifyProfile_t *reqArg
        );

/**
 * @param extended error
 */
typedef struct
{
   uint16_t  *pExtErrorCode;
} unpack_wds_SLQSModifyProfile_t;


/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_wds_SLQSModifyProfile(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSModifyProfile_t *pOutput
        );

/**
 *  @param  ProfileID
 *          - 1 to 16 for 3GPP profile
 *          - 101 to 106 for 3GPP2 profile
 *
 *  @param  ProfileType
 *          - Identifies the technology type of the profile
 *            - 0x00 - 3GPP
 *            - 0x01 - 3GPP2
 *
 *  @note   - If profileID is NULL, 3GPP profile will be fetched
 *            and index will be assigned based on availability in device.
 *  @note   - If profileID is not NULL depending on pProfileType 3GPP/3GPP2
 *            relevant profile will be fetched
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
typedef struct
{
    uint8_t   ProfileId;
    uint8_t   ProfileType;
} pack_wds_SLQSGetProfileSettings_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSGetProfileSettings(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_wds_SLQSGetProfileSettings_t *reqArg
        );




/**
 * @param extended error
 * @param profile info
*/

/**
 * This structure contains Input parameters of SLQSCreateProfile
 * and SLQSModifyProfile and output parameters of SLQSGetProfileSettings
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  \param  pProfileName
 *          - One or more uint8_ts describing the profile
 *
 *  \param  pProfilenameSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pProfileName field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pPDPType
 *          - Packet Data Protocol (PDP) type specifies the
 *            type of data payload exchanged over the air link
 *            when the packet data session is established with
 *            this profile
 *            - 0x00 - PDP-IP (IPv4)
 *            - 0x01 - PDP-PPP
 *            - 0x02 - PDP-IPV6
 *            - 0x03 - PDP-IPV4V6
 *
 *  \param  pPdpHdrCompType
 *          - PDP header compression type
 *            - 0 - PDP header compression is OFF
 *            - 1 - Manufacturer preferred compression
 *            - 2 - PDP header compression based on RFC 1144
 *            - 3 - PDP header compression based on RFC 25074
 *                  PDP header compression based on RFC 3095
 *
 *  \param  pPdpDataCompType
 *          - PDP data compression type
 *            - 0 - PDP data compression is OFF
 *            - 1 - Manufacturer preferred compression
 *            - 2 - V.42BIS data compression
 *            - 3 - V.44 data compression
 *
 *  \param  pAPNName
 *          - Access point name
 *
 *  \param  pAPNnameSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pAPNName field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pPriDNSIPv4AddPref
 *          - Primary DNS IPv4 Address Preference
 *
 *  \param  pSecDNSIPv4AddPref
 *          - Secondary DNS IPv4 Address Preference
 *
 *  \param  pUMTSReqQoS
 *          - UMTS Requested QoS
 *
 *  \param  pUMTSMinQoS
 *          - UMTS Minimum QoS
 *
 *  \param  pGPRSRequestedQoS
 *          - GPRS Minimum QoS
 *
 *  \param  pUsername
 *          - User name
 *
 *  \param  pUsernameSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pUsername field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pPassword
 *          - Password
 *
 *  \param  pPasswordSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pPassword field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pAuthenticationPref
 *          - Authentication Preference
 *             - Bit map that indicates the authentication
 *               algorithm preference
 *               - Bit 0 - PAP preference
 *                 - 0 - PAP is never performed
 *                 - 1 - PAP may be performed
 *               - Bit 1 - CHAP preference
 *                 - 0 - CHAP is never performed
 *                 - 1 - CHAP may be performed
 *               - If more than one bit is set, then the device decides
 *                 which authentication procedure is performed while setting
 *                 up the data session. For example, the device may have a
 *                 policy to select the most secure authentication mechanism.
 *
 *  \param  pIPv4AddrPref
 *          - IPv4 Address Preference
 *
 *  \param  pPcscfAddrUsingPCO
 *          - P-CSCF Address using PCO Flag
 *            - 1 - (TRUE) implies request PCSCF address using PCO
 *            - 0 - (FALSE) implies do not request
 *                  By default, this value is 0
 *
 *  \param  pPdpAccessConFlag
 *          - PDP access control flag
 *            - 0 - PDP access control none
 *            - 1 - PDP access control reject
 *            - 2 - PDP access control permission
 *
 *  \param  pPcscfAddrUsingDhcp
 *          - P-CSCF address using DHCP
 *            - 1 - (TRUE) implies Request PCSCF address using DHCP
 *            - 0 - (FALSE) implies do not request
 *                  By default, value is 0
 *
 *  \param  pImCnFlag
 *          - IM CN flag
 *            - 1 - (TRUE) implies request IM CN flag for
 *                   this profile
 *            - 0 - (FALSE) implies do not request IM CN
 *                  flag for this profile
 *
 *  \param  pTFTID1Params
 *          - Traffic Flow Template
 *
 *  \param  pTFTID2Params
 *          - Traffic Flow Template
 *
 *  \param  pPdpContext
 *          - PDP context number
 *
 *  \param  pSecondaryFlag
 *          - PDP context secondary flag
 *            - 1 - (TRUE) implies this is secondary profile
 *            - 0 - (FALSE) implies this is not secondary profile
 *
 *  \param  pPrimaryID
 *          - PDP context primary ID
 *          - function SLQSGetProfileSettings() returns a default value
 *            0xFF if this parameter is not returned by the device
 *
 *  \param  pIPv6AddPref
 *          - IPv6 address preference
 *            Preferred IPv6 address to be assigned to the TE; actual
 *            assigned address is negotiated with the network and may
 *            differ from this value; if not specified, the IPv6
 *            address is obtained automatically from the network
 *
 *  \param  pUMTSReqQoSSigInd
 *          - UMTS requested QoS with Signalling Indication flag
 *
 *  \param  pUMTSMinQoSSigInd
 *          - UMTS minimum QoS with Signalling Indication flag
 *
 *  \param  pPrimaryDNSIPv6addpref
 *          - Primary DNS IPv6 address preference
 *            - The value may be used as a preference during
 *              negotiation with the network; if not specified, the
 *              wireless device will attempt to obtain the DNS
 *              address automatically from the network; the
 *              negotiated value is provided to the host via DHCP
 *
 *  \param  pSecondaryDNSIPv6addpref
 *          - Secondary DNS IPv6 address preference
 *
 *  \param  paddrAllocationPref
 *          - DHCP/NAS preference
 *            - This enumerated value may be used to indicate
 *              the address allocation preference
 *               - 0 - NAS signaling is used for address allocation
 *               - 1 - DHCP is used for address allocation
 *
 *  \param  pQosClassID
 *          - 3GPP LTE QoS parameters
 *
 *  \param  pAPNDisabledFlag
 *          - Optional 1 uint8_t Flag indicating if the APN is disabled/enabled
 *          - If set, the profile can not be used for making data calls
 *          - Any data call is failed locally
 *          - Values:
 *            - 0 - FALSE(default)
 *            - 1 - True
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  \param  pPDNInactivTimeout
 *          - Optional 4 uint8_ts indicating the duration of inactivity timer
 *            in seconds
 *          - If the PDP context/PDN connection is inactive for this duration
 *            i.e. No data Tx/Rx occurs, the PDP context/PDN connection is
 *            disconnected
 *          - Default value of zero indicates infinite value
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  \param  pAPNClass
 *          - Optional 1 uint8_t numeric identifier representing the APN in profile
 *          - Can be set and queried but is not used by the modem
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 */
typedef struct 
{
    uint8_t                    *pProfilename;
    uint16_t                    *pProfilenameSize;
    uint8_t                    *pPDPtype;
    uint8_t                    *pPdpHdrCompType;
    uint8_t                    *pPdpDataCompType;
    uint8_t                    *pAPNName;
    uint16_t                    *pAPNnameSize;
    uint32_t                   *pPriDNSIPv4AddPref;
    uint32_t                   *pSecDNSIPv4AddPref;
    LibPackUMTSQoS          *pUMTSReqQoS;
    LibPackUMTSQoS          *pUMTSMinQoS;
    LibPackGPRSRequestedQoS *pGPRSRequestedQos;
    LibPackGPRSRequestedQoS *pGPRSMinimumQoS;
    uint8_t                    *pUsername;
    uint16_t                    *pUsernameSize;
    uint8_t                    *pPassword;
    uint16_t                    *pPasswordSize;
    uint8_t                    *pAuthenticationPref;
    uint32_t                   *pIPv4AddrPref;
    uint8_t                    *pPcscfAddrUsingPCO;
    uint8_t                    *pPdpAccessConFlag;
    uint8_t                    *pPcscfAddrUsingDhcp;
    uint8_t                    *pImCnFlag;
    LibPackTFTIDParams      *pTFTID1Params;
    LibPackTFTIDParams      *pTFTID2Params;
    uint8_t                    *pPdpContext;
    uint8_t                    *pSecondaryFlag;
    uint8_t                    *pPrimaryID;
    uint16_t                  *pIPv6AddPref;
    LibPackUMTSReqQoSSigInd *pUMTSReqQoSSigInd;
    LibPackUMTSReqQoSSigInd *pUMTSMinQosSigInd;
    uint16_t                  *pPriDNSIPv6addpref;
    uint16_t                  *pSecDNSIPv6addpref;
    uint8_t                    *pAddrAllocPref;
    LibPackQosClassID       *pQosClassID;
    uint8_t                    *pAPNDisabledFlag;
    uint32_t                   *pPDNInactivTimeout;
    uint8_t                    *pAPNClass;
}LibpackProfile3GPP;
/**
 * This structure contains the 3GPP2 profile parameters
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 *  \param  pNegoDnsSrvrPref
 *          - Negotiate DNS Server Preference
 *            - 1 - (TRUE)implies request DNS addresses
 *             from the PDSN
 *            - 0 - (FALSE)implies do not request DNS
 *             addresses from the PDSN
 *            - Default value is 1 (TRUE)
 *
 *  \param  pPppSessCloseTimerDO
 *          - PPP Session Close Timer for DO
 *            - Timer value (in seconds) on DO indicating how
 *              long the PPP Session should linger before
 *              closing down
 *
 *  \param  pPppSessCloseTimer1x
 *          - PPP Session Close Timer for 1X
 *            - Timer value (in seconds) on 1X indicating how
 *              long the PPP session should linger before closing down
 *
 *  \param  pAllowLinger
 *          - Allow/disallow lingering of interface
 *            - 1 -(TRUE) implies allow lingering
 *            - 0 -(FALSE) implies do not allow lingering
 *
 *  \param  pLcpAckTimeout
 *          - LCP ACK Timeout
 *            - Value of LCP ACK Timeout in milliseconds
 *
 *  \param  pIpcpAckTimeout
 *          - IPCP ACK Timeout
 *            - Value of IPCP ACK Timeout in milliseconds
 *
 *  \param  pAuthTimeout
 *          - AUTH Timeout
 *            - Value of Authentication Timeout in milliseconds
 *
 *  \param  pLcpCreqRetryCount
 *          - LCP Configuration Request Retry Count
 *
 *  \param  pIpcpCreqRetryCount
 *          - IPCP Configuration Request Retry Count
 *
 *  \param  pAuthRetryCount
 *          - Authentication Retry Count value
 *
 *  \param  pAuthProtocol
 *          - Authentication Protocol
 *            - 1 - PAP
 *            - 2 - CHAP
 *            - 3 - PAP or CHAP
 *
 *  \param  pUserId
 *          - User ID to be used during data network authentication
 *          - maximum length allowed is 127 uint8_ts;
 *          - QMI_ERR_ARG_TOO_LONG will be returned if the storage on the
 *            wireless device is insufficient in size to hold the value.
 *
 *  \param  pUserIdSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pUserId field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pAuthPassword
 *          - Password to be used during data network authentication;
 *          - maximum length allowed is 127 uint8_ts
 *          - QMI_ERR_ARG_TOO_LONG will be returned if the storage on the
 *            wireless device is insufficient in size to hold the value.
 *
 *  \param  pAuthPasswordSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pAuthPassword field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pDataRate
 *          - Data Rate Requested
 *            - 0 - Low (Low speed Service Options (SO15) only)
 *            - 1 - Medium (SO33 + low R-SCH)
 *            - 2 - High (SO33 + high R-SCH)
 *            - Default is 2
 *
 *  \param  pAppType
 *          - Application Type:
 *            - 0x00000001 - Default Application Type
 *            - 0x00000020 - LBS Application Type
 *            - 0x00000040 - Tethered Application Type
 *            - This parameter is not used while creating/modifying a profile
 *
 *  \param  pDataMode
 *          - Data Mode to use:
 *            - 0 - CDMA or HDR (Hybrid 1X/1xEV-DO)
 *            - 1 - CDMA Only (1X only)
 *            - 2 - HDR Only (1xEV-DO only)
 *            - Default is 0
 *
 *  \param  pAppPriority
 *          - Application Priority
 *            - Numerical 1 uint8_t value defining the application
 *              priority; higher value implies higher priority
 *            - This parameter is not used while creating/modifying a profile
 *
 *  \param  pApnString
 *          - String representing the Access Point Name
 *          - maximum length allowed is 100 uint8_ts
 *          - QMI_ERR_ARG_TOO_LONG will be returned if the APN
 *            name is too long.
 *
 *  \param  pApnStringSize;
 *          - This parameter is an input parameter and should be initialised
 *            to the size of pApnString field. Size of this parameter is 2
 *            uint8_ts.
 *
 *  \param  pPdnType
 *          - Packed Data Network Type Requested:
 *            - 0 - IPv4 PDN Type
 *            - 1 - IPv6 PDN Type
 *            - 2 - IPv4 or IPv6 PDN Type
 *            - 3 - Unspecified PDN Type (implying no preference)
 *
 *  \param  pIsPcscfAddressNedded
 *          - This boolean value is used to control if PCSCF
 *            address is requested from PDSN
 *            - 1 -(TRUE) implies request for PCSCF value from the PDSN
 *            - 0 -(FALSE) implies do not request for PCSCF value from the PDSN
 *
 *  \param  pPrimaryV4DnsAddress
 *          - IPv4 Primary DNS address
 *            - The Primary IPv4 DNS address that can be statically assigned
 *              to the UE
 *
 *  \param  pSecondaryV4DnsAddress
 *          - IPv4 Secondary DNS address
 *            - The Secondary IPv4 DNS address that can be statically assigned
 *              to the UE
 *
 *  \param  pPriV6DnsAddress
 *          - Primary IPv6 DNS address
 *            - The Primary IPv6 DNS address that can be statically assigned
 *              to the UE
 *
 *  \param  pSecV6DnsAddress
 *          - Secondary IPv6 DNS address
 *            - The Secondary IPv6 DNS address that can be statically assigned
 *              to the UE
 *
 *  \param  pRATType
 *          - Optional 1 uint8_t Flag indicating RAT Type
 *          - Values:
 *            - 1 - HRPD
 *            - 2 - EHRPD
 *            - 3 - HRPD_EHRPD
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  \param  pAPNEnabled3GPP2
 *          - Optional 1 uint8_t Flag indicating if the APN is disabled/enabled
 *          - If disabled, the profile can not be used for making data calls
 *          - Values:
 *            - 0 - Disabled
 *            - 1 - Enabled(default value)
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  \param  pPDNInactivTimeout3GPP2
 *          - Optional 4 uint8_ts indicating the duration of inactivity timer
 *            in seconds
 *          - If the PDP context/PDN connection is inactive for this duration
 *            i.e. No data Tx/Rx occurs, the PDP context/PDN connection is
 *            disconnected
 *          - Default value of zero indicates infinite value
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 *  \param  pAPNClass3GPP2
 *          - Optional 1 uint8_t numeric identifier representing the APN in profile
 *          - Can be set and queried but is not used by the modem
 *          - This parameter is currently read only and can be read by using
 *            the function SLQSGetProfileSettings().
 *
 */
typedef struct 
{
    uint8_t   *pNegoDnsSrvrPref;
    uint32_t  *pPppSessCloseTimerDO;
    uint32_t  *pPppSessCloseTimer1x;
    uint8_t   *pAllowLinger;
    uint16_t *pLcpAckTimeout;
    uint16_t *pIpcpAckTimeout;
    uint16_t *pAuthTimeout;
    uint8_t   *pLcpCreqRetryCount;
    uint8_t   *pIpcpCreqRetryCount;
    uint8_t   *pAuthRetryCount;
    uint8_t   *pAuthProtocol;
    uint8_t   *pUserId;
    uint16_t   *pUserIdSize;
    uint8_t   *pAuthPassword;
    uint16_t   *pAuthPasswordSize;
    uint8_t   *pDataRate;
    uint32_t  *pAppType;
    uint8_t   *pDataMode;
    uint8_t   *pAppPriority;
    uint8_t   *pApnString;
    uint16_t   *pApnStringSize;
    uint8_t   *pPdnType;
    uint8_t   *pIsPcscfAddressNedded;
    uint32_t  *pPrimaryV4DnsAddress;
    uint32_t  *pSecondaryV4DnsAddress;
    uint16_t *pPriV6DnsAddress;
    uint16_t *pSecV6DnsAddress;
    uint8_t   *pRATType;
    uint8_t   *pAPNEnabled3GPP2;
    uint32_t  *pPDNInactivTimeout3GPP2;
    uint8_t   *pAPNClass3GPP2;
}LibpackProfile3GPP2;

typedef union unpackWdsProfileParam
{
    LibpackProfile3GPP  SlqsProfile3GPP;
    LibpackProfile3GPP2 SlqsProfile3GPP2;
}UnpackQmiProfileInfo;

typedef struct
{
    UnpackQmiProfileInfo curProfile;
    uint16_t           *pExtErrCode;
}UnPackGetProfileSettingOut;

typedef struct
{
/*
wds_profileInfo CurProfile;
uint16_t ExtErrCode;
*/
    UnPackGetProfileSettingOut *pProfileSettings;
    uint8_t  ProfileType;
    uint16_t Tlvresult;
} unpack_wds_SLQSGetProfileSettings_t;

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput     response structure to fill
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_wds_SLQSGetProfileSettings(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_wds_SLQSGetProfileSettings_t *pOutput
        );

/**
 * @param connectionStatus state of the current packet data session
 */
typedef struct {
    uint32_t    connectionStatus;
} unpack_wds_GetSessionState_t;

/**
 * get session state pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_GetSessionState(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen
        );

/**
 * get session state unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetSessionState(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetSessionState_t *pOutput
        );

/**
 * @param profiletype profile type
 */
typedef struct {
    uint32_t profiletype;
} pack_wds_GetDefaultProfile_t;

/**
 * @param pdptype pdp type
 * @param ipaddr ip address
 * @param pridns primary dns
 * @param secdns secondry dns
 * @param ipaddrv6 ip address v6
 * @param pridnsv6 primary dns v6
 * @param secdnsv6 secondry dns v6
 * @param namesize profile name size
 * @param name profile name
 * @param apnsize apn size
 * @param apnname apn name
 * @param usersize username size
 * @param username username
 *
 */
typedef struct{
    uint32_t         pdptype;
    uint32_t         ipaddr;
    uint32_t         pridns;
    uint32_t         secdns;
    uint16_t         ipaddrv6;
    uint16_t         pridnsv6;
    uint16_t         secdnsv6;
    uint32_t         auth;
    uint8_t          namesize;
    int8_t           name[255];
    uint8_t          apnsize;
    int8_t           apnname[255];
    uint8_t          usersize;
    int8_t           username[255];
} unpack_wds_GetDefaultProfile_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_GetDefaultProfile(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_GetDefaultProfile_t *reqParam
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetDefaultProfile(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetDefaultProfile_t *pOutput
        );

/**
 * @param currentChannelTXRate Instantaneous channel Tx rate
 * @param currentChannelRXRate Instantaneous channel Rx rate
 * @param maxChannelTXRate Maximum Tx rate
 * @param maxChannelRXRate Maximum Rx rate
 */
typedef struct{
    uint32_t         currentChannelTXRate;
    uint32_t         currentChannelRXRate;
    uint32_t         maxChannelTXRate;
    uint32_t         maxChannelRXRate;
} unpack_wds_GetConnectionRate_t;

/**
 * get connection rate pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: No
 */
int pack_wds_GetConnectionRate(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen
        );

/**
 * get connection rate unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetConnectionRate(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetConnectionRate_t *pOutput
        );

/**
 * @param statmask packet statistics mask
 */
typedef struct{
    uint32_t         statmask;
} pack_wds_GetPacketStatus_t;

/**
 * @param tXPacketSuccesses Tx Packets OK
 * @param rXPacketSuccesses Rx Packets OK
 * @param tXPacketErrors Tx Packet Errors
 * @param rXPacketErrors Rx Packet Errors
 * @param tXPacketOverflows Tx Overflows
 * @param rXPacketOverflows Rx Overflows
 * @param tXOkBytesCount Tx Bytes OK
 * @param rXOkBytesCount Rx Bytes OK
 * @param tXOKBytesLastCall Last call Tx Bytes OK
 * @param rXOKBytesLastCall Last call Rx Bytes OK
 * @param tXDroppedCount Tx Packets Dropped
 * @param rXDroppedCount Rx Packets Dropped
 */
typedef struct{
    uint32_t        tXPacketSuccesses;
    uint32_t        rXPacketSuccesses;
    uint32_t        tXPacketErrors;
    uint32_t        rXPacketErrors;
    uint32_t        tXPacketOverflows;
    uint32_t        rXPacketOverflows;
    uint64_t        tXOkBytesCount;
    uint64_t        rXOkBytesCount;
    uint64_t        tXOKBytesLastCall;
    uint64_t        rXOKBytesLastCall;
    uint32_t        tXDroppedCount;
    uint32_t        rXDroppedCount;
} unpack_wds_GetPacketStatus_t;

/**
 * get packet status pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_GetPacketStatus(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_GetPacketStatus_t *reqParam
        );

/**
 * get packet status unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetPacketStatus(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetPacketStatus_t *pOutput
        );

/**
 * @param callDuration call duration in milliseconds
 */
typedef struct{
    uint64_t      callDuration;
} unpack_wds_GetSessionDuration_t;

typedef struct{

}pack_wds_GetSessionDuration_t;

/**
 * get session duration pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: No
 */
int pack_wds_GetSessionDuration(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_GetSessionDuration_t *reqParam
        );

/**
 * get session duration unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetSessionDuration(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetSessionDuration_t *pOutput
        );

/**
 * @param dormancyState dormancy status
 */
typedef struct{
    uint32_t    dormancyState;
} unpack_wds_GetDormancyState_t;

typedef struct{
    
}pack_wds_GetDormancyState_t;
/**
 * get dormancy state pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: No
 */
int pack_wds_GetDormancyState(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_GetDormancyState_t *reqParam
        );

/**
 * get dormancy state unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetDormancyState(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetDormancyState_t *pOutput
        );

/**
 * @param profileType profile type
 * @param profileIndex profile index
 */
typedef struct{
    uint8_t profileType;
    uint8_t profileIndex;
} pack_wds_SLQSDeleteProfile_t;

/**
 * @param extendedErrorCode extended error code
 */
typedef struct{
    uint16_t extendedErrorCode;
} unpack_wds_SLQSDeleteProfile_t;

/**
 * delete stored profile pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSDeleteProfile(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_SLQSDeleteProfile_t *reqParam
        );

/**
 * delete stored profile unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSDeleteProfile(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_SLQSDeleteProfile_t *pOutput
        );

/**
 * @param profileType profile type
 * @param pdpType Packet Data Protocol (PDP) type
 * @param ipAddress ip address
 * @param primaryDNS primary dns
 * @param secondaryDNS secondry dns
 * @param authentication authentication type
 * @param name name of the profile
 * @param apnname apn name
 * @param username username of the profile
 * @param password password of profile
 */
typedef struct{
    uint32_t    profileType;
    uint32_t    pdpType;
    uint32_t    ipAddress;
    uint32_t    primaryDNS;
    uint32_t    secondaryDNS;
    uint32_t    authentication;
    uint8_t     *pName;
    uint8_t     *pUsername;
    uint8_t     *pApnname;
    uint8_t     *pPassword;
} pack_wds_SetDefaultProfile_t;

/**
 * set default profile pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SetDefaultProfile(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_SetDefaultProfile_t *reqParam
        );

/**
 * set default profile unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SetDefaultProfile(
        uint8_t   *pResp,
        uint16_t  respLen
        );

/**
 * @param profileList Profile List
 * @param[out] defaultPDNEnabled 
 *        - 0 - disabled 
 *        - 1 - enabled
 * @param[out] _3gppRelease 3GPP release 
 *        - 0 - Release_99 
 *        - 1 - Release_5 
 *        - 2 - Release_6 
 *        - 3 - Release_7 
 *        - 4 - Release_8 
 *        - 5 - Release_9  (In 9x30 and toworads)
 *        - 6 - Release_10 (In 9x30 and toworads)
 *        - 7 - Release_11 (In 9x30 and toworads)
 * @param[out] LTEAttachProfileList
 *          - pointer to WORD array indiciating LTE Attach Profile List
 *              - Optional parameter
 *              - possible values: 1-24
 *              - This setting is only supported for MC/EM74xx onwards
 *              - Please provide attach profiles in order of decreasing priority in this list. 
 *
 * @param[in,out] LTEAttachProfileListLen
 *          - Number of element in pLTEAttachProfileList
 *             - valid range: 1-24
 *             - This setting is only supported for MC/EM74xx onwards
 */
typedef struct{
    uint16_t profileList[5];
    uint8_t  defaultPDNEnabled;
    uint8_t  _3gppRelease;
    uint16_t LTEAttachProfileList[MAX_WDS_3GPP_CONF_LTE_ATTACH_PROFILE_LIST_SIZE];
    uint16_t LTEAttachProfileListLen;
} unpack_wds_SLQSGet3GPPConfigItem_t;

/**
 * get 3Gpp config items pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSGet3GPPConfigItem(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen
        );

/**
 * get 3GPP config items unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSGet3GPPConfigItem(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_SLQSGet3GPPConfigItem_t *pOutput
        );

/**
 * @param profileList Profile List
 * @param defaultPDNEnabled 
 *        - 0 - disabled 
 *        - 1 - enabled
 * @param _3gppRelease 3GPP release 
 *        - 0 - Release_99 
 *        - 1 - Release_5 
 *        - 2 - Release_6 
 *        - 3 - Release_7 
 *        - 4 - Release_8
 * @param LTEAttachProfileList
 *          - pointer to WORD array indiciating LTE Attach Profile List
 *              - Optional parameter
 *              - possible values: 1-24
 *              - This setting is only supported for MC/EM74xx onwards
 *              - Please provide attach profiles in order of decreasing priority in this list. 
 *
 * @param LTEAttachProfileListLen
 *          - Number of element in pLTEAttachProfileList
 *             - valid range: 1-24
 *             - This setting is only supported for MC/EM74xx onwards
 */
typedef struct{
    uint16_t profileList[5];
    uint8_t  defaultPDNEnabled;
    uint8_t  _3gppRelease;
    uint16_t LTEAttachProfileList[24];
    uint16_t LTEAttachProfileListLen;
} pack_wds_SLQSSet3GPPConfigItem_t;

/**
 * set 3Gpp config items pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSSet3GPPConfigItem(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_SLQSSet3GPPConfigItem_t *reqParam
        );

/**
 * set 3GPP config items unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 */
int unpack_wds_SLQSSet3GPPConfigItem(
        uint8_t   *pResp,
        uint16_t  respLen
        );


/**
* @param mipMode mobile IP mode
*/
typedef struct{
uint32_t         mipMode;
} unpack_wds_GetMobileIP_t;
typedef struct{

}pack_wds_GetMobileIP_t;
/**
 * get mobile ip mode pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: No
 */
int pack_wds_GetMobileIP(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_GetMobileIP_t *pReqParam
    );

/**
 * get mobile ip mode unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
*/
int unpack_wds_GetMobileIP(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_GetMobileIP_t *pOutput
    );

/**
 * @param index mobile ip profile identifier
 */
typedef struct{
uint8_t   index;
} pack_wds_GetMobileIPProfile_t;

/**
* @param enabled mobile ip profile state enabled/disabled
* @param address mobile ip profile home address
* @param primaryHA mobile ip profile home agent primary
* @param secondaryHA mobile ip profile secondary home agent address
* @param revTunneling mobile ip profile rev tunneling
* @param naiSize mobile ip profile NAI size
* @param NAI NAI string in ASCII text.
* @param HASPI HA security parameter index.
* @param AAASPI AAA server security parameter index.
* @param HAState Mobile IP Profile HA Key State
* @param AAAState Mobile IP Profile AAA Key State
*/
typedef struct{
uint8_t   enabled;
uint32_t  address;
uint32_t  primaryHA;
uint32_t  secondaryHA;
uint8_t   revTunneling;
uint8_t   naiSize;
int8_t    NAI[255];
uint32_t  HASPI;
uint32_t  AAASPI;
uint32_t  HAState;
uint32_t  AAAState;
} unpack_wds_GetMobileIPProfile_t;

/**
 * get mobile ip profile pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_GetMobileIPProfile(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_GetMobileIPProfile_t *reqParam
    );

/**
 * get mobile ip profile unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetMobileIPProfile(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_GetMobileIPProfile_t *pOutput
    );

/**
* @param NetworkType network type 0-3GPP 1-3GPP2
* @param RATMask Radio access technology mask to indicate technology.
* @param SOMask Service Option mask to indicate the service option.
*/

typedef struct
{
uint8_t  NetworkType;
uint32_t RATMask;
uint32_t SOMask;
}currNetworkInfo;
/**
* @param prefNetwork preferred network
* @param networkInfoLen number of set of currNetworkInfo elements
* @param currNetworkInfo current network infomation.
*/
typedef struct{
uint8_t             prefNetwork;
uint8_t             networkInfoLen;
currNetworkInfo     currNetworkInfo[255];
} unpack_wds_SLQSGetCurrDataSystemStat_t;

typedef struct{

}pack_wds_SLQSGetCurrDataSystemStat_t;
/**
 * get current data system pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSGetCurrDataSystemStat(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SLQSGetCurrDataSystemStat_t *pReqParam
    );

/**
 * get current data system unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSGetCurrDataSystemStat(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_SLQSGetCurrDataSystemStat_t *pOutput
    );

/**
 * @param error last mip status 0-success >0- error code
 */
typedef struct{
uint32_t             error;
} unpack_wds_GetLastMobileIPError_t;

typedef struct{

}pack_wds_GetLastMobileIPError_t;

/**
 * get current data system pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: No
 */
int pack_wds_GetLastMobileIPError(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_GetLastMobileIPError_t *pReqParam
    );

/**
 * get current data system unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetLastMobileIPError(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_GetLastMobileIPError_t *pOutput
    );

/**
 * @param bResetStatistics Clear RM statistics
 * @param ulMask Requested statistic bit mask
 */
typedef struct
{
uint8_t   bResetStatistics;
uint32_t  ulMask;
} rmTrasnferStaticsReq;

/**
 * @param rmTrasnferStaticsReq RM Transfer Statistics Indicator
 */
typedef struct{
rmTrasnferStaticsReq   RmTrasnferStaticsReq;
} pack_wds_RMSetTransferStatistics_t;

/**
 * rm set transfer statistics pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[out] reqParam request parameter
 *
 * @note PDN Specific: No
 */
int pack_wds_RMSetTransferStatistics(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_RMSetTransferStatistics_t *reqParam
    );

typedef struct
{

}unpack_wds_RMSetTransferStatistics_t;

/**
 * rm set transfer statistics unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_RMSetTransferStatistics(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_RMSetTransferStatistics_t *pOutput
    );

/**
 * @param spc service programming code string
 * @param index Index of the profile to modify
 * @param pEnabled Enable profile 0-disable nonzero enable
 * @param pAddress Home IPv4 address
 * @param pPrimaryHA Primary home agent IPv4 address
 * @param pSecondaryHA secondary home agent IPv4 address
 * @param pRevTunneling Enable reverse tunneling 0-disable nonzero enable
 * @param pNAI Network access identifier string
 * @param pHASPI Home agent security parameter index
 * @param pAAASPI AAA server security parameter index
 * @param pMNHA MN-HA key string
 * @param pMNAAA MN-AAA key string
 */
typedef struct{
int8_t   spc[10];
uint8_t  index;
uint8_t  *pEnabled;
uint32_t *pAddress;
uint32_t *pPrimaryHA;
uint32_t *pSecondaryHA;
uint8_t  *pRevTunneling;
int8_t   *pNAI;
uint32_t *pHASPI;
uint32_t *pAAASPI;
int8_t   *pMNHA;
int8_t   *pMNAAA;
} pack_wds_SetMobileIPProfile_t;

/**
 * set mobile ip profile pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[out] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SetMobileIPProfile(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SetMobileIPProfile_t *reqParam
    );

typedef struct
{
}unpack_wds_SetMobileIPProfile_t;
/**
 * set mobile ip profile unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SetMobileIPProfile(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_SetMobileIPProfile_t *pOutput
    );

/**
 * @param contextId Context Identifier
 * @param contextType Context Type 0-3GPP 1-3GPP2
 */
typedef struct
{
uint8_t  contextId;
uint8_t  contextType;
} pack_wds_SLQSWdsSwiPDPRuntimeSettings_t;

/**
 * @param IPV6PrefixLen Length of the received IPv6 address
 * @param IPAddressV6 IPv6 address(in network byte order)
 */
struct ipv6AddressInfo
{
uint8_t   IPV6PrefixLen;
uint16_t  IPAddressV6[IPV6_ADDRESS_ARRAY_SIZE];
};

/**
 * @param contextId Context Identifier
 * @param bearerId Bearer Identity
 * @param apnName APN name associated with the context id
 * @param ipv4Address IPv4 Address
 * @param ipv4GWAddress IPv4 Gateway Address
 * @param prDNSIPv4Address Primary DNS IPv4 Address
 * @param seDNSIPv4Address Secondary DNS IPv4 Address
 * @param ipv6Address IPv6 Address
 * @param ipv6GWAddress IPv6 Gateway Address
 * @param prDNSIPv6Address Primary IPv6 DNS Address
 * @param seDNSIPv6Address Secondary IPv6 DNS Address
 * @param prPCSCFIPv4Address Primary PCSCF IPv4 Address
 * @param sePCSCFIPv4Address Secondary PCSCF IPv4 Address
 * @param prPCSCFIPv6Address Primary PCSCF IPv6 Address
 * @param sePCSCFIPv6Address Secondary PCSCF IPv6 Address
 */
typedef struct
{
uint8_t                   contextId;
uint8_t                   bearerId;
int8_t                    apnName[100];
uint32_t                  ipv4Address;
uint32_t                  ipv4GWAddress;
uint32_t                  prDNSIPv4Address;
uint32_t                  seDNSIPv4Address;
struct ipv6AddressInfo    ipv6Address;
struct ipv6AddressInfo    ipv6GWAddress;
uint16_t                  prDNSIPv6Address[IPV6_ADDRESS_ARRAY_SIZE];
uint16_t                  seDNSIPv6Address[IPV6_ADDRESS_ARRAY_SIZE];
uint32_t                  prPCSCFIPv4Address;
uint32_t                  sePCSCFIPv4Address;
uint16_t                  prPCSCFIPv6Address[IPV6_ADDRESS_ARRAY_SIZE];
uint16_t                  sePCSCFIPv6Address[IPV6_ADDRESS_ARRAY_SIZE];
} unpack_wds_SLQSWdsSwiPDPRuntimeSettings_t;

/**
 * swi pdp runtime settings pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[out] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSWdsSwiPDPRuntimeSettings(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SLQSWdsSwiPDPRuntimeSettings_t *reqParam
    );

/**
 * get current data system unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSWdsSwiPDPRuntimeSettings(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_SLQSWdsSwiPDPRuntimeSettings_t *pOutput
    );

/**
 * @param StatsPeriod Field Period between transfer statistic reports.
 * @param StatsMask requested statistic bit mask.
 */
typedef struct
{
uint8_t  StatsPeriod;
uint32_t StatsMask;
} transferStatInd;

/**
 * @param Mask  mast bits corresponding to the information requested to 1
 * @param pReportConnStatus Connect Status Indicator
 * @param pTransferStatInd Transfer Statistics Indicator
 * @param pReportDormStatus Dormancy Status Indicator
 * @param pReportDataBearerTech Current Data Bearer Technology Indicator
 * @param pReportChannelRate Channel Rate Indicator
 */
typedef struct
{
uint32_t           Mask;
uint8_t            *pReportConnStatus;
transferStatInd    *pTransferStatInd;
uint8_t            *pReportDormStatus;
uint8_t            *pReportDataBearerTech;
uint8_t            *pReportChannelRate;
} pack_wds_SLQSGetDUNCallInfo_t;

/**
 * @param MDMConnStatus Current link status 1-Disconected 2-Connected
 * @param MDMCallDuration Call duration in milliseconds
 */
typedef struct
{
uint8_t      MDMConnStatus;
uint64_t     MDMCallDuration;
} connectionStatus;

/**
* @param CurrChanTxRate instantaneous channel Tx rate in bits per second
* @param CurrChanRxRate instantaneous channel Rx rate in bits per second
* @param MaxChanTxRate maximum Tx rate that can be assigned to the device
* @param MaxChanRxRate maximum Rx rate that can be assigned to the device
*/
typedef struct
{
uint32_t CurrChanTxRate;
uint32_t CurrChanRxRate;
uint32_t MaxChanTxRate;
uint32_t MaxChanRxRate;
} dunchannelRate;

/**
* @param connectionStatus Connection Status
* @param callEndReason Last Modem Call End Reason
* @param txOKBytesCount Tx Bytes OK
* @param rxOKBytesCount Rx Bytes OK
* @param dormancyStatus Dormancy Status
* @param dataBearerTech data bearer technology
* @param channelRate data Channel Rate
* @param lastCallTXOKBytesCnt Last Call Tx Bytes OK
* @param lastCallRXOKBytesCnt Last Call Rx Bytes OK
* @param mdmCallDurationActive Call active duration
* @param lastCallDataBearerTech Last Call Data Bearer Technology
*/
typedef struct
{
connectionStatus    connectionStatus;
uint16_t            callEndReason;
uint64_t            txOKBytesCount;
uint64_t            rxOKBytesCount;
uint8_t             dormancyStatus;
uint8_t             dataBearerTech;
dunchannelRate         channelRate;
uint64_t            lastCallTXOKBytesCnt;
uint64_t            lastCallRXOKBytesCnt;
uint64_t            mdmCallDurationActive;
uint8_t             lastCallDataBearerTech;
} unpack_wds_SLQSGetDUNCallInfo_t;

/**
 * get dun call info pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[out] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSGetDUNCallInfo(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SLQSGetDUNCallInfo_t *reqParam
    );

/**
 * get dun call info unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
*/
int unpack_wds_SLQSGetDUNCallInfo(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_SLQSGetDUNCallInfo_t *pOutput
    );

/**
* @param currentNetwork current selected network
* @param Radio Access Technology (RAT) mask
* @param soMask Service Option (SO) mask
*/
typedef struct
{
uint8_t  currentNetwork;
uint32_t ratMask;
uint32_t soMask;
} qmiWDSDataBearerTechnology;

/**
* @param dataBearerMask  bit mask indicates bearer info is for current and/or last call
* @param curDataBearerTechnology current data bearer technology value
* @param lastCallDataBearerTechnology last call data bearer technology value
*/
typedef struct
{
uint8_t dataBearerMask;
qmiWDSDataBearerTechnology curDataBearerTechnology;
qmiWDSDataBearerTechnology lastCallDataBearerTechnology;
} unpack_wds_SLQSGetDataBearerTechnology_t;

typedef struct{

}pack_wds_SLQSGetDataBearerTechnology_t;
/**
 * get data bearer technology pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[out] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 * @note PDN Specific: Yes
 */
int pack_wds_SLQSGetDataBearerTechnology(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SLQSGetDataBearerTechnology_t *pReqParam
    );

/**
 * get data bearer technology unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSGetDataBearerTechnology(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_SLQSGetDataBearerTechnology_t *pOutput
    );

/**
* @param IPFamilyPreference IP Family preference
*                               - PACK_WDS_IPV4 IP Version 4
*                               - PACK_WDS_IPV6 IP Version 6
*/

typedef struct{
    uint8_t IPFamilyPreference;
}pack_wds_SLQSSetIPFamilyPreference_t;

/**
* @param Tlvresult unpack result
*/

typedef struct{
    uint16_t Tlvresult;
}unpack_wds_SLQSSetIPFamilyPreference_t;

/**
 * Set IP Family Preference pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[out] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_wds_SLQSSetIPFamilyPreference(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SLQSSetIPFamilyPreference_t *pReqParam
);

/**
 * Set IP Family Preference unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SLQSSetIPFamilyPreference(
    uint8_t   *pResp,
    uint16_t  respLen,
    unpack_wds_SLQSSetIPFamilyPreference_t *pOutput
);

typedef struct{
    uint8_t type;
    uint8_t family;
    uint8_t index;
}pack_wds_SetDefaultProfileNum_t;

/**
 * set default profile number pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_wds_SetDefaultProfileNum(
    pack_qmi_t  *pCtx,
    uint8_t     *pReqBuf,
    uint16_t    *pLen,
    pack_wds_SetDefaultProfileNum_t *pReqParam
    );

/**
 * set default profile number unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_SetDefaultProfileNum(
    uint8_t   *pResp,
    uint16_t  respLen
    );

/**
* @param type profile type
* - 0 - 3GPP
* - 1 - 3GPP2
*
* @param type profile family
* - 0 - Embedded
* - 1 - Tethered
*/
typedef struct{
    uint8_t type;
    uint8_t family;
}pack_wds_GetDefaultProfileNum_t;

/**
* @param index profile index
*/
typedef struct{
    uint8_t index;
}unpack_wds_GetDefaultProfileNum_t;

/**
 * get default profile number pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqParam request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_wds_GetDefaultProfileNum(
        pack_qmi_t  *pCtx,
        uint8_t     *pReqBuf,
        uint16_t    *pLen,
        pack_wds_GetDefaultProfileNum_t *pReqParam
        );

/**
 * get default profile number unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_wds_GetDefaultProfileNum(
        uint8_t   *pResp,
        uint16_t  respLen,
        unpack_wds_GetDefaultProfileNum_t *pOutput
        );
