/**
 * \ingroup pack
 *
 * \file uim.h
 */
#include "common.h"
#include <stdint.h>

#define UIM_UINT8_MAX_STRING_SZ 255
#define UIM_MAX_DESCRIPTION_LENGTH 255
#define UIM_MAX_NO_OF_SLOTS 5
#define UIM_MAX_NO_OF_APPLICATIONS 10
#define MAX_NO_OF_SLOTS        5
#define MAX_NO_OF_APPLICATIONS 10
#define MAX_DESCRIPTION_LENGTH 255
#define MAX_SLOTS_STATUS       255
#define MAX_ICCID_LENGTH       255


/**
 *  This structure contains Application Status Information loaded on the card.
 *
 *  \param  appType
 *          - Indicates the type of the application.
 *              - 0 - Unknown
 *              - 1 - SIM card
 *              - 2 - USIM application
 *              - 3 - RUIM card
 *              - 4 - CSIM application
 *              - 5 - ISIM application
 *          - Other values are reserved for the future and are
 *            to be handled as "Unknown".
 *
 *  \param  appState
 *          - Indicates the state of the application.
 *              - 0 - Unknown
 *              - 1 - Detected
 *              - 2 - PIN1 or UPIN is required
 *              - 3 - PUK1 or PUK for UPIN is required
 *              - 4 - Personalization state must be checked
 *              - 5 - PIN1 is blocked
 *              - 6 - Illegal
 *              - 7 - Ready
 *
 *  \param  persoState
 *          - Indicates the state of the personalization for the application.
 *              - 0 - Unknown
 *              - 1 - Personalization operation is in progress
 *              - 2 - Ready
 *              - 3 - Personalization code is required
 *              - 4 - PUK for personalization code is required
 *              - 5 - Permanently blocked
 *
 *  \param  persoFeature
 *          - Indicates the personalization feature.
 *          - This applies only when a personalization code is required to
 *            deactivate or unblock personalization.
 *              - 0 - GW network personalization
 *              - 1 - GW network subset personalization
 *              - 2 - GW service provider personalization
 *              - 3 - GW corporate personalization
 *              - 4 - GW UIM personalization
 *              - 5 - 1X network type 1 personalization
 *              - 6 - 1X network type 2 personalization
 *              - 7 - 1X HRPD personalization
 *              - 8 - 1X service provider personalization
 *              - 9 - 1X corporate personalization
 *              - 10 - 1X RUIM personalization
 *              - 11 - Unknown
 *
 *  \param  persoRetries
 *          - Indicates the number of retries remaining to disable the
 *            personalization.
 *
 *  \param  persoUnblockRetries
 *          - Indicates the number of retries remaining to unblock the
 *            personalization.
 *
 *  \param  aidLength
 *          - Number of sets of the following elements. i.e. aidVal
 *          - If zero(0) then no aidVal information exists.
 *
 *  \param  aidVal[MAX_DESCRIPTION_LENGTH]
 *          - Application identifier value.
 *
 *  \param  univPin
 *          - Indicates whether UPIN replaces PIN1.
 *              - 0 - PIN1 is used
 *              - 1 - UPIN replaces PIN1
 *
 *  \param  pin1State
 *          - Indicates the state of PIN1.
 *              - 0 - Unknown
 *              - 1 - Enabled and not verified
 *              - 2 - Enabled and verified
 *              - 3 - Disabled
 *              - 4 - Blocked
 *              - 5 - Permanently blocked
 *
 *  \param  pin1Retries
 *          - Indicates the number of retries remaining to verify PIN1.
 *
 *  \param  puk1Retries
 *          - Indicates the number of retries remaining to unblock PIN1.
 *
 *  \param  pin2State
 *          - Indicates the state of PIN2.
 *              - 0 - Unknown
 *              - 1 - Enabled and not verified
 *              - 2 - Enabled and verified
 *              - 3 - Disabled
 *              - 4 - Blocked
 *              - 5 - Permanently blocked
 *
 *  \param  pin2Retries
 *          - Indicates the number of retries remaining to verify PIN2.
 *
 *  \param  puk2Retries
 *          - Indicates the number of retries remaining to unblock PIN2.
 *
 */
typedef struct
{
    uint8_t appType;
    uint8_t appState;
    uint8_t persoState;
    uint8_t persoFeature;
    uint8_t persoRetries;
    uint8_t persoUnblockRetries;
    uint8_t aidLength;
    uint8_t aidVal[UIM_MAX_DESCRIPTION_LENGTH];
    uint8_t univPin;
    uint8_t pin1State;
    uint8_t pin1Retries;
    uint8_t puk1Retries;
    uint8_t pin2State;
    uint8_t pin2Retries;
    uint8_t puk2Retries;
} uim_appStatus;


/**
 *  This structure contains information about the SLOTS present.
 *
 *  \param  cardState
 *          - Indicates the state of the card for each slot.
 *              - 0 - Absent
 *              - 1 - Present
 *              - 2 - Error
 *
 *  \param  upinState
 *          - Indicates the state of UPIN.
 *              - 0 - Unknown
 *              - 1 - Enabled and not verified
 *              - 2 - Enabled and verified
 *              - 3 - Disabled
 *              - 4 - Blocked
 *              - 5 - Permanently blocked
 *              - 0xFF - Not Available
 *
 *  \param  upinRetries
 *          - Indicates the number of retries remaining to verify the UPIN.
 *          - If 0xFF, information not available.
 *
 *  \param  upukRetries
 *          - Indicates the number of retries remaining to unblock the UPIN.
 *          - If 0xFF, information not available.
 *
 *  \param  errorState
 *          - Indicates the reason for the card error, and is valid only when
 *            the card state is Error
 *              - 0 - Unknown
 *              - 1 - Power down
 *              - 2 - Poll error
 *              - 3 - No ATR received
 *              - 4 - Volt mismatch
 *              - 5 - Parity error
 *              - 6 - Unknown; possibly removed
 *              - 7 - Card returned technical problems
 *              - 0xFF - Not Available
 *          - Other values are possible and reserved for future use.
 *          - When an unknown value is received, it is to be handled as "Unknown".
 *
 *  \param  numApp
 *          - Indicates the number of applications available on the card.
 *          - The following block is repeated for each application. i.e. AppStatus.
 *          - If zero(0) then no AppStatus information exists.
 *
 *  \param  AppStatus
 *          - See \ref uim_appStatus for more information.
 *
 */
typedef struct
{
    uint8_t      cardState;
    uint8_t      upinState;
    uint8_t      upinRetries;
    uint8_t      upukRetries;
    uint8_t      errorState;
    uint8_t      numApp;
    uim_appStatus AppStatus[UIM_MAX_NO_OF_APPLICATIONS];
} uim_slotInfo;


/**
 *  This structure contains Card Status Information.
 *
 *  \param  indexGwPri
 *          - Index of the primary GW provisioning application.
 *          - The most significant byte indicates the slot (starting from 0),
 *            while the least significant byte indicates the application for
 *            that slot (starting from 0).
 *          - The value 0xFFFF identifies when the session does not exist.
 *
 *  \param  index1xPri
 *          - Index of the primary 1X provisioning application.
 *          - The most significant byte indicates the slot (starting from 0),
 *            while the least significant byte indicates the application for
 *            that slot (starting from 0).
 *          - The value 0xFFFF identifies when the session does not exist.
 *
 *  \param  indexGwSec
 *          - Index of the secondary GW provisioning application.
 *          - The most significant byte indicates the slot (starting from 0),
 *            while the least significant byte indicates the application for
 *            that slot (starting from 0).
 *          - The value 0xFFFF identifies when the session does not exist.
 *
 *  \param  index1xSec
 *          - Index of the secondary GW provisioning application.
 *          - The most significant byte indicates the slot (starting from 0),
 *            while the least significant byte indicates the application for
 *            that slot (starting from 0).
 *          - The value 0xFFFF identifies when the session does not exist.
 *
 *  \param  numSlot
 *          - Indicates the number of slots available on the device.
 *          - The following block is repeated for each slot. i.e. cardState
 *          - If zero(0) then no cardState information exists.
 *
 *  \param  SlotInfo
 *          - See \ref uim_slotInfo for more information.
 *
 */
typedef struct
{
    uint16_t     indexGwPri;
    uint16_t     index1xPri;
    uint16_t     indexGwSec;
    uint16_t     index1xSec;
    uint8_t     numSlot;
    uim_slotInfo SlotInfo[UIM_MAX_NO_OF_SLOTS];
} uim_cardStatus;

/**
 *  This structure contains Hot Swap Status Information.
 *
 *  \param  hotSwapLength
 *          - Number of sets of the following elements. i.e. hot_swap
 *
 *  \param  hotSwap
 *          - Indicates the status of the hot-swap switch.
 *              - 0 - Hot-swap is not supported
 *              - 1 - Hot-swap is supported, but the status of the switch
 *                    is not supported
 *              - 2 - Switch indicates that the card is present
 *              - 3 - Switch indicates that the card is not present
 *
 */
typedef struct
{
    uint8_t hotSwapLength;
    uint8_t hotSwap[UIM_MAX_DESCRIPTION_LENGTH];
} uim_hotSwapStatus;

/**
 *  This structure contains information of the response parameters associated
 *  with a Get Card Status API.
 *
 *  @param  pCardStatus(optional)
 *          - See \ref uim_cardStatus for more information.
 *
 *  @param  pHotSwapStatus(optional)
 *          - See \ref uim_hotSwapStatus for more information.
 *
 *  @note   Using NULL for the pointers would make sure that the parameter is
 *          not returned.
 */
typedef struct{
    uim_cardStatus *pCardStatus;
    uim_hotSwapStatus *pHotSwapStatus;
    uint16_t Tlvresult;
}unpack_uim_GetCardStatus_t;

/**
 *  This structure contains the encrypted PIN1 Information.
 *
 *  @param  pin1Len
 *          - Number of sets of the following elements ie encrypted PIN1 value.
 *          - If zero(0), no information follows.
 *
 *  @param  pin1Val
 *          - Encrypted PIN1 value.
 *
 *  @note   This value is returned only when PIN1 is enabled successfully
 *          and the feature is supported.
 *
 */
typedef struct
{
    uint8_t pin1Len;
    uint8_t pin1Val[UIM_MAX_DESCRIPTION_LENGTH];
} uim_encryptedPIN1;

/**
 *  This structure contains the information about the retries remaining.
 *
 *  @param  verifyLeft
 *          - Number of remaining attempts to verify the PIN.
 *          - 0xFF, if unavailable.
 *
 *  @param  unblockLeft
 *          - Number of remaining attempts to unblock the PIN.
 *          - 0xFF, if unavailable.
 *
 *  @note   This value is returned only when the enable/disable operation has
 *          failed.
 *          This information is not sent for a hidden key PIN type.
 *
 */
typedef struct
{
    uint8_t verifyLeft;
    uint8_t unblockLeft;
} uim_remainingRetries;

/**
 *  This structure contains the Session Information.
 *
 *  @param  sessionType
 *          - Indicates the session type.
 *              - 0 - Primary GW provisioning
 *              - 1 - Primary 1X provisioning
 *              - 2 - Secondary GW provisioning
 *              - 3 - Secondary 1X provisioning
 *              - 4 - Non-provisioning on slot 1
 *              - 5 - Non-provisioning on slot 2
 *              - 6 - Card on slot 1
 *              - 7 - Card on slot 2
 *              - 8 - Logical channel on slot 1
 *              - 9 - Logical channel on slot 2
 *
 *  @param  aidLength
 *          - Length of the following elements i.e. Application Identifier.
 *
 *  @param  aid
 *          - Application identifier value or channel ID.
 *          - This value is required for non-provisioning and for logical
 *            channel session types. It is ignored in all other cases.
 *
 */
typedef struct
{
    uint8_t   sessionType;
    uint8_t   aidLength;
    uint8_t   aid[UIM_MAX_DESCRIPTION_LENGTH];
} uim_sessionInformation ;

/**
 *  This structure contains the information about the pin parameters that need
 *  to be verified.
 *
 *  \param  pinID
 *          - Indicates the PIN ID to be verified.
 *              - 1 - PIN1 (also called PIN)
 *              - 2 - PIN2
 *              - 3 - Universal PIN
 *              - 4 - Hidden key
 *
 *  \param  pinLen
 *          - Length of the following elements i.e. pin value.
 *
 *  \param  pinVal[MAX_DESCRIPTION_LENGTH]
 *          - PIN value.
 *          - This value is a sequence of ASCII characters.
 *
 */
typedef struct{
    uint8_t pinID;
    uint8_t pinLen;
    uint8_t pinVal[UIM_MAX_DESCRIPTION_LENGTH];
} uim_verifyUIMPIN;

/**
 *  This structure contains the information about the unblock pin parameters.
 *
 *  @param  pinID
 *          - Indicates the PIN ID to be changed.
 *              - 1 - PIN1 (also called PIN)
 *              - 2 - PIN2
 *              - 3 - Universal PIN
 *
 *  @param  pukLen
 *          - Length of the following elements i.e. puk value.
 *
 *  @param  pukVal[UIM_MAX_DESCRIPTION_LENGTH]
 *          - PIN Unlock Key value.
 *          - This value is a sequence of ASCII characters.
 *
 *  @param  newPINLen
 *          - Length of the following elements i.e. new pin value.
 *
 *  @param  newPINVal[UIM_MAX_DESCRIPTION_LENGTH]
 *          - New PIN value.
 *          - This value is a sequence of ASCII characters.
 *
 */
typedef struct{
    uint8_t pinID;
    uint8_t pukLen;
    uint8_t pukVal[UIM_MAX_DESCRIPTION_LENGTH];
    uint8_t newPINLen;
    uint8_t newPINVal[UIM_MAX_DESCRIPTION_LENGTH];
} uim_unblockUIMPIN;

/**
 *  This structure contains the information about the card result.
 *
 *  @param  sw1
 *          - SW1 received from the card.
 *
 *  @param  sw2
 *          - SW2 received from the card.
 *
 */
typedef struct
{
    uint8_t sw1;
    uint8_t sw2;
} uim_cardResult;

/**
 *  This structure contains the information about the pin protection parameters
 *  that need to be set.
 *
 *  @param  pinID
 *          - Indicates the PIN ID to be enabled or disabled.
 *              - 1 - PIN1 (also called PIN)
 *              - 2 - PIN2
 *              - 3 - Universal PIN
 *              - 4 - Hidden key
 *
 *  @param  pinOperation
 *          - Indicates whether the PIN is enabled or disabled.
 *              - 0 - Disable the PIN
 *              - 1 - Enable the PIN
 *
 *  @param  pinLength
 *          - Length of the following elements i.e. pin value.
 *
 *  @param  pinValue[MAX_DESCRIPTION_LENGTH]
 *          - PIN value.
 *          - This value is a sequence of ASCII characters.
 *
 */
typedef struct
{
    uint8_t pinID;
    uint8_t pinOperation;
    uint8_t pinLength;
    uint8_t pinValue[UIM_MAX_DESCRIPTION_LENGTH];
} uim_setPINProtection;

/**
 *  This structure contains the information about the pin parameters that need
 *  to be verified.
 *
 *  \param  pinID
 *          - Indicates the PIN ID to be changed.
 *              - 1 - PIN1 (also called PIN)
 *              - 2 - PIN2
 *              - 3 - Universal PIN
 *              - 4 - Hidden key
 *
 *  \param  oldPINLen
 *          - Length of the following elements i.e. old pin value.
 *
 *  \param  oldPINVal[MAX_DESCRIPTION_LENGTH]
 *          - Old PIN value.
 *          - This value is a sequence of ASCII characters.
 *
 *  \param  pinLen
 *          - Length of the following elements i.e. new pin value.
 *
 *  \param  pinVal[MAX_DESCRIPTION_LENGTH]
 *          - New PIN value.
 *          - This value is a sequence of ASCII characters.
 *
 */
typedef struct{
    uint8_t pinID;
    uint8_t oldPINLen;
    uint8_t oldPINVal[UIM_MAX_DESCRIPTION_LENGTH];
    uint8_t pinLen;
    uint8_t pinVal[UIM_MAX_DESCRIPTION_LENGTH];
} uim_changeUIMPIN;

/**
 *  This structure contains paramaters for file Information
 *
 *  @param  fileID
 *          - This is Identifier to SIM files;
 *            e.g. in UIM "6F07" is Identifier of IMSI File
 *
 *  @param  pathLen
 *          - Length of file Path
 *
 *  @param  path
 *          - Path value. This value must be the complete
 *            path of the file, which is a sequence block of 2
 *            bytes (e.g., 0x3F00 0x7FFF).
 */
typedef struct {
    uint16_t  fileID;
    uint8_t  pathLen;
    uint16_t  path[UIM_UINT8_MAX_STRING_SZ];
} uim_fileInfo;


/**
 *  This structure contains the Session Information.
 *
 *  @param  sessionType
 *          - Indicates the session type.
 *              - 0 - Primary GW provisioning
 *              - 1 - Primary 1X provisioning
 *              - 2 - Secondary GW provisioning
 *              - 3 - Secondary 1X provisioning
 *              - 4 - Non-provisioning on slot 1
 *              - 5 - Non-provisioning on slot 2
 *              - 6 - Card on slot 1
 *              - 7 - Card on slot 2
 *              - 8 - Logical channel on slot 1
 *              - 9 - Logical channel on slot 2
 *
 *  @param  aidLength
 *          - Length of the following elements i.e. Application Identifier.
 *
 *  @param  aid
 *          - Application identifier value or channel ID.
 *          - This value is required for non-provisioning and for logical
 *            channel session types. It is ignored in all other cases.
 *
 */
typedef struct
{
    uint8_t   sessionType;
    uint8_t   aidLength;
    uint8_t   aid[255];
} uim_UIMSessionInformation ;

/**
 *  This structure contains the information for read operation.
 *
 *  @param  offset
 *          - Offset for the read operation.
 *
 *  @param  length
 *          - Length of the content to be read.
 *          - The value 0 is used to read the complete file.
 *
 */
typedef struct
{
    uint16_t offset;
    uint16_t length;
} uim_readTransparentInfo;


/**
 *  This structure contains the information for write operation.
 *
 *  @param  contentLen
 *          - Number of sets of content.
 *
 *  @param  content[255]
 *          - Read content.
 *          - The content is the sequence of bytes as read from the card.
 */
typedef struct
{
    uint16_t contentLen;
    uint8_t content[UIM_UINT8_MAX_STRING_SZ];
} uim_readResult;


/**
 * Get Card Status pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_GetCardStatus(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Card Status unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_GetCardStatus(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_GetCardStatus_t *pOutput
);

/**
 *  This structure contains information of the request parameters associated
 *  with a verify PIN API.
 *
 *  @param  sessionInfo
 *          - See \ref UIMSessionInformation for more information.
 *
 *  @param  verifyPIN
 *          - See \ref verifyUIMPIN for more information.
 *
 *  @param  pEncryptedPIN1(optional)
 *          - See \ref encryptedPIN1 for more information.
 *
 *  @param  pKeyReferenceID(optional)
 *          - Indicates the PIN key reference ID.
 *          - Indicates the PIN key reference ID. Valid values are from 1 to 8,
 *            respectively, for application 1 to application 8.
 *          - This TLV is used only for PIN1 and PIN2 and is ignored in all
 *            other cases.
 *
 *  @param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result must be
 *            provided in a subsequent indication.
 *          - Valid Values
 *            - 0 - Result of operation in response. Indication will not be generated by the modem
 *            - Any other positive number - Result of operation in indication.
 *              Indication will have same token value set by this function
 *
 *  @note   Using NULL for the pointers would make sure that the parameter is
 *          not added to the request.
 */
typedef struct{
    uim_encryptedPIN1    *pEncryptedPIN1;
    uint32_t *pIndicationToken;
    uint8_t *pKeyReferenceID;
    uim_sessionInformation sessionInfo;
    uim_verifyUIMPIN      verifyPIN;
    uint16_t Tlvresult;
}pack_uim_VerifyPin_t;

/**
 *  This structure contains information of the response parameters associated
 *  with a set of PIN related API's.
 *
 *  \param  pRemainingRetries(optional)
 *          - See \ref uim_remainingRetries for more information.
 *
 *  \param  pEncryptedPIN1(optional)
 *          - See \ref uim_encryptedPIN1 for more information.
 *
 *  \param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result is
 *            provided in a subsequent indication.
 *          - 0xFFFFFFFF, if unavailable
 *
 *  \note   Using NULL for the pointers would make sure that the parameter is
 *          not returned.
 */
typedef struct{
    uim_remainingRetries *pRemainingRetries;
    uim_encryptedPIN1    *pEncryptedPIN1;
    uint32_t *pIndicationToken;
    uint16_t Tlvresult;
}unpack_uim_VerifyPin_t;


/**
 * Verify Pin Status pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_VerifyPin(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_VerifyPin_t *reqArg
);

/**
 * Verify Pin unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_VerifyPin(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_VerifyPin_t *pOutput
);

/**
 *  This structure contains information of the request parameters associated
 *  with a Unblock PIN API.
 *
 *  @param  EncryptedPIN1
 *          - See \ref uim_encryptedPIN1 for more information.
 *
 *  @param  sessionInfo
 *          - See \ref uim_sessionInformation for more information.
 *
 *  @param  pinProtection
 *          - See \ref uim_unblockUIMPIN for more information.
 *
 *  @param  pKeyReferenceID(optional)
 *          - Indicates the PIN key reference ID.
 *          - Indicates the PIN key reference ID. Valid values are from 1 to 8,
 *            respectively, for application 1 to application 8.
 *          - This TLV is used only for PIN1 and PIN2 and is ignored in all
 *            other cases.
 *
 *  @param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result must be
 *            provided in a subsequent indication.
 *          - Valid Values
 *            - 0 - Result of operation in response. Indication will not be generated by the modem
 *            - Any other positive number - Result of operation in indication.
 *              Indication will have same token value set by this function
 *
 */
typedef struct{
    uim_encryptedPIN1    EncryptedPIN1;
    uint32_t *pIndicationToken;
    uint8_t *pKeyReferenceID;
    uim_sessionInformation sessionInfo;
    uim_unblockUIMPIN pinProtection;
    uint16_t Tlvresult;
}pack_uim_UnblockPin_t;

/**
 *  This structure contains information of the response parameters associated
 *  with a set of PIN related API's.
 *
 *  @param  pRemainingRetries(optional)
 *          - See \ref uim_remainingRetries for more information.
 *
 *  @param  pEncryptedPIN1(optional)
 *          - See \ref uim_encryptedPIN1 for more information.
 *
 *  @param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result is
 *            provided in a subsequent indication.
 *          - 0xFFFFFFFF, if unavailable
 *
 */
typedef struct{
    uim_remainingRetries *pRemainingRetries;
    uim_encryptedPIN1    *pEncryptedPIN1;
    uint32_t *pIndicationToken;
    uint16_t Tlvresult;
}unpack_uim_UnblockPin_t;


/**
 * Unblock Pin pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_UnblockPin(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_UnblockPin_t *reqArg
);

/**
 * Unblock Pin unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_UnblockPin(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_UnblockPin_t *pOutput
);

/**
 *  This structure contains information of the request parameters associated
 *  with a set pin protection API.
 *
 *  @param  sessionInfo
 *          - See \ref uim_sessionInformation for more information.
 *
 *  @param  pinProtection
 *          - See \ref uim_setPINProtection for more information.
 *
 *  @param  pKeyReferenceID(optional)
 *          - Indicates the PIN key reference ID.
 *          - Indicates the PIN key reference ID. Valid values are from 1 to 8,
 *            respectively, for application 1 to application 8.
 *          - This TLV is used only for PIN1 and PIN2 and is ignored in all
 *            other cases.
 *
 *  @param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result must be
 *            provided in a subsequent indication.
 *          - Valid Values
 *            - 0 - Result of operation in response. Indication will not be generated by the modem
 *            - Any other positive number - Result of operation in indication.
 *              Indication will have same token value set by this function
 *
 *  @note   Using NULL for the pointers would make sure that the parameter is
 *          not added to the request.
 */
typedef struct{
    uim_encryptedPIN1    EncryptedPIN1;
    uint32_t *pIndicationToken;
    uint8_t *pKeyReferenceID;
    uim_sessionInformation sessionInfo;
    uim_setPINProtection      pinProtection;
    uint16_t Tlvresult;
}pack_uim_SetPinProtection_t;

/**
 *  This structure contains information of the response parameters associated
 *  with a set of PIN related API's.
 *
 *  @param  pRemainingRetries(optional)
 *          - See \ref uim_remainingRetries for more information.
 *
 *  @param  pEncryptedPIN1(optional)
 *          - See \ref uim_encryptedPIN1 for more information.
 *
 *  @param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result is
 *            provided in a subsequent indication.
 *          - 0xFFFFFFFF, if unavailable
 *
 *  \note   Using NULL for the pointers would make sure that the parameter is
 *          not returned.
 */
typedef struct{
    uim_remainingRetries *pRemainingRetries;
    uim_encryptedPIN1    *pEncryptedPIN1;
    uint32_t *pIndicationToken;
    uint16_t Tlvresult;
}unpack_uim_SetPinProtection_t;


/**
 * Set Pin Protection pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_SetPinProtection(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_SetPinProtection_t *reqArg
);

/**
 * Set Pin Protection unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_SetPinProtection(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_SetPinProtection_t *pOutput
);

/**
 *  This structure contains information of the request parameters associated
 *  with a Change PIN API.
 *
 *  \param  sessionInfo
 *          - See \ref UIMSessionInformation for more information.
 *
 *  \param  changePIN
 *          - See \ref changeUIMPIN for more information.
 *
 *  \param  pKeyReferenceID(optional)
 *          - Indicates the PIN key reference ID.
 *          - Indicates the PIN key reference ID. Valid values are from 1 to 8,
 *            respectively, for application 1 to application 8.
 *          - This TLV is used only for PIN1 and PIN2 and is ignored in all
 *            other cases.
 *
 *  \param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result must be
 *            provided in a subsequent indication.
 *          - Valid Values
 *            - 0 - Result of operation in response. Indication will not be generated by the modem
 *            - Any other positive number - Result of operation in indication.
 *              Indication will have same token value set by this function
 *
 *  \note   Using NULL for the pointers would make sure that the parameter is
 *          not added to the request.
 */
typedef struct{
    uim_encryptedPIN1    EncryptedPIN1;
    uint32_t *pIndicationToken;
    uint8_t *pKeyReferenceID;
    uim_sessionInformation sessionInfo;
    uim_changeUIMPIN      changePIN;
    uint16_t Tlvresult;
}pack_uim_ChangePin_t;

/**
 *  This structure contains information of the response parameters associated
 *  with a set of PIN related API's.
 *
 *  \param  pRemainingRetries(optional)
 *          - See \ref uim_remainingRetries for more information.
 *
 *  \param  pEncryptedPIN1(optional)
 *          - See \ref uim_encryptedPIN1 for more information.
 *
 *  \param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result is
 *            provided in a subsequent indication.
 *          - 0xFFFFFFFF, if unavailable
 *
 *  \note   Using NULL for the pointers would make sure that the parameter is
 *          not returned.
 */
typedef struct{
    uim_remainingRetries *pRemainingRetries;
    uim_encryptedPIN1    *pEncryptedPIN1;
    uint32_t *pIndicationToken;
    uint16_t Tlvresult;
}unpack_uim_ChangePin_t;


/**
 * Change Pin pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_ChangePin(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_ChangePin_t *reqArg
);

/**
 * Change Pin unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_ChangePin(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_ChangePin_t *pOutput
);


/**
 *  This structure contains information of the request parameters associated
 *  with a Read Transparent API.
 *
 *  @param  sessionInfo
 *          - See \ref UIMSessionInformation for more information.
 *
 *  @param  fileIndex
 *          - See \ref fileInfo for more information.
 *
 *  @param  readTransparent
 *          - See \ref readTransparentInfo for more information.
 *
 *  @param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result must be
 *            provided in a subsequent indication.
 *          - Valid Values
 *            - 0 - Result of operation in response. Indication will not be generated by the modem
 *            - Any other positive number - Result of operation in indication.
 *              Indication will have same token value set by this function
 *
 *  @param  pEncryptData(optional)
 *          - Encrypt Data.
 *          - Indicates whether the data read from the card is to be encrypted.
 *
 *  @note   Using NULL for the pointers would make sure that the parameter is
 *          not added to the request.
 */
typedef struct{
    uim_sessionInformation sessionInfo;
    uim_fileInfo              fileIndex;
    uim_readTransparentInfo   readTransparent;
    uint32_t                 *pIndicationToken;
    uint8_t                  *pEncryptData;
    uint16_t Tlvresult;
}pack_uim_ReadTransparent_t;

/**
 *  This structure contains information of the response parameters associated
 *  with a Read Transparent API.
 *
 *  \param  pCardResult
 *          - See \ref cardResult for more information.
 *
 *  \param  pReadResult
 *          - See \ref readResult for more information.
 *
 *  \param  pIndicationToken(optional)
 *          - Response in Indication.
 *          - When this TLV is present, it indicates that the result must be
 *            provided in a subsequent indication.
 *
 *  \param  pEncryptedData(optional)
 *          - Encrypted Data.
 *          - Indicates whether the data from the card passed in read_result is encrypted.
 *
 *  \note   Using NULL for the pointers would make sure that the parameter is
 *          not added to the request.
 */
typedef struct{
    uim_cardResult  *pCardResult;
    uim_readResult  *pReadResult;
    uint32_t        *pIndicationToken;
    uint8_t         *pEncryptedData;
    uint16_t Tlvresult;
}unpack_uim_ReadTransparent_t;


/**
 * SLQS ReadTransparent pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_ReadTransparent(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_ReadTransparent_t *reqArg
);

/**
 * SLQS ReadTransparent unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_ReadTransparent(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_ReadTransparent_t *pOutput
);

/**
 * @param eventMask - bit 1 - card status
 *                  - bit 4 - physical slot status
 */
typedef struct{
    uint32_t eventMask;
}pack_uim_SLQSUIMEventRegister_t;

/**
 * UIM Status Change callback enable pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_SLQSUIMEventRegister (
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_SLQSUIMEventRegister_t *reqArg
);

/**
 * @param eventMask - bit 1 - card status
 */
typedef struct{
    uint32_t eventMask;
}unpack_uim_SLQSUIMEventRegister_t;

/**
 * UIM Status Change callback enable unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_SLQSUIMEventRegister(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_SLQSUIMEventRegister_t *pOutput
);

/**
 *  This structure contains Application Status Information loaded on the card.
 *
 *  \param  appType
 *          - Indicates the type of the application.
 *              - 0 - Unknown
 *              - 1 - SIM card
 *              - 2 - USIM application
 *              - 3 - RUIM card
 *              - 4 - CSIM application
 *              - 5 - ISIM application
 *          - Other values are reserved for the future and are
 *            to be handled as "Unknown".
 *
 *  \param  appState
 *          - Indicates the state of the application.
 *              - 0 - Unknown
 *              - 1 - Detected
 *              - 2 - PIN1 or UPIN is required
 *              - 3 - PUK1 or PUK for UPIN is required
 *              - 4 - Personalization state must be checked
 *              - 5 - PIN1 is blocked
 *              - 6 - Illegal
 *              - 7 - Ready
 *
 *  \param  persoState
 *          - Indicates the state of the personalization for the application.
 *              - 0 - Unknown
 *              - 1 - Personalization operation is in progress
 *              - 2 - Ready
 *              - 3 - Personalization code is required
 *              - 4 - PUK for personalization code is required
 *              - 5 - Permanently blocked
 *
 *  \param  persoFeature
 *          - Indicates the personalization feature.
 *          - This applies only when a personalization code is required to
 *            deactivate or unblock personalization.
 *              - 0 - GW network personalization
 *              - 1 - GW network subset personalization
 *              - 2 - GW service provider personalization
 *              - 3 - GW corporate personalization
 *              - 4 - GW UIM personalization
 *              - 5 - 1X network type 1 personalization
 *              - 6 - 1X network type 2 personalization
 *              - 7 - 1X HRPD personalization
 *              - 8 - 1X service provider personalization
 *              - 9 - 1X corporate personalization
 *              - 10 - 1X RUIM personalization
 *              - 11 - Unknown
 *
 *  \param  persoRetries
 *          - Indicates the number of retries remaining to disable the
 *            personalization.
 *
 *  \param  persoUnblockRetries
 *          - Indicates the number of retries remaining to unblock the
 *            personalization.
 *
 *  \param  aidLength
 *          - Number of sets of the following elements. i.e. aidVal
 *          - If zero(0) then no aidVal information exists.
 *
 *  \param  aidVal[MAX_DESCRIPTION_LENGTH]
 *          - Application identifier value.
 *
 *  \param  univPin
 *          - Indicates whether UPIN replaces PIN1.
 *              - 0 - PIN1 is used
 *              - 1 - UPIN replaces PIN1
 *
 *  \param  pin1State
 *          - Indicates the state of PIN1.
 *              - 0 - Unknown
 *              - 1 - Enabled and not verified
 *              - 2 - Enabled and verified
 *              - 3 - Disabled
 *              - 4 - Blocked
 *              - 5 - Permanently blocked
 *
 *  \param  pin1Retries
 *          - Indicates the number of retries remaining to verify PIN1.
 *
 *  \param  puk1Retries
 *          - Indicates the number of retries remaining to unblock PIN1.
 *
 *  \param  pin2State
 *          - Indicates the state of PIN2.
 *              - 0 - Unknown
 *              - 1 - Enabled and not verified
 *              - 2 - Enabled and verified
 *              - 3 - Disabled
 *              - 4 - Blocked
 *              - 5 - Permanently blocked
 *
 *  \param  pin2Retries
 *          - Indicates the number of retries remaining to verify PIN2.
 *
 *  \param  puk2Retries
 *          - Indicates the number of retries remaining to unblock PIN2.
 *
 */
typedef struct
{
    uint8_t appType;
    uint8_t appState;
    uint8_t persoState;
    uint8_t persoFeature;
    uint8_t persoRetries;
    uint8_t persoUnblockRetries;
    uint8_t aidLength;
    uint8_t aidVal[MAX_DESCRIPTION_LENGTH];
    uint8_t univPin;
    uint8_t pin1State;
    uint8_t pin1Retries;
    uint8_t puk1Retries;
    uint8_t pin2State;
    uint8_t pin2Retries;
    uint8_t puk2Retries;
} appStats;

/**
 *  This structure contains information about the SLOTS present.
 *
 *  \param  cardState
 *          - Indicates the state of the card for each slot.
 *              - 0 - Absent
 *              - 1 - Present
 *              - 2 - Error
 *
 *  \param  upinState
 *          - Indicates the state of UPIN.
 *              - 0 - Unknown
 *              - 1 - Enabled and not verified
 *              - 2 - Enabled and verified
 *              - 3 - Disabled
 *              - 4 - Blocked
 *              - 5 - Permanently blocked
 *              - 0xFF - Not Available
 *
 *  \param  upinRetries
 *          - Indicates the number of retries remaining to verify the UPIN.
 *          - If 0xFF, information not available.
 *
 *  \param  upukRetries
 *          - Indicates the number of retries remaining to unblock the UPIN.
 *          - If 0xFF, information not available.
 *
 *  \param  errorState
 *          - Indicates the reason for the card error, and is valid only when
 *            the card state is Error
 *              - 0 - Unknown
 *              - 1 - Power down
 *              - 2 - Poll error
 *              - 3 - No ATR received
 *              - 4 - Volt mismatch
 *              - 5 - Parity error
 *              - 6 - Unknown; possibly removed
 *              - 7 - Card returned technical problems
 *              - 0xFF - Not Available
 *          - Other values are possible and reserved for future use.
 *          - When an unknown value is received, it is to be handled as "Unknown".
 *
 *  \param  numApp
 *          - Indicates the number of applications available on the card.
 *          - The following block is repeated for each application. i.e. AppStatus.
 *          - If zero(0) then no AppStatus information exists.
 *
 *  \param  AppStatus[MAX_NO_OF_APPLICATIONS]
 *          - See \ref appStats for more information.
 *
 */
typedef struct
{
    uint8_t      cardState;
    uint8_t      upinState;
    uint8_t      upinRetries;
    uint8_t      upukRetries;
    uint8_t      errorState;
    uint8_t      numApp;
    appStats    AppStatus[MAX_NO_OF_APPLICATIONS];
} slotInf;
/**
 * This structure contains information about Status change callback.
 *
 *  @param  pCardStatus Card Status
 *          - See \ref uim_cardStatus for more information.
 *
 */
typedef struct{
    uim_cardStatus *pCardStatus;
}unpack_uim_SLQSUIMSetStatusChangeCallBack_ind_t;

/**
 * UIM Status Change indication unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * @note use pack_uim_SLQSUIMEventRegister to subscribe 
 */
int unpack_uim_SLQSUIMSetStatusChangeCallBack_ind(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_SLQSUIMSetStatusChangeCallBack_ind_t *pOutput
);

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_SLQSUIMGetSlotsStatus (
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen
);

/**
 *  This structure contains information of the response parameters associated
 *  with a Get Slots Status API.
 *
 *  \param  uPhyCardStatus
 *          - State of the card in the Pyhsical Slot Status.
 *              - 0x00 - Unknown.
 *              - 0x01 - Absent.
 *              - 0x02 - Present.
 *
 *  \param  uPhySlotStatus
 *          - State of the Physical Slot status.
 *              - 0x00 Inactive.
 *              - 0x01 Activate.
 *
 *  \param  bLogicalSlot
 *          - Logical Slot associated with this physical slot. THis is valid if the physical slot is active.
 *              - 1 - Slot 1.
 *              - 2 - Slot 2.
 *              - 3 - Slot 3.
 *              - 4 - Slot 4.
 *              - 5 - Slot 5.
 *
 *  \param  bLogicalSlot
 *          - Number of sets the sets of ICCCID
 *
 *  \param  bICCID[MAX_ICCID_LENGTH]
 *          - Contains the ICCID of the card in the physical slot.
 *
 */
typedef struct
{
    uint32_t   uPhyCardStatus;
    uint32_t   uPhySlotStatus;
    uint8_t    bLogicalSlot;
    uint8_t    bICCIDLength;
    uint8_t    bICCID[MAX_ICCID_LENGTH];
}slot_t;

typedef struct
{
    slot_t uimSlotStatus[MAX_SLOTS_STATUS];
}slots_t;

/**
 *  This structure contains information of the response parameters associated
 *  with a Get Slots Status API.
 *
 *  \param  pNumberOfPhySlot
 *          - Number of sets of the Slot Status.
 *
 *  \param  pUimSlotsStatus
 *          - Slots Status  See \ref slots_t for more information..
 */
typedef struct{
    uint8_t    *pNumberOfPhySlot;
    slots_t *pUimSlotsStatus;
}unpack_uim_SLQSUIMGetSlotsStatus_t;

/**
 * get slot status unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_SLQSUIMGetSlotsStatus(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_SLQSUIMGetSlotsStatus_t *pOutput
);

/**
 *  This structure contains information of the request parameters associated
 *  with a Switch Slot.
 *
 *  \param  bLogicalSlot
 *          - Indicates the slot to be used.
 *              - 1 - Slot 1
 *              - 2 - Slot 2
 *              - 3 - Slot 3
 *              - 4 - Slot 4
 *              - 5 - Slot 5
 *
 *  \param  ulPhysicalSlot
 *              - 1 - Slot 1
 *              - 2 - Slot 2
 *              - 3 - Slot 3
 *              - 4 - Slot 4
 *              - 5 - Slot 5
 *
 */
typedef struct{
    uint8_t bLogicalSlot;
    uint32_t ulPhysicalSlot;
}pack_uim_SLQSUIMSwitchSlot_t;

/**
 * switch slot pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_uim_SLQSUIMSwitchSlot (
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_uim_SLQSUIMSwitchSlot_t *reqArg
);

/**
 * switch slot unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_uim_SLQSUIMSwitchSlot(
    uint8_t *pResp,
    uint16_t respLen
);

/**
 *  Structure consist of card status params
 *
 *  \param  slotsstatusChange
 *          - See \ref slot_t for more information
 *
 *  \param  bNumberOfPhySlots
 *          - Number of Physical Slot(s)
 */
typedef struct{
    slots_t slotsstatusChange;
    uint8_t bNumberOfPhySlots;
}unpack_uim_SetUimSlotStatusChangeCallback_ind_t;

/**
 * UIM Slot Status Change indication unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * @note use pack_uim_SLQSUIMEventRegister to subscribe 
 */
int unpack_uim_SetUimSlotStatusChangeCallback_ind(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_uim_SetUimSlotStatusChangeCallback_ind_t *pOutput
);
