/**
 * \ingroup pack
 *
 * \file sms.h
 */
#include "common.h"
#include <stdint.h>

#define MAX_SMS_MESSAGE_SIZE 255
#define MAX_SMS_LIST_SIZE 255
#define MAX_MS_TRANSFER_ROUTE_MSG 256
#define MAX_MSE_TWS_MSG 1254
#define MAX_MSC_ADDRESS_SIZE 256
#define MAX_CDMA_ENC_MO_TXT_MSG_SIZE 255



/**
 * @param storageType
 *          - SMS message storage type
 *              - 0 - UIM - Invalid in case of CDMA device that does not
 *                          require SIM
 *              - 1 - NV
 *
 * @param messageIndex
 *          - Message index
 *
 * @param pMessageMode
 *          - 0x00 - CDMA, LTE (if network type is CDMA)
 *          - 0x01 - GW, LTE (if network type is UMTS)
 *
 */
typedef struct{
    uint32_t storageType;
    uint32_t messageIndex;
    uint8_t *pMessageMode;
} pack_sms_SLQSGetSMS_t;

/**
 * @param messageTag
 *          - Message tag
 *              - 0 - Read
 *              - 1 - Not read
 *              - 2 - Mobile originated and sent
 *              - 3 - Mobile originated but not yet sent
 *
 * @param messageFormat
 *          - Message format
 *              - 0 - CDMA (IS-637B)
 *              - 1 - 5 (Reserved)
 *              - 6 - GSM/WCDMA PP
 *
 * @param messageSize
  *          - Upon input the maximum number of bytes that can be written to the
 *            message array.
 *          - Upon successful output the actual number of bytes written to the
 *            message array.
 *
 * @param message
 *          - The message contents array
 */
typedef struct {
    uint32_t  messageTag;
    uint32_t  messageFormat;
    uint32_t  messageSize;
    uint8_t   message[MAX_SMS_MESSAGE_SIZE];
} unpack_sms_SLQSGetSMS_t;

/**
 * get sms pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param reqParam packed request
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_sms_SLQSGetSMS(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_sms_SLQSGetSMS_t *reqParam
        );

/**
 * get sms unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_sms_SLQSGetSMS(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SLQSGetSMS_t *pOutput
        );

/**
 * @param storageType
 *          - SMS message storage type
 *              - 0 - UIM - Invalid in case of CDMA device that does not
 *                          require SIM
 *              - 1 - NV
 *
 * @param requestedTag
 *          - (Optional) Message tag
 *              - 0 - Read
 *              - 1 - Not read
 *              - 2 - Mobile originated and sent
 *              - 3 - Mobile originated but not yet sent
 *
 * @param messageMode
 *          - 0x00 - CDMA, LTE (if network type is CDMA)
 *          - 0x01 - GW, LTE (if network type is UMTS)
 */
typedef struct{
    uint32_t storageType;
    uint32_t *pRequestedTag;
    uint8_t  *pMessageMode;
} pack_sms_SLQSGetSMSList_t;

/**
 * @param messageIndex
 *          - Message index of each matched message
 * @param messageTag
 *          - Messagetag
 */
typedef struct 
{
    uint32_t messageIndex;
    uint32_t messageTag;
}qmiSmsMessageList;

/**
 * @param messageListSize
 *          - Upon input the maximum number of elements that the message list
 *            array can contain.
 *          - Upon successful output the actual number of elements in the
 *            message list array.
 *
 * @param messageList
 *          - Message List
 *          - See @ref qmiSmsMessageList for more information
 */
typedef struct{
    uint32_t    messageListSize;
    qmiSmsMessageList messageList[MAX_SMS_LIST_SIZE];
} unpack_sms_SLQSGetSMSList_t;

/**
 * get sms list pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param reqParam packed request
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int pack_sms_SLQSGetSMSList(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_sms_SLQSGetSMSList_t *reqParam
        );

/**
 * get sms list unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SLQSGetSMSList(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SLQSGetSMSList_t *pOutput
        );

/**
 * @param storageType
 *          - SMS message storage type
 *              - 0 - UIM - Invalid in case of CDMA device that does not
 *                          require SIM
 *              - 1 - NV
 *
 * @param messageIndex
 *          - Message index
 *
 * @param messageTag
 *          - Message tag
 *              - 0 - Read
 *              - 1 - Not read
 *
 * @param pMessageMode
 *          - 0x00 - CDMA, LTE (if network type is CDMA)
 *          - 0x01 - GW, LTE (if network type is UMTS)
 *
 */
typedef struct{
    uint32_t storageType;
    uint32_t messageIndex;
    uint32_t messageTag;
    uint8_t *pMessageMode;
} pack_sms_SLQSModifySMSStatus_t;

/**
 * modify sms status pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param reqParam packed request
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int pack_sms_SLQSModifySMSStatus(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_sms_SLQSModifySMSStatus_t *reqParam
        );

typedef struct{

} unpack_sms_SLQSModifySMSStatus_t;

/**
 * modify sms status unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SLQSModifySMSStatus(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SLQSModifySMSStatus_t *pOutput
        );

/**
 * @param storageType
 *          - SMS message storage type
 *              - 0 - UIM - Invalid in case of CDMA device that does not
 *                          require SIM
 *              - 1 - NV
 *
 * @param pMessageIndex
 *          - (Optional) message index
 *
 * @param pMessageTag
 *          - (Optional) message tag
 *              - 0 - Read
 *              - 1 - Not read
 *              - 2 - Mobile originated and sent
 *              - 3 - Mobile originated but not yet sent
 *
 * @param pMessageMode
 *        - (Optional) message mode
 *        - this must be included if the device is capable of supporting more than one protocol
 *        - e.g. CDMA and GW
 *          - 0x00 - CDMA, LTE (if network type is CDMA)
 *          - 0x01 - GW, LTE (if network type is UMTS)
 *
 */
typedef struct{
    uint32_t storageType;
    uint32_t *pMessageIndex;
    uint32_t *pMessageTag;
    uint8_t  *pMessageMode;
} pack_sms_SLQSDeleteSMS_t;

/**
 * delete sms pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param reqParam packed request
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int pack_sms_SLQSDeleteSMS(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_sms_SLQSDeleteSMS_t *reqParam
        );

typedef struct{

} unpack_sms_SLQSDeleteSMS_t;

/**
 * delete sms unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SLQSDeleteSMS(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SLQSDeleteSMS_t *pOutput
        );

/**
 * @param messageFormat
 *          - Message format
 *              - 0 - CDMA (IS-637B)
 *              - 1 - 5 (Reserved)
 *              - 6 - GSM/WCDMA PP
 *
 * @param messageSize
 *          - The length of the message contents in bytes
 *
 * @param pLinktimer
 *          - GW SMS link open for the specified number of second
 *
 * @param pMessage
 *          - The message contents in PDU format contains SMS header and payload message
 *
 */
typedef struct{
    uint32_t messageFormat;
    uint32_t messageSize;
    uint8_t  *pMessage;
    uint8_t *pLinktimer;
} pack_sms_SendSMS_t;

/**
 * @param messageID
 *          - WMS message ID
 *
 * @param messageFailureCode
 *          - pointer to message failure code. If cause code is not provided,
 *            then value will be 0xFFFFFFFF
 *
 */
typedef struct{
    uint16_t    messageID;
    uint32_t    messageFailureCode;
} unpack_sms_SendSMS_t;

/**
 * send sms list pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param reqParam packed request
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int pack_sms_SendSMS(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_sms_SendSMS_t *reqParam
        );

/**
 * send sms  unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SendSMS(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SendSMS_t *pOutput
        );

enum eqmiCbkSetStatus{
    LIBPACK_QMI_CBK_PARAM_RESET = 0,
    LIBPACK_QMI_CBK_PARAM_SET   = 1,
    LIBPACK_QMI_CBK_PARAM_NOCHANGE
};

/**
 * @param status callback parameter
 */
typedef struct{
    enum eqmiCbkSetStatus status;
} pack_sms_SetNewSMSCallback_t;

/**
 * set new sms callback pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param reqParam packed request
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int pack_sms_SetNewSMSCallback(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_sms_SetNewSMSCallback_t reqParam
        );

typedef struct{

} unpack_sms_SetNewSMSCallback_t;

/**
 * set new sms callback unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SetNewSMSCallback(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SetNewSMSCallback_t *Output
        );
/**
 * @param storageType memory storage 0x00-UIM  0x01-NV
 * @param messageIndex MT Message index
 */
typedef struct sMSMTMessage
{
    uint32_t storageType;
    uint32_t messageIndex;
} sMSMTMessageInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param MTMessageInfo
 *          - MT Message
 *          - See @ref sMSMTMessageInfo for more information
 *
 */
struct newMTMessageTlv
{
    uint8_t    TlvPresent;
    sMSMTMessageInfo MTMessageInfo;
};
/**
 * @param ackIndicator
 *          - Parameter to indicate if ACK must be sent by the control point
 *              0x00 - Send ACK
 *              0x01 - Do not send ACK
 *
 * @param transactionID
 *          - Transaction ID of the message
 *
 * @param format
 *          - Message format
 *              0x00 - CDMA
 *              0x02 - 0x05 - Reserved
 *              0x06 - GW_PP
 *              0x07 - GW_BC
 *
 * @param length
 *          - Length of the raw message. This length should not exceed the
 *            maximum WMS payload length of 256 bytes
 *
 * @param data
 *          - Raw message data
 */
typedef struct sMSTransferRouteMTMessage
{
    uint8_t  ackIndicator;
    uint32_t transactionID;
    uint8_t  format;
    uint16_t  length;
    uint8_t  data[MAX_MS_TRANSFER_ROUTE_MSG];
} sMSTransferRouteMTMessageInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param TransferRouteMTMessageInfo
 *          - Transfer Route MT Message
 *          - See @ref sMSTransferRouteMTMessageInfo for more information
 */
struct transferRouteMessageTlv
{
    uint8_t TlvPresent;
    sMSTransferRouteMTMessageInfo TransferRouteMTMessageInfo;
};

/**
 * @param messageMode Message Mode
 */
typedef struct sMSMessageMode
{
    uint8_t  messageMode;
} sMSMessageModeInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param MessageModeInfo
 *          - Message Mode
 *          - See @ref sMSMessageModeInfo for more information
 */
struct messageModeTlv
{
    uint8_t TlvPresent;
    sMSMessageModeInfo MessageModeInfo;
};

/**
 * @param notificationType
 *          - Message mode
 *              0x00 - Primary
 *              0x01 - Secondary GSM
 *              0x02 - Secondary UMTS
 *
 * @param length
 *          - Number of sets of following elements
 *
 * @param data
 *          - Raw message data
 */

typedef struct sMSEtwsMessage
{
    uint8_t notificationType;
    uint16_t length;
    uint8_t data[MAX_MSE_TWS_MSG];
} sMSEtwsMessageInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param EtwsMessageInfo
 *          - ETWS Message
 *          - See @ref sMSEtwsMessageInfo for more information
 */
struct sMSEtwsMessageTlv
{
    uint8_t    TlvPresent;
    sMSEtwsMessageInfo EtwsMessageInfo;
};

/**
 * @param mobileCountryCode
 *          - 16 bit representation of MCC
 *              value range : 0 -999
 *
 * @param mobileNetworkCode
 *          - 16 bit representation of MNC
 *              value range : 0 -999
 *
 */
typedef struct sMSEtwsPlmn
{
    uint16_t mobileCountryCode;
    uint16_t mobileNetworkCode;
} sMSEtwsPlmnInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param ETWSPLMNInfo
 *          - ETWS PLMN Information
 *          - See @ref sMSEtwsPlmnInfo for more information
 */
struct eTWSPLMNInfoTlv
{
    uint8_t         TlvPresent;
    sMSEtwsPlmnInfo ETWSPLMNInfo;
};

/**
 * @param length
 *          - Number of sets of following element
 *
 * @param data
 *          - SMSC address
 */

typedef struct sMSCAddress
{
    uint8_t length;
    uint8_t data[MAX_MSC_ADDRESS_SIZE];
} sMSCAddressInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param SMSCInfo
 *          - SMSC Address
 *          - See @ref sMSCAddressInfo for more information
 */
struct sMSCAddressTlv
{
    uint8_t         TlvPresent;
    sMSCAddressInfo SMSCInfo;
};

/**
 * @param smsOnIMS SMS on IMS
 */
typedef struct sMSOnIMS
{
    uint8_t smsOnIMS;
}sMSOnIMSInfo;

/**
 * @param TlvPresent
 *          - Boolean indicating the presence of the TLV
 *             in the QMI response
 *
 * @param IMSInfo
 *          - SMS on IMS
 *          - See @ref sMSOnIMSInfo for more information
 */
struct sMSOnIMSTlv
{
    uint8_t      TlvPresent;
    sMSOnIMSInfo IMSInfo;
};

/**
 * @param NewMMTlv
 *          - MT message
 *
 * @param TRMessageTlv
 *          - Transfer Route MT Message
 *          - See @ref transferRouteMessageTlv for more information
 * @param MMTlv
 *          - Message mode
 *          - See @ref messageModeTlv for more information
 *
 * @param ETWSTlv
 *          - ETWS Message
 *          - See @ref sMSEtwsMessageTlv for more information
 *
 * @param ETWSPLMNTlv
 *          - ETWS PLMN Information
 *          - See @ref eTWSPLMNInfoTlv for more information
 *
 * @param SMSCTlv
 *          - SMSC Address
 *          - See @ref sMSCAddressTlv for more information
 *
 * @param IMSTlv
 *          - SMS on IMS
 *          - See @ref sMSOnIMSTlv for more information
 */
typedef struct{
    struct newMTMessageTlv         NewMMTlv;
    struct transferRouteMessageTlv TRMessageTlv;
    struct messageModeTlv          MMTlv;
    struct sMSEtwsMessageTlv       ETWSTlv;
    struct eTWSPLMNInfoTlv         ETWSPLMNTlv;
    struct sMSCAddressTlv          SMSCTlv;
    struct sMSOnIMSTlv             IMSTlv;
}unpack_sms_SetNewSMSCallback_ind_t;

/**
 * set new sms callback indication unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SetNewSMSCallback_ind(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SetNewSMSCallback_ind_t *pOutput
        );

/**
 * @param storageType
 *          - SMS message storage type
 *              - 0 - UIM - Invalid in case of CDMA device that does not
 *                          require SIM
 *              - 1 - NV
 *
 * @param messageMode
 *          - 0x00 - CDMA, LTE (if network type is CDMA)
 *          - 0x01 - GW, LTE (if network type is UMTS)
 *
 */
typedef struct{
    uint8_t storageType;
    uint8_t messageMode;
}unpack_sms_SLQSWmsMemoryFullCallBack_ind_t;

/**
 * sms full callback indication unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 */
int unpack_sms_SLQSWmsMemoryFullCallBack_ind(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_sms_SLQSWmsMemoryFullCallBack_ind_t *pOutput
        );
