/**
 * \ingroup pack
 *
 * \file nas.h
 */
#include <stdint.h>
#include "common.h"
#define NAS_OTA_MESSAGE_MAX_BUF_SIZE 2048
#define NAS_MAX_NUM_NETWORKS 30
#define NAS_MAX_DESCRIPTION_LENGTH           255
#define NAS_PLMN_LENGTH                      3
/**
 * @param len number of rssi & radio items following
 * @param rssi signal strength array
 * @param radio radio interface array
 */
typedef struct {
    uint32_t len;
    signed char rssi[8];
    uint32_t radio[8];
} unpack_nas_GetSignalStrengths_t;

/**
 * get signal strengths unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetSignalStrengths(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_GetSignalStrengths_t *pOutput
        );

/**
 * get signal strengths pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetSignalStrengths(
        pack_qmi_t *pCtx,
        uint8_t *pReq,
        uint16_t *pLen
        );

/**
 *  @param  pEmerMode
 *          - Optional parameter specifying the emergency Mode
 *          - Values:
 *              - 0 - OFF (normal)
 *              - 1 - ON (Emergency)
 *
 *  @param  pModePref
 *          - Optional parameter
 *          - Bit Mask indicating the radio technology mode preference
 *          - Bit values:
 *              - Bit 0 - cdma2000 1x
 *              - Bit 1 - cdma2000 HRPD(1xEV-DO)
 *              - Bit 2 - GSM
 *              - Bit 3 - UMTS
 *              - Bit 4 - LTE
 *
 *  @param  pBandPref
 *          - Optional parameter
 *          - Bit mask representing the band preference
 *          - Bit values:
 *              - Bit 0 - Band Class 0, A-System
 *              - Bit 1 - Band Class 0, B-System, Band Class 0
 *                        AB, GSM 850 Band
 *              - Bit 2 - Band Class 1, all blocks
 *              - Bit 3 - Band Class 2 place holder
 *              - Bit 4 - Band Class 3, A-System
 *              - Bit 5 - Band Class 4, all blocks
 *              - Bit 6 - Band Class 5, all blocks
 *              - Bit 7 - GSM_DCS_1800 band
 *              - Bit 8 - GSM Extended GSM (E-GSM) 900 band
 *              - Bit 9 - GSM Primary GSM (P-GSM) 900 band
 *              - Bit 10 - Band Class 6
 *              - Bit 11 - Band Class 7
 *              - Bit 12 - Band Class 8
 *              - Bit 13 - Band Class 9
 *              - Bit 14 - Band Class 10
 *              - Bit 15 - Band Class 11
 *              - Bit 16 - GSM 450 band
 *              - Bit 17 - GSM 480 band
 *              - Bit 18 - GSM 750 band
 *              - Bit 19 - GSM 850 band
 *              - Bit 20 - GSM Railways GSM 900 Band
 *              - Bit 21 - GSM PCS 1900 band
 *              - Bit 22 - WCDMA Europe, Japan, and China IMT 2100 band
 *              - Bit 23 - WCDMA U.S. PCS 1900 band
 *              - Bit 24 - WCDMA Europe and China DCS 1800 band
 *              - Bit 25 - WCDMA U.S. 1700 band
 *              - Bit 26 - WCDMA U.S. 850 band
 *              - Bit 27 - WCDMA Japan 800 band
 *              - Bit 28 - Band Class 12
 *              - Bit 29 - Band Class 14
 *              - Bit 30 - Reserved
 *              - Bit 31 - Band Class 15
 *              - Bit 32 to 47 - Reserved
 *              - Bit 48 - WCDMA Europe 2600 band
 *              - Bit 49 - WCDMA Europe and Japan 900 band
 *              - Bit 50 - WCDMA Japan 1700 band
 *              - Bit 51 to 55 - Reserved
 *              - Bit 56 - Band Class 16
 *              - Bit 57 - Band Class 17
 *              - Bit 58 - Band Class 18
 *              - Bit 59 - Band Class 19
 *              - Bit 60 to 64 - Reserved
 *
 *  @param  pPRLPref
 *          - Optional parameter indicating the CDMA PRL Preference
 *          - Values:
 *              - 0x0001 - Acquire available system only on the A side
 *              - 0x0002 - Acquire available system only on the B side
 *              - 0x3FFF - Acquire any available systems
 *
 *  @param  pRoamPref
 *          - Optional parameter indicating the roaming Preference
 *          - Values:
 *              - 0x01 - Acquire only systems for which the roaming indicator
 *                       is off
 *              - 0x02 - Acquire a system as long as its roaming indicator is
 *                       not off
 *              - 0x03 - Acquire only systems for which the roaming indicator
 *                       is off or solid on, i.e. not flashing; CDMA only
 *              - 0xFF - Acquire systems, regardless of their roaming indicator
 *
 *  @param  pLTEBandPref
 *          - Optional parameter
 *          - Bit mask representing the LTE band preference
 *          - Bit Values
 *              - Bit 0 - E-UTRA Operating Band 1
 *              - Bit 1 - E-UTRA Operating Band 2
 *              - Bit 2 - E-UTRA Operating Band 3
 *              - Bit 3 - E-UTRA Operating Band 4
 *              - Bit 4 - E-UTRA Operating Band 5
 *              - Bit 5 - E-UTRA Operating Band 6
 *              - Bit 6 - E-UTRA Operating Band 7
 *              - Bit 7 - E-UTRA Operating Band 8
 *              - Bit 8 - E-UTRA Operating Band 9
 *              - Bit 9 - E-UTRA Operating Band 10
 *              - Bit 10 - E-UTRA Operating Band 11
 *              - Bit 11 - E-UTRA Operating Band 12
 *              - Bit 12 - E-UTRA Operating Band 13
 *              - Bit 13 - E-UTRA Operating Band 14
 *              - Bit 16 - E-UTRA Operating Band 17
 *              - Bit 17 - E-UTRA Operating Band 18
 *              - Bit 18 - E-UTRA Operating Band 19
 *              - Bit 19 - E-UTRA Operating Band 20
 *              - Bit 20 - E-UTRA Operating Band 21
 *              - Bit 32 - E-UTRA Operating Band 33
 *              - Bit 33 - E-UTRA Operating Band 34
 *              - Bit 34 - E-UTRA Operating Band 35
 *              - Bit 35 - E-UTRA Operating Band 36
 *              - Bit 36 - E-UTRA Operating Band 37
 *              - Bit 37 - E-UTRA Operating Band 38
 *              - Bit 38 - E-UTRA Operating Band 39
 *              - Bit 39 - E-UTRA Operating Band 40
 *              - All other bits are reserved

 *  @param  pNetSelPref
 *          - Optional parameter indicating network selection preference
 *          - Values:
 *              - 0x00 - Automatic network selection
 *              - 0x01 - Manual network selection.
 *
 *  @param  pSrvDomainPref
 *          - Optional parameter indicating Service domain preference
 *          - Values:
 *              - 0x00 - Circuit switched only
 *              - 0x01 - Packet switched only
 *              - 0x02 - Circuit switched and packet switched
 *              - 0x03 - Packet switched attach
 *              - 0x04 - Packet switched detach
 *
 *  @param  pGWAcqOrderPref
 *          - Optional parameter indicating GSM/WCDMA Acquisition
 *            order Preference
 *          - Values:
 *              - 0x00 - Automatic
 *              - 0x01 - GSM then WCDMA
 *              - 0x02 - WCDMA then GSM
 *
 */
typedef struct
{
    uint8_t              *pEmerMode;
    uint16_t             *pModePref;
    uint64_t   *pBandPref;
    uint16_t             *pPRLPref;
    uint16_t             *pRoamPref;
    uint64_t   *pLTEBandPref;
    uint8_t              *pNetSelPref;
    uint32_t        *pSrvDomainPref;
    uint32_t        *pGWAcqOrderPref;
} unpack_nas_SLQSGetSysSelectionPref_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSGetSysSelectionPref(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSGetSysSelectionPref(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSGetSysSelectionPref_t *pOutput
        );

/**
 *  Contain the network selection preference.
 *
 *  @param  netReg
 *          - specifies one of the following actions:
 *              - 0x00 - Automatic registration
 *                     - Device registers according to its provisioning; mcc
 *                       and mnc fields are ignored
 *              - 0x01 - Manual Registration
 *                     - Device registers to specified network; mcc and mnc
 *                       must contain valid values
 *
 *  @param  mcc
 *          - MCC value. Range 0 to 999
 *
 *  @param  mnc
 *          - MNC value. Range 0 to 999
 *
 */
struct nas_netSelectionPref
{
    uint8_t netReg;
    uint16_t mcc;
    uint16_t mnc;
};

/**
 *  Contain the Acquisition Order Preference.
 *
 *  @param  acqOrdeLen
 *          - Number of sets of the following elements.
 *
 *  @param  pAcqOrder
 *          - Acquisition order preference to be set. Values:
 *            - 0x01 - NAS_RADIO_IF_CDMA_1X - cdma2000 1X
 *            - 0x02 - NAS_RADIO_IF_CDMA_1XEVDO - cdma2000 HRPD (1xEV-DO)
 *            - 0x04 - NAS_RADIO_IF_GSM - GSM
 *            - 0x05 - NAS_RADIO_IF_UMTS - UMTS
 *            - 0x08 - NAS_RADIO_IF_LTE - LTE
 *            - 0x09 - NAS_RADIO_IF_TDSCDMA - TD-SCDMA
 *
 */
struct nas_acqOrderPref
{
    uint8_t acqOrdeLen;
    uint8_t *pAcqOrder;
};

/**
 *  Contain the CSGID.
 *
 *  @param  mcc
 *          - MCC value. Range 0 to 999
 *
 *  @param  mnc
 *          - MNC value. Range 0 to 999
 *
 *   @param  mncPcsDigits
 *           - TRUE - MNC is a three-digit value; e.g., a reported
 *                   value of 90 corresponds to an MNC value of 090
 *           - FALSE - MNC is a two-digit value; e.g., a reported
 *                   value of 90 corresponds to an MNC value of 90
 *   @param   id
 *            - Closed subscriber group identifier.
 *
 *   @param   rat
 *            - Radio interface technology of the CSG network.
 *              Values:
 *              - 0x04 - RADIO_IF_GSM - GSM
 *              - 0x05 - RADIO_IF_UMTS - UMTS
 *              - 0x08 - RADIO_IF_LTE - LTE
 *              - 0x09 - RADIO_IF_TDSCDMA - TDS
 *
 */
struct nas_CSGID
{
    uint16_t  mcc;
    uint16_t  mnc;
    uint8_t  mncPcsDigits;
    uint32_t id;
    uint8_t  rat;
};

/**
 *  Contain the system selection preferences.
 *
 *  @param  pEmerMode
 *          - Optional parameter specifying the emergency Mode
 *          - Values:
 *              - 0 - OFF (normal)
 *              - 1 - ON (Emergency)
 *
 *  @param  pModePref
 *          - Optional parameter
 *          - Bit Mask indicating the radio technology mode preference
 *          - Bit values:
 *              - Bit 0 - cdma2000 1x
 *              - Bit 1 - cdma2000 HRPD(1xEV-DO)
 *              - Bit 2 - GSM
 *              - Bit 3 - UMTS
 *              - Bit 4 - LTE
 *
 *  @param  pBandPref
 *          - Optional parameter
 *          - Bit mask representing the band preference
 *          - Bit values:
 *              - Bit 0 - Band Class 0, A-System
 *              - Bit 1 - Band Class 0, B-System, Band Class 0
 *                        AB, GSM 850 Band
 *              - Bit 2 - Band Class 1, all blocks
 *              - Bit 3 - Band Class 2 place holder
 *              - Bit 4 - Band Class 3, A-System
 *              - Bit 5 - Band Class 4, all blocks
 *              - Bit 6 - Band Class 5, all blocks
 *              - Bit 7 - GSM_DCS_1800 band
 *              - Bit 8 - GSM Extended GSM (E-GSM) 900 band
 *              - Bit 9 - GSM Primary GSM (P-GSM) 900 band
 *              - Bit 10 - Band Class 6
 *              - Bit 11 - Band Class 7
 *              - Bit 12 - Band Class 8
 *              - Bit 13 - Band Class 9
 *              - Bit 14 - Band Class 10
 *              - Bit 15 - Band Class 11
 *              - Bit 16 - GSM 450 band
 *              - Bit 17 - GSM 480 band
 *              - Bit 18 - GSM 750 band
 *              - Bit 19 - GSM 850 band
 *              - Bit 20 - GSM Railways GSM 900 Band
 *              - Bit 21 - GSM PCS 1900 band
 *              - Bit 22 - WCDMA Europe, Japan, and China IMT 2100 band
 *              - Bit 23 - WCDMA U.S. PCS 1900 band
 *              - Bit 24 - WCDMA Europe and China DCS 1800 band
 *              - Bit 25 - WCDMA U.S. 1700 band
 *              - Bit 26 - WCDMA U.S. 850 band
 *              - Bit 27 - WCDMA Japan 800 band
 *              - Bit 28 - Band Class 12
 *              - Bit 29 - Band Class 14
 *              - Bit 30 - Reserved
 *              - Bit 31 - Band Class 15
 *              - Bit 32 to 47 - Reserved
 *              - Bit 48 - WCDMA Europe 2600 band
 *              - Bit 49 - WCDMA Europe and Japan 900 band
 *              - Bit 50 - WCDMA Japan 1700 band
 *              - Bit 51 to 55 - Reserved
 *              - Bit 56 - Band Class 16
 *              - Bit 57 - Band Class 17
 *              - Bit 58 - Band Class 18
 *              - Bit 59 - Band Class 19
 *              - Bit 60 to 64 - Reserved
 *
 *  @param  pPRLPref
 *          - Optional parameter indicating the CDMA PRL Preference
 *          - Values:
 *              - 0x0001 - Acquire available system only on the A side
 *              - 0x0002 - Acquire available system only on the B side
 *              - 0x3FFF - Acquire any available systems
 *
 *  @param  pRoamPref
 *          - Optional parameter indicating the roaming Preference
 *          - Values:
 *              - 0x01 - Acquire only systems for which the roaming indicator
 *                       is off
 *              - 0x02 - Acquire a system as long as its roaming indicator is
 *                       not off
 *              - 0x03 - Acquire only systems for which the roaming indicator
 *                       is off or solid on, i.e. not flashing; CDMA only
 *              - 0xFF - Acquire systems, regardless of their roaming indicator
 *
 *  @param  pLTEBandPref
 *          - Optional parameter
 *          - Bit mask representing the LTE band preference
 *          - Bit Values
 *              - Bit 0 - E-UTRA Operating Band 1
 *              - Bit 1 - E-UTRA Operating Band 2
 *              - Bit 2 - E-UTRA Operating Band 3
 *              - Bit 3 - E-UTRA Operating Band 4
 *              - Bit 4 - E-UTRA Operating Band 5
 *              - Bit 5 - E-UTRA Operating Band 6
 *              - Bit 6 - E-UTRA Operating Band 7
 *              - Bit 7 - E-UTRA Operating Band 8
 *              - Bit 8 - E-UTRA Operating Band 9
 *              - Bit 9 - E-UTRA Operating Band 10
 *              - Bit 10 - E-UTRA Operating Band 11
 *              - Bit 11 - E-UTRA Operating Band 12
 *              - Bit 12 - E-UTRA Operating Band 13
 *              - Bit 13 - E-UTRA Operating Band 14
 *              - Bit 16 - E-UTRA Operating Band 17
 *              - Bit 17 - E-UTRA Operating Band 18
 *              - Bit 18 - E-UTRA Operating Band 19
 *              - Bit 19 - E-UTRA Operating Band 20
 *              - Bit 20 - E-UTRA Operating Band 21
 *              - Bit 32 - E-UTRA Operating Band 33
 *              - Bit 33 - E-UTRA Operating Band 34
 *              - Bit 34 - E-UTRA Operating Band 35
 *              - Bit 35 - E-UTRA Operating Band 36
 *              - Bit 36 - E-UTRA Operating Band 37
 *              - Bit 37 - E-UTRA Operating Band 38
 *              - Bit 38 - E-UTRA Operating Band 39
 *              - Bit 39 - E-UTRA Operating Band 40
 *              - All other bits are reserved
 *
 *  @param  pNetSelPref - netSelectionPref
 *          - Optional parameter for specifying Network Selection Preference
 *          - Modem selects networks based on this parameter(if present).
 *          - see \ref netSelectionPref for more information
 *
 *  @param  pChgDuration
 *          - Optional parameter specifying the duration of the change
 *          - Values:
 *              - 0x00 - Power cycle - Remains active until the next device
 *                                     power cycle
 *              - 0x01 - Permanent - Remains active through power cycles until
 *                                   changed by client
 *              - Device will use "0x01 - permanent" as default if this
 *                parameter is omitted
 *
 *  @param  pMNCIncPCSDigStat
 *          - Optional parameter indicating if MNC includes PCS digit
 *          - Values:
 *              - TRUE - MNC is a 3 digit value; e.g., a reported value of
 *                       90 corresponds to an MNC value of 090
 *              - FALSE - MNC is a 2-digit value; e.g., a reported value of
 *                        90 corresponds to an MNC value of 90
 *
 *  @param  pSrvDomainPref
 *          - Optional parameter indicating Service domain preference
 *          - Values:
 *              - 0x00 - Circuit switched only
 *              - 0x01 - Packet switched only
 *              - 0x02 - Circuit switched and packet switched
 *              - 0x03 - Packet switched attach
 *              - 0x04 - Packet switched detach
 *
 *  @param  pGWAcqOrderPref
 *          - Optional parameter indicating GSM/WCDMA Acquisition
 *            order Preference
 *          - Values:
 *              - 0x00 - Automatic
 *              - 0x01 - GSM then WCDMA
 *              - 0x02 - WCDMA then GSM
 */
typedef struct
{
    uint8_t                     *pEmerMode;
    uint16_t                    *pModePref;
    uint64_t          *pBandPref;
    uint16_t                    *pPRLPref;
    uint16_t                    *pRoamPref;
    uint64_t          *pLTEBandPref;
    struct nas_netSelectionPref *pNetSelPref;
    uint8_t                     *pChgDuration;
    uint8_t                     *pMNCIncPCSDigStat;
    uint32_t               *pSrvDomainPref;
    uint32_t               *pGWAcqOrderPref;
    uint64_t          *pTdscdmaBandPref;
    struct nas_acqOrderPref     *pAcqOrderPref;
    uint32_t               *pSrvRegRestriction;
    struct nas_CSGID            *pCSGID;
    unsigned char               *pRAT;
} pack_nas_SLQSSetSysSelectionPref_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request Parameters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSSetSysSelectionPref(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSSetSysSelectionPref_t *pReqParam
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSSetSysSelectionPref(
        uint8_t *pResp,
        uint16_t respLen
        );
/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  band preference
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSSetBandPreference(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        uint64_t bandPref
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 */
int unpack_nas_SLQSSetBandPreference(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 * 
 *  @param  pSystemSelectionInd [Optional]
 *          - System Selection Preference indication registration. The following
 *            callbacks would not be invoked if the indication is disabled.\n
 *            \ref tFNRoamingIndicator \ref tFNDataCapabilities and
 *            \ref tFNServingSystem
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pDDTMInd [Optional]
 *          - DDTM (Data Dedicated Transmission Mode) indication registration.
 *            The following callbacks would not be invoked if the indication
 *            is disabled.\n
 *            \ref tFNDDTM
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pServingSystemInd [Optional]
 *          - Serving System indication registration. The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNBandPreference
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pDualStandByPrefInd [Optional]
 *          - Dual Standby Preference indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNDualStandByPref
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pSubscriptionInfoInd [Optional]
 *          - Subscription Information indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNSubscriptionInfo
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pNetworkTimeInd [Optional]
 *          - Network Time indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNNetworkTime
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pSysInfoInd [Optional]
 *          - System Information indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNSysInfo
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pSignalStrengthInd [Optional]
 *          - Signal Strength indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNSigInfo
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pErrorRateInd [Optional]
 *          - Error Rate indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNErrRate
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pHDRNewUATIAssInd [Optional]
 *          - HDR New UATI Assigned indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNHDRUATIUpdate
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pHDRSessionCloseInd [Optional]
 *          - HDR Session Closed indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNHDRSessionClose
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pManagedRoamingInd [Optional]
 *          - Managed Roaming indication registration.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNManagedRoaming
 *              - 0x00 - Disable
 *              - 0x01 - Enable
 *
 *  @param  pLTECphyCa [Optional]
 *          - LTE Physical Carrier Aggregation Information.The following callbacks
 *            would not be invoked if the indication is disabled.\n
 *            \ref tFNManagedRoaming
 *              - 0x00 - Disable (default value)
 *              - 0x01 - Enable
 */
typedef struct
{
    uint8_t *pSystemSelectionInd;
    uint8_t *pDDTMInd;
    uint8_t *pServingSystemInd;
    uint8_t *pDualStandByPrefInd;
    uint8_t *pSubscriptionInfoInd;
    uint8_t *pNetworkTimeInd;
    uint8_t *pSysInfoInd;
    uint8_t *pSignalStrengthInd;
    uint8_t *pErrorRateInd;
    uint8_t *pHDRNewUATIAssInd;
    uint8_t *pHDRSessionCloseInd;
    uint8_t *pManagedRoamingInd;
    uint8_t *pLTECphyCa;
} pack_nas_SLQSNasIndicationRegisterExt_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request Parameters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSNasIndicationRegisterExt(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSNasIndicationRegisterExt_t *pReqParam
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasIndicationRegisterExt(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 * @param radioInterface radio interface technology
 * @param activeBandClass active band class
 * @param activeChannel active channel
 */
typedef struct {
    uint8_t  radioInterface;
    uint16_t activeBandClass;
    uint16_t activeChannel;
} RFBandInfoElements;

/**
 * @param instancesSize number of elements in RF info instances array.
 * @param RFBandInfoElements RF info instances array
 */
typedef struct {
    uint8_t     instancesSize;
    RFBandInfoElements RFBandInfoElements[255];
} unpack_nas_GetRFInfo_t;

/**
 * get rf info pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetRFInfo(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );
/**
 * get rf info unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetRFInfo(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_GetRFInfo_t *pOutput
        );

/**
 * @param rssi RSSI in dBm.
 * @param ecio ECIO value representing negative 0.5 dBm increment
 */
typedef struct
{
    int8_t  rssi;
    int16_t ecio;
} cdmaSSInfo;

/**
 * @param rssi RSSI in dBm.
 * @param ecio ECIO value representing negative 0.5 dBm increment
 * @param sinr SINR level.
 * @param io Received IO in dBm.
 */
typedef struct
{
    int8_t  rssi;
    int16_t ecio;
    uint8_t  sinr;
    int32_t io;
} hdrSSInfo;

/**
 * @param rssi RSSI in dBm.
 * @param rsrq RSRQ value in dB
 * @param rsrp Current RSRP in dBm as measured by L1.
 * @param snr SNR level as a scaled integer in units of 0.1 dB.
 */
typedef struct
{
    int8_t  rssi;
    int8_t  rsrq;
    int16_t rsrp;
    int16_t snr;
} lteSSInfo;

/**
 * @param rssi RSSI in dBm.
 * @param rsrq RSRQ value in dB
 * @param rsrp Current RSRP in dBm as measured by L1.
 * @param snr SNR level as a scaled integer in units of 0.1 dB.
 */
typedef struct
{
    float rssi;
    float rscp;
    float ecio;
    float sinr;
} tdscdmaSigInfoExt;

/**
 * @param CDMASSInfo CDMA Signal Strength Information
 * @param HDRSSInfo HDR Signal Strength Information
 * @param GSMSSInfo GSM signal strength is the RSSI in dBm.
 * @param WCDMASSInfo WCDMA Signal Strength Information
 * @param LTESSInfo LTE Signal Strength Information
 */
typedef struct {
    cdmaSSInfo CDMASSInfo;
    hdrSSInfo  HDRSSInfo;
    int8_t     GSMSSInfo;
    cdmaSSInfo WCDMASSInfo;
    lteSSInfo  LTESSInfo;
} unpack_nas_SLQSNasGetSigInfo_t;

/**
 * get sig info pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSNasGetSigInfo(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * get sig info unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasGetSigInfo(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSNasGetSigInfo_t *pOutput
        );

/**
 *
 *  @param  pCDMASigInfo CDMA SS info
 *  @param  pHDRSigInfo HDR SS info
 *  @param  pGSMSigInfo GSM signal info
 *  @param  pWCDMASigInfo WCDMA signal info
 *  @param  pLTESigInfo LTE signal info
 *  @param  pRscp RSCP of the Primary Common Control Physical Channel
 *  @param  pTDSCDMASigInfoExt extra CDMA sig info
 */
typedef struct
{
    cdmaSSInfo        *pCDMASigInfo;
    hdrSSInfo         *pHDRSigInfo;
    int8_t            *pGSMSigInfo;
    cdmaSSInfo        *pWCDMASigInfo;
    lteSSInfo         *pLTESigInfo;
    int8_t            *pRscp;
    tdscdmaSigInfoExt *pTDSCDMASigInfoExt;
} unpack_nas_SLQSNasSigInfoCallback_t;

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      sig info indication unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasSigInfoCallback(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSNasSigInfoCallback_t *pOutput
        );

/**
 * @param mcc mobile country code
 * @param mnc mobile network code
 * @param name network name or description
 * @param sid home network system id only applies to cdma2000
 * @param nid home network id Only applies to cdma2000
 */
typedef struct {
    uint16_t    mcc;
    uint16_t    mnc;
    char        name[255];
    uint16_t    sid;
    uint16_t    nid;
} unpack_nas_GetHomeNetwork_t;

/**
 * get home network unpack
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetHomeNetwork(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_GetHomeNetwork_t *pOutput
        );

/**
 * get home network pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetHomeNetwork(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );
/**
 *  Structure for storing the service status information for CDMA and HDR
 *  networks.
 *
 *  @param  srvStatus
 *          - Service status of the system.
 *              - 0x00 - No service
 *              - 0x01 - Limited service
 *              - 0x02 - Service
 *              - 0x03 - Limited regional service
 *              - 0x04 - Power save
 *              - 0xFF - Not Available
 *
 *  @param  isPrefDataPath
 *          - Whether the RAT is the preferred data path.
 *              - 0x00 - Not preferred
 *              - 0x01 - Preferred
 *              - 0xFF - Not Available
 *
 */
typedef struct
{
    uint8_t srvStatus;
    uint8_t isPrefDataPath;
} nas_SrvStatusInfo;

/**
 *  Structure for storing the service status information for GSM, WCDMA and
 *  LTE networks.
 *
 *  @param  srvStatus
 *          - Service status of the system.
 *              - 0x00 - No service
 *              - 0x01 - Limited service
 *              - 0x02 - Service
 *              - 0x03 - Limited regional service
 *              - 0x04 - Power save
 *              - 0xFF - Not Available
 *
 *  @param  trueSrvStatus
 *          - True service status of the system.
 *          - Not applicable to CDMA/HDR.
 *              - 0x00 - No service
 *              - 0x01 - Limited service
 *              - 0x02 - Service
 *              - 0x03 - Limited regional service
 *              - 0x04 - Power save
 *              - 0xFF - Not Available
 *
 *  @param  isPrefDataPath
 *          - Whether the RAT is the preferred data path.
 *              - 0x00 - Not preferred
 *              - 0x01 - Preferred
 *              - 0xFF - Not Available
 *
 */
typedef struct
{
    uint8_t srvStatus;
    uint8_t trueSrvStatus;
    uint8_t isPrefDataPath;
} nas_GSMSrvStatusInfo;

/**
 *  Structure for storing the System Information common to CDMA, HDR, GSM,
 *  WCDMA and LTE networks.
 *
 *  @param  srvDomainValid
 *          - Indicates whether the service domain is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  srvDomain
 *          - Service domain registered on the system.
 *              - 0x00 - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *              - 0x04 - Camped
 *              - 0xFF - Not Available
 *
 *  @param  srvCapabilityValid
 *          - Indicates whether the service capability is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  srvCapability
 *          - Current system's service capability.
 *              - 0x00 - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *              - 0x04 - Camped
 *              - 0xFF - Not Available
 *
 *  @param  roamStatusValid
 *          - Indicates whether the roaming status is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  roamStatus
 *          - Current roaming status.
 *              - 0x00 - Off
 *              - 0x01 - On
 *              - 0x02 - Blinking
 *              - 0x03 - Out of the neighborhood
 *              - 0x04 - Out of the building
 *              - 0x05 - Preferred system
 *              - 0x06 - Available system
 *              - 0x07 - Alliance partner
 *              - 0x08 - Premium partner
 *              - 0x09 - Full service
 *              - 0x0A - Partial service
 *              - 0x0B - Banner is on
 *              - 0x0C - Banner is off
 *              - 0x0D to 0x3F - Reserved for Standard Enhanced Roaming
 *                               Indicator Numbers
 *              - 0x40 to 0x7F - Reserved for Non-Standard Enhanced Roaming
 *                               Indicator Numbers
 *              - 0x40 to 0xFF - Reserved.
 *              - 0xFF - Not Available
 *          - Values from 0x02 onward are only applicable for 3GPP2
 *
 *  @param  isSysForbiddenValid
 *          - Indicates whether the forbidden system is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  isSysForbidden
 *          - Whether the system is forbidden.
 *              - 0x00 - Not forbidden
 *              - 0x01 - Forbidden
 *              - 0xFF - Not Available
 *
 */
typedef struct
{
    uint8_t srvDomainValid;
    uint8_t srvDomain;
    uint8_t srvCapabilityValid;
    uint8_t srvCapability;
    uint8_t roamStatusValid;
    uint8_t roamStatus;
    uint8_t isSysForbiddenValid;
    uint8_t isSysForbidden;
} nas_sysInfoCommon;

/**
 *  Structure for storing the CDMA System Information.
 *
 *  @param  sysInfoCDMA
 *          - See \ref sysInfoCommon for more information.
 *
 *  @param  isSysPrlMatchValid
 *          - Indicates whether the system PRL match is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  isSysPrlMatch
 *          - Indicates whether the system is in a PRL.
 *          - Only applies to CDMA/HDR.
 *              - 0x00 - System is not in a PRL
 *              - 0x01 - System is in a PRL
 *              - 0xFF - Not Available
 *          - If the system is not in a PRL, roam_status carries the value from
 *            the default roaming indicator in the PRL.
 *          - If the system is in a PRL, roam_status is set to the value based
 *            on the standard specification.
 *
 *  @param  pRevInUseValid
 *          - Indicates whether the P_Rev in use is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  pRevInUse
 *          - The lesser of the base station P_Rev and mobile P_Rev
 *          - Only applicable for CDMA.
 *              - 0xFF - Not Available
 *
 *  @param  bsPRevValid
 *          - Indicates whether the base station P_Rev is valid
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  bsPRev
 *          - Base station P_Rev.
 *          - Only applicable for CDMA.
 *              - 0xFF - Not Available
 *
 *  @param  ccsSupportedValid
 *          - Indicates whether the supported concurrent service is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  ccsSupported
 *          - Whether concurrent service is supported.
 *          - Only applicable for CDMA.
 *              - 0x00 - Not supported
 *              - 0x01 - Supported
 *              - 0xFF - Not Available
 *
 *  @param  cdmaSysIdValid
 *          - Indicates whether the CDMA system ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  systemID
 *          - System ID.
 *              - 0xFFFF - Not Available
 *
 *  @param  networkID
 *          - Network ID.
 *              - 0xFFFF - Not Available
 *
 *  @param  bsInfoValid
 *          - Indicates whether the base station information is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  baseLat
 *          - Base station latitude in units of 0.25 sec.
 *          - Expressed as a two's complement signed number with positive
 *            numbers signifying North latitudes.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  baseLong
 *          - Base station longitude in units of 0.25 sec.
 *          - Expressed as a two's complement signed number with positive
 *            numbers signifying East latitudes.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  packetZoneValid
 *          - Indicates whether the packet zone is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  packetZone
 *          - Packet zone (8-bit).
 *              - 0xFFFF indicates no packet zone.
 *          - Only applicable for CDMA.
 *
 *  @param  networkIdValid
 *          - Indicates whether the network ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  MCC[PLMN_LENGTH]
 *          - Mobile Country Code.
 *          - MCC digits in ASCII characters
 *          - MCC wildcard value is returned as {'3', 0xFF, 0xFF}.
 *
 *  @param  MNC[PLMN_LENGTH]
 *          - Mobile Network Code.
 *          - MNC digits in ASCII characters
 *          - An unused byte is set to 0xFF.
 *          - MNC wildcard value is returned as {'7', 0xFF, 0xFF}.
 *
 */
typedef struct
{
    nas_sysInfoCommon sysInfoCDMA;
    uint8_t          isSysPrlMatchValid;
    uint8_t          isSysPrlMatch;
    uint8_t          pRevInUseValid;
    uint8_t          pRevInUse;
    uint8_t          bsPRevValid;
    uint8_t          bsPRev;
    uint8_t          ccsSupportedValid;
    uint8_t          ccsSupported;
    uint8_t          cdmaSysIdValid;
    uint16_t          systemID;
    uint16_t          networkID;
    uint8_t          bsInfoValid;
    uint16_t               baseId;
    uint32_t              baseLat;
    uint32_t              baseLong;
    uint8_t          packetZoneValid;
    uint16_t               packetZone;
    uint8_t          networkIdValid;
    uint8_t          MCC[3];
    uint8_t          MNC[3];
} nas_CDMASysInfo;

/**
 *  Structure for storing the HDR System Information.
 *
 *  @param  sysInfoHDR
 *          - See \ref sysInfoCommon for more information.
 *
 *  @param  isSysPrlMatchValid
 *          - Indicates whether the system PRL match is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  isSysPrlMatch
 *          - Indicates whether the system is in a PRL.
 *          - Only applies to CDMA/HDR.
 *              - 0x00 - System is not in a PRL
 *              - 0x01 - System is in a PRL
 *              - 0xFF - Not Available
 *          - If the system is not in a PRL, roam_status carries the value from
 *            the default roaming indicator in the PRL.
 *          - If the system is in a PRL, roam_status is set to the value based
 *            on the standard specification.
 *
 *  @param  hdrPersonalityValid
 *          - Indicates whether the HDR personality is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  hdrPersonality
 *          - HDR personality information.
 *          - Only applicable for HDR.
 *              - 0x00 - None
 *              - 0x02 - HRPD
 *              - 0x03 - eHRPD
 *              - 0xFF - Not Available
 *
 *  @param  hdrActiveProtValid
 *          - Indicates whether the HDR active protocol revision information
 *            is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  hdrActiveProt
 *          - HDR active protocol revision information .
 *          - Only applicable for HDR.
 *              - 0x00 - None
 *              - 0x02 - HDR Rel 0
 *              - 0x03 - HDR Rel A
 *              - 0x04 - HDR Rel B
 *              - 0xFF - Not Available
 *
 *  @param  is856SysIdValid
 *          - Indicates whether the IS-856 system ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  is856SysId[SLQS_SYSTEM_ID_SIZE]
 *          - IS-856 system ID.
 *          - Only applicable for HDR.
 *
 */
typedef struct
{
   nas_sysInfoCommon sysInfoHDR;
   uint8_t          isSysPrlMatchValid;
   uint8_t          isSysPrlMatch;
   uint8_t          hdrPersonalityValid;
   uint8_t          hdrPersonality;
   uint8_t          hdrActiveProtValid;
   uint8_t          hdrActiveProt;
   uint8_t          is856SysIdValid;
   uint8_t          is856SysId[16];
} nas_HDRSysInfo;

/**
 *  Structure for storing the GSM System Information.
 *
 *  @param  sysInfoGSM
 *          - See \ref sysInfoCommon for more information.
 *
 *  @param  lacValid
 *          - Indicates whether the location area code is valid..
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  lac
 *          - Location area code.
 *          - Only applies to 3GPP.
 *              - 0xFFFF - Not Available
 *
 *  @param  cellIdValid
 *          - Indicates whether the cell ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  cellId
 *          - Cell ID.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  regRejectInfoValid
 *          - Indicates whether the registration reject information is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  rejectSrvDomain
 *          - Type of service domain in which the registration is rejected.
 *              - 0x00 - SYS_SRV_DOMAIN_NO_SRV - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *              - 0x04 - Camped
 *              - 0xFF - Not Available
 *
 *  @param  rejCause
 *          - Reject cause values sent are specified in
 *            [3GPP TS 24.008, Section 10.5.3.6].
 *              - 0xFF - Not Available
 *
 *
 *  @param  networkIdValid
 *          - Indicates whether the network ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  MCC[PLMN_LENGTH]
 *          - Mobile Country Code.
 *          - MCC digits in ASCII characters
 *
 *  @param  MNC[PLMN_LENGTH]
 *          - Mobile Network Code.
 *          - MNC digits in ASCII characters
 *          - An unused byte is set to 0xFF.
 *          - In case of two-digit MNC values, the third (unused) digit is set to 0xFF.
 *            For example, 15 (a two-digit MNC) is reported using the byte stream
 *            0x31 0x35 0xFF.
 *
 *  @param  egprsSuppValid
 *          - Indicates whether the EGPRS support is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  egprsSupp
 *          - EGPRS support indication.
 *          - Only applicable for GSM.
 *              - 0x00 - Not available
 *              - 0x01 - Available
 *              - 0xFF - Not Available
 *
 *  @param  dtmSuppValid
 *          - Indicates whether Dual Transfer mode support is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  dtmSupp
 *          - Dual Transfer mode support indication.
 *          - Only applicable for GSM.
 *              - 0x00 - Not available
 *              - 0x01 - Available
 *              - 0xFF - Not Available
 *
 */
typedef struct
{
     nas_sysInfoCommon sysInfoGSM;
     uint8_t          lacValid;
     uint16_t          lac;
     uint8_t          cellIdValid;
     uint32_t         cellId;
     uint8_t          regRejectInfoValid;
     uint8_t          rejectSrvDomain;
     uint8_t          rejCause;
     uint8_t          networkIdValid;
     uint8_t          MCC[3];
     uint8_t          MNC[3];
     uint8_t          egprsSuppValid;
     uint8_t          egprsSupp;
     uint8_t          dtmSuppValid;
     uint8_t          dtmSupp;
} nas_GSMSysInfo;

/**
 *  Structure for storing the WCDMA System Information.
 *
 *  @param  sysInfoWCDMA
 *          - See \ref sysInfoCommon for more information.
 *
 *  @param  lacValid
 *          - Indicates whether the location area code is valid..
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  lac
 *          - Location area code.
 *          - Only applies to 3GPP.
 *              - 0xFFFF - Not Available
 *
 *  @param  cellIdValid
 *          - Indicates whether the cell ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  cellId
 *          - Cell ID.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  regRejectInfoValid
 *          - Indicates whether the registration reject information is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  rejectSrvDomain
 *          - Type of service domain in which the registration is rejected.
 *              - 0x00 - SYS_SRV_DOMAIN_NO_SRV - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *              - 0x04 - Camped
 *              - 0xFF - Not Available
 *
 *  @param  rejCause
 *          - Reject cause values sent are specified in
 *            [3GPP TS 24.008, Section 10.5.3.6].
 *              - 0xFF - Not Available
 *
 *
 *  @param  networkIdValid
 *          - Indicates whether the network ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  MCC[PLMN_LENGTH]
 *          - Mobile Country Code.
 *          - MCC digits in ASCII characters
 *
 *  @param  MNC[PLMN_LENGTH]
 *          - Mobile Network Code.
 *          - MNC digits in ASCII characters
 *          - An unused byte is set to 0xFF.
 *          - In case of two-digit MNC values, the third (unused) digit is set to 0xFF.
 *            For example, 15 (a two-digit MNC) is reported using the byte stream
 *            0x31 0x35 0xFF.
 *
 *  @param  hsCallStatusValid
 *          - Indicates whether the high-speed call status is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  hsCallStatus
 *          - Call status on high speed.
 *          - Only applicable for WCDMA.
 *              - 0x00 - HSDPA and HSUPA are unsupported
 *              - 0x01 - HSDPA is supported
 *              - 0x02 - HSUPA is supported
 *              - 0x03 - HSDPA and HSUPA are supported
 *              - 0x04 - HSDPA+ is supported
 *              - 0x05 - HSDPA+ and HSUPA are supported
 *              - 0x06 - Dual-cell HSDPA+ is supported
 *              - 0x07 - Dual-cell HSDPA+ and HSUPA are supported
 *              - 0xFF - Not Available
 *
 *  @param  hsIndValid
 *          - Indicates whether high-speed service indication is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  hsInd
 *          - High-speed service indication
 *          - Only applicable for WCDMA.
 *              - 0x00 - HSDPA and HSUPA are unsupported
 *              - 0x01 - HSDPA is supported
 *              - 0x02 - HSUPA is supported
 *              - 0x03 - HSDPA and HSUPA are supported
 *              - 0x04 - HSDPA+ is supported
 *              - 0x05 - HSDPA+ and HSUPA are supported
 *              - 0x06 - Dual-cell HSDPA+ is supported
 *              - 0x07 - Dual-cell HSDPA+ and HSUPA are supported
 *              - 0xFF - Not Available
 *
 *  @param  pscValid
 *          - Indicates whether primary scrambling code is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  psc
 *          - Primary scrambling code.
 *              - 0xFFFF - Not Available
 *
 */
typedef struct
{
    nas_sysInfoCommon sysInfoWCDMA;
    uint8_t          lacValid;
    uint16_t          lac;
    uint8_t          cellIdValid;
    uint32_t         cellId;
    uint8_t          regRejectInfoValid;
    uint8_t          rejectSrvDomain;
    uint8_t          rejCause;
    uint8_t          networkIdValid;
    uint8_t          MCC[3];
    uint8_t          MNC[3];
    uint8_t          hsCallStatusValid;
    uint8_t          hsCallStatus;
    uint8_t          hsIndValid;
    uint8_t          hsInd;
    uint8_t          pscValid;
    uint16_t          psc;
} nas_WCDMASysInfo;

/**
 *  Structure for storing the LTE System Information.
 *
 *  @param  sysInfoLTE
 *          - See \ref sysInfoCommon for more information.
 *
 *  @param  lacValid
 *          - Indicates whether the location area code is valid..
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  lac
 *          - Location area code.
 *          - Only applies to 3GPP.
 *              - 0xFFFF - Not Available
 *
 *  @param  cellIdValid
 *          - Indicates whether the cell ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  cellId
 *          - Cell ID.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  regRejectInfoValid
 *          - Indicates whether the registration reject information is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  rejectSrvDomain
 *          - Type of service domain in which the registration is rejected.
 *              - 0x00 - SYS_SRV_DOMAIN_NO_SRV - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *              - 0x04 - Camped
 *              - 0xFF - Not Available
 *
 *  @param  rejCause
 *          - Reject cause values sent are specified in
 *            [3GPP TS 24.008, Section 10.5.3.6].
 *              - 0xFF - Not Available
 *
 *
 *  @param  networkIdValid
 *          - Indicates whether the network ID is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  MCC[PLMN_LENGTH]
 *          - Mobile Country Code.
 *          - MCC digits in ASCII characters
 *
 *  @param  MNC[PLMN_LENGTH]
 *          - Mobile Network Code.
 *          - MNC digits in ASCII characters
 *          - An unused byte is set to 0xFF.
 *          - In case of two-digit MNC values, the third (unused) digit is set to 0xFF.
 *            For example, 15 (a two-digit MNC) is reported using the byte stream
 *            0x31 0x35 0xFF.
 *
 *  @param  tacValid
 *          - Indicates whether tracking area code is valid.
 *              - 0x00 - Invalid
 *              - 0x01 - Valid
 *              - 0xFF - Not Available
 *
 *  @param  tac
 *          - Tracking area code.
 *          - Only applicable for LTE.
 *              - 0xFFFF - Not Available
 *
 */
typedef struct
{
    nas_sysInfoCommon sysInfoLTE;
    uint8_t          lacValid;
    uint16_t          lac;
    uint8_t          cellIdValid;
    uint32_t         cellId;
    uint8_t          regRejectInfoValid;
    uint8_t          rejectSrvDomain;
    uint8_t          rejCause;
    uint8_t          networkIdValid;
    uint8_t          MCC[3];
    uint8_t          MNC[3];
    uint8_t          tacValid;
    uint16_t          tac;
} nas_LTESysInfo;

/**
 *  Structure for storing the Additional CDMA System Information.
 *
 *  @param  geoSysIdx
 *          - System table index referencing the beginning of the geo in which
 *            the current serving system is present.
 *          - When the system index is not known, 0xFFFF is used.
 *
 *  @param  regPrd
 *          - Registration period after the CDMA system is acquired.
 *          - When the CDMA registration period is not valid, 0xFFFF is used.
 *
 */
typedef struct
{
    uint16_t geoSysIdx;
    uint16_t regPrd;
} nas_AddCDMASysInfo;

/**
 *  Structure for storing the Additional GSM and WCDMA System Information.
 *
 *  @param  geoSysIdx
 *          - System table index referencing the beginning of the geo in which
 *            the current serving system is present.
 *          - When the system index is not known, 0xFFFF is used.
 *
 *  @param  cellBroadcastCap
 *          - Cell broadcast capability of the serving system.
 *          - When the CDMA registration period is not valid, 0xFFFF is used.
 *              - NAS_CELL_BROADCAST_CAP_UNKNOWN - Cell broadcast support is unknown
 *              - NAS_CELL_BROADCAST_CAP_OFF - Cell broadcast is not supported
 *              - NAS_CELL_BROADCAST_CAP_ON - Cell broadcast is supported
 *
 */
typedef struct
{
    uint16_t  geoSysIdx;
    uint32_t cellBroadcastCap;
} nas_AddSysInfo;

/**
 *  Structure for storing the GSM and WCDMA Call Barring System Information.
 *
 *  @param  csBarStatus
 *          - Call barring status for circuit-switched calls.
 *              - NAS_CELL_ACCESS_NORMAL_ONLY - Cell access is allowed for normal calls only
 *              - NAS_CELL_ACCESS_EMERGENCY_ONLY - Cell access is allowed for emergency calls only
 *              - NAS_CELL_ACCESS_NO_CALLS - Cell access is not allowed for any call type
 *              - NAS_CELL_ACCESS_ALL_CALLS - Cell access is allowed for all call types
 *              - NAS_CELL_ACCESS_UNKNOWN - Cell access type is unknown
 *
 *  @param  psBarStatus
 *          - Call barring status for packet-switched calls.
 *              - NAS_CELL_ACCESS_NORMAL_ONLY - Cell access is allowed for normal calls only
 *              - NAS_CELL_ACCESS_EMERGENCY_ONLY - Cell access is allowed for emergency calls only
 *              - NAS_CELL_ACCESS_NO_CALLS - Cell access is not allowed for any call type
 *              - NAS_CELL_ACCESS_ALL_CALLS - Cell access is allowed for all call types
 *              - NAS_CELL_ACCESS_UNKNOWN - Cell access type is unknown
 *
 */
typedef struct
{
    uint32_t csBarStatus;
    uint32_t psBarStatus;
} nas_CallBarringSysInfo;

/**
 *
 *  @param  pCDMASrvStatusInfo
 *          - See \ref SrvStatusInfo for more information.
 *
 *  @param  pHDRSrvStatusInfo
 *          - See \ref SrvStatusInfo for more information.
 *
 *  @param  pGSMSrvStatusInfo
 *          - See \ref GSMSrvStatusInfo for more information.
 *
 *  @param  pWCDMASrvStatusInfo
 *          - See \ref GSMSrvStatusInfo for more information.
 *
 *  @param  pLTESrvStatusInfo
 *          - See \ref GSMSrvStatusInfo for more information.
 *
 *  @param  pCDMASysInfo
 *          - See \ref CDMASysInfo for more information.
 *
 *  @param  pHDRSysInfo
 *          - See \ref HDRSysInfo for more information.
 *
 *  @param  pGSMSysInfo
 *          - See \ref GSMSysInfo for more information.
 *
 *  @param  pWCDMASysInfo
 *          - See \ref WCDMASysInfo for more information.
 *
 *  @param  pLTESysInfo
 *          - See \ref LTESysInfo for more information.
 *
 *  @param  pAddCDMASysInfo
 *          - See \ref AddCDMASysInfo for more information.
 *
 *  @param  pAddHDRSysInfo
 *          - System table index referencing the beginning of the geo in which
 *            the current serving system is present.
 *          - When the system index is not known, 0xFFFF is used.
 *
 *  @param  pAddGSMSysInfo
 *          - See \ref AddSysInfo for more information.
 *
 *  @param  pAddWCDMASysInfo
 *          - See \ref AddSysInfo for more information.
 *
 *  @param  pAddLTESysInfo
 *          - System table index referencing the beginning of the geo in which
 *            the current serving system is present.
 *          - When the system index is not known, 0xFFFF is used.
 *
 *  @param  pGSMCallBarringSysInfo
 *          - See \ref CallBarringSysInfo for more information.
 *
 *  @param  pWCDMACallBarringSysInfo
 *          - See \ref CallBarringSysInfo for more information.
 *
 *  @param  pLTEVoiceSupportSysInfo
 *          - Indicates voice support status on LTE.
 *              - 0x00 - Voice is not supported
 *              - 0x01 - Voice is supported
 *
 *  @param  pGSMCipherDomainSysInfo
 *          - Ciphering on the service domain.
 *              - 0x00 - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *
 *  @param  pWCDMACipherDomainSysInfo
 *          - Ciphering on the service domain.
 *              - 0x00 - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *
 */
typedef struct
{
    nas_SrvStatusInfo      *pCDMASrvStatusInfo;
    nas_SrvStatusInfo      *pHDRSrvStatusInfo;
    nas_GSMSrvStatusInfo   *pGSMSrvStatusInfo;
    nas_GSMSrvStatusInfo   *pWCDMASrvStatusInfo;
    nas_GSMSrvStatusInfo   *pLTESrvStatusInfo;
    nas_CDMASysInfo        *pCDMASysInfo;
    nas_HDRSysInfo         *pHDRSysInfo;
    nas_GSMSysInfo         *pGSMSysInfo;
    nas_WCDMASysInfo       *pWCDMASysInfo;
    nas_LTESysInfo         *pLTESysInfo;
    nas_AddCDMASysInfo     *pAddCDMASysInfo;
    uint16_t               *pAddHDRSysInfo;
    nas_AddSysInfo         *pAddGSMSysInfo;
    nas_AddSysInfo         *pAddWCDMASysInfo;
    uint16_t               *pAddLTESysInfo;
    nas_CallBarringSysInfo *pGSMCallBarringSysInfo;
    nas_CallBarringSysInfo *pWCDMACallBarringSysInfo;
    uint8_t                *pLTEVoiceSupportSysInfo;
    uint8_t                *pGSMCipherDomainSysInfo;
    uint8_t                *pWCDMACipherDomainSysInfo;
} unpack_nas_SLQSGetSysInfo_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSGetSysInfo(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSGetSysInfo(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSGetSysInfo_t *pOutput
        );

/**
 *  @param  pCDMASrvStatusInfo
 *          - See \ref SrvStatusInfo for more information.
 *
 *  @param  pHDRSrvStatusInfo
 *          - See \ref SrvStatusInfo for more information.
 *
 *  @param  pGSMSrvStatusInfo
 *          - See \ref GSMSrvStatusInfo for more information.
 *
 *  @param  pWCDMASrvStatusInfo
 *          - See \ref GSMSrvStatusInfo for more information.
 *
 *  @param  pLTESrvStatusInfo
 *          - See \ref GSMSrvStatusInfo for more information.
 *
 *  @param  pCDMASysInfo
 *          - See \ref CDMASysInfo for more information.
 *
 *  @param  pHDRSysInfo
 *          - See \ref HDRSysInfo for more information.
 *
 *  @param  pGSMSysInfo
 *          - See \ref GSMSysInfo for more information.
 *
 *  @param  pWCDMASysInfo
 *          - See \ref WCDMASysInfo for more information.
 *
 *  @param  pLTESysInfo
 *          - See \ref LTESysInfo for more information.
 *
 *  @param  pAddCDMASysInfo
 *          - See \ref AddCDMASysInfo for more information.
 *
 *  @param  pAddHDRSysInfo
 *          - System table index referencing the beginning of the geo in which
 *            the current serving system is present.
 *          - When the system index is not known, 0xFFFF is used.
 *
 *  @param  pAddGSMSysInfo
 *          - See \ref AddSysInfo for more information.
 *
 *  @param  pAddWCDMASysInfo
 *          - See \ref AddSysInfo for more information.
 *
 *  @param  pAddLTESysInfo
 *          - System table index referencing the beginning of the geo in which
 *            the current serving system is present.
 *          - When the system index is not known, 0xFFFF is used.
 *
 *  @param  pGSMCallBarringSysInfo
 *          - See \ref CallBarringSysInfo for more information.
 *
 *  @param  pWCDMACallBarringSysInfo
 *          - See \ref CallBarringSysInfo for more information.
 *
 *  @param  pLTEVoiceSupportSysInfo
 *          - Indicates voice support status on LTE.
 *              - 0x00 - Voice is not supported
 *              - 0x01 - Voice is supported
 *
 *  @param  pGSMCipherDomainSysInfo
 *          - Ciphering on the service domain.
 *              - 0x00 - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *
 *  @param  pWCDMACipherDomainSysInfo
 *          - Ciphering on the service domain.
 *              - 0x00 - No service
 *              - 0x01 - Circuit-switched only
 *              - 0x02 - Packet-switched only
 *              - 0x03 - Circuit-switched and packet-switched
 *
 *  @param  pSysInfoNoChange
 *          - System Info No Change.
 *          - Flag used to notify clients that a request to select a network
 *            ended with no change in the PLMN.
 *              - 0x01 - No change in system information
 *
 */
typedef struct
{
    nas_SrvStatusInfo      *pCDMASrvStatusInfo;
    nas_SrvStatusInfo      *pHDRSrvStatusInfo;
    nas_GSMSrvStatusInfo   *pGSMSrvStatusInfo;
    nas_GSMSrvStatusInfo   *pWCDMASrvStatusInfo;
    nas_GSMSrvStatusInfo   *pLTESrvStatusInfo;
    nas_CDMASysInfo        *pCDMASysInfo;
    nas_HDRSysInfo         *pHDRSysInfo;
    nas_GSMSysInfo         *pGSMSysInfo;
    nas_WCDMASysInfo       *pWCDMASysInfo;
    nas_LTESysInfo         *pLTESysInfo;
    nas_AddCDMASysInfo     *pAddCDMASysInfo;
    uint16_t               *pAddHDRSysInfo;
    nas_AddSysInfo         *pAddGSMSysInfo;
    nas_AddSysInfo         *pAddWCDMASysInfo;
    uint16_t               *pAddLTESysInfo;
    nas_CallBarringSysInfo *pGSMCallBarringSysInfo;
    nas_CallBarringSysInfo *pWCDMACallBarringSysInfo;
    uint8_t               *pLTEVoiceSupportSysInfo;
    uint8_t               *pGSMCipherDomainSysInfo;
    uint8_t               *pWCDMACipherDomainSysInfo;
    uint8_t               *pSysInfoNoChange;
} unpack_nas_SLQSSysInfoCallback_t;

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasSysInfoCallback(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSSysInfoCallback_t *pOutput
        );

/**
 * @param RegistrationState registration state
 * @param CSDomain CS domain
 * @param PSDomain PS domain
 * @param RAN radio access network
 * @param RadioIfacesSize radio interface size
 * @param RadioIfaces radio interface list
 * @param Roaming romaing indicator
 * @param MCC Mobile country code
 * @param MNC Mobile network code
 * @param nameSize network name size
 * @param Name network name
 * @param DataCapsLen data capabilities len
 * @param DataCap data capabilities
 */
typedef struct {
    uint32_t   RegistrationState;
    uint32_t   CSDomain;
    uint32_t   PSDomain;
    uint32_t   RAN;
    uint8_t    RadioIfacesSize;
    uint8_t    RadioIfaces[255];
    uint32_t   Roaming;
    uint16_t   MCC;
    uint16_t   MNC;
    uint8_t    nameSize;
    uint8_t    Name[255];
    uint8_t    DataCapsLen;
    uint8_t    DataCaps[255];
} unpack_nas_GetServingNetwork_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetServingNetwork(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetServingNetwork(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_GetServingNetwork_t *pOutput
        );

/**
 * @param DataCapsLen data capabilities len
 * @param DataCap data capabilities
 */
typedef struct {
    uint8_t    DataCapsLen;
    uint8_t    DataCaps[255];
} unpack_nas_GetServingNetworkCapabilities_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetServingNetworkCapabilities(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetServingNetworkCapabilities(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_GetServingNetworkCapabilities_t *pOutput
        );

/**
 *  @param  MCC Mobile Country Code
 *  @param  MNC Mobile Network Code
 *  @param  InUse current network or not
 *  @param  Roaming Home/Roam Status of the Network
 *  @param  Forbidden Network in the forbidden network list or not
 *  @param  Preferred Network in the Preferred network list or not
 *  @param  Desription Network Name/Description
 */
typedef struct
{
    uint16_t          MCC;
    uint16_t          MNC;
    uint32_t          InUse;
    uint32_t          Roaming;
    uint32_t          Forbidden;
    uint32_t          Preferred;
    char              Desription[255];
}nas_QmiNas3GppNetworkInfo;

/**
 *  Contain the 3GPP radio access technology information.
 *
 *  \param  MCC
 *          - Mobile Country Code
 *
 *  \param  MNC
 *          - Mobile Network Code
 *
 *  \param  RAT
 *          - Radio Access Technology
 *              - 0x04 - GERAN
 *              - 0x05 - UMTS
 *              - 0x08 - LTE
 *              - 0x09 - TD-SCDMA
 *
 */
typedef struct 
{
    uint16_t MCC;
    uint16_t MNC;
    uint8_t RAT;
}nas_QmiNas3GppNetworkRAT;


/**
 *  Contain the PCS Digit information
 *
 *  \param  MCC
 *          - Mobile Country Code
 *
 *  \param  MNC
 *          - Mobile Network Code
 *
 *  \param  includes_pcs_digit
 *          - this field is use to interpret the length of corresponding MNC reported
 *          - 0x01 - MNC is a three-digit value
 *          - 0x00 - MNC is a two-digit value
 *
 */
typedef struct 
{
    uint16_t MCC;
    uint16_t MNC;
    uint8_t includes_pcs_digit;
}nas_QmisNasPcsDigit;

/**
 * @param InstanceSize total instances
 * @param Instances info for instances
 */
typedef struct
{
    uint8_t   *p3GppNetworkInstanceSize;
    nas_QmiNas3GppNetworkInfo  *p3GppNetworkInfoInstances;
    uint8_t   *pRATInstanceSize;
    nas_QmiNas3GppNetworkRAT *pRATINstance;
    uint8_t   *pPCSInstanceSize;
    nas_QmisNasPcsDigit *pPCSInstance;
    uint32_t  *pScanResult;
}unpack_nas_PerformNetworkScan_t;


/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_PerformNetworkScan(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_PerformNetworkScan(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_PerformNetworkScan_t *pOutput
        );


/**
 *  @param  ValidityCW0[OUT]
 *          - Values
 *              - 0- Invalid.
 *              - 1- Valid.
 *
 *  @param  CQIValueCW0[OUT]
 *          - Values
 *              - Range 0~15
 *
 *  @param  ValidityCW1[OUT]
 *          - Values
 *              - 0- Invalid.
 *              - 1- Valid.
 *
 *  @param  CQIValueCW1[OUT]
 *          - Values
 *              - Range 0~15
 */
typedef struct
{
    uint8_t   ValidityCW0;
    uint8_t   CQIValueCW0;
    uint8_t   ValidityCW1;
    uint8_t   CQIValueCW1;
}unpack_nas_SLQSSwiGetLteCQI_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSSwiGetLteCQI(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSSwiGetLteCQI(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSSwiGetLteCQI_t *pOutput
        );

/**
 *  Structure for storing the common information for the device.
 *
 *  @param  temperature
 *          - Temperature.
 *              - 8-bit signed integer
 *              - 0xFF - Not Available.
 *
 *  @param  modemMode
 *          - Modem Operating Mode.
 *              - 0x00 - POWERING OFF
 *              - 0x01 - FACTORY TEST
 *              - 0x02 - OFFLINE
 *              - 0x03 - OFFLINE_AMPS
 *              - 0x04 - OFFLINE_CDMA
 *              - 0x05 - ONLINE
 *              - 0x06 - LOW POWER MODE
 *              - 0x07 - RESETTING
 *              - 0x08 - NETWORK TEST
 *              - 0x09 - OFFLINE REQUEST
 *              - 0x0A - PSEUDO ONLINE
 *              - 0x0B - RESETTING MODEM
 *              - 0xFF - Unknown
 *
 *  @param  systemMode
 *          - System Acquisition Mode.
 *              - 0x00 - No service
 *              - 0x01 - AMPS
 *              - 0x02 - CDMA
 *              - 0x03 - GSM
 *              - 0x04 - HDR
 *              - 0x05 - WCDMA
 *              - 0x06 - GPS
 *              - 0x08 - WLAN
 *              - 0x09 - LTE
 *              - 0xFF - Unknown
 *
 *  @param  imsRegState
 *          - IMS Registration State.
 *              - 0x00 - NO SRV
 *              - 0x01 - IN PROG
 *              - 0x02 - FAILED
 *              - 0x03 - LIMITED
 *              - 0x04 - FULL SRV
 *              - 0xFF - Unknown
 *
 *  @param  psState
 *          - PS Attach State.
 *              - 0x00 - Attached
 *              - 0x01 - Detached
 *              - 0xFF - Unknown
 *
 */
typedef struct
{
    int8_t temperature;
    uint8_t modemMode;
    uint8_t systemMode;
    uint8_t imsRegState;
    uint8_t psState;
} nas_CommInfo;

/**
 *  Structure for storing the LTE information for the device.
 *
 *  @param  band
 *          - LTE Band
 *              - 1 ~ 40 (Band in decimal)
 *              - 0xFF - Invalid
 *
 *
 *  @param  bandwidth
 *          - BandWidth.
 *              - 0x00 - 1.4 MHz
 *              - 0x01 - 3 MHz
 *              - 0x02 - 5 MHz
 *              - 0x03 - 10 MHz
 *              - 0x04 - 15 MHz
 *              - 0x05 - 20 MHz
 *              - 0x06 - Invalid
 *              - 0xFF - Unknown
 *
 *  @param  RXChan
 *          - RX channel number in decimal
 *              - 0xFFFF - Not Available
 *
 *  @param  TXChan
 *          - TX channel number in decimal
 *              - 0xFFFF - Not Available
 *
 *  @param  emmState
 *          - EMM State.
 *              - 0x00 - Deregistered
 *              - 0x01 - Reg Initiated
 *              - 0x02 - Registered
 *              - 0x03 - TAU Initiated
 *              - 0x04 - SR Initiated
 *              - 0x05 - Dereg Initiated
 *              - 0x06 - Invalid
 *              - 0xFF - Unknown
 *
 *  @param  emmSubState
 *          - EMM Sub State.
 *              - 0xFF - NOT Applicable
 *          - When EMM_state is 0x00:
 *              - 0x00 - No IMSI
 *              - 0x01 - PLMN Search
 *              - 0x02 - Attach Needed
 *              - 0x03 - No Cell
 *              - 0x04 - Attaching
 *              - 0x05 - Normal Service
 *              - 0x06 - Limited Service
 *              - 0x07 - Waiting for PDN
 *          - When EMM_state is 0x01:
 *              - 0x00 - Waiting for NW
 *              - 0x01 - Waiting for ESM
 *          - When EMM_state is 0x02:
 *              - 0x00 - Normal Service
 *              - 0x01 - Update Needed
 *              - 0x02 - Attempt Update
 *              - 0x03 - No Cell
 *              - 0x04 - PLMN Search
 *              - 0x05 - Limited Service
 *              - 0x06 - MM Update
 *              - 0x07 - IMSI Detach
 *              - 0x08 - Waiting for ESM
 *
 *  @param  emmConnState
 *          - EMM Connected Mode State.
 *              - 0x00 - RRC Idle
 *              - 0x01 - Waiting RRC Cfm
 *              - 0x02 - RRC Connected
 *              - 0x03 - RRC Releasing
 *              - 0xFF - Unknown
 *
 */
typedef struct
{
    uint8_t band;
    uint8_t bandwidth;
    uint16_t RXChan;
    uint16_t TXChan;
    uint8_t emmState;
    uint8_t emmSubState;
    uint8_t emmConnState;
} nas_LTEInfo;

/**
 *  Structure for storing the SLQS Nas Swi Modem Status response parameters.
 *
 *  @param  commonInfo (mandatory)
 *          - See \ref CommInfo for more information
 *
 *
 *  @param  pLTEInfo (optional)
 *          - See \ref LTEInfo for more information
 *
 */
typedef struct
{
    nas_CommInfo commonInfo;
    nas_LTEInfo  *pLTEInfo;
} unpack_nas_SLQSNasSwiModemStatus_t;


/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSNasSwiModemStatus(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasSwiModemStatus(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSNasSwiModemStatus_t *pOutput
        );


/**
 * This structure contains the Serving System parameters
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param regState
 *        - Registration state - Registration state of the mobile
 *        - Values:
 *            - 0 - Not Registered; mobile is not currently searching
 *                  for a new network to provide service
 *            - 1 - Registered with a network
 *            - 2 - Not registered, but mobile is currently searching
 *                  for a new network to provide service
 *            - 3 - Registration denied by visible network
 *            - 4 - Registration state is unknown
 *
 * @param csAttachState
 *        - CS Attach State - Circuit-switched domain attach state of the mobile
 *        - Values:
 *            - 0 - Unknown or not applicable
 *            - 1 - Attached
 *            - 2 - Detached
 *
 * @param psAttachState
 *        - PS Attach State - Packet-switched domain attach state of the mobile
 *        - Values:
 *            - 0 - Unknown or not applicable
 *            - 1 - Attached
 *            - 2 - Detached
 *
 * @param selNetwork
 *        - Selected Network - Type of selected radio access network
 *        - Values:
 *            - 0 - Unknown
 *            - 1 - 3GPP2 network
 *            - 2 - 3GPP network
 *
 * @param numRadioInterfaces
 *        - In Use Radio Interfaces Number
 *            - Number of radio interfaces currently in use
 *            - defaults to zero
 *
 * @param radioInterface
 *        - Radio Interface currently in use
 *        - Values:
 *            - 0x00 - RADIO_IF_NO_SVC - None(no service)
 *            - 0x01 - RADIO_IF_CDMA_1X - cdma2000 1X
 *            - 0x02 - RADIO_IF_CDMA_1XEVDO - cdma2000 HRPD (1xEV-DO)
 *            - 0x03 - RADIO_IF_AMPS - AMPS
 *            - 0x04 - RADIO_IF_GSM - GSM
 *            - 0x05 - RADIO_IF_UMTS - UMTS
 *            - 0x08 - RADIO_IF_LTE - LTE
 *
 */
typedef struct {
    uint8_t regState;
    uint8_t csAttachState;
    uint8_t psAttachState;
    uint8_t selNetwork;
    uint8_t numRadioInterfaces;
    uint8_t radioInterface[32];
}nas_servSystem;

/**
 * This structure contains the data services capability
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param dataCapabilitiesLen
 *        - Length of data capabilities list
 *        - Defaults to zero
 *
 * @param dataCapabilities
 *        - List of data capabilities
 *        - Values:
 *            - 0x01 - GPRS
 *            - 0x02 - EDGE
 *            - 0x03 - HSDPA
 *            - 0x04 - HSUPA
 *            - 0x05 - WCDMA
 *            - 0x06 - CDMA
 *            - 0x07 - EV-DO Rev0
 *            - 0x08 - EV-DO RevA
 *            - 0x09 - GSM
 *            - 0x0A - EV-DO Rev B
 *            - 0x0B - LTE
 *            - 0x0C - HSDPA+
 *            - 0x0D - DC-HSDPA+
 *
 */
typedef struct {
    uint8_t dataCapabilitiesLen;
    uint8_t dataCapabilities[32];
}nas_dataSrvCapabilities;

/**
 * This structure contains the current PLMN parameters
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param MCC
 *        - mobile country code
 *            - A 16 bit representation of MCC
 *            - Range 0 to 999
 *
 * @param MNC
 *        - mobile network code
 *            - A 16 bit representation of MNC
 *            - Range 0 to 999
 *
 * @param netDescrLength
 *        - Length of Network description field
 *        - Defaults to zero
 *
 * @param netDescr
 *        - Network Description
 *            - optional string containing network name or description
 *
 */
typedef struct {
    uint16_t MCC;
    uint16_t MNC;
    uint8_t netDescrLength;
    uint8_t netDescr[255];
}nas_currentPLMN;

/**
 * This structure contains the Roaming Indicator List
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param numInstances
 *        - number of sets of radio interface currently in use and roaming
 *          indicator
 *            - defaults to zero
 *
 * @param radioInterface
 *        - Radio Interface currently in use
 *        - Values:
 *            - 0x01 - RADIO_IF_CDMA_1X - cdma2000 1X
 *            - 0x02 - RADIO_IF_CDMA_1XEVDO - cdma2000 HRPD (1xEV-DO)
 *            - 0x03 - RADIO_IF_AMPS - AMPS
 *            - 0x04 - RADIO_IF_GSM - GSM
 *            - 0x05 - RADIO_IF_UMTS - UMTS
 *            - 0x08 - RADIO_IF_LTE - LTE
 *
 * @param roamIndicator
 *        - Roaming Indicator
 *        - Values:
 *            - 0x00 - Roaming
 *            - 0x01 - Home
 *
 */
typedef struct {
    uint8_t numInstances;
    uint8_t radioInterface[32];
    uint8_t roamIndicator[32];
}nas_roamIndList;

/**
 * This structure contains the 3GPP2TimeZone parameters
 *
 * @param leapSeconds
 *        - leap seconds - Number of leap seconds since the start of
 *                          CDMA system time.
 *
 * @param localTimeOffset
 *        - Local Time Offset - Offset of system time in units of 30 minutes;
 *                             the value in this field conveys as 8 bit 2's
 *                             compliment number.
 *
 * @param daylightSavings
 *        - Day Light Savings Indicator
 *            - 0x00 - OFF (daylight savings not in effect)
 *            - 0x01 - ON (daylight savings in effect)
 *
 */
typedef struct {
    uint8_t leapSeconds;
    uint8_t localTimeOffset;
    uint8_t daylightSavings;
}nas_qaQmi3Gpp2TimeZone;

/**
 * This structure contains Detailed Service information
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param srvStatus
 *        - Service status
 *        - Values:
 *            - 0x00 - No service
 *            - 0x01 - Limited service
 *            - 0x02 - Service available
 *            - 0x03 - Limited regional service
 *            - 0x04 - MS in power save or deep sleep
 *
 * @param srvCapability
 *        - System's service capability
 *        - Values:
 *            - 0x00 - No Service
 *            - 0x01 - Circuit-switched only
 *            - 0x02 - Packet-switched only
 *            - 0x03 - Circuit-switched and packet-switched
 *            - 0x04 - MS found the right system but not yet
 *                     registered/attached
 *
 * @param hdrSrvStatus
 *        - HDR service status
 *        - Values:
 *            - 0x00 - No service
 *            - 0x01 - Limited service
 *            - 0x02 - Service available
 *            - 0x03 - Limited regional service
 *            - 0x04 - MS in power save or deep sleep
 *
 * @param hdrHybrid
 *        - HDR hybrid information
 *        - Values:
 *            - 0x00 - System is not hybrid
 *            - 0x01 - System is hybrid
 *
 * @param isSysForbidden
 *        - Forbidden system information
 *        - Values:
 *            - 0x00 - System is not a forbidden system
 *            - 0x01 - System is a forbidden system
 *
 */
typedef struct {
    uint8_t srvStatus;
    uint8_t srvCapability;
    uint8_t hdrSrvStatus;
    uint8_t hdrHybrid;
    uint8_t isSysForbidden;
}nas_detailSvcInfo;

/**
 * This structure contains CDMA system information extension
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param MCC
 *        - Mobile Country Code
 *
 * @param imsi_11_12
 *        - IMSI_11_12
 */
typedef struct {
    uint16_t MCC;
    uint8_t imsi_11_12;
}nas_CDMASysInfoExt;

/**
 * This structure contains Call Barring Status.
 *  - Parameter values default to their data type's maximum unsigned value
 *    unless explicitly stated otherwise.
 *
 * @param csBarStatus
 *        - Call Barring Status for circuit-switched calls.
 *        - Values:
 *        - NAS_CELL_ACCESS_NORMAL_ONLY - Cell access is allowed for normal
 *                                        calls only
 *        - NAS_CELL_ACCESS_EMERGENCY_ONLY - Cell access is allowed for
 *                                           emergency calls only
 *        - NAS_CELL_ACCESS_NO_CALLS - Cell access is not allowed for any
 *                                     call type
 *        - NAS_CELL_ACCESS_ALL_CALLS - Cell access is allowed for all call
 *                                      types
 *        - NAS_CELL_ACCESS_UNKNOWN - Cell access type is unknown
 *
 * @param psBarStatus
 *        - Call Barring Status for packet-switched calls.
 *        - Values:
 *            - NAS_CELL_ACCESS_NORMAL_ONLY - Cell access is allowed for normal
 *                                            calls only
 *            - NAS_CELL_ACCESS_EMERGENCY_ONLY - Cell access is allowed for
 *                                               emergency calls only
 *            - NAS_CELL_ACCESS_NO_CALLS - Cell access is not allowed for any
 *                                         call type
 *            - NAS_CELL_ACCESS_ALL_CALLS - Cell access is allowed for all call
 *                                          types
 *            - NAS_CELL_ACCESS_UNKNOWN - Cell access type is unknown
 */
typedef struct {
    uint32_t csBarStatus;
    uint32_t psBarStatus;
}nas_callBarStatus;

/**
 *  @param  ServingSystem serving system info
 *  @param  RoamIndicatorVal roaming indicator value
 *  @param  DataSrvCapabilities data servcie capabilities
 *  @param  CurrentPLMN current PLMN info
 *  @param  SystemID system id
 *  @param  NetworkID network id
 *  @param  BasestationID base station id
 *  @param  BasestationLatitude base station latitude
 *  @param  BasestationLongitude base station longitude
 *  @param  RoamingIndicatorList roaming indicator list
 *  @param  DefaultRoamInd default roaming indicator
 *  @param  3Gpp2TimeZone 3Gpp2 time zone
 *  @param  pCDMA_P_Rev cdma P_Rev in use
 *  @param  3GppTimeZone 3Gpp time zone
 *  @param  GppNetworkDSTAdjustment 3GPP network daylight saving adjustment
 *  @param  Lac location area code
 *  @param  CellID 3GPP cell id
 *  @param  ConcSvcInfo 3GPP2 concurrent servcie info
 *  @param  PRLInd 3GPP2 PRL indicator
 *  @param  DTMInd DTM indicator(GSM)
 *  @param  DetailedSvcInfo detail servcie info
 *  @param  CDMASystemInfoExt extra cdma system info
 *  @param  HdrPersonality hdr personality
 *  @param  TrackAreaCode track area code
 *  @param  CallBarStatus call barring status
 */
typedef struct
{
    nas_servSystem              ServingSystem;
    uint8_t                     RoamIndicatorVal;
    nas_dataSrvCapabilities     DataSrvCapabilities;
    nas_currentPLMN             CurrentPLMN;
    uint16_t                    SystemID;
    uint16_t                    NetworkID;
    uint16_t                    BasestationID;
    uint32_t                    BasestationLatitude;
    uint32_t                    BasestationLongitude;
    nas_roamIndList             RoamingIndicatorList;
    uint8_t                     DefaultRoamInd;
    nas_qaQmi3Gpp2TimeZone      Gpp2TimeZone;
    uint8_t                     CDMA_P_Rev;
    uint8_t                     GppTimeZone;
    uint8_t                     GppNetworkDSTAdjustment;
    uint16_t                    Lac;
    uint32_t                    CellID;
    uint8_t                     ConcSvcInfo;
    uint8_t                     PRLInd;
    uint8_t                     DTMInd;
    nas_detailSvcInfo           DetailedSvcInfo;
    nas_CDMASysInfoExt          CDMASystemInfoExt;
    uint8_t                     HdrPersonality;
    uint16_t                    TrackAreaCode;
    nas_callBarStatus           CallBarStatus;
} unpack_nas_SLQSGetServingSystem_t;


/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSGetServingSystem(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSGetServingSystem(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSGetServingSystem_t *pOutput
        );


/**
 * This structure contains the Received Signal Strength Information
 *
 *  @param  rxSignalStrength
 *          - Received signal strength in dBm
 *              - For CDMA and UMTS, this indicates forward link pilotEc.
 *              - For GSM, the received signal strength.
 *              - For LTE, this indicates the total received wideband power
 *                observed by UE.
 *
 *  @param  radioIf
 *          - Radio interface technology of the signal being radio_if measured
 *              - 0x00 – RADIO_IF_NO_SVC – None (no service)
 *              - 0x01 – RADIO_IF_CDMA_1X – cdma2000® 1X
 *              - 0x02 – RADIO_IF_CDMA_1XEVDO – cdma2000 HRPD (1xEV-DO)
 *              - 0x03 – RADIO_IF_AMPS – AMPS
 *              - 0x04 – RADIO_IF_GSM – GSM
 *              - 0x05 – RADIO_IF_UMTS – UMTS
 *              - 0x08 – RADIO_IF_LTE – LTE
 *
 *  \note   First elemnet of the RSSI list always contains the current Signal
 *          strength and Radio Interface.
 */
typedef struct 
{
    int16_t rxSignalStrength;
    uint8_t  radioIf;
} nas_rxSignalStrengthListElement;

/**
 * This structure contains the ECIO Information
 *
 *  @param  ecio
 *          - ECIO value in dBm
 *
 *  @param  radioIf
 *          - Radio interface technology of the signal being measured
 *              - 0x00 – RADIO_IF_NO_SVC – None (no service)
 *              - 0x01 – RADIO_IF_CDMA_1X – cdma2000® 1X
 *              - 0x02 – RADIO_IF_CDMA_1XEVDO – cdma2000 HRPD (1xEV-DO)
 *              - 0x03 – RADIO_IF_AMPS – AMPS
 *              - 0x04 – RADIO_IF_GSM – GSM
 *              - 0x05 – RADIO_IF_UMTS – UMTS
 */
typedef struct 
{
    int16_t ecio;
    uint8_t  radioIf;
}nas_ecioListElement;

/**
 * This structure contains the Error Rate Information
 *
 *  @param  errorRate
 *          - Error rate value corresponds to the RAT that is currently
 *            registered.
 *              - For CDMA, the error rate reported is Frame Error Rate:
 *                  - Valid error rate values between 1 and 10000 are returned
 *                    to indicate percentage, e.g., a value of 300 means the
 *                    error rate is 3%
 *                  - A value of 0xFFFF indicates that the error rate is
 *                    unknown or unavailable
 *              - For HDR, the error rate reported is Packet Error Rate:
 *                  - Valid error rate values between 1 and 10000 are returned
 *                    to indicate percentage, e.g., a value of 300 means the
 *                    error rate is 3%
 *                  - A value of 0xFFFF indicates that the error rate is
 *                    unknown or unavailable
 *              - For GSM, the error rate reported is Bit Error Rate:
 *                  - Valid values are 0, 100, 200, 300, 400, 500, 600, and 700
 *                    The reported value divided by 100 gives the error rate as
 *                    an RxQual value, e.g.,a value of 300 represents an RxQual
 *                    value of 3.
 *                  - A value of 25500 indicates No Data
 *              - For WCDMA, the error rate reported is Block Error Rate (BLER):
 *                  - Valid values are 1 to 10000
 *                  - The reported value divided by 100 provides the error rate
 *                    in percentages, e.g., a value of 300 represents a BLER of
 *                    3%.
 *                  - A value of 0 indicates No Data
 *
 *  @param  radioIf
 *          - Radio interface technology of the signal being measured
 *              - 0x00 – RADIO_IF_NO_SVC – None (no service)
 *              - 0x01 – RADIO_IF_CDMA_1X – cdma2000® 1X
 *              - 0x02 – RADIO_IF_CDMA_1XEVDO – cdma2000 HRPD (1xEV-DO)
 *              - 0x03 – RADIO_IF_AMPS – AMPS
 *              - 0x04 – RADIO_IF_GSM – GSM
 *              - 0x05 – RADIO_IF_UMTS – UMTS
 */
typedef struct 
{
    uint16_t errorRate;
    uint8_t   radioIf;
} nas_errorRateListElement;

/**
 * This structure contains the RSRQ Information
 *
 *  @param  rsrq
 *          - RSRQ value in dB (signed integer value); valid range is -3
 *            to -20 (-3 means -3 dB, -20 means -20 dB)
 *
 *  @param  radioIf
 *          - Radio interface technology of the signal being measured
 *              - 0x08 – LTE
 */
typedef struct 
{
    int8_t rsrq;
    uint8_t radioIf;
} nas_rsrqInformation;

/**
 * This structure contains the LTE SNR Information
 *
 *  @param  snrlevel
 *          - LTE SNR level as a scaled integer in units of 0.1dB
 *             e.g. -16dB has a value of -160 and 24.6dB has value of 246.
 */
typedef struct
{
    int16_t snrlevel;
} nas_lteSnrinformation;

/**
 * This structure contains the LTE RSRP Information
 *
 *  @param  rsrplevel
 *          - LTE RSRP in dBm as a mesaured by L1.
 *            Range: -44 to -140(-44 means -44dBm, -140 means -140dBm).
 */
typedef struct
{
    int16_t rsrplevel;
} nas_lteRsrpinformation;


/**
 *  @param  rxSignalStrengthListLen number of elements in Receive Signal Strength List
 *  @param  rxSignalStrengthList signal strength list
 *  @param  ecioListLen number of elements in ECIO List
 *  @param  ecioList ecio list
 *  @param  Io received Io in dBm; IO is only applicable for 1xEV-DO
 *  @param  sinr SINR level; SINR is only applicable for 1xEV-DO
 *  @param  errorRateListLen number of elements in Error Rate List
 *  @param  errorRateList error rate list
 *  @param  rsrqInfo rsrq info
 *  @param  ltesnr lte snr info
 *  @param  ltersrp lte srp info
 */
typedef struct {
    uint16_t   signalStrengthReqMask; //don't use this value
    uint16_t   rxSignalStrengthListLen;
    nas_rxSignalStrengthListElement rxSignalStrengthList[18];
    uint16_t   ecioListLen;
    nas_ecioListElement ecioList[18];
    int32_t    Io;
    uint8_t    sinr;
    uint16_t   errorRateListLen;
    nas_errorRateListElement errorRateList[18];
    nas_rsrqInformation rsrqInfo;
    int16_t    ltesnr;
    int16_t    ltersrp;
} unpack_nas_SLQSGetSignalStrength_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqMask request mask for fetching extra signal info
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSGetSignalStrength(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        uint16_t reqMask
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOuput      response unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSGetSignalStrength(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SLQSGetSignalStrength_t *pOutput
        );

/**
 *  @param  rxSignalStrengthDelta RSSI delta(in dBm) at which an event report indication
 *  @param  ecioDelta ecio delta
 *  @param  ioDelta io delta
 *  @param  sinrDelta sinr delta
 *  @param  rsrqDelta rsrq delta
 *  @param  ecioThresholdListLen
 *  @param  ecioThresholdList
 *  @param  sinrThresholdListLen
 *  @param  sinrThresholdList
 *  @param  lteSnrDelta lte snr delta
 *  @param  lteRsrpDelta lte rsrp delta
 */
typedef struct
{
    uint8_t  rxSignalStrengthDelta;
    uint8_t  ecioDelta;
    uint8_t  ioDelta;
    uint8_t  sinrDelta;
    uint8_t  rsrqDelta;
    uint8_t  ecioThresholdListLen;
    int16_t  ecioThresholdList[10];
    uint8_t  sinrThresholdListLen;
    uint8_t  sinrThresholdList[5];
    uint16_t lteSnrDelta;
    uint8_t  lteRsrpDelta;
} nas_SLQSSignalStrengthsIndReq;

/**
 *  @param  bEnable 0/1 to disable/enable RSSI signal stregth indication
 *  @param  pSigIndReq parameters to control signal strength indication
 */
typedef struct
{
    uint8_t bEnable;
    nas_SLQSSignalStrengthsIndReq *pSigIndReq;
} pack_nas_SLQSSetSignalStrengthsCallback_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] pReqParam request prarmeters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSSetSignalStrengthsCallback(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSSetSignalStrengthsCallback_t *pReqParam
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSSetSignalStrengthsCallback(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] bEnable 0/1 value to disable/enable indication respectively
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SetRFInfoCallback(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        uint8_t *pBenable
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SetRFInfoCallback(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] bEnable 0/1 value to disable/enable indication respectively
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SetLURejectCallback(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        uint8_t *pBenable
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SetLURejectCallback(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 *  @param  rxSignalStrengthInfo signal strength info list
 *  @param  ecioInfo ecio info list
 *  @param  io received IO in dBm; IO is only applicable for 1xEV-DO
 *  @param  sinr SINR level
 *  @param  errorRateInfo error rate info
 *  @param  rsrqInfo rsrq info
 *  @param  lteSnrinfo lte Snr information
 *  @param  lteRsrpinfo lte rsrp info
 */
typedef struct{
    nas_rxSignalStrengthListElement rxSignalStrengthInfo;
    nas_ecioListElement             ecioInfo;
    uint32_t                   io;
    uint8_t                         sinr;
    nas_errorRateListElement        errorRateInfo;
    nas_rsrqInformation             rsrqInfo;
    nas_lteSnrinformation           lteSnrinfo;
    nas_lteRsrpinformation          lteRsrpinfo;
} nas_SLQSSignalStrengthsInformation;


/**
 *  @param TlvPresent indicating the presence of the TLV in the QMI ind
 *  @param serviceDomain service domain
 *  @param rejectCause cause of reject
 */
typedef struct 
{
    uint8_t TlvPresent;
    uint32_t serviceDomain;
    uint32_t rejectCause;
} nas_RejectReasonTlv;

/**
 *  @param TlvPresent indicating the presence of the TLV in the QMI ind
 *  @param signalStrength signal strength
 *  @param radioInterface radio interface technology of the signal being measured
 */
typedef struct 
{
    uint8_t  TlvPresent;
    int8_t  signalStrength;
    uint32_t radioInterface;
} nas_SignalStrengthTlv;

/**
 *  @param TlvPresent indicating the presence of the TLV in the QMI ind
 *  @param radioInterface radio interface technology of the signal being measured
 *  @param activeBandClass active band class
 *  @param  activeChannel active channel
 */
typedef struct 
{
    uint8_t  TlvPresent;
    uint8_t  radioInterfaceSize;
    uint32_t radioInterface[255];
    uint32_t activeBandClass[255];
    uint32_t activeChannel[255];
} nas_RFInfoTlv;

/**
 *  @param TlvPresent indicating the presence of the TLV in the QMI ind
 *  @param sSLQSSignalStrengthsInfo signal strength info
 */
typedef struct 
{
    uint8_t                                  TlvPresent;
    nas_SLQSSignalStrengthsInformation sSLQSSignalStrengthsInfo;
} nas_SLQSSignalStrengthsTlv;

/**
 *  @param  SSTlv signal strength tlv
 *  @param  RFTlv RF tlv
 *  @param  RRTlv RR tlv
 *  @param  SLQSSSTlv signal strength complete info tlv
 */
typedef struct {
    nas_SignalStrengthTlv      SSTlv;
    nas_RFInfoTlv              RFTlv;
    nas_RejectReasonTlv        RRTlv;
    nas_SLQSSignalStrengthsTlv SLQSSSTlv;
} unpack_nas_SetEventReportInd_t;

int unpack_nas_SetEventReportInd(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SetEventReportInd_t *pOutput
        );

/**
 *  @param  SCI slot cycle index
 *  @param  SCM station class mark
 *  @param  RegHomeSID register on home sid
 *  @param  RegForeignSID register on foreign sid
 *  @param  RegForeignNID register on foreign nid
 *  @param  ForceRev0 force header revision
 *  @param  CustomSCP custom SCP
 *  @param  Protocol protocol
 *  @param  Booadcast broadcast
 *  @param  Application application
 *  @param  Roaming roaming
 */
typedef struct {
    uint8_t  SCI;
    uint8_t  SCM;
    uint8_t  RegHomeSID;
    uint8_t  RegForeignSID;
    uint8_t  RegForeignNID;
    uint8_t  ForceRev0;
    uint8_t  CustomSCP;
    uint32_t Protocol;
    uint32_t Broadcast;
    uint32_t Application;
    uint32_t Roaming;
} unpack_nas_GetCDMANetworkParameters_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetCDMANetworkParameters(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput     qmi output parameters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetCDMANetworkParameters(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_GetCDMANetworkParameters_t *pOutput
        );

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetANAAAAuthenticationStatus(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp          qmi response from modem
 * @param[in]   respLen        qmi response length
 * @param[out]  authStatus     auth status
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetANAAAAuthenticationStatus(
        uint8_t *pResp,
        uint16_t respLen,
        uint32_t *pAuthStatus
        );

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_GetACCOLC(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * @param[in]   pResp          qmi response from modem
 * @param[in]   respLen        qmi response length
 * @param[out]  pAccolc        accolc
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_GetACCOLC(
        uint8_t *pResp,
        uint16_t respLen,
        uint8_t *pAccolc
        );

/**
 *  @param  spc servcie programming code
 *  @param  accolc accolc
 */
typedef struct
{
   int8_t spc[6];
   uint8_t accolc;    
}  pack_nas_SetACCOLC_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request Parameters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SetACCOLC(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SetACCOLC_t reqParam
        );

/**
 * @param[in]   pResp          qmi response from modem
 * @param[in]   respLen        qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SetACCOLC(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 *  This structure contains CDMA RSSI threshold related parameters.
 *
 *  @param  CDMARSSIThreshListLen
 *          - Length of the CDMARSSI threshold list parameter to follow
 *
 *  @param  pCDMARSSIThreshList
 *          - Array of RSSI thresholds (in units of 0.1 dBm)
 *          - maximum of 32 values.
 *          - Range for RSSI values:-105 to -21 (in dBm).
 *
 */
typedef struct
{
    uint8_t  CDMARSSIThreshListLen;
    int16_t  *pCDMARSSIThreshList;
}nas_CDMARSSIThresh;

/**
 *  This structure contains CDMA ECIO threshold related parameters.
 *
 *  @param  CDMAECIOThreshListLen
 *          - Length of the CDMA ECIO threshold list parameter to follow
 *
 *  @param  pCDMAECIOThreshList
 *          - Array of ECIO thresholds (in units of 0.1 dB)
 *          - Maximum of 32 values.
 *          Range for ECIO values: -31.5 to 0 (in dB)
 *
 */
typedef struct
{
    uint8_t  CDMAECIOThreshListLen;
    int16_t  *pCDMAECIOThreshList;
}nas_CDMAECIOThresh;

/**
 *  This structure contains HDR RSSI threshold related parameters.
 *
 *  @param  HDRRSSIThreshListLen
 *          - Length of the HDR RSSI threshold list parameter to follow
 *
 *  @param  pHDRRSSIThreshList
 *          - Array of RSSI thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values.
 *          - Range for RSSI values: -118 to -13 (in dBm).
 *
 */
typedef struct
{
    uint8_t  HDRRSSIThreshListLen;
    int16_t *pHDRRSSIThreshList;
}nas_HDRRSSIThresh;

/**
 *  This structure contains HDR ECIO threshold related parameters.
 *
 *  @param  HDRECIOThreshListLen
 *          - Length of the HDR ECIO threshold list parameter to follow
 *
 *  @param  pHDRECIOThreshList
 *          - Array of ECIO thresholds (in units of 0.1 dB)
 *          - Maximum of 32 values
 *          - Range for ECIO values: -31.5 to 0 (in dB).
 */
typedef struct
{
    uint8_t  HDRECIOThreshListLen;
    int16_t  *pHDRECIOThreshList;
}nas_HDRECIOThresh;

/**
 *  This structure contains HDR SINR threshold related parameters.
 *
 *  @param  HDRSINRThreshListLen
 *          - Length of the HDR ECIO threshold list parameter to follow
 *
 *  @param  pHDRSINRThreshList
 *          - Array of SINR level thresholds (in units of 1)
 *          - maximum of 32 values.
 *          - Valid levels are 0 to 8
 *            - 0x00 - SINR_LEVEL_0 is -9 dB
 *            - 0x01 - SINR_LEVEL_1 is -6 dB
 *            - 0x02 - SINR_LEVEL_2 is -4.5 dB
 *            - 0x03 - SINR_LEVEL_3 is -3 dB
 *            - 0x04 - SINR_LEVEL_4 is -2 dB
 *            - 0x05 - SINR_LEVEL_5 is +1 dB
 *            - 0x06 - SINR_LEVEL_6 is +3 dB
 *            - 0x07 - SINR_LEVEL_7 is +6 dB
 *            - 0x08 - SINR_LEVEL_8 is +9 dB
*/
typedef struct
{
    uint8_t  HDRSINRThreshListLen;
    uint16_t  *pHDRSINRThreshList;
}nas_HDRSINRThreshold;

/**
 *  This structure contains HDR IO threshold related parameters.
 *
 *  @param  HDRIOThreshListLen
 *          - Length of the HDR IO threshold list parameter to follow
 *
 *  @param  pHDRIOThreshList
 *          - Array of IO thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values
 *          - Range for IO values: -128 to -13 (in dBm).
 */

typedef struct
{
    uint8_t  HDRIOThreshListLen;
    int16_t  *pHDRIOThreshList;
}nas_HDRIOThresh;

/**
 *  This structure contains GSM RSSI threshold related parameters.
 *
 *  @param  GSMRSSIThreshListLen
 *          - Length of the GSM RSSI threshold list parameter to follow
 *
 *  @param  pGSMRSSIThreshList
 *          - Array of RSSI thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values
 *          - Range for RSSI values: -111 to -48 (in dBm)
 */

typedef struct
{
    uint8_t  GSMRSSIThreshListLen;
    int16_t  *pGSMRSSIThreshList;
}nas_GSMRSSIThresh;

/**
 *  This structure contains WCDMA RSSI threshold related parameters.
 *
 *  @param  WCDMARSSIThreshListLen
 *          - Length of the WCDMA RSSI threshold list parameter to follow
 *
 *  @param  pWCDMARSSIThreshList
 *          - Array of RSSI thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values.
 *          - Range for RSSI values: -121 to 0 (in dBm)
 */
typedef struct
{
    uint8_t  WCDMARSSIThreshListLen;
    int16_t  *pWCDMARSSIThreshList;
}nas_WCDMARSSIThresh;

/**
 *  This structure contains WCDMA ECIO threshold related parameters.
 *
 *  @param  WCDMAECIOThreshListLen
 *          - Length of the WCDMA ECIO threshold list parameter to follow
 *
 *  @param  pWCDMAECIOThreshList
 *          - Array of ECIO thresholds (in units of 0.1 dB)
 *          - Maximum of 32 values
 *          - Range for ECIO values: -31.5 to 0 (in dB)
 */
typedef struct
{
    uint8_t  WCDMAECIOThreshListLen;
    int16_t  *pWCDMAECIOThreshList;
}nas_WCDMAECIOThresh;

/**
 *  This structure contains LTE RSSI threshold related parameters.
 *
 *  @param  LTERSSIThreshListLen
 *          - Length of the LTE RSSI threshold list parameter to follow
 *
 *  @param  pLTERSSIThreshList
 *          - Array of RSSI thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values.
 *          - Range for RSSI values: -120 to 0 (in dBm)
 */
typedef struct
{
    uint8_t  LTERSSIThreshListLen;
    int16_t  *pLTERSSIThreshList;
}nas_LTERSSIThresh;

/**
 *  This structure contains LTE SNR threshold related parameters.
 *
 *  @param  LTESNRThreshListLen
 *          - Length of the LTE SNR threshold list parameter to follow
 *
 *  @param  pLTESNRThreshList
 *          - Array of SNR thresholds (in units of 0.1 dB)
 *          - Maximum of 32 values
 *          - Range for SNR values: -20 to 30 (in dB).
 */
typedef struct
{
    uint8_t  LTESNRThreshListLen;
    int16_t  *pLTESNRThreshList;
}nas_LTESNRThreshold;

/**
 *  This structure contains LTE RSRQ threshold related parameters.
 *
 *  @param  LTERSRQThreshListLen
 *          - Length of the LTE RSRQ threshold list parameter to follow
 *
 *  @param  pLTERSRQThreshList
 *          - Array of RSRQ thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values.
 *          - Range for RSRQ values: -20 to -3 (in dBm)
 */
typedef struct
{
    uint8_t  LTERSRQThreshListLen;
    int16_t  *pLTERSRQThreshList;
}nas_LTERSRQThresh;

/**
 *  This structure contains LTE RSRP threshold related parameters.
 *
 *  @param  LTERSRPThreshListLen
 *          - Length of the LTE RSRP threshold list parameter to follow
 *
 *  @param  pLTERSRPThreshList
 *          - Array of RSRP thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values
 *          - Range for RSRP values: -140 to -44 (in dBm).
 */
typedef struct
{
    uint8_t  LTERSRPThreshListLen;
    int16_t  *pLTERSRPThreshList;
}nas_LTERSRPThresh;

/**
 *  This structure contains LTE RSRP threshold related parameters.
 *
 *  @param  rptRate
 *          - Rate on how often the LTE signal must be checked for reporting
 *          Values
 *          - 0 - Report using the default configuration
 *          - 1 - Report every 1 sec
 *          - 2 - Report every 2 sec
 *          - 3 - Report every 3 sec
 *          - 4 - Report every 4 sec
 *          - 5 - Report every 5 sec
 *
 *  @param  avgPeriod
 *          - Averaging period to be used for the LTE signal.
 *          - Values
 *           - 0 - Average using the default configuration
 *           - 1 - Average over 1 sec
 *           - 2 - Average over 2 sec
 *           - 3 - Average over 3 sec
 *           - 4 - Average over 4 sec
 *           - 5 - Average over 5 sec
 *           - 6 - Average over 6 sec
 *           - 7 - Average over 7 sec
 *           - 8 - Average over 8 sec
 *           - 9 - Average over 9 sec
 *           - 10 - Average over 10 sec
 */
typedef struct
{
    uint8_t  rptRate;
    uint8_t  avgPeriod;
}nas_LTESigRptConfig;

/**
 *  This structure contains TDSCDMA RSCP threshold related parameters.
 *
 *  @param  TDSCDMARSCPThreshListLen
 *          - Length of the TDSCDMA RSCP threshold list parameter to follow
 *
 *  @param  pTDSCDMARSCPThreshList
 *          - Array of RSCP thresholds (in units of 0.1 dBm)
 *          - Maximum of 32 values
 *          - Range for RSCP values: -120 to -25 (in dBm).
 */
typedef struct
{
    uint8_t  TDSCDMARSCPThreshListLen;
    int16_t  *pTDSCDMARSCPThreshList;
}nas_TDSCDMARSCPThresh;

/**
 *  This structure contains TDSCDMA RSSI threshold related parameters.
 *
 *  @param  TDSCDMARSSIThreshListLen
 *          - Length of the TDSCDMA RSSI threshold list parameter to follow
 *
 *  @param  pTDSCDMARSSIThreshList
 *          - Array of RSSI thresholds (in dBm) used by TD-SCDMA
 *          - Maximum of 32 values.
 */
typedef struct
{
    uint8_t   TDSCDMARSSIThreshListLen;
    float  *pTDSCDMARSSIThreshList;
}nas_TDSCDMARSSIThresh;

/**
 *  This structure contains TDSCDMA ECIO threshold related parameters.
 *
 *  @param  TDSCDMAECIOThreshListLen
 *          - Length of the TDSCDMA ECIO threshold list parameter to follow
 *
 *  @param  pTDSCDMAECIOThreshList
 *          - Array of ECIO thresholds (in dB) used by TD-SCDMA
 *          - Maximum of 32 values.
 */
typedef struct
{
    uint8_t   TDSCDMAECIOThreshListLen;
    float  *pTDSCDMAECIOThreshList;
}nas_TDSCDMAECIOThresh;

/**
 *  This structure contains TDSCDMA SINR threshold related parameters.
 *
 *  @param  TDSCDMASINRThreshListLen
 *          - Length of the TDSCDMA SINR threshold list parameter to follow
 *
 *  @param  pTDSCDMASINRThreshList
 *          - Array of SINR thresholds (in dB) used by TD-SCDMA
 *          - Maximum of 32 values
 */
typedef struct
{
    uint8_t   TDSCDMASINRThreshListLen;
    float  *pTDSCDMASINRThreshList;
}nas_TDSCDMASINRThresh;

/**
 *
 *  @param  pCDMARSSIThresh
 *          - CDMA RSSI threshold List
 *
 *  @param  pCDMARSSIDelta
 *          - RSSI delta (in units of 0.1 dBm).
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pCDMAECIOThresh
 *          - CDMA ECIO Threshold List
 *
 *  @param  pCDMAECIODelta
 *          - ECIO delta (in units of 0.1 dB).
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pHDRRSSIThresh
 *          - HDR RSSI Threshold List
 *
 *  @param  pHDRRSSIDelta
 *          - RSSI delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pHDRECIOThresh
 *          - HDR ECIO Threshold List
 *
 *  @param  pHDRECIODelta
 *          - ECIO delta (in units of 0.1 dB)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pHDRSINRThresh
 *          - HDR SINR Threshold List
 *
 *  @param  pHDRSINRDelta
 *          - SINR delta (in units of 1 SINR level)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pHDRIOThresh
 *          - HDR IO Threshold List
 *
 *  @param  pHDRIODelta
 *          - IO delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pGSMRSSIThresh
 *          - GSM RSSI Threshold List
 *          - See \ref GSMRSSIThresh for more details
 *
 *  @param  pGSMRSSIDelta
 *          - RSSI delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pWCDMARSSIThresh
 *          - WCDMA RSSI Threshold List
 *          - See \ref WCDMARSSIThresh for more details
 *
 *  @param  pWCDMARSSIDelta
 *          - RSSI delta (in units of 0.1 dBm).
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pWCDMAECIOThresh
 *          - WCDMA ECIO Threshold List
 *
 *  @param  pWCDMAECIODelta
 *          - ECIO delta (in units of 0.1 dB)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pLTERSSIThresh
 *          - LTE RSSI Threshold List
 *
 *  @param  pLTERSSIDelta
 *          - RSSI delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pLTESNRThresh
 *          - LTE SNR Threshold List
 *
 *  @param  pLTESNRDelta
 *          - SNR delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pLTERSRQThresh
 *          - LTE RSRQ Threshold List
 *
 *  @param  pLTERSRQDelta
 *          - RSRQ delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pLTERSRPThresh
 *          - LTE RSRP Threshold List
 *
 *  @param  pLTERSRPDelta
 *          - RSRP delta (in units of 0.1 dBm).
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pLTESigRptConfig
 *          - LTE Signal Report Config
 *
 *  @param  pTDSCDMARSCPThresh
 *          - TDSCDMA RSCP Threshold List
 *
 *  @param  pTDSCDMARSCPDelta
 *          - RSCP delta (in units of 0.1 dBm)
 *          - A value of 0 is rejected with a QMI_ERR_INVALID_ARG error.
 *
 *  @param  pTDSCDMARSSIThresh
 *          - TDSCDMA RSSI Threshold List
 *
 *  @param  pTDSCDMARSSIDelta
 *          - RSSI delta (in dBm) used by TD-SCDMA.
 *
 *  @param  pTDSCDMAECIOThresh
 *          - TDSCDMA ECIO Threshold List
 *
 *  @param  pTDSCDMAECIODelta
 *          - ECIO delta (in dB) used by TD-SCDMA
 *
 *  @param  pTDSCDMASINRThresh
 *          - TDSCDMA SINR Threshold List
 *
 *  @param  pTDSCDMASINRDelta
 *          - SINR delta (in dB) used by TD-SCDMA.
 *
 */
typedef struct
{
    nas_CDMARSSIThresh    *pCDMARSSIThresh;
    uint16_t              *pCDMARSSIDelta;
    nas_CDMAECIOThresh    *pCDMAECIOThresh;
    uint16_t              *pCDMAECIODelta;
    nas_HDRRSSIThresh     *pHDRRSSIThresh;
    uint16_t              *pHDRRSSIDelta;
    nas_HDRECIOThresh     *pHDRECIOThresh;
    uint16_t              *pHDRECIODelta;
    nas_HDRSINRThreshold  *pHDRSINRThresh;
    uint16_t              *pHDRSINRDelta;
    nas_HDRIOThresh       *pHDRIOThresh;
    uint16_t              *pHDRIODelta;
    nas_GSMRSSIThresh     *pGSMRSSIThresh;
    uint16_t              *pGSMRSSIDelta;
    nas_WCDMARSSIThresh   *pWCDMARSSIThresh;
    uint16_t              *pWCDMARSSIDelta;
    nas_WCDMAECIOThresh   *pWCDMAECIOThresh;
    uint16_t              *pWCDMAECIODelta;
    nas_LTERSSIThresh     *pLTERSSIThresh;
    uint16_t              *pLTERSSIDelta;
    nas_LTESNRThreshold   *pLTESNRThresh;
    uint16_t              *pLTESNRDelta;
    nas_LTERSRQThresh     *pLTERSRQThresh;
    uint16_t              *pLTERSRQDelta;
    nas_LTERSRPThresh     *pLTERSRPThresh;
    uint16_t              *pLTERSRPDelta;
    nas_LTESigRptConfig   *pLTESigRptConfig;
    nas_TDSCDMARSCPThresh *pTDSCDMARSCPThresh;
    uint16_t              *pTDSCDMARSCPDelta;
    nas_TDSCDMARSSIThresh *pTDSCDMARSSIThresh;
    float             *pTDSCDMARSSIDelta;
    nas_TDSCDMAECIOThresh *pTDSCDMAECIOThresh;
    float             *pTDSCDMAECIODelta;
    nas_TDSCDMASINRThresh *pTDSCDMASINRThresh;
    float             *pTDSCDMASINRDelta;
} pack_nas_SLQSNasConfigSigInfo2_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  reqParam request Parameters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSNasConfigSigInfo2(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSNasConfigSigInfo2_t *pReqParam
        );

/**
 * @param[in]   pResp          qmi response from modem
 * @param[in]   respLen        qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasConfigSigInfo2(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 * @param dataCapsSize Number of Data Capabilities
 * @param dataCaps Data Capabilities
 */
typedef struct
{
   uint8_t dataCapsSize;
   uint8_t dataCaps[255];
} unpack_nas_SetDataCapabilitiesCallback_ind_t;

/**
 * Data Capabilities indication unpack
 * @param[in]   pResp       qmi indication from modem
 * @param[in]   respLen     qmi indication length
 * @param[out]  pOuput      indication unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SetDataCapabilitiesCallback_ind(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SetDataCapabilitiesCallback_ind_t *pOutput
        );

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 */

int pack_nas_GetNetworkPreference(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen
);

/**
 *  \param  TechnologyPref[OUT]
 *          - Bitmask representing the radio technology preference set.
 *          - No bits set indicates to the device to automatically
 *            determine the technology to use
 *          - Values:
 *              - Bit 0 - Technology is 3GPP2
 *              - Bit 1 - Technology is 3GPP
 *          - Any combination of the following may be returned:
 *              - Bit 2 - Analog - AMPS if 3GPP2, GSM if 3GPP
 *              - Bit 3 - Digital - CDMA if 3GPP2, WCDMA if 3GPP
 *              - Bit 4 - HDR
 *              - Bit 5 - LTE
 *              - Bits 6 to 15 - Reserved
 *
 *  \param  Duration[OUT]
 *          - Duration of active preference
 *              - 0 - Permanent
 *              - 1 - Power cycle
 *              - 2 - Until the end of the next call or a power cycle
 *              - 3 - Until the end of the next call, a specified time, or a power cycle
 *              - 4 to 6 - Until the end of the next call
 *
 *  \param  PersistentTechnologyPref[OUT]
 *          - Bit field representing persistent radio technology preference
 *              - Same representation as the pTechnologyPref parameter
 * \param Tlvresult 
            - unpack result
 */
typedef struct
{
    uint32_t ActiveTechPref;
    uint32_t Duration;
    uint32_t PersistentTechPref;
    uint16_t Tlvresult;
} unpack_nas_GetNetworkPreference_t;

int unpack_nas_GetNetworkPreference(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_GetNetworkPreference_t *pOutput
);

/**
 *  \param  TechnologyPref[IN]
 *          - Bitmask representing the radio technology preference set.
 *          - No bits set indicates to the device to automatically
 *            determine the technology to use
 *          - Values:
 *              - Bit 0 - Technology is 3GPP2
 *              - Bit 1 - Technology is 3GPP
 *          - Any combination of the following may be returned:
 *              - Bit 2 - Analog - AMPS if 3GPP2, GSM if 3GPP
 *              - Bit 3 - Digital - CDMA if 3GPP2, WCDMA if 3GPP
 *              - Bit 4 - HDR
 *              - Bit 5 - LTE
 *              - Bits 6 to 15 - Reserved
 *
 *  \param  Duration[IN]
 *          - Duration of active preference
 *              - 0 - Permanent
 *              - 1 - Power cycle
 *              - 2 - Until the end of the next call or a power cycle
 *              - 3 - Until the end of the next call, a specified time, or a power cycle
 *              - 4 to 6 - Until the end of the next call
 *
 * \param Tlvresult 
            - pack result
 */
typedef struct
{
    uint32_t TechnologyPref;
    uint32_t Duration;
    uint16_t Tlvresult;
} pack_nas_SetNetworkPreference_t;


/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] pack default prototype
 */

int pack_nas_SetNetworkPreference(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_nas_SetNetworkPreference_t *reqArg
);

/**
 *  \param  TechnologyPref[OUT]
 *          - Bitmask representing the radio technology preference set.
 *          - No bits set indicates to the device to automatically
 *            determine the technology to use
 *          - Values:
 *              - Bit 0 - Technology is 3GPP2
 *              - Bit 1 - Technology is 3GPP
 *          - Any combination of the following may be returned:
 *              - Bit 2 - Analog - AMPS if 3GPP2, GSM if 3GPP
 *              - Bit 3 - Digital - CDMA if 3GPP2, WCDMA if 3GPP
 *              - Bit 4 - HDR
 *              - Bit 5 - LTE
 *              - Bits 6 to 15 - Reserved
 *
 *  \param  Duration[OUT]
 *          - Duration of active preference
 *              - 0 - Permanent
 *              - 1 - Power cycle
 *              - 2 - Until the end of the next call or a power cycle
 *              - 3 - Until the end of the next call, a specified time, or a power cycle
 *              - 4 to 6 - Until the end of the next call
 *
 *  \param  PersistentTechnologyPref[OUT]
 *          - Bit field representing persistent radio technology preference
 *              - Same representation as the pTechnologyPref parameter
 * \param Tlvresult 
            - unpack result
 */
typedef struct
{
    uint16_t Tlvresult;
} unpack_nas_SetNetworkPreference_t;

int unpack_nas_SetNetworkPreference(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SetNetworkPreference_t *pOutput
);


/**
 *  @param  roaming
 *          - Roaming Indication\n
 *              - 0  - Roaming\n
 *              - 1  - Home\n
 *              - 2  - Roaming partner\n
 *              - >2 - Operator defined values
 */
typedef struct
{
   uint8_t roaming;
} unpack_nas_SetRoamingIndicatorCallback_ind_t;

/**
 * Roaming indication unpack
 * @param[in]   pResp       qmi indication from modem
 * @param[in]   respLen     qmi indication length
 * @param[out]  pOuput      indication unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SetRoamingIndicatorCallback_ind(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_nas_SetRoamingIndicatorCallback_ind_t *pOutput
        );


#define  NAS_SERVING_SYSTEM_INFO_MAX_RADIO_INTERFACE_LIST 255
/**
 *  This structure will hold the serving system parameters information
 *
 *  \param  registrationState - Registration state of the mobile
 *          - 0 - QMI_NAS_NOT_REGISTERED  Not registered;mobile is not
 *                currently searching for a new network to provide service
 *          - 1 - QMI_NAS_REGISTERED  Registered with a network
 *          - 2 - QMI_NAS_NOT_REGISTERED_SEARCHING  Not registered, but mobile
 *                is currently searching for a new network to provide service
 *          - 3 - QMI_NAS_REGISTRATION_DENIED Registration denied by the
 *                visible network
 *          - 4 - QMI_NAS_REGISTRATION_UNKNOWN Registration state is unknown
 *
 *  \param  csAttachState  - Circuit Switch domain attach state of the mobile
 *          - 0 - Unknown or not applicable
 *          - 1 - Attached
 *          - 2 - Detached
 *
 *  \param  psAttachState  - Packet domain attach state of the mobile
 *          - 0 - Unknown or not applicable
 *          - 1 - Attached
 *          - 2 - Detached
 *
 *  \param  selectedNetwork - Type of selected radio access network
 *          - 0x00 - Unknown
 *          - 0x01 - 3GPP2 network
 *          - 0x02 - 3GPP network
 *
 *  \param  radioInterfaceNo - Number of radio interfaces currently in use;
 *                             this  indicates how many radio_if identifiers
 *                             follow this field
 *
 *  \param  radioInterfaceList - Radio interface currently in use
 *                               (each is 1 byte)
 *          - 0x00 - None (no service)
 *          - 0x01 - cdma2000 1X
 *          - 0x02 - cdma2000 HRPD (1xEV-DO)
 *          - 0x03 - AMPS
 *          - 0x04 - GSM
 *          - 0x05 - UMTS
 *          - 0x08 - LTE
 *
 *  \param  hdrPersonality - HDR personality information (valid only for EVDO)
 *          - 0x00 - Unknown
 *          - 0x01 - HRPD
 *          - 0x02 - eHRPD
 *
 *  Note:   None
 */
typedef struct 
{
    uint8_t registrationState;
    uint8_t csAttachState;
    uint8_t psAttachState;
    uint8_t selectedNetwork;
    uint8_t radioInterfaceNo;
    uint8_t radioInterfaceList[NAS_SERVING_SYSTEM_INFO_MAX_RADIO_INTERFACE_LIST];
    uint8_t hdrPersonality;
}NASServingSystemInfo;


/**
 *  @param  SSInfo
 *          - Serving system parameters information\n
 *              - See \ref NASServingSystemInfo for more details
 *  @param Tlvresult 
 *          - unpack result
 */
typedef struct
{
   NASServingSystemInfo SSInfo;
   uint16_t Tlvresult;
} unpack_nas_SetServingSystemCallback_ind_t;

int unpack_nas_SetServingSystemCallback_ind(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SetServingSystemCallback_ind_t *pOutput
);


/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 */

int pack_nas_SlqsGetLTECphyCAInfo(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen
);

/*
 * NAS LTE CPHY Scell States.
 */
typedef enum {
    eLIBPACK_NAS_LTE_CPHY_SCELL_STATE_DECONFIGURED=0x00,
    eLIBPACK_NAS_LTE_CPHY_SCELL_STATE_CONFIGURED_DEACTIVATED=0x01,
    eLIBPACK_NAS_LTE_CPHY_SCELL_STATE_CONFIGURED_ACTIVATED=0x02,
}LIBPACK_NAS_LTE_CPHY_SCELL_STATE;

/*
 * NAS LTE CPHY Downlink Values.
 */
typedef enum {
    eLIBPACK_NAS_LTE_CPHY_CA_BW_NRB_6=0x00,
    eLIBPACK_NAS_LTE_CPHY_CA_BW_NRB_15=0x01,
    eLIBPACK_NAS_LTE_CPHY_CA_BW_NRB_25=0x02,
    eLIBPACK_NAS_LTE_CPHY_CA_BW_NRB_50=0x03,
    eLIBPACK_NAS_LTE_CPHY_CA_BW_NRB_75=0x04,
    eLIBPACK_NAS_LTE_CPHY_CA_BW_NRB_100=0x05,
}LIBPACK_NAS_LTE_CPHY_CA_BW_NRB;


/**
 * This structure contains the parameters for Physical Carrier
 * aggregation of Scell Indeicator Type.
 *
 *  \param  pci
 *          - Physical cell ID of the SCell Range.
 *          - Range for ID values: 0 to 503.
 *
 *  \param  freq
 *          - Frequency of the absolute cell Range.
 *          - Range for ID values: 0 to 65535.
 *
 *  \param  scell_state
 *          - Scell state Values.
 *          - See \ref NAS_LTE_CPHY_SCELL_STATE for more information.
 *
 *  \param  TlvPresent
 *          - Tlv Present.
 */

typedef struct{
    uint32_t pci;
    uint32_t freq;
    LIBPACK_NAS_LTE_CPHY_SCELL_STATE scell_state;
    uint8_t TlvPresent;
}NASPhyCaAggScellIndType;

/**
 * This structure contains the parameters for Physical Carrier
 * aggregation Downlink Bandwidth of Scell.
 *
 *  \param  dl_bw_value
 *          - Downlink Bandwidth Values.
 *          - See \ref NAS_LTE_CPHY_CA_BW_NRB for more information.
 *
 */
typedef struct{
    LIBPACK_NAS_LTE_CPHY_CA_BW_NRB dl_bw_value;
    uint8_t TlvPresent;
}NASPhyCaAggScellDlBw;

/**
 * This structure contains the parameters for Physical Carrier
 * aggregation of Scell Information.
 *
 *  \param  pci
 *          - Physical cell ID of the SCell Range.
 *          - Range for ID values: 0 to 503.
 *
 *  \param  freq
 *          - Frequency of the absolute cell Range.
 *          - Range for ID values: 0 to 65535.
 *
 *  \param  dl_bw_value
 *          - Downlink Bandwidth Values.
 *          - See \ref NAS_LTE_CPHY_CA_BW_NRB for more information.
 *
 *  \param  iLTEbandValue
 *          - Band value.
 *          - Range for LTE Band class 120 to 160.
 *
 *
 *  \param  scell_state
 *          - Scell state Values.
 *          - See \ref NAS_LTE_CPHY_SCELL_STATE for more information.
 *
 *  \param  TlvPresent
 *          - Tlv Present.
 */
typedef struct{
    uint32_t pci;
    uint32_t freq;
    LIBPACK_NAS_LTE_CPHY_CA_BW_NRB dl_bw_value;
    uint32_t iLTEbandValue;
    LIBPACK_NAS_LTE_CPHY_SCELL_STATE scell_state;
    uint8_t TlvPresent;
}NASPhyCaAggScellInfo;

/**
 * This structure contains the parameters for Physical Carrier
 * aggregation of Pcell Information.
 *
 *  \param  pci
 *          - Physical cell ID of the SCell Range.
 *          - Range for ID values: 0 to 503.
 *
 *  \param  freq
 *          - Frequency of the absolute cell Range.
 *          - Range for ID values: 0 to 65535.
 *
 *  \param  dl_bw_value
 *          - Downlink Bandwidth Values.
 *          - See \ref NAS_LTE_CPHY_CA_BW_NRB for more information.
 *
 *  \param  scell_state
 *          - Scell state Values.
 *          - See \ref NAS_LTE_CPHY_SCELL_STATE for more information.
 *
 *  \param  TlvPresent
 *          - Tlv Present.
 */
typedef struct{
    uint32_t pci;
    uint32_t freq;
    LIBPACK_NAS_LTE_CPHY_CA_BW_NRB dl_bw_value;
    uint32_t iLTEbandValue;
    uint8_t TlvPresent;
}NASPhyCaAggPcellInfo;

/**
 * This structure contains the parameters for Physical Carrier
 * aggregation of Scell Index.
 *
 *  \param  scell_idx
 *          - Physical cell ID of the SCell Range.
 *          - Range for ID values: 0 to 503.
 *
 *  \param  TlvPresent
 *          - Tlv Present.
 */
typedef struct{
    uint8_t scell_idx;
    uint8_t TlvPresent;
}NASPhyCaAggScellIndex;


typedef struct
{
    NASPhyCaAggScellIndType    PhyCaAggScellIndType;
    NASPhyCaAggScellDlBw       PhyCaAggScellDlBw;
    NASPhyCaAggScellInfo       PhyCaAggScellInfo;
    NASPhyCaAggPcellInfo       PhyCaAggPcellInfo;
    NASPhyCaAggScellIndex      PhyCaAggScellIndex;
} NasGetLTECphyCaInfo;

/**
 *  @param  LTECphyCa
 *          - Carrier aggregation event information\n
 *              - See \ref NasGetLTECphyCaInfo for more details
 *  @param Tlvresult 
 *          - unpack result
 */
typedef struct
{
   NasGetLTECphyCaInfo LTECphyCaInfo;
   uint16_t Tlvresult;
} unpack_nas_SlqsGetLTECphyCAInfo_t;

int unpack_nas_SlqsGetLTECphyCAInfo(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SlqsGetLTECphyCAInfo_t *pOutput
);

/*
 * Name:    NASEmergencyModeTlv
 *
 * Purpose: Structure used to store Emergency Mode TLV Value.
 *
 * Members: TlvPresent - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          EmerMode   - Emergency Mode Value
 */
typedef struct 
{
    uint8_t TlvPresent;
    uint8_t EmerMode;
}NASEmergencyModeTlv;

/*
 * Name:    NASModePreferenceTlv
 *
 * Purpose: Structure used to store Mode Preference TLV Value.
 *
 * Members: TlvPresent - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          ModePref   - Mode Preference Value
 */
typedef struct
{
    uint8_t TlvPresent;
    uint16_t ModePref;
}NASModePreferenceTlv;

/*
 * Name:    NASBandPreferenceTlv
 *
 * Purpose: Structure used to store all Band Preference TLV Value.
 *
 * Members: TlvPresent - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          band_pref  - Roaming Indication Value
 */
typedef struct 
{
    uint8_t      TlvPresent;
    uint64_t band_pref;
}NASBandPreferenceTlv;

/*
 * Name:    NASPRLPreferenceTlv
 *
 * Purpose: Structure used to store CDMA PRL Preference TLV Value.
 *
 * Members: TlvPresent - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          PRLPref    - CDMA PRL Preference Value
 */
typedef struct
{
    uint8_t TlvPresent;
    uint16_t PRLPref;
}NASPRLPreferenceTlv;

/*
 * Name:    NASRoamPreferenceTlv
 *
 * Purpose: Structure used to store Roaming Preference TLV Value.
 *
 * Members: TlvPresent - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          RoamPref   - Roaming Preference Value
 */
typedef struct
{
    uint8_t TlvPresent;
    uint16_t RoamPref;
}NASRoamPreferenceTlv;

/*
 * Name:    NASLTEBandPreferenceTlv
 *
 * Purpose: Structure used to store LTE Band Preference TLV Value.
 *
 * Members: TlvPresent  - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          LTEBandPref - LTE Band Preference Value
 */
typedef struct 
{
    uint8_t      TlvPresent;
    uint64_t LTEBandPref;
}NASLTEBandPreferenceTlv;

/*
 * Name:    NASNetSelPreferenceTlv
 *
 * Purpose: Structure used to store Network Selection Preference TLV Value.
 *
 * Members: TlvPresent - Boolean indicating the presence of the TLV
 *                       in the QMI response
 *
 *          NetSelPref - Network Selection Preference Value
 */
typedef struct
{
    uint8_t TlvPresent;
    uint8_t NetSelPref;
}NASNetSelPreferenceTlv;

/*
 * Name:    NASServDomainPrefTlv
 *
 * Purpose: Structure used to store Service domain preference TLV Value.
 *
 * Members: TlvPresent    - Boolean indicating the presence of the TLV
 *                          in the QMI response
 *
 *          SrvDomainPref - Service Domain Preference Value
 */
typedef struct 
{
    uint8_t TlvPresent;
    uint32_t SrvDomainPref;
}NASServDomainPrefTlv;

/*
 * Name:    NASGWAcqOrderPrefTlv
 *
 * Purpose: Structure used to store GSM/WCDMA acquisition order preference
 *          TLV Value.
 *
 * Members: TlvPresent     - Boolean indicating the presence of the TLV
 *                          in the QMI response
 *
 *          GWAcqOrderPref - GSM/WCDMA acquisition order Preference Value
 */
typedef struct 
{
    uint8_t TlvPresent;
    uint32_t GWAcqOrderPref;
}NASGWAcqOrderPrefTlv;

/*
 * Name:    QmiCbkNasSystemSelPrefInd
 *
 * Purpose: Structure used to store all QMI Notification parameters.
 *
 * Members: EMTlv    - Emergency mode Tlv
 *          MPTlv    - Mode preference Tlv
 *          BPTlv    - Band preference Tlv
 *          PRLPTlv  - PRL preference Tlv
 *          RPTlv    - Roaming preference Tlv
 *          LBPTlv   - LTE band preference Tlv
 *          NSPTlv   - Network selection preference Tlv
 *          SDPTlv   - Service domain preference Tlv
 *          GWAOPTlv - GSM WCDMA Acquisition Order Preference Tlv
 *
 * Note:    None
 */
typedef struct{
    NASEmergencyModeTlv     EMTlv;
    NASModePreferenceTlv    MPTlv;
    NASBandPreferenceTlv    BPTlv;
    NASPRLPreferenceTlv     PRLPTlv;
    NASRoamPreferenceTlv    RPTlv;
    NASLTEBandPreferenceTlv LBPTlv;
    NASNetSelPreferenceTlv  NSPTlv;
    NASServDomainPrefTlv    SDPTlv;
    NASGWAcqOrderPrefTlv    GWAOPTlv;
}NASQmiCbkNasSystemSelPrefInd;


/**
 *  @param Info 
 *          - Structure used to store all QMI Notification Info.
 *              - See \ref NASQmiCbkNasSystemSelPrefInd for more details
 *  @param Tlvresult 
 *          - unpack result
 */
typedef struct
{
    NASQmiCbkNasSystemSelPrefInd Info;
    uint16_t Tlvresult;
} unpack_nas_SLQSSetSysSelectionPrefCallBack_ind_t;

/**
 * System Selection Preference indication unpack
 * @param[in]   pResp       qmi indication from modem
 * @param[in]   respLen     qmi indication length
 * @param[out]  pOuput      indication unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_nas_SLQSSetSysSelectionPrefCallBack_ind(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SLQSSetSysSelectionPrefCallBack_ind_t *pOutput
);

typedef struct  {
    uint8_t    TlvPresent;
    uint32_t    message_type;
    uint16_t    data_len;
    uint8_t    data_buf[NAS_OTA_MESSAGE_MAX_BUF_SIZE];
}NASOTAMessageTlv;

typedef struct {
    uint8_t    TlvPresent;
    uint8_t    nas_release;
    uint8_t    nas_major;
    uint8_t    nas_minor;
}NASLteNasReleaseInfoTlv;

typedef struct {
    uint8_t    TlvPresent;
    uint64_t time;
}NASTimeInfoTlv;

typedef struct  {
    NASOTAMessageTlv otaMsgTlv;
    NASLteNasReleaseInfoTlv nasRelInfoTlv;
    NASTimeInfoTlv timeTlv;
}NASQmiCbkNasSwiOTAMessageInd;


/**
 *  @param Info 
 *          - Structure used to store all QMI Notification Info.
 *              - See \ref NASQmiCbkNasSwiOTAMessageInd for more details
 *  @param Tlvresult 
 *          - unpack result
 */
typedef struct
{
    NASQmiCbkNasSwiOTAMessageInd Info;
    uint16_t Tlvresult;
} unpack_nas_SLQSNasSwiOTAMessageCallback_ind_t;

/**
 * OTA message indication unpack
 * @param[in]   pResp       qmi indication from modem
 * @param[in]   respLen     qmi indication length
 * @param[out]  pOuput      indication unpacked
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasSwiOTAMessageCallback_ind(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SLQSNasSwiOTAMessageCallback_ind_t *pOutput
);

/**
 *  Structure contains Manual Network Register Information parameters
 *
 *  \param  mcc
 *          - A 16-bit integer representation of Mobile Country Code.
 *            Range - 0 to 999.
 *
 *  \param  mnc
 *          - A 16-bit integer representation of Mobile Network Code.
 *            Range - 0 to 999.
 *
 *  \param  rat
 *          - Radio access technology for which to register.
 *              - 0x04 - RADIO_IF_GSM
 *              - 0x05 - RADIO_IF_UMTS
 *              - 0x08 - RADIO_IF_LTE
 */
typedef struct
{
    uint16_t  mcc;
    uint16_t  mnc;
    uint32_t rat;
} nas_MNRInfo;

/**
 *  This structure contains Initiate Network Registration request parameters
 *
 *  \param  regAction
 *          - Specifies one of the following register actions :
 *            - AUTO_REGISTER - Device registers according to its provisioning
 *                  and optional parameters supplied with the command are ignored.
 *            - MANUAL_REGISTER - Device registers to a specified network and
 *                  the optional Manual Network Register Information parameter
 *                  pMNRInfo must also be included for the command to process
 *                  successfully and supported only for 3GPP.
 *
 *  \param  pMNRInfo [Optional]
 *          - Pointer to structure MNRInfo
 *              - See \ref nas_MNRInfo for more information
 *
 *  \param  pChangeDuration [Optional]
 *          - Duration of the change.
 *              - 0x00 - Power cycle - Remains active until the next device power
 *                       cycle
 *              - 0x01 - Permanent - Remains active through power cycles until
 *                       changed by the client
 *
 * \param  pMncPcsDigitStatus [Optional]
 *         - MNC PCS Digit Include Status
 *             - True - MNC is a 3-digit value.
 *             - False - MNC is a 2-digit value.
 */
typedef struct
{
    uint32_t    regAction;
    nas_MNRInfo *pMNRInfo;
    uint32_t    *pChangeDuration;
    uint8_t     *pMncPcsDigitStatus;
} pack_nas_SLQSInitiateNetworkRegistration_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in]  pReqParam request Parameters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSInitiateNetworkRegistration(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSInitiateNetworkRegistration_t *pReqParam
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSInitiateNetworkRegistration(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 *  This structure contains the OTA message indication.
 *
 *  \param  lteEsmUl
 *          - 0 - do not report
 *          - 1 - report LTE ESM uplink messages
 *
 *  \param  lteEsmDl
 *          - 0 - do not report
 *          - 1 - report LTE ESM downlink messages
 *
 *  \param  lteEmmUl
 *          - 0 - do not report
 *          - 1 - report LTE EMM uplink messages
 *
 *  \param  lteEmmDl
 *          - 0 - do not report
 *          - 1 - report GSM/UMTS uplink messages
 *
 *  \param  gsmUmtsUl
 *          - 0 - do not report
 *          - 1 - report GSM/UMTS uplink messages
 *
 *  \param  gsmUmtsDl
 *          - 0 - do not report
 *          - 1 - report GSM/UMTS downlink messages
 *
 *  \param  pRankIndicatorInd
 *          - 0 - do not report
 *          - 1 - report Rank Indicator messages
 */
typedef struct
{
    uint8_t lteEsmUl;
    uint8_t lteEsmDl;
    uint8_t lteEmmUl;
    uint8_t lteEmmDl;
    uint8_t gsmUmtsUl;
    uint8_t gsmUmtsDl;
    uint8_t *pRankIndicatorInd;
}pack_nas_SLQSNasSwiOTAMessageCallback_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReq qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] pReqParam request prarmeters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSNasSwiOTAMessageCallback(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSNasSwiOTAMessageCallback_t *pReqParam
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasSwiOTAMessageCallback(
        uint8_t *pResp,
        uint16_t respLen
        );

/**
 *  @param  mcc
 *          - A 16-bit integer representation of MCC. Range: 0 to 999
 *
 *  @param  mnc
 *          - A 16-bit integer representation of MNC. Range: 0 to 999
 *
 *  @param  pMncPcsStatus
 *          - MNC PCS Digit Include Status
 *          - Used to interpret the length of the corresponding MNC reported
 *            in the PLMN TLV(0x01).
 *          - Values
 *            - TRUE -  MNC is a three-digit value. e.g. a reported value of 90
 *                      corresponds to an MNC value of 090
 *            - FALSE - MNC is a two-digit value. e.g. a reported value of 90
 *                      corresponds to an MNC value of 90
 *
 *  \note    If pMncPcsStatus is not present, an MNC smaller than 100
 *           is assumed to be a two-digit value, and an MNC greater than
 *           or equal to 100 is assumed to be a three digit value.
 */
typedef struct
{
    uint16_t mcc;
    uint16_t mnc;
    uint8_t  *pMncPcsStatus;
}pack_nas_SLQSGetPLMNName_t;

/*
 *  @param  spnEncoding
 *          - Coding scheme used for service provider name. This value is
 *            ignored if spn_len is zero
 *            Values:
 *              - 0x00 - SMS default 7-bit coded alphabet as defined in
 *                3GPP TS 23.038 with bit 8 set to 0
 *              - 0x01 - UCS2 (16 bit, little-endian) 3GPP TS 23.038
 *
 *  @param  spnLength
 *          - Length of SPN which follows
 *
 *  @param  spn
 *          - Service Provider name string
 *
 *  @param  shortNameEn
 *          - Coding scheme used for PLMN short name. This value is
 *            ignored if PLMN short name length is zero
 *            Values:
 *              - 0x00 - SMS default 7-bit coded alphabet as defined in
 *                3GPP TS 23.038 with bit 8 set to 0
 *              - 0x01 - UCS2 (16 bit, little-endian) 3GPP TS 23.038
 *
 *  @param  shortNameCI
 *          - Indicates whether the country initials are to be added to the
 *            shortName. This value is ignored if shortNameLen is zero. Values:
 *              - 0x00 - Do not add the letters for the countrys initials
 *                to the name
 *              - 0x01 - Add the countrys initials and a text string to
 *                the name
 *              - 0xFF - Not specified
 *
 *  @param  shortNameSB
 *          - PLMN short name spare bits. This value is ignored if
 *            shortNameLen is zero. Values:
 *              - 0x01 - Bit 8 is spare and set to 0 in octet
 *              - 0x02 - Bits 7 and 8 are spare and set to 0 in octet n
 *              - 0x03 - Bits 6 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x04 - Bits 5 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x05 - Bits 4 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x06 - Bits 3 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x07 - Bits 2 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x00 - Carries no information about the number of spare
 *                bits in octet n
 *
 *  @param  shortNameLen
 *          - Length of shortName which follows
 *
 *  @param  shortName
 *          - PLMN short name
 *
 *  @param  longNameEn
 *          - Coding scheme used for PLMN long name. This value is
 *            ignored if PLMN long name length is zero
 *            Values:
 *              - 0x00 - SMS default 7-bit coded alphabet as defined in
 *                3GPP TS 23.038 with bit 8 set to 0
 *              - 0x01 - UCS2 (16 bit, little-endian) 3GPP TS 23.038
 *
 *  @param  longNameCI
 *          - Indicates whether the country initials are to be added to the
 *            longName. This value is ignored if longNameLen is zero. Values:
 *              - 0x00 - Do not add the letters for the countrys initials
 *                to the name
 *              - 0x01 - Add the countrys initials and a text string to
 *                the name
 *              - 0xFF - Not specified
 *
 *
 *  @param  longNameSB
 *          - PLMN long name spare bits. This value is ignored if
 *            longNameLen is zero. Values:
 *              - 0x01 - Bit 8 is spare and set to 0 in octet
 *              - 0x02 - Bits 7 and 8 are spare and set to 0 in octet n
 *              - 0x03 - Bits 6 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x04 - Bits 5 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x05 - Bits 4 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x06 - Bits 3 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x07 - Bits 2 to 8 (inclusive) are spare and set to 0 in
 *                octet n
 *              - 0x00 - Carries no information about the number of spare
 *                bits in octet n
 *
 *  @param  longNameLen
 *          - Length of longName which follows
 *
 *  @param  longName
 *          - PLMN long name
 */
typedef struct
{
    uint8_t spnEncoding;
    uint8_t spnLength;
    char spn[255];
    uint8_t shortNameEn;
    uint8_t shortNameCI;
    uint8_t shortNameSB;
    char shortNameLen;
    uint8_t shortName[255];
    uint8_t longNameEn;
    uint8_t longNameCI;
    uint8_t longNameSB;
    uint8_t longNameLen;
    char longName[255];
} unpack_nas_SLQSGetPLMNName_t;

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request prarmeters
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSGetPLMNName(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        pack_nas_SLQSGetPLMNName_t *reqArg
        );

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput     unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSGetPLMNName(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SLQSGetPLMNName_t *pOutput
);

/**
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_nas_SLQSNasGetCellLocationInfo(
        pack_qmi_t *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );
/**
 *  This structure contains information about the Network Measurement Report
 *  (NMR) Cell Information.
 *
 *  @param  nmrCellID
 *          - Cell ID.
 *          - 0xFFFFFFFF indicates cell ID information is not present.
 *
 *  @param  nmrPlmn[NAS_PLMN_LENGTH]
 *          - MCC/MNC information coded as octet 3, 4, and 5.
 *          - This field is ignored when nmrCellID is not present.
 *
 *  @param  nmrLac
 *          - Location area code.
 *          - This field is ignored when nmrCellID is not present.
 *              - 0xFFFF - Not Available
 *
 *  @param  nmrArfcn
 *          - Absolute RF channel number.
 *              - 0xFFFF - Not Available
 *
 *  @param  nmrBsic
 *          - Base station identity code.
 *              - 0xFF - Not Available
 *
 *  @param  nmrRxLev
 *          - Cell Rx measurement.
 *          - Values range between 0 and 63.
 *          - Mapped to a measured signal level:
 *              - Rxlev 0 is a signal strength less than -110 dBm
 *              - Rxlev 1 is -110 dBm to -109 dBm
 *              - Rxlev 2 is -109 dBm to -108 dBm
 *              - ...
 *              - Rxlev 62 is -49 dBm to -48 dBm
 *              - Rxlev 63 is greater than -48 dBm
 *              - 0xFFFF - Not Available
 *
 */
typedef struct
{
    uint32_t nmrCellID;
    uint8_t  nmrPlmn[NAS_PLMN_LENGTH];
    uint16_t  nmrLac;
    uint16_t  nmrArfcn;
    uint8_t  nmrBsic;
    uint16_t  nmrRxLev;
} nas_nmrCellInfo;

/**
 *  This structure contains information about the GERAN Network.
 *
 *  @param  cellID
 *          - Cell ID.
 *          - 0xFFFFFFFF indicates cell ID information is not present.
 *
 *  @param  plmn[NAS_PLMN_LENGTH]
 *          - MCC/MNC information coded as octet 3, 4, and 5.
 *          - This field is ignored when nmrCellID is not present.
 *
 *  @param  lac
 *          - Location area code.
 *          - This field is ignored when nmrCellID is not present.
 *              - 0xFFFF - Not Available
 *
 *  @param  arfcn
 *          - Absolute RF channel number.
 *              - 0xFFFF - Not Available
 *
 *  @param  bsic
 *          - Base station identity code.
 *              - 0xFF - Not Available
 *
 *  @param  timingAdvance
 *          - Measured delay (in bit periods; 1 bit period = 48/13 microsecond)
 *            of access burst transmission on RACH or PRACH to the expected
 *            signal from an MS at zero distance under static channel
 *            conditions.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  rxLev
 *          - Serving Cell Rx measurement.
 *          - Values range between 0 and 63.
 *          - Mapped to a measured signal level:
 *              - Rxlev 0 is a signal strength less than -110 dBm
 *              - Rxlev 1 is -110 dBm to -109 dBm
 *              - Rxlev 2 is -109 dBm to -108 dBm
 *              - ...
 *              - Rxlev 62 is -49 dBm to -48 dBm
 *              - Rxlev 63 is greater than -48 dBm
 *              - 0xFFFF - Not Available
 *
 *  @param  nmrInst
 *          - Provides the number of set of instances which follow.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  insNmrCellInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_nmrCellInfo for more information.
 *
 */
typedef struct
{
    uint32_t        cellID;
    uint8_t         plmn[NAS_PLMN_LENGTH];
    uint16_t        lac;
    uint16_t        arfcn;
    uint8_t         bsic;
    uint32_t        timingAdvance;
    uint16_t        rxLev;
    uint8_t         nmrInst;
    nas_nmrCellInfo insNmrCellInfo[NAS_MAX_DESCRIPTION_LENGTH];
} nas_GERANInfo;

/**
 *  This structure contains information about the GERAN Instances in UMTS
 *  Network.
 *
 *  @param  geranArfcn
 *          - Absolute RF channel number.
 *
 *  @param  geranBsicNcc
 *          - Base station identity code network color code.
 *          - 0xFF indicates information is not present.
 *
 *  @param  geranBsicBcc
 *          - Base station identity code base station color code.
 *          - 0xFF indicates information is not present.
 *
 *  @param  geranRssi
 *          - Received signal strength indicator.
 *
 */
typedef struct
{
    uint16_t geranArfcn;
    uint8_t  geranBsicNcc;
    uint8_t  geranBsicBcc;
    int16_t  geranRssi;
} nas_geranInstInfo;

/**
 *  This structure contains information about the UMTS Instances in UMTS
 *  Network.
 *
 *  @param  umtsUarfcn
 *          - UTRA absolute RF channel number.
 *
 *  @param  umtsPsc
 *          - Primary scrambling code.
 *
 *  @param  umtsRscp
 *          - Received signal code power.
 *
 *  @param  umtsEcio
 *          - ECIO(Signal-to-Interference-ratio).
 *
 */
typedef struct
{
    uint16_t umtsUarfcn;
    uint16_t umtsPsc;
    int16_t umtsRscp;
    int16_t umtsEcio;
} nas_UMTSinstInfo;

/**
 *  This structure contains information about the UMTS Network.
 *
 *  @param  cellID
 *          - Cell ID.
 *          - 0xFFFFFFFF indicates cell ID information is not present.
 *
 *  @param  plmn[NAS_PLMN_LENGTH]
 *          - MCC/MNC information coded as octet 3, 4, and 5.
 *          - This field is ignored when nmrCellID is not present.
 *
 *  @param  lac
 *          - Location area code.
 *          - This field is ignored when nmrCellID is not present.
 *              - 0xFFFF - Not Available
 *
 *  @param  uarfcn
 *          - UTRA absolute RF channel number.
 *              - 0xFFFF - Not Available
 *
 *  @param  psc
 *          - Primary scrambling code.
 *              - 0xFFFF - Not Available
 *
 *  @param  rscp
 *          - Received signal code power.
 *              - 0xFFFF - Not Available
 *
 *  @param  ecio
 *          - ECIO(Signal-to-Interference-ratio).
 *              - 0xFFFF - Not Available
 *
 *  @param  umtsInst
 *          - Provides the number of set of UMTS info instances.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  UMTSInstInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_UMTSinstInfo for more information.
 *
 *  @param  geranInst
 *          - Provides the number of set of GERAN info instances.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  GeranInstInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_geranInstInfo for more information.
 *
 */
typedef struct
{
    uint16_t          cellID;
    uint8_t           plmn[NAS_PLMN_LENGTH];
    uint16_t          lac;
    uint16_t          uarfcn;
    uint16_t          psc;
    int16_t           rscp;
    int16_t           ecio;
    uint8_t           umtsInst;
    nas_UMTSinstInfo  UMTSInstInfo[NAS_MAX_DESCRIPTION_LENGTH];
    uint8_t           geranInst;
    nas_geranInstInfo GeranInstInfo[NAS_MAX_DESCRIPTION_LENGTH];
} nas_UMTSInfo;

/**
 *  This structure contains information about the CDMA Network.
 *
 *  @param  sid
 *          - System ID.
 *              - 0xFFFF - Not Available
 *
 *  @param  nid
 *          - Network ID.
 *              - 0xFFFF - Not Available
 *
 *  @param  baseId
 *          - Base station ID.
 *              - 0xFFFF - Not Available
 *
 *  @param  refpn
 *          - Reference PN.
 *              - 0xFFFF - Not Available
 *
 *  @param  baseLat
 *          - Latitude of the current base station in units of 0.25 sec.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  baseLong
 *          - Longitude of the current base station in units of 0.25 sec.
 *              - 0xFFFFFFFF - Not Available
 *
 */
typedef struct
{
    uint16_t  sid;
    uint16_t  nid;
    uint16_t  baseId;
    uint16_t  refpn;
    uint32_t  baseLat;
    uint32_t  baseLong;
} nas_CDMAInfo;

/**
 *  This structure contains information about the Cell parameters.
 *
 *  @param  pci
 *          - Physical cell ID.
 *          - Range: 0 to 503.
 *
 *  @param  rsrq
 *          - Current RSRQ in 1/10 dB as measured by L1.
 *          - Range: -20.0 dB to -3.0 dB.
 *
 *  @param  rsrp
 *          - Current RSRP in 1/10 dBm as measured by L1.
 *          - Range: -140.0 dBm to -44.0 dBm.
 *
 *  @param  rssi
 *          - Current RSSI in 1/10 dBm as measured by L1.
 *          - Range: -120.0 dBm to 0.
 *
 *  @param  srxlev
 *          - Cell selection Rx level (Srxlev) value.
 *          - Range: -128 to 128.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 */
typedef struct
{
    uint16_t pci;
    int16_t rsrq;
    int16_t rsrp;
    int16_t rssi;
    int16_t srxlev;
} nas_cellParams;

/**
 *  This structure contains information about the LTE Intra-Frequency Network.
 *
 *  @param  ueInIdle
 *          - TRUE if the UE is in Idle mode, otherwise FALSE.
 *              - 0xFF - Not Available
 *
 *  @param  plmn[NAS_PLMN_LENGTH]
 *          - PLMN ID coded as octet 3, 4, and 5.
 *
 *  @param  tac
 *          - Tracking area code.
 *              - 0xFFFF - Not Available
 *
 *  @param  globalCellId
 *          - Global cell ID in the system information block.
 *              - 0xFFFFFFFF - Not Available
 *
 *  @param  earfcn
 *          - E-UTRA absolute radio frequency channel number of the serving
 *            cell.
 *          - Range: 0 to 65535.
 *              - 0xFFFF - Not Available
 *
 *  @param  servingCellId
 *          - LTE serving cell ID.
 *          - Range: 0 to 503.
 *          - This is the cell ID of the serving cell and can be found in the
 *            cell list.
 *              - 0xFFFF - Not Available
 *
 *  @param  cellReselPriority
 *          - Priority for serving frequency.
 *          - Range: 0 to 7.
 *          - This field is only valid when ue_in_idle is TRUE.
 *              - 0xFF - Not Available
 *
 *  @param  sNonIntraSearch
 *          - S non-intra search threshold to control non-intrafrequency
 *            searches.
 *          - Range: 0 to 31.
 *          - This field is only valid when ue_in_idle is TRUE.
 *              - 0xFF - Not Available
 *
 *  @param  threshServingLow
 *          - Serving cell low threshold.
 *          - Range: 0 to 31.
 *          - This field is only valid when ue_in_idle is TRUE.
 *              - 0xFF - Not Available
 *
 *  @param  sIntraSearch
 *          - S Intra search threshold.
 *          - Range: 0 to 31.
 *          - The current cell measurement must fall below this threshold to
 *            consider intrafrequency for reselection.
 *          - This field is only valid when ue_in_idle is TRUE.
 *              - 0xFF - Not Available
 *
 *  @param  cellsLen
 *          - Provides the number of set of cell params.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  CellParams[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_cellParams for more information.
 *
 */
typedef struct
{
    uint8_t        ueInIdle;
    uint8_t        plmn[NAS_PLMN_LENGTH];
    uint16_t       tac;
    uint32_t       globalCellId;
    uint16_t       earfcn;
    uint16_t       servingCellId;
    uint8_t        cellReselPriority;
    uint8_t        sNonIntraSearch;
    uint8_t        threshServingLow;
    uint8_t        sIntraSearch;
    uint8_t        cellsLen;
    nas_cellParams CellParams[NAS_MAX_DESCRIPTION_LENGTH];
} nas_LTEInfoIntrafreq;

/**
 *  This structure contains information about the inter-frequency.
 *
 *  @param  earfcn
 *          - E-UTRA absolute radio frequency channel number of the serving
 *            cell.
 *          - Range: 0 to 65535.
 *
 *  @param  threshXLow
 *          - Cell Srxlev low threshold.
 *          - Range: 0 to 31.
 *          - When the serving cell does not exceed thresh_serving_low, the
 *            value of an evaluated cell must be smaller than this value to be
 *            considered for re-selection.
 *
 *  @param  threshXHigh
 *          - Cell Srxlev high threshold.
 *          - Range: 0 to 31.
 *          - When the serving cell exceeds thresh_serving_low, the value of an
 *            evaluated cell must be greater than this value to be considered
 *            for re-selection.
 *
 *  @param  cell_resel_priority
 *          - Cell re-selection priority
 *          - Range: 0 to 7.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  cells_len
 *          - Provides the number of set of cell params.
 *
 *  @param  cellInterFreqParams[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_cellParams for more information.
 *
 */
typedef struct
{
    uint16_t       earfcn;
    uint8_t        threshXLow;
    uint8_t        threshXHigh;
    uint8_t        cell_resel_priority;
    uint8_t        cells_len;
    nas_cellParams cellInterFreqParams[NAS_MAX_DESCRIPTION_LENGTH];
} nas_infoInterFreq;

/**
 *  This structure contains information about the LTE Inter-Frequency Network.
 *
 *  @param  ueInIdle
 *          - TRUE if the UE is in Idle mode, otherwise FALSE.
 *              - 0xFF - Not Available
 *
 *  @param  freqsLen
 *          - Provides the number of set of inter frequency information.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  InfoInterfreq[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_infoInterFreq for more information.
 *
 */
typedef struct
{
    uint8_t          ueInIdle;
    uint8_t          freqsLen;
    nas_infoInterFreq InfoInterfreq[NAS_MAX_DESCRIPTION_LENGTH];
} nas_LTEInfoInterfreq;

/**
 *  This structure contains information about the GSM Cell.
 *
 *  @param  arfcn
 *          - GSM frequency being reported.
 *          - Range: 0 to 1023.
 *
 *  @param  band1900
 *          - Band indicator for the GSM ARFCN
 *          - This field is only valid if arfcn is in the overlapping region.
 *          - If TRUE and the cell is in the overlapping region, the ARFCN is
 *            on the 1900 band.
 *          - If FALSE, it is on the 1800 band.
 *
 *  @param  cellIdValid
 *          - Flag indicating whether the base station identity code ID is
 *            valid.
 *
 *  @param  bsicId
 *          - Base station identity code ID, including base station color code
 *            and network color code.
 *          - The lower 6 bits can be set to any value.
 *
 *  @param  rssi
 *          - Measured RSSI value in 1/10 dB.
 *          - Range: -200.0 dB to 0
 *
 *  @param  srxlev
 *          - Cell selection Rx level (Srxlev) value.
 *          - Range: -128 to 128.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 */
typedef struct
{
    uint16_t arfcn;
    uint8_t  band1900;
    uint8_t  cellIdValid;
    uint8_t  bsicId;
    int16_t  rssi;
    int16_t  srxlev;
} nas_gsmCellInfo;

/**
 *  This structure contains information about the LTE GSM Cell.
 *
 *  @param  cellReselPriority
 *          - Priority of this frequency group.
 *          - Range: 0 to 7.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  threshGsmHigh
 *          - Reselection threshold for high priority layers.
 *          - Range: 0 to 31.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  threshGsmLow
 *          - Reselection threshold for low priority layers.
 *          - Range: 0 to 31.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  nccPermitted
 *          - Bitmask specifying whether a neighbor with a specific network
 *            color code is to be reported.
 *          - Range: 0 to 255.
 *          - Bit n set to 1 means a neighbor with NCC n must be included in
 *            the report. This flag is synonymous with a blacklist in other
 *            RATs.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  cells_len
 *          - Provides the number of set of gsm cells.
 *
 *  @param  GsmCellInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_gsmCellInfo for more information.
 *
 */
typedef struct
{
    uint8_t         cellReselPriority;
    uint8_t         threshGsmHigh;
    uint8_t         threshGsmLow;
    uint8_t         nccPermitted;
    uint8_t         cells_len;
    nas_gsmCellInfo GsmCellInfo[NAS_MAX_DESCRIPTION_LENGTH];
} nas_lteGsmCellInfo;

/**
 *  This structure contains information about the LTE Neighboring GSM Network.
 *
 *  @param  ueInIdle
 *          - TRUE if the UE is in Idle mode, otherwise FALSE.
 *              - 0xFF - Not Available
 *
 *  @param  freqsLen
 *          - Provides the number of set of LTE GSM cell information.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  LteGsmCellInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_lteGsmCellInfo for more information.
 *
 */
typedef struct
{
    uint8_t            ueInIdle;
    uint8_t            freqsLen;
    nas_lteGsmCellInfo LteGsmCellInfo[NAS_MAX_DESCRIPTION_LENGTH];
} nas_LTEInfoNeighboringGSM;

/**
 *  This structure contains information about the WCDMA Cell.
 *
 *  @param  psc
 *          - Primary scrambling code.
 *          - Range: 0 to 511.
 *
 *  @param  cpich_rscp
 *          - Absolute power level (in 1/10 dBm) of the common pilot channel
 *            as received by the UE.
 *          - Range: -120.0 dBm to -25.0 dBm
 *
 *  @param  cpich_ecno
 *          - CPICH Ec/No; ratio (in 1/10 dB) of the received energy per PN
 *            chip for the CPICH to the total received power spectral density
 *            at the UE antenna connector.
 *          - Range: -50.0 dB to 0.
 *
 *  @param  srxlev
 *          - Cell selection Rx level (Srxlev) value.
 *          - Range: -128 to 128.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 */
typedef struct
{
    uint16_t psc;
    int16_t  cpich_rscp;
    int16_t  cpich_ecno;
    int16_t  srxlev;
} nas_wcdmaCellInfo;

/**
 *  This structure contains information about the LTE WCDMA Cell.
 *
 *  @param  uarfcn
 *          - WCDMA layer frequency.
 *          - Range: 0 to 16383.
 *
 *  @param  cellReselPriority
 *          - Cell re-selection priority.
 *          - Range: 0 to 7.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  threshXhigh
 *          - Re-selection low threshold.
 *          - Range: 0 to 31.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  threshXlow
 *          - Re-selection high threshold.
 *          - Range: 0 to 31.
 *          - This field is only valid when ue_in_idle is TRUE.
 *
 *  @param  cellsLen
 *          - Provides the number of set of WCDMA cells.
 *
 *  @param  WCDMACellInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref wcdmaCellInfo for more information.
 *
 */
typedef struct
{
    uint16_t          uarfcn;
    uint8_t           cellReselPriority;
    uint16_t          threshXhigh;
    uint16_t          threshXlow;
    uint8_t           cellsLen;
    nas_wcdmaCellInfo WCDMACellInfo[NAS_MAX_DESCRIPTION_LENGTH];
} nas_lteWcdmaCellInfo;

/**
 *  This structure contains information about the LTE Neighboring WCDMA Network.
 *
 *  @param  ueInIdle
 *          - TRUE if the UE is in Idle mode, otherwise FALSE.
 *              - 0xFF - Not Available
 *
 *  @param  freqsLen
 *          - Provides the number of set of LTE WCDMA cell information.
 *          - If 0(zero), then no information follows it.
 *
 *  @param  LTEWCDMACellInfo[NAS_MAX_DESCRIPTION_LENGTH]
 *          - See @ref nas_lteWcdmaCellInfo for more information.
 *
 */
typedef struct
{
    uint8_t             ueInIdle;
    uint8_t             freqsLen;
    nas_lteWcdmaCellInfo LTEWCDMACellInfo[NAS_MAX_DESCRIPTION_LENGTH];
} nas_LTEInfoNeighboringWCDMA;

/**
 *  This structure contains information about the UMTS LTE neighbour Cell.
 *
 *  @param  earfcn
 *          - E-UTRA absolute RF channel number of the detected cell.
 *
 *  @param  pci
 *          - Physical cell ID of the detected cell.
 *          - Range is defined in 3GPP TS 36.211
 *
 *  @param  rsrp
 *          - Current received signal strength indication (in dBm)
 *            of the detected cell.
 *
 *  @param  rsrq
 *          - Current reference signal received quality (in dB) of
 *            the detected cell.
 *
 *  @param  srxlev
 *          - Cell selection Rx level (Srxlev) value of the detected
 *            cell in linear scale.
 *          - This field is only valid when wcdma_rrc_state is not
 *            NAS_WCDMA_RRC_STATE_CEL_FACH or NAS_WCDMA_RRC_STATE_CELL_DCH.
 *
 *  @param  cellIsTDD
 *          - TRUE if the cell is TDD; FALSE if the cell is FDD.
 *
 */
typedef struct
{
    uint16_t  earfcn;
    uint16_t  pci;
    uint32_t  rsrp;
    uint32_t  rsrq;
    int16_t   srxlev;
    uint8_t   cellIsTDD;
} nas_umtsLTENbrCell;

/**
 *  This structure contains information about the WCDMA - LTE Neighboring
 *  Cell Info Set.
 *
 *  @param  wcdmaRRCState
 *          - WCDMA RRC states.
 *          - Defined in 3GPP TS 25.331
 *          - Values:
 *              - 0x00 - NAS_WCDMA_RRC_STATE_DISCONNECTED
 *                         - WCDMA RRC State is IDLE
 *              - 0x01 - NAS_WCDMA_RRC_STATE_CELL_PCH
 *                         - WCDMA RRC state is CELL_PCH
 *              - 0x02 - NAS_WCDMA_RRC_STATE_URA_PCH
 *                         - WCDMA RRC state is URA_PCH
 *              - 0x03 - NAS_WCDMA_RRC_STATE_CELL_FACH
 *                         - WCDMA RRC state is CELL_FACH
 *              - 0x04 - NAS_WCDMA_RRC_STATE_CELL_DCH
 *                         - WCDMA RRC state is CELL_DCH
 *
 *  @param  umtsLTENbrCellLen
 *          - Number of sets of UMTS LTE Neighbors.
 *
 *  @param  UMTSLTENbrCell
 *          - See @ref nas_umtsLTENbrCell for more information.
 *
 */
typedef struct
{
    uint32_t           wcdmaRRCState;
    uint8_t            umtsLTENbrCellLen;
    nas_umtsLTENbrCell UMTSLTENbrCell[NAS_MAX_DESCRIPTION_LENGTH];
} nas_WCDMAInfoLTENeighborCell;

/**
 *  This structure contains information about the Get Cell Location response
 *  parameters.
 *
 *  @param  pGERANInfo
 *          - See @ref nas_GERANInfo for more information.
 *
 *  @param  pUMTSInfo
 *          - See @ref nas_UMTSInfo for more information.
 *
 *  @param  pCDMAInfo
 *          - See @ref nas_CDMAInfo for more information.
 *
 *  @param  pLTEInfoIntrafreq
 *          - See @ref nas_LTEInfoIntrafreq for more information.
 *
 *  @param  pLTEInfoInterfreq
 *          - See @ref nas_LTEInfoInterfreq for more information.
 *
 *  @param  pLTEInfoNeighboringGSM
 *          - See @ref nas_LTEInfoNeighboringGSM for more information.
 *
 *  @param  pLTEInfoNeighboringWCDMA
 *          - See @ref nas_LTEInfoNeighboringWCDMA for more information.
 *
 *  @param  pUMTSCellID
 *          - Cell ID.
 *          - 0xFFFFFFFF indicates cell ID information is not present.
 *
 *  @param  pWCDMAInfoLTENeighborCell
 *          - See @ref nas_WCDMAInfoLTENeighborCell for more information.
 */
typedef struct
{
    nas_GERANInfo                *pGERANInfo;
    nas_UMTSInfo                 *pUMTSInfo;
    nas_CDMAInfo                 *pCDMAInfo;
    nas_LTEInfoIntrafreq         *pLTEInfoIntrafreq;
    nas_LTEInfoInterfreq         *pLTEInfoInterfreq;
    nas_LTEInfoNeighboringGSM    *pLTEInfoNeighboringGSM;
    nas_LTEInfoNeighboringWCDMA  *pLTEInfoNeighboringWCDMA;
    uint32_t                     *pUMTSCellID;
    nas_WCDMAInfoLTENeighborCell *pWCDMAInfoLTENeighborCell;
} unpack_nas_SLQSNasGetCellLocationInfo_t;

/**
 * @param[in]   pResp       qmi response from modem
 * @param[in]   respLen     qmi response length
 * @param[out]  pOutput     unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_nas_SLQSNasGetCellLocationInfo(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_nas_SLQSNasGetCellLocationInfo_t *pOutput
);
