/**
 * \ingroup pack
 *
 * \file loc.h
 */

#ifndef _LOC_HEADER_
#define _LOC_HEADER_
#include "common.h"
#include <stdint.h>

#define LOC_UINT8_MAX_STRING_SZ 255

/*!
 * \def LOCEVENTMASKPOSITIONREPORT
 * The control point must enable this mask to receive 
 * position report event indications.
 */
#define LOCEVENTMASKPOSITIONREPORT            0x00000001
/*!
 * \def LOCEVENTMASKGNSSSVINFO
 * The control point must enable this mask to receive 
 * satellite report event indications. These reports 
 * are sent at a 1 Hz rate.
 */
#define LOCEVENTMASKGNSSSVINFO                0x00000002
/*!
 * \def LOCEVENTMASKNMEA
 * The control point must enable this mask to receive 
 * NMEA reports for position and satellites in view. 
 * The report is at a 1 Hz rate.
 */
#define LOCEVENTMASKNMEA                      0x00000004
/*!
 * \def LOCEVENTMASKNINOTIFYVERIFYREQ
 * The control point must enable this mask to receive 
 * NI Notify/Verify request event indications.
 */
#define LOCEVENTMASKNINOTIFYVERIFYREQ         0x00000008
/*!
 * \def LOCEVENTMASKINJECTTIMEREQ
 * The control point must enable this mask to receive 
 * time injection request event indications.
 */
#define LOCEVENTMASKINJECTTIMEREQ             0x00000010
/*!
 * \def LOCEVENTMASKINJECTPREDICTEDORBITSREQ
 * The control point must enable this mask to receive 
 * predicted orbits request event indications.
 */
#define LOCEVENTMASKINJECTPREDICTEDORBITSREQ  0x00000020
/*!
 * \def LOCEVENTMASKINJECTPOSITIONREQ
 * The control point must enable this mask to receive 
 * position injection request event indications.
 */
#define LOCEVENTMASKINJECTPOSITIONREQ         0x00000040
/*!
 * \def LOCEVENTMASKENGINESTATE
 * The control point must enable this mask to receive 
 * engine state report event indications.
 */
#define LOCEVENTMASKENGINESTATE               0x00000080
/*!
 * \def LOCEVENTMASKFIXSESSIONSTATE
 * The control point must enable this mask to receive 
 * fix session status report event indications.
 */
#define LOCEVENTMASKFIXSESSIONSTATE           0x00000100
/*! 
 * \def LOCEVENTMASKWIFIREQ
 * The control point must enable this mask to receive 
 * Wi-Fi position request event indications.
 */
#define LOCEVENTMASKWIFIREQ                   0x00000200
/*!
 \def LOCEVENTMASKSENSORSTREAMINGREADYSTATUS
 * The control point must enable this mask to receive 
 * notifications from the location engine indicating 
 * its readiness to accept data from the 
 * sensors (accelerometer, gyroscope, etc.).
 */
#define LOCEVENTMASKSENSORSTREAMINGREADYSTATUS  0x00000400
/*!
 * \def LOCEVENTMASKTIMESYNCREQ
 * The control point must enable this mask to receive 
 * time sync requests from the GPS engine. 
 * Time sync enables the GPS engine to synchronize 
 * its clock with the sensor processor's clock.
 */
#define LOCEVENTMASKTIMESYNCREQ               0x00000800
/*!
 * \def LOCEVENTMASKSETSPISTREAMINGREPORT
 * The control point must enable this mask to receive 
 * Stationary Position Indicator (SPI) 
 * streaming report indications.
 */
#define LOCEVENTMASKSETSPISTREAMINGREPORT     0x00001000
/*!
 * \def LOCEVENTMASKLOCATIONSERVERCONNECTIONREQ
 * The control point must enable this mask
 * to receive location server requests. These
 * requests are generated when the service
 * wishes to establish a connection with a
 * location server.
 */
#define LOCEVENTMASKLOCATIONSERVERCONNECTIONREQ  0x00002000
/*!
 * \def LOCEVENTMASKNIGEOFENCENOTIFICATION
 * The control point must
 * enable this mask to receive notifications
 * related to network-initiated Geofences.
 * These events notify the client when a
 * network-initiated Geofence is added,
 * deleted, or edited.
 */
#define LOCEVENTMASKNIGEOFENCENOTIFICATION    0x00004000
/*!
 * \def LOCEVENTMASKGEOFENCEGENALERT
 * The control point must
 * enable this mask to receive Geofence
 * alerts. These alerts are generated to
 * inform the client of the changes that may
 * affect a Geofence, for example, if GPS is
 * turned off or if the network is
 * unavailable.
 */
#define LOCEVENTMASKGEOFENCEGENALERT          0x00008000
/*!
 * \def LOCEVENTMASKGEOFENCEBREACHNOTIFICATION
 * The control point must enable this mask to
 * receive notifications when a Geofence is
 * breached. These events are generated
 * when a UE enters or leaves the perimeter
 * of a Geofence. This breach report is for a
 * single Geofence.
 */
#define LOCEVENTMASKGEOFENCEBREACHNOTIFICATION  0x00010000
/*!
 * \def LOCEVENTMASKPEDOMETERCONTROL
 * The control point must
 * enable this mask to register for
 * pedometer control requests from the
 * location engine. The location engine
 * sends this event to control the injection
 * of pedometer reports.
 */
#define LOCEVENTMASKPEDOMETERCONTROL          0x00020000
/*!
 * \def LOCEVENTMASKMOTIONDATACONTROL
 * The control point must
 * enable this mask to register for motion
 * data control requests from the location
 * engine. The location engine sends this
 * event to control the injection of motion
 * data.
 */
#define LOCEVENTMASKMOTIONDATACONTROL         0x00040000
/*!
 * \def LOCEVENTMASKBATCHFULLNOTIFICATION
 * The control point must enable this
 * mask to receive notification when a
 * batch is full. The location engine sends
 * this event to notify of Batch Full for
 * ongoing batching session.
 */
#define LOCEVENTMASKBATCHFULLNOTIFICATION     0x00080000
/*!
 * \def LOCEVENTMASKLIVEBATCHEDPOSITIONREPORT
 * The control point must
 * enable this mask to receive position
 * report indications along with an ongoing
 * batching session. The location engine
 * sends this event to notify the batched
 * position report while a batching session
 * is ongoing.
 */
#define LOCEVENTMASKLIVEBATCHEDPOSITIONREPORT 0x00100000
/*!
 * \def LOCEVENTMASKINJECTWIFIAPDATAREQ
 * The control point must enable this mask
 * to receive Wi-Fi Access Point (AP) data
 * inject request event indications.
 */
#define LOCEVENTMASKINJECTWIFIAPDATAREQ       0x00200000
/*!
 * \def LOCEVENTMASKGEOFENCEBATCHBREACHNOTIFICATION
 * The control point must enable this mask to
 * receive notifications when a Geofence is
 * breached. These events are generated
 * when a UE enters or leaves the perimeter
 * of a Geofence. This breach notification
 * is for multiple Geofences. Breaches
 * from multiple Geofences are all batched
 * and sent in the same notification.
 */
#define LOCEVENTMASKGEOFENCEBATCHBREACHNOTIFICATION 0x00400000
/*!
 * \def LOCEVENTMASKVEHICLEDATAREADYSTATUS
 * The control point must
 * enable this mask to receive notifications
 * from the location engine indicating its
 * readiness to accept vehicle data (vehicle
 * accelerometer, vehicle angular rate,
 * vehicle odometry, etc.).
 */
#define LOCEVENTMASKVEHICLEDATAREADYSTATUS    0x00800000
/*! 
 \def LOCEVENTMASKGNSSMEASUREMENTREPORT
 * The control point must
 * enable this mask to receive system clock
 * and satellite measurement report events
 * (system clock, SV time, Doppler, etc.).
 * Reports are generated only for the GNSS
 * satellite constellations that are enabled
 * using QMI_LOC_SET_GNSS_
 * CONSTELL_REPORT_CONFIG(Not yet supported).
 */
#define LOCEVENTMASKGNSSMEASUREMENTREPORT     0x01000000
/*!
 * \def LOCEVENTMASKINVALIDVALUE
 * Invalid Event Mask
 */
#define LOCEVENTMASKINVALIDVALUE              0xFFFFFFFF

enum {
    eQMI_LOC_SESS_STATUS_SUCCESS=0,
    eQMI_LOC_SESS_STATUS_IN_PROGRESS=1,
    eQMI_LOC_SESS_STATUS_FAILURE=2,
    eQMI_LOC_SESS_STATUS_TIMEOUT=3,
};


/**
 * This structure contains the Application Information
 *
 * \param appProviderLength
 *        - Length of the Application Provider
 *
 * \param pAppProvider
 *        - Application Provider
 *        - Depends upon the Length of application Provider
 *
 * \param appNameLength
 *        - Length of Application Name
 *
 * \param pAppName
 *        - Application Name
 *        - Depends upon the Length of application Name
 *
 * \param appVersionValid
 *        - Specifies whether the application version string contains a
 *        valid value
 *
 *        - 0x00 (FALSE) Application version string is invalid
 *        - 0x01 (TRUE) Application version string is valid
 *
 * \param appVersionLength
 *        - Length of Application Version
 *
 * \param pAppVersion
 *        - Application Version
 *        - Depends upon the Length of application Version
 *
 */

typedef struct 
{
    uint8_t appProviderLength;
    uint8_t *pAppProvider;
    uint8_t appNameLength;
    uint8_t *pAppName;
    uint8_t appVersionValid;
    uint8_t appVersionLength;
    uint8_t *pAppVersion;
}loc_LocApplicationInfo;

/**
 * This structure contains the Delete LOC SV Info
 *
 * \param id
 *        - LOC SV ID of the satellite whose data is to be
 *          deleted
 *        - Range:
 *           - For GPS: 1 to 32
 *           - For SBAS: 33 to 64
 *           - For GLONASS: 65 to 96
 *
 * \param system
 *        - Indicates to which constellation this loc_SV belongs
 *        - Valid values:
 *           - eQMI_LOC_SV_SYSTEM_GPS (1) - GPS satellite
 *           - eQMI_LOC_SV_SYSTEM_GALILEO (2) - GALILEO satellite
 *           - eQMI_LOC_SV_SYSTEM_SBAS (3) - SBAS satellite
 *           - eQMI_LOC_SV_SYSTEM_COMPASS (4) - COMPASS satellite
 *           - eQMI_LOC_SV_SYSTEM_GLONASS (5) - GLONASS satellite
 *           - eQMI_LOC_SV_SYSTEM_BDS (6) - BDS satellite
 *
 * \param mask
 *        - Indicates if the ephemeris or almanac for a satellite
 *          is to be deleted
 *        - Valid values:
 *           - 0x01 - DELETE_EPHEMERIS
 *           - 0x02 - DELETE_ALMANAC
 *
 */

typedef struct
{
    uint16_t    id;
    uint32_t   system;
    uint8_t    mask;
}loc_SV;

/**
 * This structure contains the elements of Delete LOC SV Info
 *
 * \param len
 *        - Number of sets of the following elements in struct loc_SV:
 *           - gnssSvId
 *           - system
 *           - deleteSvInfoMask
 *
 * \param pSV
 *        - Pointer to struct loc_SV. See \ref loc_SV for more information
 *
 */

typedef struct
{
    uint8_t    len;
    loc_SV      *pSV;
}loc_SVInfo;

/**
 * This structure contains the GNSS data
 *
 * \param mask
 *        - Mask for the GNSS data that is to be deleted
 *        - Valid values:
 *           - QMI_LOC_MASK_DELETE_GPS_SVDIR (0x00000001) - Mask to
 *             delete GPS SVDIR
 *           - QMI_LOC_MASK_DELETE_GPS_SVSTEER (0x00000002) - Mask to
 *             delete GPS SVSTEER 
 *           - QMI_LOC_MASK_DELETE_GPS_TIME (0x00000004) - Mask to
 *             delete GPS time
 *           - QMI_LOC_MASK_DELETE_GPS_ALM_CORR (0x00000008) - Mask to
 *             delete almanac correlation
 *           - QMI_LOC_MASK_DELETE_GLO_SVDIR (0x00000010) - Mask to
 *             delete GLONASS SVDIR
 *           - QMI_LOC_MASK_DELETE_GLO_SVSTEER (0x00000020) - Mask to
 *             delete GLONASS SVSTEER
 *           - QMI_LOC_MASK_DELETE_GLO_TIME (0x00000040) - Mask to
 *             delete GLONASS time
 *           - QMI_LOC_MASK_DELETE_GLO_ALM_CORR (0x00000080) - Mask to
 *             delete GLONASS almanac correlation
 *           - QMI_LOC_MASK_DELETE_SBAS_SVDIR (0x00000100) - Mask to
 *             delete SBAS SVDIR
 *           - QMI_LOC_MASK_DELETE_SBAS_SVSTEER (0x00000200) - Mask to
 *             delete SBAS SVSTEER
 *           - QMI_LOC_MASK_DELETE_POSITION (0x00000400) - Mask to
 *             delete position estimate
 *           - QMI_LOC_MASK_DELETE_TIME (0x00000800) - Mask to
 *             delete time estimate
 *           - QMI_LOC_MASK_DELETE_IONO (0x00001000) - Mask to
 *             delete IONO
 *           - QMI_LOC_MASK_DELETE_UTC (0x00002000) - Mask to
 *             delete UTC estimate
 *           - QMI_LOC_MASK_DELETE_HEALTH (0x00004000) - Mask to
 *             delete SV health record
 *           - QMI_LOC_MASK_DELETE_SADATA (0x00008000) - Mask to
 *             delete SADATA
 *           - QMI_LOC_MASK_DELETE_RTI (0x00010000) - Mask to
 *             delete RTI
 *           - QMI_LOC_MASK_DELETE_SV_NO_EXIST (0x00020000) - Mask to
 *             delete SV_NO_EXIST
 *           - QMI_LOC_MASK_DELETE_FREQ_BIAS_EST (0x00040000) - Mask to
 *             delete frequency bias estimate
 *           - QMI_LOC_MASK_DELETE_BDS_SVDIR (0x00080000) - Mask to
 *             delete BDS SVDIR
 *           - QMI_LOC_MASK_DELETE_BDS_SVSTEER (0x00100000) - Mask to
 *             delete BDS SVSTEER
 *           - QMI_LOC_MASK_DELETE_BDS_TIME (0x00200000) - Mask to
 *             delete BDS time
 *           - QMI_LOC_MASK_DELETE_BDS_ALM_CORR (0x00400000) - Mask to
 *             delete BDS almanac correlation
 *           - QMI_LOC_MASK_DELETE_GNSS_SV_BLACKLIST_GPS (0x00800000) -
 *             Mask to delete GNSS SV blacklist GPS
 *           - QMI_LOC_MASK_DELETE_GNSS_SV_BLACKLIST_GLO (0x01000000) -
 *             Mask to delete GNSS SV blacklist GLO
 *           - QMI_LOC_MASK_DELETE_GNSS_SV_BLACKLIST_BDS (0x02000000) -
 *             Mask to delete GNSS SV blacklist BDS
 *
 */
 
typedef struct
{
    uint64_t   mask;
}loc_GnssData;

/**
 * This structure contains the cell database
 *
 * \param mask
 *        - Mask for the cell database assistance data that is to be deleted
 *        - Valid values:
 *           - 0x00000001 - DELETE_CELLDB_POS
 *           - 0x00000002 - DELETE_CELLDB_LATEST_GPS_POS
 *           - 0x00000004 - DELETE_CELLDB_OTA_POS
 *           - 0x00000008 - DELETE_CELLDB_EXT_REF_POS
 *           - 0x00000010 - DELETE_CELLDB_TIMETAG
 *           - 0x00000020 - DELETE_CELLDB_CELLID
 *           - 0x00000040 - DELETE_CELLDB_CACHED_CELLID
 *           - 0x00000080 - DELETE_CELLDB_LAST_SRV_CELL
 *           - 0x00000100 - DELETE_CELLDB_CUR_SRV_CELL
 *           - 0x00000200 - DELETE_CELLDB_NEIGHBOR_INFO
 *
 */

typedef struct
{
    uint32_t   mask;
}loc_CellDb;

/**
 * This structure contains the clock Info
 *
 * \param mask
 *        - Mask for the clock information assistance data that is to be deleted
 *        - Valid bitmasks:
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_TIME_EST (0x00000001) - Mask to
 *             delete time estimate from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_FREQ_EST (0x00000002) - Mask to
 *             delete frequency estimate from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_WEEK_NUMBER (0x00000004) - Mask to
 *             delete week number from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_RTC_TIME (0x00000008) - Mask to
 *             delete RTC time from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_TIME_TRANSFER (0x00000010) - Mask to
 *             delete time transfer from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GPSTIME_EST (0x00000020) - Mask to
 *             delete GPS time estimate from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GLOTIME_EST (0x00000040) - Mask to
 *             delete GLONASS time estimate from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GLODAY_NUMBER (0x00000080) - Mask to
 *             delete GLONASS day number from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GLO4YEAR_NUMBER (0x00000100) - Mask to
 *             delete GLONASS four year number from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GLO_RF_GRP_DELAY (0x00000200) - Mask to
 *             delete GLONASS RF GRP delay from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_DISABLE_TT (0x00000400) - Mask to
 *             delete disable TT from clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GG_LEAPSEC (0x00000800) - Mask to
 *             delete a BDS time estimate from the clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GG_GGTB (0x00001000) - Mask to
 *             delete a BDS time estimate from the clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_BDSTIME_EST (0x00002000) - Mask to
 *             delete a BDS time estimate from the clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_GB_GBTB (0x00004000) - Mask to delete
 *             Glonass-to-BDS time bias-related information from the clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_BG_BGTB (0x00008000) - Mask to delete
 *             BDS-to-GLONASS time bias-related information from the clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_BDSWEEK_NUMBER (0x00010000) - Mask to
 *             delete the BDS week number from the clock information
 *           - QMI_LOC_MASK_DELETE_CLOCK_INFO_BDS_RF_GRP_DELAY (0x00020000) - Mask to
 *             delete the BDS RF GRP delay from the clock information
 *
 */

typedef struct
{
    uint32_t   mask;
}loc_ClkInfo;

/**
 * This structure contains the BDS SV Info
 *
 * \param id
 *        - SV ID of the satellite whose data is to be deleted.
 *           - Range for BDS: 201 to 237
 *
 * \param mask
 *        - Indicates if the ephemeris or almanac for a satellite is to be deleted
 *        - Valid values:
 *           - QMI_LOC_MASK_DELETE_EPHEMERIS (0x01) - Delete ephemeris for the satellite
 *           - QMI_LOC_MASK_DELETE_ALMANAC (0x02) - Delete almanac for the satellite
 *
*/

typedef struct
{
    uint16_t    id;
    uint8_t    mask;
}loc_BdsSV;

/**
 * This structure contains the number of sets of the BDS SVN Info
 *
 * \param len
 *        - Number of sets of the following elements:
 *           - gnssSvId
 *           - deleteSvInfoMask
 *
 * \param pSV
 *        - Pointer to struct loc_BdsSV. See \ref loc_BdsSV for more information
 *
*/

typedef struct
{
    uint8_t    len;
    loc_BdsSV   *pSV;
}loc_BdsSVInfo;

/**
 *  This structure contains the Parameter for RegisterEvents
 *
 *  \param  eventRegister
 *          - Specifies the events that the control point is interested
 *          in receiving.
 *          -Values
 *             - 0x00000001 - to receive position report event
 *                            indications
 *             - 0x00000002 - to receive satellite report event
 *                            indications
 *             - 0x00000004 - to receive NMEA reports for position
 *                            and satellites in view
 *             - 0x00000008 - to receive NI Notify/Verify request
 *                            event indications
 *             - 0x00000010 - to receive time injection request
 *                            event indications.
 *             - 0x00000020 - to receive predicted orbits request
 *                            event indications.
 *             - 0x00000040 - to receive position injection request
 *                            event indications.
 *             - 0x00000080 - to receive fix session status report
 *                            event indications.
 *             - 0x00000200 - to receive Wi-Fi position request
 *                            event indications.
 *             - 0x00000400 - to receive notifications from the
 *                            location engine indicating its readiness
 *                            to accept data from the sensors
 *             - 0x00000800 - to receive time sync requests from
 *                            the GPS engine. Time sync enables the
 *                            GPS engine to synchronize its clock
 *                            with the sensor processor’s clock.
 *             - 0x00001000 - to receive Stationary Position Indicator (SPI)
 *                            streaming report indications.
 *             - 0x00002000 - to receive location server requests.
 *                            These requests are generated when
 *                            the service wishes to establish a
 *                            connection with a location server.
 *             - 0x00004000 - to receive notifications related
 *                            to network-initiated Geofences. These
 *                            events notify the client when a network-initiated
 *                            Geofence is added, deleted, or edited.
 *             - 0x00008000 - to receive Geofence alerts. These alerts
 *                            are generated to inform the client of
 *                            the changes that may affect a Geofence,
 *                            e.g., if GPS is turned off or if the
 *                            network is unavailable.
 *             - 0x00010000 - to receive notifications when a Geofence
 *                            is breached. These events are generated
 *                            when a UE enters or leaves the perimeter
 *                            of a Geofence. This breach report is for
 *                            a single Geofence.
 *             - 0x00020000 - to register for pedometer control requests
 *                            from the location engine. The location
 *                            engine sends this event to control the
 *                            injection of pedometer reports.
 *             - 0x00040000 - to register for motion data control requests
 *                            from the location engine. The location
 *                            engine sends this event to control the
 *                            injection of motion data.
 *             - 0x00080000 - to receive notification when a batch is full.
 *                            The location engine sends this event to
 *                            notify of Batch Full for ongoing batching
 *                            session.
 *             - 0x00100000 - to receive position report indications
 *                            along with an ongoing batching session.
 *                            The location engine sends this event to
 *                            notify the batched position report while
 *                            a batching session is ongoing.
 *             - 0x00200000 - to receive Wi-Fi Access Point (AP) data
 *                            inject request event indications.
 *             - 0x00400000 - to receive notifications when a Geofence
 *                            is breached. These events are generated
 *                            when a UE enters or leaves the perimeter
 *                            of a Geofence. This breach notification
 *                            is for multiple Geofences. Breaches from
 *                            multiple Geofences are all batched and
 *                            sent in the same notification.
 *             - 0x00800000 - to receive notifications from the location
 *                            engine indicating its readiness to accept
 *                            vehicle data (vehicle accelerometer,
 *                            vehicle angular rate, vehicle odometry, etc.).
 *             - 0x01000000 - to receive system clock and satellite
 *                            measurement report events (system clock,
 *                            SV time, Doppler, etc.).
 *             - 0x02000000 - to receive satellite position reports
 *                            as polynomials.
 * \param Tlvresult
 *        - Pack result.
 *
 */
 
typedef struct{
    uint64_t eventRegister;
    uint16_t Tlvresult;
}pack_loc_EventRegister_t;

/**
 * This structure contains Event Register unpack
 *
 * \param Tlvresult
 *        - Unpack result.
 *
 */

typedef struct{
    uint16_t Tlvresult;
}unpack_loc_EventRegister_t;


/**
 * Event Register pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_loc_EventRegister(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_loc_EventRegister_t     *reqArg
);

/**
 * Event Register unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_EventRegister(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_EventRegister_t *pOutput
);

/**
 *  This structure contains the Parameter External Power Source State pack.
 *
 *  \param  extPowerState
 *          - Specifies the Power state; injected by the control
 *            point.
 *          - Values
 *             - 0 - Device is not connected to an external power
 *                   source
 *             - 1 - Device is connected to an external power source
 *             - 2 - Unknown external power state
 * \param Tlvresult
 *        - Pack result.
 */


typedef struct{
    uint32_t extPowerState;
    uint16_t Tlvresult;
}pack_loc_SetExtPowerState_t;

/**
 * This structure contains Set Ext Power State unpack
 *
 * \param Tlvresult
 *        - Unpack result.
 */

typedef struct{
    uint16_t Tlvresult;
}unpack_loc_SetExtPowerState_t;


/**
 * Set Ext Power State pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_loc_SetExtPowerState(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_loc_SetExtPowerState_t *reqArg
);

/**
 * Set Ext Power State unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_SetExtPowerState(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_SetExtPowerState_t *pOutput
);

/**
 * This structure contains the LOC Start pack
 *
 * \param SessionId
 *        - ID of the session as identified by the control point.
 *        - Range: 0 to 255
 *
 * \param pRecurrenceType
 *        - Specifies the type of session in which the control point
 *        is interested.
 *        - Defaults to SINGLE.
 *        -Values
 *           - 1 - Request periodic position fixes
 *           - 2 - Request a single position fix
 *
 * \param pHorizontalAccuracyLvl
 *        - Specifies the horizontal accuracy level required by the
 *        control point.
 *        - Defaults to LOW
 *        - Values
 *           - 1 - Low accuracy
 *           - 2 - Medium accuracy
 *           - 3 - High accuracy
 *
 * \param pIntermediateReportState
 *        - Specifies if the control point is interested
 *        in receiving intermediate reports.
 *        - ON by default.
 *        - Values
 *          - 1 - Intermediate reports are turned on
 *          - 2 - Intermediate reports are turned off
 *
 * \param appVersionValid
 *        - Specifies whether the application version string contains a
 *        valid value
 *
 *        - 0x00 (FALSE) – Application version string is invalid
 *        - 0x01 (TRUE) – Application version string is valid
 *
 * \param LocApplicationInfo
 *        - LOC Application Parameters
 *        - See \ref loc_LocApplicationInfo for more information
 *
 * \param pConfigAltitudeAssumed
 *        - Configuration for Altitude Assumed Info in GNSS SV
 *        Info Event
 *        - Defaults to ENABLED.
 *        - Values
 *          - 1 - Enable Altitude Assumed information in GNSS SV
 *          Info Event
 *          - 2 - Disable Altitude Assumed information in GNSS SV
 *          Info Event
 */

typedef struct{
    uint8_t    SessionId;
    uint32_t   *pRecurrenceType;
    uint32_t   *pHorizontalAccuracyLvl;
    uint32_t   *pIntermediateReportState;
    uint32_t   *pMinIntervalTime;
    loc_LocApplicationInfo *pApplicationInfo;
    uint32_t   *pConfigAltitudeAssumed;
    uint16_t    Tlvresult;
}pack_loc_Start_t;

/**
 * This structure contains Start LOC unpack
 *
 * \param Tlvresult
 *        - Unpack result.
 *
 */

typedef struct{
    uint16_t Tlvresult;
}unpack_loc_Start_t;


/**
 * LOC Start pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_loc_Start(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_loc_Start_t *reqArg
);

/**
 * Loc Start  unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_Start(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_Start_t *pOutput
);


/**
 * This structure contains Stop LOC pack
 *
 * \param sessionId
 *        - ID of the session as identified by the control point.
 *        - Range: 0 to 255
 *
 * \param Tlvresult
 *        - Unpack result.
 */


typedef struct{
    uint8_t     SessionId;
    uint16_t    Tlvresult;
}pack_loc_Stop_t;

/**
 * This structure contains Stop LOC unpack
 *
 * \param Tlvresult
 *        - Unpack result.
 */

typedef struct{
    uint16_t Tlvresult;
}unpack_loc_Stop_t;


/**
 * Loc Stop pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_loc_Stop(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_loc_Stop_t *reqArg
);

/**
 * Loc Stop unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_Stop(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_Stop_t *pOutput
);

/**
 * This structure contains Set Operation Mode pack
 *
 * \param mode
 *        - 0 - Default Mode.
 *
 * \param Tlvresult
 *        - Pack result.
 *
 */

typedef struct{
    uint32_t mode;
    uint16_t Tlvresult;
}pack_loc_SetOperationMode_t;

/**
 * This structure contains Set Operation Mode unpack
 *
 * \param Tlvresult
 *        - Unpack result.
 */

typedef struct{
    uint16_t Tlvresult;
}unpack_loc_SetOperationMode_t;


/**
 * Set Operation Mode pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_loc_SetOperationMode(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_loc_SetOperationMode_t *reqArg
);

/**
 * Set Operation Mode unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_SetOperationMode(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_SetOperationMode_t *pOutput
);


/**
 * This structure contains LOC delete assist data pack
 *
 * \param pSVInfo
 *        - Pointer to struct loc_SVInfo. See \ref loc_SVInfo for more information
 *
 * \param pGnssData
 *        - Pointer to struct loc_GnssData. See \ref loc_GnssData for more information
 *
 * \param pCellDb
 *        - Pointer to struct loc_CellDb. See \ref loc_CellDb for more information
 *
 * \param pClkInfo
 *        - Pointer to struct loc_ClkInfo. See \ref loc_ClkInfo for more information
 *
 * \param pBdsSVInfo
 *        - Pointer to struct loc_BdsSVInfo. See \ref loc_BdsSVInfo for more information
 *
 * \param Tlvresult
 *        - Pack delete assist data request result.
 */

typedef struct{
    loc_SVInfo *pSVInfo;
    loc_GnssData *pGnssData;
    loc_CellDb *pCellDb;
    loc_ClkInfo *pClkInfo;
    loc_BdsSVInfo *pBdsSVInfo;
    uint16_t Tlvresult;
}pack_loc_Delete_Assist_Data_t;

/**
 * This structure contains LOC delete assist data unpack
 *
 * \param Tlvresult
 *        - Unpack result.
 */
 
typedef struct{
    uint16_t Tlvresult;
}unpack_loc_Delete_Assist_Data_t;



/**
 * Delete Assistant Data pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_loc_DeleteAssistData(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_loc_Delete_Assist_Data_t *reqArg
);

/**
 * Delete Assistant Data unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_DeleteAssistData(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_Delete_Assist_Data_t *pOutput
);

/**
 *  This structure contains Dilution of precision associated with this position.
 *
 *  \param  PDOP
 *          - Position dilution of precision.
 *          - Range - 1 (highest accuracy) to 50 (lowest accuracy)
 *          - PDOP = square root of (Square of HDOP + Square of VDOP2 )
 *  \param  HDOP
 *          - Horizontal dilution of precision.
 *          - Range - 1 (highest accuracy) to 50 (lowest accuracy)
 *  \param  VDOP
 *          - Vertical dilution of precision.
 *          - Range-  1 (highest accuracy) to 50 (lowest accuracy)
 *
 */

typedef struct {
     uint32_t    PDOP;
     uint32_t    HDOP;
     uint32_t    VDOP;
}loc_precisionDilution;

/**
 *  This structure contains Sensor Data Usage info.
 *
 *  \param  usageMask
 *          - Specifies which sensors were used in calculating the position
 *            in the position report.
 *        - Value
 *          - 0x00000001 - Accelerometer used
 *          - 0x00000002 - Gyroscope used
 *  \param  aidingIndicatorMask
 *          - Specifies which results were aided by sensors.
 *        - Value
 *         - 0x00000001 - AIDED_HEADING
 *         - 0x00000002 - AIDED_SPEED
 *         - 0x00000004 - AIDED_POSITION
 *         - 0x00000008 - AIDED_VELOCITY
 *
 */

typedef struct {
     uint32_t     usageMask;
     uint32_t     aidingIndicatorMask;
}loc_sensorDataUsage;

/**
 *  This structure contains SVs Used to Calculate the Fix.
 *
 *  \param  gnssSvUsedList_len
 *          - Number of sets of gnssSvUsedList
 *  \param  pGnssSvUsedList
 *          - Entry in the list contains the SV ID of a satellite
 *            used for calculating this position report.
 *          - Following information is associated with each SV ID:
 *            - GPS - 1 to 32
 *            - SBAS - 33 to 64
 *            - GLONASS - 65 to 96
 *            - QZSS - 193 to 197
 *            - BDS - 201 to 237
 *
 */

typedef struct {
     uint8_t     gnssSvUsedList_len ;
     uint16_t     gnssSvUsedList[LOC_UINT8_MAX_STRING_SZ];
}loc_svUsedforFix;

/**
 *  This structure contains GPS Time info.
 *
 *  \param  gpsWeek
 *          - Current GPS week as calculated from midnight, Jan. 6, 1980.
 *          - Units - Weeks
 *  \param  gpsTimeOfWeekMs
 *          - Amount of time into the current GPS week.
 *          - Units - Milliseconds
 *
 */

typedef struct {
     uint16_t     gpsWeek;
     uint32_t    gpsTimeOfWeekMs;
}loc_gpsTime;

/**
 *  This structure contains Event Position Report Indication unpack
 *
 *  \param  sessionStatus
 *          - Values
 *            - 0 - Session was successful
 *            - 1 - Session is still in progress; further position reports will be generated
 *                  until either the fix criteria specified by the client are met or the client
 *                  response timeout occurs.
 *            - 2 - Session failed..
 *            - 3 - Fix request failed because the session timed out.
 *            - 4 - Fix request failed because the session was ended by the user.
 *            - 5 - Fix request failed due to bad parameters in the request.
 *            - 6 - Fix request failed because the phone is offline.
 *            - 7 - Fix request failed because the engine is locked
 *  \param  sessionId
 *             - ID of the session that was specified in the Start request
 *             - Range - 0 to 255
 *  \param  pLatitude
 *             - Latitude (specified in WGS84 datum)
 *             - Type - Floating point
 *             - Units - Degrees
 *             - Range - -90.0 to 90.0
 *             - Positive values indicate northern latitude
 *             - Negative values indicate southern latitude
 *  \param  pLongitude
 *             - Longitude (specified in WGS84 datum)
 *             - Type - Floating point
 *             - Units - Degrees
 *             - Range - -180.0 to 180.0
 *             - Positive values indicate eastern latitude
 *             - Negative values indicate western latitude
 *   \param  pHorUncCircular
 *             - Horizontal position uncertainty.
 *             - Units - Meters
 *   \param  pHorUncEllipseSemiMinor
 *             - Semi-minor axis of horizontal elliptical uncertainty.
 *             - Units - Meters
 *   \param  pHorUncEllipseSemiMajor
 *             - Semi-major axis of horizontal elliptical uncertainty.
 *             - Units: Meters
 *   \param  pHorUncEllipseOrientAzimuth
 *             - Elliptical horizontal uncertainty azimuth of orientation.
 *             - Units - Decimal degrees
 *             - Range - 0 to 180
 *   \param  pHorConfidence
 *             - Horizontal uncertainty confidence.
 *             - If both elliptical and horizontal uncertainties
 *               are specified in this message, the confidence
 *               corresponds to the elliptical uncertainty.
 *             - Units - Percentage
 *             - Range 0-99
 *   \param  pHorReliability
 *          - Values
 *            - 0 - Location reliability is not set.
 *            - 1 - Location reliability is very low; use it at
 *                  your own risk
 *            - 2 - Location reliability is low; little or no
 *                  cross-checking is possible.
 *            - 3 - Location reliability is medium; limited cross-check passed
 *            - 4 - Location reliability is high; strong cross-check passed
 *   \param  pSpeedHorizontal
 *             - Horizontal speed.
 *             - Units - Meters/second
 *   \param  pSpeedUnc
 *             - 3-D Speed uncertainty.
 *             - Units - Meters/second.
 *   \param  pAltitudeWrtEllipsoid
 *             - Altitude With Respect to WGS84 Ellipsoid.
 *             - Units - Meters
 *             - Range -500 to 15883
 *   \param  pAltitudeWrtMeanSeaLevel
 *             - Altitude With Respect to Sea Level.
 *             - Units - Meters
 *   \param  pVertUnc
 *             - Vertical uncertainty.
 *             - Units - Meters
 *   \param  pVertConfidence
 *             - Vertical uncertainty confidence.
 *             - Units - Percentage
 *             - Range 0 to 99
 *   \param  pVertReliability
 *          - Values
 *             - 0 - Location reliability is not set.
 *             - 1 - Location reliability is very low;
 *                   use it at your own risk.
 *             - 2 - Location reliability is low; little or no
 *                   cross-checking is possible
 *             - 3 - Location reliability is medium; limited
 *                   cross-check passed
 *             - 4 - Location reliability is high; strong
 *                   cross-check passed
 *   \param  pSpeedVertical
 *             - Vertical speed.
 *             - Units - Meters/second
 *   \param  pHeading
 *             - Heading.
 *             - Units - Degree
 *             - Range 0 to 359.999
 *   \param  pHeadingUnc
 *             - Heading uncertainty.
 *             - Units - Degree
 *             - Range 0 to 359.999
 *   \param  pMagneticDeviation
 *             - Difference between the bearing to true north and the
 *               bearing shown on a magnetic compass. The deviation is
 *               positive when the magnetic north is east of true north.
 *   \param  pTechnologyMask
 *          - Values
 *             - 0x00000001 - Satellites were used to generate the fix
 *             - 0x00000002 - Cell towers were used to generate the fix
 *             - 0x00000004 - Wi-Fi access points were used to generate the fix
 *             - 0x00000008 - Sensors were used to generate the fix
 *             - 0x00000010 - Reference Location was used to generate the fix
 *             - 0x00000020 - Coarse position injected into the location engine
 *                            was used to generate the fix
 *             - 0x00000040 - AFLT was used to generate the fix
 *             - 0x00000080 - GNSS and network-provided measurements were
 *                            used to generate the fix
 *  \param  -pPrecisionDilution
 *        - See \ref loc_precisionDilution for more information
 *   \param  pTimestampUtc
 *             - UTC timestamp
 *             - Units - Milliseconds since Jan. 1, 1970
 *   \param  pLeapSeconds
 *             - Leap second information. If leapSeconds is not available,
 *               timestampUtc is calculated based on a hard-coded value
 *               for leap seconds.
 *             - Units - Seconds
 *  \param  -pGpsTime
 *        - See \ref loc_gpsTime for more information
 *   \param  pTimeUnc
 *             - Time uncertainty.
 *             - Units - Milliseconds
 *   \param  pTimeSrc
 *          - Values
 *             - 0 - Invalid time.
 *             - 1 - Time is set by the 1X system.
 *             - 2 - Time is set by WCDMA/GSM time tagging.
 *             - 3 - Time is set by an external injection.
 *             - 4 - Time is set after decoding over-the-air GPS navigation
 *                   data from one GPS satellite.
 *             - 5 - Time is set after decoding over-the-air GPS navigation
 *                   data from multiple satellites.
 *             - 6 - Both time of the week and the GPS week number
 *                   are known.
 *             - 7 - Time is set by the position engine after the
 *                   fix is obtained
 *             - 8 - Time is set by the position engine after performing SFT,
 *                   this is done when the clock time uncertainty is large.
 *             - 9 - Time is set after decoding GLO satellites.
 *             - 10- Time is set after transforming the GPS to GLO time
 *             - 11- Time is set by the sleep time tag provided by
 *                   the WCDMA network.
 *             - 12- Time is set by the sleep time tag provided by the
 *                   GSM network
 *             - 13- Source of the time is unknown
 *             - 14- Time is derived from the system clock (better known
 *                   as the slow clock); GNSS time is maintained
 *                   irrespective of the GNSS receiver state
 *             - 15- Time is set after decoding QZSS satellites.
 *             - 16- Time is set after decoding BDS satellites.
 *  \param  -pSensorDataUsage
 *        - See \ref loc_sensorDataUsage for more information
 *   \param  pFixId
 *             - Fix count for the session. Starts with 0 and increments
 *               by one for each successive position report for a
 *               particular session.
 *  \param  -pSvUsedforFix
 *        - See \ref loc_svUsedforFix for more information
 *   \param  pAltitudeAssumed
 *             - Indicates whether altitude is assumed or calculated.
 *          - Value
 *             - 0x00 - Altitude is calculated
 *             - 0x01 - Altitude is assumed
 *
 */

typedef struct{
    uint32_t                    sessionStatus;
    uint8_t                     sessionId;
    uint64_t                    *pLatitude;
    uint64_t                    *pLongitude;
    uint32_t                    *pHorUncCircular;
    uint32_t                    *pHorUncEllipseSemiMinor;
    uint32_t                    *pHorUncEllipseSemiMajor;
    uint32_t                    *pHorUncEllipseOrientAzimuth;
    uint8_t                     *pHorConfidence;
    uint32_t                    *pHorReliability;
    uint32_t                    *pSpeedHorizontal;
    uint32_t                    *pSpeedUnc;
    uint32_t                    *pAltitudeWrtEllipsoid;
    uint32_t                    *pAltitudeWrtMeanSeaLevel;
    uint32_t                    *pVertUnc;
    uint8_t                     *pVertConfidence;
    uint32_t                    *pVertReliability;
    uint32_t                    *pSpeedVertical;
    uint32_t                    *pHeading;
    uint32_t                    *pHeadingUnc;
    uint32_t                    *pMagneticDeviation;
    uint32_t                    *pTechnologyMask;
    loc_precisionDilution       *pPrecisionDilution;
    uint64_t                    *pTimestampUtc;
    uint8_t                     *pLeapSeconds;
    loc_gpsTime                 *pGpsTime;
    uint32_t                    *pTimeUnc;
    uint32_t                    *pTimeSrc;
    loc_sensorDataUsage         *pSensorDataUsage;
    uint32_t                    *pFixId;
    loc_svUsedforFix            *pSvUsedforFix;
    uint8_t                     *pAltitudeAssumed;
    uint16_t Tlvresult;
}unpack_loc_PositionRpt_Ind_t;

/**
 * Loc Position Report Indication unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_loc_PositionRpt_Ind(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_loc_PositionRpt_Ind_t *pOutput
);

#endif
