/**
 * \ingroup pack
 *
 * \file dms.h
 */
#include "common.h"
#include <stdint.h>
#define DMS_UINT8_MAX_STRING_SZ 255
#define DMS_MAX_CUST_ID_LEN   64
#define DMS_MAX_CUST_VALUE_LEN 8
#define DMS_IMGDETAILS_LEN    16

/**
 * @param modelid device model id
 */
typedef struct{
    char modelid[255];
    uint16_t  Tlvresult;
} unpack_dms_GetModelID_t;

typedef struct{
    char imsi[255];
    uint16_t  Tlvresult;
} unpack_dms_GetIMSI_t;

#define SLQSFWINFO_MODELID_SZ     20
#define SLQSFWINFO_BOOTVERSION_SZ 85
#define SLQSFWINFO_APPVERSION_SZ  85
#define SLQSFWINFO_SKU_SZ         15
#define SLQSFWINFO_PACKAGEID_SZ   85
#define SLQSFWINFO_CARRIER_SZ     20
#define SLQSFWINFO_PRIVERSION_SZ  10
#define SLQSFWINFO_CUR_CARR_NAME  17
#define SLQSFWINFO_CUR_CARR_REV   13

#define MAX_BUILD_ID_LEN  255
#define UNIQUE_ID_LEN     16
#define SLQS_MAX_DYING_GASP_CFG_SMS_CONTENT_LENGTH 160
#define SLQS_MAX_DYING_GASP_CFG_SMS_NUMBER_LENGTH 20

/**
 * @param modelid_str Mode ID String.
 * @param bootversion_str Boot Version.
 * @param appversion_str Application Version String.
 * @param sku_str SKU String.
 * @param packageid_str
 *        - Package ID String.
 *        - deprecated on EM/MC74xx(9x30) devices
 * @param carrier_str Carrier String.
 * @param priversion_str PRI Version String.
 * @param priversion_str PRI Version String.
 * @param cur_carr_name Current Carrier Name String.
 * @param cur_carr_rev Current Carrier Revision String.
 * @param Tlvresult Tlv Result.
 */

typedef struct {
    char modelid_str[SLQSFWINFO_MODELID_SZ];
    char bootversion_str[SLQSFWINFO_BOOTVERSION_SZ];
    char appversion_str[SLQSFWINFO_APPVERSION_SZ];
    char sku_str[SLQSFWINFO_SKU_SZ];
    char packageid_str[SLQSFWINFO_PACKAGEID_SZ];
    char carrier_str[SLQSFWINFO_CARRIER_SZ];
    char priversion_str[SLQSFWINFO_PRIVERSION_SZ];
    char cur_carr_name[SLQSFWINFO_CUR_CARR_NAME];
    char cur_carr_rev[SLQSFWINFO_CUR_CARR_REV];
    uint16_t  Tlvresult;
} unpack_dms_GetFirmwareInfo_t;

/**
 * @param OperationMode operating mode
 * @param OfflineReason offline reason
 * @param HardwareControlledMode hardware restricted mode
 */
typedef struct {
    uint32_t OperationMode;
    uint32_t OfflineReason;
    uint32_t HardwareControlledMode;
    uint16_t  Tlvresult;
} unpack_dms_GetPower_t;

/**
 * get IMSI pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetIMSI(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );

/**
 * get model id unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetIMSI(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_GetIMSI_t *pOutput
        );

/**
 * get model id pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetModelID(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );

/**
 * get model id unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetModelID(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_GetModelID_t *pOutput
        );

/**
 * get firmware info pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetFirmwareInfo(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );

/**
 * get firmware info unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetFirmwareInfo(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_GetFirmwareInfo_t *pOutput
        );

/**
 * get power pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetPower(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
        );

/**
 * get power unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetPower(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_GetPower_t *pOutput
        );

/**
 * @param esn Electronic Serial Number of the device
 * @param imei_no International Mobile Equipment Identity of the device.
 * @param meid Mobile Equipment Identifier of the device.
 * @param imeisv_svn imei software version revision
 */
typedef struct{
    char esn[255];
    char imei_no[255];
    char meid[255];
    char imeisv_svn[255];
} unpack_dms_GetSerialNumbers_t;

/**
 * get serial numbers pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetSerialNumbers(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );

/**
 * get serial numbers unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetSerialNumbers(
       uint8_t *pResp,
       uint16_t respLen,
       unpack_dms_GetSerialNumbers_t *pOutput
       );

/**
 * @param hwVer hardware vesion
 */
typedef struct{
    char hwVer[255];
} unpack_dms_GetHardwareRevision_t;

/**
 * get hardware revision pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetHardwareRevision(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );
/**
 * get hardware revision unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetHardwareRevision(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_GetHardwareRevision_t *pOutput
        );

/**
 * @param bandCapability band capability
 */
typedef struct{
    uint64_t bandCapability;
} unpack_dms_SLQSGetBandCapability_t;

/**
 * get band capability pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SLQSGetBandCapability(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );

/**
 * get band capability unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SLQSGetBandCapability(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_SLQSGetBandCapability_t *pOutput
        );
/**
 * @param maxTxChannelRate Maximum Tx transmission rate in bits per second.
 * @param maxRxChannelRate Maximum Rx transmission rate in bits per second
 * @param dataServiceCaCapability data service capability
 * @param simCapability SIM Capability
 * @param radioIfacesSize radio interface length
 * @param RadioIfaces radio interfaces
 */
typedef struct{
    uint32_t  maxTxChannelRate;
    uint32_t  maxRxChannelRate;
    uint32_t  dataServiceCaCapability;
    uint32_t  simCapability;
    uint32_t  radioIfacesSize;
    uint8_t        RadioIfaces[255];
} unpack_dms_GetDeviceCapabilities_t;

/**
 * get device capability pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetDeviceCapabilities(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen,
        void     *reqArg
        );
/**
 * get device capability unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetDeviceCapabilities(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_GetDeviceCapabilities_t *pOutput
        );
/**
 * @param amssstring AMSS revision string
 * @param bootstring boot code revision string
 * @param pristring PRI revision string
 */
typedef struct{
    uint8_t    amssSize;
    char    AMSSString[DMS_UINT8_MAX_STRING_SZ];
    uint8_t    bootSize;
    char    BootString[DMS_UINT8_MAX_STRING_SZ];
    uint8_t    priSize;
    char    PRIString[DMS_UINT8_MAX_STRING_SZ];
    uint16_t  Tlvresult;
}unpack_dms_GetFirmwareRevisions_t;
        
        
/**
 * get Firmware Revisions pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
    
int pack_dms_GetFirmwareRevisions(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * get Firmware Revisions unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetFirmwareRevisions(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetFirmwareRevisions_t *pOutput
);

typedef struct{
    uint8_t    amssSize;
    char    AMSSString[DMS_UINT8_MAX_STRING_SZ];
    uint16_t  Tlvresult;
}unpack_dms_GetFirmwareRevision_t;
        
        
/**
 * get Firmware Revision pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
    
int pack_dms_GetFirmwareRevision(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
    
);

/**
 * get Firmware Revision unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetFirmwareRevision(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetFirmwareRevision_t *pOutput
);


typedef struct{
    uint8_t    esnSize;
    char    ESNString[DMS_UINT8_MAX_STRING_SZ];
    uint8_t    imeiSize;
    char    IMEIString[DMS_UINT8_MAX_STRING_SZ];
    uint8_t    meidSize;
    char    MEIDString[DMS_UINT8_MAX_STRING_SZ];
    uint8_t imeiSvnSize;
    char    ImeiSvnString[DMS_UINT8_MAX_STRING_SZ];
    uint16_t  Tlvresult;
}unpack_dms_GetDeviceSerialNumbers_t;


/**
 * get Device Serial Number pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetDeviceSerialNumbers(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);


/**
 * get Device Serial Number unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetDeviceSerialNumbers(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetDeviceSerialNumbers_t *pOutput
);

typedef struct{
    uint8_t    u8PRLPreference;
    uint16_t   u16PRLVersion;
    uint16_t  Tlvresult;
}unpack_dms_GetPRLVersion_t;

/**
 * Get PRL Versions pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetPRLVersion(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get PRL Versions unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetPRLVersion(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetPRLVersion_t *pOutput
);

/**
 *  @param  source
 *          - Source of timestamp\n
 *            0 - 32 kHz device clock\n
 *            1 - CDMA network\n
 *            2 - cdma2000 1xEV-DO network

 *  @param  timestamp
 *          - Count of 1.25 ms that have elapsed from the start of GPS time
 *            (Jan 6, 1980)
 */
typedef struct{
    uint16_t source;
    uint64_t timestamp;
    uint16_t  Tlvresult;
}unpack_dms_GetNetworkTime_t;

/**
 * Get Network Time pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetNetworkTime(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Network Time unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetNetworkTime(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetNetworkTime_t *pOutput
);

typedef struct{
    uint8_t voiceNumberSize;
    char    VoiceNumber[DMS_UINT8_MAX_STRING_SZ];
    uint8_t minSize;
    char    MIN[DMS_UINT8_MAX_STRING_SZ];
    uint16_t  Tlvresult;
}unpack_dms_GetVoiceNumber_t;

/**
 * Get Voice Number pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetVoiceNumber(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Voice Number unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetVoiceNumber(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetVoiceNumber_t *pOutput
);


typedef struct{
    uint8_t    stringSize;
    char    String[DMS_UINT8_MAX_STRING_SZ];
    uint16_t  Tlvresult;
}unpack_dms_GetDeviceHardwareRev_t;

/**
 * Get Hardware Revision pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetDeviceHardwareRev(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Hardware Revision unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetDeviceHardwareRev(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetDeviceHardwareRev_t *pOutput
);

typedef struct{
    char    String[DMS_UINT8_MAX_STRING_SZ];
    uint16_t  Tlvresult;
}unpack_dms_GetFSN_t;

/**
 * Get FSN pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetFSN(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get FSN unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetFSN(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetFSN_t *pOutput
);

typedef struct{
    uint32_t MaxTXChannelRate;
    uint32_t MaxRXChannelRate;
    uint32_t DataServiceCapability;
    uint32_t SimCapability;
    uint32_t RadioIfacesSize;
    uint8_t  RadioIfaces[64];
    uint16_t Tlvresult;
}unpack_dms_GetDeviceCap_t;

/**
 * Get Device Capabilities pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetDeviceCap(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Device Capabilities unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetDeviceCap(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetDeviceCap_t *pOutput
);


#define DMS_PM_ONLINE         0x00 /* Online */
#define DMS_PM_LOW            0x01 /* Low Power */
#define DMS_PM_FACTORY        0x02 /* Factory Test Mode */
#define DMS_PM_OFFLINE        0x03 /* Offline */
#define DMS_PM_RESET          0x04 /* Reset */
#define DMS_PM_SHUT_DOWN      0x05 /* Shut Down */
#define DMS_PM_PERSISTENT_LOW 0x06 /* Persistent Low Power */


typedef struct{
    uint32_t  mode;
    uint16_t  Tlvresult;
}pack_dms_SetPower_t;

typedef struct{
    uint16_t  Tlvresult;
}unpack_dms_SetPower_t;

/**
 * Set Power pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SetPower(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_SetPower_t *reqArg
);

/**
 * Set Power unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SetPower(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SetPower_t *pOutput
);

typedef struct{
    uint32_t BandCapability;
    uint16_t  Tlvresult;
}unpack_dms_GetBandCapability_t;

/**
 * Get Band Capability pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetBandCapability(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Band Capabilities unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetBandCapability(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetBandCapability_t *pOutput
);

typedef struct{
    uint8_t  USBComp[255];
    uint8_t  NumSupUSBComps;
    uint8_t  SupUSBComps;
    uint16_t  Tlvresult;
}unpack_dms_GetUSBComp_t;

/**
 * Get USB Comp pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetUSBComp(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get USB Comp unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetUSBComp(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetUSBComp_t *pOutput
);

typedef struct{
    uint8_t  USBComp;
    uint16_t  Tlvresult;
}pack_dms_SetUSBComp_t;

typedef struct{
    uint16_t  Tlvresult;
}unpack_dms_SetUSBComp_t;

/**
 * Set USB Comp pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SetUSBComp(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_SetUSBComp_t *reqArg
);

/**
 * Set USB Comp unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SetUSBComp(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SetUSBComp_t *pOutput
);



typedef struct{
    uint32_t GpsEnable;
    uint8_t  DisableIMSI;
    uint16_t  IPFamSupport;
    uint8_t  RMAutoConnect;
    uint8_t  GPSSel;
    uint8_t  SMSSupport;
    uint8_t  IsVoiceEnabled;
    uint8_t  DHCPRelayEnabled;
    uint8_t  GPSLPM;
}pack_dms_SetCustFeature_t;


typedef struct{
    uint16_t  Tlvresult;
}unpack_dms_SetCustFeature_t;

/**
 * Set Custom Feature pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SetCustFeature(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_SetCustFeature_t *reqArg
);

/**
 * Set Custom Feature unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SetCustFeature(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SetCustFeature_t *pOutput
);


typedef struct{
    uint32_t GpsEnable;
    uint8_t  DisableIMSI;
    uint16_t  IPFamSupport;
    uint8_t  RMAutoConnect;
    uint8_t  GPSSel;
    uint8_t  SMSSupport;
    uint8_t  IsVoiceEnabled;
    uint8_t  DHCPRelayEnabled;
    uint8_t  GPSLPM;
    uint16_t  Tlvresult;
}unpack_dms_GetCustFeature_t;

/**
 * Get Custom Feature pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetCustFeature(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Custom Feature unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetCustFeature(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetCustFeature_t *pOutput
);


typedef struct{
    uint16_t  Tlvresult;
}unpack_dms_SetFirmwarePreference_t;

/**
 * Set Firmware Preference pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SetFirmwarePreference(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Set Firmware Preference unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SetFirmwarePreference(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SetFirmwarePreference_t *pOutput
);

typedef struct{
    uint8_t DevCrashState;
    uint16_t Tlvresult;
}unpack_dms_GetCrashAction_t;

/**
 * Get Crash Action pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetCrashAction(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Crash Action unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetCrashAction(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetCrashAction_t *pOutput
);

typedef struct{
    uint8_t stringSize;
    char    String[DMS_UINT8_MAX_STRING_SZ];
    uint16_t Tlvresult;
}unpack_dms_GetDeviceMfr_t;

/**
 * Get Manufacture pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetDeviceMfr(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    void     *reqArg
);

/**
 * Get Manufacture unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetDeviceMfr(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetDeviceMfr_t *pOutput
);

#define DMS_SET_REPORT_ENABLE 1
#define DMS_SET_REPORT_DISABLE 0


typedef struct{
    uint8_t mode;
}pack_dms_SetEventReport_t;


typedef struct{
    uint16_t Tlvresult;
}unpack_dms_SetEventReport_t;

/**
 * Set Event Report pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SetEventReport(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_SetEventReport_t *reqArg
);

/**
 * Set Event Report unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SetEventReport(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SetEventReport_t *pOutput
);

typedef struct 
{
    uint16_t    TlvPresent;
    uint32_t   operatingMode;
}dms_OperatingModeTlv;

typedef struct 
{
    uint16_t    TlvPresent;
    uint32_t    activationStatus;
}dms_ActivationStatusTlv;

typedef struct{
    dms_ActivationStatusTlv ActivationStatusTlv;
    dms_OperatingModeTlv OperatingModeTlv;
    uint16_t Tlvresult;
}unpack_dms_SetEventReport_ind_t;

/**
 * Event Report Indication unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_dms_SetEventReport_ind(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_SetEventReport_ind_t *pOutput
);

/**
 * This structure contains UIM Get ICCID pack
 *
 * \param Tlvresult
 *        - Pack result.
 *
 */
typedef struct{
    uint16_t Tlvresult;
}pack_dms_UIMGetICCID_t;

/**
 * This structure contains Get ICCID pack
 *
 * \param stringSize
 *        - Size of String.
 *
 * \param String
 *        - ICCID String.
 *
 * \param Tlvresult
 *        - Pack result.
 *
 */
 
typedef struct{
    uint8_t stringSize;
    uint8_t String[DMS_UINT8_MAX_STRING_SZ];
    uint16_t Tlvresult;
}unpack_dms_UIMGetICCID_t;

/**
 * Packs the UIMGetICCID response message to a
 * user-provided response structure.
 * @param [in] pResp qmi response
 * @param [in] pespLen qmi response length
 * @param [out] pOutput unpacked response
 * @param [in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_UIMGetICCID(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_UIMGetICCID_t *reqArg
);

/**
 * Unpacks the UIMGetICCID response message to a
 * user-provided response structure.
 * @param [in] pResp qmi response
 * @param [in] pespLen qmi response length
 * @param [out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_UIMGetICCID(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_UIMGetICCID_t *pOutput
);

/**
 *  This structure contains customization settings set to modem pack
 *
 *  \param  cust_id
 *          - Customization ID (Maximum 64 bytes)
 *
 *  \param  value_length
 *          - length of cust_value field
 *
 *  \param  cust_value
 *          - Customization Setting Value (Maximum 8 bytes)
 *
 *  \param  Tlvresult
 *          - Pack Result
 */

typedef struct
{
    uint8_t cust_id[DMS_MAX_CUST_ID_LEN+1];
    uint16_t value_length;
    uint8_t cust_value[DMS_MAX_CUST_VALUE_LEN+1];
    uint16_t Tlvresult;
} pack_dms_SetCustFeaturesV2_t;

/**
 * Set Cust Features pack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 * @param[in] reqArg request parameter
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SetCustFeaturesV2( 
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_SetCustFeaturesV2_t *reqArg);

/**
 *  This structure contains customization settings set to modem unpack
 *
 *  \param  Tlvresult
 *          - Unpack Result
 */
typedef struct 
{
    uint16_t Tlvresult;
}unpack_dms_SetCustFeaturesV2_t;

/**
 * Set Cust features unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SetCustFeaturesV2(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SetCustFeaturesV2_t *pOutput
);

/**
 *  This structure contains which customization id or the list type want to retrieve
 *  from modem. This TLV is only applicable for 9x30 modules so far
 *
 *  \param  cust_id
 *          - Customization ID (Maximum 64 bytes)
 * 
 *  \param  list_type
 *          - list type requested
 *
 *  \param  Tlvresult
 *          - Pack Result
 *
 */
typedef struct
{
    uint8_t cust_id[DMS_MAX_CUST_ID_LEN+1];
    uint8_t list_type;
    uint16_t Tlvresult;
} pack_dms_GetCustFeaturesV2_t;

int pack_dms_GetCustFeaturesV2( 
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_GetCustFeaturesV2_t *reqArg);


/**
 *  This structure contains which customization id or the list type want to retrieve
 *  from modem. This TLV is only applicable for 9x30 modules so far
 *
 *  \param  cust_id
 *          - Customization ID (Maximum 64 bytes)
 * 
 *  \param  list_type
 *          - list type requested
 *
 */
typedef struct
{
    uint8_t cust_id[DMS_MAX_CUST_ID_LEN+1];
    uint8_t list_type;
} DMSgetCustomInput;

 /**
 *  This structure contains information about Customization Setting.
 *  This TLV is only applicable for 9x30 modules so far
 *
 *  \param  id_length
 *          - length of cust_id field
 *
 *  \param  cust_id
 *          - Customization ID (Maximum 64 bytes)
 *
 *  \param  value_length
 *          - length of cust_value field
 *
 *  \param  cust_value
 *          - Customization Setting Value (Maximum 8 bytes)
 *
 *  \param  cust_attr
 *          - Customization Setting attribute through QMI
 *              - bit 0: Values:
 *                  - 0 - read only
 *                  - 1 - read/write 
 *
 */
typedef struct
{
    uint16_t id_length;
    uint8_t cust_id[DMS_MAX_CUST_ID_LEN+1];
    uint16_t value_length;
    uint8_t cust_value[DMS_MAX_CUST_VALUE_LEN+1];
    uint16_t cust_attr;
} DMScustSettingInfo;

/**
 *  This structure contains the fields of TLV Customization Setting List. 
 *  This TLV is only applicable for 9x30 modules so far
 *
 *  \param  list_type
 *          - list type requested
 *
 *  \param  num_instances
 *          - number of instances of customization setting
 *
 *  \param  custSetting
 *          -  See \ref custSettingInfo for more information
 *
 */
typedef struct
{
    uint8_t list_type;
    uint16_t num_instances;
    DMScustSettingInfo custSetting[DMS_UINT8_MAX_STRING_SZ+1];
} DMScustSettingList;

/**
 *  This struture contains the TLV required to get the Customization Info and
 *  customization list.
 *
 *  \param[IN]  pGetCustomInput
 *          -  See \ref getCustomInput for more information
 *
 *  \param[OUT]  pCustSettingInfo
 *          -  See \ref custSettingInfo for more information
 *
 *  \param[OUT]  pCustSettingList
 *          -  See \ref custSettingList for more information
 *
 */
typedef struct
{
    DMSgetCustomInput *pGetCustomInput;
    DMScustSettingInfo *pCustSettingInfo;
    DMScustSettingList *pCustSettingList;
} DMSgetCustomFeatureV2;


/**
 *  This structure contains customization settings set to modem unpack
 *
 *  \param  Tlvresult
 *          - Unpack Result
 */
typedef struct 
{
    DMSgetCustomFeatureV2 GetCustomFeatureV2;
    uint16_t Tlvresult;
}unpack_dms_GetCustFeaturesV2_t;

int unpack_dms_GetCustFeaturesV2(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetCustFeaturesV2_t *pOutput
);

/**
 *  @param  pActivationState[OUT]
 *          -  Service Activation Code\n
 *              0 - Service not activated\n
 *              1 - Service activated\n
 *              2 - Activation connecting\n
 *              3 - Activation connected\n
 *              4 - OTASP security authenticated\n
 *              5 - OTASP NAM downloaded\n
 *              6 - OTASP MDN downloaded\n
 *              7 - OTASP IMSI downloaded\n
 *              8 - OTASP PRL downloaded\n
 *              9 - OTASP SPC downloaded\n
 *              10 - OTASP settings committed
 */
typedef struct 
{
    uint8_t state;
}unpack_dms_GetActivationState_t;

/**
 * Get Activation State pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_GetActivationState(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen
);

/**
 * Get Activation State unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_GetActivationState(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_GetActivationState_t *pOutput
);

typedef struct
{
    uint8_t imageType;
    uint8_t uniqueID[UNIQUE_ID_LEN];
    uint8_t buildIDLen;
    uint8_t buildID[MAX_BUILD_ID_LEN];
} image_info_t;

/**
 * @param  numEntries[IN/OUT]
 *         - Number of entries in the image list to follow
 *         - The size of the list pCurrImgInfo must be specified
 *           when calling the API
 *
 * @param  pCurrImgInfo[OUT]
 *         - Currently Active Image List
 *
 *  @param  priver[OUT]
 *          - PRI version of the currently running firmware
 *
 *  @param  pkgver[OUT]
 *          - Package version of the currently running firmware
 *
 *  @param  fwvers[OUT]
 *          - firmware version of the currently running firmware
 *
 *  @param  carrier[OUT]
 *          - Carrier string of the currently running firmware
 */
typedef struct {
    uint8_t       numEntries;
    image_info_t *pCurrImgInfo;
    char          priver[DMS_IMGDETAILS_LEN];
    char          pkgver[DMS_IMGDETAILS_LEN];
    char          fwvers[DMS_IMGDETAILS_LEN];
    char          carrier[DMS_IMGDETAILS_LEN];
} unpack_dms_SLQSSwiGetFirmwareCurr_t;

/**
 * get currently active image pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int pack_dms_SLQSSwiGetFirmwareCurr(
        pack_qmi_t  *pCtx,
        uint8_t *pReqBuf,
        uint16_t *pLen
        );

/**
 * get currently active image unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */
int unpack_dms_SLQSSwiGetFirmwareCurr(
        uint8_t *pResp,
        uint16_t respLen,
        unpack_dms_SLQSSwiGetFirmwareCurr_t *pOutput
        );

/**
 * @param  pDestSMSNum[IN]
 *         - SMS Destination Number as string of 8 bit ASCII Characters Max 20 chars.
 *         - Optional parameter.
 *
 * @param  pDestSMSContent[IN]
 *         - SMS COntent as a string of 8 bit ASCII text characters Max 160 chars.
 *         - Optional parameter.
 */

typedef struct {
    uint8_t *pDestSMSNum;
    uint8_t *pDestSMSContent;
}pack_dms_SLQSSwiSetDyingGaspCfg_t;


/**
 * Set Dying GASP Config pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int pack_dms_SLQSSwiSetDyingGaspCfg(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen,
    pack_dms_SLQSSwiSetDyingGaspCfg_t *reqArg
);
/**
 *  This structure contains set Dying GASP Config unpack
 *
 *  \param  Tlvresult
 *          - Unpack Result
 */

typedef struct {
    uint16_t Tlvresult;
}unpack_dms_SLQSSwiSetDyingGaspCfg_t;

/**
 * Set Dying GASP Config unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_dms_SLQSSwiSetDyingGaspCfg(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SLQSSwiSetDyingGaspCfg_t *pOutput
);

/**
 * Clear Dying GASP Statistics pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int pack_dms_SLQSSwiClearDyingGaspStatistics(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen
);

/**
 *  This structure contains Clear Dying GASP unpack
 *
 *  \param  Tlvresult
 *          - Unpack Result
 */

typedef struct {
    uint16_t Tlvresult;
}unpack_dms_SLQSSwiClearDyingGaspStatistics_t;

/**
 * Clear Dying GASP Statistics unpack
 * @param[in] pResp qmi response
 * @param[in] pespLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_dms_SLQSSwiClearDyingGaspStatistics(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SLQSSwiClearDyingGaspStatistics_t *pOutput
);

/**
 * Get Dying GASP Statistics pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int pack_dms_SLQSSwiGetDyingGaspStatistics(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen);

/**
 * @param  TimeStamp[OUT]
 *         - Time Stamp.
 *
 * @param  SMSAttemptedFlag[OUT]
 *         - SMS Attempted Flag.
 */

typedef struct
{
    uint32_t *pTimeStamp;
    uint8_t  *pSMSAttemptedFlag;
} packgetDyingGaspStatistics;

/**
 *  This structure contains Get Dying GASP Statistics.
 *
 *  \param  Tlvresult
 *          - Unpack Result
 */

typedef struct {
    packgetDyingGaspStatistics *pGetDyingGaspStatistics;
    uint16_t Tlvresult;
}unpack_dms_SLQSSwiGetDyingGaspStatistics_t;

/**
 * Get Dying GASP Statistics unpack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_dms_SLQSSwiGetDyingGaspStatistics(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SLQSSwiGetDyingGaspStatistics_t *pOutput);

/**
 * Get Dying GASP Config pack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int pack_dms_SLQSSwiGetDyingGaspCfg(
    pack_qmi_t  *pCtx,
    uint8_t *pReqBuf,
    uint16_t *pLen);

/**
 * @param  pDestSMSNum[IN]
 *         - SMS Destination Number as string of 8 bit ASCII Characters Max 20 chars.
 *         - Optional parameter.
 *
 * @param  pDestSMSContent[IN]
 *         - SMS COntent as a string of 8 bit ASCII text characters Max 160 chars.
 *         - Optional parameter.
 */


typedef struct
{
    uint8_t *pDestSMSNum;
    uint8_t *pDestSMSContent;
} packgetDyingGaspCfg;

/**
 *  This structure contains Get Dying GASP Config unpack
 *
 *  \param  Tlvresult
 *          - Unpack Result
 */

typedef struct {
    packgetDyingGaspCfg *pGetDyingGaspCfg;
    uint16_t Tlvresult;
}unpack_dms_SLQSSwiGetDyingGaspCfg_t;

/**
 * Get Dying GASP Config unpack
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 */

int unpack_dms_SLQSSwiGetDyingGaspCfg(
    uint8_t *pResp,
    uint16_t respLen,
    unpack_dms_SLQSSwiGetDyingGaspCfg_t *pOutput);


