/*************
 *
 * Filename:    fwDld_9x15.c
 *
 * Purpose:     Contains routines needed to update MDM 9x15 based devices
 *
 * Copyright: © 2013 Sierra Wireless Inc., all rights reserved
 *
 **************/
#include "fwDld_9x15.h"
#include <syslog.h>
#include <dirent.h>

#define FW_DOWNLOAD_TIMEOUT_SEC 180
extern int g_kill_sdk;

static volatile BOOL fwDwlDone = FALSE;
static BOOL fwdwfail = FALSE;

/*
 * Name:     DevStateChgCbk
 *
 * Purpose:  Device State change callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void DevStateChgCbk_9x15(eDevState devstatus)
{
    /* If device is ready to communicate */
    if( devstatus ==  DEVICE_STATE_READY )
    {
        fprintf( stderr, "\nDevice Ready\n" );

        /* Unsubscribe from the callback */
        SetDeviceStateChangeCbk(NULL);
        fwDwlDone = TRUE;
    }
}

/*
 * Name:     FwDwldCbk_9x15
 *
 * Purpose:  Firmware download completion callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void FwDwldCbk_9x15(ULONG status)
{
    switch (status)
    {
        case eQCWWAN_ERR_NONE:
        {
            fprintf( stderr, "\nFirmware Download Completed\n" );
            /* set firmware complete to true */
            fwDwlDone = TRUE;
            break;
        }
        case eQCWWAN_ERR_SWIIM_FIRMWARE_NOT_DOWNLOADED:
        {
            fprintf( stderr, "\nFirmware Not Downloaded" );
            fwdwfail = FALSE; // This is not an error as the modem did not need the FW to be downloaded
            /* set firmware complete to true */
            fwDwlDone = TRUE;
            break;
        }
        case eQCWWAN_ERR_SWIIM_FW_ENTER_DOWNLOAD_MODE:
        {
            fprintf( stderr, "\nEnter Download Mode\n" );
            return;
        }
        case eQCWWAN_ERR_SWIIM_FW_FLASH_COMPLETE:
        {
            fprintf( stderr, "\nFlash Complete, Waiting for Modem to Reboot\n" );
            return;
        }
        default:
        {
            fprintf( stderr, "ERROR: Firmware Download failed. Callback status %u", status);
            fwdwfail = TRUE;
            /* set firmware complete to true */
            fwDwlDone = TRUE;
            break;
        }
    }

    /* Unsubscribe from the callback */
    SetFwDldCompletionCbk(NULL);
}

BOOL IsValidPath(CHAR *pImgPath)
{
    struct dirent *pDirent;
    DIR *pDir;
    BOOL retval =0;
    
    pDir = opendir (pImgPath);
    if (pDir == NULL)
        return 0;
    else
    {
        while ((pDirent = readdir(pDir)) != NULL)
        {
            if (pDirent->d_name[0]!='.')
            {        
                if ((strstr(pDirent->d_name, ".cwe")) ||
                    (strstr(pDirent->d_name, ".nvu")) ||
                    (strstr(pDirent->d_name, ".spk")))
                {
                    retval = 1;  
                }
            }
        }
    }
    closedir (pDir);
    return retval;
}


/*
 * Name:     FwDloader_9x15_NoGIM
 *
 * Purpose:  Download firmware on a 9x15 based device using non-GIM mode
 *
 * Return:   None
 *
 * Notes:    none
 */
void FwDloader_9x15_NoGIM( CHAR *pImagePath )
{
    ULONG rc = 0;
    CHAR  completeImagePath[512];
    ULONG len = 0;

    /* Reset the firmware download completion and device ready flags */
    fwDwlDone = FALSE;
    fwdwfail = FALSE;

    rc = SetFwDldCompletionCbk( FwDwldCbk_9x15 );
    if( eQCWWAN_ERR_NONE != rc )
    {
        fprintf( stderr, "ERROR: Failed to register firmware Download Completion Callback\n"\
                         "Failure Code: %d\n", rc );
        quit_app(eFWDWL_ERR_SET_CBK);
    }

    /* Concatenate image Path with '/' */
    memset( completeImagePath, 0, sizeof(completeImagePath) );
    
    len = strlen(pImagePath);
    if (len > (sizeof(completeImagePath) - 2) )
    {
        fprintf( stderr, "Error: ImagePath too long to process\n" );
        fprintf( stderr, "Failure Code:%d\n", eFWDWL_ERR_PATH_TOO_LONG );
        quit_app(eFWDWL_ERR_PATH_TOO_LONG);
    }

    if (len == 0)
    {
        fprintf( stderr, "Error: empty ImagePath\n" );
        fprintf( stderr, "Failure Code:%d\n", eFWDWL_ERR_PATH_NOT_SPECIFIED );
        quit_app(eFWDWL_ERR_PATH_NOT_SPECIFIED);
    }

    strncpy( completeImagePath,
             pImagePath,
             len );
    CHAR *pr = &completeImagePath[len - 1];
    if( *pr != '/' )
    {
        DIR *pDir = opendir(completeImagePath);
        if(pDir != NULL)
        {
            closedir(pDir);
            *(pr + 1) = '/';
            if (!IsValidPath(completeImagePath))
            {
                fprintf( stderr,
                         "ERROR: Failed to retrieve both PRI or firmware image information"\
                         " - Please check path and try again\n");
                fprintf( stderr, "Failure Code:%d\n", eFWDWL_ERR_INVALID_PATH );
                quit_app(eFWDWL_ERR_INVALID_PATH);
            }
        }
    }
    /* Start downloading the firmware */
    rc = UpgradeFirmware2k( completeImagePath );
    if( eQCWWAN_ERR_NONE != rc )
    {
        /* some firmware of MC7700 may reboot immediately after receiving
         * set firmware id command without sending out respond,
         * neglect this error to bypass issue*/
        if(eQCWWAN_ERR_NO_DEVICE !=rc )
        {
            fprintf( stderr, "Firmware Download Failed\n"\
                             "Failure Code: %d\n", rc );
            syslog (LOG_DEBUG, "%s: Failure Code: %d", __func__, rc);
            quit_app(eFWDWL_ERR_SDK);
        }
        return;
    }

    fprintf( stderr, "\n\nDownloading Firmware");
    while( !fwDwlDone )
    {
        /* Display "." while firmware downloads */
        fprintf( stderr, ".");
        sleep(2);
    }

    if (fwdwfail)
    {
        fprintf( stderr, "ERROR: Firmware Download Failed\n");
        quit_app(eFWDWL_ERR_SDK);
    }

    fprintf( stderr, "INFO: Firmware Download Succeeded\n");
}

/*
 * Name:     FwDloader_9x15_GIM
 *
 * Purpose:  Download firmware on a 9x15 based device using GIM mode
 *
 * Return:   None
 *
 * Notes:    none
 */
void FwDloader_9x15_GIM( CHAR *pImagePath )
{
    ULONG              resultCode = 0;
    struct qmifwinfo_s FwImgInfo, NVImgInfo;
    CurrentImgList     CurrImgList;
    CurrImageInfo      currImgInfo[5];
    BYTE               numEntries  = 5;
    CHAR               priVer[16];
    CHAR               pkgVer[SLQSFWINFO_PACKAGEID_SZ];
    CHAR               carrier[SLQSFWINFO_PACKAGEID_SZ];
    CHAR               AppVer[SLQSFWINFO_PACKAGEID_SZ];
    CHAR               completeImagePath[512];
    ULONG              len = 0;
    BOOL               skipPRIChk = FALSE;
    BOOL               skipMDMChk = FALSE;
    FirmwareUpdatStat  FirmwareUpdatStatus;
    BYTE               ImgType     = 0;
    ULONG              RefData     = 0;
    BYTE               RefStrLen   = 0xFF;
    BYTE               RefStr[255] = "\0";
    BYTE               LogStrLen   = 0xFF;
    BYTE               LogStr[255] = "\0";
    int                count =0;
    int NVImgInfoAvail = 0;

    memset(&FirmwareUpdatStatus,0,sizeof(FirmwareUpdatStat));
    FirmwareUpdatStatus.pImgType      = &ImgType;
    FirmwareUpdatStatus.pRefData      = &RefData;
    FirmwareUpdatStatus.pRefStringLen = &RefStrLen;
    FirmwareUpdatStatus.pRefString    = RefStr;
    FirmwareUpdatStatus.pLogStringLen = &LogStrLen;
    FirmwareUpdatStatus.pLogString    = LogStr;

    
    /* Initialize variables */
    memset( priVer, 0, sizeof(priVer) );
    memset( pkgVer, 0, sizeof(pkgVer) );
    memset( carrier, 0, sizeof(carrier) );
    memset( AppVer, 0, sizeof(AppVer) );

    CurrImgList.pCurrImgInfo = currImgInfo;
    CurrImgList.numEntries   = numEntries;

    memset( completeImagePath, 0, sizeof(completeImagePath) );
    len = strlen(pImagePath);
    if (len > (sizeof(completeImagePath) - 2) )
    {
        fprintf( stderr, "Error: ImagePath too long to process\n" );
        fprintf( stderr, "Failure Code:%d\n", eFWDWL_ERR_PATH_TOO_LONG );
        quit_app(eFWDWL_ERR_PATH_TOO_LONG);
    }
    strncpy( completeImagePath,
             pImagePath,
             strlen(pImagePath) );
    len = strlen(completeImagePath);

    if (len == 0)
    {
        fprintf( stderr, "Error: empty ImagePath\n" );
        fprintf( stderr, "Failure Code:%d\n", eFWDWL_ERR_PATH_NOT_SPECIFIED );
        quit_app(eFWDWL_ERR_PATH_NOT_SPECIFIED);
    }

    CHAR *pr = &completeImagePath[len - 1];
    if( *pr != '/' )
    {
        DIR *pDir = opendir(completeImagePath);
        if(pDir != NULL)
        {
            closedir(pDir);
            *(pr + 1) = '/';
        }
    }

    /* Get Firmware image details */
    resultCode = SLQSGetImageInfo_9x15( completeImagePath,
                                        IMG_FW_TYPE_CWE,
                                        &FwImgInfo.dev.s );

    if (eQCWWAN_ERR_NONE != resultCode)
    {
        resultCode = SLQSGetImageInfo_9x15( completeImagePath,
                                     IMG_FW_TYPE_SPK,
                                     &FwImgInfo.dev.s );
    }

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr,
                 "INFO: no firmware info, use device's (0x%x)\n", resultCode);
        syslog (LOG_DEBUG, "%s: no firmware info, use device's: %d", __func__, resultCode);
        skipMDMChk = TRUE;
    }
    else
    {
        /* Retrieve appversion( firmware version ) from fw image */
        strncpy( AppVer,
                 FwImgInfo.dev.s.appversion_str,
                 strlen(FwImgInfo.dev.s.appversion_str) );
    }

    /* Get PRI image details */
    resultCode = SLQSGetImageInfo_9x15( completeImagePath,
                                        IMG_PRI_TYPE_NVU,
                                        &NVImgInfo.dev.s );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr,
                 "INFO: no PRI info, use device's (0x%x)\n", resultCode);
        syslog (LOG_DEBUG, "%s: no PRI info, use device's: %d", __func__, resultCode);
        skipPRIChk = TRUE;
    }
    else
    {
        NVImgInfoAvail = 1;
        /* Save the firmware details for the image being downloaded */
        /* Retrieve PRI version, pkg version, carrier from PRI image */
        strncpy( priVer,
                 NVImgInfo.dev.s.priversion_str,
                 strlen(NVImgInfo.dev.s.priversion_str) );

        strncpy( pkgVer,
                 NVImgInfo.dev.s.packageid_str,
                 strlen(NVImgInfo.dev.s.packageid_str) );

        strncpy( carrier,
                 NVImgInfo.dev.s.carrier_str,
                 strlen(NVImgInfo.dev.s.carrier_str) );
    }

    /* If both images were not found, it indicates an issue */
    if( TRUE == skipMDMChk && TRUE == skipPRIChk )
    {
        fprintf( stderr,
                 "ERROR: Failed to retrieve both PRI and firmware image information"\
                 " - Please check path and try again\n");
        fprintf( stderr, "Failure Code:%d\n", eFWDWL_ERR_INVALID_PATH );
        syslog (LOG_DEBUG, "%s: ERROR: Failed to retrieve both PRI and firmware image information", __func__);
        quit_app(eFWDWL_ERR_SDK);
    }

    if ( NVImgInfoAvail && strncmp( NVImgInfo.dev.s.sku_str, "9999999", 7) )
    {
        fprintf( stderr, "INFO: OEM PRI\n");
        skipPRIChk = TRUE;
    }

    /* Reaching here means, image has been found */
    /* Subscribe to Device State Change callback */
    resultCode = SetDeviceStateChangeCbk(DevStateChgCbk_9x15);
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr, "ERROR: Failed to register to device State Callback\n"\
                         "Failure Code: %d\n", resultCode );
        syslog (LOG_DEBUG, "%s: ERROR: Failed to register to device State Callback: %d", __func__, resultCode);
        quit_app(eFWDWL_ERR_SDK);
    }

    /* Subscribe to Firmware Download completion callback */
    resultCode = SetFwDldCompletionCbk(FwDwldCbk_9x15);
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr, "ERROR: Failed to register firmware Download Completion Callback\n"\
                         "Failure Code : %u\n", resultCode );
        syslog (LOG_DEBUG, "%s: ERROR: Failed to register firmware Download Completion Callback: %d", __func__, resultCode);
        quit_app(eFWDWL_ERR_SDK);
    }

    /* Start downloading the firmware */
    resultCode = UpgradeFirmware2k( completeImagePath );
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr, "ERROR: Firmware Download Failed\n"\
                         "Failure Code : %u\n", resultCode );

        /* Deregister device state change and firmware download completion
         * callback.
         */
        SetFwDldCompletionCbk( NULL );
        SetDeviceStateChangeCbk( NULL );
        syslog (LOG_DEBUG, "%s: ERROR: Firmware Download Failed: %d", __func__, resultCode);
        quit_app(eFWDWL_ERR_SDK);
    }

    /* Keep displaying "." until fimrware downloads complete */
    fprintf( stderr, "Downloading Firmware");
    fwDwlDone = FALSE;
    while( !fwDwlDone )
    {
        fprintf( stderr, ".");
        sleep(1);
    }

    /* Intialize current running image structure */
    memset( (void *)&CurrImgList, 0, sizeof( CurrImgList ) );
    memset( (void *)&currImgInfo, 0, sizeof( currImgInfo ) );

    CurrImgList.numEntries = numEntries;
    CurrImgList.pCurrImgInfo = currImgInfo;
    count = 0;
    do
    {
        sleep(1);
        resultCode = SLQSSwiGetFwUpdateStatus( &FirmwareUpdatStatus );
        if(resultCode==eQCWWAN_ERR_NONE)
        {
            fprintf(stderr,"\n");
            break;
        }
        if(count==0)
        {
            fprintf(stderr,"\nWaiting Device Ready!\n");
        }
        else if(count>=(FW_DOWNLOAD_TIMEOUT_SEC-1))
        {
            fprintf(stderr,"\nWaiting Device Timeout!\n");
        }
        fprintf(stderr,". ");
    }while(count++<FW_DOWNLOAD_TIMEOUT_SEC);

    if(FirmwareUpdatStatus.pLogString)
    {
        if(strlen((char*)FirmwareUpdatStatus.pLogString)>0)
        {
            syslog (LOG_DEBUG, "Get firmware update status\n"\
              "LogString: %s\n", FirmwareUpdatStatus.pLogString);
            fprintf(stderr,"\nGet firmware update status:%s!\n",FirmwareUpdatStatus.pLogString);
        }
    }

    resultCode = SLQSSwiGetFirmwareCurr( &CurrImgList );
    if( eQCWWAN_ERR_NONE != resultCode)
    {
        fprintf( stderr, "ERROR: Failed to get firmware details after download\n"\
                         "Failure Code: %d\n", resultCode );
        syslog (LOG_DEBUG, "%s: ERROR: Failed to get firmware details after download: %d", __func__, resultCode);
        quit_app(eFWDWL_ERR_SDK);
    }

    /* Check if the firmware download is success */
    /* Compare PRI, package, carrier and app version strings */
    if( FALSE == skipPRIChk )
    {
        if( 0 != strncmp( CurrImgList.priver, priVer, sizeof(priVer) ) ||
            0 != strncmp( CurrImgList.pkgver, pkgVer, sizeof(pkgVer) ) ||
            0 != strncmp( CurrImgList.carrier, carrier, sizeof(carrier) ) )
        {
            fprintf( stderr, "CurrImgList.priver, priVer: %s/%s\n", CurrImgList.priver, priVer);
            fprintf( stderr, "CurrImgList.pkgVer, pkgVer: %s/%s\n", CurrImgList.pkgver, pkgVer);
            fprintf( stderr, "CurrImgList.carrier, carrier: %s/%s\n", CurrImgList.carrier, carrier);
            fprintf( stderr, "ERROR: PRI Upgrade Failed!!!\n" );
            quit_app(eFWDWL_ERR_FW_UPGRADE);
        }
        fprintf( stderr, "INFO: PRI Upgrade successful!!!\n" );
    }
    if( FALSE == skipMDMChk )
    {
        if(  0 != strncmp( CurrImgList.fwvers, AppVer, sizeof(AppVer) ) )
        {
            fprintf( stderr, "ERROR: Firmware Upgrade Failed!!!\n" );
            quit_app(eFWDWL_ERR_FW_UPGRADE);
        }
        fprintf( stderr, "INFO: Firmware Upgrade successful!!!\n" );
    }

    quit_app(eFWDWL_SUCCESS);
}

/*
 * Name:     FwDloader_9x15
 *
 * Purpose:  Download firmware on a 9x15 based device
 *
 * Return:   None
 *
 * Notes:    none
 */
void FwDloader_9x15( CHAR *pImagePath )
{
    ULONG              resultCode = 0;
    CurrentImgList     CurrImgList;
    CurrImageInfo      currImgInfo[5];
    BYTE               numEntries  = 5;
    CHAR               ModelId[128] = {0};
    BYTE               stringsize = sizeof(ModelId);

    memset( (void *)&CurrImgList, 0, sizeof( CurrImgList ) );
    memset( (void *)&currImgInfo, 0, sizeof( currImgInfo ) );
    CurrImgList.pCurrImgInfo = currImgInfo;
    CurrImgList.numEntries   = numEntries;

    GetModelID( stringsize, ModelId );

    if (strstr (ModelId, "AR") ||
        strstr (ModelId, "WP"))
    {
        /* for all AR series, treat ot as No GIM since the AR firmware does not
           use Goib Image Switching */
        FwDloader_9x15_NoGIM( pImagePath );
    }
    else
    {
        /* There are 2 possible scenario's determined by calling SLQSSwiGetFwCurr
         * 1) Device does not support Gobi IM - In this case use same procedure as
         *    FwDloader_9x00. In this case, we need only one file( spkg format).
         * 2) Device supports GIM but data returned is blank. Use normal procedure. */
        resultCode = SLQSSwiGetFirmwareCurr( &CurrImgList );
        if( eQCWWAN_ERR_NONE != resultCode)
        {
            /* Assume that device is in non GIM Mode. Device issues can be detected
             * on subsequent SLQS API calls */
            FwDloader_9x15_NoGIM( pImagePath );
        }
        else
        {
            /* Device supports GIM. We do not need the firmware details as of now. */
            FwDloader_9x15_GIM( pImagePath );
        }
    }
}
