/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/kobject.h>
#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/etherdevice.h>
#include <linux/list.h>
#include <linux/netdevice.h>
#include <linux/slab.h>
#include <linux/list.h>
#include <linux/phy.h>
#include <linux/timer.h>
#include <linux/workqueue.h>

#define ATH_HLEN	4

#define ATH_HDR_MOD_VER "0.0.1"

/* Bit field to determine which source ports will get their own vlan */

static unsigned long portvlan_bitfield;

static int ath_rcv(struct sk_buff *skb, struct net_device *dev,
		   struct packet_type *pt, struct net_device *orig_dev)
{
	u8 *ath_header;
	int source_port;

	skb = skb_unshare(skb, GFP_ATOMIC);
	if (skb == NULL)
		goto out;

	if (unlikely(!pskb_may_pull(skb, ATH_HLEN)))
		goto out_drop;

	/*
	 * The ethertype field is part of the DSA header.
	 */
	ath_header = skb->data - 2;

	/*
	 * Check that frame type is right version 
	 */
	if ((ath_header[0] & 0xc0) != 0x80)
		goto out_drop;

	/*
	 * Determine source port.
	 */
	source_port = ath_header[3] & 0x3;

	/*
	 * Convert the DSA header to an 802.1q header if the port is
	 * set in the portvlan_bitfield. If not, 
	 * delete the DSA header entirely.
	 * Port 0 is CPU, port 1 is trunk.
	 */
	if (source_port > 1 && ((1<<(source_port-2)) & portvlan_bitfield)) {
		u8 new_header[4];

		/*
		 * Insert 802.1q ethertype and copy the VLAN-related
		 * fields, but clear the bit that will hold CFI (since
		 * DSA uses that bit location for another purpose).
		 */
		new_header[0] = (ETH_P_8021Q >> 8) & 0xff;
		new_header[1] = ETH_P_8021Q & 0xff;
		new_header[2] = 0;
		new_header[3] = source_port - 1;

		/*
		 * Update packet checksum if skb is CHECKSUM_COMPLETE.
		 */
		if (skb->ip_summed == CHECKSUM_COMPLETE) {
			__wsum c = skb->csum;
			c = csum_add(c, csum_partial(new_header + 2, 2, 0));
			c = csum_sub(c, csum_partial(ath_header + 2, 2, 0));
			skb->csum = c;
		}

		memcpy(ath_header, new_header, ATH_HLEN);
	} else {
		/*
		 * Remove DSA tag and update checksum.
		 */
		skb_pull_rcsum(skb, ATH_HLEN);
		memmove(skb->data - ETH_HLEN,
			skb->data - ETH_HLEN - ATH_HLEN,
			2 * ETH_ALEN);
	}

	skb_push(skb, ETH_HLEN);
	skb->pkt_type = PACKET_HOST;
	skb->protocol = eth_type_trans(skb, skb->dev);
	netif_receive_skb(skb);

	return 0;

out_drop:
	kfree_skb(skb);
out:
	return 0;
}

/*
 * Create a simple subdirectory in sysfs called
 * /sys/kernel/ath-header  In that directory, 2 files are created:
 * "set" and "reset". If an integer is written to these files, it can be
 * later read out of it.
 * The bit field written to 'set' forces the corresponding source port packets to be 
 * rewritten with a vlan header for that port. Bits not set have no effect.
 * To force ports off, write one values in the bitfield in 'reset'. Bits not
 * set in reset have no effect
 * The current bit field can be read by reading either the set or reset file.
 */

static ssize_t portvlan_show(struct kobject *kobj, struct kobj_attribute *attr,
		      char *buf)
{
	return sprintf(buf, "0x%lX\n", portvlan_bitfield);
}

static ssize_t portvlan_store(struct kobject *kobj, struct kobj_attribute *attr,
		       const char *buf, size_t count)
{
	unsigned long var;

	var = simple_strtoul(buf, 0, 0);
	if (strcmp(attr->attr.name, "set") == 0)
		portvlan_bitfield |= var;
	else if (strcmp(attr->attr.name, "reset") == 0)
		portvlan_bitfield &= ~var;
	else
		count = 0;
	return count;
}

static struct kobj_attribute set_attribute =
	__ATTR(set, 0666, portvlan_show, portvlan_store);
static struct kobj_attribute reset_attribute =
	__ATTR(reset, 0666, portvlan_show, portvlan_store);

/*
 * Create a group of attributes so that we can create and destroy them all
 * at once.
 */
static struct attribute *attrs[] = {
	&set_attribute.attr,
	&reset_attribute.attr,
	NULL,	/* need to NULL terminate the list of attributes */
};

/*
 * An unnamed attribute group will put all of the attributes directly in
 * the kobject directory.  If we specify a name, a subdirectory will be
 * created for the attributes with the directory being the name of the
 * attribute group.
 */
static struct attribute_group attr_group = {
	.attrs = attrs,
};

static struct kobject *tagath_kobj;

static struct packet_type ath_packet_type __read_mostly = {
	.type	= cpu_to_be16(0xAAAA),
	.func	= ath_rcv,
};

static int __init ath_init_module(void)
{
	int retval;

	/*
	 * Create a simple kobject with the name of "ath-header",
	 * located under /sys/kernel/
	 *
	 * As this is a simple directory, no uevent will be sent to
	 * userspace.  That is why this function should not be used for
	 * any type of dynamic kobjects, where the name and number are
	 * not known ahead of time.
	 */
	tagath_kobj = kobject_create_and_add("ath-header", kernel_kobj);
	if (!tagath_kobj)
		return -ENOMEM;

	/* Create the files associated with this kobject */
	retval = sysfs_create_group(tagath_kobj, &attr_group);
	if (retval) {
		kobject_put(tagath_kobj);
		return retval;
	}
	dev_add_pack(&ath_packet_type);
	return 0;
}
module_init(ath_init_module);

static void __exit ath_cleanup_module(void)
{
	kobject_put(tagath_kobj);
	dev_remove_pack(&ath_packet_type);
}
module_exit(ath_cleanup_module);

MODULE_DESCRIPTION("ATH-HEADER: Munge the atheros header");
MODULE_LICENSE("Dual BSD/GPL");
MODULE_VERSION(ATH_HDR_MOD_VER);

