/* dummy.c: a dummy net driver

	The purpose of this driver is to provide a device to point a
	route through, but not to actually transmit packets.

	Why?  If you have a machine whose only connection is an occasional
	PPP/SLIP/PLIP link, you can only connect to your own hostname
	when the link is up.  Otherwise you have to use localhost.
	This isn't very consistent.

	One solution is to set up a dummy link using PPP/SLIP/PLIP,
	but this seems (to me) too much overhead for too little gain.
	This driver provides a small alternative. Thus you can do

	[when not running slip]
		ifconfig dummy slip.addr.ess.here up
	[to go to slip]
		ifconfig dummy down
		dip whatever

	This was written by looking at Donald Becker's skeleton driver
	and the loopback driver.  I then threw away anything that didn't
	apply!	Thanks to Alan Cox for the key clue on what to do with
	misguided packets.

			Nick Holloway, 27th May 1994
	[I tweaked this explanation a little but that's all]
			Alan Cox, 30th May 1994
*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/init.h>
#include <linux/moduleparam.h>
#include <linux/rtnetlink.h>
#include <net/rtnetlink.h>
#include <linux/u64_stats_sync.h>
#include <linux/phy.h>

static int numdummies = 1;
#define MAX_SIZE_OF_BUS_PHY_ID	20
static char bus_phy_id[MAX_SIZE_OF_BUS_PHY_ID];

struct dummpy_priv {
	struct net_device		*ndev;
	struct device			*dev;
	struct phy_device		*phy;
	int 					phy_if;
};

static int dummy_set_address(struct net_device *dev, void *p)
{
	struct sockaddr *sa = p;

	if (!is_valid_ether_addr(sa->sa_data))
		return -EADDRNOTAVAIL;

	memcpy(dev->dev_addr, sa->sa_data, ETH_ALEN);
	return 0;
}

/* fake multicast ability */
static void set_multicast_list(struct net_device *dev)
{
}

struct pcpu_dstats {
	u64			tx_packets;
	u64			tx_bytes;
	struct u64_stats_sync	syncp;
};

static struct rtnl_link_stats64 *dummy_get_stats64(struct net_device *dev,
						   struct rtnl_link_stats64 *stats)
{
	int i;

	for_each_possible_cpu(i) {
		const struct pcpu_dstats *dstats;
		u64 tbytes, tpackets;
		unsigned int start;

		dstats = per_cpu_ptr(dev->dstats, i);
		do {
			start = u64_stats_fetch_begin(&dstats->syncp);
			tbytes = dstats->tx_bytes;
			tpackets = dstats->tx_packets;
		} while (u64_stats_fetch_retry(&dstats->syncp, start));
		stats->tx_bytes += tbytes;
		stats->tx_packets += tpackets;
	}
	return stats;
}

static netdev_tx_t dummy_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct pcpu_dstats *dstats = this_cpu_ptr(dev->dstats);

	u64_stats_update_begin(&dstats->syncp);
	dstats->tx_packets++;
	dstats->tx_bytes += skb->len;
	u64_stats_update_end(&dstats->syncp);

	dev_kfree_skb(skb);
	return NETDEV_TX_OK;
}
static int dummy_open(struct net_device *ndev)
{
	struct dummpy_priv *priv = netdev_priv(ndev);
	if(bus_phy_id[0])
	{
		priv->phy = phy_connect(priv->ndev, bus_phy_id,
					 NULL, 0, priv->phy_if);
		if (IS_ERR(priv->phy)) {
			dev_info(priv->dev, "phy %s not found\n",bus_phy_id );
			priv->phy = NULL;
		} else {
			dev_info(priv->dev, "phy %s found\n", bus_phy_id);
		}
	}
	return 0;
}
static int dummy_stop(struct net_device *ndev)
{
	struct dummpy_priv *priv = netdev_priv(ndev);

	if (!priv->phy)
		return 0;
	phy_stop(priv->phy);
	phy_disconnect(priv->phy);
	priv->phy = NULL;
	return 0;
}

static int dummy_do_ioctl(struct net_device *ndev, struct ifreq *ifrq,
		int cmd)
{
	struct dummpy_priv *priv = netdev_priv(ndev);
	if (!(netif_running(ndev)))
		return -EINVAL;
	if(priv->phy ==NULL) return -ENXIO;

	return phy_mii_ioctl(priv->phy, ifrq, cmd);
}

static int dummy_dev_init(struct net_device *dev)
{
	dev->dstats = alloc_percpu(struct pcpu_dstats);
	if (!dev->dstats)
		return -ENOMEM;

	return 0;
}

static void dummy_dev_free(struct net_device *dev)
{
	free_percpu(dev->dstats);
	free_netdev(dev);
}

static const struct net_device_ops dummy_netdev_ops = {
	.ndo_init		= dummy_dev_init,
	.ndo_open		= dummy_open,
	.ndo_stop		= dummy_stop,
	.ndo_start_xmit		= dummy_xmit,
	.ndo_validate_addr	= eth_validate_addr,
	.ndo_set_rx_mode	= set_multicast_list,
	.ndo_set_mac_address	= dummy_set_address,
	.ndo_do_ioctl		= dummy_do_ioctl,
	.ndo_get_stats64	= dummy_get_stats64,
};

static void dummy_setup(struct net_device *dev)
{
	ether_setup(dev);

	/* Initialize the device structure. */
	dev->netdev_ops = &dummy_netdev_ops;
	dev->destructor = dummy_dev_free;

	/* Fill in device structure with ethernet-generic values. */
	dev->tx_queue_len = 0;
	dev->flags |= IFF_NOARP;
	dev->flags &= ~IFF_MULTICAST;
	dev->features	|= NETIF_F_SG | NETIF_F_FRAGLIST | NETIF_F_TSO;
	dev->features	|= NETIF_F_NO_CSUM | NETIF_F_HIGHDMA | NETIF_F_LLTX;
	random_ether_addr(dev->dev_addr);
}

static int dummy_validate(struct nlattr *tb[], struct nlattr *data[])
{
	if (tb[IFLA_ADDRESS]) {
		if (nla_len(tb[IFLA_ADDRESS]) != ETH_ALEN)
			return -EINVAL;
		if (!is_valid_ether_addr(nla_data(tb[IFLA_ADDRESS])))
			return -EADDRNOTAVAIL;
	}
	return 0;
}

static struct rtnl_link_ops dummy_link_ops __read_mostly = {
	.kind		= "dummy",
	.setup		= dummy_setup,
	.validate	= dummy_validate,
};


/* Number of dummy devices to be set up by this module. */
module_param(numdummies, int, 0);
MODULE_PARM_DESC(numdummies, "Number of dummy pseudo devices");
module_param_string(bus_phy_id, bus_phy_id, sizeof(bus_phy_id), 0644);
MODULE_PARM_DESC(bus_phy_id, "BUS:PHY ID, it can be found in /sys/bus/mdio_bus/devices");

static int __init dummy_init_one(void)
{
	struct net_device *dev_dummy;
	int err;
	struct dummpy_priv *priv;

	dev_dummy = alloc_netdev(sizeof(struct dummpy_priv), "dummy%d", dummy_setup);
	if (!dev_dummy)
		return -ENOMEM;
	priv = netdev_priv(dev_dummy);

	priv->ndev = dev_dummy;
	priv->dev  = &dev_dummy->dev;
	priv->phy  = NULL;
	priv->phy_if = PHY_INTERFACE_MODE_RGMII;

	dev_dummy->rtnl_link_ops = &dummy_link_ops;
	err = register_netdevice(dev_dummy);
	if (err < 0)
		goto err;
	return 0;

err:
	free_netdev(dev_dummy);
	return err;
}

static int __init dummy_init_module(void)
{
	int i, err = 0;

	rtnl_lock();
	err = __rtnl_link_register(&dummy_link_ops);

	for (i = 0; i < numdummies && !err; i++)
		err = dummy_init_one();
	if (err < 0)
		__rtnl_link_unregister(&dummy_link_ops);
	rtnl_unlock();

	return err;
}

static void __exit dummy_cleanup_module(void)
{
	rtnl_link_unregister(&dummy_link_ops);
}

module_init(dummy_init_module);
module_exit(dummy_cleanup_module);
MODULE_LICENSE("GPL");
MODULE_ALIAS_RTNL_LINK("dummy");
