/*
 * 8 bit, BCH8-only NAND driver for TI am335x. Contains workarounds
 * for hardware bugs.
 *
 * There is a hardware bug that occasionally causes the first data byte
 * after a NAND RNDOUT command to be read as 0xFF, depending on the address
 * pattern in the RNDOUT command.
 *
 * This interfereres with the ECC syndrome calculation, since the first of
 * the parity bytes reads as 0xFF. The BCH8 ECC is capable of correcting
 * up to 8 bit errors per subpage, including parity bytes.
 *
 * If a parity byte appears as 0xFF, all 0-bits of the original value now
 * show up as bit errors that need correction. Thus, between 0 and 8 bits
 * of error correction capability is wasted to fix the parity byte, leaving
 * none for the actual data in 1/256 of the cases.
 *
 * The workaround involves using a debug register in the NAND controller
 * to feed data bytes into the parity generator without reading it from NAND.
 *
 * The algorithm reads the whole OOB area first. The structure of this area
 * is as follows:
 *
 * +00, 2 bytes, 0xFF 0xFF Bad block marker
 * +02, 13 bytes, Parity for subpage 0
 * +15, 1 byte, 0x00 alignment
 * ...
 * ..., 13 bytes, Parity for last subpage
 * ..., 1 byte, 0x00 alignment
 * ..., Spare bytes in OOB
 *
 * When reading the whole OOB, the RNDOUT bug can only hit the bad block
 * marker, which by definition must be 0xFF anyway. The rest of the OOB
 * reads correctly.
 *
 * Then, the data subpages are read individually:
 *	* reset the ECC parity generator
 *	* read the 512 bytes of subpage data
 *	* force the respective 13 parity bytes into the ECC controller
 *	* read and store the generated ECC syndrome
 *
 * When all subpages are done, the stored syndromes are processed via ELM
 * as usual to extract the byte and bit offsets of real bit errors.
 *
 * Iwo
 *
 * Copyright © 2004 NetComm Wireless, Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 * Copyright © 2004 Texas Instruments, Jian Zhang <jzhang@ti.com>
 * Copyright © 2004 Micron Technology Inc.
 * Copyright © 2004 David Brownell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/jiffies.h>
#include <linux/sched.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/io.h>
#include <linux/slab.h>

#include <plat/dma.h>
#include <plat/gpmc.h>
#include <plat/nand.h>
#include <plat/elm.h>

#include <linux/debugfs.h>

#define	DRIVER_NAME	"omap2-nand"
#define	OMAP_NAND_TIMEOUT_MS	5000

#define NAND_Ecc_P1e		(1 << 0)
#define NAND_Ecc_P2e		(1 << 1)
#define NAND_Ecc_P4e		(1 << 2)
#define NAND_Ecc_P8e		(1 << 3)
#define NAND_Ecc_P16e		(1 << 4)
#define NAND_Ecc_P32e		(1 << 5)
#define NAND_Ecc_P64e		(1 << 6)
#define NAND_Ecc_P128e		(1 << 7)
#define NAND_Ecc_P256e		(1 << 8)
#define NAND_Ecc_P512e		(1 << 9)
#define NAND_Ecc_P1024e		(1 << 10)
#define NAND_Ecc_P2048e		(1 << 11)

#define NAND_Ecc_P1o		(1 << 16)
#define NAND_Ecc_P2o		(1 << 17)
#define NAND_Ecc_P4o		(1 << 18)
#define NAND_Ecc_P8o		(1 << 19)
#define NAND_Ecc_P16o		(1 << 20)
#define NAND_Ecc_P32o		(1 << 21)
#define NAND_Ecc_P64o		(1 << 22)
#define NAND_Ecc_P128o		(1 << 23)
#define NAND_Ecc_P256o		(1 << 24)
#define NAND_Ecc_P512o		(1 << 25)
#define NAND_Ecc_P1024o		(1 << 26)
#define NAND_Ecc_P2048o		(1 << 27)

#define TF(value)	(value ? 1 : 0)

#define P2048e(a)	(TF(a & NAND_Ecc_P2048e)	<< 0)
#define P2048o(a)	(TF(a & NAND_Ecc_P2048o)	<< 1)
#define P1e(a)		(TF(a & NAND_Ecc_P1e)		<< 2)
#define P1o(a)		(TF(a & NAND_Ecc_P1o)		<< 3)
#define P2e(a)		(TF(a & NAND_Ecc_P2e)		<< 4)
#define P2o(a)		(TF(a & NAND_Ecc_P2o)		<< 5)
#define P4e(a)		(TF(a & NAND_Ecc_P4e)		<< 6)
#define P4o(a)		(TF(a & NAND_Ecc_P4o)		<< 7)

#define P8e(a)		(TF(a & NAND_Ecc_P8e)		<< 0)
#define P8o(a)		(TF(a & NAND_Ecc_P8o)		<< 1)
#define P16e(a)		(TF(a & NAND_Ecc_P16e)		<< 2)
#define P16o(a)		(TF(a & NAND_Ecc_P16o)		<< 3)
#define P32e(a)		(TF(a & NAND_Ecc_P32e)		<< 4)
#define P32o(a)		(TF(a & NAND_Ecc_P32o)		<< 5)
#define P64e(a)		(TF(a & NAND_Ecc_P64e)		<< 6)
#define P64o(a)		(TF(a & NAND_Ecc_P64o)		<< 7)

#define P128e(a)	(TF(a & NAND_Ecc_P128e)		<< 0)
#define P128o(a)	(TF(a & NAND_Ecc_P128o)		<< 1)
#define P256e(a)	(TF(a & NAND_Ecc_P256e)		<< 2)
#define P256o(a)	(TF(a & NAND_Ecc_P256o)		<< 3)
#define P512e(a)	(TF(a & NAND_Ecc_P512e)		<< 4)
#define P512o(a)	(TF(a & NAND_Ecc_P512o)		<< 5)
#define P1024e(a)	(TF(a & NAND_Ecc_P1024e)	<< 6)
#define P1024o(a)	(TF(a & NAND_Ecc_P1024o)	<< 7)

#define P8e_s(a)	(TF(a & NAND_Ecc_P8e)		<< 0)
#define P8o_s(a)	(TF(a & NAND_Ecc_P8o)		<< 1)
#define P16e_s(a)	(TF(a & NAND_Ecc_P16e)		<< 2)
#define P16o_s(a)	(TF(a & NAND_Ecc_P16o)		<< 3)
#define P1e_s(a)	(TF(a & NAND_Ecc_P1e)		<< 4)
#define P1o_s(a)	(TF(a & NAND_Ecc_P1o)		<< 5)
#define P2e_s(a)	(TF(a & NAND_Ecc_P2e)		<< 6)
#define P2o_s(a)	(TF(a & NAND_Ecc_P2o)		<< 7)

#define P4e_s(a)	(TF(a & NAND_Ecc_P4e)		<< 0)
#define P4o_s(a)	(TF(a & NAND_Ecc_P4o)		<< 1)

#define MAX_HWECC_BYTES_OOB_64     24
#define JFFS2_CLEAN_MARKER_OFFSET  0x2

#define BCH_ECC_POS			0x2
#define BCH_JFFS2_CLEAN_MARKER_OFFSET	0x3a
#define OMAP_BCH8_ECC_SECT_BYTES	14

#define DLEVEL 0
#include <linux/debug.h>

/* oob info generated runtime depending on ecc algorithm and layout selected */
static struct nand_ecclayout omap_oobinfo;
/* Define some generic bad / good block scan pattern which are used
 * while scanning a device for factory marked good / bad blocks
 */
static uint8_t scan_ff_pattern[] = { 0xff };
static struct nand_bbt_descr bb_descrip_flashbased = {
	.options = NAND_BBT_SCANEMPTY | NAND_BBT_SCANALLPAGES,
	.offs = 0,
	.len = 1,
	.pattern = scan_ff_pattern,
};


struct omap_nand_info {
	struct nand_hw_control		controller;
	struct omap_nand_platform_data	*pdata;
	struct mtd_info			mtd;
	struct nand_chip		nand;
	struct platform_device		*pdev;

	int				gpmc_cs;
	unsigned long			phys_base;
	struct completion		comp;
	int				dma_ch;
	int				gpmc_irq;
	enum {
		OMAP_NAND_IO_READ = 0,	/* read */
		OMAP_NAND_IO_WRITE,	/* write */
	} iomode;
	u_char				*buf;
	int				buf_len;
	int				ecc_opt;
	int (*ctrlr_suspend) (void);
	int (*ctrlr_resume) (void);
};

/**
 * omap_hwcontrol - hardware specific access to control-lines
 * @mtd: MTD device structure
 * @cmd: command to device
 * @ctrl:
 * NAND_NCE: bit 0 -> don't care
 * NAND_CLE: bit 1 -> Command Latch
 * NAND_ALE: bit 2 -> Address Latch
 *
 * NOTE: boards may use different bits for these!!
 */
static void omap_hwcontrol(struct mtd_info *mtd, int cmd, unsigned int ctrl)
{
	struct omap_nand_info *info = container_of(mtd,
					struct omap_nand_info, mtd);

	if (cmd != NAND_CMD_NONE) {
		if (ctrl & NAND_CLE)
			gpmc_nand_write(info->gpmc_cs, GPMC_NAND_COMMAND, cmd);

		else if (ctrl & NAND_ALE)
			gpmc_nand_write(info->gpmc_cs, GPMC_NAND_ADDRESS, cmd);

		else /* NAND_NCE */
			gpmc_nand_write(info->gpmc_cs, GPMC_NAND_DATA, cmd);
	}
}

/**
 * omap_read_buf8 - read data from NAND controller into buffer
 * @mtd: MTD device structure
 * @buf: buffer to store date
 * @len: number of bytes to read
 */
static void omap_read_buf8(struct mtd_info *mtd, u_char *buf, int len)
{
	struct nand_chip *nand = mtd->priv;

	ioread8_rep(nand->IO_ADDR_R, buf, len);
}

/**
 * omap_write_buf8 - write buffer to NAND controller
 * @mtd: MTD device structure
 * @buf: data buffer
 * @len: number of bytes to write
 */
static void omap_write_buf8(struct mtd_info *mtd, const u_char *buf, int len)
{
	struct omap_nand_info *info = container_of(mtd,
						struct omap_nand_info, mtd);
	u_char *p = (u_char *)buf;
	u32	status = 0;

	while (len--) {
		iowrite8(*p++, info->nand.IO_ADDR_W);
		/* wait until buffer is available for write */
		do {
			status = gpmc_read_status(GPMC_STATUS_BUFFER);
		} while (!status);
	}
}

/**
 * omap_read_buf16 - read data from NAND controller into buffer
 * @mtd: MTD device structure
 * @buf: buffer to store date
 * @len: number of bytes to read
 */
static void omap_read_buf16(struct mtd_info *mtd, u_char *buf, int len)
{
	struct nand_chip *nand = mtd->priv;

	ioread16_rep(nand->IO_ADDR_R, buf, len / 2);
}

/**
 * omap_write_buf16 - write buffer to NAND controller
 * @mtd: MTD device structure
 * @buf: data buffer
 * @len: number of bytes to write
 */
static void omap_write_buf16(struct mtd_info *mtd, const u_char * buf, int len)
{
	struct omap_nand_info *info = container_of(mtd,
						struct omap_nand_info, mtd);
	u16 *p = (u16 *) buf;
	u32	status = 0;
	/* FIXME try bursts of writesw() or DMA ... */
	len >>= 1;

	while (len--) {
		iowrite16(*p++, info->nand.IO_ADDR_W);
		/* wait until buffer is available for write */
		do {
			status = gpmc_read_status(GPMC_STATUS_BUFFER);
		} while (!status);
	}
}

/**
 * omap_read_buf_pref - read data from NAND controller into buffer
 * @mtd: MTD device structure
 * @buf: buffer to store date
 * @len: number of bytes to read
 */
static void omap_read_buf_pref(struct mtd_info *mtd, u_char *buf, int len)
{
	struct omap_nand_info *info = container_of(mtd,
						struct omap_nand_info, mtd);
	uint32_t r_count = 0;
	int ret = 0;
	u32 *p = (u32 *)buf;

	/* take care of subpage reads */
	if (len % 4) {
		if (info->nand.options & NAND_BUSWIDTH_16)
			omap_read_buf16(mtd, buf, len % 4);
		else
			omap_read_buf8(mtd, buf, len % 4);
		p = (u32 *) (buf + len % 4);
		len -= len % 4;
	}

	/* configure and start prefetch transfer */
	ret = gpmc_prefetch_enable(info->gpmc_cs,
			PREFETCH_FIFOTHRESHOLD_MAX, 0x0, len, 0x0);
	if (ret) {
		/* PFPW engine is busy, use cpu copy method */
		if (info->nand.options & NAND_BUSWIDTH_16)
			omap_read_buf16(mtd, (u_char *)p, len);
		else
			omap_read_buf8(mtd, (u_char *)p, len);
	} else {
		do {
			r_count = gpmc_read_status(GPMC_PREFETCH_FIFO_CNT);
			r_count = r_count >> 2;
			ioread32_rep(info->nand.IO_ADDR_R, p, r_count);
			p += r_count;
			len -= r_count << 2;
		} while (len);
		/* disable and stop the PFPW engine */
		gpmc_prefetch_reset(info->gpmc_cs);
	}
}

/**
 * omap_write_buf_pref - write buffer to NAND controller
 * @mtd: MTD device structure
 * @buf: data buffer
 * @len: number of bytes to write
 */
static void omap_write_buf_pref(struct mtd_info *mtd,
					const u_char *buf, int len)
{
	struct omap_nand_info *info = container_of(mtd,
						struct omap_nand_info, mtd);
	uint32_t w_count = 0;
	int i = 0, ret = 0;
	u16 *p = (u16 *)buf;
	unsigned long tim, limit;

	/* take care of subpage writes */
	if (len % 2 != 0) {
		writeb(*buf, info->nand.IO_ADDR_W);
		p = (u16 *)(buf + 1);
		len--;
	}

	/*  configure and start prefetch transfer */
	ret = gpmc_prefetch_enable(info->gpmc_cs,
			PREFETCH_FIFOTHRESHOLD_MAX, 0x0, len, 0x1);
	if (ret) {
		/* PFPW engine is busy, use cpu copy method */
		if (info->nand.options & NAND_BUSWIDTH_16)
			omap_write_buf16(mtd, (u_char *)p, len);
		else
			omap_write_buf8(mtd, (u_char *)p, len);
	} else {
		while (len) {
			w_count = gpmc_read_status(GPMC_PREFETCH_FIFO_CNT);
			w_count = w_count >> 1;
			for (i = 0; (i < w_count) && len; i++, len -= 2)
				iowrite16(*p++, info->nand.IO_ADDR_W);
		}
		/* wait for data to flushed-out before reset the prefetch */
		tim = 0;
		limit = (loops_per_jiffy *
					msecs_to_jiffies(OMAP_NAND_TIMEOUT_MS));
		while (gpmc_read_status(GPMC_PREFETCH_COUNT) && (tim++ < limit))
			cpu_relax();

		/* disable and stop the PFPW engine */
		gpmc_prefetch_reset(info->gpmc_cs);
	}
}

/*
 * omap_nand_dma_cb: callback on the completion of dma transfer
 * @lch: logical channel
 * @ch_satuts: channel status
 * @data: pointer to completion data structure
 */
static void omap_nand_dma_cb(int lch, u16 ch_status, void *data)
{
	complete((struct completion *) data);
}

/*
 * omap_nand_dma_transfer: configer and start dma transfer
 * @mtd: MTD device structure
 * @addr: virtual address in RAM of source/destination
 * @len: number of data bytes to be transferred
 * @is_write: flag for read/write operation
 */
static inline int omap_nand_dma_transfer(struct mtd_info *mtd, void *addr,
					unsigned int len, int is_write)
{
	struct omap_nand_info *info = container_of(mtd,
					struct omap_nand_info, mtd);
	enum dma_data_direction dir = is_write ? DMA_TO_DEVICE :
							DMA_FROM_DEVICE;
	dma_addr_t dma_addr;
	int ret;
	unsigned long tim, limit;

	/* The fifo depth is 64 bytes max.
	 * But configure the FIFO-threshold to 32 to get a sync at each frame
	 * and frame length is 32 bytes.
	 */
	int buf_len = len >> 6;

	if (addr >= high_memory) {
		struct page *p1;

		if (((size_t)addr & PAGE_MASK) !=
			((size_t)(addr + len - 1) & PAGE_MASK))
			goto out_copy;
		p1 = vmalloc_to_page(addr);
		if (!p1)
			goto out_copy;
		addr = page_address(p1) + ((size_t)addr & ~PAGE_MASK);
	}

	dma_addr = dma_map_single(&info->pdev->dev, addr, len, dir);
	if (dma_mapping_error(&info->pdev->dev, dma_addr)) {
		dev_err(&info->pdev->dev,
			"Couldn't DMA map a %d byte buffer\n", len);
		goto out_copy;
	}

	if (is_write) {
	    omap_set_dma_dest_params(info->dma_ch, 0, OMAP_DMA_AMODE_CONSTANT,
						info->phys_base, 0, 0);
	    omap_set_dma_src_params(info->dma_ch, 0, OMAP_DMA_AMODE_POST_INC,
							dma_addr, 0, 0);
	    omap_set_dma_transfer_params(info->dma_ch, OMAP_DMA_DATA_TYPE_S32,
					0x10, buf_len, OMAP_DMA_SYNC_FRAME,
					OMAP24XX_DMA_GPMC, OMAP_DMA_DST_SYNC);
	} else {
	    omap_set_dma_src_params(info->dma_ch, 0, OMAP_DMA_AMODE_CONSTANT,
						info->phys_base, 0, 0);
	    omap_set_dma_dest_params(info->dma_ch, 0, OMAP_DMA_AMODE_POST_INC,
							dma_addr, 0, 0);
	    omap_set_dma_transfer_params(info->dma_ch, OMAP_DMA_DATA_TYPE_S32,
					0x10, buf_len, OMAP_DMA_SYNC_FRAME,
					OMAP24XX_DMA_GPMC, OMAP_DMA_SRC_SYNC);
	}
	/*  configure and start prefetch transfer */
	ret = gpmc_prefetch_enable(info->gpmc_cs,
			PREFETCH_FIFOTHRESHOLD_MAX, 0x1, len, is_write);
	if (ret)
		/* PFPW engine is busy, use cpu copy method */
		goto out_copy;

	init_completion(&info->comp);

	omap_start_dma(info->dma_ch);

	/* setup and start DMA using dma_addr */
	wait_for_completion(&info->comp);
	tim = 0;
	limit = (loops_per_jiffy * msecs_to_jiffies(OMAP_NAND_TIMEOUT_MS));
	while (gpmc_read_status(GPMC_PREFETCH_COUNT) && (tim++ < limit))
		cpu_relax();

	/* disable and stop the PFPW engine */
	gpmc_prefetch_reset(info->gpmc_cs);

	dma_unmap_single(&info->pdev->dev, dma_addr, len, dir);
	return 0;

out_copy:
	if (info->nand.options & NAND_BUSWIDTH_16)
		is_write == 0 ? omap_read_buf16(mtd, (u_char *) addr, len)
			: omap_write_buf16(mtd, (u_char *) addr, len);
	else
		is_write == 0 ? omap_read_buf8(mtd, (u_char *) addr, len)
			: omap_write_buf8(mtd, (u_char *) addr, len);
	return 0;
}

/**
 * omap_read_buf_dma_pref - read data from NAND controller into buffer
 * @mtd: MTD device structure
 * @buf: buffer to store date
 * @len: number of bytes to read
 */
static void omap_read_buf_dma_pref(struct mtd_info *mtd, u_char *buf, int len)
{
	if (len <= mtd->oobsize)
		omap_read_buf_pref(mtd, buf, len);
	else
		/* start transfer in DMA mode */
		omap_nand_dma_transfer(mtd, buf, len, 0x0);
}

/**
 * omap_write_buf_dma_pref - write buffer to NAND controller
 * @mtd: MTD device structure
 * @buf: data buffer
 * @len: number of bytes to write
 */
static void omap_write_buf_dma_pref(struct mtd_info *mtd,
					const u_char *buf, int len)
{
	if (len <= mtd->oobsize)
		omap_write_buf_pref(mtd, buf, len);
	else
		/* start transfer in DMA mode */
		omap_nand_dma_transfer(mtd, (u_char *) buf, len, 0x1);
}

/*
 * omap_nand_irq - GMPC irq handler
 * @this_irq: gpmc irq number
 * @dev: omap_nand_info structure pointer is passed here
 */
static irqreturn_t omap_nand_irq(int this_irq, void *dev)
{
	struct omap_nand_info *info = (struct omap_nand_info *) dev;
	u32 bytes;
	u32 irq_stat;

	irq_stat = gpmc_read_status(GPMC_GET_IRQ_STATUS);
	bytes = gpmc_read_status(GPMC_PREFETCH_FIFO_CNT);
	bytes = bytes  & 0xFFFC; /* io in multiple of 4 bytes */
	if (info->iomode == OMAP_NAND_IO_WRITE) { /* checks for write io */
		if (irq_stat & 0x2)
			goto done;

		if (info->buf_len && (info->buf_len < bytes))
			bytes = info->buf_len;
		else if (!info->buf_len)
			bytes = 0;
		iowrite32_rep(info->nand.IO_ADDR_W,
						(u32 *)info->buf, bytes >> 2);
		info->buf = info->buf + bytes;
		info->buf_len -= bytes;

	} else {
		ioread32_rep(info->nand.IO_ADDR_R,
						(u32 *)info->buf, bytes >> 2);
		info->buf = info->buf + bytes;

		if (irq_stat & 0x2)
			goto done;
	}
	gpmc_cs_configure(info->gpmc_cs, GPMC_SET_IRQ_STATUS, irq_stat);

	return IRQ_HANDLED;

done:
	complete(&info->comp);
	/* disable irq */
	gpmc_cs_configure(info->gpmc_cs, GPMC_ENABLE_IRQ, 0);

	/* clear status */
	gpmc_cs_configure(info->gpmc_cs, GPMC_SET_IRQ_STATUS, irq_stat);

	return IRQ_HANDLED;
}

/*
 * omap_read_buf_irq_pref - read data from NAND controller into buffer
 * @mtd: MTD device structure
 * @buf: buffer to store date
 * @len: number of bytes to read
 */
static void omap_read_buf_irq_pref(struct mtd_info *mtd, u_char *buf, int len)
{
	struct omap_nand_info *info = container_of(mtd,
						struct omap_nand_info, mtd);
	int ret = 0;

	if (len <= mtd->oobsize) {
		omap_read_buf_pref(mtd, buf, len);
		return;
	}

	info->iomode = OMAP_NAND_IO_READ;
	info->buf = buf;
	init_completion(&info->comp);

	/*  configure and start prefetch transfer */
	ret = gpmc_prefetch_enable(info->gpmc_cs,
			PREFETCH_FIFOTHRESHOLD_MAX/2, 0x0, len, 0x0);
	if (ret)
		/* PFPW engine is busy, use cpu copy method */
		goto out_copy;

	info->buf_len = len;
	/* enable irq */
	gpmc_cs_configure(info->gpmc_cs, GPMC_ENABLE_IRQ,
		(GPMC_IRQ_FIFOEVENTENABLE | GPMC_IRQ_COUNT_EVENT));

	/* waiting for read to complete */
	wait_for_completion(&info->comp);

	/* disable and stop the PFPW engine */
	gpmc_prefetch_reset(info->gpmc_cs);
	return;

out_copy:
	if (info->nand.options & NAND_BUSWIDTH_16)
		omap_read_buf16(mtd, buf, len);
	else
		omap_read_buf8(mtd, buf, len);
}

/*
 * omap_write_buf_irq_pref - write buffer to NAND controller
 * @mtd: MTD device structure
 * @buf: data buffer
 * @len: number of bytes to write
 */
static void omap_write_buf_irq_pref(struct mtd_info *mtd,
					const u_char *buf, int len)
{
	struct omap_nand_info *info = container_of(mtd,
						struct omap_nand_info, mtd);
	int ret = 0;
	unsigned long tim, limit;

	if (len <= mtd->oobsize) {
		omap_write_buf_pref(mtd, buf, len);
		return;
	}

	info->iomode = OMAP_NAND_IO_WRITE;
	info->buf = (u_char *) buf;
	init_completion(&info->comp);

	/* configure and start prefetch transfer : size=24 */
	ret = gpmc_prefetch_enable(info->gpmc_cs,
			(PREFETCH_FIFOTHRESHOLD_MAX * 3) / 8, 0x0, len, 0x1);
	if (ret)
		/* PFPW engine is busy, use cpu copy method */
		goto out_copy;

	info->buf_len = len;
	/* enable irq */
	gpmc_cs_configure(info->gpmc_cs, GPMC_ENABLE_IRQ,
			(GPMC_IRQ_FIFOEVENTENABLE | GPMC_IRQ_COUNT_EVENT));

	/* waiting for write to complete */
	wait_for_completion(&info->comp);
	/* wait for data to flushed-out before reset the prefetch */
	tim = 0;
	limit = (loops_per_jiffy *  msecs_to_jiffies(OMAP_NAND_TIMEOUT_MS));
	while (gpmc_read_status(GPMC_PREFETCH_COUNT) && (tim++ < limit))
		cpu_relax();

	/* disable and stop the PFPW engine */
	gpmc_prefetch_reset(info->gpmc_cs);
	return;

out_copy:
	if (info->nand.options & NAND_BUSWIDTH_16)
		omap_write_buf16(mtd, buf, len);
	else
		omap_write_buf8(mtd, buf, len);
}

/**
 * omap_verify_buf - Verify chip data against buffer
 * @mtd: MTD device structure
 * @buf: buffer containing the data to compare
 * @len: number of bytes to compare
 */
static int omap_verify_buf(struct mtd_info *mtd, const u_char * buf, int len)
{
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);
	u16 *p = (u16 *) buf;

	len >>= 1;
	while (len--) {
		if (*p++ != cpu_to_le16(readw(info->nand.IO_ADDR_R)))
			return -EFAULT;
	}

	return 0;
}


/* Debugfs entries for statistics */
static struct dentry *dfe = NULL;

/* DEBUGFS statistics for NAND page access */
static struct dfstats {
	unsigned rd_total;  /* Total number of NAND sub-page reads */
	unsigned rd_ff;     /* Erased sub-pages */
	unsigned rd_OK;     /* Error-free sub-pages */
	unsigned rd_ECC;    /* sub-pages required data error correction */
	unsigned rd_ECCBUG; /* sub-pages triggered OOB parity bug */
	unsigned rd_BUG8;   /* sub-pages with OOB parity bug with no ECC protection left */
} dfs = { 0 };

#define STAT_INC(name) dfs.name++

static int debugfs_init(void)
{
	dfe = debugfs_create_dir("NAND", NULL);
	if (IS_ERR(dfe)) return PTR_ERR(dfe);

	#define EXPORT_32RO(name) debugfs_create_u32(#name, 0444, dfe, &dfs.name)
	EXPORT_32RO(rd_total);
	EXPORT_32RO(rd_ff);
	EXPORT_32RO(rd_OK);
	EXPORT_32RO(rd_ECC);
	EXPORT_32RO(rd_ECCBUG);
	EXPORT_32RO(rd_BUG8);

	return 0;
}

static void debugfs_exit(void)
{
	debugfs_remove(dfe);
}


/* Returns 1 if array only contains values v */
static int allbytes(const u_char *array, const u_char v, unsigned count)
{
	uint32_t w = ((uint32_t)v << 24) | (v << 16) | (v << 8) | (v << 0);
	/* Unaligned head */
	while ((unsigned)array & 0x3) {
		if ( *array != v ) return 0;
		array++;
		count--;
	}
	/* Aligned words */
	while (count >= 4) {
		if ( *(uint32_t*)array != w ) return 0;
		array += 4;
		count -= 4;
	}
	/* Unaligned tail */
	while (count) {
		if ( *array != v ) return 0;
		array++;
		count--;
	}
	return 1;
}


/**
 * omap_read_page_bch - BCH ecc based page read function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	buffer to store read data
 * @page:	page number to read
 *
 * For BCH ECC scheme, GPMC used for syndrome calculation and ELM module
 * used for error correction.
 */
static int omap_read_page_bch(struct mtd_info *mtd, struct nand_chip *chip,
				uint8_t *buf, int page)
{
	int i, j, eccsize = chip->ecc.size;
	int eccbytes = chip->ecc.bytes;
	int eccsteps = chip->ecc.steps;
	uint8_t *p = buf;
	uint8_t *ecc_calc = chip->buffers->ecccalc;
	uint8_t *ecc_code = chip->buffers->ecccode;
	uint32_t *eccpos = chip->ecc.layout->eccpos;
	uint8_t *oob = &chip->oob_poi[eccpos[0]];

	uint32_t data_pos;
	uint32_t oob_offs;

	MTDDP(mtd, page, MR);

	/* Start read comand */
	chip->ecc.hwctl(mtd, NAND_ECC_READ);

	/* Read whole OOB area */
	chip->cmdfunc(mtd, NAND_CMD_RNDOUT, eccsize * eccsteps, page);
	chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);

	/* Reposition NAND pointer for data read */
	chip->cmdfunc(mtd, NAND_CMD_RNDOUT, 0, page); /* Data area */

	/* Read data areas */
	for (i = 0; i < eccsteps; i++) {

		data_pos = i * eccsize;  /* offset of current subpage */
		oob_offs = i * eccbytes; /* offset of subpage parity information within parity block */
		p = &buf[data_pos];      /* pointer into page data buffer */
		oob = &chip->oob_poi[eccpos[0]+oob_offs]; /* pointer into parity info */

		gpmc_clear_ECC(); /* Reset the parity generator */

		/* read subpage data */
		chip->read_buf(mtd, p, eccsize);

		/* Force-feed parity bytes into ECC to calculate syndrome. */
		for (j = 0; j < BCH8_ECC_OOB_BYTES; j++) {
			gpmc_force_ECC(oob[j]);
		}

		/* read & store syndrome. */
		chip->ecc.calculate(mtd, p, &ecc_calc[oob_offs]);
	}

	/* Copy syndrome data for MTD */
	for (i = 0; i < chip->ecc.total; i++)
		ecc_code[i] = chip->oob_poi[eccpos[i]];

	/* Bit error correction based on syndrome. chip->ecc.correct is the
	 * omap_correct_data function below. */

	eccsteps = chip->ecc.steps;
	p = buf;
	for (i = 0 ; eccsteps; eccsteps--, i += eccbytes, p += eccsize) {
		int stat;

		stat = chip->ecc.correct(mtd, p, &ecc_code[i], &ecc_calc[i]);

		if (stat < 0)
			mtd->ecc_stats.failed++;
		else
			mtd->ecc_stats.corrected += stat;
	}

	return 0;
}

/* Returns the number of 0-bits in buffer, up to MAX_ZEROBITS */
static int check_erased(uint8_t *buf, int size)
{
	#define MAX_ZEROBITS 33
	int zerobits = 0;
	uint32_t *wbuf = (uint32_t*)buf;
	/* We assume the buffer and size are 4-byte aligned */
	BUG_ON((long)buf & 0x3 || size & 0x3);
	while (size > 0) {
		if (unlikely(*wbuf != 0xffffffffu)) {
			uint32_t v = *wbuf;
			int i;
			for (i=0; i<32; i++) {
				zerobits += !(v & 0x80000000);
				v <<= 1;
			}
			if (zerobits >= MAX_ZEROBITS) break;
		}
		size-=4;
		wbuf++;
	}
	return zerobits>MAX_ZEROBITS?MAX_ZEROBITS:zerobits;
}

/**
 * omap_correct_data - Performs bit-error correction based on BCH8 syndrome
 * @mtd: MTD device structure
 * @dat: page data
 * @read_ecc: ecc read from nand flash
 * @calc_ecc: ecc read from HW ECC registers
 */
static int omap_correct_data(struct mtd_info *mtd, u_char *dat,
				u_char *read_ecc, u_char *calc_ecc)
{
	int j, count;
	unsigned int err_loc[8];
	int real_bitflips = 0;
	int zerobits;

	/* This function is called for every subpage of info->nand.ecc.size
	 * bytes, usually 512. ECC is calculated and stored separately for
	 * each subpage. The u_char * points at the start of each range. */

	STAT_INC(rd_total);

	/* The ECC calculation provides parity bits on write and a syndrome on
	 * read. Syndrome is 0x00 when no bit errors are encountered. */
	if (allbytes(calc_ecc, 0x00, BCH8_ECC_OOB_BYTES)) {
		STAT_INC(rd_OK);
		return 0;
	}

	/* If we reach this point, the syndrome has flagged bit errors.
	 * The options at this point are:
	 * 
	 * 1) We have encountered the OOB read bug, where the first byte
	 *    of the parity data is corrupted and reads as 0xFF. This
	 *    always shows up as 0-8 bit errors at offset 512.
	 *
	 * 2) Correctable bit errors in data area or parity bytes
	 *
	 * 3) Uncorrectable bit errors
	 */

	/* ELM calculates the offsets of bit errors from the syndrome. */
	count  = omap_elm_decode_bch_error(0, calc_ecc, err_loc);
	if (count < 0) {
		/* So we have uncorrectable bit errors. According to ELM.
		 * This may be true, but we could also deal with an erased
		 * page right now, maybe even with a few bit errors */

		/* Find out how many zero bits are in the page */
		zerobits = check_erased(dat, BCH8_ECC_BYTES);
		if (zerobits) {
			if (zerobits < MAX_ZEROBITS) {
				memset(dat, 0xff, BCH8_ECC_BYTES);
				MTDD_PRINT("Corrected %d erased page bitflips\n", zerobits);
				STAT_INC(rd_ECC);
				return zerobits;
			} else {
				/* Can't fix this mess */
				MTDD_PRINT("Uncorrectable bit errors\n");
				return -1;
			}
		} else {
			/* Page is erased, no bitflips */
			STAT_INC(rd_ff);
			return 0;
		}
	}

	/* If a NAND read reports bit errors, UBI will torture the block until
	 * it admits to being bad. If the confession is forthcoming, the block
	 * is retired and a spare block is used instead.
	 *
	 * In the case of the OOB read bug, every read will show bit errors,
	 * so sooner or later UBI will run out of spare blocks.
	 *
	 * To evade the inquisition, we keep quiet about bit errors due to the
	 * bug and only account for real bit errors. */

	real_bitflips = 0;
	for (j = 0; j < count; j++) {
		u32 bit_pos, byte_pos;

		/* Sanity check for bit errors reported beyond the data+parity.
		 * Doesn't make sense, but original TI code checks for this. */
		if (err_loc[j] >= BCH8_ECC_MAX) continue;

		bit_pos   = err_loc[j] % 8;
		byte_pos  = (BCH8_ECC_MAX - err_loc[j] - 1) / 8;

		if (byte_pos < BCH8_ECC_BYTES) {
			/* Bit error in data area, correcting. */
			STAT_INC(rd_ECC);
			MTDD_PRINT("Correcting bit at offset %d:%d\n", byte_pos, bit_pos);
			dat[byte_pos] ^= 1 << bit_pos;
			real_bitflips++;
		} else if (byte_pos == BCH8_ECC_BYTES && read_ecc[0] == 0xFF) {
			/* OOB read bug. If bit error is flagged in first parity byte and
			 * the byte is 0xFF, we ignore it and don't report it. */
			MTDD_PRINT("OOB bug %d:%d\n", byte_pos, bit_pos);
		} else {
			/* Real bit error in parity area. Reporting, but no need to correct. */
			STAT_INC(rd_ECC);
			MTDD_PRINT("Flipped bit in parity %d:%d\n", byte_pos - BCH8_ECC_BYTES, bit_pos);
			real_bitflips++;
		}
	}

	/* debugfs statistics update */
	if (count > real_bitflips) {
		STAT_INC(rd_ECCBUG);
		if (count - real_bitflips == 8) {
			STAT_INC(rd_BUG8);
		}
	}

	return real_bitflips;
}


/**
 * omap_calcuate_ecc - Read the ECC parity/syndrome bytes.
 * @mtd: MTD device structure
 * @dat: The pointer to data on which ecc is computed
 * @ecc_code: The ecc_code buffer
 *
 */
static int omap_calculate_ecc(struct mtd_info *mtd, const u_char *dat,
				u_char *ecc_code)
{
	int rval;

	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);

	rval = gpmc_calculate_ecc(info->ecc_opt, info->gpmc_cs, dat, ecc_code);

	/* Detecting erased pages and forcing the calculated ECC to be 0xFF. */
	{
		const u_char ErasedECC[BCH8_ECC_OOB_BYTES] = {
			0xF3, 0xDB, 0x14, 0x16, 0x8B, 0xD2, 0xBE,
			0xCC, 0xAC, 0x6B, 0xFF, 0x99, 0x7B };

		/* For speed, we don't compare the data area, but check the
		 * ECC syndrome values. Probability of false positives is 1:2^112. */
		if (memcmp(ecc_code, ErasedECC, BCH8_ECC_OOB_BYTES)==0) {
			memset(ecc_code, 0xFF, info->nand.ecc.bytes); /* This is one more byte than BCH8_ECC_OOB_BYTES */
		}
	}

	return rval;
}

/**
 * omap_enable_hwecc - This function enables the hardware ecc functionality
 * @mtd: MTD device structure
 * @mode: Read/Write mode
 */
static void omap_enable_hwecc(struct mtd_info *mtd, int mode)
{
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);
	struct nand_chip *chip = mtd->priv;
	unsigned int dev_width = (chip->options & NAND_BUSWIDTH_16) ? 1 : 0;

	gpmc_enable_hwecc(info->ecc_opt, info->gpmc_cs, mode,
				dev_width, info->nand.ecc.size);
}

/**
 * omap_wait - wait until the command is done
 * @mtd: MTD device structure
 * @chip: NAND Chip structure
 *
 * Wait function is called during Program and erase operations and
 * the way it is called from MTD layer, we should wait till the NAND
 * chip is ready after the programming/erase operation has completed.
 *
 * Erase can take up to 400ms and program up to 20ms according to
 * general NAND and SmartMedia specs
 */
static int omap_wait(struct mtd_info *mtd, struct nand_chip *chip)
{
	struct nand_chip *this = mtd->priv;
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);
	unsigned long timeo = jiffies;
	int status, state = this->state;

	if (state == FL_ERASING)
		timeo += (HZ * 400) / 1000;
	else
		timeo += (HZ * 20) / 1000;

	gpmc_nand_write(info->gpmc_cs,
			GPMC_NAND_COMMAND, (NAND_CMD_STATUS & 0xFF));
	while (time_before(jiffies, timeo)) {
		status = gpmc_nand_read(info->gpmc_cs, GPMC_NAND_DATA);
		if (status & NAND_STATUS_READY)
			break;
		cond_resched();
	}

	status = gpmc_nand_read(info->gpmc_cs, GPMC_NAND_DATA);
	return status;
}

/**
 * omap_dev_ready - calls the platform specific dev_ready function
 * @mtd: MTD device structure
 */
static int omap_dev_ready(struct mtd_info *mtd)
{
	unsigned int val = 0;
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);

	val = gpmc_read_status(GPMC_GET_IRQ_STATUS);
	if ((val & 0x100) == 0x100) {
		/* Clear IRQ Interrupt */
		val |= 0x100;
		val &= ~(0x0);
		gpmc_cs_configure(info->gpmc_cs, GPMC_SET_IRQ_STATUS, val);
	} else {
		unsigned int cnt = 0;
		while (cnt++ < 0x1FF) {
			if  ((val & 0x100) == 0x100)
				return 0;
			val = gpmc_read_status(GPMC_GET_IRQ_STATUS);
		}
	}

	return 1;
}

static int __devinit omap_nand_probe(struct platform_device *pdev)
{
	struct omap_nand_info		*info;
	struct omap_nand_platform_data	*pdata;
	int				err;
	int				i, offset;

	pdata = pdev->dev.platform_data;
	if (pdata == NULL) {
		dev_err(&pdev->dev, "platform data missing\n");
		return -ENODEV;
	}

	info = kzalloc(sizeof(struct omap_nand_info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	platform_set_drvdata(pdev, info);

	spin_lock_init(&info->controller.lock);
	init_waitqueue_head(&info->controller.wq);

	info->pdev = pdev;

	info->gpmc_cs		= pdata->cs;
	info->phys_base		= pdata->phys_base;

	info->mtd.priv		= &info->nand;
	info->mtd.name		= dev_name(&pdev->dev);
	info->mtd.owner		= THIS_MODULE;
	info->ecc_opt		= pdata->ecc_opt;

	info->nand.options	= pdata->devsize;
	info->nand.options	|= NAND_SKIP_BBTSCAN;

	/*
	 * If ELM feature is used in OMAP NAND driver, then configure it
	 */
	if (pdata->elm_used) {
		if (pdata->ecc_opt == OMAP_ECC_BCH8_CODE_HW)
			omap_configure_elm(&info->mtd, OMAP_BCH8_ECC);
	}

	if (pdata->ctrlr_suspend)
		info->ctrlr_suspend = pdata->ctrlr_suspend;
	if (pdata->ctrlr_resume)
		info->ctrlr_resume = pdata->ctrlr_resume;

	/* NAND write protect off */
	gpmc_cs_configure(info->gpmc_cs, GPMC_CONFIG_WP, 0);

	if (!request_mem_region(info->phys_base, NAND_IO_SIZE,
				pdev->dev.driver->name)) {
		err = -EBUSY;
		goto out_free_info;
	}

	info->nand.IO_ADDR_R = ioremap(info->phys_base, NAND_IO_SIZE);
	if (!info->nand.IO_ADDR_R) {
		err = -ENOMEM;
		goto out_release_mem_region;
	}

	info->nand.controller = &info->controller;

	info->nand.IO_ADDR_W = info->nand.IO_ADDR_R;
	info->nand.cmd_ctrl  = omap_hwcontrol;

	/*
	 * If RDY/BSY line is connected to OMAP then use the omap ready
	 * funcrtion and the generic nand_wait function which reads the status
	 * register after monitoring the RDY/BSY line.Otherwise use a standard
	 * chip delay which is slightly more than tR (AC Timing) of the NAND
	 * device and read status register until you get a failure or success
	 */
	if (pdata->dev_ready) {
		info->nand.dev_ready = omap_dev_ready;
		info->nand.chip_delay = 0;
	} else {
		info->nand.waitfunc = omap_wait;
		info->nand.chip_delay = 50;
	}

	switch (pdata->xfer_type) {
	case NAND_OMAP_PREFETCH_POLLED:
		info->nand.read_buf   = omap_read_buf_pref;
		info->nand.write_buf  = omap_write_buf_pref;
		break;

	case NAND_OMAP_POLLED:
		if (info->nand.options & NAND_BUSWIDTH_16) {
			info->nand.read_buf   = omap_read_buf16;
			info->nand.write_buf  = omap_write_buf16;
		} else {
			info->nand.read_buf   = omap_read_buf8;
			info->nand.write_buf  = omap_write_buf8;
		}
		break;

	case NAND_OMAP_PREFETCH_DMA:
		err = omap_request_dma(OMAP24XX_DMA_GPMC, "NAND",
				omap_nand_dma_cb, &info->comp, &info->dma_ch);
		if (err < 0) {
			info->dma_ch = -1;
			dev_err(&pdev->dev, "DMA request failed!\n");
			goto out_release_mem_region;
		} else {
			omap_set_dma_dest_burst_mode(info->dma_ch,
					OMAP_DMA_DATA_BURST_16);
			omap_set_dma_src_burst_mode(info->dma_ch,
					OMAP_DMA_DATA_BURST_16);

			info->nand.read_buf   = omap_read_buf_dma_pref;
			info->nand.write_buf  = omap_write_buf_dma_pref;
		}
		break;

	case NAND_OMAP_PREFETCH_IRQ:
		err = request_irq(pdata->gpmc_irq,
				omap_nand_irq, IRQF_SHARED, "gpmc-nand", info);
		if (err) {
			dev_err(&pdev->dev, "requesting irq(%d) error:%d",
							pdata->gpmc_irq, err);
			goto out_release_mem_region;
		} else {
			info->gpmc_irq	     = pdata->gpmc_irq;
			info->nand.read_buf  = omap_read_buf_irq_pref;
			info->nand.write_buf = omap_write_buf_irq_pref;
		}
		break;

	default:
		dev_err(&pdev->dev,
			"xfer_type(%d) not supported!\n", pdata->xfer_type);
		err = -EINVAL;
		goto out_release_mem_region;
	}

	info->nand.verify_buf = omap_verify_buf;

	info->nand.ecc.bytes     = OMAP_BCH8_ECC_SECT_BYTES;
	info->nand.ecc.size      = 512;
	info->nand.ecc.read_page = omap_read_page_bch;

	info->nand.ecc.calculate = omap_calculate_ecc;
	info->nand.ecc.hwctl     = omap_enable_hwecc;
	info->nand.ecc.correct   = omap_correct_data;
	info->nand.ecc.mode      = NAND_ECC_HW;

	if (nand_scan_ident(&info->mtd, 1, NULL)) {
		err = -ENXIO;
		goto out_release_mem_region;
	}

	if (info->mtd.writesize == 2048 && info->mtd.oobsize > 64) {
		pr_info("NAND reported OOB of %d bytes, shrinking to 64\n", info->mtd.oobsize);
		info->mtd.oobsize = 64;
	}
	if (info->mtd.writesize == 4096 && info->mtd.oobsize > 128) {
		pr_info("NAND reported OOB of %d bytes, shrinking to 128\n", info->mtd.oobsize);
		info->mtd.oobsize = 128;
	}


	info->nand.badblock_pattern = &bb_descrip_flashbased;

	omap_oobinfo.eccbytes = info->nand.ecc.bytes *
		info->mtd.writesize / info->nand.ecc.size;

	offset = BCH_ECC_POS; /* Synchronize with U-boot */

	omap_oobinfo.oobfree->offset = offset +
		omap_oobinfo.eccbytes;

	omap_oobinfo.oobfree->length = info->mtd.oobsize -
				offset - omap_oobinfo.eccbytes;

	for (i = 0; i < omap_oobinfo.eccbytes; i++)
		omap_oobinfo.eccpos[i] = i+offset;

	info->nand.ecc.layout = &omap_oobinfo;

	/* second phase scan */
	if (nand_scan_tail(&info->mtd)) {
		err = -ENXIO;
		goto out_release_mem_region;
	}


	/* We force the subpage size to full page size here. This is
	 * for compatibility with other platforms only, there is no
	 * reason why this system can't support subpages. Iwo. */

	info->mtd.subpage_sft = 0;
	info->nand.subpagesize = info->mtd.writesize >>
		info->mtd.subpage_sft;

	mtd_device_parse_register(&info->mtd, NULL, 0,
			pdata->parts, pdata->nr_parts);

	platform_set_drvdata(pdev, &info->mtd);

	return 0;

out_release_mem_region:
	release_mem_region(info->phys_base, NAND_IO_SIZE);
out_free_info:
	kfree(info);

	return err;
}

static int omap_nand_remove(struct platform_device *pdev)
{
	struct mtd_info *mtd = platform_get_drvdata(pdev);
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);

	platform_set_drvdata(pdev, NULL);
	if (info->dma_ch != -1)
		omap_free_dma(info->dma_ch);

	if (info->gpmc_irq)
		free_irq(info->gpmc_irq, info);

	/* Release NAND device, its internal structures and partitions */
	nand_release(&info->mtd);
	iounmap(info->nand.IO_ADDR_R);
	release_mem_region(info->phys_base, NAND_IO_SIZE);
	kfree(&info->mtd);
	return 0;
}

#ifdef CONFIG_PM
static int omap_nand_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct mtd_info *mtd = platform_get_drvdata(pdev);
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);

	mtd->suspend(mtd);

	if (info->ctrlr_suspend)
		info->ctrlr_suspend();

	return 0;
}

static int omap_nand_resume(struct platform_device *pdev)
{
	struct mtd_info *mtd = platform_get_drvdata(pdev);
	struct omap_nand_info *info = container_of(mtd, struct omap_nand_info,
							mtd);

	if (info->ctrlr_resume)
		info->ctrlr_resume();

	return 0;
}
#endif

static struct platform_driver omap_nand_driver = {
	.probe		= omap_nand_probe,
	.remove		= omap_nand_remove,
#ifdef CONFIG_PM
	.suspend	= omap_nand_suspend,
	.resume		= omap_nand_resume,
#endif
	.driver		= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
	},
};

static int __init omap_nand_init(void)
{
	int rval;

	pr_info("%s driver initializing\n", DRIVER_NAME);

	rval = debugfs_init();
	if (rval) {
		msg("Debugfs failed, rval=%d (%s)\n", rval, ErrStr(rval));
		return rval;
	}

	return platform_driver_register(&omap_nand_driver);
}

static void __exit omap_nand_exit(void)
{
	platform_driver_unregister(&omap_nand_driver);

	debugfs_exit();
}

module_init(omap_nand_init);
module_exit(omap_nand_exit);

MODULE_ALIAS("platform:" DRIVER_NAME);
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("NAND flash on TI AM335x");
MODULE_AUTHOR("Iwo Mergler <Iwo.Mergler@netcommwireless.com");
