/*
 * LED Kernel CDCS trigger set
 *
 * Sysfs entries:
 *
 *	brightness_on  = LED brightness in the 'on/triggered' state
 *	brightness_off = LED brighness in the 'off/idle' state
 *	force          = 'trigger' triggered operation, 'trigger_cont' triggered operation continously, 'on'/'off solid - trigger disabled.
 *	trigger_pulse  = [ms] trigger pulse
 *	trigger_dead   = [ms] trigger dead time
 *	trigger_now    = run the trigger action now
 *	help           = Helpful hints
 *
 * Currently supported triggers: cdcs-wwan, cdcs-wlan, cdcs-pots
 * To add more triggers, see comments in the first ~100 lines.
 *
 * <Iwo Mergler> Iwo@call-direct.com.au
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#include <linux/module.h>
#include <linux/jiffies.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/spinlock.h>
#include <linux/device.h>
#include <linux/timer.h>
#include <linux/ctype.h>
#include <linux/leds.h>
#include "leds.h"

#define DLEVEL 0
#include <linux/debug.h>

#define MODULE_NAME "ledtrig-cdcs: "

/* Generic function prototypes, for use by the per-trigger specific ones. */
static void ledtrig_cdcs(int index);
static void trig_activate(int index, struct led_classdev *led_cdev);
static void trig_deactivate(int index, struct led_classdev *led_cdev);

/*
vvvvvvvvvvvv Edit below this line to add more triggers vvvvvvvvvvvv
*/

#define CDCS_WWAN   0
#define CDCS_WAN    1
#define CDCS_WLANAP 2
#define CDCS_WLANST 3
#define CDCS_POTS0  4
#define CDCS_POTS1  5
#define CDCS_ZIGBEE 6
#define CDCS_BEAT   7
#define CDCS_NONE   8

#define CDCS_LAST 9

/* Trigger functions to be used by other drivers. Only use
if CONFIG_LEDS_TRIGGER_CDCS is defined. */
void ledtrig_cdcs_wwan(void) { ledtrig_cdcs(CDCS_WWAN); }
EXPORT_SYMBOL(ledtrig_cdcs_wwan);
void ledtrig_cdcs_wan(void) { ledtrig_cdcs(CDCS_WAN); }
EXPORT_SYMBOL(ledtrig_cdcs_wan);
void ledtrig_cdcs_wlanap(void) { ledtrig_cdcs(CDCS_WLANAP); }
EXPORT_SYMBOL(ledtrig_cdcs_wlanap);
void ledtrig_cdcs_wlanst(void) { ledtrig_cdcs(CDCS_WLANST); }
EXPORT_SYMBOL(ledtrig_cdcs_wlanst);
void ledtrig_cdcs_pots0(void) { ledtrig_cdcs(CDCS_POTS0); }
EXPORT_SYMBOL(ledtrig_cdcs_pots0);
void ledtrig_cdcs_pots1(void) { ledtrig_cdcs(CDCS_POTS1); }
EXPORT_SYMBOL(ledtrig_cdcs_pots1);
void ledtrig_cdcs_zigbee(void) { ledtrig_cdcs(CDCS_ZIGBEE); }
EXPORT_SYMBOL(ledtrig_cdcs_zigbee);
void ledtrig_cdcs_none(void) { }
EXPORT_SYMBOL(ledtrig_cdcs_none);


/* Setup/teardown functions, one per trigger */
static void trig_activate_wwan(struct led_classdev *led_cdev) { trig_activate(CDCS_WWAN,led_cdev); }
static void trig_deactivate_wwan(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_WWAN,led_cdev); }

static void trig_activate_wan(struct led_classdev *led_cdev) { trig_activate(CDCS_WAN,led_cdev); }
static void trig_deactivate_wan(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_WAN,led_cdev); }

static void trig_activate_wlanap(struct led_classdev *led_cdev) { trig_activate(CDCS_WLANAP,led_cdev); }
static void trig_deactivate_wlanap(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_WLANAP,led_cdev); }

static void trig_activate_wlanst(struct led_classdev *led_cdev) { trig_activate(CDCS_WLANST,led_cdev); }
static void trig_deactivate_wlanst(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_WLANST,led_cdev); }

static void trig_activate_pots0(struct led_classdev *led_cdev) { trig_activate(CDCS_POTS0,led_cdev); }
static void trig_deactivate_pots0(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_POTS0,led_cdev); }

static void trig_activate_pots1(struct led_classdev *led_cdev) { trig_activate(CDCS_POTS1,led_cdev); }
static void trig_deactivate_pots1(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_POTS1,led_cdev); }

static void trig_activate_zigbee(  struct led_classdev *led_cdev) {trig_activate(  CDCS_ZIGBEE, led_cdev);}
static void trig_deactivate_zigbee(struct led_classdev *led_cdev) {trig_deactivate(CDCS_ZIGBEE, led_cdev);}

static void trig_activate_beat(struct led_classdev *led_cdev) { trig_activate(CDCS_BEAT,led_cdev); }
static void trig_deactivate_beat(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_BEAT,led_cdev); }

static void trig_activate_none(struct led_classdev *led_cdev) { trig_activate(CDCS_NONE,led_cdev); }
static void trig_deactivate_none(struct led_classdev *led_cdev)   { trig_deactivate(CDCS_NONE,led_cdev); }


/* Trigger names and pointers to setup/teardown functions */
static struct led_trigger led_triggers[] = {
	{.name = "cdcs-wwan",   .activate = trig_activate_wwan, .deactivate = trig_deactivate_wwan},
	{.name = "cdcs-wan",    .activate = trig_activate_wan, .deactivate = trig_deactivate_wan},
	{.name = "cdcs-wlanap", .activate = trig_activate_wlanap, .deactivate = trig_deactivate_wlanap},
	{.name = "cdcs-wlanst", .activate = trig_activate_wlanst, .deactivate = trig_deactivate_wlanst},
	{.name = "cdcs-pots0",  .activate = trig_activate_pots0, .deactivate = trig_deactivate_pots0},
	{.name = "cdcs-pots1",  .activate = trig_activate_pots1, .deactivate = trig_deactivate_pots1},
	{.name = "cdcs-zigbee", .activate = trig_activate_zigbee, .deactivate = trig_deactivate_zigbee},
    {.name = "cdcs-beat",   .activate = trig_activate_beat,   .deactivate = trig_deactivate_beat},
	{.name = "cdcs-none",   .activate = trig_activate_none,   .deactivate = trig_deactivate_none},
	{NULL, NULL, NULL}
};


/*
^^^^^^^^^^^^ No further changes are required for more triggers, below this line ^^^^^^^^^^^^
*/

/* 'force' control values */
#define F_TRIG 0 /* LED uses trigger */
#define F_OFF  1 /* LED is solid (brightness_on)  */
#define F_ON   2 /* LED is solid (brightness_off) */
#define F_TRIG_CONT 3 /* LED runs trigger continously */

/* Defaults - These are the startup values used when a trigger is enabled.
They can be changed later via sysfs (/sys/class/leds/led-name/varname) . */
#define BRIGHTNESS_ON  LED_FULL
#define BRIGHTNESS_OFF LED_OFF
#define FORCE          F_TRIG
#define TRIGGER_PULSE  100
#define TRIGGER_DEAD   10

#define PATTERN_VALUE(td) ((td->pattern >> td->pattern_shift) & 0x1)

struct cdcs_trig_data {
	unsigned long brightness_on;   /* LED brightness in the 'on/triggered' state */
	unsigned long brightness_off;  /* LED brighness in the 'off/idle' state */

	int force;                     /* Solid at on/off/trigger/trigger_cont (F_TRIG,F_TRIG_CONT,F_OFF,F_ON) */

	unsigned long trigger_pulse;   /* [ms] trigger pulse */
	unsigned long trigger_dead;    /* [ms] trigger dead time */
	unsigned long trigger_now;     /* run trigger action now */
	unsigned long pattern;         /* bit pattern for cdcs-beat */

	int index;

	unsigned int last_activity;    /* Remembers last counter value */
	unsigned int activity;         /* Wrapping trigger event counter */
	int pattern_shift;             /* The current pattern shift */
	int trigger_continous;         /* run trigger action continous */

	int state;                     /* State machine */
#define STATE_OFF    0 /* Trigger is inactive (e.g. forced or unused). */
#define STATE_IDLE   1 /* No trigger. LED at brightness_off. */
#define STATE_PULSE  2 /* LED is at brightness_on */
#define STATE_DEAD   3 /* LED is at brightness_off (in deadzone) */

	struct timer_list timer;       /* Kernel timer to drive LED */
};
static struct cdcs_trig_data cdcs_triggers[CDCS_LAST]; /* We keep a local copy per trigger. */

static void led_timer_function(unsigned long data);

/* Initialises one trigger */
static void init_trigger(int index, struct led_classdev *led_cdev)
{
	struct cdcs_trig_data *td=&cdcs_triggers[index]; /* led_cdev may be NULL */
	debug1("(index=%d, led_cdev=%p)\n",index,(void*)led_cdev);
	td=&cdcs_triggers[index];
	td->brightness_on  = BRIGHTNESS_ON;
	td->brightness_off = BRIGHTNESS_OFF;
	td->force          = FORCE;
	td->trigger_pulse  = TRIGGER_PULSE;
	td->trigger_dead   = TRIGGER_DEAD;
	td->index          = index;
	td->last_activity  = 0;
	td->activity       = 0;
	td->pattern        = 0;
	td->pattern_shift  = -1;
	td->trigger_continous = 0;
	td->state          = (led_cdev)?STATE_IDLE:STATE_OFF;
	init_timer(&td->timer);
	td->timer.function = led_timer_function;
	td->timer.data     = (unsigned long)led_cdev;
}

static void deinit_trigger(int index, struct led_classdev *led_cdev)
{
	struct cdcs_trig_data *td = led_cdev->trigger_data;
	debug1("(index=%d, led_cdev=%p)\n",index,(void*)led_cdev);
	if (td) {
		td->state = STATE_OFF;
		del_timer(&td->timer);
		td->timer.data = 0;
	}
}

/* Generic trigger event, can be called in interrupt. */
static void ledtrig_cdcs(int index)
{
	struct cdcs_trig_data *td = &cdcs_triggers[index];
	if (td->state != STATE_OFF) {
		td->activity++;
		/* This is a race, but missing a trigger is not a big problem. */
		if (!timer_pending(&td->timer))
			mod_timer(&td->timer, jiffies + msecs_to_jiffies(10)); /* = soon. */
	}
}

/* Timer driven state machine. */
static void led_timer_function(unsigned long data)
{
	struct led_classdev *led_cdev = (struct led_classdev *) data;
	struct cdcs_trig_data *td;

	if (unlikely(!led_cdev)) return;
	td=led_cdev->trigger_data;
	if (unlikely(!td)) return;

	switch (td->state) {
		case STATE_OFF :
			break;
		case STATE_IDLE :
		case STATE_DEAD :
            if (td->pattern == 0) {
                /* Decide to activate the LED */
                if (td->activity != td->last_activity || td->trigger_continous) {
                    td->state = STATE_PULSE;
                    led_set_brightness(led_cdev,td->brightness_on);
                    mod_timer(&td->timer, jiffies +
                              msecs_to_jiffies(td->trigger_pulse));
                } else {
                    td->state = STATE_IDLE;
                    led_set_brightness(led_cdev,td->brightness_off);
                }
                break;
            } else {
                td->pattern_shift = -1;
                td->state = STATE_PULSE;
                /* Fall through to STATE_PULSE */
            }
	    case STATE_PULSE :
            if (td->pattern && (td->pattern >> ++td->pattern_shift)) {
                /*
                 * Pattern is enabled and have not reached the end of the
                 * pattern yet. Set the led based on the current pattern bit.
                 */
                if (PATTERN_VALUE(td)) {
                    led_set_brightness(led_cdev, td->brightness_on);
                } else {
                    led_set_brightness(led_cdev, td->brightness_off);
                }
                mod_timer(&td->timer,
                          jiffies + msecs_to_jiffies(td->trigger_pulse));
            } else {
                /* Start pulse dead zone. */
                td->state = STATE_DEAD;
                led_set_brightness(led_cdev,td->brightness_off);
                mod_timer(&td->timer, jiffies +
                          msecs_to_jiffies(td->trigger_dead));
            }
            break;
	}
	td->last_activity = td->activity;
}


/*
Sysfs variable control.
*/

/* Declare the variable 'show' functions for sysfs. */
/* led_brightness_on_show, led_brightness_off_show, */
/* led_trigger_pulse_show,led_trigger_dead_show */
#define DECLARE_SYSFS_SHOW(varname) \
	static ssize_t led_##varname##_show(struct device *dev, \
			struct device_attribute *attr, char *buf) \
	{ \
		struct led_classdev *led_cdev = dev_get_drvdata(dev); \
		struct cdcs_trig_data *td = led_cdev->trigger_data; \
		debug1("led_cdev=%p, td->index=%d\n",(void*)led_cdev,td->index); \
		return sprintf(buf, "%lu\n", td->varname); \
	}
DECLARE_SYSFS_SHOW(brightness_on)
DECLARE_SYSFS_SHOW(brightness_off)
DECLARE_SYSFS_SHOW(trigger_pulse)
DECLARE_SYSFS_SHOW(trigger_dead)
DECLARE_SYSFS_SHOW(trigger_now)
DECLARE_SYSFS_SHOW(pattern)

static ssize_t led_force_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct cdcs_trig_data *td = led_cdev->trigger_data;
	const char *fmt;
	switch (td->force) {
		case F_TRIG :
			fmt="[%s] %s %s %s\n";
			break;
		case F_TRIG_CONT :
			fmt="%s [%s] %s %s\n";
			break;
		case F_OFF :
			fmt="%s %s [%s] %s\n";
			break;
		case F_ON :
			fmt="%s %s %s [%s]\n";
			break;
		default :
			fmt="%s %s %s %s [???]\n";
			break;
	}
	return sprintf(buf, fmt, "trigger", "trigger_cont", "off", "on");
}

/* Macro to create a variable based store */
#define DECLARE_SYSFS_STORE(varname,updatecode) \
	static ssize_t led_##varname##_store(struct device *dev, \
			struct device_attribute *attr, const char *buf, size_t size) \
	{ \
		struct led_classdev *led_cdev = dev_get_drvdata(dev); \
		struct cdcs_trig_data *td = led_cdev->trigger_data; \
		int ret = -EINVAL; \
		char *after; \
		unsigned long newval = simple_strtoul(buf, &after, 10); \
		size_t count = after - buf; \
		if (*after && isspace(*after)) \
			count++; \
		if (count == size) { \
			if (td->varname != newval) { \
				/* the new value differs from the previous */ \
				td->varname = newval; \
				/* Update LED state */ \
				updatecode \
			} \
			ret = count; \
		} \
		return ret; \
	}

/* Macro to create a store with no backing variable/value */
#define DECLARE_SYSFS_STORE_NOVAL(varname,updatecode) \
	static ssize_t led_##varname##_store(struct device *dev, \
			struct device_attribute *attr, const char *buf, size_t size) \
	{ \
		struct led_classdev *led_cdev = dev_get_drvdata(dev); \
		struct cdcs_trig_data *td = led_cdev->trigger_data; \
		/* Update LED state */ \
		updatecode \
		return size; \
	}

/* Declare 'store' functions for sysfs. */
DECLARE_SYSFS_STORE(brightness_on, { \
    if (td->force == F_ON) { \
        led_set_brightness(led_cdev,td->brightness_on); \
    } else if (td->state == STATE_PULSE) { \
        if ((td->pattern == 0) || PATTERN_VALUE(td)) { \
            led_set_brightness(led_cdev,td->brightness_on); \
        } else { \
            led_set_brightness(led_cdev,td->brightness_off); \
        } \
    } \
})

DECLARE_SYSFS_STORE(brightness_off, { \
    if ((td->force == F_OFF) || (td->state != STATE_PULSE) || \
        ((td->state == STATE_PULSE) && (td->pattern != 0) && \
         !PATTERN_VALUE(td))) { \
        led_set_brightness(led_cdev,td->brightness_off); \
    } \
})

DECLARE_SYSFS_STORE(trigger_pulse,{})
DECLARE_SYSFS_STORE(trigger_dead,{})

DECLARE_SYSFS_STORE(pattern, { \
    if (!timer_pending(&td->timer)) \
        mod_timer(&td->timer, jiffies + msecs_to_jiffies(10)); \
})

DECLARE_SYSFS_STORE_NOVAL(trigger_now, {\
	td->state = STATE_IDLE;                             \
	led_set_brightness(led_cdev, td->brightness_off);   \
	ledtrig_cdcs(td->index);                            \
})

static ssize_t led_force_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t size)
{
	struct led_classdev *led_cdev = dev_get_drvdata(dev);
	struct cdcs_trig_data *td = led_cdev->trigger_data;
	int ret = -EINVAL;
	int newval = -1;
	const char *p;

	p=buf;
	while (*p && isspace(*p)) p++;
	if      (strncmp("trigger_cont",p,sizeof("trigger_cont")-1)==0) newval = F_TRIG_CONT;
	else if (strncmp("trigger",p,sizeof("trigger")-1)==0) newval = F_TRIG;
	else if (strncmp("off",p,sizeof("off")-1)==0) newval = F_OFF;
	else if (strncmp("on",p,sizeof("on")-1)==0) newval = F_ON;
	else newval=-1;

	if (newval != -1) {
		if (td->force != newval) {
			td->force = newval;

			if (td->force == F_TRIG_CONT) {
				td->trigger_continous = 1;
			}
			else {
				td->trigger_continous = 0;
			}

			switch (newval) {
				case F_TRIG_CONT:
				case F_TRIG :
					td->state = STATE_IDLE;
					led_set_brightness(led_cdev,td->brightness_off);
					ledtrig_cdcs(td->index);
					break;
				case F_OFF :
					td->state = STATE_OFF;
					if (timer_pending(&td->timer)) del_timer_sync(&td->timer);
					led_set_brightness(led_cdev,td->brightness_off);
					break;
				case F_ON :
					td->state = STATE_OFF;
					if (timer_pending(&td->timer)) del_timer_sync(&td->timer);
					led_set_brightness(led_cdev,td->brightness_on);
					break;
			}
		}
		ret=size;
	}
	return ret;
}

#define DECLARE_SYSFS_ATTR(varname) \
	static DEVICE_ATTR(varname, S_IRUGO | S_IWUSR, led_##varname##_show, led_##varname##_store);
DECLARE_SYSFS_ATTR(brightness_on)
DECLARE_SYSFS_ATTR(brightness_off)
DECLARE_SYSFS_ATTR(trigger_pulse)
DECLARE_SYSFS_ATTR(trigger_dead)
DECLARE_SYSFS_ATTR(trigger_now)
DECLARE_SYSFS_ATTR(force)
DECLARE_SYSFS_ATTR(pattern)

/* Help function */
static ssize_t led_help_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	int len=0;
	int i;
	len+=sprintf(buf+len,"\nUsage of the CDCS trigger group (");
	for (i=0; i<CDCS_LAST-1; i++)
		len+=sprintf(buf+len,"%s ",led_triggers[i].name);
	len+=sprintf(buf+len,"%s):\n\n",led_triggers[i].name);
	len+=sprintf(buf+len,"\tbrightness_on  = LED brightness in the 'on/triggered' state\n");
	len+=sprintf(buf+len,"\tbrightness_off = LED brighness in the 'off/idle' state\n");
	len+=sprintf(buf+len,"\tforce          = 'trigger' triggered operation, 'trigger_cont' triggered operation continously, 'on'/'off solid - trigger disabled.\n");
	len+=sprintf(buf+len,"\ttrigger_pulse  = [ms] trigger pulse\n");
	len+=sprintf(buf+len,"\ttrigger_dead   = [ms] trigger dead time (minimum off)\n");
	len+=sprintf(buf+len,"\ttrigger_now    = run the trigger action now\n");
    len+=sprintf(buf+len,"\tpattern        = bit pattern for cdcs-beat\n");
	len+=sprintf(buf+len,"\thelp           = This stuff\n");
	len+=sprintf(buf+len,"\n");
	return len;
}
static DEVICE_ATTR(help, S_IRUGO, led_help_show, NULL);

/* Generic setup/teardown  functions */

static void cdcs_unravel(int stage, int index, struct led_classdev *led_cdev)
{
	/* Undo initialisation steps, falling through a case statement. */
	if (stage) printk(KERN_INFO MODULE_NAME "unraveling from stage %d\n",stage);
	switch (stage) {
#define STAGE_all  0
		default :

#define STAGE_pattern 9
            if (index == CDCS_BEAT) {
                device_remove_file(led_cdev->dev, &dev_attr_pattern);
            }
		case STAGE_pattern :

#define STAGE_help 8
			device_remove_file(led_cdev->dev, &dev_attr_help);
		case STAGE_help :

#define STAGE_trigger_now 7
			device_remove_file(led_cdev->dev, &dev_attr_trigger_now);
		case STAGE_trigger_now :

#define STAGE_trigger_dead 6
			device_remove_file(led_cdev->dev, &dev_attr_trigger_dead);
		case STAGE_trigger_dead :

#define STAGE_trigger_pulse 5
			device_remove_file(led_cdev->dev, &dev_attr_trigger_pulse);
		case STAGE_trigger_pulse :

#define STAGE_force 4
			device_remove_file(led_cdev->dev, &dev_attr_force);
		case STAGE_force :

#define STAGE_brightness_off 3
			device_remove_file(led_cdev->dev, &dev_attr_brightness_off);
		case STAGE_brightness_off :

#define STAGE_brightness_on 2
			device_remove_file(led_cdev->dev, &dev_attr_brightness_on);
		case STAGE_brightness_on :

#define STAGE_init 1
			deinit_trigger(index,led_cdev);
		case STAGE_init :

			/* nothing */
			break;
	}
}

static void trig_activate(int index, struct led_classdev *led_cdev)
{
	struct cdcs_trig_data *td;
	int rc;

	td=&cdcs_triggers[index];
	led_cdev->trigger_data=td;

	init_trigger(index,led_cdev);

#define CREATE_SYSFS(name) \
	rc = device_create_file(led_cdev->dev, &dev_attr_##name); \
	if (rc) { \
		cdcs_unravel(STAGE_##name, index, led_cdev); \
		goto error; \
	}

	CREATE_SYSFS(brightness_on)
	CREATE_SYSFS(brightness_off)
	CREATE_SYSFS(force)
	CREATE_SYSFS(trigger_pulse)
	CREATE_SYSFS(trigger_dead)
	CREATE_SYSFS(trigger_now)
	CREATE_SYSFS(help)

    if (index == CDCS_BEAT) {
	    CREATE_SYSFS(pattern)
    }

	return;
error:
	printk(KERN_INFO MODULE_NAME "Failed to activate trigger %s\n",led_triggers[index].name);
}

static void trig_deactivate(int index, struct led_classdev *led_cdev)
{
	cdcs_unravel(STAGE_all,index,led_cdev);
}

static int __init cdcs_trig_init(void)
{
	int rval=0;
	int index;
	printk(KERN_INFO MODULE_NAME "Registering CDCS LED trigger group. (%s %s)\n",__DATE__,__TIME__);
	for (index=0; index<CDCS_LAST && rval==0; index++) {
		init_trigger(index,NULL); /* Makes sure there is no race condition with trigger function */
		rval=led_trigger_register(&led_triggers[index]);
	}
	if (rval != 0) {
		printk(KERN_INFO MODULE_NAME "Registering triggers failed at index=%d, rval=%d\n",index,rval);
	}
	return rval;
}

static void __exit cdcs_trig_exit(void)
{
	int index;
	printk(KERN_INFO MODULE_NAME "Unregistering CDCS LED trigger group.\n");
	for (index=0; index<CDCS_LAST; index++) {
		led_trigger_unregister(&led_triggers[index]);
	}
}

module_init(cdcs_trig_init);
module_exit(cdcs_trig_exit);

MODULE_AUTHOR("Iwo Mergler <Iwo@call-direct.com.au>");
MODULE_DESCRIPTION("CDCS LED trigger set");
MODULE_LICENSE("GPL");
