/*
 * Qualcomm PHY drivers + ethtest driver
 *
 */
#include <common.h>
#include <command.h>
#include <exports.h>
#include <config.h>

#include <net.h>
#include <miiphy.h>
#include <netdev.h>

#include <phy.h>
#include "ethtest.h"

//QUALQOMM_ETH_TEST defined since the test signal is not implemented here, JZ 09/06/2015
#define QUALQOMM_ETH_TEST

/* Accessing Qualcomm debug registers */

__attribute__((unused)) static int debug_read(struct phy_device *phydev, unsigned short dreg, unsigned short *val)
{
	int rval;
	rval = phy_write(phydev, phydev->addr, 0x1D, dreg & 0x3f); /* Select pair */
	printf("deb read addr %d dreg 0x%x rval 0x%x ", phydev->addr, dreg, rval);
	/* if (!rval) */ rval = phy_read(phydev, phydev->addr, 0x1E); /* Register Value */
	printf("rval out 0x%x\n", rval);
	if (rval >= 0) *val=rval & 0xFFFF;
	return rval;
}

__attribute__((unused)) static int debug_write(struct phy_device *phydev, unsigned short dreg, unsigned short val)
{
	int rval;
	rval = phy_write(phydev, phydev->addr, 0x1D, dreg & 0x3f); /* Select pair */
	printf("deb write addr %d dreg 0x%x rval 0x%x att val 0x%x\n", phydev->addr, dreg, rval, val);
	/* if (!rval) */ rval = phy_write(phydev, phydev->addr, 0x1E, val); /* Register Value */
	return rval;
}

unsigned int
athrs17_reg_read(struct phy_device *phydev,unsigned int reg_addr)
{
    uint32_t reg_word_addr;
    uint32_t phy_addr, tmp_val, reg_val;
    uint16_t phy_val;
    uint8_t phy_reg;
    int tempAD= phydev->addr;

    /* change reg_addr to 16-bit word address, 32-bit aligned */
    reg_word_addr = (reg_addr & 0xfffffffc) >> 1;

    /* configure register high address */
    phy_addr = 0x18;
    phy_reg = 0x0;
    phy_val = (uint16_t) ((reg_word_addr >> 8) & 0x3ff);  /* bit16-8 of reg address */
    phydev->addr = phy_addr ;
    phy_write(phydev, phy_addr, phy_reg, phy_val);

    /* For some registers such as MIBs, since it is read/clear, we should */
    /* read the lower 16-bit register then the higher one */

    /* read register in lower address */
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
    phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
    phydev->addr = phy_addr ;
    reg_val = (uint32_t) phy_read(phydev, phy_addr, phy_reg);

    /* read register in higher address */
    reg_word_addr++;
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit7-5 of reg address */
    phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit4-0 of reg address */
    phydev->addr = phy_addr ;
    tmp_val = (uint32_t) phy_read(phydev, phy_addr, phy_reg);
    reg_val |= (tmp_val << 16);
    
    phydev->addr=tempAD;
    
//    printf("[%s,(%s),%d] Location read: %x:%x\n",__FILE__,__func__,__LINE__,reg_addr,reg_val);
    return reg_val;
}


void athrs17_reg_write(struct phy_device *phydev,unsigned int reg_addr, unsigned int reg_val)
{
    uint32_t reg_word_addr;
    uint32_t phy_addr;
    uint16_t phy_val;
    uint8_t phy_reg;
    int tempAD= phydev->addr; // Save the Default address of the driver and load it back when the setting is finsihed.

    /* change reg_addr to 16-bit word address, 32-bit aligned */
    reg_word_addr = (reg_addr & 0xfffffffc) >> 1;

    
    /* configure register high address */
    phy_addr = 0x18;
    phy_reg = 0x0;
    // Bit 18-9
    phy_val = (uint16_t) ((reg_word_addr >> 8) & 0x3ff);  /* bit18-9 of reg address */
//    phy_write(phydev,(uint8_t)phy_addr, phy_reg, phy_val);
    phydev->addr = phy_addr ;
    phy_write(phydev,(uint8_t)phy_addr, phy_reg, phy_val);
    

    /* For S17 registers such as ARL and VLAN, since they include BUSY bit */
    /* in higher address, we should write the lower 16-bit register then the */
    /* higher one */

    /* write register in lower address */
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit8-6 of reg address */
    phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit5-1 of reg address */
    phy_val = (uint16_t) (reg_val & 0xffff);
    phydev->addr = phy_addr ;
    phy_write(phydev,(uint8_t)phy_addr, phy_reg, phy_val);
    
    /* read register in higher address */
    reg_word_addr++;
    phy_addr = 0x10 | ((reg_word_addr >> 5) & 0x7); /* bit8-6 of reg address */
    phy_reg = (uint8_t) (reg_word_addr & 0x1f);   /* bit5-1 of reg address */
    phy_val = (uint16_t) ((reg_val >> 16) & 0xffff);
    phydev->addr = phy_addr ;
    phy_write(phydev,(uint8_t)phy_addr, phy_reg, phy_val);
    
    phydev->addr=tempAD;
    
//    printf("[%s,(%s),%d] Location write: %x.%x:%x\n",__FILE__,__func__,__LINE__,phy_addr,reg_addr,reg_val);    
}


static int qca8337_config(struct phy_device *phydev)
{
      /* configure the RGMII */
	printf("Enabling RGMII\n");
	printf("[%s,(%s),%d] Location \n",__FILE__,__func__,__LINE__);
	athrs17_reg_write(phydev,0x624 , 0x007f7f7f);
	athrs17_reg_write(phydev,0x10  , 0x40000000);
	athrs17_reg_write(phydev,0x4   , 0x07600000);
	athrs17_reg_write(phydev,0xc   , 0x01000000);
	athrs17_reg_write(phydev,0x7c  , 0x0000007e);
//	return genphy_config(phydev);
	return 0;
}

/* Reset */
static int qca8337_reset(struct phy_device *phydev,char phyNo)
{
	int v;
	int tempValue=phydev->addr;
	phydev->addr=phyNo;

//	printf("[%s,(%s),%d] Location Phy no: %d\n",__FILE__,__func__,__LINE__,phyNo);
	/* Software reset, autoneg / speed setting survives
	 * reset, MDIX and a few other setting applies at reset. */
	v = phy_read(phydev, (uint8_t)phyNo, MII_BMCR);
	printf("[%s,%d] Phyno: %x, v= %x\n",__FILE__,__LINE__,phyNo,v);
	phy_write(phydev, (uint8_t)phyNo, MII_BMCR, v | BMCR_RESET );
	while (phy_read(phydev, (uint8_t)phyNo, MII_BMCR) & BMCR_RESET);
	
	phydev->addr=tempValue;
	return 0;
}


/* Reset & Enable autonegotiation */
static int qca8337_aneg(struct phy_device *phydev, char phyNo)
{
	int v;
	int tempValue=phydev->addr;
	phydev->addr=phyNo;

//	printf("[%s,(%s),%d] Location Phy no: %d\n",__FILE__,__func__,__LINE__,phyNo);
	/* Enable auto-MDIX. The testmodes may disable it. */
	v = phy_read(phydev, (uint8_t)phyNo, 0x10);
	phy_write(phydev, (uint8_t)phyNo, 0x10, v | 0x60);

	/* Enable autoneg. */
	v = BMCR_FULLDPLX | BMCR_ANENABLE | BMCR_SPEED100;
	phy_write(phydev, (uint8_t)phyNo, MII_BMCR, v);

	qca8337_reset(phydev,phyNo);

	/* Restart aneg */
	phy_write(phydev, (uint8_t)phyNo, MII_BMCR, v | BMCR_ANRESTART );

	/* We don't wait here, the network stack will poll link state anyway. */	
	phydev->addr=tempValue;
	
	return 0;
}

/* This is called once, on first use. We use the opportunity
 * to set up the RGMII delays. Shouldn't do anything else.
 * WARNING: Also called by 1G test mode below, to revert a hack. */
static int qca8337_probe(struct phy_device *phydev)
{
//	printf("[%s,(%s),%d] Location \n",__FILE__,__func__,__LINE__);
	phy_write(phydev, 0, MII_BMCR, BMCR_RESET);
	/*cpu port en*/
	athrs17_reg_write(phydev,0x620, (athrs17_reg_read(phydev,0x620)|0x400));

	///* enable L3 offload */
//	athrs17_reg_write(phydev,0x30, (athrs17_reg_read(phydev,0x30) | 0x4));
// reset signal comming by processor to restart the switch, leads the boot strap to configure it in unknown condition !!! JZ 9/6/2015

	athrs17_reg_write(phydev,0x624 , 0x007f7f7f);
	athrs17_reg_write(phydev,0x4   , 0x07600000);	// RGMII on port 0, Delayed
	athrs17_reg_write(phydev,0x8   , 0x01000000);	// enable the RX clock ....
	athrs17_reg_write(phydev,0xc   , 0x04200000);	// RGMII on port pad 6, only RX Clk Delay
	athrs17_reg_write(phydev,0x7c  , 0x007e);	// port 0 status..
	athrs17_reg_write(phydev,0x94  , 0x007e);	// port 6 status

	// leds settings
	// Amber:
	// on  -- 1000Mbpx
	// off -- 100/10 Mbps or no cable
	athrs17_reg_write(phydev,0x50  , 0xcc05);
	// Green:
	// on -- link up
	// blink -- tx/rx
	// off -- no link
	athrs17_reg_write(phydev,0x54  , 0xcf35);
	athrs17_reg_write(phydev,0x58  , 0xcf35);

	// only enable leds for port1 and port2
	athrs17_reg_write(phydev,0x5c  , 0xfff00);

	printf("[%s,(%s),%d] Disabled port1(WAN/LAN port)\n",__FILE__,__func__,__LINE__);
	athrs17_reg_write(phydev, 0x420, 0x10001);
	athrs17_reg_write(phydev, 0x424, 0x48);
	athrs17_reg_write(phydev, 0x660, 0x14017C); //PORT0_LOOKUP_CTRL: PORT_VID_MEM_0 - b1111100
	athrs17_reg_write(phydev, 0x428, 0x10001);
	athrs17_reg_write(phydev, 0x42C, 0x1048);
	athrs17_reg_write(phydev, 0x66C, 0x14007D); //PORT1_LOOKUP_CTRL: PORT_VID_MEM_1 - b1111101
	athrs17_reg_write(phydev, 0x430, 0x10001);
	athrs17_reg_write(phydev, 0x434, 0x1048);
	athrs17_reg_write(phydev, 0x678, 0x140079); //PORT2_LOOKUP_CTRL: PORT_VID_MEM_2 - b1111001
	athrs17_reg_write(phydev, 0x450, 0x10001);
	athrs17_reg_write(phydev, 0x454, 0x48);
	athrs17_reg_write(phydev, 0x6A8, 0x14013D); //PORT6_LOOKUP_CTRL: PORT_VID_MEM_6 - b0111101

	int temasd=phydev->addr ;
	int i;	// work around ....
	for ( i=0;i< 5 ; i++)
	{
	  phydev->addr = i ;
	  /* For 100M waveform */
	  phy_write(phydev, (uint8_t)i, 0x1d, 0x0);
	  phy_write(phydev, (uint8_t)i, 0x1e, 0x02ea);
	  /* Turn On Gigabit Clock */
	  phy_write(phydev, (uint8_t)i, 0x1d, 0x3d);
	  phy_write(phydev, (uint8_t)i, 0x1e, 0x68a0);
	  
	  qca8337_aneg(phydev,i);	  	// considering setting address to "->addr", help to each phy take setting and autonegotiation ....
	  
	}
	phydev->addr=temasd;
	return 0;
}


static int qca8337_startup(struct phy_device *phydev)
{
	/* The genphy link function always waits for autoneg, even if nothing
	 * is connected. This takes too long. We assume autoneg has happened already.
	 * Reading status twice to clear latched state. */
//	printf("[%s,(%s),%d] Location \n",__FILE__,__func__,__LINE__);
	phy_read(phydev, phydev->addr, MII_BMSR);
	phydev->link = !!(phy_read(phydev, phydev->addr, MII_BMSR) & BMSR_LSTATUS);
	/* Work out speed & duplex */
	return genphy_parse_link(phydev);
}

static int qca8337_shutdown(struct phy_device *phydev)
{
	return genphy_shutdown(phydev);
}


struct phy_driver QCA8337_driver =  {
	.name = "QCA8337",
	.uid =  0x004dd036,
	.mask = 0x00fffff0,
	.features = PHY_GBIT_FEATURES,
	.config = qca8337_config,
	.probe = qca8337_probe,
	.startup = qca8337_startup,
	.shutdown = qca8337_shutdown,
};

#ifdef QUALQOMM_ETH_TEST
static int ethtest_probe(struct ethtest_driver *ethtest)
{
	/* Nothing to do here atm., but if we have to match multiple IDs
	 * for different PHYs, this is the place to do it. */
	return 0;
}

/* Time domain reflectometry cable test */
static int ethtest_cable(struct ethtest_driver *ethtest)
{
	unsigned v, dist;
	int val;
	struct phy_device *phydev = ethtest->phydev;

	printf("	Cable test\n");
	/* Iterate over cable pairs */
	for (v=0; v<4; v++) {
		printf("		Pair %d, ", v);
		phy_write(phydev, phydev->addr, 0x16, (v << 8) | 0 ); /* Select pair */
		phy_write(phydev, phydev->addr, 0x16, (v << 8) | 1 ); /* start test */
		do {
			val = phy_read(phydev, phydev->addr, 0x16);
		} while (val & 1); /* Wait until done */
		val = phy_read(phydev, phydev->addr, 0x1C); /* Read result */
		switch ((val >> 8) & 0x3) {
			case  0: printf("OK, "); break;
			case  1: printf("SHORT, "); break;
			case  2: printf("OPEN, "); break;
			case  3: printf("FAIL, "); break;
		}
		val &= 0xFF;
		dist = (float)val*84.2;
		printf("Distance: %d.%d (%d)\n", dist/100, dist%100, val);
	}
	return 0;
}

/* 10 MBit test modes */
static int ethtest_test10(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	if (mode != PT_MODE_NORMAL) {
		/* Force 10MBit/s, full duplex */
		phy_write(phydev, phydev->addr, MII_BMCR, BMCR_FULLDPLX );

		/* Force MDI */
		v = phy_read(phydev, phydev->addr, 0x10);
		phy_write(phydev, phydev->addr, 0x10, (v & ~0x60));

		/* MDIX setting takes effect after reset only */
		qca8337_reset(phydev, phydev->addr);
	}

	v = 0x0C; /* These bits are default 1, read only, but the datasheet is lying. */
	switch (mode) {
		case PT10_MODE_10MHZ : v |= 0x01; break;
		case PT10_MODE_5MHZ : v |= 0x20; break;
		case PT10_MODE_RANDOM : v |= 0x02; break;
		case PT10_MODE_LINK : v |= 0x03; break;
		default : v |= 0; break;
	}

	debug_write(phydev, 0x12, v);

	/* Reset PHY to return to sanity */
	if (mode == PT_MODE_NORMAL) {
		qca8337_aneg(phydev, phydev->addr);
	}

	return 0;
}

/* 100 MBit test modes */
static int ethtest_test100(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	if (mode != PT_MODE_NORMAL) {
		/* Force 100MBit/s, full duplex */
		phy_write(phydev, phydev->addr, MII_BMCR,
				BMCR_SPEED100 | BMCR_FULLDPLX );
		/* Force MDI */
		v = phy_read(phydev, phydev->addr, 0x10);
		phy_write(phydev, phydev->addr, 0x10, (v & ~0x60));

		/* MDIX setting takes effect after reset only */
		qca8337_reset(phydev, phydev->addr);
	}

	switch (mode) {
		case PT100_MODE_JITTER: v = 0x80; break;
		case PT100_MODE_OVERSHOOT: v = 0x40; break;
		case PT100_MODE_DCD: v = 0x20; break;
		default : v = 0; break;
	}

	debug_write(phydev, 0x10, v);

	/* Reset PHY to return to sanity */
	if (mode == PT_MODE_NORMAL) {
		qca8337_aneg(phydev, phydev->addr);
	}

	return 0;
}

/* Gigabit test modes */
static int ethtest_test1G(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	if (mode != PT_MODE_NORMAL) {
		/* Force 1GBit/s, full duplex */
		phy_write(phydev, phydev->addr, MII_BMCR,
				BMCR_SPEED1000 | BMCR_FULLDPLX );

		qca8337_reset(phydev, phydev->addr);

		/* WARNING: Hack for undocumented 'feature'. We have to
		 * set a few debug register bits which are marked as
		 * 'reserved' and 'read only' in the datasheet.
		 * Without them, Gigabit test modes don't work.
		 * This breaks the RGMII TxCLK delay setting, so we
		 * fix that later. */
		debug_write(phydev, 0xB, 5);
	}

	/* All duplex bits, manual master */
	switch (mode) {
		case PT1G_MODE_WAVE: v = 0x1F00 | (0x1 << 13); break;
		case PT1G_MODE_JITTER_MASTER: v = 0x1F00 | (0x2 << 13); break;
		case PT1G_MODE_JITTER_SLAVE:  v = 0x1F00 | (0x3 << 13); break;
		case PT1G_MODE_DISTORTION:  v = 0x1F00 | (0x4 << 13); break;
		default: v = 0x0200; /* Normal, auto everything */ break;
	}

	phy_write(phydev, phydev->addr, 0x09, v);

	/* Reset PHY to return to sanity */
	if (mode == PT_MODE_NORMAL) {
		/* The undocumented hack above has garbled the TxCLK delay.
		 * We call the probe function because it restores the delay. */
		qca8337_probe(phydev);
		qca8337_aneg(phydev, phydev->addr);
	}

	return 0;
}

struct ethtest_driver ethtdrv = {
	.name = "qualcomm",
	.uid =  0x004dd036,
	.mask = 0x00fffff0,
	.probe = ethtest_probe,
	.cable = ethtest_cable,
	.test10 = ethtest_test10,
	.test100 = ethtest_test100,
	.test1G = ethtest_test1G,
};

#endif

int phy_qualcomm_ethtest_init(void)
{
	phy_register(&QCA8337_driver);
#ifdef QUALQOMM_ETH_TEST
	ethtest_register(&ethtdrv);
#endif	
	return 0;
}
