/*
 * Algorithm library for U-Boot environment handling. Supports legacy
 * U-Boot format (CRC32,DATA), redundant format (CRC32,FLAG,DATA) and
 * Netcomm high-reliability format (redundant over arbitrary number of
 * blocks & sub-blocks).
 *
 * A U-Boot environment block has the following structure:
 *
 * CRC32  4   CRC of data area, without optional flag field.
 * [flag] 1   Optional flag (sequence) field, present on redundant blocks only.
 * NAME   N   Name of variable
 * '='    1   name/value separator
 * VALUE  M   Value of variable
 * '\0'   1   field separator
 * ...        More NAME=VALUE\0 fields
 * '\0'   1   End of env block, makes a double \0 with the end of the last field.
 *
 * A legacy environment area contains a single environment block without the flag field.
 * Any change to the environment requires a block erase and a re-write. Between the
 * beginning of the erase and the end of the write this scheme is vulnerable to
 * complete loss in case of a power cut. Beyond that, this method can place a
 * heavy endurance load on the one eraseblock in case of frequent changes and is
 * unable to cope with a bad block at its location.
 *
 * A redundant environment uses two separate eraseblocks, each containing one
 * environment block with a value 0-255 in the flag field. The flag value acts
 * as a sequence counter. When reading, the block with the highest value (considering
 * wrap-around!) is used, when writing, the one with the lower one is replaced.
 * This scheme is not vulnerable to power cut and provides better endurance than
 * the legacy one. Bad eraseblocks are not handled well, due to a bug in U-Boot.
 *
 * The Netcomm high-reliability format (NHRF) allows the use of more than 2
 * erase blocks. It also makes use of the whole erase block by writing mutiple
 * versions of the environment into the same block. This significantly increases
 * the life span of the storage, since only erase cycles count towards the endurance
 * limit. The individual environment blocks are of the 'redundant' type. In case
 * of bad blocks, the NHRF algorithm deals with them gracefully.
 *
 * The master copy of this file lives in the cdcs_apps/uenv. Copies may be used
 * within the U-Boot source, but must be unmodified.
 *
 * Iwo Mergler <Iwo@netcommwireless.com>
 */
#ifndef _UENV_ALG_H
#define _UENV_ALG_H

#ifndef __KERNEL__
	#include <stdint.h> /* *int*_t types */
	#include <sys/types.h> /* For loff_t */
#else
	#ifndef abs
		#define abs(i) ({typeof(i) ii = (i); ii < 0 ? -ii : ii; })
	#endif
#endif

/* Environment entry in a format suitable for manipulation. This can be forwards or backwards
 * converted from/to a simple buffer representation. */
struct env_record {
	char *name;  /* Points to copy of name. */
	char *value; /* Points to copy of value. */
	struct env_record *prev;
	struct env_record *next;
	/* Storage for name & value copies follows. */
};

struct env_records {
	struct env_record *begin; /* Start of record list. */
	struct env_record *end;   /* End of record list. */
};

/* Single uenv entry descriptor. Contains the per-environment information
 * but not the data. */
struct uenv_desc {
	struct uenv_desc *next; /* List handling */

	loff_t   offset; /* Env offset in NAND */
	uint32_t size; /* Detected env size */

	#define UF_ERROR   (1<<0) /* We got an uncorrectable ECC error while reading this block */
	#define UF_ERASED  (1<<1) /* When bit is set, field up to envsize is erased */
	#define UF_CRCOK   (1<<2) /* When bit set, field is active, CRC was correct. */
	#define UF_REDFLAG (1<<3) /* When bit set, the entry has redundancy flag field. */

	#define UF_REDWR   (1<<4) /* When writing, write this block as redundant */
	#define UF_ERASE   (1<<5) /* Erase underlying erasblock before writing env to this. */
	#define UF_WRITE   (1<<6) /* Set when this block is picked as write target. */
	/* For writing, write redundant if UF_REDFLAG is set, else legacy. */
	uint8_t flags; /* Env type flags */

	uint8_t flag;  /* Redundancy flag of the env, only valid when flags & UF_REDFLAG. */
};

/* Whole environment, including descriptor list. */
struct uenv {
	/* Env settings, type defaults to use for writing. */
	loff_t   envbase;    /* start of environment area */
	unsigned envsize;    /* Used for writing. */
	unsigned eraseblock; /* NAND eraseblock size */
	unsigned stride;     /* Distance between envs, same as eraseblock for leg & red */
	unsigned totsize;    /* Total size of env area */
	unsigned menvsize;   /* Max. size of one env block. */
	#define UT_AUTO      (1<<0) /* Autodetect env type. */
	#define UT_FORCE     (1<<1) /* If set, force type for reads too. */
	#define UT_LEGACY    (1<<2) /* Use legacy env type for writing. envbase/envsize. */
	#define UT_REDUNDANT (1<<3) /* Use redundant env type for writing. envbase & envbase+stride */
	#define UT_NHRF      (1<<4) /* Use Netcomm env type for writing. Considers whole envbase/totsize */
	unsigned type;

	/* Autodetected global env Information. */
	struct uenv_desc *read_env;   /* pointer to descriptor for env to read */
	struct uenv_desc *write_env;  /* pointer to descriptor for env to write */

	uint8_t *buffer; /* Buffer for one whole max. (menvsize) environment, including CRC & flag */

	struct uenv_desc *ud_start; /* Linked list for env descriptors. */
	struct uenv_desc *ud_end;
};


/** Environment operations function pointer, provided by user.
 *
 * This function pointer is passed as an argument into several uenv functions to
 * provide NAND access fuctionality. start/size will never stradle an eraseblock
 * boundary.
 *
 * @param buf   Buffer to read to / write from.
 * @param start Start offset of env block in NAND
 * @param size  Size of env block in NAND
 * @param flags Flags field determine the operations to execute. Multiple flags
 *              can be specified simultaneously, in which case the operations are
 *              executed in the order of this definition:
 *
 *	EO_ERASE  = Erase underlying erase block
 *	EO_WRITE  = Write current buffer to start/size
 *	EO_VERIFY = Verify / Compare
 *	EO_READ   = Read env block at start/size into buffer
 *	EO_MRKBD  = Mark underlying erase block as bad
 *
 * @return Returns 0 on success and != 0 on failure. In case multiple operations
 *         are specified, the exact operation that failed is considered unknown.
 */
typedef int (*envop_fun)(uint8_t *buf, loff_t start, unsigned size, unsigned flags);
#define EO_ERASE  (1<<0)
#define EO_WRITE  (1<<1)
#define EO_VERIFY (1<<2)
#define EO_READ   (1<<3)
#define EO_MRKBD  (1<<4)


/*
 *   Env data management functions.
 */

/** Allocate struct uenv.
 *
 * Please set the settings fields (envbase, envsize, eraseblock, stride
 * totsize, menvsize, type, readenv) to sensible values.
 *
 * @param max_size Maximum expected size for any environment block
 *
 * @return Pointer to an environment descriptor or NULL in case of error.
 */
struct uenv * uenv_init(unsigned max_size);


/** Releases all resources used by a uenv structure.
 *
 * @param ue Pointer to uenv to free.
 */
void uenv_free(struct uenv *ue);

/** Debug - dumps whole uenv structure in human readable form
 *
 * This only works if DEBUG or TEST macros are defined during compile.
 *
 * @param ue Pointer to uenv data structure to dump.
 */
void uenv_dump(struct uenv *ue, int (*pr)(const char *fmt, ...));


/** Set up env for autodetect
 *
 * This function is an example of how to set up uenv. This particular
 * configuration sets up for autodetection of environment. In this mode,
 * existing environments are found and can be overwritten, but not
 * created. Writes will mirror the type and size of the found blocks.
 *
 * TODO: Move the doc into the include file.
 *
 * @param ue         Allocated struct uenv.
 * @param envbase    Base offset of env area on storage
 * @param stride     Stride of possible env block locations.
 * @param totsize    Total size of environment area
 */
static inline void uenv_set_autoscan(struct uenv *ue, loff_t envbase, unsigned stride, unsigned totsize)
{
	/* Base address of the environment area. All possible environment blocks
	 * must be part of the area. */
	ue->envbase = envbase;

	/* Default environment block size. This value is only used to write to new
	 * locations. We set it 0 here, since we won't write to new locations
	 * in automatic mode. */
	ue->envsize = 0;

	/* NAND erase block size, usually 128 or 256K. Unless writing in Netcomm mode,
	 * this should be the same or smaller than stride. */
	ue->eraseblock = 128*1024;

	/* This defines the search grid used for environment blocks. The base address of
	 * every env block can be expressed as ( ue->envbase + ue->stride * n), n=0...
	 * For legacy and redundant environments, this stride is identical or larger
	 * than the NAND eraseblock, for Netcomm env it can be smaller than the erase block. */
	ue->stride = stride;

	/* Total size of the environment area. All possible environment blocks
	 * must be part of the area. The totsize together with stride control
	 * the memory usage of the env structure (const+28*totsize/stride). */
	ue->totsize = totsize;

	/* Controls what env types are detected or written. This is a bit field:
	 *
	 * UT_AUTO = When set, automatically adjusts write parameters to match
	 *           the detected env blocks. This will never write new blocks.
	 *           When not set, new blocks will be written according to mode
	 *           flags below.
	 *
	 * UT_FORCE = When set, filter reads, only see selected env type.
	 *
	 * UT_LEGACY = Mode flag. When set, legacy mode mode will be used for write at
	 *             envbase|envsize. Combined with UT_FORCE also ignores redundant
	 *             env blocks during scan.
	 *
	 * UT_REDUNDANT = Mode flag. When set, 2-block redundant mode is used for write
	 *                at envbase|envsize and (envbase+stride)|envsize. Combined with
	 *                UT_FORCE also ignores legacy env block during scan.
	 *
	 * UT_NHRF = Mode flag. When set, many-block redundant mode is used for write
	 *           at (envbase + stride*n)|envsize. Combined with UT_FORCE also ignores
	 *           legacy env block during scan.
	 */
	ue->type = UT_AUTO;
}


/*
 *    Environment data manipulation
 */


/** Check the CRC in an env block.
 *
 * Evaluates a given memory buffer to see if it contains a valid environment.
 * The return value indicates the type of env block found. The erase detection
 * is not a guarantee, it just means that both CRC and the following 4 bytes
 * are 0xff.
 *
 * @param buf Pointer to start of environment block
 * 
 * @return Bitfield with flags. UF_ERASED=erased, UF_CRCOK=CRC matched,
 *         UF_REDFLAG=redundant structure, else legacy. Returns 0 for
 *         non-env data area.
 */
uint8_t env_check(const uint8_t *buf, unsigned size);


/** Check environment block with unknown size
 *
 * @param buf  Pointer to start of environment block
 * @param size Returns found environment size
 * @param min  Minimum expected env block size. Must be > 8
 * @param max  Maximum expected env block size. Must be > min
 * 
 * @return Bitfield with flags. UF_ERASED=erased, UF_CRCOK=CRC matched,
 *         UF_REDFLAG=redundant structure, else legacy. Returns 0 for
 *         non-env data area.
 */
uint8_t env_check_scan(const uint8_t *buf, unsigned *size, unsigned min, unsigned max);


/** Update CRC in an env block
 *
 * Update the CRC in a provided env block.
 *
 * @param buf Buffer holding the environment block. The updated CRC
 *            is written into the buffer.
 * @param size Environment size, including CRC and optional flag field.
 * @param flag If value is between 0-255 inclusive, write redundant
 *             env type with flag value. If not (e.g. -1), write
 *             legacy env. type.
 */
void env_update_crc(uint8_t *buf, unsigned size, int flag);


/** Allocate and init an empty records structure
 *
 * @return Pointer to structure or NULL on memory error.
 */
struct env_records * env_alloc_records(void);

/** Parsing an env block into a linked list of records
 *
 * The block must be valid. The elements of the returned list of records contains
 * a copy of the names & values.
 *
 * buf gets modified on-the-fly, but restored afterwards.
 *
 * @param buf       Buffer holding the environment block.
 * @param redundant Boolean, if set assume redundant env, else legacy
 *
 * @return pointer to allocated record list. May be empty.
 */
struct env_records * env_parse_records(uint8_t *buf, int redundant);


/** Drop an entire env record structure
 *
 * @param er  Env records structure to drop
 */
void env_drop_records(struct env_records *er);


/** Iterate over all records, in no particular order.
 *
 * @param er      Env records structure to iterate over
 * @param iterand Function to call on every record entry. If this function
 *                returns !=0, abort iteration.
 *
 * @returns 0 or the value the iterand function has returned in case of abort.
 */
int env_iterate_records(struct env_records *er, int (*iterand)(const char *name, const char *value));


/** Generate an env block from record structure.
 *
 * This also sets the (optional) flag field and the CRC.
 *
 * @param buf   Buffer holding the target environment block.
 * @param er    Env records to transform into env block
 * @param flag  When between 0-255 inclusive, write redundant env with this
 *              as the flag vale, otherwise (-1) write legacy env.
 * @param size  Size of environment to write. buf must be large enough.
 *
 * @return 0 on success, -1 on error (size exceeded)
 */
int env_generate(uint8_t *buf, const struct env_records *er, int flag, unsigned size);


/** Generate an env block from record structure.
 *
 * This is identical to env_generate, but extracts the necessary parameters
 * from the uenv write candidate entry and updates the flag field for redundant
 * envs.
 *
 * @param ue    uenv structure
 * @param er    Env records to transform into env block
 *
 * @return 0 on success, -1 on error (size exceeded)
 */
int env_update(struct uenv *ue, const struct env_records *er);


/** Read value of an env record
 *
 * The returned pointer points into the record's data area. Only valid as long
 * as the record element exists.
 *
 * @param er    Record list
 * @param name  Name of variable, 0-terminated
 *
 * @return Pointer to record value.
 */
const char * env_read_record(struct env_records *er, const char *name);


/** Delete env record
 *
 * @param er    Record list
 * @param name  Name of variable, 0-terminated
 *
 * @return 0 on success, 1 variable not found.
 */
int env_del_record(struct env_records *er, const char *name);


/** Adds / Replaces a record in the env records structure
 *
 * @param er    Record list
 * @param name  Name of variable, 0-terminated
 * @param value Value of variable, 0-terminated
 *
 * @return 0 on success, -1 on failure (OOM).
 */
int env_write_record(struct env_records *er, const char *name, const char *value);


/*
 *   Environment scanning & classification
 */


/** Scan env area for all potential env blocks
 *
 * The settings in ue must be valid. Creates an env descriptor entry for
 * every env entry and stride offset that isn't part of an entry.
 *
 * If envop returns an error during read, the while NAND block is considered
 * bad and will be skipped.
 *
 * @param ue Pointer to env structure.
 * @param envop Ptr to function capable of NAND operations.
 *
 * @return Number of found valid envs.
 */
int full_env_scan(struct uenv *ue, envop_fun envop );


/** Detects which env to use for reading and writing, respectively.
 *
 * Logic:
 *
 * For reading environments, redundant envs take precedence over legacy
 * ones, unless forced (UT_FORCE). For writing, we can make automatic
 * decisions or force an override mode, depending on ue->type.
 *
 * Automatic decisions are as follows:
 *
 * Leg | Red | Read target                | Write Target (auto)
 * ----+-----+----------------------------+-------------------------------------
 *  0  |  0  | Error                      | none
 *  1  |  0  | Legacy                     | Replace Legacy
 * >1  |  0  | First Legacy               | Replace First legacy
 * any |  1  | Redundant                  | Replace Redundant
 * any |  2  | Newest Redundant           | Oldest redundant
 * any | >2  | Newest Redundant           | Free space or oldest redundant
 *
 * In case of ambiguous candidates (bad wraps, etc.), the read candidate is set
 * to a unspecified valid env entry, the write candidate is set to NULL. In such
 * a case, perform a read & scrub.
 *
 * @param ue uenv data structure
 *
 * @return 0 on success, -1 on type flag error
 */
int env_analyse(struct uenv *ue);


/** Temporarily mark the current write candidate in the scan structure as bad.
 *
 * This is intended to remove an entry from the next run of env_analyze, in case
 * we don't have a markbad implementation. This marker only exists until the next
 * full_env_scan().
 *
 * @param ue    uenv data structure
 *
 * @return 0 on success. -1 on error (no matching entry found).
 */
int env_disable(struct uenv *ue);


/** Write entire environment area from current buffer
 * 
 * This writes the current buffer into all possible env locations. It's
 * intended to be used for error recovery and format conversion. Uses
 * the format as specified in ue->type, using size ue->envsize.
 *
 * NOTE: The resulting env locations will all have the same sequence
 * number. This saves a little time.
 *
 * @param ue    uenv data structure
 * @param envop Ptr to function capable of NAND operations.
 * @param flag  When between 0-255 inclusive, write redundant env with this
 *              as the sequence number, otherwise (-1) write legacy header.
 *
 * @return 0 on success, -1 on error (all writes have failed)
 */
int env_scrub(const struct uenv *ue, envop_fun envop, int flag);



/*
 *   Environment storage operations
 */


/** Read current read env block into uenv internal buffer
 *
 * @param ue Uenv structure, contains buffer & info about read source.
 * @param envop Ptr to function capable of NAND operations.
 *
 * @return 0 on success, else error passed from envop.
 */
int env_fetch(struct uenv *ue, envop_fun envop );


/** Write the internal uenv buffer into current write offset.
 *
 * @param ue Uenv structure, contains buffer & info about write target.
 * @param envop Ptr to function capable of NAND operations.
 *
 * @return 0 on success, else error passed from envop.
 */
int env_store(struct uenv *ue, envop_fun envop );


#endif /* _UENV_ALG_H */
