/*
 * LED command line support for Netcomm/Falcon Board
 *
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */


/*
	This uboot LED module has to remain portable from board to board and from platform to platform.

	This LED module is ported from Matthew's Freescale implementation and exteneded to be portable across platforms.
	At this moment, this file is not shared by platforms but eventually there will be a build script structure to
	share this actual file for all platforms. Thus, please keep this file compile-able across platforms.

	* for a new board, provide the following functions
		static void setup(void)					: setup all LEDs output (prepare LEDs to use)
		static inline void setup_read_pin(const unsigned pin)	: setup read pin (prepare a read pin to use)
		static void ledctl(unsigned led, unsigned color)	: turn on or off green or red LED
		static inline unsigned read_pin(const unsigned pin)	: return value 1=not pressed, 0=pressed

	* V variables
		V_BOARD_xxxx
		V_LEDFUN_xxxx

	* u-boot commands
		leds	: test LEDs
		rstb	: full test of LEDs and Reset button

	Yong Kim <yong.kim@netcommwireless.com>
*/

#include <cdcs_variant.h>

#if defined V_LED_RESET_y

#include <common.h>
#include <command.h>
#include <asm/byteorder.h>

#if defined(V_BOARD_falcon)

#include <asm/arch/imx-regs.h>

#elif defined(V_BOARD_nguni) || defined(V_BOARD_kewel)

#include "mux.h"

#define mdelay(x)	udelay((x)*1000)

#else

#error "This Board is unsupported. Check V_BOARD"

#endif

#include "gpio.h"
#include <cdcs_variant.h>

struct led {
	unsigned green;
	unsigned red;
};

/* define virtual PIN - this pin information may or may not be understood by hardware */
#ifndef PIN
#define PIN(bank,bit)     ((bank)*32+(bit))
#endif

#define BANK(x)		((x)/32)
#define BANK_PIN(x)	((x)%32)


#if defined(V_LEDFUN_falcon_with_aux)

#define LEDN 8

static const struct led leds[LEDN] = {
	{ PIN(1,9), PIN(1,10) },
	{ PIN(1,16), PIN(1,17) },
	{ PIN(2,2), PIN(2,3) },
	{ PIN(1,18), PIN(1,25) },
	{ PIN(3,30), PIN(1,27) },
	{ PIN(0,21), PIN(0,17) },
	{ PIN(2,6), PIN(2,7) },   /* FIXME: 2,6 doesn't work on REV A */
	{ PIN(1,11), PIN(1,22) },
};

#elif defined(V_LEDFUN_falcon_default)

#define LEDN 7

static const struct led leds[LEDN] = {
	{ PIN(1,9), PIN(1,10) },
	{ PIN(1,16), PIN(1,17) },
	{ PIN(1,18), PIN(1,25) },
	{ PIN(3,30), PIN(1,27) },
	{ PIN(0,21), PIN(0,17) },
	{ PIN(2,6), PIN(2,7) },   /* FIXME: 2,6 doesn't work on REV A */
	{ PIN(3,2), PIN(3,3) },
};

#elif defined(V_LEDFUN_falcon_1101)

#define LEDN 3

static const struct led leds[LEDN] = {
	{ PIN(1,27), PIN(1,27) },
	{ PIN(2,6), PIN(2,6) },
	{ PIN(2,7), PIN(2,7) },
};

#elif defined(V_LEDFUN_ntx01)

#define LEDN 4

static const struct led leds[LEDN] = {
	{ PIN(3,4), PIN(3,5) },
	{ PIN(2,6), PIN(2,7) },
	{ PIN(3,2), PIN(3,3) },
	{ PIN(0,21), PIN(0,17) },
};


#elif defined(V_LEDFUN_140wx) || defined(V_LEDFUN_140) || defined(V_LEDFUN_nwl22) || defined(V_LEDFUN_nwl22w) || defined(V_LEDFUN_145w)

#define LEDN 8

#if defined(V_LEDFUN_nwl22w) || defined(V_LEDFUN_145w)
static const struct led leds[LEDN] = {
	// Note: LEDx below numbered per PCB layout
	{
		PIN(3,14),	/* gpio3_14 LED8 grn */
		PIN(3,15)	/* gpio3_15 LED8 red */
	},
	{
		PIN(0,19),	/* gpio0_19 LED5 grn */
		PIN(0,20)	/* gpio0_20 LED5 red */
	},
	{
		PIN(3,18),	/* gpio3_18 LED4 grn */
		PIN(3,17)	/* gpio3_17 LED4 red */
	},
	{
		PIN(3,16),	/* gpio3_16 LED6 grn */
		PIN(0,10)       /* gpio0_10 LED6 red */
	},
	{
		PIN(1,12),	/* gpio1_12 LED7 grn */
		PIN(1,13)	/* gpio1_13 LED7 red */
	},
	{
		PIN(1,14),	/* gpio1_14 LED1 grn */
		PIN(1,15)	/* gpio1_15 LED1 red */
	},
	{
		PIN(0,26),	/* gpio0_26 LED2 grn */
		PIN(0,27)	/* gpio0_27 LED2 red */
	},
	{
		PIN(0,22),	/* gpio0_22 LED3 grn */
		PIN(0,23)	/* gpio0_23 LED3 red */
	}
};
#else
static const struct led leds[LEDN] = {
	{
		PIN(0,19),	/* gpio0_19 LED1 grn */
		PIN(0,20)	/* gpio0_20 LED1 red */
	},
	{
		PIN(1,14),	/* gpio1_14 LED2 grn */
		PIN(1,15)	/* gpio1_15 LED2 red */
	},
	{
		PIN(1,12),	/* gpio1_12 LED3 grn */
		PIN(1,13)	/* gpio1_13 LED3 red */
	},
	{
		PIN(0,26),	/* gpio0_26 LED4 grn */
		PIN(0,27)	/* gpio0_27 LED4 red */
	},
	{
		PIN(0,22),	/* gpio0_22 LED5 grn */
		PIN(0,23)	/* gpio0_23 LED5 red */
	},
	{
		PIN(3,14),	/* gpio3_14 LED6 grn */
		PIN(3,18)	/* gpio3_18 LED6 red */
	},
	{
		PIN(3,17),	/* gpio3_17 LED7 grn */
		PIN(3,16)	/* gpio3_16 LED7 red */
	},
	{
		PIN(0,10),	/* gpio0_10 LED8 grn */
		PIN(3,15)	/* gpio3_15 LED8 red */
	}
};
#endif
#elif defined(V_LEDFUN_ledNrg8)
/* Placeholder for the moment. Not implemented yet. */
#warning No U-Boot LED function implementation for this board yet

#define LEDN 0

static const struct led leds[1] = {0}; /* Dummy */

#else
#error LED function is not defined!
#endif

/*
 * A very rudimentary atoi function, only for use in below get_pin function
 */
static int atoi(char *string)
{
	char *tmp = string;
	int result = 0;
	while (*tmp != '\0') {
		result =  (result * 10) + (*tmp - '0');
		tmp++;
	}
	return result;
}

#if defined(V_BOARD_falcon)
static inline void setup_read_pin(const unsigned pin)
{
	/*
		TODO: setup read pin on the falcon

		on the falcon, gpio initial procedure sets up the reset read pin right now
	*/
}

static inline unsigned read_pin(const unsigned pin)
{
	return gpio_read(pin);
}
#elif defined(V_BOARD_nguni) || defined(V_BOARD_kewel)
static inline void setup_read_pin(const unsigned pin)
{
	GPIO_dir(BANK(pin), BANK_PIN(pin), GPIO_DIR_IN);
}

static inline unsigned read_pin(const unsigned pin)
{
	return GPIO_in(BANK(pin),BANK_PIN(pin));
}
#else
#error "This Board is unsupported. Check V_BOARD"
#endif

/* Expects a string of the form x:y, where x and y are both integers in the
 * GPIO range. x and y should be positive and no more than 2 digits.
 * Used to convert the V_RECBUTTON string into the same output
 * as provided by the PIN macro in gpio.h.
 */
#define MAX_STRING_SIZE 10
static unsigned get_pin(char *string)
{
	char *token;
	int x,y;
	char string_array[MAX_STRING_SIZE];

	strncpy(string_array, string, MAX_STRING_SIZE);
	token = strtok(string_array, ":");
	x = atoi(token);
	token = strtok(NULL, ":");
	y = atoi(token);
	return PIN(x,y);
}

static unsigned int reset_button;

#if defined(V_BOARD_falcon)
static void setup(void)
{
	unsigned i;
	struct pininfo pr = { M_OUT, V33, A12, DO(1) }; /* All leds off */

	for (i=0; i<LEDN; i++) {
		set_pininfo(leds[i].green, &pr);
		set_pininfo(leds[i].red, &pr);
	}
}
#elif defined(V_BOARD_nguni) || defined(V_BOARD_kewel)
static void setup_phys(int led)
{
	struct pinstate pr= {.v = 0, .oe = 1};	/*  GPIO_DIR_OUT and output 0 */

	pr.bank=BANK(led);
	pr.pin=BANK_PIN(led);
	GPIO_setstate(&pr);
}
static void setup(void)
{
	int i;

	for (i=0; i<LEDN; i++) {
		setup_phys(leds[i].green);
		setup_phys(leds[i].red);
	}
}
#else
#error "This Board is unsupported. Check V_BOARD"
#endif

static int randbit(void)
{
	static unsigned state=0x12345678;
	state <<=1;
	state |= (((state >> 31) & 1) ^ ((state >> 28) & 1));
	return (state & 1);
}

#define BLACK 0
#define GREEN 1
#define RED 2
#define AMBER 3
#define BLINK_TIME 50
#define COUNTDOWN_DELAY 625 /* 5000/8 milliseconds */

#if defined(V_BOARD_falcon)
static void ledctl(unsigned led, unsigned color)
{
	if (color & GREEN) gpio_set(leds[led].green, 0);
	else gpio_set(leds[led].green, 1);
	if (color & RED) gpio_set(leds[led].red, 0);
	else gpio_set(leds[led].red, 1);
}
#elif defined(V_BOARD_nguni) || defined(V_BOARD_kewel)
static void ledctl_phys(unsigned led, unsigned val)
{
	/* reverse output */
	GPIO_out(BANK(led),BANK_PIN(led), !val);
}
static void ledctl(unsigned led, unsigned color)
{
	ledctl_phys(leds[led].green, color & GREEN);
	ledctl_phys(leds[led].red, color & RED);
}
#else
#error "This Board is unsupported. Check V_BOARD"
#endif


static void step(void)
{
	static int led = 0;
	static unsigned color = 0;
	static int dir=-1;

	ledctl(led, color);

	led += dir;
	if (led<0) {
		led=0;
		dir=1;
	}
	if (led>LEDN-1) {
		led=LEDN-1;
		dir=-1;
	}

	color = (randbit() << 1) | randbit();

}

void LED_all (int colour)
{
	int i;
	for (i = 0; i < LEDN; i++) {
		ledctl(i, colour);
	}
}

void LED_blink (unsigned int colour, unsigned int time)
{
	LED_all(BLACK);
	while(time > 0) {
		LED_all(colour);
		mdelay(BLINK_TIME);
		LED_all(BLACK);
		mdelay(BLINK_TIME);
		time--;
	}
}

static int read_reset_button(void)
{
	return read_pin(reset_button);
}

int LED_countdown (unsigned int colour, unsigned int delay,int (*read_pin)(void))
{
	int i;

	LED_all(colour);
	for (i = 0; i < LEDN; i++) {
		mdelay(delay);
		if (read_pin && (read_pin() != 0)) {
			return 1;
		}
		ledctl(i, BLACK);
	}
	mdelay(delay);
	return 0;
}
/*
 * Handle the reset button logic:
 *     1. If reset button held less than X1 seconds, normal reset.
 *     2. If reset button held between X1-X2 seconds, recovery mode.
 *     3. If reset button held less than X1 seconds, factory reset.
 *
 * returns 1 if a reset button event occurred (proceed straight to boot)
 *         0 if no events occurred, proceed as normal.
 */

#define STATE_NORMAL 0
#define STATE_RECOVERY 1
#define STATE_FACTORY 2

int handle_reset_button (void)
{
	int current_state = STATE_NORMAL;
	unsigned int button_pressed = 0; /*  has it ever been pressed? */
	char *emergency;

	reset_button = get_pin(V_RECBUTTON);
	setup();
	setup_read_pin(reset_button);

	printf("Hit SPACEBAR to stop at the end of lap.\n");

	while(read_pin(reset_button) == 0) {
		LED_all(BLACK);
		/* disable the watchdog so we can wait for button release */
#if 0
		/* Not currently used on Falcon */
		if (current_state == 0) {
			cmdline_wdtoff(NULL, 0, 0, NULL);
		}
#endif
		button_pressed = 1;
		if (LED_countdown(GREEN, COUNTDOWN_DELAY,read_reset_button)) {
			printf("normal boot detected\n");
			current_state = STATE_NORMAL;
			break;
		}
		if (LED_countdown(AMBER, COUNTDOWN_DELAY * 2,read_reset_button)) {
			printf("recovery mode detected\n");
			current_state = STATE_RECOVERY;
			break;
		}
		if (LED_countdown(RED, COUNTDOWN_DELAY,read_reset_button)) {
			printf("factory reset mode detected\n");
			current_state = STATE_FACTORY;
			break;
		}

		/* cancel */
		if (tstc() && getc()==' ') {
			printf("break\n");
			break;
		}
	}

	/* Check what state the button was released in, to determine
	 * the action that should be taken.
	 */
	if (button_pressed == 0) {
		LED_all(BLACK);

		/* Read emergency variable and set LED */
		emergency = getenv("emergency");
		if (emergency && (strcmp(emergency, "button") == 0)) {
			ledctl(0, AMBER);
		} else {
			ledctl(0, GREEN);
		}
	} else {
		LED_all(BLACK);
		switch (current_state % 3) {
		case 0:
			// Normal Boot - flash green
			LED_blink(GREEN, 3);

			/* Make sure the power LED is left on */
			ledctl(0, GREEN);
			break;
		case 1:
			// Recovery mode - flash yellow
			LED_blink(AMBER, 3);
			setenv("emergency", "button");
			/* if "recboot" is not empty here, "bootcheck" will erase "emergency"
			 * and the unit will boot to main system
			 */
			setenv("recboot", "");
			saveenv();

			/* Make sure the power LED is left on */
			ledctl(0, AMBER);
			break;
		case 2:
			// Factory reset - flash red
			LED_blink(RED, 3);
			setenv("emergency", "reset");
			/* same reason as above */
			setenv("recboot", "");
			saveenv();

			/* Make sure the power LED is left on */
			ledctl(0, RED);
			break;
		default:
			// This should never happen

			/* Make sure the power LED is left on */
			ledctl(0, GREEN);
			break;
		}
	}

	return current_state;
}


/* Set gpio muxmode. Args: bank bit mux0|mux1|mux2|in|out */
static int fun_leds(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	/* setup read pins */
	reset_button=get_pin(V_RECBUTTON);
	setup_read_pin(reset_button);

	/* print info */
	printf(
		"* reset button status (V_RECBUTTON=%s,PIN=%d,BANK=%d,BANK_PIN=%d,value=%d)\n",V_RECBUTTON,reset_button,BANK(reset_button),BANK_PIN(reset_button),read_pin(reset_button)
	);

	/* setup leds */
	setup();

	/* test count down */
	printf("* led dance - green\n");
	LED_countdown(GREEN, COUNTDOWN_DELAY,NULL);
	printf("* led dance - amber\n");
	LED_countdown(AMBER, COUNTDOWN_DELAY,NULL);
	printf("* led dance - red\n");
	LED_countdown(RED, COUNTDOWN_DELAY,NULL);

	/* random dance */
	printf("* led dance - starwars (Hit SPACEBAR to stop demo)\n");
	while (1) {
		step();
		mdelay(100); /* 100 ms */
		if (tstc() && getc()==' ') {
			break;
		}
	}

	return 0;
}

static int fun_rstb(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	handle_reset_button();

	return 0;
}

U_BOOT_CMD(leds, CONFIG_SYS_MAXARGS, 1, fun_leds,
           "LED demo",
           "Hit SPACEBAR to stop.\n"
          );

U_BOOT_CMD(rstb, CONFIG_SYS_MAXARGS, 0, fun_rstb,
           "Reset button demo",
           "Perform reset button procedure.\n"
          );

#endif /* V_LED_RESET_y */
