/*
 * cmd_mux.c
 *
 * am335x Pin Multiplexer command line support
 *
 * Iwo.Mergler@netcommwireless.com
 * Copyright (C) 2012 NetComm Wireless Limited - http://www.netcommwireless.com/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <common.h>
#include <command.h>
#include <config.h>
#include <asm/io.h>
#include "common_def.h"

#include "mux.h"

#ifndef CONFIG_SPL_BUILD

/* Array of pin names, must match struct pad_signals in mux.h
   The order reflects the pad control registers in the processor. */
static const char *pin_names[] = {
	"gpmc_ad0",
	"gpmc_ad1",
	"gpmc_ad2",
	"gpmc_ad3",
	"gpmc_ad4",
	"gpmc_ad5",
	"gpmc_ad6",
	"gpmc_ad7",
	"gpmc_ad8",
	"gpmc_ad9",
	"gpmc_ad10",
	"gpmc_ad11",
	"gpmc_ad12",
	"gpmc_ad13",
	"gpmc_ad14",
	"gpmc_ad15",
	"gpmc_a0",
	"gpmc_a1",
	"gpmc_a2",
	"gpmc_a3",
	"gpmc_a4",
	"gpmc_a5",
	"gpmc_a6",
	"gpmc_a7",
	"gpmc_a8",
	"gpmc_a9",
	"gpmc_a10",
	"gpmc_a11",
	"gpmc_wait0",
	"gpmc_wpn",
	"gpmc_be1n",
	"gpmc_csn0",
	"gpmc_csn1",
	"gpmc_csn2",
	"gpmc_csn3",
	"gpmc_clk",
	"gpmc_advn_ale",
	"gpmc_oen_ren",
	"gpmc_wen",
	"gpmc_be0n_cle",
	"lcd_data0",
	"lcd_data1",
	"lcd_data2",
	"lcd_data3",
	"lcd_data4",
	"lcd_data5",
	"lcd_data6",
	"lcd_data7",
	"lcd_data8",
	"lcd_data9",
	"lcd_data10",
	"lcd_data11",
	"lcd_data12",
	"lcd_data13",
	"lcd_data14",
	"lcd_data15",
	"lcd_vsync",
	"lcd_hsync",
	"lcd_pclk",
	"lcd_ac_bias_en",
	"mmc0_dat3",
	"mmc0_dat2",
	"mmc0_dat1",
	"mmc0_dat0",
	"mmc0_clk",
	"mmc0_cmd",
	"mii1_col",
	"mii1_crs",
	"mii1_rxerr",
	"mii1_txen",
	"mii1_rxdv",
	"mii1_txd3",
	"mii1_txd2",
	"mii1_txd1",
	"mii1_txd0",
	"mii1_txclk",
	"mii1_rxclk",
	"mii1_rxd3",
	"mii1_rxd2",
	"mii1_rxd1",
	"mii1_rxd0",
	"rmii1_refclk",
	"mdio_data",
	"mdio_clk",
	"spi0_sclk",
	"spi0_d0",
	"spi0_d1",
	"spi0_cs0",
	"spi0_cs1",
	"ecap0_in_pwm0_out",
	"uart0_ctsn",
	"uart0_rtsn",
	"uart0_rxd",
	"uart0_txd",
	"uart1_ctsn",
	"uart1_rtsn",
	"uart1_rxd",
	"uart1_txd",
	"i2c0_sda",
	"i2c0_scl",
	"mcasp0_aclkx",
	"mcasp0_fsx",
	"mcasp0_axr0",
	"mcasp0_ahclkr",
	"mcasp0_aclkr",
	"mcasp0_fsr",
	"mcasp0_axr1",
	"mcasp0_ahclkx",
	"xdma_event_intr0",
	"xdma_event_intr1",
	"nresetin_out",
	"porz",
	"nnmi",
	"osc0_in",
	"osc0_out",
	"rsvd1",
	"tms",
	"tdi",
	"tdo",
	"tck",
	"ntrst",
	"emu0",
	"emu1",
	"osc1_in",
	"osc1_out",
	"pmic_power_en",
	"rtc_porz",
	"rsvd2",
	"ext_wakeup",
	"enz_kaldo_1p8v",
	"usb0_dm",
	"usb0_dp",
	"usb0_ce",
	"usb0_id",
	"usb0_vbus",
	"usb0_drvvbus",
	"usb1_dm",
	"usb1_dp",
	"usb1_ce",
	"usb1_id",
	"usb1_vbus",
	"usb1_drvvbus",
	"ddr_resetn",
	"ddr_csn0",
	"ddr_cke",
	"ddr_ck",
	"ddr_nck",
	"ddr_casn",
	"ddr_rasn",
	"ddr_wen",
	"ddr_ba0",
	"ddr_ba1",
	"ddr_ba2",
	"ddr_a0",
	"ddr_a1",
	"ddr_a2",
	"ddr_a3",
	"ddr_a4",
	"ddr_a5",
	"ddr_a6",
	"ddr_a7",
	"ddr_a8",
	"ddr_a9",
	"ddr_a10",
	"ddr_a11",
	"ddr_a12",
	"ddr_a13",
	"ddr_a14",
	"ddr_a15",
	"ddr_odt",
	"ddr_d0",
	"ddr_d1",
	"ddr_d2",
	"ddr_d3",
	"ddr_d4",
	"ddr_d5",
	"ddr_d6",
	"ddr_d7",
	"ddr_d8",
	"ddr_d9",
	"ddr_d10",
	"ddr_d11",
	"ddr_d12",
	"ddr_d13",
	"ddr_d14",
	"ddr_d15",
	"ddr_dqm0",
	"ddr_dqm1",
	"ddr_dqs0",
	"ddr_dqsn0",
	"ddr_dqs1",
	"ddr_dqsn1",
	"ddr_vref",
	"ddr_vtp",
	"ddr_strben0",
	"ddr_strben1",
	"ain7",
	"ain6",
	"ain5",
	"ain4",
	"ain3",
	"ain2",
	"ain1",
	"ain0",
	"vrefp",
	"vrefn",
	NULL
};

#define PINS (sizeof(pin_names)/sizeof(const char *))


/* Returns register offset for given pin name. Returns < 0 if not found. */
static int name2offset(const char *s)
{
	const char **p = pin_names;
	int num = 0;
	while(*p) {
		if (strcmp(s,*p)==0) break;
		num++;
		p++;
	}
	if (*p) {
		return PAD_CTRL_BASE + num * 4;
	} else {
		return -1;
	}
}

static void printpad(uint32_t v)
{

	if ( v & SLEWCTRL ) printf("slow, ");
	else printf("fast, ");

	if ( v & RXACTIVE ) printf(" rx, ");
	else printf("nrx, ");

	if (v & PULLUDDIS ) {
		printf("np, ");
	} else {
		if (v & PULLUP_EN) {
			printf("pu, ");
		} else {
			printf("pd, ");
		}
	}

	printf("Mode%d", v & 0x7);
}

static int do_mux(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int off;
	uint32_t v;

	if (argc==1) {
		const char **p = pin_names;
		int count = 0;
		while(*p) {
			off = name2offset(*p);
			v = __raw_readl(CTRL_BASE + off);
			printf("%3d (0x%04x),%20s [0x%08x] ", count, off, *p, v);
			printpad(v);
			putc('\n');
			count++;
			p++;
		}
		return 0;
	}

	off = name2offset(argv[1]);
	if (off < 0) {
		printf("Pin not found\n");
		return -1;
	}

	if (argc==2) {
		v = __raw_readl(CTRL_BASE + off);
		printf("%20s [0x%08x] ", argv[1], v);
		printpad(v);
		putc('\n');
		return 0;
	}

	/* Set command */
	{
		char * const *av = &argv[2];
		int ac = argc - 2;
		v = __raw_readl(CTRL_BASE + off);
		while (ac--) {
			if (strcmp("pu", *av)==0) {
				v &= ~PULLUDDIS;
				v |= PULLUP_EN;
			} else if (strcmp("pd", *av)==0) {
				v &= ~PULLUDDIS;
				v &= ~PULLUP_EN;
			} else if (strcmp("np", *av)==0) {
				v |= PULLUDDIS;
			} else if (strcmp("fast", *av)==0) {
				v &= ~SLEWCTRL;
			} else if (strcmp("slow", *av)==0) {
				v |= SLEWCTRL;
			} else if (strcmp("rx", *av)==0) {
				v |= RXACTIVE;
			} else if (strcmp("nrx", *av)==0) {
				v &= ~RXACTIVE;
			} else {
				/* muxmode */
				v &= ~0x7;
				v |= 0x7 & simple_strtoul(*av,NULL,16);
			}
			av++;
		}
		__raw_writel(v, CTRL_BASE + off);
	}

	return 0;
}


U_BOOT_CMD(
	mux,	7,	1,	do_mux,
	"Pin Mux commands",
	"Dump all mux settings\n"
	"mux <name> [np|pu|pd] [fast|slow] [rx|nrx] muxmode = Set mux\n"
	"mux <name> = read mux\n"
);

#endif /* CONFIG_SPL_BUILD */
