/*
 *
 * Filename:    SMSSampleApp.c
 *
 * Purpose:     SMS Application
 *
 * Copyright: © 2011-2013 Sierra Wireless Inc., all rights reserved
 *
 **************/

#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include "SMSSampleApp.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <unistd.h>

/****************************************************************
 *                    GLOBAL DATA
 ****************************************************************/

/* path to sdk binary */
static char *sdkbinpath = NULL;

/* Device connectivity */
static device_info_t devices[1] = { { {'\0'}, {'\0'} } };
static device_info_t *pdev = &devices[0];

/* SMS Message List */
messageLists *pMessageListsNV,*pMessageListsUIM;

const CHAR *memoryNV  = "NV";
const CHAR *memoryUIM = "UIM";

/* Stores value of parameters in case of callback is invoked */
ULONG gStorageType[10]  = {0};
ULONG gMessageIndex[10] = {0};
INT8  gNotificationFlag = 0;
BYTE  messageMode       = INVALID_TECH;
ULONG messageCount      = 0;
ULONG gMsgCountTag      = 0;
ULONG gMessageTag       = 0 ;
indexMapMessage indexmaps[256];

/******************************************************************************
                       Forward Declarations
******************************************************************************/
/* Read SMS with the provided Index */
void WGetSMS4Index();

/****************************************************************************
*                       Functions
*****************************************************************************/
/*
 * Name:     IsDeviceMC7750
 *
 * Purpose:  To check if device is MC7750 or else
 *
 * Return:   TRUE if device is MC7750 , FALSE otherwise
 *
 * Notes:    None
 */
BOOL IsDeviceMC7750()
{
    BYTE  stringSize = MAX_FIELD_SIZE;
    CHAR  modelId[MAX_FIELD_SIZE];
    CHAR  *pStr = NULL;

    /* Get the model Id of the device */
    GetModelID( stringSize, modelId );

    /* Search for a MC7750 string in the received model id */
    pStr = strstr( modelId, "MC7750" );

    /* If the device is a MC7750 device */
    if ( pStr != NULL )
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

/*
 * Name:     IsDeviceMC7800
 *
 * Purpose:  To check if device is MC7800 or else
 *
 * Return:   TRUE if device is MC7800 , FALSE otherwise
 *
 * Notes:    None
 */
BOOL IsDeviceMC7800()
{
    BYTE  stringSize = MAX_FIELD_SIZE;
    CHAR  modelId[MAX_FIELD_SIZE];
    CHAR  *pStr = NULL;

    /* Get the model Id of the device */
    GetModelID( stringSize, modelId );

    /* Search for a MC7800 string in the received model id */
    pStr = strstr( modelId, "MC7800" );

    /* If the device is a MC7800 device */
    if ( pStr != NULL )
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}

/*
 * Name:     GetMsgProtocol
 *
 * Purpose:  Fetch the message protocol used in device
 *
 * Return:   None
 *
 * Notes:    None
 */
void GetMsgProtocol()
{
    ULONG nRet;
    ULONG RegistrationState = 0;
    ULONG CSDomain =0 ;
    ULONG PSDomain = 0;
    ULONG RAN = 0 ;
    BYTE  RadioIfacesSize = nMaxStrLen;
    BYTE  RadioIfaces[nMaxStrLen];
    ULONG Roaming = 0;
    WORD  MCC = 0;
    WORD  MNC = 0;
    BYTE  nameSize = nMaxStrLen;
    CHAR  Name[nMaxStrLen];
    smsMsgprotocolResp MsgProtocol;

    nRet = SLQSSmsGetMessageProtocol(&MsgProtocol);
    if ( eQCWWAN_ERR_NONE == nRet )
    {
        if ( CDMA == MsgProtocol.msgProtocol )
        {
            messageMode = CDMA;
        }
        else if ( UMTS == MsgProtocol.msgProtocol )
        {
            messageMode = UMTS;
        }
        else
        {
            fprintf(stderr, "MessageProtocol not supported \n");
            exit(EXIT_FAILURE);
        }
    }
    else
    {
    nRet = GetServingNetwork(&RegistrationState,
                             &CSDomain,
                             &PSDomain,
                             &RAN,
                             &RadioIfacesSize,
                             &RadioIfaces[0],
                             &Roaming,
                             &MCC,
                             &MNC,
                             nameSize,
                             &Name[0]);

        if (!nRet)
        {
            if ( IsDeviceMC7750() || IsDeviceMC7800() )
            {
                messageMode = CDMA;
            }
            else
            {
                switch( RadioIfaces[0] )
                {
                    case CDMA_1xRTT:
                    case CDMA_1xEVDO:
                        messageMode = CDMA;
                        break;

                    case GSM_RI:
                    case UMTS_RI:
                        messageMode = UMTS;
                    break;

                    case LTE_RI:
                        if( CDMA2000_NET == RAN  )
                        {
                            messageMode = CDMA ;
                        }
                        else if ( UMTS_NET == RAN )
                        {
                             messageMode = UMTS;
                        }
                        else if( UNKNOWN_NET == RAN )
                        {
                            free(sdkbinpath);
                            fprintf(stderr, "Unknown Radio Access Network\n");
                            exit(EXIT_FAILURE);
                        }
                        break;

                    default:
                        free(sdkbinpath);
                        fprintf(stderr, "RF Technology is not supported\n");
                        exit(EXIT_FAILURE);
                 }
             }
         }
         else
         {
            free(sdkbinpath);
            fprintf(stderr, "RF Technology could not be determined,please verify SIM\
                             functionality (at+cpin) and device settings\n");
            exit(EXIT_FAILURE);
         }
    }
}

/*
 * Name:     ReadNewSMS
 *
 * Purpose:  Displays received SMS
 *
 * Return:   None
 *
 * Notes:
 */
void ReadNewSMS()
{
    ULONG messageSize = nMaxStrLen + 2;
    ULONG messageTag = 0,messageFormat = 0;
    BYTE  message[nMaxStrLen + 2];
    CHAR  ascii[MAXTEXT]= {0};
    BYTE  mcTimeStamp[8] = {0};
    ULONG nRet;
    BYTE  SenderAddrLength = 16;
    BYTE  TextMsgLength    = 160;
    BYTE  SMSCAddrLength   = 16;
    CHAR  SenderAddr[16]   = {'\0'};
    WORD  TextMsg[160]     = {'\0'};
    CHAR  SMSCAddr[16]     = {'\0'};

    /* Execute GetSMS API with storage index */
    nRet = SLQSGetSMS( gStorageType[gNotificationFlag],
                       gMessageIndex[gNotificationFlag],
                       &messageTag,
                       &messageFormat,
                       (ULONG *)&messageSize,
                       &message[0],
                       &messageMode);
    if ( !nRet )
    {
         fprintf( stderr, "SMS read is successful\n" );
    }
    else
    {
        fprintf(stderr,"\n UIM memory does not contain any messages\n");
    }

    if ( UMTS == messageMode )
    {
        /* Structure to decode PDU message to ASCII */
        struct wcdmaMsgDecodingParams WcdmaMsgDecodingParams;

        WcdmaMsgDecodingParams.pMessage          = message;
        WcdmaMsgDecodingParams.pSenderAddrLength = &SenderAddrLength;
        WcdmaMsgDecodingParams.pTextMsgLength    = &TextMsgLength;
        WcdmaMsgDecodingParams.pScAddrLength     = &SMSCAddrLength;

        WcdmaMsgDecodingParams.pSenderAddr = SenderAddr;
        WcdmaMsgDecodingParams.pTextMsg    = ascii;
        WcdmaMsgDecodingParams.pScAddr     = SMSCAddr;

        nRet = SLQSWCDMADecodeMTTextMsg( &WcdmaMsgDecodingParams );

        if( eQCWWAN_ERR_NONE != nRet )
        {
            fprintf( stderr,"Failed to Read SMS \n");
            fprintf( stderr,"Failure Code: %lu\n", nRet );
        }

        switch( nRet )
        {
            case eQCWWAN_ERR_SWISMS_MSG_CORRUPTED:
                fprintf( stderr,"Message is not available or is corrupted\n" );
                break;

            case eQCWWAN_ERR_SWISMS_SMSC_NUM_CORRUPTED:
                fprintf( stderr," SMSC/Mobile number is not correct\n" );
                break;

            case eQCWWAN_ERR_BUFFER_SZ:
                fprintf( stderr," Internal Error - Can not read SMS \n" );
                break;

            case eQCWWAN_ERR_SWISMS_MSG_LEN_TOO_LONG:
                fprintf( stderr," Message exceeds 160 characters\n" );
                break;

            case eQCWWAN_ERR_NONE:
                fprintf( stderr, "\nService Center Number = %s\n",
                         WcdmaMsgDecodingParams.pScAddr );
                fprintf( stderr, "Date = %s\n",
                         WcdmaMsgDecodingParams.Date );
                fprintf( stderr, "Time = %s\n",
                         WcdmaMsgDecodingParams.Time );
                fprintf( stderr, "Message: %s\n",
                         WcdmaMsgDecodingParams.pTextMsg );
                //printMsg();
                break;

            default:
                break;
        }
       fprintf(stderr,"\n");
    }
    if ( CDMA == messageMode )
    {
        SwiCDMADecodeMTTextMsg( messageSize,
                                message,
                                NULL,
                                &SenderAddrLength,
                                SenderAddr,
                                &TextMsgLength,
                                (WORD*)&TextMsg,
                                NULL,
                                NULL,
                                NULL,
                                mcTimeStamp,
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                NULL );

        fprintf(stderr,"Sender Number: %s\n",SenderAddr);
        fprintf(stderr,"SMS Text: %s\n",(CHAR*)TextMsg);
        fprintf(stderr,"Time Stamp: %d:%d:%d:%d:%d:%d\n",
                            mcTimeStamp[0],mcTimeStamp[1],
                            mcTimeStamp[2],mcTimeStamp[3],
                            mcTimeStamp[4],mcTimeStamp[5]);
    }

    /* Modify the tag of the Message that has been read */
    if( TAG_MSG_MT_UNREAD == messageTag )
    {
        SLQSModifySMSStatus( gStorageType[gNotificationFlag],
                             gMessageIndex[gNotificationFlag],
                             TAG_MSG_MT_READ,
                             &messageMode );
    }
    gStorageType[gNotificationFlag]  = 0;
    gMessageIndex[gNotificationFlag] = 0;
}

/*
 * Name:     FlushStdinStream
 *
 * Purpose:  Flush the stdin stream
 *
 * Return:   None
 *
 * Notes:    fflush does not work for input stream.
 */
void FlushStdinStream( )
{
    ULONG inputChar;

    /* keep on reading until an <New Line> or end of file is received */
    do
    {
        inputChar = getchar();

#ifdef DBG
    fprintf( stderr,  "inputChar: %lx\n", inputChar );
#endif
    }
    while ( ( inputChar != ENTER_KEY ) &&
            ( inputChar != (ULONG)EOF )       &&
            ( inputChar != ESCAPE_KEY) );
}

/*
 * Name:     StartSDK
 *
 * Purpose:  It starts the SDK by setting the SDK path, enumerates the device
 *           and connects to the SDK.
 *
 * Return:   eQCWWAN_ERR_NONE on successfully starting SDK, else error code
 *
 * Notes:    none
 */
ULONG StartSDK()
{
    ULONG resultCode = 0;
    BYTE  devicesSize = 1;

    /* Set SDK image path */
    if( eQCWWAN_ERR_NONE !=
       (resultCode = SetSDKImagePath(sdkbinpath)) )
    {
        fprintf(stderr, "Failed to set the SDK Image path (%lx)\n", resultCode);
        return resultCode;
    }

    if( eQCWWAN_ERR_SWICM_SOCKET_IN_USE == (resultCode = SLQSStart()) )
    {
        fprintf(stderr, "Another APP is currently using the SDK (%lx)\n", resultCode);
        return resultCode;
    }
    else if( eQCWWAN_ERR_NONE != resultCode )
    {
        return resultCode;
    }

    /* Enumerate the device */
    while (QCWWAN2kEnumerateDevices(&devicesSize, (BYTE *)pdev) != 0)
    {
        printf ("\nUnable to find device..\n");
        sleep(1);
    }

#ifdef DBG
    fprintf( stderr,  "#devices: %d\ndeviceNode: %s\ndeviceKey: %s\n",
            devicesSize,
            pdev->deviceNode,
            pdev->deviceKey );
#endif

    /* Connect to the SDK */
    resultCode = QCWWANConnect( pdev->deviceNode,
                                pdev->deviceKey );
    return resultCode;
}

/*
 * Name:     ListAllMessages
 *
 * Purpose:  Allocated memory and stores SMS List
 *
 * Return:   None
 *
 * Notes:    none
 */
void ListAllMessages()
{
    messageLists *pAllMessages;
    ULONG        tempSizeNV  = 0;
    ULONG        tempSizeUIM = 0;
    USHORT       count;
    USHORT       msgCount;
    ULONG        RequestedTag;
    ULONG        messageListSizeNV  = nMaxStrLen;
    ULONG        messageListSizeUIM = nMaxStrLen;
    ULONG        nRet;
    memset(indexmaps, 0, 256*sizeof(indexMapMessage));

    /* Allocate memory for NV storage type */
    pAllMessages = (messageLists *)malloc(4*nMaxStrLen*sizeof(messageLists));

    /* Check if memory allocation is successful or not */
    if( !pAllMessages )
    {
        fprintf(stderr,"Memory allocation failed:");
        return;
    }

    for( count = 0; count < 4 ; count++)
    {
        RequestedTag       = count;
        messageListSizeNV  = nMaxStrLen;
        messageListSizeUIM = nMaxStrLen;

        nRet = SLQSGetSMSList(STORAGEINDEX_NV,
                              &RequestedTag,
                              &messageListSizeNV,
                              (BYTE*)(pAllMessages+ tempSizeNV),
                              &messageMode );
        if( nRet )
        {
             messageListSizeNV = 0;
        }
        tempSizeNV = tempSizeNV + messageListSizeNV;
    }

    if(tempSizeNV)
    {
        RequestedTag       = count;
        messageListSizeNV  = nMaxStrLen;
        messageListSizeUIM = nMaxStrLen;
        for(msgCount = 0; msgCount < tempSizeNV; msgCount++)
        {
            indexmaps[msgCount].memoryType    = "NV";
            indexmaps[msgCount].originalIndex =
                                pAllMessages[msgCount].messageIndex;
            indexmaps[msgCount].mappedIndex   = msgCount;
            indexmaps[msgCount].messageTag    =
                                pAllMessages[msgCount].messageTag;
        }
    }

    /* Set memory buffer to zero to gather list for UIM */
    memset(pAllMessages,0,tempSizeNV);

    /* Get messages from UIM */
    for( count = 0; count < 4 ; count++)
    {
        RequestedTag       = count;
        messageListSizeNV  = nMaxStrLen;
        messageListSizeUIM = nMaxStrLen;

        nRet = SLQSGetSMSList(STORAGEINDEX_UIM,
                              &RequestedTag,
                              &messageListSizeUIM,
                              (BYTE*)(pAllMessages + tempSizeUIM),
                              &messageMode );
        if( nRet )
        {
            messageListSizeUIM = 0;
        }
        tempSizeUIM = tempSizeUIM + messageListSizeUIM;
    }

    if(tempSizeUIM)
    {
        for(msgCount = 0; msgCount < tempSizeUIM; msgCount++)
        {
             indexmaps[msgCount+tempSizeNV].memoryType = "UIM";
             indexmaps[msgCount+tempSizeNV].originalIndex =
                                        pAllMessages[msgCount].messageIndex;
             indexmaps[msgCount+tempSizeNV].mappedIndex =
                                            msgCount+tempSizeNV;
             indexmaps[msgCount+tempSizeNV].messageTag =
                                        pAllMessages[msgCount].messageTag;
        }
    }
    /* Free the allocated memory for message list */
    free( pAllMessages );

    messageCount = tempSizeNV + tempSizeUIM;
}

/*
 * Name:     Displaymsgs
 *
 * Purpose:  Display the message list depending on message tag
 *
 * Params:   pMessageTag[IN] - Tag of message to be displayed
 *
 * Return:   None
 *
 * Notes:    none
 */

ULONG Displaymsgs( ULONG *pMessageTag )
{
    ULONG msgCount;
    ULONG msgCountTag = 0;
    fprintf(stderr,"\nStorage type\t  messageIndex\t   Message tag\n");
    for(msgCount = 0; msgCount < messageCount; msgCount++)
    {
        if ( indexmaps[msgCount].messageTag == *pMessageTag )
        {
            fprintf(stderr,
                           "%6s",
                           indexmaps[msgCount].memoryType);
            fprintf(stderr,
                           "%14ld",
                           indexmaps[msgCount].mappedIndex);
            fprintf(stderr,
                           "%18ld\n",
                           indexmaps[msgCount].messageTag);
            msgCountTag++;
        }

        if( 4 == *pMessageTag )
        {
            fprintf(stderr,
                           "%6s",
                           indexmaps[msgCount].memoryType);
            fprintf(stderr,
                           "%14ld",
                           indexmaps[msgCount].mappedIndex);
            fprintf(stderr,
                          "%18ld\n",
                          indexmaps[msgCount].messageTag);
        }
    }
    if ( 0 == *pMessageTag ||
         1 == *pMessageTag ||
         2 == *pMessageTag ||
         3 == *pMessageTag )
    {
        /* Print Total size if list */
        fprintf(stderr,
                "Total SMS messages in the corresponding list = %lu\n",
                msgCountTag);
    }

    gMsgCountTag = msgCountTag;
    if( 4 == *pMessageTag )
    {
        /* Print Total size if list */
        fprintf(stderr,
                "Total SMS messages in the corresponding list = %lu\n",
                messageCount);
    }
    return messageCount;
}

/*
 * Name:     ShowList
 *
 * Purpose:  Display the message list depending on memory type
 *
 * Params:   pMemoryType[IN] - Type of memory from which user want to see
 *                             the message
 *
 * Return:   None
 *
 * Notes:    None
 */
ULONG ShowList( CHAR *pMemoryType )
{
    ULONG msgCount;
    ULONG msgCountTag = 0;
    fprintf(stderr,"\nStorage type\t  messageIndex\t   Message tag\n");

    /* Update list with recently received message */
    ListAllMessages();
    for(msgCount = 0; msgCount < messageCount; msgCount++)
    {
        if ( *indexmaps[msgCount].memoryType == *pMemoryType )
        {
            fprintf(stderr,
                           "%6s",
                           indexmaps[msgCount].memoryType);
            fprintf(stderr,
                           "%14ld",
                           indexmaps[msgCount].mappedIndex);
            fprintf(stderr,
                           "%18ld\n",
                           indexmaps[msgCount].messageTag);
            msgCountTag++;
        }
    }
    return msgCountTag;
}

/*
 * Name:     NewSMSCallback
 *
 * Purpose:  CallBack function for NewSMSCallback
 *
 * Return:   None
 *
 * Notes:    none
 */
void NewSMSCallback(
    ULONG storageType,
    ULONG messageIndex )
{
    gStorageType[gNotificationFlag]  = storageType ;
    gMessageIndex[gNotificationFlag] = messageIndex ;
    gNotificationFlag++;
    if ( gNotificationFlag > 9 )
    {
         gNotificationFlag--;
    }
    fprintf(stderr,
            "\nAn incoming SMS message has been received,"
            " would you like to read the message (y/n):\n");
}

/*
 * Name:     ActivateSMSCallback
 *
 * Purpose:  To Activate NewSMSCallback
 *
 * Return:   None
 *
 * Notes:    none
 */
void ActivateSMSCallback()
{
    ULONG nRet;
    nRet = SetNewSMSCallback(&NewSMSCallback);
    if ( nRet != 0 )
    {
        fprintf( stderr,
                 "Registration for incoming SMS message"
                 " notifications failed: %lu\n", nRet );
    }
    else
    {
        fprintf( stderr,
                 "Registration for incoming SMS message"
                 " notifications succeeded\n");
    }
}


void getEncodingscheme( CHAR *encodingscheme )
{
    CHAR  *pEndOfLine = NULL;
    BOOL  bFlag = FALSE;
    while(1)
    {
        fprintf( stderr,
                 "Enter the encoding scheme for message(0 for default 7Bit, 1 for 8 Bit, 2 for UCS2 Encoding )"\
                 " followed by <Enter> key:\n" );

        fgets( encodingscheme, OPTION_LEN, stdin);
        switch( encodingscheme[0] )
        {
            case '0':
            case '1':
            case '2':
                bFlag = TRUE;
                break;
            default:
                fprintf(stderr," Encoding Scheme entered is invalid!\n");
                break;
        }
        if( TRUE == bFlag )
        {
            break;
        }
    }

    pEndOfLine = strchr( encodingscheme, ENTER_KEY );
    if( NULL == pEndOfLine )
    {
        FlushStdinStream();
    }

    if( encodingscheme == pEndOfLine )
        return;
}

/*
 * Name:     WsendSMS
 *
 * Purpose:  Handle user-selected send SMS option
 *
 * Return:   None
 *
 * Notes:    none
 */
void WsendSMS()
{
    BYTE  hexa[200]={0};
    CHAR  *pdu;
    CHAR  *pEndOfLine = NULL;
    ULONG nRet;
    ULONG messageFailureCode;
    CHAR  messageFormat[4];
    CHAR  mobileNo[22] = {'\0'};
    CHAR  smsMessage[162] = {'\0'};
    BYTE  MessageSize = 240;
    BYTE  EncodingAlphabet = 2;
    BYTE *pMessage;
    ULONG idx;
    ULONG flag = 0;
    CHAR  encodingscheme[OPTION_LEN] = {'\0'};

    while(1)
    {
        /* Take mobile as input from user */
        fprintf( stderr,
                 "Enter mobile number with <Country Code Digits>"
                 " followed by ENTER KEY:\n" );

        fgets( mobileNo, 22, stdin );

        /* If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        pEndOfLine = strchr( mobileNo, ENTER_KEY );
        if( NULL == pEndOfLine )
        {
            FlushStdinStream();
        }
        else
        {
            *pEndOfLine = 0;
        }
        if( mobileNo == pEndOfLine )
            return;

        for(idx = 0; idx <= 2; idx++)
        {
            if( !isdigit(mobileNo[idx]) )
            {
                if ( gNotificationFlag )
                {
                    gNotificationFlag--;
                    if ( 'y' == mobileNo[idx] ||'Y' == mobileNo[idx] )
                    {
                        ReadNewSMS();
                    }
                    return;
                }

                fprintf(stderr,"Please enter valid digits!\n\n");
                flag = 1;
                break;
            }
            if ( gNotificationFlag )
            {
                if ( isdigit(mobileNo[idx]) )
                {
                    return;
                }
            }
        }
        if ( 1 == flag )
        {
            flag = 0;
            continue;
        }
        if( strlen(mobileNo) > 20 )
        {
            fprintf(stderr,"Please enter a valid mobile number\n\n");
            mobileNo[20] = '\0';
            continue;
        }
        break;
    }

    fprintf(stderr,
                  "Enter the message to send (160 character limit)"\
                  " followed by the <Enter> key\n" );

    fgets( smsMessage, 162, stdin);
    if( strlen(smsMessage) > 160  )
    {
        fprintf(stderr,"Maximum character limit of 160 has been reached; "
                       " excess characters will be discarded\n\n");
        smsMessage[160] = '\0';
    }

    /* If '/n' character is not read, there are more characters in input
     * stream. Clear the input stream.
     */
    pEndOfLine = strchr( smsMessage, ENTER_KEY );
    if( NULL == pEndOfLine )
    {
        FlushStdinStream();
    }
    else
    {
        *pEndOfLine = 0;
    }

    if( smsMessage == pEndOfLine )
        return;

    while(1)
    {
        fprintf( stderr,
                 "Enter the message format(0 for CDMA, 6 for GSM/WCDMA)"\
                 " followed by <Enter> key:\n" );

        fgets( messageFormat, OPTION_LEN, stdin);
        if( '6' != messageFormat[0] && '0' != messageFormat[0] )
        {
            fprintf(stderr,"Message format entered is invalid!\n");
            continue;
        }
        break;
    }
    pEndOfLine = strchr( messageFormat, ENTER_KEY );
    if( NULL == pEndOfLine )
    {
        FlushStdinStream();
    }

    if( messageFormat == pEndOfLine )
        return;

    if ( '6' == messageFormat[0] )
    {
         getEncodingscheme(encodingscheme);

         /* Allocate pdu buffer memory */
         pdu = (CHAR*)malloc(800*sizeof(CHAR));

         nRet = SwiMakePdu(mobileNo,smsMessage,strlen(smsMessage),pdu, 0, encodingscheme[0]);
         if( eQCWWAN_ERR_NONE != nRet )
         {
             fprintf( stderr,
                      "Failed to send SMS: %lu\n", nRet );
             return;
         }
        SwiAsc2Hex(hexa, pdu);

         /* Send SMS */
         nRet = SendSMS( (ULONG)(messageFormat[0] - 0x30),
                         strlen(pdu)/2,
                         hexa,
                         &messageFailureCode);
         if ( !nRet )
         {
             fprintf(stderr, "SMS sent successfully\n");
         }
         else if ( messageFailureCode )
         {
             fprintf( stderr,
                      "Failed to send SMS: Failure Code: 0x%lx Failure Cause:0x%lx\n"
                      ,nRet, messageFailureCode);
         }

         /* release pdu buffer memory */
         free(pdu);
     }

    if ( '0' == messageFormat[0])
    {
        /* Allocate pdu buffer memory */
        pMessage = (BYTE*)malloc(240*sizeof(CHAR));
        SwiCDMAEncodeMOTextMsg( &MessageSize,
                             pMessage,
                             0,
                             mobileNo,
                             NULL,
                             strlen(smsMessage),
                             (wchar_t*)smsMessage,
                             NULL,
                             &EncodingAlphabet,
                             NULL );
         /* Send SMS */
         nRet = SendSMS( (ULONG)(messageFormat[0] - 0x30),
                         MessageSize,
                         pMessage,
                         &messageFailureCode);
         if ( !nRet )
         {
             fprintf(stderr, "SMS sent successfully\n");
         }
         else if ( messageFailureCode )
         {
             fprintf( stderr,
                      "Failed to send SMS: Failure Code: 0x%lx Failure Cause:0x%lx\n"
                       ,nRet, messageFailureCode);
         }

         /* release pdu buffer memory */
         free(pMessage);
    }
 }

/*
 * Name:     WgetSMSList
 *
 * Purpose:  Wrapper function for getSMSList
 *
 * Return:   None
 *
 * Notes:    none
 */

void WgetSMSList()
{
    ULONG  RequestedTag;
    CHAR   *pEndOfLine;
    CHAR   selOption[OPTION_LEN];

    while(1)
    {
        /* Update list with recently received message */
        ListAllMessages();
        fprintf( stderr,
                 "\nSelect one of the lists below or press <Enter>"
                 " to go back to the main menu:\n"\
                 "1. Read messages\n"\
                 "2. Unread messages\n"\
                 "3. Mobile originated messages that have been sent"
                 " to the recipient\n"\
                 "4. Mobile originated messages that have yet to be"
                 " sent to the recipient\n"\
                 "5. All messages\n"\
                 "Option: " );

        /* Receive the input from the user */
        fgets( selOption, ( OPTION_LEN ), stdin );

        /* If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        pEndOfLine = strchr( selOption, ENTER_KEY );
        if( NULL == pEndOfLine )
        {
            FlushStdinStream();
        }
        /* If user has entered an invalid input, prompt again */
        if( 2 < strlen(selOption) )
        {
            continue;
        }

        if( selOption[0] > '5' || selOption[0] == ENTER_KEY )
        {
            fprintf( stderr, "Back to main Menu!\n");
            return;
        }
        else if( selOption[0] <= '4' )
        {
            RequestedTag = selOption[0] - eREAD_MESSAGES;
            Displaymsgs(&RequestedTag);
        }
        else if ( selOption[0] == '5' )
        {
            RequestedTag = selOption[0] - eREAD_MESSAGES;
            Displaymsgs(&RequestedTag);
        }
    }
}

/*
 * Name:     WReadSMS
 *
 * Purpose:  Wrapper function for API GetSMS
 *
 * Return:   None
 *
 * Notes:    none
 */

void WReadSMS(void)
{
    ULONG  RequestedTag;
    CHAR   *pEndOfLine;
    CHAR   selOption[OPTION_LEN];

    while(1)
    {
        /* Update the list with recently received Messages */
        ListAllMessages();
        fprintf( stderr,
                 "\nSelect the type of message you want to read"\
                 " press <enter> to go back to main menu:\n"\
                 "1. Read messages\n"\
                 "2. Unread messages\n"\
                 "3. Mobile originated messages that have been sent to"
                 " the recipient\n"\
                 "4. Mobile originated messages that have yet to be sent"
                 " to the recipient\n"\
                 "Option: " );

        /* Receive the input from the user */
        fgets( selOption, ( OPTION_LEN ), stdin );

        /*
         * If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        pEndOfLine = strchr( selOption, ENTER_KEY );
        if( NULL == pEndOfLine )
        {
            FlushStdinStream();
        }

        /* If user has entered an invalid input, prompt again */
        if( 2 < strlen(selOption) )
        {
            continue;
        }

        if( selOption[0] > '4' || selOption[0] == ENTER_KEY )
        {
            fprintf( stderr, "Back to main Menu!\n");
            return;
        }

        RequestedTag = selOption[0] - eREAD_MESSAGES;
        gMessageTag = RequestedTag;
        if( 0 == Displaymsgs( &RequestedTag) )
            continue;

        WGetSMS4Index();

    }
}

/*
 * Name:     WGetSMS4Index
 *
 * Purpose:  Read SMS with the provided Index
 *
 * Return:   None
 *
 * Notes:    none
 */
void WGetSMS4Index()
{
    ULONG idx;
    ULONG nRet;
    ULONG flag = 0;
    ULONG messageTag;
    ULONG messageFormat;
    ULONG messageIndex;
    ULONG messageSize = nMaxStrLen + 2;;
    BYTE  message[nMaxStrLen + 2]= {'\0'};
    CHAR  Option[OPTION_LEN];
    CHAR  ascii[MAXTEXT]= {'\0'};
    BYTE  SenderAddrLength = 16;
    BYTE  mcTimeStamp[8] = {0};
    BYTE  TextMsgLength = 160;
    CHAR  SenderAddr[16] = {'\0'};
    WORD  TextMsg[160] = {'\0'};
    BYTE  SMSCAddrLength   = 16;
    CHAR  SMSCAddr[16]     = {'\0'};

    while(1)
    {
        if ( 0 == gMsgCountTag )
        {
            break;
        }
        fprintf(stderr,"\nEnter the 'Message Index' of the SMS"
                       " message to read or press <Enter> to go back:\n");

        /* Receive the input from the user */
        fgets( Option, ( OPTION_LEN ), stdin );

        /* If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        if( Option[0] == ENTER_KEY)
        {
            return;
        }
        if( isdigit(Option[0]))
        {
            messageIndex = atoi(Option);
            for( idx = 0; idx < messageCount ; idx++ )
            {
                if (indexmaps[idx].mappedIndex == messageIndex)
                {
                    messageIndex = indexmaps[idx].originalIndex;
                    flag = 1;
                    break;
                }
            }
            if ( !flag )
            {
                fprintf(stderr,"This Index not present in list!\n");
                continue;
            }
            flag = 0;
        }
        else
        {
            fprintf(stderr,"Index entered is not correct!\n");
            continue;
        }

        /* Execute SLQSGetSMS API with storage index 1 */
        nRet = SLQSGetSMS( STORAGEINDEX_NV,
                           messageIndex,
                           &messageTag,
                           &messageFormat,
                           (ULONG *)&messageSize,
                           &message[0],
                           &messageMode );
        if ( !nRet )
        {
            fprintf( stderr,
                     "Read from NV memory successful\nMessage Tag: %lu\n"\
                     "Message Format: %lu\n",
                      gMessageTag, messageFormat );

            if ( UMTS == messageMode )
            {
                /* Structure to decode PDU message to ASCII */
                struct wcdmaMsgDecodingParams WcdmaMsgDecodingParams;

                WcdmaMsgDecodingParams.pMessage          = message;
                WcdmaMsgDecodingParams.pSenderAddrLength = &SenderAddrLength;
                WcdmaMsgDecodingParams.pTextMsgLength    = &TextMsgLength;
                WcdmaMsgDecodingParams.pScAddrLength     = &SMSCAddrLength;

                WcdmaMsgDecodingParams.pSenderAddr = SenderAddr;
                WcdmaMsgDecodingParams.pTextMsg    = ascii;
                WcdmaMsgDecodingParams.pScAddr     = SMSCAddr;

                nRet = SLQSWCDMADecodeMTTextMsg( &WcdmaMsgDecodingParams );

                if( eQCWWAN_ERR_NONE != nRet )
                {
                    fprintf( stderr,"Failed to Read SMS \n");
                    fprintf( stderr,"Failure Code: %lu\n", nRet );
                }

                switch( nRet )
                {
                    case eQCWWAN_ERR_SWISMS_MSG_CORRUPTED:
                        fprintf( stderr,"Message is not available or is corrupted\n" );
                        break;

                    case eQCWWAN_ERR_SWISMS_SMSC_NUM_CORRUPTED:
                        fprintf( stderr," SMSC/Mobile number is not correct\n" );
                        break;

                    case eQCWWAN_ERR_BUFFER_SZ:
                        fprintf( stderr," Internal Error - Can not read SMS \n" );
                        break;

                    case eQCWWAN_ERR_SWISMS_MSG_LEN_TOO_LONG:
                        fprintf( stderr," Message exceeds 160 characters\n" );
                        break;

                    case eQCWWAN_ERR_NONE:
                        fprintf( stderr, "\nService Center Number = %s\n",
                                 WcdmaMsgDecodingParams.pScAddr );
                        fprintf( stderr, "Date = %s\n",
                                 WcdmaMsgDecodingParams.Date );
                        fprintf( stderr, "Time = %s\n",
                                 WcdmaMsgDecodingParams.Time );
                        fprintf( stderr, "Message: %s\n",
                                 WcdmaMsgDecodingParams.pTextMsg );
                        //printMsg();
                        break;

                    default:
                        break;
                }
               fprintf(stderr,"\n");
            }

            if ( CDMA == messageMode )
            {
                SwiCDMADecodeMTTextMsg( messageSize,
                                        message,
                                        NULL,
                                        &SenderAddrLength,
                                        SenderAddr,
                                        &TextMsgLength,
                                        (WORD*)&TextMsg,
                                        NULL,
                                        NULL,
                                        NULL,
                                        mcTimeStamp,
                                        NULL,
                                        NULL,
                                        NULL,
                                        NULL,
                                        NULL,
                                        NULL,
                                        NULL,
                                        NULL);

                /* Print SMS details */
                fprintf(stderr,"Sender Number: %s\n",SenderAddr);
                fprintf(stderr,"SMS Text: %s\n",(CHAR*)TextMsg);
                fprintf(stderr,"Time Stamp: %d:%d:%d:%d:%d:%d\n",
                                mcTimeStamp[0],mcTimeStamp[1],
                                mcTimeStamp[2],mcTimeStamp[3],
                                mcTimeStamp[4],mcTimeStamp[5]);
            }

            /* Modify the tag of the Message that has been read */
            if( TAG_MSG_MT_UNREAD == gMessageTag )
            {
                SLQSModifySMSStatus( STORAGEINDEX_NV,
                                     messageIndex,
                                     TAG_MSG_MT_READ,
                                     &messageMode );
            }

            /* Update the list of messages as tags have been modified*/
            ListAllMessages();

            /* Display Message list */
            Displaymsgs(&gMessageTag);
        }

        messageSize = nMaxStrLen;
        if(nRet)
        {
            /* Execute GetSMS API with storage index 0 */
            nRet = SLQSGetSMS( STORAGEINDEX_UIM,
                               messageIndex,
                               &messageTag,
                               &messageFormat,
                               (ULONG *)&messageSize,
                               &message[0],
                               &messageMode);
            if ( !nRet )
            {
                fprintf( stderr,
                         "Read UIM memory successful\nMessage Tag: %lu\n"\
                         "Message Format: %lu\n",
                         gMessageTag, messageFormat );

                if ( UMTS == messageMode )
                {
                    /* Structure to decode PDU message to ASCII */
                    struct wcdmaMsgDecodingParams WcdmaMsgDecodingParams;

                    WcdmaMsgDecodingParams.pMessage          = message;
                    WcdmaMsgDecodingParams.pSenderAddrLength = &SenderAddrLength;
                    WcdmaMsgDecodingParams.pTextMsgLength    = &TextMsgLength;
                    WcdmaMsgDecodingParams.pScAddrLength     = &SMSCAddrLength;

                    WcdmaMsgDecodingParams.pSenderAddr = SenderAddr;
                    WcdmaMsgDecodingParams.pTextMsg    = ascii;
                    WcdmaMsgDecodingParams.pScAddr     = SMSCAddr;

                    nRet = SLQSWCDMADecodeMTTextMsg( &WcdmaMsgDecodingParams );

                    if( eQCWWAN_ERR_NONE != nRet )
                    {
                        fprintf( stderr,"Failed to Read SMS \n");
                        fprintf( stderr,"Failure Code: %lu\n", nRet );
                    }

                    switch( nRet )
                    {
                        case eQCWWAN_ERR_SWISMS_MSG_CORRUPTED:
                            fprintf( stderr,"Message is not available or is corrupted\n" );
                            break;

                        case eQCWWAN_ERR_SWISMS_SMSC_NUM_CORRUPTED:
                            fprintf( stderr," SMSC/Mobile number is not correct\n" );
                            break;

                        case eQCWWAN_ERR_BUFFER_SZ:
                            fprintf( stderr," Internal Error - Can not read SMS \n" );
                            break;

                        case eQCWWAN_ERR_SWISMS_MSG_LEN_TOO_LONG:
                            fprintf( stderr," Message exceeds 160 characters\n" );
                            break;

                        case eQCWWAN_ERR_NONE:
                            fprintf( stderr, "\nService Center Number = %s\n",
                                     WcdmaMsgDecodingParams.pScAddr );
                            fprintf( stderr, "Date = %s\n",
                                     WcdmaMsgDecodingParams.Date );
                            fprintf( stderr, "Time = %s\n",
                                     WcdmaMsgDecodingParams.Time );
                            fprintf( stderr, "Message: %s\n",
                                     WcdmaMsgDecodingParams.pTextMsg );
                            //printMsg();
                            break;

                        default:
                            break;
                    }
                   fprintf(stderr,"\n");
                }
                if ( CDMA == messageMode )
                {
                    SwiCDMADecodeMTTextMsg( messageSize,
                                            message,
                                            NULL,
                                            &SenderAddrLength,
                                            SenderAddr,
                                            &TextMsgLength,
                                            (WORD*)&TextMsg,
                                            NULL,
                                            NULL,
                                            NULL,
                                            mcTimeStamp,
                                            NULL,
                                            NULL,
                                            NULL,
                                            NULL,
                                            NULL,
                                            NULL,
                                            NULL,
                                            NULL);
                    /* Print SMS details */
                    fprintf(stderr,"Sender Number: %s\n",SenderAddr);
                    fprintf(stderr,"SMS Text: %s\n",(CHAR*)TextMsg);
                    fprintf(stderr,"Time Stamp: %d:%d:%d:%d:%d:%d\n",
                            mcTimeStamp[0],mcTimeStamp[1],
                            mcTimeStamp[2],mcTimeStamp[3],
                            mcTimeStamp[4],mcTimeStamp[5]);
                }

                /* Modify the tag of the Message that has been read */
                if( TAG_MSG_MT_UNREAD == gMessageTag )
                {
                    SLQSModifySMSStatus( STORAGEINDEX_UIM,
                                         messageIndex,
                                         TAG_MSG_MT_READ,
                                         &messageMode );
                }

                /* Update the list of messages as tags have been modified*/
                ListAllMessages();

                /* Display Message list */
                Displaymsgs(&gMessageTag);
            }
        }
    }
}

/*
 * Name:     ExecuteDeleteSMS
 *
 * Purpose:  Deletes SMS with provided parameter
 *
 * Params:   select[IN] - Indicates below memory type
 *                        1 - UIM memory
 *                        2 - NV memory
 * Return:   None
 *
 * Notes:    none
 */

void ExecuteDeleteSMS( CHAR select )
{
    ULONG nRet;
    ULONG messageIndex;
    ULONG StorageType;
    CHAR  Option[4];
    ULONG flag = 0;
    ULONG idx = 0;

    /* Execute only when option 2 is selected */
    while(1)
    {
        if ( select <= '2' )
        {
            if ( '1' == select )
            {
                 if( 0 == ShowList("UIM") )
                 {
                     fprintf(stderr," UIM Memory does not contain any messages!\n");
                     return;
                 }
                 else
                 {
                     /* Initialize storage type with UIM memory */
                     StorageType = STORAGEINDEX_UIM;
                 }
            }
            if ( '2' == select )
            {
                 if( 0 == ShowList("NV") )
                 {
                     fprintf(stderr," NV Memory does not contain any messages!\n");
                     return;
                 }
                 else
                 {
                     /* Initialize storage type with NV memory */
                     StorageType = STORAGEINDEX_NV;
                 }

            }
            fprintf(stderr,"Enter Index of message to delete"
                           " or press <Enter> to exit:\n");

            /* Receive the input from the user */
            fgets( Option, ( OPTION_LEN ), stdin );

            if( !isdigit(Option[0]) )
            {
                if ( gNotificationFlag )
                {
                    gNotificationFlag--;
                    if ( 'y' == Option[0] )
                    {
                        ReadNewSMS();
                    }
                    return;
                }

            }

            /* If '/n' character is not read, there are more characters in input
             * stream. Clear the input stream.
             */
            if ( Option[0] == ENTER_KEY )
            {
                return;
            }
            if( isdigit(Option[0]))
            {
                messageIndex = atoi(Option);
                for( idx = 0; idx < messageCount ; idx++ )
                {
                    if ( indexmaps[idx].mappedIndex == messageIndex )
                    {
                        messageIndex = indexmaps[idx].originalIndex;
                        flag = 1;
                        break;
                    }
                }
                if ( !flag )
                {
                    fprintf(stderr,"This Index not present in list!\n");
                    continue;
                }
                flag = 0;
            }
            else
            {
                fprintf(stderr,"Index entered is not correct!\n");
            }

            nRet = SLQSDeleteSMS( StorageType,
                                  &messageIndex,
                                  NULL,
                                  &messageMode );
            if (!nRet)
            {
                fprintf(stderr,"Message Deleted\n");
            }
            else
            {
                fprintf(stderr,"Message Deletion unsuccessful\n");
            }
        }
        else
        {
            break;
        }

    }

    if ( '3' == select )
    {
        StorageType = 0;
        nRet = SLQSDeleteSMS( StorageType,
                              NULL,
                              NULL,
                             &messageMode );
           if (!nRet)
       {
           fprintf(stderr,"All Messages in UIM Memory Deleted\n");
       }
       else
       {
           fprintf(stderr,"SMS Deletion unsuccessful\n");
       }
   }
   if ( '4' == select )
   {
       StorageType = 1;
       nRet = SLQSDeleteSMS( StorageType,
                             NULL,
                             NULL,
                            &messageMode );
       if (!nRet)
       {
           fprintf(stderr,"All Messages in NV Memory Deleted\n");
       }
       else
       {
           fprintf(stderr,"SMS Deletion unsuccessful\n");
       }
  }

}

/*
 * Name:     WDeleteMessage
 *
 * Purpose:  Displays Menu for delete message
 *
 * Return:   None
 *
 * Notes:    none
 */

void WDeleteMessage()
{
    CHAR  *pEndOfLine;
    CHAR  selOption[OPTION_LEN];

    while(1)
    {
        /* Update List with recently received message */
        ListAllMessages();
        fprintf( stderr,
                 "\nSelect the type of message you want to delete"\
                 " press <enter> to go back to main menu:\n"\
                 "1. Delete a specific SMS message from UIM memory\n"\
                 "2. Delete a specific SMS message from NV memory\n"\
                 "3. Delete all the messages stored in UIM memory\n"\
                 "4. Delete all the messages stored in NV memory\n"\
                 "Option: " );

        /* Receive the input from the user */
        fgets( selOption, ( OPTION_LEN ), stdin );

        /*
         * If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        pEndOfLine = strchr( selOption, ENTER_KEY );
        if( NULL == pEndOfLine )
        {
            FlushStdinStream();
        }

        /* If user has entered an invalid input, prompt again */
        if( 2 < strlen(selOption) )
        {
            continue;
        }

        if( selOption[0] > '4' || selOption[0] == ENTER_KEY )
        {
            fprintf( stderr, "Back to main Menu!\n");
            return;
        }
        ExecuteDeleteSMS(selOption[0]);
    }
}

/*
 * Name:     main
 *
 * Purpose:  Entry point of the application
 *
 * Return:   EXIT_SUCCESS, EXIT_FAILURE on unexpected error
 *
 * Notes:    none
 */
int main(int argc, const char *argv[])
{
    ULONG resultCode = 0;
    CHAR  selOption[OPTION_LEN];
    CHAR  *pEndOfLine = NULL;

   if( argc < 2 )
    {
        fprintf( stderr, "usage: %s <path to sdk binary>\n", argv[0] );
        exit( EXIT_SUCCESS );
    }

    if( NULL == (sdkbinpath = malloc(strlen(argv[1]) + 1)) )
    {
        perror(__func__);
        exit( EXIT_FAILURE );
    }

    strncpy( sdkbinpath, argv[1], strlen(argv[1]) + 1);

    /* Start the SDK */
    resultCode = StartSDK();
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        free(sdkbinpath);

        /* Display the failure reason */
        fprintf( stderr,
                 "Failed to start SDK: Exiting App\n"\
                 "Failure Code: %lu\n", resultCode );

        /* Failed to start SDK, exit the application */
        exit( EXIT_FAILURE );
    }

    GetMsgProtocol();

    while(1)
    {
         /* Checks for Notificationflag to read next message */
         if ( gNotificationFlag )
         {
             fprintf(stderr,"Please enter (y/n)message is in queue \n");
         }
         else
         {
         fprintf( stderr,
                  "\nPlease select one of the following options or"
                  " press <Enter> to exit:\n"\
                  "1. Register CallBack\n"\
                  "2. Send SMS\n"\
                  "3. Get SMS List\n"\
                  "4. Read SMS\n"\
                  "5. Delete SMS\n"\
                  "Option: " );
         }

        /* Receive the input from the user */
        fgets( selOption, ( OPTION_LEN ), stdin );

        /* If '/n' character is not read, there are more characters in input
         * stream. Clear the input stream.
         */
        pEndOfLine = strchr( selOption, ENTER_KEY );
        if( NULL == pEndOfLine )
        {
            FlushStdinStream();
        }
        /* If user has entered an invalid input, prompt again */
        if( 2 < strlen(selOption) )
        {
            if ( !gNotificationFlag )
            {
                fprintf(stderr,"Please choose valid option from menu \n");
            }
            continue;
        }

        /* Execute the option provided by user */
        switch( selOption[0] )
        {
            case eREGISTER_CALLBACK:
                ActivateSMSCallback();
                break;

            case eSEND_SMS:
                WsendSMS();
                break;

            case eGETSMS_LIST:
                WgetSMSList();
                break;

            case eREADSMS:
                WReadSMS();
                break;

            case eDELETESMS:
                WDeleteMessage();
                break;

            case 'y':
            case 'Y':
                gNotificationFlag--;
                ReadNewSMS();
                break;

            case 'n':
            case 'N':
                gNotificationFlag-- ;
                break;

            case eEXIT_APP:
                free(sdkbinpath);
                QCWWANDisconnect();
                exit(EXIT_SUCCESS);
                break;

            default:
               break;
        }
    }
}
