/*
 * Atheros PHY drivers + ethtest driver
 *
 */
#include <common.h>
#include <command.h>
#include <exports.h>
#include <config.h>

#include <net.h>
#include <miiphy.h>
#include <netdev.h>

#include <phy.h>
#include "ethtest.h"


/* Accessing Atheros debug registers */

__attribute__((unused)) static int debug_read(struct phy_device *phydev, unsigned short dreg, unsigned short *val)
{
	int rval;
	rval = phy_write(phydev, MDIO_DEVAD_NONE, 0x1D, dreg & 0x3f); /* Select pair */
	if (!rval) rval = phy_read(phydev, MDIO_DEVAD_NONE, 0x1E); /* Register Value */
	if (rval >= 0) *val=rval & 0xFFFF;
	return rval;
}

__attribute__((unused)) static int debug_write(struct phy_device *phydev, unsigned short dreg, unsigned short val)
{
	int rval;
	rval = phy_write(phydev, MDIO_DEVAD_NONE, 0x1D, dreg & 0x3f); /* Select pair */
	if (!rval) rval = phy_write(phydev, MDIO_DEVAD_NONE, 0x1E, val); /* Register Value */
	return rval;
}


static int ar8035_config(struct phy_device *phydev)
{
	return genphy_config(phydev);
}

/* Reset */
static int ar8035_reset(struct phy_device *phydev)
{
	int v;

	/* Software reset, autoneg / speed setting survives
	 * reset, MDIX and a few other setting applies at reset. */
	v = phy_read(phydev, MDIO_DEVAD_NONE, MII_BMCR);
	phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR, v | BMCR_RESET );
	while (phy_read(phydev, MDIO_DEVAD_NONE, MII_BMCR) & BMCR_RESET);

	return 0;
}


/* Reset & Enable autonegotiation */
static int ar8035_aneg(struct phy_device *phydev)
{
	int v;

	/* Enable auto-MDIX. The testmodes may disable it. */
	v = phy_read(phydev, MDIO_DEVAD_NONE, 0x10);
	phy_write(phydev, MDIO_DEVAD_NONE, 0x10, v | 0x60);

	/* Enable autoneg. */
	v = BMCR_FULLDPLX | BMCR_ANENABLE | BMCR_SPEED100;
	phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR, v);

	ar8035_reset(phydev);

	/* Restart aneg */
	phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR, v | BMCR_ANRESTART );

	/* We don't wait here, the network stack will poll link state anyway. */

	return 0;
}

/* This is called once, on first use. We use the opportunity
 * to set up the RGMII delays. Shouldn't do anything else.
 * WARNING: Also called by 1G test mode below, to revert a hack. */
static int ar8035_probe(struct phy_device *phydev)
{
	/* Adjusting RGMII Tx Clock delay */

	debug_write(phydev, 0x05, 0x2d47);
	//debug_write(phydev, 0x0B, 0x3c00); /* 0.7ns, Nguni: works badly, glitchy. */
	//debug_write(phydev, 0x0B, 0x3c20); /* 1.3ns, Nguni: seems OK */
	debug_write(phydev, 0x0B, 0x3c40); /* 2.4ns, Nguni: seems OK. Optimal value, furthest away from trouble. */
	//debug_write(phydev, 0x0B, 0x3c60); /* 3.4ns, Nguni: works badly when freezer sprayed, not otherwise. */

	return 0;
}


static int ar8035_startup(struct phy_device *phydev)
{
	/* The genphy link function always waits for autoneg, even if nothing
	 * is connected. This takes too long. We assume autoneg has happened already.
	 * Reading status twice to clear latched state. */
	phy_read(phydev, MDIO_DEVAD_NONE, MII_BMSR);
	phydev->link = !!(phy_read(phydev, MDIO_DEVAD_NONE, MII_BMSR) & BMSR_LSTATUS);
	/* Work out speed & duplex */
	return genphy_parse_link(phydev);
}

static int ar8035_shutdown(struct phy_device *phydev)
{
	return genphy_shutdown(phydev);
}


struct phy_driver AR8035_driver =  {
	.name = "AR8035",
	.uid =  0x004dd072,
	.mask = 0x00fffff0,
	.features = PHY_GBIT_FEATURES,
	.config = ar8035_config,
	.probe = ar8035_probe,
	.startup = ar8035_startup,
	.shutdown = ar8035_shutdown,
};

static int ethtest_probe(struct ethtest_driver *ethtest)
{
	/* Nothing to do here atm., but if we have to match multiple IDs
	 * for different PHYs, this is the place to do it. */
	return 0;
}

/* Time domain reflectometry cable test */
static int ethtest_cable(struct ethtest_driver *ethtest)
{
	unsigned v, dist;
	int val;
	struct phy_device *phydev = ethtest->phydev;

	printf("	Cable test\n");
	/* Iterate over cable pairs */
	for (v=0; v<4; v++) {
		printf("		Pair %d, ", v);
		phy_write(phydev, MDIO_DEVAD_NONE, 0x16, (v << 8) | 0 ); /* Select pair */
		phy_write(phydev, MDIO_DEVAD_NONE, 0x16, (v << 8) | 1 ); /* start test */
		do {
			val = phy_read(phydev, MDIO_DEVAD_NONE, 0x16);
		} while (val & 1); /* Wait until done */
		val = phy_read(phydev, MDIO_DEVAD_NONE, 0x1C); /* Read result */
		switch ((val >> 8) & 0x3) {
			case  0: printf("OK, "); break;
			case  1: printf("SHORT, "); break;
			case  2: printf("OPEN, "); break;
			case  3: printf("FAIL, "); break;
		}
		val &= 0xFF;
		dist = (float)val*84.2;
		printf("Distance: %d.%d (%d)\n", dist/100, dist%100, val);
	}
	return 0;
}

/* 10 MBit test modes */
static int ethtest_test10(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	if (mode != PT_MODE_NORMAL) {
		/* Force 10MBit/s, full duplex */
		phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR, BMCR_FULLDPLX );

		/* Force MDI */
		v = phy_read(phydev, MDIO_DEVAD_NONE, 0x10);
		phy_write(phydev, MDIO_DEVAD_NONE, 0x10, (v & ~0x60));

		/* MDIX setting takes effect after reset only */
		ar8035_reset(phydev);
	}

	v = 0x0C; /* These bits are default 1, read only, but the datasheet is lying. */
	switch (mode) {
		case PT10_MODE_10MHZ : v |= 0x01; break;
		case PT10_MODE_5MHZ : v |= 0x20; break;
		case PT10_MODE_RANDOM : v |= 0x02; break;
		case PT10_MODE_LINK : v |= 0x03; break;
		default : v |= 0; break;
	}

	debug_write(phydev, 0x12, v);

	/* Reset PHY to return to sanity */
	if (mode == PT_MODE_NORMAL) {
		ar8035_aneg(phydev);
	}

	return 0;
}

/* 100 MBit test modes */
static int ethtest_test100(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	if (mode != PT_MODE_NORMAL) {
		/* Force 100MBit/s, full duplex */
		phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR,
				BMCR_SPEED100 | BMCR_FULLDPLX );
		/* Force MDI */
		v = phy_read(phydev, MDIO_DEVAD_NONE, 0x10);
		phy_write(phydev, MDIO_DEVAD_NONE, 0x10, (v & ~0x60));

		/* MDIX setting takes effect after reset only */
		ar8035_reset(phydev);
	}

	switch (mode) {
		case PT100_MODE_JITTER: v = 0x80; break;
		case PT100_MODE_OVERSHOOT: v = 0x40; break;
		case PT100_MODE_DCD: v = 0x20; break;
		default : v = 0; break;
	}

	debug_write(phydev, 0x10, v);

	/* Reset PHY to return to sanity */
	if (mode == PT_MODE_NORMAL) {
		ar8035_aneg(phydev);
	}

	return 0;
}

/* Gigabit test modes */
static int ethtest_test1G(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	if (mode != PT_MODE_NORMAL) {
		/* Force 1GBit/s, full duplex */
		phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR,
				BMCR_SPEED1000 | BMCR_FULLDPLX );

		ar8035_reset(phydev);

		/* WARNING: Hack for undocumented 'feature'. We have to
		 * set a few debug register bits which are marked as
		 * 'reserved' and 'read only' in the datasheet.
		 * Without them, Gigabit test modes don't work.
		 * This breaks the RGMII TxCLK delay setting, so we
		 * fix that later. */
		debug_write(phydev, 0xB, 5);
	}

	/* All duplex bits, manual master */
	switch (mode) {
		case PT1G_MODE_WAVE: v = 0x1F00 | (0x1 << 13); break;
		case PT1G_MODE_JITTER_MASTER: v = 0x1F00 | (0x2 << 13); break;
		case PT1G_MODE_JITTER_SLAVE:  v = 0x1F00 | (0x3 << 13); break;
		case PT1G_MODE_DISTORTION:  v = 0x1F00 | (0x4 << 13); break;
		default: v = 0x0200; /* Normal, auto everything */ break;
	}

	phy_write(phydev, MDIO_DEVAD_NONE, 0x09, v);

	/* Reset PHY to return to sanity */
	if (mode == PT_MODE_NORMAL) {
		/* The undocumented hack above has garbled the TxCLK delay.
		 * We call the probe function because it restores the delay. */
		ar8035_probe(phydev);
		ar8035_aneg(phydev);
	}

	return 0;
}

struct ethtest_driver ethtdrv = {
	.name = "atheros",
	.uid =  0x004dd072,
	.mask = 0x00fffff0,
	.probe = ethtest_probe,
	.cable = ethtest_cable,
	.test10 = ethtest_test10,
	.test100 = ethtest_test100,
	.test1G = ethtest_test1G,
};

int phy_atheros_ethtest_init(void)
{
	phy_register(&AR8035_driver);
	ethtest_register(&ethtdrv);
	return 0;
}
