
#include <common.h>
#include <command.h>
#include <exports.h>
#include <config.h>

#include <net.h>
#include <miiphy.h>
#include <netdev.h>

#include "ethtest.h"

//#define ETHT_DEBUG

#ifdef ETHT_DEBUG
	#define D(...) printf("D:" __VA_ARGS__)
#else
	#define D(...) do{}while(0)
#endif

static int phytest_cable(struct ethtest_driver *edrv, int dummy);
static int phytest_off(struct ethtest_driver *edrv, int dummy);
static int phytest_10M(struct ethtest_driver *edrv, int mode);
static int phytest_100M(struct ethtest_driver *edrv, int mode);
static int phytest_1G(struct ethtest_driver *edrv, int mode);

struct modes {
	const char *string;
	const char *doc;
	int (*msfun)(struct ethtest_driver *edrv, int mode); /* Mode switch function to use */
	int mode;
};
static const struct modes am[] = {
	/* Built-in cable tester */
	{ "cable", "TDM measurements",
		phytest_cable, 0 },

	/* Switch any tests off */
	{ "off",   "Normal operation",
		phytest_off, PT_MODE_NORMAL },

	/* 10MBit test modes */
	{ "10mhz", "Harmonic test, All-1",
		phytest_10M, PT10_MODE_10MHZ },
	{ "5mhz",  "Sine",
		phytest_10M, PT10_MODE_5MHZ },
	{ "rand",  "TP_IDLE, Jitter, DiffV",
		phytest_10M, PT10_MODE_RANDOM },
	{ "link",  "Link pulses",
		phytest_10M, PT10_MODE_LINK },

	/* 100MBit test modes */
	{ "jitt",  "Jitter",
		phytest_100M, PT100_MODE_JITTER },
	{ "over",  "Overshoot",
		phytest_100M, PT100_MODE_OVERSHOOT },
	{ "dcd",   "DCD",
		phytest_100M, PT100_MODE_DCD },

	/* 1GBit test modes */
	{ "wave",  "Mode1, Waveform",
		phytest_1G, PT1G_MODE_WAVE },
	{ "jittm", "Mode2, Jitter MASTER",
		phytest_1G, PT1G_MODE_JITTER_MASTER },
	{ "jitts", "Mode3, Jitter SLAVE",
		phytest_1G, PT1G_MODE_JITTER_SLAVE },
	{ "dist",  "Mode4, Distortion",
		phytest_1G, PT1G_MODE_DISTORTION },

	/* End marker */
	{ NULL, NULL, 0 }
};

static LIST_HEAD(etht_drivers);

static int (*undofun)(struct ethtest_driver *ethtest, enum ethtest_mode mode); /* Last used test function */

/**
 * Generic functions, used if driver doesn't register its own.
 **/

/* Default probe function always succeeds */
static int gentest_probe(struct ethtest_driver *ethtest)
{
	return 0;
}

/* There is no generic way to generate 10MBit test signals.
 * It may be possible to send specific data packets istead. */
static int gentest_test10(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	printf("No %s in driver %s\n","test10",ethtest->name);
	return 1;
}

/* There is no generic way to generate 100MBit test signals.
 * It may be possible to send specific data packets istead. */
static int gentest_test100(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	printf("No %s in driver %s\n","test100",ethtest->name);
	return 1;
}

/* This one is part of the standard, should always work.
 * Exporting, in case drivers need to work around stuff. */
int gentest_test1G(struct ethtest_driver *ethtest, enum ethtest_mode mode)
{
	struct phy_device *phydev = ethtest->phydev;
	unsigned short v;

	D("Generic GBit test implementation\n");

	if (mode != PT_MODE_NORMAL) {
		/* Force 1GBit/s, full duplex */
		phy_write(phydev, MDIO_DEVAD_NONE, MII_BMCR,
				BMCR_SPEED1000 | BMCR_FULLDPLX );
	}

	/* All duplex bits, manual master */
	switch (mode) {
		case PT1G_MODE_WAVE: v = 0x1F00 | (0x1 << 13); break;
		case PT1G_MODE_JITTER_MASTER: v = 0x1F00 | (0x2 << 13); break;
		case PT1G_MODE_JITTER_SLAVE:  v = 0x1F00 | (0x3 << 13); break;
		case PT1G_MODE_DISTORTION:  v = 0x1F00 | (0x4 << 13); break;
		default: v = 0x0200; /* Normal, auto everything */ break;
	}

	phy_write(phydev, MDIO_DEVAD_NONE, 0x09, v);

	if (mode == PT_MODE_NORMAL) {
		/* Return to normal */
		phy_reset(phydev);
		phy_config(phydev);
	}
	return 0;
}

/* All we can do here is display link status */
int gentest_cable(struct ethtest_driver *ethtest)
{
	struct phy_device *phydev = ethtest->phydev;
	D("Generic cable test\n");
	phy_startup(phydev);
	printf("Link %s, ", phydev->link?"up":"down");
	switch(phydev->speed) {
		case SPEED_10: printf("%d", 10); break;
		case SPEED_100: printf("%d", 100); break;
		case SPEED_1000: printf("%d", 1000); break;
		default: printf("???"); break;
	}
	printf("MBit, ");
	switch(phydev->duplex) {
		case DUPLEX_HALF: printf("HDX\n"); break;
		default : printf("FDX\n"); break;
	}
	return 0;
}

/* Dummy driver, is used if no other driver matches. */
static struct ethtest_driver gentest_default = {
	.name = "default",
	.uid = 0,
	.mask = 0,
	.probe = gentest_probe,
	.cable = gentest_cable,
	.test10 = gentest_test10,
	.test100 = gentest_test100,
	.test1G = gentest_test1G,
};

/**
 * API
 **/

int ethtest_register(struct ethtest_driver *edrv)
{
	D("ethtest Register '%s'\n", edrv->name);

	/* Setting defaults for uninitialised methods */
	if (!edrv->probe) edrv->probe = gentest_probe;
	if (!edrv->test10) edrv->test10 = gentest_test10;
	if (!edrv->test100) edrv->test100 = gentest_test100;
	if (!edrv->test1G) edrv->test1G = gentest_test1G;
	if (!edrv->cable) edrv->cable = gentest_cable;

	INIT_LIST_HEAD(&edrv->list);
	list_add_tail(&edrv->list, &etht_drivers);
	return 0;
}

/**
 * Call trampolines
 **/

#ifdef ETHT_DEBUG
static const char *fun_id(struct ethtest_driver *edrv,
		int (*funp)(struct ethtest_driver *ethtest, enum ethtest_mode mode))
{
	static const char *rstr;

	if (funp == edrv->test10) rstr = "test10";
	else if (funp == edrv->test100) rstr = "test100";
	else if (funp == edrv->test1G) rstr = "test1G";
	else rstr = "ERROR";

	return rstr;
}
#endif

static int phytest_cable(struct ethtest_driver *edrv, int dummy)
{
	D("%s:%d:\n", __func__, __LINE__);

	/* The cable test driver is expected to leave the PHY in
	 * normal operation mode. */
	undofun = NULL;
	return edrv->cable(edrv);
}

static int phytest_off(struct ethtest_driver *edrv, int dummy)
{
	D("%s:%d:\n", __func__, __LINE__);

	if (undofun) {
		D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[PT_MODE_NORMAL]);
		undofun(edrv, PT_MODE_NORMAL);
	}
	undofun = NULL;

	return 0;
}

static int phytest_10M(struct ethtest_driver *edrv, int mode)
{
	int rval;
	D("%s:%d:\n", __func__, __LINE__);

	/* Disable previous testmode */
	if (undofun) {
		D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[PT_MODE_NORMAL]);
		undofun(edrv, PT_MODE_NORMAL);
	}

	undofun = edrv->test10;
	D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[mode]);
	rval = edrv->test10(edrv, mode);
	return rval;
}

static int phytest_100M(struct ethtest_driver *edrv, int mode)
{
	int rval;
	D("%s:%d:\n", __func__, __LINE__);

	/* Disable previous testmode */
	if (undofun) {
		D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[PT_MODE_NORMAL]);
		undofun(edrv, PT_MODE_NORMAL);
	}

	undofun = edrv->test100;
	D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[mode]);
	rval = edrv->test100(edrv, mode);
	return rval;
}

static int phytest_1G(struct ethtest_driver *edrv, int mode)
{
	int rval;
	D("%s:%d:\n", __func__, __LINE__);

	/* Disable previous testmode */
	if (undofun) {
		D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[PT_MODE_NORMAL]);
		undofun(edrv, PT_MODE_NORMAL);
	}

	undofun = edrv->test1G;
	D("Calling %s(%s)\n", fun_id(edrv, undofun), modestr[mode]);
	rval = edrv->test1G(edrv, mode);
	return rval;
}

/**
 * UI
 **/

/* List available modes */
static void dump_modes(void)
{
	const struct modes *m = am;
	while (m->string) {
		printf("  %10s ", m->string);
		if (m->msfun == phytest_10M)       printf("(10MBit) ");
		else if (m->msfun == phytest_100M) printf("(100MBit)");
		else if (m->msfun == phytest_1G)   printf("(1GBit)  ");
		else                               printf("         ");
		printf(" %s\n", m->doc);
		m++;
	}
}

static int fun_ethtest(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	const char *devname;
	struct mii_dev *bus;
	struct phy_device *phydev;
	struct ethtest_driver *edrv;
	unsigned short addr;
	struct list_head *entry;
	const struct modes *amp;

	if (argc==1) {
		/* List available modes */
		printf("Available modes for ethtest:\n\n");
		dump_modes();
		return 0;
	}

	if (argc==2) {
		printf("bad args.\n");
		return -1; /* Either no arguments or at least 2 */
	}

#if defined(CONFIG_MII_INIT)
	mii_init ();
#endif

	D("%d: '%s','%s','%s','%s'\n", argc, argv[0], argv[1], argv[2], argv[3]);

	devname = miiphy_get_current_dev();

	D("Devname: '%s'\n", devname);

	bus = miiphy_get_dev_by_name(devname);

	D("bus:%p\n", bus);

	/* Get PHY ID */
	addr = simple_strtoul (argv[1], NULL, 0);

	D("addr: %d\n", addr);

	phydev = get_phy_device(bus, addr, PHY_INTERFACE_MODE_NONE);

	if (phydev == NULL) {
		printf("No phy at addr %d (try mii info)\n", addr);
		return -1;
	}

	D("phydev: %p\n", phydev);
	D("id: %08x\n", phydev->phy_id);

	edrv = NULL;
	list_for_each(entry, &etht_drivers) {
		edrv = list_entry(entry, struct ethtest_driver, list);
		if ((edrv->uid & edrv->mask) == (phydev->phy_id & edrv->mask)) {
			edrv->phydev = phydev;
			/* Check the probe function if there is one */
			if (edrv->probe(edrv) != 0) continue; /* Skip if probe says so */
			D("match: '%s'\n", edrv->name);
			break;
		}
	}
	D("edrv: %p\n", edrv);

	/* At this point, edrv is either the correct driver or NULL. If NULL,
	   we force the default methods. */
	if (!edrv) {
		printf("Using generic ethtest!\n");
		edrv = &gentest_default;
	}

	/* argv[2] is command, look up table */
	amp = am;
	while (amp->string != NULL) {
		if (strcmp(argv[2], amp->string)==0) break;
		amp++;
	}

	if (amp->string == NULL) {
		printf("Unknown command: '%s'\n", argv[2]);
		return -1;
	}

	/* Run access function */
	amp->msfun(edrv, amp->mode);

	return 0;
}
U_BOOT_CMD(ethtest, CONFIG_SYS_MAXARGS, 1, fun_ethtest,
	"Ethernet test modes",
	        "            - list all valid modes\n"
	"ethtest ADDR cable  - run PHY cable test\n"
	"ethtest ADDR off    - Disable testmode, normal operation\n"
	"ethtest ADDR MODE   - Enable specified test mode\n"
	"ADDR is the PHY address, as listed by mii info\n"
);
