/*
 * Algorithm library for U-Boot environment handling. Supports legacy
 * U-Boot format (CRC32,DATA), redundant format (CRC32,FLAG,DATA) and
 * Netcomm high-reliability format (redundant over arbitrary number of
 * blocks & sub-blocks).
 *
 * The master copy of this file lives in the cdcs_apps/uenv. Copies may be used
 * within the U-Boot source, but must be unmodified.
 *
 * Iwo Mergler <Iwo@netcommwireless.com>
 */

/* U-Boot and kernel code define __KERNEL__ */
#ifndef __KERNEL__
	#include <stdio.h>  /* printf */
	#include <string.h> /* memset */
	#include <stdlib.h> /* malloc/free */
	#include <stdint.h> /* *int*_t types */
	#include <sys/types.h> /* For loff_t */
#else
	#include <common.h>
	#include <malloc.h>
#endif

#include "uenv_alg.h"

#ifndef ASSERT
	#if defined(TEST)
		#define ASSERT(cond) do { if (!(cond)) fprintf(stderr,"%s:%d: assert(" #cond ") failed\n", \
			__func__, __LINE__); }while(0)
	#else
		#define ASSERT(cond) do{}while(0)
	#endif
#endif

#if defined(DEBUG)
	#define dbg(...) fprintf(stderr, __VA_ARGS__)
#else
	#define dbg(...) do{}while(0)
#endif


/* Add a uenv descriptor to list
 *
 */
static void uenv_addlist(struct uenv *ue, struct uenv_desc * const ud)
{
	ud->next = NULL;
	if (ue->ud_end) ue->ud_end->next = ud;
	ue->ud_end = ud;
	if (!ue->ud_start) ue->ud_start = ud;
}

/* Delete all uenv descriptors.
 */
static void uenv_desc_del(struct uenv *ue)
{
	struct uenv_desc *ud;
	ud = ue->ud_start;
	while (ud) {
		ue->ud_start = ud->next;
		free(ud);
		ud = ue->ud_start;
	}
	ue->ud_end = NULL;
}

/** Allocate struct uenv.
 *
 * Please set the settings fields (envbase, envsize, eraseblock, stride
 * totsize, menvsize, type, readenv) to sensible values.
 *
 * @param max_size Maximum expected size for any environment block
 *
 * @return Pointer to an environment descriptor or NULL in case of error.
 */
struct uenv * uenv_init(unsigned max_size)
{
	struct uenv *ue;

	/* There is already one struct uenv_desc in struct struct uenv */
	ue = malloc(sizeof(struct uenv));
	if (ue) {
		memset(ue, 0, sizeof(struct uenv));
		ue->menvsize = max_size;
		ue->buffer = malloc(max_size);
		if (!ue->buffer) {
			free(ue);
			ue=NULL;
		}
	}
	return ue;
}


/** Releases all resources used by a uenv structure.
 *
 * @param ue Pointer to uenv to free.
 */
void uenv_free(struct uenv *ue)
{
	if (ue) {
		uenv_desc_del(ue);
		if (ue->buffer) free(ue->buffer);
		ue->buffer=NULL;
		free(ue);
	}
}


/* Compatibility code. Older U-Boot can't handle 64-bit printf formatting (%llx). */
static const char *hex64str(uint64_t v)
{
	static char space[16+1]="";
	uint32_t high, low;
	high = (uint32_t)(v >> 32);
	low = (uint32_t)(v);
	if (high) {
		sprintf(space,"%x%x", high, low);
	} else {
		sprintf(space,"%x", low);
	}
	return space;
}


/** Debug - dumps whole uenv structure in human readable form
 *
 * This only works if DEBUG or TEST macros are defined during compile.
 *
 * @param ue Pointer to uenv data structure to dump.
 * @param pr Print function similar to printf
 */
void uenv_dump(struct uenv *ue, int (*pr)(const char *fmt, ...))
{
	struct uenv_desc *ud;
	unsigned repcount;
	uint8_t oldflags=0xff;
	unsigned descount=0;

	if (!ue) {
		pr("UENV: %p\n", (void*)ue);
		return;
	}

	pr("UENV: envbase=0x%s, envsize=0x%x, eraseblock=0x%x, stride=0x%x\n",
		hex64str(ue->envbase), ue->envsize, ue->eraseblock, ue->stride);
	pr("      totsize=0x%x, menvsize=0x%x, type=%s%s%s%s%s\n",
		ue->totsize, ue->menvsize,
		(ue->type & UT_AUTO)?"AUTO ":"", (ue->type & UT_FORCE)?"FORCE ":"",
		(ue->type & UT_LEGACY)?"LEGACY ":"", (ue->type & UT_REDUNDANT)?"REDUNDANT ":"",
		(ue->type & UT_NHRF)?"NHRF ":"");

	repcount = 0;
	ud = ue->ud_start;
	while (ud) {
		descount++;
		if (ud->flags == oldflags) {
			if (oldflags & UF_ERASED || oldflags == 0) {
				repcount++;
				ud = ud->next;
				continue;
			}
		} else {
			if (repcount) {
				pr("        || == Repeated %d times ==\n", repcount);
				repcount = 0;
			}
		}
		oldflags = ud->flags;

		pr("        %c%c ofs=0x%s, size=0x%x, flag=%02x, flags=%s%s%s%s%s%s%s%s\n",
			(ud == ue->read_env)?'R':'|', (ud == ue->write_env)?'W':'|',
			hex64str(ud->offset), ud->size, ud->flag,
			(ud->flags & UF_ERROR)?"ERROR ":"",
			(ud->flags & UF_ERASED)?"ERASED ":"",
			(ud->flags & UF_CRCOK)?"CRCOK ":"",
			(ud->flags & UF_REDFLAG)?"REDFLAG ":"",
			(ud->flags & UF_REDWR)?"REDWR ":"",
			(ud->flags & UF_ERASE)?"ERASE ":"",
			(ud->flags & UF_WRITE)?"WRITE ":"",
			(ud->flags == 0)?"NOISE ":"");
		ud=ud->next;
	}
	if (repcount) {
		pr("        || == Repeated %d times ==\n", repcount);
	}
	pr("      Total %d descriptors.\n", descount);
}


/* CRC32 lookup table. Avoids the shift-xor loops required otherwise. */
static uint32_t crc32lut[256] = {
	0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f,
	0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
	0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2,
	0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
	0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
	0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
	0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c,
	0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
	0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423,
	0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
	0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106,
	0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
	0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d,
	0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
	0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
	0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
	0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7,
	0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
	0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa,
	0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
	0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81,
	0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
	0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84,
	0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
	0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
	0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
	0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e,
	0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
	0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55,
	0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
	0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28,
	0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
	0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f,
	0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
	0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
	0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
	0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69,
	0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
	0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc,
	0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
	0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693,
	0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
	0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};


/* Single CRC32 step.
 *
 * To calculate full CRC32 over a number of bytes,
 * start with crc=~0U and then take the 1-complement when done.
 *
 * @param crc Ptr to current CRC value, gets updated.
 * @param d   Data byte for this step.
 */
static void crc32step(uint32_t *crc, uint8_t d)
{
	uint32_t c = *crc;
	c = crc32lut[ ( c ^ d ) & 0xFF ] ^ ( c >> 8 );
	*crc = c;
}


/* Read 32-bit CRC value in U-Boot format.
 *
 * @param buf pointer to beginning of env block
 *
 * @return 32-bit CRC value in machine endianness
 */
static uint32_t read_crc(const uint8_t *buf)
{
	return (uint32_t)buf[0] << 0 |
	       (uint32_t)buf[1] << 8 |
	       (uint32_t)buf[2] << 16 |
	       (uint32_t)buf[3] << 24;
}


/* Write a U-Boot environment CRC
 *
 * @param buf pointer to start of U-Boot env block
 * @param crc CRC value in machine endianness
 */
static void write_crc(uint8_t *buf, uint32_t crc)
{
	buf[0]=((crc >> 0) & 0xFF);
	buf[1]=((crc >> 8) & 0xFF);
	buf[2]=((crc >> 16) & 0xFF);
	buf[3]=((crc >> 24) & 0xFF);
}


/* Compares two flag fields
 *
 * Flag fields represent 8-bit sequence numbers, wrapping around.
 *
 * @param fA First flag value
 * @param fB Second flag value
 *
 * @return <0 when fA is older than fB, =0 when equal, >0 when fA is newer than fB.
 */
static int env_flcomp(const uint8_t fA, const uint8_t fB)
{
	int r = (int)fA - (int)fB;
	/* Wraping around. We use separate checks for the two extremes, such
	 * that the magnitude is correct for env_fldist() below. */
	if (r > 127) r = r - 256;
	if (r < -128) r = r + 256;
	return r;
}


/*
 * Calculates the distance between two flag values
 */
static int env_fldist(const uint8_t fA, const uint8_t fB)
{
	return abs(env_flcomp(fA, fB));
}


/** Check the CRC in an env block.
 *
 * Evaluates a given memory buffer to see if it contains a valid environment.
 * The return value indicates the type of env block found. The erase detection
 * is not a guarantee, it just means that both CRC and the following 4 bytes
 * are 0xff.
 *
 * @param buf Pointer to start of environment block
 * 
 * @return Bitfield with flags. UF_ERASED=erased, UF_CRCOK=CRC matched,
 *         UF_REDFLAG=redundant structure, else legacy. Returns 0 for
 *         non-env data area.
 */
uint8_t env_check(const uint8_t *buf, unsigned size)
{
	uint32_t ecrc; /* CRC field in buffer */
	uint32_t crc_l, crc_r; /* Legacy and redundant crc hypothesis */

	ASSERT(size > 8);

	/* The ECC32 steps result in the 1-complement of the real CRC,
	 * so we modify our expectations accordingly. */
	ecrc = ~read_crc(buf);

	crc_l = crc_r = ~0UL;
	size -= 4; /* Skip CRC */
	buf  += 4;

	/* We don't have to check far to determine that this is erased,
	 * 0xff is not used in valid environments. This also reduces the
	 * chance of mis-detecting an erased page if there are correctable
	 * bit errors. * N.B. ecrc is the 1-complement of the CRC field,
	 * so 0x0 means the CRC field is 0xFFFFFFFF. */
	if (ecrc == 0 && *(uint32_t*)buf == 0xffffffff ) {
		return UF_ERASED;
	}

	/* CRC_R skips the first (flag) byte, as it's not under CRC */
	crc32step(&crc_l, *buf++);
	size--;

	while (size--) {
		crc32step(&crc_l, *buf);
		crc32step(&crc_r, *buf);
		buf++;
	}

	/* Environment must end with double 0 */
	buf -= 2;
	if (buf[0] != 0 || buf[1] != 0) {
		dbg("not ended in 00\n");
		return 0;
	}

	if (crc_r == ecrc) {
		return UF_CRCOK | UF_REDFLAG;
	} else if (crc_l == ecrc) {
		return UF_CRCOK;
	} else {
		dbg("CRC error, expected 0x%08x, got 0x%08x|0x%08x\n", ecrc, crc_l, crc_r);
		return 0;
	}
}


/** Check environment block with unknown size
 *
 * @param buf  Pointer to start of environment block
 * @param size Returns found environment size
 * @param min  Minimum expected env block size. Must be > 8
 * @param max  Maximum expected env block size. Must be > min
 * 
 * @return Bitfield with flags. UF_ERASED=erased, UF_CRCOK=CRC matched,
 *         UF_REDFLAG=redundant structure, else legacy. Returns 0 for
 *         non-env data area.
 */
uint8_t env_check_scan(const uint8_t *buf, unsigned *size, unsigned min, unsigned max)
{
	uint32_t ecrc; /* CRC field in buffer */
	uint32_t crc_l, crc_r; /* Legacy and redundant crc hypothesis */
	unsigned ofs = 0;
	unsigned zf;

	ASSERT(min <= max && min > 8);

	/* We reduce the min size by two, to simplify code in case
	 * an env is exactly min bytes long. */
	min-=2;

	/* The ECC32 steps result in the 1-complement of the real CRC,
	 * so we modify our expectations accordingly. */
	ecrc = ~read_crc(buf);

	crc_l = crc_r = ~0U;
	ofs += 4; /* Skip CRC */
	buf += 4;

	/* We don't have to check far to determine that this is potentially
	 * erased, 0xff is not used in valid environments. N.B. ecrc is the
	 * 1-complement of the CRC field, so 0x0 means the CRC field is 0xFFFFFFFF. */
	if (ecrc == 0 && *(uint32_t*)buf == 0xffffffff ) {
		return UF_ERASED;
	}

	/* CRC_R skips the first (flag) byte, as it's not under CRC */
	crc32step(&crc_l, *buf++);
	ofs++;

	/* Eat the minimum number of bytes */
	while (ofs < min) {
		crc32step(&crc_l, *buf);
		crc32step(&crc_r, *buf);
		buf++;
		ofs++;
	}

	/* Continue, while checking */
	zf = 0;
	while (ofs < max) {
		crc32step(&crc_l, *buf);
		crc32step(&crc_r, *buf);
		zf = (zf << 1) | !(*buf); /* 0-byte history flags */
		buf++; ofs++;
		/* Environment can only end with 00 */
		if ( (zf & 0x3) != 0x3 ) continue;

		/* This could be the end, check CRCs */
		if ( crc_r == ecrc ) {
			*size = ofs;
			return UF_CRCOK | UF_REDFLAG;
		}
		if ( crc_l == ecrc ) {
			*size = ofs;
			return UF_CRCOK;
		}
	}

	/* If we get here we didn't find anything. */
	return 0;
}


/** Update CRC in an env block
 *
 * Update the CRC in a provided env block.
 *
 * @param buf Buffer holding the environment block. The updated CRC
 *            is written into the buffer.
 * @param size Environment size, including CRC and optional flag field.
 * @param flag If value is between 0-255 inclusive, write redundant
 *             env type with flag value. If not (e.g. -1), write
 *             legacy env. type.
 */
void env_update_crc(uint8_t *buf, unsigned size, int flag)
{
	unsigned i;
	uint32_t crc = ~0U;
	uint8_t *b = buf;

	if (flag < 0 || flag > 255) {
		i=4;
	} else {
		i=5;
		buf[4] = flag;
	}

	b = buf + i;

	while (i < size) {
		crc32step(&crc, *b);
		i++; b++;
	}

	write_crc(buf, ~crc); /* Using the 1-complement of CRC32 */
}


/** Allocate and init an empty records structure
 *
 * @return Pointer to structure or NULL on memory error.
 */
struct env_records * env_alloc_records(void)
{
	struct env_records *er;
	er = malloc(sizeof(struct env_records));
	if (er) {
		er->begin = NULL;
		er->end = NULL;
	}
	return er;
}


/*
 * Search by name, return pointer to env record.
 *
 * @return NULL if not found.
 */
static struct env_record * env_find_record(struct env_records *er, const char *name)
{
	struct env_record *p = er->begin;
	while (p) {
		if (strcmp(p->name,name)==0) break;
		p=p->next;
	}
	return p;
}


/* Delete an env record from list. */
static void env_delete_record(struct env_records *er, struct env_record *item)
{
	if (item->next) item->next->prev = item->prev;
	else er->end = item->prev;
	if (item->prev) item->prev->next = item->next;
	else er->begin = item->next;
	free(item);
}


/* Add record to end of records list
 *
 * No checking is performed, could create a duplicate.
 *
 * @return 0 on success, -1 on out of memeory.
 */
static int env_add_record(struct env_records *er, const char *name, const char *value)
{
	struct env_record *item;
	int lh,ln,lv;

	lh = sizeof(struct env_record);
	ln = strlen(name)+1;
	lv = strlen(value)+1;

	item = malloc(lh + ln + lv);
	if (!item) return -1;

	item->name = (char*)item + lh;
	strcpy(item->name, name);

	item->value = (char*)item + lh + ln;
	strcpy(item->value, value);

	/* Insert at end. */
	if (er->end) er->end->next = item;

	item->next = NULL;
	item->prev = er->end;

	er->end = item;
	if (!er->begin) er->begin = item;
	return 0;
}

/* Delete all env records and free data structure.
 */
static void env_delete_all_records(struct env_records *er)
{
	struct env_record *item;
	while (er->begin) {
		item = er->begin;
		er->begin = item->next;
		free(item);
	}
	er->begin = er->end = NULL;
	free(er);
}


/** Parsing an env block into a linked list of records
 *
 * The block must be valid. The elements of the returned list of records contains
 * a copy of the names & values.
 *
 * @param buf       Buffer holding the environment block.
 * @param redundant Boolean, if set assume redundant env, else legacy
 *
 * @return pointer to allocated record list
 */
struct env_records * env_parse_records(uint8_t *buf, int redundant)
{
	struct env_records *er;
	char *name;
	char *value;
	int rval;

	er = env_alloc_records();
	if (!er) return er;

	if (redundant) buf+=5;
	else buf += 4;

	while (*buf) {
		name=(char*)buf;
		while (*buf && *buf != '=') buf++;
		if (*buf != '=') continue; /* Skipping variable without value */
		*buf = '\0';
		buf++;
		value = (char*)buf;
		while (*buf) buf++;
		rval = env_add_record(er, name, value);
		value[-1] = '='; /* Restore marker */
		buf++;
		if (rval) {
			/* env_add_record failed, we're out of memeory */
			env_delete_all_records(er);
			er = NULL;
			break;
		}
	}

	return er;
}


/** Drop an entire env record structure
 *
 * @param er  Env records structure to drop
 */
void env_drop_records(struct env_records *er)
{
	env_delete_all_records(er);
}


/** Iterate over all records, in no particular order.
 *
 * @param er      Env records structure to iterate over
 * @param iterand Function to call on every record entry. If this function
 *                returns !=0, abort iteration.
 *
 * @returns 0 or the value the iterand function has returned in case of abort.
 */
int env_iterate_records(struct env_records *er, int (*iterand)(const char *name, const char *value))
{
	struct env_record *r = er->begin;
	int rval = 0;

	while (r) {
		rval = iterand(r->name, r->value);
		if (rval) break;
		r = r->next;
	}

	return rval;
}

/** Generate an env block from record structure.
 *
 * This also sets the (optional) flag field and the CRC.
 *
 * @param buf   Buffer holding the target environment block.
 * @param er    Env records to transform into env block
 * @param flag  When between 0-255 inclusive, write redundant env with this
 *              as the flag vale, otherwise (-1) write legacy env.
 * @param size  Size of environment to write. buf must be large enough.
 *
 * @return 0 on success, -1 on error (size exceeded)
 */
int env_generate(uint8_t *buf, const struct env_records *er, int flag, unsigned size)
{
	uint8_t *bp = buf+4;
	struct env_record *r = er->begin;
	char *sp;
	unsigned csize = 4;

	dbg("env_generate(buf=%p, flag=%02x, size=0x%x)\n", (void*)buf, flag, size);

	if (flag >= 0 && flag <= 255) {
		*bp++ = flag;
		csize++;
	}

	while (r && csize < size-1) {
		sp = (char*)r->name;
		while (*sp && csize < size - 1) { /* Copy name */
			*bp++ = *sp++;
			csize++;
		}
		*bp++ = '=';
		csize++;
		if (csize >= size - 1) break;
		sp = (char*)r->value;
		while (*sp && csize < size - 1) { /* Copy value */
			*bp++ = *sp++;
			csize++;
		}
		*bp++ = '\0';
		csize++;
		r = r->next;
	}

	if (csize >= size - 1) {
		/* Size exceeded */
		return -1;
	}

	/* Fill the remainder with 0 */
	while (csize < size) {
		*bp++ = '\0';
		csize++;
	}

	env_update_crc(buf, size, flag);

	return 0;
}

/** Generate an env block from record structure.
 *
 * This is identical to env_generate, but extracts the necessary parameters
 * from the uenv write candidate entry.
 *
 * @param ue    uenv structure
 * @param er    Env records to transform into env block
 *
 * @return 0 on success, -1 on error (size exceeded)
 */
int env_update(struct uenv *ue, const struct env_records *er)
{
	int flag;
	struct uenv_desc *we = ue->write_env;

	if (!we) {
		return -1;
	}

	if (we->flags & UF_REDWR) {
		/* For redundant write, the flag field must be one higher
		* than the current read field. If we don't have a read
		* field, we start at 0 */
		if (ue->read_env && ue->read_env->flags & UF_REDFLAG)
			flag = (uint8_t)(ue->read_env->flag + 1);
		else
			flag = 0;
	} else {
		/* Legacy env, no flag field. */
		flag = -1;
	}

	return env_generate(ue->buffer, er, flag, we->size);
}


/** Read value of an env record
 *
 * The returned pointer points into the record's data area. Only valid as long
 * as the record element exists.
 *
 * @param er    Record list
 * @param name  Name of variable, 0-terminated
 *
 * @return Pointer to record value or NULL if not found.
 */
const char * env_read_record(struct env_records *er, const char *name)
{
	struct env_record *r;

	r = env_find_record(er, name);

	if (r)
		return (const char *)r->value;
	else
		return NULL;
}


/** Delete env record
 *
 * @param er    Record list
 * @param name  Name of variable, 0-terminated
 *
 * @return 0 on success, 1 variable not found.
 */
int env_del_record(struct env_records *er, const char *name)
{
	struct env_record *r;

	r = env_find_record(er, name);

	if (r) {
		env_delete_record(er, r);
		return 0;
	} else {
		return 1; /* Not found */
	}
}


/** Adds / Replaces a record in the env records structure
 *
 * @param er    Record list
 * @param name  Name of variable, 0-terminated
 * @param value Value of variable, 0-terminated
 *
 * @return 0 on success, -1 on failure (OOM).
 */
int env_write_record(struct env_records *er, const char *name, const char *value)
{
	env_del_record(er, name);
	return env_add_record(er, name, value);
}


/*
 *   Environment scanning & classification
 */


/** Scan env area for all potential env blocks
 *
 * The settings in ue must be all valid.
 * Creates an env descriptor entry for every valid env and every
 * stride offset.
 *
 * @param ue Pointer to env structure.
 * @param envop Ptr to function capable of NAND operations.
 *
 * @return Number of found valid envs.
 */

int full_env_scan(struct uenv *ue, envop_fun envop )
{
	int rval=0;
	loff_t offset, foffset;
	unsigned env=0;
	unsigned scansize = ue->menvsize;
	unsigned totsize = ue->totsize;
	unsigned stride = ue->stride;
	unsigned minenv;
	uint8_t ueflags;
	unsigned size;
	unsigned blk_offs; /* Offset into erase block */
	struct uenv_desc *ud;

	/* Clean out old values */
	uenv_desc_del(ue);
	ue->read_env = NULL;
	ue->write_env = NULL;

	/* Minimal env size is not terribly relevant. We
	 * hardcode it to something sensible.  */
	minenv = 256;

	/* Stepping through env range */
	offset = 0;
	while (offset <= totsize - stride) {

		/* Offset into storage */
		foffset = ue->envbase + offset;

		/* Creat new descriptor, pre-fill contents. */
		ud = malloc(sizeof(struct uenv_desc));
		if (ud == NULL) {
			dbg("Out of memory\n");
			uenv_desc_del(ue);
			rval = -1; goto exit;
		}
		ud->next = NULL;
		ud->offset = foffset;
		ud->size   = 0;
		ud->flags  = 0;
		ud->flag   = 0;

		uenv_addlist(ue, ud);

		/* Adjust the scan size such that it doesn't cross block
		 * boundaries or exceed total size. */
		blk_offs = foffset % ue->eraseblock;
		scansize = ue->menvsize;
		if (offset + scansize > totsize)
			scansize = totsize - offset;
		if (blk_offs + scansize > ue->eraseblock)
			scansize = ue->eraseblock - blk_offs;

		rval = envop(ue->buffer, foffset, scansize, EO_READ);
		if (rval) {
			/* Error reported, we assume this is a bad block. We create a
			 * descriptor that covers the whole eraseblock. */
			ud->size = ue->eraseblock;
			ud->flags = UF_ERROR;
			offset += ue->eraseblock;
			dbg("Found bad block at (start=%lld, size=%d)\n", foffset, scansize);
			continue;
		}

		size = 0;
		ueflags = env_check_scan(ue->buffer, &size, minenv, scansize);

		/* We create an env descriptor for every stride position, but skip
		 * stride positions that fall inside a detected env. */

		ud->size   = size;
		ud->flags  = ueflags;

		if (!(ueflags & UF_CRCOK)) {
			/* Not a valid env, skip to next candidate pos. */
			ud->size = stride;
			offset += stride;
			continue;
		}

		/* Two basic env types. */
		if (ueflags & UF_REDFLAG) {
			dbg("Found redundant env at 0x%08llxx, size=0x%x\n", foffset, size);
			ud->flag = ue->buffer[4];
		} else {
			dbg("Found legacy env at 0x%08llx, size=0x%x\n", foffset, size);
			ud->flag = 0;
		}
		env++;

		/* Skipping to end of detected env. If the size would break our stride,
		 * we step by stride only and risk noise. */
		offset += ((size-1) / stride + 1) * stride;
	}
	rval = env;

exit:
	return rval;
}


/* Calculates the offset of the erase block that contains the
 * specified offset.
 *
 * @param ofs Offset into NAND
 * @param blocksize Erase block size of NAND device. Must be a
 *                  2^n, where n is an integer.
 *
 * @return Start address of erase block containing the offset
 */
static inline loff_t eraseblock(loff_t ofs, unsigned blocksize)
{
	return ofs & ~(loff_t)(blocksize - 1);
}


/** Detects which env to use for reading and writing, respectively.
 *
 * Logic:
 *
 * For reading environments, redundant envs take precedence over legacy
 * ones, unless forced (UT_FORCE). For writing, we can make automatic
 * decisions or force an override mode, depending on ue->type.
 *
 * Automatic decisions are as follows:
 *
 * Leg | Red | Read target                | Write Target (auto)
 * ----+-----+----------------------------+-------------------------------------
 *  0  |  0  | Error                      | none
 *  1  |  0  | Legacy                     | Replace Legacy
 * >1  |  0  | First Legacy               | Replace First legacy
 * any |  1  | Redundant                  | Replace Redundant
 * any |  2  | Newest Redundant           | Oldest redundant
 * any | >2  | Newest Redundant           | Free space or oldest redundant
 *
 * In case of ambiguous candidates (bad wraps, etc.), the read candidate is set
 * to a unspecified valid env entry, the write candidate is set to NULL. In such
 * a case, perform a read & scrub.
 *
 * @param ue uenv data structure
 *
 * @return 0 on success, -1 on type flag error, -2 on settings error.
 */
int env_analyse(struct uenv *ue)
{
	struct uenv_desc *ud;
	int found_red_envs = 0;
	uint8_t f;

	/* The sequence number for redundant entries uses a single byte
	 * sequence number. This means that a wrap-around of sequence
	 * numbers is likely during the product lifetime.
	 *
	 * This leads to a potential pathological case in which random
	 * sequence numbers can lead to incorrect read & write candidates.
	 * Subsequent writes can get stuck in a single location.
	 *
	 * In such ambiguous cases, it is not possible to relibly find
	 * the oldest or newest entry. However, we can detect the fact that
	 * we are in an ambiguous case and kill the write candidate, thus
	 * forcing a scrub.
	 *
	 * Theoretically, as long as all sequence number values are within
	 * the same half of the value circle, the situation is not ambiguous.
	 * However, this turns out to be quite hard to compute.
	 *
	 * The algorithm used here takes the first sequence number value
	 * and then checks if its distance to all other values is within
	 * a quarter circle (64). This ensures that all values are within
	 * a half circle (worst case is a quarter circle each side). This
	 * is not optimal and may lead to a one-off false negative, but
	 * is otherwise safe.
	 */
	int ch_first   = -1; /* Will hold the flag value of first redundant entry */
	int ch_maxdist = 0;  /* Max. found distance between ch_first and all other sequence values */

	struct uenv_desc *rdcand = NULL; /* Env read candidate */
	struct uenv_desc *wrcand = NULL; /* Env write candidate */

	/* Not all type combinations are allowed */
	switch (ue->type) {
		case UT_AUTO :                 /* Auto read & write */
		case UT_FORCE | UT_LEGACY :    /* legacy mode, forced read */
		case UT_FORCE | UT_REDUNDANT : /* redundant mode, forced read */
		case UT_FORCE | UT_NHRF :      /* nhrf mode, forced read */
		case UT_LEGACY :               /* legacy mode, auto read */
		case UT_REDUNDANT :            /* redundant mode, auto read */
		case UT_NHRF :                 /* nhrf mode, auto read */
			/* OK */
			break;
		default :
			dbg("Invalid flag combination: ue->type = 0x%02x\n", ue->type);
			return -1;
			break;
	}

	/* We may have write/erase flags from a previous run, clearing them out. */
	ud = ue->ud_start;
	while (ud) {
		ud->flags &= UF_ERROR | UF_ERASED | UF_CRCOK | UF_REDFLAG;
		ud = ud->next;
	}

	/* Searching for redundant env blocks first, unless we are forced to
	 * only read legacy blocks. */
	if (!(ue->type & UT_FORCE && ue->type & UT_LEGACY)) {
		ud = ue->ud_start;
		while (ud) {
			f = ud->flags;
			if (f & UF_CRCOK && f & UF_REDFLAG) {
				/* Find the newest / oldest candidate.*/
				dbg("[rdc=%02x, wrc=%02x, cur=%02x] =>",
					rdcand?rdcand->flag:0x100,
					wrcand?wrcand->flag:0x100,
					ud->flag );
				if (!rdcand || env_flcomp(rdcand->flag, ud->flag) <= 0) {
					dbg("RDC ");
					rdcand = ud;
				}
				if (!wrcand || env_flcomp(wrcand->flag, ud->flag) >= 0) {
					dbg("WRC ");
					wrcand = ud;
				}
				dbg("=> [rdc=%02x, wrc=%02x, cur=%02x]\n",
					rdcand?rdcand->flag:0x100,
					wrcand?wrcand->flag:0x100,
					ud->flag );
				/* Count redundant environments that are the expected size */
				if (ud->size == ue->envsize) {
					found_red_envs++;
				}
				/* Tracking maximum distance */
				{
					int d;
					if (ch_first < 0) ch_first = ud->flag;
					d = env_fldist(ch_first, ud->flag);
					if (d > ch_maxdist) ch_maxdist = d;
				}
			}
			ud = ud->next;
		}
	}

	/* Searching for legacy env blocks only if no redundant ones were found
	 * or searched for. We don't go further than the first one. */
	if (!rdcand) {
		ud = ue->ud_start;
		while (ud) {
			f = ud->flags;
			if (f & UF_CRCOK && !(f & UF_REDFLAG)) {
				/* found one, read and write here. */
				dbg("Found legacy env at %p\n", (void*)ud);
				rdcand = wrcand = ud;
				break;
			}
			ud = ud->next;
		}
	}

	dbg("Analyse: rdcand=%p (%02x), wrcand=%p (%02x)\n",
		(void*)rdcand, rdcand ? rdcand->flag:0x100,
		(void*)wrcand, wrcand?wrcand->flag:0x100);

	/* At this point, rdcand contains a candidate read location and wrcand
	 * a candidate write location. Depending on mode, the write location
	 * must be adjusted. */

	/* In auto mode, we only overwrite existing env blocks
	 * (unless we've found more than 2, in which case we do NHRF) */
	if (ue->type & UT_AUTO && found_red_envs < 3) {
		if (wrcand) {
			if ( wrcand->flags & UF_REDFLAG)
				wrcand->flags |= UF_REDWR;
			wrcand->flags |= UF_ERASE | UF_WRITE;
		}
		dbg("Dual redundant env\n");
		goto exit;
	}

	/* In legacy mode, we overwrite existing env block or create
	 * a new one at the start of the env area. */
	if (ue->type & UT_LEGACY) {
		if (!wrcand) {
			wrcand = ue->ud_start;
			wrcand->size = ue->envsize;
		}
		wrcand->flags |= UF_ERASE | UF_WRITE;
		dbg("Legacy env block\n");
		goto exit;
	}

	/* In 2x redundant mode, we write to one of two possible locations,
	 * envbase or envbase+stride. */
	if (ue->type & UT_REDUNDANT) {
		/* No candiate? -> write to first location */
		if (!wrcand) {
			wrcand = ue->ud_start;
			wrcand->size = ue->envsize;
		}
		/* Scan only found one env? -> use alternate location */
		if (wrcand == rdcand) {
			/* Finding the first location which is not the
			 * read location. */
			ud = ue->ud_start;
			if (ud == rdcand) ud = ud->next;
			wrcand = ud;
			wrcand->size = ue->envsize;
		}
		wrcand->flags |= UF_REDWR | UF_ERASE | UF_WRITE;
		dbg("Dual redundant env\n");
		goto exit;
	}

	/* If we reach this point we are asked to write (or have detected) a
	 * Netcomm env structure.
	 *
	 * = stride may be less than eraseblock, we can write more than one env
	 *   block into one erase block. That is, we don't always have to erase.
	 *
	 * = There can be more than 2 eraseblocks in the env area.
	 */

	/* Sanity checking the settings */
	if ( ue->totsize / ue->stride > 64) {
		/* We have to limit the maximum number of env entries to avoid
		 * sequence number ambiguity. */
		dbg("Exeeded maximum number of prossible env blocks (%d>64)\n",
			ue->totsize / ue->stride);
		return -2;
	}

	/* Find an empty write location. If found, we can write to it, without
	 * erase. */
	ud = ue->ud_start;
	while (ud) {
		if ( ud->flags & UF_ERASED ) {
			/* Yay! */
			wrcand = ud;
			wrcand->size = ue->envsize;
			wrcand->flags |= UF_REDWR | UF_WRITE;
			dbg("NHRF: Found empty location to write at 0x%llx\n", wrcand->offset);
			goto exit;
		}
		ud = ud->next;
	}

	/* Garbage collection. We may have entries that are not erased,
	 * but contain either random data or legacy environments. We reclaim
	 * them here, but making sure we are not sharing an erase block
	 * with the current read candidate. */
	if (rdcand) {
		loff_t read_eb = eraseblock(rdcand->offset, ue->eraseblock);
		loff_t c_eb;

		ud = ue->ud_start;
		while (ud) {
			c_eb = eraseblock(ud->offset, ue->eraseblock);
			if (c_eb != read_eb && (ud->flags == UF_CRCOK || ud->flags == 0)) {
				wrcand = ud;
				wrcand->size = ue->envsize;
				/* We delete any CRCOK flag here - there is no difference
				 * between legacy env and noise at this point. */
				wrcand->flags = UF_ERASE | UF_WRITE | UF_REDWR;
				dbg("NHRF: Garbage collected at 0x%llx\n", wrcand->offset);
				goto exit;
			}
			ud = ud->next;
		}
	}

	/* Out of empty locations, force an erase on the block of the
	 * current write candidate (oldest redundant entry)  */
	if (wrcand) wrcand->flags |= UF_ERASE | UF_WRITE | UF_REDWR;
	dbg("NHRF: Forced erase\n");

exit:

	ue->read_env = rdcand;
	ue->write_env = wrcand;

	if ( ch_maxdist > 63 ) {
		dbg("Ambiguous sequence numbers (d=%d), cancelling write candidate.\n", ch_maxdist);
		ue->write_env = NULL;
	}

	return 0;
}

/** Temporarily mark a entry in the scan structure as bad.
 *
 * This is intended to remove an entry from the next run of env_analyze, in case
 * we don't have a markbad implementation. This marker only exists until the next
 * full_env_scan().
 *
 * @param ue    uenv data structure
 * @param start Start address. The exact extends of the affected area depend
 *              on the item's size in the structure.
 *
 * @return 0 on success. -1 on error (no valid write candidate).
 */
int env_disable(struct uenv *ue)
{
	if (!ue->write_env) return -1;
	ue->write_env->flags = UF_ERROR;
	return 0;
}


/** Write entire environment area from current buffer
 * 
 * This writes the current buffer into all possible env locations. It's
 * intended to be used for error recovery and format conversion. Uses
 * ue->envsize.
 *
 * NOTE: The resulting env locations will all have the same sequence
 * number. This saves a little time.
 *
 * @param ue    uenv data structure
 * @param envop Ptr to function capable of NAND operations.
 * @param flag  When between 0-255 inclusive, write redundant env with this
 *              as the sequence number, otherwise (-1) write legacy header.
 *
 * @return 0 on success, -1 on error (all writes have failed)
 */
int env_scrub(const struct uenv *ue, envop_fun envop, int flag)
{
	unsigned i, j, envs_per_block;
	loff_t offset;
	int rval;
	int ok = 0; /* Counts the number of successful writes/verifies */

	/* We could specifically skip the block of the current read candidate
	 * and write that last, but the small safety benefit is not worth the
	 * effort. This function is not intended to be used a lot.
	 *
	 * NOTE: 
	 */

	if ( ue->stride && ue->stride < ue->eraseblock ) {
		envs_per_block = ue->eraseblock / ue->stride;
	} else {
		envs_per_block = 1;
	}

	dbg("scrubbing: envs/block=%u, flag=%d\n", envs_per_block, flag);

	env_update_crc(ue->buffer, ue->envsize, flag);

	/* Loop over all erase blocks */
	for (i = 0; i < ue->totsize; i += ue->eraseblock) {
		offset = ue->envbase + i;
		envop(ue->buffer, offset, ue->eraseblock, EO_ERASE);
		/* Loop over all env storage locations within page */
		for (j = 0; j < envs_per_block; j ++) {
			offset = ue->envbase + i + j * ue->stride;
			rval = envop(ue->buffer, offset, ue->envsize, EO_WRITE | EO_VERIFY);
			if (rval == 0) ok++;
		}
	}

	/* Return error if not a single write was successful. */
	if (ok) return 0;
	else return -1;
}


/** Read current env block into uenv internal buffer
 *
 * @param ue Uenv structure, contains buffer & info about read source.
 * @param envop Ptr to function capable of NAND operations.
 */
int env_fetch(struct uenv *ue, envop_fun envop )
{
	int rval;

	if (ue->read_env) {
		/* We set the buffer to all 0, in case we need to increase env size later */
		memset(ue->buffer, 0, ue->menvsize);
		rval = envop(ue->buffer, ue->read_env->offset, ue->read_env->size, EO_READ);
	} else {
		rval = -1;
	}

	return rval;
}

/** Write an env block back to storage 
 *
 * Writes the env in the current buffer back to storage.
 *
 * @param ue Uenv structure, contains buffer & info about write target.
 * @param envop Ptr to function capable of NAND operations.
 */
int env_store(struct uenv *ue, envop_fun envop )
{
	int rval = 0;
	unsigned flags;
	struct uenv_desc *we = ue->write_env;

	if (!we) {
		rval = -1;
		goto exit;
	}

	/* Attempting to write */
	flags = 0;
	if (we->flags & UF_ERASE)
		flags |= EO_ERASE;

	/* We (optionally) erase, write and verify. */
	flags |= EO_WRITE | EO_VERIFY;

	rval = envop(ue->buffer, we->offset, we->size, flags);

	/* We attempt a forced erase and try again. */
	if (rval) {
		rval = envop(ue->buffer, we->offset, we->size, flags | EO_ERASE);
	}

	/* Losing hope, marking bad. */
	if (rval) {
		rval = envop(ue->buffer, we->offset, we->size, EO_MRKBD);
		/* Mark local data structure. To write to the next location,
		 * re-analyse. This also works if the MARKBAD envop is unimplemented. */
		env_disable(ue);
		/* We return error if we had to mark bad */
		rval = -1;
	}

exit:
	return rval;
}

#ifdef TEST
/* Regression test, build this file standalone with
 * gcc -std=c99 -Wall -pedantic -DTEST uenv_alg.c -o uenv_alg_test
 */

#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdarg.h>

static uint8_t *buffer; /* Current env image */
static size_t flen=0;

/*
 * Calculates a CRC32 over the buffer, we use this to check
 * results in a generic way.
 */
static int buffer_check(uint32_t crc)
{
	uint32_t c;
	size_t i;
	uint8_t *bp = buffer;

	c = ~0U;
	for (i = 0; i < flen; i++) {
		crc32step(&c, *bp++);
	}
	c = ~c;

	if (crc != c) {
		printf("Image CRC failed, expected 0x%08x, got 0x%08x\n", crc, c);
		return -1;
	} else {
		return 0;
	}
}

/*
 * MTD/NAND operations simulation
 */
static int test_envop(uint8_t *buf, loff_t start, unsigned size, unsigned flags)
{
	unsigned i;
	loff_t blk_base;
	unsigned blk_offs;
	/* global: buffer */
	#define EBLOCK_SIZE (128*1024)

	/* This is the erase block belonging to 'start' */
	blk_base = (start / EBLOCK_SIZE) * EBLOCK_SIZE;
	blk_offs = (start % EBLOCK_SIZE);

	/*
	printf("envop(buf=%p, start=0x%llx, size=0x%x, flags=%s%s%s%s%s)\n",
		(void*)buf, start, size,
		(flags & EO_ERASE)?"ERASE ":"", (flags & EO_WRITE)?"WRITE ":"",
		(flags & EO_VERIFY)?"VERIFY ":"", (flags & EO_READ)?"READ ":"",
		(flags & EO_MRKBD)?"EO_MRKBD ":"");
	*/

	if (start+size > flen) {
		printf("	envop Access (start=0x%llx, size=0x%x) out of bounds (flen=0x%x)\n",
			start, size, flen );
		return -2;
	}

	if (blk_offs+size > EBLOCK_SIZE) {
		printf("	envop Access accross block boundary (0x%x + 0x%x)\n", blk_offs, size);
		return -3;
	}

	if (strcmp("BADBLOCK", (char*)&buffer[blk_base])==0) {
		printf("	envop Bad block @ 0x%llx\n", blk_base);
		return -1;
	}

	if (flags & EO_ERASE) {
		printf("	envop Erasing 0x%llx|0x%x\n", blk_base, EBLOCK_SIZE);
		/* Simulating erase by writing all '1' to buffer */
		for (i = 0; i < EBLOCK_SIZE; i++) {
			buffer[blk_base + i] = 0xFF;
		}
	}

	if (flags & EO_WRITE) {
		unsigned warn = 0;
		printf("	envop Writing 0x%llx|0x%x\n", start, size);
		/* Simulating write by writing a logical-AND */
		for (i = 0; i < size; i++) {
			if (buffer[start+i] != 0xff) warn++;
			buffer[start+i] &= buf[i];
		}
		if (warn) {
			printf("	envop Found %d non-0xFF bytes\n", warn);
		}
		
	}

	if (flags & EO_VERIFY) {
		printf("	envop Verify 0x%llx|0x%x\n", start, size);
		/* Comparing buffer & buf */
		for (i = 0; i < size; i++) {
			if (buffer[start+i] != buf[i]) {
				printf("	envop Verify error (%02x,%02x)\n", buffer[start+i], buf[i]);
				return -1;
			}
		}
	}

	if (flags & EO_READ) {
		//printf("	Reading 0x%llx|0x%x\n", start, size);
		for (i = 0; i < size; i++) {
			buf[i] = buffer[start+i];
		}
	}

	if (flags & EO_MRKBD) {
		printf("	envop Marking BAD 0x%llx|0x%x\n", blk_base, EBLOCK_SIZE);
		/* Marking block as bad. */
		memset(&buffer[blk_base], 0, EBLOCK_SIZE);
		strcpy((char*)&buffer[blk_base], "BADBLOCK");
	}

	return 0;
}
static int test_envop_error(uint8_t *buf, loff_t start, unsigned size, unsigned flags)
{
	return -3;
}


/*
 * File access, read from file. Length 0 means read to file end
 * and return read bytes. if file is to short for length, return
 * NULL. Free when not needed anymore.
 */
static uint8_t * readfile(const char *name, loff_t offset, size_t *length)
{
	uint8_t *rval = NULL;
	uint8_t *rptr;
	struct stat sb;
	int err;
	int fd;
	ssize_t rsize;
	size_t s;

	err = stat(name, &sb);
	if (err) {
		printf("Can't stat file '%s'\n", name);
		return rval;
	}

	if (sb.st_size < offset + *length) {
		printf("File '%s' is not long enough for offset=%lld, length=%d\n",
			name, offset, *length);
	}

	if (*length == 0) *length = sb.st_size - offset;

	rval = malloc(*length);
	rptr = rval;

	if (rval) {
		fd = open(name, O_RDONLY);
		if (fd<0) {
			perror("Failed to open file\n");
			goto error;
		}
		s = *length;
		do {
			rsize = read(fd, rptr, s);
			if (rsize < 0) {
				perror("Failed to read file\n");
				goto error;
			}
			s -= rsize;
			rptr += rsize;
		} while(rsize!=0 && s != 0);
		close(fd);
	} else {
		printf("Can't allocate %d bytes\n", *length);
	}

	printf("Reading '%s', ofs=%lld, len=%d\n", name, offset, *length);

	return rval;
error:
	if (rval) free(rval);
	return NULL;
}

/* Use to dump buffer for external analysis */
void writefile(const char * name)
{
	int fd;
	ssize_t wsize;
	size_t s = flen;
	uint8_t *wptr = buffer;

	printf("Dumping buffer to file: '%s'\n", name);
	fd = open(name, O_WRONLY | O_CREAT, 0644);
	if (fd<0) {
		perror("Failed to open file\n");
		return;
	}

	do {
		wsize = write(fd, wptr, s);
		if (wsize < 0) {
			perror("Failed to read file\n");
			close(fd);
			return;
		}
		s -= wsize;
		wptr += wsize;
	} while(wsize!=0 && s != 0);

	close(fd);
}

static unsigned iterate = 0;
static int test_iterand(const char *name, const char *value)
{
	if (!iterate) return -47;
	iterate--;
	printf("	%s=%s\n", name, value);
	return 0;
}

static int errpr(const char *fmt, ...)
{
	va_list ap;
	int rval;
	va_start(ap, fmt);
	rval = vfprintf(stderr, fmt, ap);
	va_end(ap);
	return rval;
}

static unsigned quiet = 0; /* Set this temporarily to 1 to suppress OK messages */

/* Assert condition, print message if failed. */
#define CHECK(cond,...) do{ \
	if (cond) { \
		if (!quiet) printf("  [%d] => OK (" #cond ")\n", __LINE__); \
	} else { \
		printf("  [%d] ", __LINE__); \
		printf("FAILED (" #cond "): " __VA_ARGS__); \
		uenv_dump(ue, errpr); \
		errors++; \
	} }while(0)


/* Scans/analyses/parses/(over)writes/generates/writeback.  */
int complete_write_cycle(struct uenv *ue, const char *name, const char *value)
{
	struct env_records *er;
	int rint;
	unsigned errors=0;

	printf("complete_write_cycle: '%s'='%s'\n", name, value);

	quiet = 1;

	rint = full_env_scan(ue, test_envop);
	CHECK(rint>=0, "full_env_scan returned %d.\n", rint);
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	rint = env_fetch(ue, test_envop);
	CHECK(rint==0, "Unexpected error %d\n", rint);
	er = env_parse_records(ue->buffer, ue->read_env && ue->read_env->flags & UF_REDFLAG);
	CHECK(er!=NULL, "env_parse_records failed\n");
	rint = env_write_record(er, name, value);
	CHECK(rint==0, "Unexpected failure to create\n");
	rint = env_update(ue, er);
	CHECK(rint==0, "Env encoding returned error %d\n", rint);
	rint = env_store(ue, test_envop);
	CHECK(rint==0, "Saving env failed");
	env_drop_records(er);

	quiet = 0;

	return errors;
}

void complete_rescan_dump(struct uenv *ue)
{
	full_env_scan(ue, test_envop);
	env_analyse(ue);
	uenv_dump(ue, errpr);
}

int main(void)
{
	unsigned errors=0;
	uint8_t rv;
	int rint;
	const char *rchr;
	unsigned size;
	size=0;
	struct uenv *ue;
	struct env_records *er;

	#define STRIDE 2048
	#define MAX_ENV (128*1024)
	printf("Allocating env structure, max size %d...\n", MAX_ENV);
	ue = uenv_init(MAX_ENV);
	uenv_set_autoscan(ue, 0, STRIDE, MAX_ENV);
	CHECK(ue!=NULL, "uenv_init failed, NULL returned.\n");

	/* Tests on env_flcomp(fA, fB)
	 * Should be <0 when fA is older than fB, =0 when equal, >0 when fA is newer than fB.
	 */
	printf("env_flcomp() tests...\n");
	#define CHECKFLCOMP(a, b, exp) \
	CHECK(env_flcomp(a, b) exp, "env_flcomp(" #a ", " #b ") returned %d\n", env_flcomp(a, b) )
	CHECKFLCOMP(0, 0, ==0);
	CHECKFLCOMP(1, 1, ==0);
	CHECKFLCOMP(127, 127, ==0);
	CHECKFLCOMP(0xFF, 0xFF, ==0);
	CHECKFLCOMP(0, 1, <0);
	CHECKFLCOMP(0, 2, <0);
	CHECKFLCOMP(0, 127, <0);
	CHECKFLCOMP(1, 2, <0);
	CHECKFLCOMP(1, 127, <0);
	CHECKFLCOMP(127, 128, <0);
	CHECKFLCOMP(127, 129, <0);
	CHECKFLCOMP(128, 129, <0);
	CHECKFLCOMP(128, 255, <0);
	CHECKFLCOMP(129, 0, <0);
	CHECKFLCOMP(255, 0, <0);
	CHECKFLCOMP(255, 1, <0);
	CHECKFLCOMP(255, 30, <0);
	CHECKFLCOMP(255, 125, <0);
	CHECKFLCOMP(1, 0, >0);
	CHECKFLCOMP(2, 0, >0);
	CHECKFLCOMP(127, 0, >0);
	CHECKFLCOMP(2, 1, >0);
	CHECKFLCOMP(127, 1, >0);
	CHECKFLCOMP(128, 127, >0);
	CHECKFLCOMP(129, 127, >0);
	CHECKFLCOMP(129, 128, >0);
	CHECKFLCOMP(129, 128, >0);
	CHECKFLCOMP(255, 128, >0);
	CHECKFLCOMP(0, 129, >0);
	CHECKFLCOMP(0, 255, >0);
	CHECKFLCOMP(1, 255, >0);
	CHECKFLCOMP(30, 255, >0);
	CHECKFLCOMP(125, 255, >0);

	/* Tests env_fldist(fA, fB) */
	#define CHECKFLDIST(a, b, exp) \
	CHECK(env_fldist(a, b) exp, "env_fldist(" #a ", " #b ") returned %d\n", env_fldist(a, b) )
	CHECKFLDIST(3, 2, == 1); /* Trivial cases... */
	CHECKFLDIST(3, 3, == 0);
	CHECKFLDIST(3, 4, == 1);
	CHECKFLDIST(0, 128, == 128); /* Extreme distance... */
	CHECKFLDIST(64, 192, == 128);
	CHECKFLDIST(192, 64, == 128);
	CHECKFLDIST(224, 32, == 64); /* Crossing wrap point */
	CHECKFLDIST(32, 224, == 64);

	/* Tests on random data - error handling */
	flen=0;
	buffer = readfile("img_random.bin", 0, &flen);
	ue->totsize = flen;

	CHECK(buffer_check(0x192819ae)==0, "Input file changed\n");

	printf("Checking random 'environment'...\n");
	rv = env_check(buffer, flen);
	CHECK(rv==0, "Unexpected success: rv=0x%02x\n", rv);

	printf("Scanning random 'environment'...\n");
	rv = env_check_scan(buffer, &size, 16, flen);
	CHECK(rv==0, "Unexpected success: rv=0x%02x, size=%d\n", rv, size);

	printf("Searching random 'environment'...\n");
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==0, "full_env_scan returned %d for random env.\n", rint);

	printf("Analysing random 'environment'...\n");
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	CHECK(ue->read_env==NULL && ue->write_env==NULL, "Unexpected read or write candidate from random env\n");

	/* This shouldn't normally be called if CRC fails */
	printf("Checking record conversion error handling...\n");
	er = env_parse_records(ue->buffer, 1);
	CHECK(er != NULL, "Failed to allocate records structure\n");
	CHECK(er->begin == NULL, "Unexpected content\n");
	env_drop_records(er);

	printf("Checking search error handling...\n");
	rint = full_env_scan(ue, test_envop_error);
	CHECK(rint==0, "full_env_scan found something despite envop errors.\n");

	free(buffer);

	/* Tests on a large block with one legacy env */
	flen=0;
	buffer = readfile("img_oldstyle_partition.bin", 0, &flen);
	ue->totsize = flen;

	CHECK(buffer_check(0xdb6730f2)==0, "Input file changed\n");

	printf("Checking legacy environemnt...\n");
	rv = env_check(&buffer[0x260000], 0x2000);
	CHECK(rv!=0, "Unexpected failure: rv=0x%02x\n", rv);

	printf("Scanning legacy environment...\n");
	rv = env_check_scan(&buffer[0x260000], &size, 16, flen-0x260000);
	CHECK(rv!=0, "Unexpected failure: rv=0x%02x, size=%d\n", rv, size);

	printf("Searching legacy environment...\n");
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==1, "full_env_scan returned %d for legacy env.\n", rint);

	printf("Analysing legacy environment...\n");
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	CHECK(ue->read_env!=NULL && ue->write_env!=NULL, "Failed to produce read or write candidate.\n");

	printf("Reading read candidate\n");
	rint = env_fetch(ue, test_envop);
	CHECK(rint==0, "Unexpected error %d\n", rint);

	printf("Parsing environment...\n");
	er = env_parse_records(ue->buffer, 0);
	CHECK(er!=NULL, "Failed to allocate records structure\n");
	CHECK(er->begin != NULL, "Expected content\n");

	printf("Dumping first 10 environment records...\n");
	iterate = 10;
	rint = env_iterate_records(er, test_iterand);
	CHECK(rint==-47, "env_iterate_records didn't return iterand error: %d\n", rint);

	printf("Reading env entry 'POSTFLASH'...\n");
	rchr = env_read_record(er, "POSTFLASH");
	CHECK(rchr!=NULL, "Expected to find entry\n");
	printf("	Entry: '%s'\n", rchr);

	printf("Overwriting env entry 'snextra'...\n");
	rint = env_write_record(er, "snextra", "Test Value");
	CHECK(rint==0, "Unexpected failure to overwrite\n");

	printf("Creating new entry 'testvar'...\n");
	rint = env_write_record(er, "testvar", "Testvar Value");
	CHECK(rint==0, "Unexpected failure to create\n");

	printf("Deleting env entry 'POSTFLASH'...\n");
	rint = env_del_record(er, "POSTFLASH");
	CHECK(rint==0, "Expected to find and delete entry\n");

	printf("Reading (deleted) env entry 'POSTFLASH'...\n");
	rchr = env_read_record(er, "POSTFLASH");
	CHECK(rchr==NULL, "Entry wasn't deleted\n");

	printf("Reading env entry 'snextra'...\n");
	rchr = env_read_record(er, "snextra");
	CHECK(rchr!=NULL, "Expected to find entry\n");
	printf("	Entry: '%s'\n", rchr);

	printf("Reading env entry 'testvar'...\n");
	rchr = env_read_record(er, "testvar");
	CHECK(rchr!=NULL, "Expected to find entry\n");
	printf("	Entry: '%s'\n", rchr);

	printf("Re-encoding env, testing for overflow...\n");
	rint = env_generate(ue->buffer, er, -1, 100); /* Not enough space for env */
	CHECK(rint==-1, "Unexpected success: %d\n", rint);

	printf("Re-encoding env...\n");
	rint = env_update(ue, er);
	CHECK(rint==0, "Env encoding returned error %d\n", rint);

	printf("Saving env...\n");
	rint = env_store(ue, test_envop);
	CHECK(rint==0, "Saving env failed");

	printf("Checking encoded env...\n");
	CHECK(buffer_check(0x686efda3)==0, "Bad env_update or env_store result\n");

	env_drop_records(er); er=NULL;

	//writefile("test.bin");

	free(buffer);

	/* Tests on a redundant env, two copies */
	flen=0;
	buffer = readfile("img_betzy.bin", 0, &flen);
	ue->totsize = flen;

	CHECK(buffer_check(0xcf84bf34)==0, "Input file changed\n");

	printf("Searching redundant environment...\n");
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==2, "full_env_scan returned %d for redundant env.\n", rint);

	printf("Analysing redundant environment...\n");
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	CHECK(ue->read_env!=NULL && ue->write_env!=NULL, "Failed to produce read or write candidate.\n");

	printf("Creating empty env records...\n");
	er = env_alloc_records();
	CHECK(er!=NULL, "\n");

	printf("Record reading failure test...\n");
	rchr = env_read_record(er, "nosuch");
	CHECK(rchr==NULL, "Shouldn't have found anything\n");

	printf("Record deleting failure test...\n");
	rint = env_del_record(er, "nosuch");
	CHECK(rint==1, "Returned %d\n", rint);

	printf("Adding a few records...\n");
	rint = env_write_record(er, "name1", "value1");
	CHECK(rint==0, "Returned %d\n", rint);
	rint = env_write_record(er, "name1", "over1");
	CHECK(rint==0, "Returned %d\n", rint);
	rint = env_write_record(er, "name2", "value2");
	CHECK(rint==0, "Returned %d\n", rint);
	rint = env_write_record(er, "name3", "value3");
	CHECK(rint==0, "Returned %d\n", rint);

	printf("Updating environment...\n");
	rint = env_update(ue, er);
	CHECK(rint==0, "Returned %d\n", rint);

	printf("Writing redundant...\n");
	rint = env_store(ue, test_envop);
	CHECK(rint==0, "Returned %d\n", rint);

	env_drop_records(er);

	//writefile("test.bin");
	CHECK(buffer_check(0x90dcf9a3)==0, "Redundant write failed\n");

	printf("Re-scanning/analysing/reading/parsing redundant...\n");
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==2, "full_env_scan returned %d for redundant env.\n", rint);
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	rint = env_fetch(ue, test_envop);
	CHECK(rint==0, "Unexpected error %d\n", rint);
	er = env_parse_records(ue->buffer, 1);
	CHECK(er!=NULL, "Failed to allocate records structure\n");
	CHECK(er->begin != NULL, "Expected content\n");

	printf("Record readback verify...\n");
	rchr = env_read_record(er, "name1");
	CHECK(rchr && strcmp("over1",rchr)==0, "Unexpected read value\n");
	rchr = env_read_record(er, "name2");
	CHECK(rchr && strcmp("value2",rchr)==0, "Unexpected read value\n");
	rchr = env_read_record(er, "name3");
	CHECK(rchr && strcmp("value3",rchr)==0, "Unexpected read value\n");

	env_drop_records(er);

	free(buffer);

	/* Tests on a hybrid env, two copies redundant, one legacy */
	flen=0;
	buffer = readfile("img_mix.bin", 0, &flen);
	ue->totsize = flen;
	ue->stride = 0x2000;

	CHECK(buffer_check(0xd4be6432)==0, "Input file changed\n");

	printf("Checking mixed environemnt...\n");
	rv = env_check_scan(&buffer[0x80000], &size, 16, flen - 0x80000);
	CHECK(rv!=0, "Unexpected failure: rv=0x%02x\n", rv);
	printf("size=0x%x\n", size);

	printf("Searching mixed environment...\n");
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==3, "full_env_scan returned %d for redundant env.\n", rint);

	printf("Analysing mixed environment...\n");
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	CHECK(ue->read_env && ue->read_env->flag == 0x0d, "Bad read candidate.\n");
	CHECK(ue->write_env && ue->write_env->flag == 0x0c, "Bad write candidate.\n");

	/* Using the same hybrid env, we test analysis and write of Netcomm env */

	printf("Analysing Netcomm environment...\n");
	ue->type = UT_NHRF;
	ue->envsize = 0x1000;  /* Reduced write size, to check if this works. */
	ue->stride = 0x10000;  /* Stride adjusted to place two envs per erase block. */
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==3, "full_env_scan returned %d for redundant env.\n", rint);
	rint = env_analyse(ue);
	CHECK(rint==0, "Bad flags\n");
	CHECK(ue->read_env && ue->write_env,
		"Failed to produce read or write candidate.\n");
	CHECK(ue->write_env->flags == (UF_ERASED | UF_WRITE | UF_REDWR),
		"write candidate has incorrect flags.\n");
	CHECK(ue->write_env->size == 0x1000,
		"incorrect write size\n");

	complete_rescan_dump(ue);

	errors += complete_write_cycle(ue, "test1", "value1");
	errors += complete_write_cycle(ue, "filla", "vala");
	errors += complete_write_cycle(ue, "fillb", "valb");
	errors += complete_write_cycle(ue, "fillc", "valc");
	errors += complete_write_cycle(ue, "test2", "value2");

	complete_rescan_dump(ue);

	errors += complete_write_cycle(ue, "test1", "over1");

	complete_rescan_dump(ue);

	errors += complete_write_cycle(ue, "test3", "value3");

	complete_rescan_dump(ue);

	errors += complete_write_cycle(ue, "test4", "value4");

	complete_rescan_dump(ue);


	printf("Testing scrubbing\n");

	full_env_scan(ue, test_envop);
	env_analyse(ue);
	env_fetch(ue, test_envop);

	env_scrub(ue, test_envop, 0);
	rint = full_env_scan(ue, test_envop);
	CHECK(rint==10, "Unexpected layout after scrub.\n");

	printf("Testing env_disable\n");
	env_analyse(ue);
	uenv_dump(ue, errpr);

	CHECK(env_disable(ue)==0, "env_disable failed\n");
	env_analyse(ue);
	CHECK(env_disable(ue)==0, "env_disable failed\n");
	env_analyse(ue);

	env_analyse(ue);
	uenv_dump(ue, errpr);

	free(buffer);

	printf("Checking ambiguous sequence handling...\n");
	flen=0;
	buffer = readfile("img_sequencebug.bin", 0, &flen);
	buffer[0x000f8004] = 0x80; /* Inserting bad sequence number */
	ue->totsize = 0x100000; /* 1MB */
	ue->envsize = 0x1000;   /* 4K */
	ue->stride  = 0x4000;   /* 4 per eraseblock */
	CHECK(buffer_check(0x57213791)==0, "Input file changed\n");
	full_env_scan(ue, test_envop);
	env_analyse(ue);
	CHECK(ue->read_env && ue->read_env->flag == 0x80, "Unexpected read candidate\n");
	CHECK(ue->write_env == NULL, "Write candate should have been NULL\n");
	/* Scrubbing */
	{
		int f = ue->read_env->flags & UF_REDFLAG ? ue->read_env->flag : -1;
		CHECK(env_fetch(ue, test_envop)==0, "Reading failed\n");
		CHECK(env_scrub(ue, test_envop, f)==0, "Scrubbing failed\n");
	}
	CHECK(full_env_scan(ue, test_envop)==64, "Incorrect number of envs found\n");
	CHECK(buffer_check(0x1c0beb72)==0, "Input file changed\n");
	//complete_rescan_dump(ue);
	//writefile("test.bin");

	free(buffer);

	printf("Freeing descriptors...\n");
	uenv_free(ue);
	if (errors) {
		printf("\nENV Regression Test: failed, %d Errors.\n\n", errors);
		return -1;
	} else {
		printf("\nENV Regression Test: all tests succeeded\n\n");
		return 0;
	}

	
}

#endif /* TEST */
