/*
 * Netcomm High Reliability environment storage in NAND flash.
 *
 * This uses the usual U-Boot redundant environment format
 * (CRC32 flag data), but supports an arbitrary number of
 * erase blocks, multiple writes per block and can handle
 * bad blocks.
 *
 * Like the original env_nand driver, only the first nand
 * chip is supported for environment storage.
 *
 * We don't support environment blocks larger than eraseblocks,
 * or even env blocks that straddle an eraseblock boundary.
 *
 * 2012 Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <command.h>
#include <environment.h>
#include <linux/stddef.h>
#include <malloc.h>
#include <nand.h>
#include <search.h>
#include <errno.h>

#include "uenv_alg.h"

//#define DEBUG
#ifdef DEBUG
	#define msg(fmt, ...) printf("%s:%d: " fmt, __func__, __LINE__, ##__VA_ARGS__)
#else
	#define msg(fmt, ...) printf("ENV_REL: " fmt, ##__VA_ARGS__)
#endif

/* references to names in env_common.c */
extern uchar default_environment[];

char *env_name_spec = "NAND";

#if defined(ENV_IS_EMBEDDED)
extern uchar environment[];
env_t *env_ptr = (env_t *)(&environment[0]);
#elif defined(CONFIG_NAND_ENV_DST)
env_t *env_ptr = (env_t *)CONFIG_NAND_ENV_DST;
#else /* ! ENV_IS_EMBEDDED */
env_t *env_ptr = 0;
#endif /* ENV_IS_EMBEDDED */

DECLARE_GLOBAL_DATA_PTR;

uchar env_get_char_spec (int index)
{
	return ( *((uchar *)(gd->env_addr + index)) );
}


struct uenv *ue = NULL;

/* Erase block cache, write through. */
static uint8_t *cache = NULL; /* Cached data. */
static loff_t  cblock = -1;   /* Current eraseblock in cache */

/* We're paranoid and make sure we get at least *one* successful
 * MTD read, before we use the default environment. Otherwise we
 * may overwrite a perfectly good env with the default one later.
 *
 * This variable counts the number of successful reads: */
static int successful_reads = 0;

/* NAND access function for use by uenv_alg library. The order in which
 * the different commands in flags are executed is important.
 *
 * We use erase block caching to speed up scanning for environments, which
 * steps through each eraseblock, but never stradles an erase block boundary.
 */
static int envop(uint8_t *buf, loff_t start, unsigned size, unsigned flags)
{
	loff_t block_base;
	unsigned block_offset;
	unsigned block_size = nand_info[0].erasesize;
	unsigned page_size = nand_info[0].writesize;
	unsigned pageno, pages;

	int rval;
	size_t len;

	/* Sanity check, start & size must align with page boundary. */
	if (size & (page_size - 1) || start & (page_size - 1)) {
		msg("Alignment error (envop): start=%llx, size=%x\n", start, size);
		return -1;
	}

	/* This is the erase block belonging to 'start' */
	block_base   = (start / block_size) * block_size;
	block_offset = start % block_size;

	/* Page number within block, number of pages requested. */
	pageno = block_offset / page_size;
	pages  = size / page_size;
	(void)pageno;
	(void)pages;

	/* Blocks marked as bad must not be touched */
	if (nand_block_isbad(&nand_info[0], start)) {
		msg("Block is bad (%llx | %x)\n", start, size);
		return -1;
	}

	/* Invalidate & reload the cache if we have a cache miss */
	if (cblock != block_base) {
		//msg("Cache fill 0x%llx | 0x%x\n", block_base, block_size);
		memset(cache, 0x0, block_size);
		len = block_size;
		rval = nand_read(&nand_info[0], block_base, &len, cache);
		if (rval) {
			/* At this point the whole eraseblock read has failed.
			 * We try again, but reading one page at a time. */
			unsigned po, pno;
			msg("Read error %d, trying individual pages\n", rval);
			po=0; pno = 0;
			while (po < block_size) {
				len = page_size;
				rval = nand_read(&nand_info[0], block_base + po, &len, &cache[po]);
				if (rval) msg("  page @ %x: %d\n", po, rval);
				else successful_reads++;
				po += page_size;
				pno++;
			}
		} else {
			successful_reads++;
		}
		cblock = block_base;
	}

	/* Erasing whole block and get the cache to reflect that. We avoid
	 * reading after erase since an erased page may have a few bit errors
	 * which can't be corrected because the ECC bytes don't exist yet.
	 */
	if (flags & EO_ERASE) {
		msg("erase %llx | %x\n", block_base, block_size);
		rval = nand_erase(&nand_info[0], block_base, block_size);
		memset(cache, 0xFF, block_size);
	}

	/* Writing. We write straight to flash and then re-read to update
	 * the cache. This allows us to catch bad pages. */
	if (flags & EO_WRITE) {
		msg("write %llx | %x\n", start, size);
		len = size;
		rval = nand_write(&nand_info[0], start, &len, buf);
		if (rval) {
			msg("Write failed\n");
			return -1;
		}
		/* Read back to update cache. */
		len = size;
		rval = nand_read(&nand_info[0], start, &len, &cache[block_offset]);
		if (rval) {
			msg("Readback failed\n");
			return -1;
		}
	}

	/* We verify against the cache. */
	if (flags & EO_VERIFY) {
		msg("verify %llx | %x\n", start, size);
		if (memcmp(&cache[block_offset], buf, size) != 0) {
			msg("Verify failed\n");
			return -1;
		}
	}

	/* Reading from the cache, but returning errors where appropriate. */
	if (flags & EO_READ) {
		memcpy(buf, &cache[block_offset], size);
	}

	/* I'm too scared to implement this. This could give a bug the power to
	 * mark the whole environment area as bad and brick the device. The algorithm
	 * already skips existing bad blocks and should be perfectly capable of handling
	 * errors without aquiring more bad block markers. */
	if (flags & EO_MRKBD) {
		msg("markbad (not implemented) %llx | %x\n", block_base, block_size);
	}

	return 0;
}


/* Allocate resources and setup if not done yet. */
static int env_alloc(void)
{
	if (cache && ue) {
		return 0;
	}

	/* Block cache data area */
	cache = malloc(nand_info[0].erasesize);
	cblock = -1;
	if (!cache) {
		msg("Out of memory\n");
		goto error_exit;
	}

	/* Maximum supported env size for read is the same as the write
	 * size we use. uenv_alg can find and read environments shorter
	 * than this. */
	ue = uenv_init(CONFIG_ENV_SIZE);
	if (!ue) goto error_exit;

	/* Start of env area */
	ue->envbase = CONFIG_ENV_OFFSET;
	/* size of env to write */
	ue->envsize = CONFIG_ENV_SIZE;
	/* Size of erase block */
	ue->eraseblock = nand_info[0].erasesize;
	/* Stride of potential env blocks in area */
	ue->stride = ue->eraseblock / CONFIG_ENV_PERBLOCK;
	/* Size of env area */
	ue->totsize = CONFIG_ENV_MAX;
	/* We read any type, but write only NHRF */
	ue->type = UT_NHRF;

	return 0;
error_exit:
	return -1;
}


/* This is called before nand_init() so we can't read NAND
 *
 * Mark it OK for now. env_relocate() in env_common.c will call our
 * relocate function which does the real initialisation.
 */
int env_init(void)
{
	gd->env_addr	= (ulong)&default_environment[0];
	gd->env_valid	= 1;
	return 0;
}

/* This scans the whole env area, collecting layout information and
 * analysing that for suitable read and write candidate locations.
 *
 * Returns the number of environments found.
 */
static int scan_env(void)
{
	int rval;
	struct uenv_desc *ud;
	int flags;

	rval = env_alloc();
	if (rval) return rval;

	successful_reads = 0; /* Modified by envop */
	rval = full_env_scan(ue, envop);

	if (successful_reads == 0) {
		msg("Unable to read a single NAND page.\n");
		msg("Cowardly refusing to continue. Infinite loop.\n");
		while(1);
	}

	if (rval < 0) return -1;

	/* Walk the whole list of environments checking to see if they're all
	 * garbage (!(flags & (UF_ERASED | UF_CRCOK))) or all read errors
	 * (flags & UF_ERROR)
	 */
	ud = ue->ud_start;
	flags = ud->flags;
	if (!(flags & (UF_ERASED | UF_CRCOK)) || (flags & UF_ERROR)) {
		while (ud) {
			if (ud->flags != flags) break;
			ud = ud->next;
		}
		if (!ud) {
			/* We ran through the whole list */
			msg("Unable to read a single valid NAND page.\n");
		}
	}

	if (env_analyse(ue)) return -1;

	if (ue->read_env != NULL && ue->write_env==NULL) {
		int f = ue->read_env->flags & UF_REDFLAG ? ue->read_env->flag : -1;
		/* Read but no write candidate means that the sequence numbers are broken.
		 * scrubbing... */
		msg("Bad sequence numbers, scrubbing...\n");

		if (env_fetch(ue, envop)) return -1;
		if (env_scrub(ue, envop, f)) return -1;
		rval = full_env_scan(ue, envop);
		if (rval < 0) return -1;
		if (env_analyse(ue)) return -1;
	}

	return rval;
}


/* Command line interface for debugging, this function forces a scan &
 * analysis of the env layout and dumps the result.
 */
int do_scanenv(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int rval;

	rval = scan_env();
	msg("scan_env()=>%d\n", rval);
	uenv_dump(ue, printf);
	if (rval > 0) return 0;
	return rval;
}
U_BOOT_CMD(scanenv, 1, 1, do_scanenv,
	"Scan and dump env info",
	""
);


/* Scans & analyses the env area and reads the current read candidate into
 * U-Boot's hash table.
 */
int readenv(void)
{
	int rval;
	int upgrade_env=0;

	rval = scan_env();
	if (rval < 1) {
		msg("Scan failed to find env\n");
		return -1;
	} else if (rval < 3) {
		msg("Upgrading environment to NetComm High Reliability format\n");
		upgrade_env = 1;
	}

	rval = env_fetch(ue, envop);
	if (rval) return rval;

	/* We can't check here, size could be wrong. */
	rval = env_import((char*)ue->buffer, 0);
	if (rval <= 0) return -1;

	if (upgrade_env) {
		saveenv();
	}

	return 0;
}


/* Command line interface to force a env read.
 */
int do_rereadenv(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int rval;

	rval = readenv();
	if (rval==0) {
		msg("Env read OK.\n");
	} else {
		msg("Env read failed, %d\n", rval);
	}

	return rval;
}
U_BOOT_CMD(rereadenv, 1, 1, do_rereadenv,
	"Force env read",
	""
);


/* U-Boot API function to save the current content of the hash table.
 * It re-scans the env area and writes to the write candidate location.
 *
 * While we support all types of environments during read, we only write
 * NHRF environment blocks.
 */
int saveenv(void)
{
	int rval;
	char *dptr;
	unsigned retry = CONFIG_ENV_PERBLOCK + 1;
	int write_all = 0;
	int f;

	/* In order to deal with errors, we have to retry writing until it sticks,
	 * but because I'm paranoid about bugs, I'll limit this to
	 * CONFIG_ENV_PERBLOCK + 1 tries.
	 *
	 * But in the special case of an empty flash, or upgrading from an old uboot
	 * we want to write lots of environments (mainly because userspace will only
	 * write to locations already in use).
	 */

	/* Initial read, this is expected to give us a read candidate */
	rval = scan_env();
	if (rval < 0) {
		msg("Unexpected scan error (%d).\n", rval);
	}

	if (rval < 3) {
		msg("Writing to every environment location\n");
		write_all = 1;
	}

	while (retry--) {

		if (env_analyse(ue) < 0) {
			msg("Env failure\n");
			return -1;
		}

		/* Environment into buffer */
		/* Skipping the header, dptr is now a pointer to the data area. */
		dptr = (char*)ue->buffer + ENV_HEADER_SIZE;
		/* ENV_SIZE is CONFIG_ENV_SIZE - ENV_HEADER_SIZE */
		rval = hexport_r(&env_htab, '\0', &dptr, ENV_SIZE);
		if (rval < 0) {
			msg("Cannot export environment: errno = %d\n", errno);
			return 1;
		}

		if (ue->write_env) {
			/* The flag value must be larger (with wrap) than the one
			 * in the read candidate location. If we don't have a read
			 * candidate, we start at 0. */
			f = ue->read_env ? (uint8_t)(ue->read_env->flag + 1) : 0;
			env_update_crc(ue->buffer, CONFIG_ENV_SIZE, f);
		} else {
			/* Shouldn't happen. */
			msg("No write candidate\n");
			return -1;
		}

		if (write_all) {
			if (env_scrub(ue, envop, f)) return -1;
			rval = full_env_scan(ue, envop);
			if (rval < 0) return -1;
			if (env_analyse(ue)) return -1;
			return 0;
		} else {
			rval = env_store(ue, envop);
		}

		if (rval < 0) {
			msg("Failed to store\n");
			/* Retrying... */
		} else {
			/* Success */
			return 0;
		}
	}

	msg("Maximum write retries exceeded, giving up.\n");
	return 1;
}


/* This is the real initialisation function. By the time it is called, the
 * MTD driver should be initialised, so we can get our environment data.
 */
void env_relocate_spec(void)
{
	int rval;

	rval = env_alloc();
	if (rval) goto error_exit;

	rval = readenv();
	if (rval) goto error_exit;

	return;

error_exit:
	/* In case of error we revert to default environment. */
	set_default_env("!env_nand_rel failed");
	return;
}
