/*
 * mux.c
 *
 * Pin multiplexer setup for am335x processor
 *
 * Copyright (C) 2012 NetComm Wireless Limited - http://www.netcommwireless.com/
 *
 * Based on mux.c
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include <common.h>
#include <config.h>
#include <asm/io.h>
#include "common_def.h"
#include <asm/arch/hardware.h>

#include "mux.h"


/* Pin multiplexer setup organised by functional modules.
 * NOTE: Pin mux arrays must be terminated with an {-1} entry. */

/* Betzy processor board extras */
static const struct module_pin_mux betzyboard_pin_mux[] = {
	{OFFSET(gpmc_a5), (MODE(7) | RXACTIVE)},	/* Debug LED on Betzy board */

	{-1},
};

/* Nguni processor board extras */
static const struct module_pin_mux nguniboard_pin_mux[] = {

	{-1},
};

/* Kewel processor board extras */
static const struct module_pin_mux kewelboard_pin_mux[] = {

	{-1},
};


static const struct module_pin_mux uart0_pin_mux[] = {
	{OFFSET(uart0_rxd), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* UART0_RXD */
	{OFFSET(uart0_txd), (MODE(0) | PULLUDEN)},		/* UART0_TXD */

	{-1},
};

static const struct module_pin_mux nand_pin_mux[] = {
	{OFFSET(gpmc_ad0), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD0 */
	{OFFSET(gpmc_ad1), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD1 */
	{OFFSET(gpmc_ad2), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD2 */
	{OFFSET(gpmc_ad3), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD3 */
	{OFFSET(gpmc_ad4), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD4 */
	{OFFSET(gpmc_ad5), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD5 */
	{OFFSET(gpmc_ad6), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD6 */
	{OFFSET(gpmc_ad7), (MODE(0) | PULLUP_EN | RXACTIVE)},	/* NAND AD7 */
	{OFFSET(gpmc_wait0), (MODE(0) | RXACTIVE | PULLUP_EN)}, /* NAND WAIT */
	{OFFSET(gpmc_wpn), (MODE(7) | PULLUP_EN | RXACTIVE)},	/* NAND_WPN */
	{OFFSET(gpmc_csn0), (MODE(0) | PULLUDEN)},	/* NAND_CS0 */
	{OFFSET(gpmc_advn_ale), (MODE(0) | PULLUDEN)}, /* NAND_ADV_ALE */
	{OFFSET(gpmc_oen_ren), (MODE(0) | PULLUDEN)},	/* NAND_OE */
	{OFFSET(gpmc_wen), (MODE(0) | PULLUDEN)},	/* NAND_WEN */
	{OFFSET(gpmc_be0n_cle), (MODE(0) | PULLUDEN)},	/* NAND_BE_CLE */

	{-1},
};

static const struct module_pin_mux i2c0_pin_mux[] = {
	{OFFSET(i2c0_sda), (MODE(0) | RXACTIVE | PULLUDEN | SLEWCTRL)},	/* I2C_DATA */
	{OFFSET(i2c0_scl), (MODE(0) | RXACTIVE | PULLUDEN | SLEWCTRL)},	/* I2C_SCLK */

	{-1},
};

static const struct module_pin_mux i2c1_pin_mux[] = {
	{OFFSET(spi0_d1), (MODE(2) | RXACTIVE | PULLUDEN | SLEWCTRL)},	/* I2C_DATA */
	{OFFSET(spi0_cs0), (MODE(2) | RXACTIVE | PULLUDEN | SLEWCTRL)},	/* I2C_SCLK */

	{-1},
};


#ifdef CONFIG_SPL_BUILD

/* PHY strap pin setup, specific to Atheros AR8035 PHY. See gpio.c for details */
static const struct module_pin_mux rgmii1_pin_mux[] = {

	{OFFSET(mii1_txclk), MODE(2) | PULLUDDIS},		/* RGMII1_TCLK */
	{OFFSET(mii1_txen), MODE(2) | PULLUDDIS},		/* RGMII1_TCTL */
	{OFFSET(mii1_txd3), MODE(2) | PULLUDDIS},		/* RGMII1_TD3 */
	{OFFSET(mii1_txd2), MODE(2) | PULLUDDIS},		/* RGMII1_TD2 */
	{OFFSET(mii1_txd1), MODE(2) | PULLUDDIS},		/* RGMII1_TD1 */
	{OFFSET(mii1_txd0), MODE(2) | PULLUDDIS},		/* RGMII1_TD0 */

	{OFFSET(mii1_rxclk), MODE(7) | RXACTIVE | PULLUDDIS},	/* (NC for 3.3V) RGMII1_RCLK */
	{OFFSET(mii1_rxdv), MODE(7) | RXACTIVE | PULLUDDIS},	/* (M0 GPIO0) RGMII1_RCTL */
	{OFFSET(mii1_rxd3), MODE(7) | RXACTIVE | PULLUDDIS},	/* (M3 GPIO1) RGMII1_RD3 */
	{OFFSET(mii1_rxd2), MODE(7) | RXACTIVE | PULLUDDIS},	/* (M1 GPIO1) RGMII1_RD2 */
	{OFFSET(mii1_rxd1), MODE(7) | RXACTIVE | PULLUDDIS},	/* (PYA1 GPIO0) RGMII1_RD1 */
	{OFFSET(mii1_rxd0), MODE(7) | RXACTIVE | PULLUDDIS},	/* (PYA0 GPIO0) RGMII1_RD0 */

	{-1},
};

/* PHY strap pin setup */
static const struct module_pin_mux rgmii2_pin_mux[] = {

	{OFFSET(gpmc_a6), MODE(2) | PULLUDDIS},		/* RGMII2_TCLK */
	{OFFSET(gpmc_a0), MODE(2) | PULLUDDIS},		/* RGMII2_TCTL */
	{OFFSET(gpmc_a2), MODE(2) | PULLUDDIS},		/* RGMII2_TD3 */
	{OFFSET(gpmc_a3), MODE(2) | PULLUDDIS},		/* RGMII2_TD2 */
	{OFFSET(gpmc_a4), MODE(2) | PULLUDDIS},		/* RGMII2_TD1 */
	{OFFSET(gpmc_a5), MODE(2) | PULLUDDIS},		/* RGMII2_TD0 */
	
	{OFFSET(gpmc_a7), MODE(7) | RXACTIVE | PULLUDDIS},	/* RGMII2_RCLK */
	{OFFSET(gpmc_a1), MODE(7) | RXACTIVE | PULLUDDIS},	/* RGMII2_RCTL */
	{OFFSET(gpmc_a8), MODE(7) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD3 */
	{OFFSET(gpmc_a9), MODE(7) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD2 */
	{OFFSET(gpmc_a10), MODE(7) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD1 */
	{OFFSET(gpmc_a11), MODE(7) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD0 */

	{-1},
};

/* MDIO bus */
static const struct module_pin_mux mdio_pin_mux[] = {
#ifdef CONFIG_BITBANGMII
	/* GPIO, when bitbanging */
	{OFFSET(mdio_data), MODE(7) | RXACTIVE | PULLUP_EN | SLEWCTRL}, /* MDIO_DATA */
	{OFFSET(mdio_clk), MODE(7) | RXACTIVE | PULLUP_EN | SLEWCTRL},	/* MDIO_CLK */
#else /* ndef CONFIG_BITBANGMII */
	{OFFSET(mdio_data), MODE(0) | RXACTIVE | PULLUP_EN | SLEWCTRL}, /* MDIO_DATA */
	{OFFSET(mdio_clk), MODE(0) | PULLUP_EN | SLEWCTRL},	/* MDIO_CLK */
#endif /* CONFIG_BITBANGMII */

	{-1},
};

#else /* ndef CONFIG_SPL_BUILD */

/* Normal operation setup */
static const struct module_pin_mux rgmii1_pin_mux[] = {

	/* We are at the edge of allowable RGMII Tx clock timing. Slower
	 * slew rate helps a little. */
	{OFFSET(mii1_txclk), MODE(2) | PULLUDDIS | SLEWCTRL},	/* RGMII1_TCLK */
	{OFFSET(mii1_txen), MODE(2) | PULLUDDIS | SLEWCTRL},	/* RGMII1_TCTL */
	{OFFSET(mii1_txd3), MODE(2) | PULLUDDIS},	/* RGMII1_TD3 */
	{OFFSET(mii1_txd2), MODE(2) | PULLUDDIS},	/* RGMII1_TD2 */
	{OFFSET(mii1_txd1), MODE(2) | PULLUDDIS},	/* RGMII1_TD1 */
	{OFFSET(mii1_txd0), MODE(2) | PULLUDDIS},	/* RGMII1_TD0 */

	{OFFSET(mii1_rxclk), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII1_RCLK */
	{OFFSET(mii1_rxdv), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII1_RCTL */
	{OFFSET(mii1_rxd3), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII1_RD3 */
	{OFFSET(mii1_rxd2), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII1_RD2 */
	{OFFSET(mii1_rxd1), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII1_RD1 */
	{OFFSET(mii1_rxd0), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII1_RD0 */

	{-1},
};

static const struct module_pin_mux rgmii2_pin_mux[] = {
	/* We are at the edge of allowable RGMII Tx clock timing. Slower
	 * slew rate helps a little. */
	{OFFSET(gpmc_a6), MODE(2) | PULLUDDIS | SLEWCTRL},	/* RGMII2_TCLK */
	{OFFSET(gpmc_a0), MODE(2) | PULLUDDIS | SLEWCTRL},	/* RGMII2_TCTL */
	{OFFSET(gpmc_a2), MODE(2) | PULLUDDIS},			/* RGMII2_TD3 */
	{OFFSET(gpmc_a3), MODE(2) | PULLUDDIS},			/* RGMII2_TD2 */
	{OFFSET(gpmc_a4), MODE(2) | PULLUDDIS},			/* RGMII2_TD1 */
	{OFFSET(gpmc_a5), MODE(2) | PULLUDDIS},			/* RGMII2_TD0 */

	{OFFSET(gpmc_a7), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII2_RCLK */
	{OFFSET(gpmc_a1), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII2_RCTL */
	{OFFSET(gpmc_a8), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD3 */
	{OFFSET(gpmc_a9), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD2 */
	{OFFSET(gpmc_a10), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD1 */
	{OFFSET(gpmc_a11), MODE(2) | RXACTIVE | PULLUDDIS},	/* RGMII2_RD0 */

	{-1},
};

static const struct module_pin_mux mdio_pin_mux[] = {
#ifdef CONFIG_BITBANGMII
	/* GPIO, when bitbanging */
	{OFFSET(mdio_data), MODE(7) | RXACTIVE | PULLUP_EN | SLEWCTRL}, /* MDIO_DATA */
	{OFFSET(mdio_clk), MODE(7) | RXACTIVE | PULLUP_EN | SLEWCTRL},	/* MDIO_CLK */
#else /* ndef CONFIG_BITBANGMII */
	{OFFSET(mdio_data), MODE(0) | RXACTIVE | PULLUP_EN | SLEWCTRL}, /* MDIO_DATA */
	{OFFSET(mdio_clk), MODE(0) | PULLUP_EN | SLEWCTRL},	/* MDIO_CLK */
#endif /* CONFIG_BITBANGMII */

	{-1},
};

#endif /* ndef CONFIG_SPL_BUILD */


static const struct module_pin_mux rmii1_pin_mux[] = {

	{OFFSET(mii1_txen), MODE(1) | PULLUDDIS },           /* RMII1_TXEN */
	{OFFSET(mii1_rxerr), MODE(1) | PULLUDDIS | RXACTIVE},   /* RMII1_RXERR */
	{OFFSET(mii1_crs), MODE(1) | PULLUDDIS | RXACTIVE},     /* RMII1_CRS */
	{OFFSET(mii1_rxd0), MODE(1) | PULLUDDIS | RXACTIVE},    /* RMII1_RXD0 */
	{OFFSET(mii1_rxd1), MODE(1) | PULLUDDIS | RXACTIVE},    /* RMII1_RXD1 */
	{OFFSET(mii1_txd0), MODE(1) | PULLUDDIS},           /* RMII1_TXD0 */
	{OFFSET(mii1_txd1), MODE(1) | PULLUDDIS},           /* RMII1_TXD1 */
	{OFFSET(rmii1_refclk), MODE(0) | PULLUDDIS | RXACTIVE}, /* RMII1_REFCLK */

	{-1},
};

/* Miscellaneous pin mux settings for koala/blinky ioboard */
static const struct module_pin_mux misc_pin_mux_blinky[] = {

	{OFFSET(lcd_pclk), MODE(7) | PULLUP_EN}, /* 2:24 3G_PSW */
	{OFFSET(gpmc_ad9), MODE(7) | PULLUDEN},  /* 0:23 nDisable to module */
	{OFFSET(lcd_ac_bias_en), MODE(7) | PULLUDEN | RXACTIVE}, /* 2:25 3G STATUS */
	{OFFSET(gpmc_ad13), MODE(7) | PULLUDEN},  /* 1:13 LEDACT */
	{OFFSET(lcd_hsync), MODE(7) | PULLUDEN | RXACTIVE}, /* 2:23 SIMPRES */

	{-1},
};

/* Miscellaneous pin mux settings for elaine/ntc_30 ioboard */
static const struct module_pin_mux misc_pin_mux_elaine[] = {

	{OFFSET(xdma_event_intr0), MODE(7) | PULLUDDIS }, /* LED1 - Power */
	{OFFSET(xdma_event_intr1), MODE(7) | PULLUDDIS }, /* LED2 - Tx/Rx */
	{OFFSET(mcasp0_ahclkx), MODE(7) | PULLUDDIS }, /* LED3 - DCD */
	{OFFSET(gpmc_ad8), MODE(7) | PULLUDDIS }, /* LED4 - Service */
	{OFFSET(mmc0_dat1), MODE(7) | PULLUDDIS }, /* LED5 - RSSI */

	{-1},
};

/* Miscellaneous pin mux settings for newman ioboard */
static const struct module_pin_mux misc_pin_mux_newman[] = {

	{-1},
};


/* Miscellaneous pin mux settings for Nguni (ntc_140wx) IO-board */
static const struct module_pin_mux misc_pin_mux_nguni[] = {

	/* Ethernet PHY reset lines */
	{OFFSET(lcd_vsync), MODE(7) | PULLUDDIS },		/* gpio2_22 RGMII2 */
	{OFFSET(lcd_hsync), MODE(7) | PULLUDDIS },		/* gpio2_23 RGMII1 */

	/* LEDs, dual red+green LEDS, active low. */
	{OFFSET(xdma_event_intr0), MODE(7) | PULLUDDIS },	/* gpio0_19 LED1 grn */
	{OFFSET(xdma_event_intr1), MODE(7) | PULLUDDIS },	/* gpio0_20 LED1 red */
	{OFFSET(gpmc_ad14), MODE(7) | PULLUDDIS },		/* gpio1_14 LED2 grn */
	{OFFSET(gpmc_ad15), MODE(7) | PULLUDDIS },		/* gpio1_15 LED2 red */
	{OFFSET(gpmc_ad12), MODE(7) | PULLUDDIS },		/* gpio1_12 LED3 grn */
	{OFFSET(gpmc_ad13), MODE(7) | PULLUDDIS },		/* gpio1_13 LED3 red */
	{OFFSET(gpmc_ad10), MODE(7) | PULLUDDIS },		/* gpio0_26 LED4 grn */
	{OFFSET(gpmc_ad11), MODE(7) | PULLUDDIS },		/* gpio0_27 LED4 red */
	{OFFSET(gpmc_ad8), MODE(7) | PULLUDDIS },		/* gpio0_22 LED5 grn */
	{OFFSET(gpmc_ad9), MODE(7) | PULLUDDIS },		/* gpio0_23 LED5 red */
	{OFFSET(mcasp0_aclkx), MODE(7) | PULLUDDIS },		/* gpio3_14 LED6 grn */
	{OFFSET(mcasp0_aclkr), MODE(7) | PULLUDDIS },		/* gpio3_18 LED6 red */
	{OFFSET(mcasp0_ahclkr), MODE(7) | PULLUDDIS },		/* gpio3_17 LED7 grn */
	{OFFSET(mcasp0_axr0), MODE(7) | PULLUDDIS },		/* gpio3_16 LED7 red */
	{OFFSET(lcd_data14), MODE(7) | PULLUDDIS },		/* gpio0_10 LED8 grn */
	{OFFSET(mcasp0_fsx), MODE(7) | PULLUDDIS },		/* gpio3_15 LED8 red */

	{-1},
};

/* Miscellaneous pin mux settings for Kudu (ntc_22) IO-board */
static const struct module_pin_mux misc_pin_mux_kudu[] = {

	/* Ethernet PHY reset lines */
	{OFFSET(lcd_hsync), MODE(7) | PULLUDDIS },		/* gpio2_23 Switch reset signal */

	/* LEDs, dual red+green LEDS, active low. */
	{OFFSET(gpmc_ad14), MODE(7) | PULLUDDIS },		/* gpio1_14 LED1 grn */
	{OFFSET(gpmc_ad15), MODE(7) | PULLUDDIS },		/* gpio1_15 LED1 red */
	{OFFSET(gpmc_ad10), MODE(7) | PULLUDDIS },		/* gpio0_26 LED2 grn */
	{OFFSET(gpmc_ad11), MODE(7) | PULLUDDIS },		/* gpio0_27 LED2 red */
	{OFFSET(gpmc_ad8), MODE(7) | PULLUDDIS },		/* gpio0_22 LED3 grn */
	{OFFSET(gpmc_ad9), MODE(7) | PULLUDDIS },		/* gpio0_23 LED3 red */
	{OFFSET(mcasp0_aclkr), MODE(7) | PULLUDDIS },		/* gpio3_18 LED4 grn */
	{OFFSET(mcasp0_ahclkr), MODE(7) | PULLUDDIS },		/* gpio3_17 LED4 red */
	{OFFSET(xdma_event_intr0), MODE(7) | PULLUDDIS },	/* gpio0_19 LED5 grn */
	{OFFSET(xdma_event_intr1), MODE(7) | PULLUDDIS },	/* gpio0_20 LED5 red */
	{OFFSET(mcasp0_axr0), MODE(7) | PULLUDDIS },		/* gpio3_16 LED6 grn */
	{OFFSET(lcd_data14), MODE(7) | PULLUDDIS },		/* gpio0_10 LED6 red */
	{OFFSET(gpmc_ad12), MODE(7) | PULLUDDIS },		/* gpio1_12 LED7 grn */
	{OFFSET(gpmc_ad13), MODE(7) | PULLUDDIS },		/* gpio1_13 LED7 red */
	{OFFSET(mcasp0_aclkx), MODE(7) | PULLUDDIS },		/* gpio3_14 LED8 grn */
	{OFFSET(mcasp0_fsx), MODE(7) | PULLUDDIS },		/* gpio3_15 LED8 red */

	{-1},
};

/* Miscellaneous pin mux settings for Hamster (ntc_2000) IO-board */
static const struct module_pin_mux misc_pin_mux_hamster[] = {

	/* Ethernet PHY reset lines */
	{OFFSET(lcd_hsync), MODE(7) | PULLUDDIS }, /* gpio2_23 RGMII1 */

	/* LEDs, dual red+green LEDS, active low. */
	/* TODO: Add LEDs */

	{-1},
};


#ifdef CONFIG_MMC
static const struct module_pin_mux mmc0_pin_mux[] = {

	{OFFSET(mmc0_dat3), (MODE(0) | RXACTIVE | PULLUP_EN)},	/* MMC0_DAT3 */
	{OFFSET(mmc0_dat2), (MODE(0) | RXACTIVE | PULLUP_EN)},	/* MMC0_DAT2 */
	{OFFSET(mmc0_dat1), (MODE(0) | RXACTIVE | PULLUP_EN)},	/* MMC0_DAT1 */
	{OFFSET(mmc0_dat0), (MODE(0) | RXACTIVE | PULLUP_EN)},	/* MMC0_DAT0 */
	{OFFSET(mmc0_clk), (MODE(0) | RXACTIVE | PULLUP_EN)},	/* MMC0_CLK */
	{OFFSET(mmc0_cmd), (MODE(0) | RXACTIVE | PULLUP_EN)},	/* MMC0_CMD */
#if 0 
	{OFFSET(mcasp0_aclkr), (MODE(4) | RXACTIVE)},		/* MMC0_WP */
#endif
	{OFFSET(spi0_cs1), (MODE(5) | RXACTIVE | PULLUP_EN)},	/* MMC0_CD */

	{-1},
};
#endif /* CONFIG_MMC */

#ifdef CONFIG_SPI
static const struct module_pin_mux spi0_pin_mux[] = {

	{OFFSET(spi0_sclk), MODE(0) | PULLUDEN | RXACTIVE},	/*SPI0_SCLK */
	{OFFSET(spi0_d0), MODE(0) | PULLUDEN | PULLUP_EN |
							RXACTIVE}, /*SPI0_D0 */
	{OFFSET(spi0_d1), MODE(0) | PULLUDEN |
							RXACTIVE}, /*SPI0_D1 */
	{OFFSET(spi0_cs0), MODE(0) | PULLUDEN | PULLUP_EN | RXACTIVE},	/*SPI0_CS0 */

	{-1},
};
#endif /* CONFIG_SPI */


/* Per board pin mux setting groups. The profile entry is
 * a combined bitfield of all relevant IOBOARDs
 * NOTE: The last entry must have a NULL {0} pointer as the
 * pinmux array. */

struct am335x_pin_mux {
	const struct module_pin_mux *mod_pin_mux;
	unsigned short profile; /* IO Boards bitmap */
};

/* Betzy processor board (for Koala/Blinky, Elaine & Newman IOBoards) */
static const struct am335x_pin_mux betzy_pin_mux[] = {

	{betzyboard_pin_mux, PROFILE_ALL},

	{uart0_pin_mux, PROFILE_ALL},
	{i2c1_pin_mux, PROFILE_ALL},
	{nand_pin_mux, PROFILE_ALL},

	{mdio_pin_mux, PROFILE_ALL}, /* MDIO bus */

	{rgmii1_pin_mux, PROFILE_0}, /* Koala/Blinky IOBoard Ethernet */
	{misc_pin_mux_blinky, PROFILE_0},

	{rmii1_pin_mux, PROFILE_1}, /* Elaine/ntc_30 IOBoard */
	{misc_pin_mux_elaine, PROFILE_1},

	{misc_pin_mux_newman, PROFILE_2}, /* Newman IOBoard */

#ifdef CONFIG_MMC
	{mmc0_pin_mux, PROFILE_2}, /* Newman has MMC slot */
#endif
#ifdef CONFIG_SPI
	{spi0_pin_mux, PROFILE_ALL},
#endif

	{0},
};

/* Nguni processor board (for Nguni IOBoard) */
static const struct am335x_pin_mux nguni_pin_mux[] = {

	{nguniboard_pin_mux, PROFILE_ALL},

	{uart0_pin_mux, PROFILE_ALL},
	{i2c1_pin_mux, PROFILE_ALL},
	{nand_pin_mux, PROFILE_ALL},

	{mdio_pin_mux, PROFILE_ALL},
	{rgmii1_pin_mux, PROFILE_3}, /* Nguni IOBoard */
	{rgmii2_pin_mux, PROFILE_3}, /* Nguni IOBoard */

	{misc_pin_mux_nguni, PROFILE_3},

#ifdef CONFIG_MMC
	{mmc0_pin_mux, PROFILE_3}, /* Nguni has MMC slot */
#endif
#ifdef CONFIG_SPI
	{spi0_pin_mux, PROFILE_ALL},
#endif

	{0},
};

/* Kewel processor board (for Hamster and Kudu IOBoards) */
static const struct am335x_pin_mux kewel_pin_mux[] = {

	{kewelboard_pin_mux, PROFILE_ALL},

	{uart0_pin_mux, PROFILE_ALL},
	{i2c1_pin_mux, PROFILE_ALL},
	{nand_pin_mux, PROFILE_ALL},

	{mdio_pin_mux, PROFILE_ALL},
	{rgmii1_pin_mux, PROFILE_5}, /* Hamster board has single Ethernet on rgmii 1 */
	{rgmii2_pin_mux, PROFILE_4}, /* Kudu board has ETH switch and fixed link on rgmii2 */

	{misc_pin_mux_kudu , PROFILE_4},
	{misc_pin_mux_hamster , PROFILE_5},

#ifdef CONFIG_MMC
	{mmc0_pin_mux, PROFILE_4 | PROFILE_5}, /* Hamster & Kudu have MMC slot */
#endif
#ifdef CONFIG_SPI
	{spi0_pin_mux, PROFILE_ALL},
#endif

	{0},
};

/* Index here is board (not ioboard type) */
static const struct am335x_pin_mux *am335x_board_pin_mux[] = {
	[BETZY_BOARD] = betzy_pin_mux,
	[NGUNI_BOARD] = nguni_pin_mux,
	[KEWEL_BOARD] = kewel_pin_mux
};

/* Dump one pinmux entry */
static void _dump_pinmux(const char *name, unsigned offset)
{
	unsigned v = __raw_readl(CTRL_BASE+offset);
	printf("PMUX %-20s: %08x ", name, v);
	printf("MUX%d ", v & 0x7);
	if (v & (1<<4)) {
		printf("PU");
	} else {
		printf("PD");
	}
	if (!(v & (1<<3))) { /* 0 enables pull-up/down */
		printf("en ");
	} else {
		printf("dis ");
	}
	if (v & (1<<5)) {
		printf("RXon ");
	} else {
		printf("RXoff ");
	}
	if (v & (1<<6)) {
		printf("slow ");
	} else {
		printf("fast ");
	}
	printf("\n");
}
#define DUMP_PINMUX(pin) _dump_pinmux(#pin, OFFSET(pin))

void dumpmux(void)
{
	DUMP_PINMUX(mdio_data);
	DUMP_PINMUX(mdio_clk);
/*
	DUMP_PINMUX(mii1_txen);
	DUMP_PINMUX(mii1_rxerr);
	DUMP_PINMUX(mii1_crs);
	DUMP_PINMUX(mii1_rxd0);
	DUMP_PINMUX(mii1_rxd1);
	DUMP_PINMUX(mii1_txd0);
	DUMP_PINMUX(mii1_txd1);
	DUMP_PINMUX(rmii1_refclk);
*/
/*
	DUMP_PINMUX(mii1_rxclk);
	DUMP_PINMUX(gpmc_a7);
*/
}


/*
 * Configure the pin mux for one module
 */
static void configure_module_pin_mux(const struct module_pin_mux *mod_pin_mux)
{
	int i;

	if (!mod_pin_mux)
		return;

	for (i = 0; mod_pin_mux[i].reg_offset != -1; i++)
		MUX_CFG(mod_pin_mux[i].val, mod_pin_mux[i].reg_offset);
}

/*
 * Check each module in the daughter board(first argument) whether it is
 * available in the selected profile(second argument). If the module is not
 * available in the selected profile, skip the corresponding configuration.
 */
static void set_board_pin_mux(const struct am335x_pin_mux *pin_mux,
			int prof)
{
	int i;
	if (!pin_mux)
		return;

	for (i = 0; pin_mux[i].mod_pin_mux != 0; i++)  {
		if ((pin_mux[i].profile & prof) ||
					(prof == PROFILE_NONE)) {
			configure_module_pin_mux(pin_mux[i].mod_pin_mux);
			printf("MUX: %d, profile match\n", i);
		} else {
			printf("MUX: %d, no match\n", i);
		}
	}
	dumpmux();
}

void configure_evm_pin_mux(unsigned char brd_id, char version[4], unsigned short
		profile, unsigned int daughter_board_flag)
{
	if (brd_id > LAST_BOARD)
		return;

	set_board_pin_mux(am335x_board_pin_mux[brd_id], profile);
}

void enable_i2c0_pin_mux(void)
{
	configure_module_pin_mux(i2c0_pin_mux);
}

void enable_uart0_pin_mux(void)
{
	configure_module_pin_mux(uart0_pin_mux);
}
