/*
 * gpio.c
 *
 * Copyright (C) 2012 NetComm Wireless Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Iwo.Mergler@netcommwireless.com
 */
#include <common.h>
#include <command.h>
#include <exports.h>
#include <config.h>
#include <asm/io.h>
#include "common_def.h"
#include <asm/arch/hardware.h>
#include "gpio.h"


/* Relevant clock module registers, GPIO0 is always on */

#define CM_PER_BASE   0x44e00000
#define CM_PER_GPIO1  (CM_PER_BASE + 0xac)
#define CM_PER_GPIO2  (CM_PER_BASE + 0xb0)
#define CM_PER_GPIO3  (CM_PER_BASE + 0xb4)

void init_gpio(void)
{
	__raw_writel(0x40002, CM_PER_GPIO1);
	__raw_writel(0x40002, CM_PER_GPIO2);
	__raw_writel(0x40002, CM_PER_GPIO3);
}


/* Processor board related GPIOs that need setting during U-Boot */
static const struct pinstate betzyboard_pins[] = {

	{.bank = 1, .pin = 21, .v = 1, .oe = 1}, /* Debug LED (D2) on processor board. */
#ifdef CONFIG_BITBANGMII
	{.bank = 0, .pin = 0, .v = 1, .oe = 0}, /* MDIO */
	{.bank = 0, .pin = 1, .v = 1, .oe = 1}, /* MDC */
#endif

	{.end = 1}
};


static const struct pinstate nguniboard_pins[] = {

#ifdef CONFIG_BITBANGMII
	{.bank = 0, .pin = 0, .v = 1, .oe = 0}, /* MDIO */
	{.bank = 0, .pin = 1, .v = 1, .oe = 1}, /* MDC */
#endif

	{.end = 1}
};

static const struct pinstate kewelboard_pins[] = {

#ifdef CONFIG_BITBANGMII
	{.bank = 0, .pin = 0, .v = 1, .oe = 0}, /* MDIO */
	{.bank = 0, .pin = 1, .v = 1, .oe = 1}, /* MDC */
#endif

	{.end = 1}
};


/* IO Board specific GPIOs that need setting during U-Boot */

/* Profile 0, NBN/Koala/Blinky board */
static const struct pinstate blinky_pins[] = {

	{.bank = 0, .pin = 23, .v = 0, .oe = 1 }, /* nDisable to module, low disables. */
	{.bank = 2, .pin = 24, .v = 1, .oe = 1 }, /* Power to module, active low */
	{.bank = 2, .pin = 25, .v = 0, .oe = 1 }, /* Module status, pulled down, input */
	{.bank = 1, .pin = 13, .v = 0, .oe = 1 }, /* module activity LED & GPIO */
	{.bank = 2, .pin = 23, .v = 0, .oe = 1 }, /* SIM presence detect */

	{.end = 1}
};


/* Profile 1, ntc_30/40/elaine board */
static const struct pinstate elaine_pins[] = {

	{.end = 1}
};


/* Profile 2, newman board, for production. */
static const struct pinstate newman_pins[] = {

	{.end = 1}
};


/* Profile 3, Nguni (ntc_140wx) IOboard */
static const struct pinstate nguni_pins[] = {

	/* LEDs 5x dual red+green LEDS, active low. SPL is all red, U-Boot is all amber. */
#ifdef CONFIG_SPL_BUILD
	{.bank = 0, .pin = 19, .v = 1, .oe = 1},	/* gpio0_19 LED1 grn */
	{.bank = 0, .pin = 20, .v = 0, .oe = 1},	/* gpio0_20 LED1 red */
	{.bank = 1, .pin = 14, .v = 1, .oe = 1},	/* gpio1_14 LED2 grn */
	{.bank = 1, .pin = 15, .v = 0, .oe = 1},	/* gpio1_15 LED2 red */
	{.bank = 1, .pin = 12, .v = 1, .oe = 1},	/* gpio1_12 LED3 grn */
	{.bank = 1, .pin = 13, .v = 0, .oe = 1},	/* gpio1_13 LED3 red */
	{.bank = 0, .pin = 26, .v = 1, .oe = 1},	/* gpio0_26 LED4 grn */
	{.bank = 0, .pin = 27, .v = 0, .oe = 1},	/* gpio0_27 LED4 red */
	{.bank = 0, .pin = 22, .v = 1, .oe = 1},	/* gpio0_22 LED5 grn */
	{.bank = 0, .pin = 23, .v = 0, .oe = 1},	/* gpio0_23 LED5 red */

	{.bank = 3, .pin = 14, .v = 1, .oe = 1},	/* gpio3_14 LED6 grn */
	{.bank = 3, .pin = 18, .v = 0, .oe = 1},	/* gpio3_18 LED6 red */
	{.bank = 3, .pin = 17, .v = 1, .oe = 1},	/* gpio3_17 LED7 grn */
	{.bank = 3, .pin = 16, .v = 0, .oe = 1},	/* gpio3_16 LED7 red */
	{.bank = 0, .pin = 10, .v = 1, .oe = 1},	/* gpio0_10 LED8 grn */
	{.bank = 3, .pin = 15, .v = 0, .oe = 1},	/* gpio3_15 LED8 red */

#else
	{.bank = 0, .pin = 19, .v = 0, .oe = 1},	/* gpio0_19 LED1 grn */
	{.bank = 0, .pin = 20, .v = 0, .oe = 1},	/* gpio0_20 LED1 red */
	{.bank = 1, .pin = 14, .v = 0, .oe = 1},	/* gpio1_14 LED2 grn */
	{.bank = 1, .pin = 15, .v = 0, .oe = 1},	/* gpio1_15 LED2 red */
	{.bank = 1, .pin = 12, .v = 0, .oe = 1},	/* gpio1_12 LED3 grn */
	{.bank = 1, .pin = 13, .v = 0, .oe = 1},	/* gpio1_13 LED3 red */
	{.bank = 0, .pin = 26, .v = 0, .oe = 1},	/* gpio0_26 LED4 grn */
	{.bank = 0, .pin = 27, .v = 0, .oe = 1},	/* gpio0_27 LED4 red */
	{.bank = 0, .pin = 22, .v = 0, .oe = 1},	/* gpio0_22 LED5 grn */
	{.bank = 0, .pin = 23, .v = 0, .oe = 1},	/* gpio0_23 LED5 red */

	{.bank = 3, .pin = 14, .v = 0, .oe = 1},	/* gpio3_14 LED6 grn */
	{.bank = 3, .pin = 18, .v = 0, .oe = 1},	/* gpio3_18 LED6 red */
	{.bank = 3, .pin = 17, .v = 0, .oe = 1},	/* gpio3_17 LED7 grn */
	{.bank = 3, .pin = 16, .v = 0, .oe = 1},	/* gpio3_16 LED7 red */
	{.bank = 0, .pin = 10, .v = 0, .oe = 1},	/* gpio0_10 LED8 grn */
	{.bank = 3, .pin = 15, .v = 0, .oe = 1},	/* gpio3_15 LED8 red */
#endif

	/* Release Ethernet PHY reset lines. */
	{.bank = 2, .pin = 22, .v = 1, .oe = 1}, /* gpio2_22 RGMII2 */
	{.bank = 2, .pin = 23, .v = 1, .oe = 1}, /* gpio2_23 RGMII1 */

	{.end = 1}
};


/* Profile 4, Kudu (nwl22) IOboard */
static const struct pinstate kudu_pins[] = {

	/* LEDs 5x dual red+green LEDS, active low. SPL is all red, U-Boot is all amber. */
#ifdef CONFIG_SPL_BUILD
	{.bank = 1, .pin = 14, .v = 1, .oe = 1},	/* gpio1_14 LED1 grn */
	{.bank = 1, .pin = 15, .v = 0, .oe = 1},	/* gpio1_15 LED1 red */
	{.bank = 0, .pin = 26, .v = 1, .oe = 1},	/* gpio0_26 LED2 grn */
	{.bank = 0, .pin = 27, .v = 0, .oe = 1},	/* gpio0_27 LED2 red */
	{.bank = 0, .pin = 22, .v = 1, .oe = 1},	/* gpio0_22 LED3 grn */
	{.bank = 0, .pin = 23, .v = 0, .oe = 1},	/* gpio0_23 LED3 red */
	{.bank = 3, .pin = 18, .v = 1, .oe = 1},	/* gpio3_18 LED4 grn */
	{.bank = 3, .pin = 17, .v = 0, .oe = 1},	/* gpio3_17 LED4 red */
	{.bank = 0, .pin = 19, .v = 1, .oe = 1},	/* gpio0_19 LED5 grn */
	{.bank = 0, .pin = 20, .v = 0, .oe = 1},	/* gpio0_20 LED5 red */
	{.bank = 3, .pin = 16, .v = 1, .oe = 1},	/* gpio3_16 LED6 grn */
	{.bank = 0, .pin = 10, .v = 0, .oe = 1},	/* gpio0_10 LED6 red */
	{.bank = 1, .pin = 12, .v = 1, .oe = 1},	/* gpio1_12 LED7 grn */
	{.bank = 1, .pin = 13, .v = 0, .oe = 1},	/* gpio1_13 LED7 red */
	{.bank = 3, .pin = 14, .v = 1, .oe = 1},	/* gpio3_14 LED8 grn */
	{.bank = 3, .pin = 15, .v = 0, .oe = 1},	/* gpio3_15 LED8 red */

#else
	{.bank = 1, .pin = 14, .v = 0, .oe = 1},	/* gpio1_14 LED1 grn */
	{.bank = 1, .pin = 15, .v = 0, .oe = 1},	/* gpio1_15 LED1 red */
	{.bank = 0, .pin = 26, .v = 0, .oe = 1},	/* gpio0_26 LED2 grn */
	{.bank = 0, .pin = 27, .v = 0, .oe = 1},	/* gpio0_27 LED2 red */
	{.bank = 0, .pin = 22, .v = 0, .oe = 1},	/* gpio0_22 LED3 grn */
	{.bank = 0, .pin = 23, .v = 0, .oe = 1},	/* gpio0_23 LED3 red */
	{.bank = 3, .pin = 18, .v = 0, .oe = 1},	/* gpio3_18 LED4 grn */
	{.bank = 3, .pin = 17, .v = 0, .oe = 1},	/* gpio3_17 LED4 red */
	{.bank = 0, .pin = 19, .v = 0, .oe = 1},	/* gpio0_19 LED5 grn */
	{.bank = 0, .pin = 20, .v = 0, .oe = 1},	/* gpio0_20 LED5 red */
	{.bank = 3, .pin = 16, .v = 0, .oe = 1},	/* gpio3_16 LED6 grn */
	{.bank = 0, .pin = 10, .v = 0, .oe = 1},	/* gpio0_10 LED6 red */
	{.bank = 1, .pin = 12, .v = 0, .oe = 1},	/* gpio1_12 LED7 grn */
	{.bank = 1, .pin = 13, .v = 0, .oe = 1},	/* gpio1_13 LED7 red */
	{.bank = 3, .pin = 14, .v = 0, .oe = 1},	/* gpio3_14 LED8 grn */
	{.bank = 3, .pin = 15, .v = 0, .oe = 1},	/* gpio3_15 LED8 red */
#endif

	/* Release Ethernet switch reset line. */
	{.bank = 2, .pin = 23, .v = 1, .oe = 1}, /* gpio2_23 RGMII2 */

	{.end = 1}
};


/* Profile 5, Hamster (ntc_2000) IOBoard. */
static const struct pinstate hamster_pins[] = {

	/* TODO: LEDs */

	/* Release PHY reset line */
	{.bank = 2, .pin = 23, .v = 1, .oe = 1}, /* gpio2_23 RGMII1 */

	{.end = 1}
};



void config_gpioset(const struct pinstate pins[])
{
	const struct pinstate *p = pins;
	while (!p->end) {
		printf("GPIO: %d:%d ", p->bank, p->pin);
		if (p->oe) printf("out %d\n", p->v);
		else printf("in\n");
		GPIO_setstate(p);
		p++;
	}
}


/* Set up strap pins for atheros ar8035 PHYs.
 * n is the number of PHYs, can be 1 or 2, which are set up in parallel. */
static void ar8035_strap_pins(unsigned n)
{
#ifdef CONFIG_SPL_BUILD
	/* In SPL, the RGMII lines are configured as GPIO to control the
	strap pins. This selects different MDIO addresses for the two PHYs. */

	#define PIN_ERST0 2,23 /* Reset line for first PHY */
	#define PIN_ERST1 2,22 /* Reset line for second PHY */
	const struct pinstate *sp;

	/* Bank / Pin / value settings for pin strapping */
	
	const struct pinstate strap0[] = {
		/* PHY0 / RGMII1 */
		{.bank = 3, .pin = 10, .v = 1 }, /* 1.8V/1.5V sel (EARCLK) */
		{.bank = 3, .pin =  4, .v = 0 }, /* M0 (EARCTL) */
		{.bank = 2, .pin = 18, .v = 1 }, /* M3 (EARX3) */
		{.bank = 2, .pin = 19, .v = 1 }, /* M1 (EARX2) */
		{.bank = 2, .pin = 20, .v = 0 }, /* PHYAD1 (EARX1) */
		{.bank = 2, .pin = 21, .v = 0 }, /* PHYAD0 (EARX0) */
		{.end = 1}
	};

	const struct pinstate strap1[] = {
		/* PHY1 / RGMII2 */
		{.bank = 1, .pin = 23, .v = 1 }, /* 1.8V/1.5V sel (EBRCLK) */
		{.bank = 1, .pin = 17, .v = 0 }, /* M0 (EBRCTL) */
		{.bank = 1, .pin = 24, .v = 1 }, /* M3 (EBRX3) */
		{.bank = 1, .pin = 25, .v = 1 }, /* M1 (EBRX2) */
		{.bank = 1, .pin = 26, .v = 0 }, /* PHYAD1 (EBRX1) */
		{.bank = 1, .pin = 27, .v = 1 }, /* PHYAD0 (EBRX0) */
		{.end = 1}
	};

	/* No PHYs to set up */
	if (n==0) return;

	/* Activate resets on PHY0 */
	GPIO_out(PIN_ERST0, 0);
	GPIO_dir(PIN_ERST0, GPIO_DIR_OUT);

	if (n>=2) {
		/* Activate resets on PHY1 */
		GPIO_out(PIN_ERST1, 0);
		GPIO_dir(PIN_ERST1, GPIO_DIR_OUT);

		/* Drive all PHY1 strap pins */
		for (sp = strap1; !sp->end; sp++) {
			GPIO_out(sp->bank, sp->pin, sp->v);
			GPIO_dir(sp->bank, sp->pin, GPIO_DIR_OUT);
		}
	}

	udelay(100000); /* Wait 100 ms reset time */

	/* Release PHY0 reset */
	GPIO_out(PIN_ERST0, 1);

	if (n>=2) {
		/* Release PHY1 reset */
		GPIO_out(PIN_ERST1, 1);
		puts("2");
	} else {
		puts("1");
	}
	puts("xPHY strap\n");

	udelay(1000); /* 1ms strap pin latching */

	/* Release all PHY0 strap pins */
	for (sp = strap0; !sp->end; sp++) {
		GPIO_dir(sp->bank, sp->pin, GPIO_DIR_IN);
	}

	if (n>=2) {
		/* Release all PHY1 strap pins */
		for (sp = strap1; !sp->end; sp++) {
			GPIO_dir(sp->bank, sp->pin, GPIO_DIR_IN);
		}
	}
#endif
}

/* Set up strap pins for atheros qca8337 Switch */
static void qca8337_strap_pins(void)
{
#ifdef CONFIG_SPL_BUILD
	/* In SPL, the RGMII lines are configured as GPIO to control the
	strap pins.  */

	#define PIN_EBRST 2,23 /* Reset line for switch */

	const struct pinstate *sp;

	/* Bank / Pin / value settings for pin strapping */
	const struct pinstate strap[] = {
		/* Switch RGMII2 */
		{.bank = 1, .pin = 27, .v = 0 }, /* RXD0_0 SPI_EN */
		{.bank = 1, .pin = 26, .v = 0 }, /* RXD1_0 SPI_SIZE */
		{.bank = 1, .pin = 25, .v = 1 }, /* RXD2_0 LED_OPEN_DRAIN */
		{.bank = 1, .pin = 24, .v = 0 }, /* RXD3_0 CABLE_DIAG */
		{.bank = 1, .pin = 17, .v = 1 }, /* RXDV_0 CTRL_DAC_0 */

		{.end = 1}
	};

	/* Activate reset */
	GPIO_out(PIN_EBRST, 0);
	GPIO_dir(PIN_EBRST, GPIO_DIR_OUT);

	/* Drive all strap pins */
	for (sp = strap; !sp->end; sp++) {
		GPIO_out(sp->bank, sp->pin, sp->v);
		GPIO_dir(sp->bank, sp->pin, GPIO_DIR_OUT);
	}

	udelay(100000); /* Wait 100 ms reset time */
	puts("Switch strap\n");

	/* Release reset */
	GPIO_out(PIN_EBRST, 1);

	udelay(1000); /* 1ms strap pin latching */

	/* Release all strap pins */
	for (sp = strap; !sp->end; sp++) {
		GPIO_dir(sp->bank, sp->pin, GPIO_DIR_IN);
	}
#endif
}

void configure_gpio(unsigned board, unsigned profile)
{
	init_gpio();

	switch (board) {
		case BETZY_BOARD:
			printf("Betzy Board init\n");
			config_gpioset(betzyboard_pins);
			break;
		case NGUNI_BOARD:
			printf("Nguni Board init\n");
			config_gpioset(nguniboard_pins);
			break;
		case KEWEL_BOARD:
			printf("Kewel Board init\n");
			config_gpioset(kewelboard_pins);
			break;
		default:
			printf("Unknown Board: %u\n", board);
			break;
	}

	if (profile & PROFILE_0) {
		printf("Blinky GPIO init\n");
		config_gpioset(blinky_pins);
	}

	if (profile & PROFILE_1) {
		printf("Elaine GPIO init\n");
		config_gpioset(elaine_pins);
	}

	if (profile & PROFILE_2) {
		printf("Newman GPIO init\n");
		config_gpioset(newman_pins);
	}

	if (profile & PROFILE_3) {
		printf("Nguni GPIO init\n");
		config_gpioset(nguni_pins);
		ar8035_strap_pins(2);
	}
	
	if (profile & PROFILE_4) {
		printf("Kudu GPIO init\n");
		config_gpioset(kudu_pins);
		qca8337_strap_pins();
	}

	if (profile & PROFILE_5) {
		printf("Hamster GPIO init\n");
		config_gpioset(hamster_pins);
		ar8035_strap_pins(1);
	}
}

#ifndef CONFIG_SPL_BUILD


#define NBANKS 4


/* GPIO read */
static int fun_gpior(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	static int bank = -1; /* Remember bank for repeat command */
	static int pin = -1;  /* Remember pin from repeat command */
	int pin_valid;

	char* bank_pin_param;
	char* p;

	/* use the previous pin if repeated */
	if(flag & CMD_FLAG_REPEAT) {
		pin_valid=1;
	}
	/* gpior bank pin - backward compatibility */
	else if(argc==3) {
		bank = simple_strtoul(argv[1],NULL,0);
		pin = simple_strtoul(argv[2],NULL,0);
		pin_valid=1;
	}
	/* gpior bank:pin */
	else if(argc==2) {
		bank_pin_param=strdup(argv[1]);
		/* get bank */
		if( (p = strtok(bank_pin_param, ":"))!=NULL ) {
			bank=simple_strtoul(p,NULL,0);
			/* get bit */
			if( (p = strtok(NULL, ":"))!=NULL ) {
				pin=simple_strtoul(p,NULL,0);
				pin_valid=1;
			}
		}
		free(bank_pin_param);
	}
	else {
		pin_valid=0;
	}

	/* bypass if not valid */
	if (!pin_valid) return -1;

	GPIO_dir(bank, pin, GPIO_DIR_IN);

	if (bank >= 0) {
		if (GPIO_in(bank,pin)) {
			printf("1\r\n");
			return 0;
		} else {
			printf("0\r\n");
			return 1;
		}
	} else {
		printf("read bank/pin first before repeat\r\n");
	}
	return 0;
}
U_BOOT_CMD(gpior, CONFIG_SYS_MAXARGS, 1, fun_gpior,
	"GPIO read",
	"BANK BIT [repeatable]\n"
	"gpior BANK:BIT [repeatable]"
);


/* GPIO write */
static int fun_gpiow(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	int bank;
	int pin;

	if (argc != 4) return -1;

	bank = simple_strtoul(argv[1],NULL,0);
	pin = simple_strtoul(argv[2],NULL,0);

	GPIO_out(bank, pin, simple_strtol(argv[3], NULL, 0));
	GPIO_dir(bank, pin, GPIO_DIR_OUT);

	return 0;
}
U_BOOT_CMD(gpiow, CONFIG_SYS_MAXARGS, 1, fun_gpiow,
	"GPIO write",
	"BANK BIT VALUE\n"
);


static int fun_gpiot(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	static int bank  = -1;
	static int pin = -1;
	struct pinstate p = {0};

	if (argc>1 && !(flag & CMD_FLAG_REPEAT)) {
		bank = simple_strtoul(argv[1],NULL,0);
		pin = simple_strtoul(argv[2],NULL,0);
	}

	if (argc != 0 && argc != 3) return -1;

	if (pin > 0) {
		p.bank = bank; p.pin = pin;
		GPIO_getstate(&p);
		p.oe = 1;
		p.v = !p.v;
		GPIO_setstate(&p);
	} else {
		printf("read bank/pin first before repeat\r\n");
	}
	return 0;
}
U_BOOT_CMD(gpiot, CONFIG_SYS_MAXARGS, 1, fun_gpiot,
	"GPIO toggle",
	"BANK BIT, can be repeated\n"
);

static void printgpio(unsigned bank, unsigned pin)
{
	struct pinstate p = { 0 };
	p.bank = bank;
	p.pin  = pin;
	GPIO_getstate(&p);
	printf("%u:%02u ", bank, pin);
	if (p.oe) printf("OUT ");
	else printf(" IN ");
	printf("%u,%u\n", p.v, p.inv);
}

static int do_gpiodump(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	unsigned bank, pin;

	for (bank = 0; bank < NBANKS; bank++) {
		for (pin = 0; pin < 32; pin++) {
			printgpio(bank, pin);
		}
	}

	return 0;
}
U_BOOT_CMD(
	gpiodump,	5,	1,	do_gpiodump,
	"GPIO dump",
	"Dump all gpio state\n"
);

#endif /* CONFIG_SPL_BUILD */
