/*
 * Copyright (C) 2010 Texas Instruments
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ----------------------------------------------------------------------------
 *
 */

#ifndef _DDR_DEFS_H
#define _DDR_DEFS_H

#include <config.h>
#include <asm/arch/hardware.h>

#ifdef CONFIG_TI816X_EVM_DDR3

	#define CONFIG_TI816X_DDR3_400 /* Values supported 400,531,675,796 */
	#define CONFIG_TI816X_DDR3_SW_LEVELING	/* Enable software leveling as part of DDR3 init*/


	/*
	* DDR3 force values.  These are board dependent
	*/

	/* EVM 400 MHz clock Settings
	* EVM has only a single RANK (chip select) */
	#define N_RANK			1

	/*
	* Invert clock adds an additional half cycle delay on the command
	* interface.  The additional half cycle, is usually meant to enable
	* leveling in the situation that DQS is later than CK on the board.  It
	* also helps provide some additional margin for leveling.
	*
	* For the EVM this is helping us with additional room for the write
	* leveling.  Since the dqs delays are very small.
	*/
	#define INVERT_CLOCK		1

	/*
	* CMD_SLAVE_RATIO determines where is the command placed with respect
	* to the clock edge.  This is a ratio, implying 0x100 is one cycle.
	* Ideally the command is centered so - this should be half cycle
	* delay (0x80).  But if invert clock is in use, an additional half
	* cycle must be added
	*/
	#define CMD_SLAVE_FROM_INV_CLOCK(i) (((i)==0) ? 0x80 : 0x100)
	#define CMD_SLAVE_RATIO		CMD_SLAVE_FROM_INV_CLOCK(INVERT_CLOCK)

	#ifdef TI816X_DDR3_PG_1_0
		/*
		* EMIF PHY allows for controlling write DQS delay w.r.t. clock.  The
		* value may be forced or use values determined from the leveling
		* process.  Since we are doing the leveling - these are actually
		* don't care and are not used.  The force is in delay units
		*/
		#define WR_DQS_FORCE_3	0x00000010
		#define WR_DQS_FORCE_2	0x00000010
		#define WR_DQS_FORCE_1	0x00000010
		#define WR_DQS_FORCE_0	0x00000010

		/*
		* EMIF PHY allows for controlling how much the read DQS must be
		* delayed to sample the data correctly.  Ideally this is about half a
		* cycle.  The force here is delay units.  The value here is in use -
		* as the current leveling process may not obtain this reliably.  The
		* value has been determined via various tests on the EVM and changing
		* this setting is no recomended.
		*/
		#define RD_DQS_FORCE_0		0x00000028
		#define RD_DQS_FORCE_1		0x00000028
		#define RD_DQS_FORCE_2		0x00000028
		#define RD_DQS_FORCE_3		0x00000028

		/*
		* read gate represents the round trip delay from command to read data
		* on the board + some package delay.  This is the period for which
		* the bust may be tristated between a write and a read command and
		* hence must not be sampled (gated off).  This is actually determined
		* via the read leveling process and hence this value is a don't care
		* for the EVM
		*/
		#define RD_GATE_FORCE_3	0x44
		#define RD_GATE_FORCE_2	0x44
		#define RD_GATE_FORCE_1	0x44
		#define RD_GATE_FORCE_0	0x44

	#endif

	/*
	* This represents the initial value for the leveling process.  The
	* value is a ratio - so 0x100 represents one cycle.  The real delay
	* is determined through the leveling process.
	*
	* During the leveling process, 0x20 is subtracted from the value, so
	* we have added that to the value we want to set.  We also set the
	* values such that byte3 completes leveling after byte2 and byte1
	* after byte0.
	*/
	#define WR_DQS_RATIO_0		0x20
	#define WR_DQS_RATIO_1		0x20
	#define WR_DQS_RATIO_2		0x20
	#define WR_DQS_RATIO_3		0x20

	#ifdef TI816X_DDR3_PG_1_0
		/*
		* read dqs ratio is only used in DDR2
		*/
		#define RD_DQS_RATIO_0		0x40
		#define RD_DQS_RATIO_1		0x40
		#define RD_DQS_RATIO_2		0x40
		#define RD_DQS_RATIO_3		0x40
	#endif

	/*
	* This represents the initial value for the leveling process.  The
	* value is a ratio - so 0x100 represents one cycle.  The real delay
	* is determined through the leveling process.
	*
	* During the leveling process, 0x20 is subtracted from the value, so
	* we have added that to the value we want to set.  We also set the
	* values such that byte3 completes leveling after byte2 and byte1
	* after byte0.
	*/
	#define RD_GATE_RATIO_0	0x20
	#define RD_GATE_RATIO_1	0x20
	#define RD_GATE_RATIO_2	0x20
	#define RD_GATE_RATIO_3	0x20

	#ifdef TI816X_DDR3_PG_1_0
		/*
		* currently there is an issue with the automatic training process for
		* DDR3 by setting the initial leveling ratios appropriately we are
		* able to work arround write leveling and read gate leveling.  How
		* ever the automatic process may not work well for the read eye
		* training (determining rd dqs delay).  To work arround this - we
		* leverage the pre-knowledge of a working RD DQS delay and make the
		* leveling process complete by forcing good and bad values
		* This is enabled via HACK_EYE_TRAINING
		*/
		#define HACK_EYE_TRAINING	0

		/*
		* only the rd dqs delay needs to be forced.  Others are determined via the leveling process
		*/
		#define USE_WR_DQS_FORCE	0
		#define USE_RD_DQS_FORCE	HACK_EYE_TRAINING
		#define USE_RD_GATE_FORCE	0

	#endif
	/*
	* data rate in MHz.  The DDR clock will be 1/2 of this value
	*/
	#define DDR_DATA_RATE		800

	#define USE_EMIF0		1
	#define USE_EMIF1		1

	/*
	* EMIF Paramters.  Refer the EMIF register documentation and the
	* memory datasheet for details
	*/
	/* For 400 MHz */
	#if defined(CONFIG_TI816X_DDR3_400)
		#define EMIF_TIM1    0x0CCCE524
		#define EMIF_TIM2    0x30308023
		#define EMIF_TIM3    0x009F82CF
		#define EMIF_SDREF   0x10000C30
		#define EMIF_SDCFG   0x62A41032
		#define EMIF_PHYCFG  0x0000010B

		#if defined(CONFIG_TI816X_DDR3_SW_LEVELING)
			/* These values are obtained from the CCS app */
			#define RD_DQS_GATE	0x12A
			#define RD_DQS		0x3B
			#define WR_DQS		0xA6
		#endif

	#endif	/* CONFIG_TI816X_DDR3_400 */

	/* For 531 MHz */
	#if defined(CONFIG_TI816X_DDR3_531)
		#define EMIF_TIM1    0x0EF136AC
		#define EMIF_TIM2    0x30408063
		#define EMIF_TIM3    0x009F83AF
		#define EMIF_SDREF   0x1000102E
		#define EMIF_SDCFG   0x62A51832
		#define EMIF_PHYCFG  0x0000010C

		#if defined(CONFIG_TI816X_DDR3_SW_LEVELING)
			/* These values are obtained from the CCS app */
			#define RD_DQS_GATE	0x13D
			#define RD_DQS		0x39
			#define WR_DQS		0xB4
		#endif

	#endif /* CONFIG_TI816X_DDR_531 */

	/* For 675 MHz */
	#if defined(CONFIG_TI816X_DDR3_675)
		#define EMIF_TIM1    0x13358875
		#define EMIF_TIM2    0x5051806C
		#define EMIF_TIM3    0x009F84AF
		#define EMIF_SDREF   0x10001491
		#define EMIF_SDCFG   0x62A63032
		#define EMIF_PHYCFG  0x0000010F

		#if defined(CONFIG_TI816X_DDR3_SW_LEVELING)
			/* These values are obtained from the CCS app */
			#define RD_DQS_GATE	0x196
			#define RD_DQS		0x39
			#define WR_DQS		0x91

		#endif

	#endif /* CONFIG_TI816X_DDR3_675 */

	/* For 796 MHz */
	#if defined(CONFIG_TI816X_DDR3_796)
		#define EMIF_TIM1   0x1779C9FE
		#define EMIF_TIM2   0x50608074
		#define EMIF_TIM3   0x009F857F
		#define EMIF_SDREF  0x10001841
		#define EMIF_SDCFG  0x62A73832
		#define EMIF_PHYCFG 0x00000110

		#if defined(CONFIG_TI816X_DDR3_SW_LEVELING)
			/* These values are obtained from the CCS app */
			#define RD_DQS_GATE	0x1B3
			#define RD_DQS		0x35
			#define WR_DQS		0x93

		#endif

	#endif /* CONFIG_TI816X_DDR_796 */


	#if defined(CONFIG_TI816X_DDR3_SW_LEVELING)
		#define WR_DQS_RATIO_BYTE_LANE3	((WR_DQS << 10) | WR_DQS)
		#define WR_DQS_RATIO_BYTE_LANE2	((WR_DQS << 10) | WR_DQS)
		#define WR_DQS_RATIO_BYTE_LANE1	((WR_DQS << 10) | WR_DQS)
		#define WR_DQS_RATIO_BYTE_LANE0	((WR_DQS << 10) | WR_DQS)

		#define WR_DATA_RATIO_BYTE_LANE3	(((WR_DQS + 0x40) << 10) | (WR_DQS + 0x40))
		#define WR_DATA_RATIO_BYTE_LANE2	(((WR_DQS + 0x40) << 10) | (WR_DQS + 0x40))
		#define WR_DATA_RATIO_BYTE_LANE1	(((WR_DQS + 0x40) << 10) | (WR_DQS + 0x40))
		#define WR_DATA_RATIO_BYTE_LANE0	(((WR_DQS + 0x40) << 10) | (WR_DQS + 0x40))

		#define RD_DQS_RATIO			((RD_DQS << 10) | RD_DQS)

		#define DQS_GATE_BYTE_LANE0		((RD_DQS_GATE << 10) | RD_DQS_GATE)
		#define DQS_GATE_BYTE_LANE1		((RD_DQS_GATE << 10) | RD_DQS_GATE)
		#define DQS_GATE_BYTE_LANE2		((RD_DQS_GATE << 10) | RD_DQS_GATE)
		#define DQS_GATE_BYTE_LANE3		((RD_DQS_GATE << 10) | RD_DQS_GATE)

	#endif	/* CONFIG_TI816X_DDR3_SW_LEVELING */

#endif	/* CONFIG_TI816X_EVM_DDR3 */

#ifdef CONFIG_TI816X_EVM_DDR2

	#define INVERT_CLK_OUT		0x0
	#define CMD_SLAVE_RATIO		0x80
	/*
	* DDR2 ratio values.  These are board dependent
	* obtained from sweep experiments
	*/

	/* EVM 400 MHz clock Settings */

	#define WR_DQS_RATIO_BYTE_LANE3	((0x4a << 10) | 0x4a)
	#define WR_DQS_RATIO_BYTE_LANE2	((0x4a << 10) | 0x4a)
	#define WR_DQS_RATIO_BYTE_LANE1	((0x4a << 10) | 0x4a)
	#define WR_DQS_RATIO_BYTE_LANE0	((0x4a << 10) | 0x4a)

	#define WR_DATA_RATIO_BYTE_LANE3	(((0x4a + 0x40) << 10) | (0x4a + 0x40))
	#define WR_DATA_RATIO_BYTE_LANE2	(((0x4a + 0x40) << 10) | (0x4a + 0x40))
	#define WR_DATA_RATIO_BYTE_LANE1	(((0x4a + 0x40) << 10) | (0x4a + 0x40))
	#define WR_DATA_RATIO_BYTE_LANE0	(((0x4a + 0x40) << 10) | (0x4a + 0x40))

	#define RD_DQS_RATIO			((0x40 << 10) | 0x40)

	#define DQS_GATE_BYTE_LANE0		((0x13a << 10) | 0x13a)
	#define DQS_GATE_BYTE_LANE1		((0x13a << 10) | 0x13a)
	#define DQS_GATE_BYTE_LANE2		((0x13a << 10) | 0x13a)
	#define DQS_GATE_BYTE_LANE3		((0x13a << 10) | 0x13a)

	/*
	* EMIF Paramters
	*/
	#define EMIF_TIM1    0xAAB15E2
	#define EMIF_TIM2    0x423631D2
	#define EMIF_TIM3    0x80032F
	#define EMIF_SDREF   0x10000C30
	#define EMIF_SDCFG   0x43801A3A  /* 32 bit ddr2, CL=6, CWL=5, 13 rows, 8 banks, 10 bit column, 2 CS */
	#define EMIF_PHYCFG  0x0000010B  /* local odt = 1, read latency = 11 (max = 12, min=6) */

#endif	/* CONFIG_TI816X_EVM_DDR2 */

#ifdef CONFIG_TI814X_EVM_DDR2
	/* ti814x specific settings to be added */
#endif

/* AM335X EMIF Register values For Netcomm products. */
#ifdef CONFIG_NTC_BETZY

	#if defined V_SDRAMTYPE_DDR3
		// DDR3 Memory setup. TODO: Clean this up - the type definitions don't belong here */
		/**
		* Encapsulates DDR DATA registers.
		*/

		struct ddr_data {
			unsigned long datardsratio0;	//	0xc8
			unsigned long datawdsratio0;	//	0xdc
			unsigned long datawiratio0;	//	0xf0
			unsigned long datagiratio0;	//
			unsigned long datafwsratio0;	//	0x108
			unsigned long datawrsratio0;	//
			unsigned long datauserank0delay;//
			unsigned long datadldiff0;	//

			unsigned long datardsratio1;	//	0xc8
			unsigned long datawdsratio1;	//	0xdc
			unsigned long datawiratio1;	//	0xf0
			unsigned long datagiratio1;	//
			unsigned long datafwsratio1;	//	0x108
			unsigned long datawrsratio1;	//
			unsigned long datauserank1delay;//
			unsigned long datadldiff1;	//
		};

		/**
		* Encapsulates DDR CMD control registers.
		*/
		struct cmd_control {
			unsigned long cmd0csratio;
			unsigned long cmd0csforce;
			unsigned long cmd0csdelay;
			unsigned long cmd0dldiff;
			unsigned long cmd0iclkout;
			unsigned long cmd1csratio;
			unsigned long cmd1csforce;
			unsigned long cmd1csdelay;
			unsigned long cmd1dldiff;
			unsigned long cmd1iclkout;
			unsigned long cmd2csratio;
			unsigned long cmd2csforce;
			unsigned long cmd2csdelay;
			unsigned long cmd2dldiff;
			unsigned long cmd2iclkout;
		};

		/* Structure containing shadow of important registers in EMIF
		* The calculation function fills in this structure to be later used for
		* initialization and DVFS
		*/
		struct emif_regs {
			u32 freq;
			u32 sdram_config_init;
			u32 sdram_config;
			u32 sdram_config2;
			u32 ref_ctrl;
			u32 sdram_tim1;
			u32 sdram_tim2;
			u32 sdram_tim3;
			u32 read_idle_ctrl;
			u32 zq_config;
			u32 temp_alert_config;
			u32 emif_ddr_phy_ctlr_1_init;
			u32 emif_ddr_phy_ctlr_1;
			u32 emif_ddr_ext_phy_ctrl_1;
			u32 emif_ddr_ext_phy_ctrl_2;
			u32 emif_ddr_ext_phy_ctrl_3;
			u32 emif_ddr_ext_phy_ctrl_4;
			u32 emif_ddr_ext_phy_ctrl_5;
			u32 emif_rd_wr_lvl_rmp_win;
			u32 emif_rd_wr_lvl_rmp_ctl;
			u32 emif_rd_wr_lvl_ctl;
			u32 emif_rd_wr_exec_thresh;
		};

		/* Micron MT41J128M16JT-125 */

// #define DDR3_303MHZ
//   28   24   20   16   12    8    4    0
// 0110 0001 1100 0000 0101 0010 1011 0010
// 61c052b2
// 0110 0001 1100 0000 0100 1010 1011 0010
// 61c04ab2
		#define MT_DDR		( 3 << 29)
		#define MT_IbankPos	( 0 << 27)
		#define MT_DDR_TERM	( 1 << 24)	//1
		#define MT_DDR2_DDQS	( 1 << 23)
		#define MT_DYN_ODT	( 2 << 21)	//2
		#define MT_DIS_DLL	( 0 << 20)	//0
		#define MT_SDRAM_DRV	( 0 << 18)	//0
		#define MT_CAS_WR_LTN	( 0 << 16)	//0
		#define MT_NARROW	( 1 << 14)
#ifdef DDR3_303MHZ
		#define MT_CAS_LTN	( 2 << 10)
#else
		#define MT_CAS_LTN	( 4 << 10)
#endif
		#define MT_ROW_SIZE	( 5 << 07)
		#define MT_I_BANK	( 3 << 04)
		#define MT_E_BANK	( 0 << 02)
		#define MT_PAGE_SIZE	( 2 << 00)

		#define MT41J128MJT125_EMIF_SDCFG  (MT_DDR|MT_IbankPos|MT_DDR_TERM|MT_DDR2_DDQS|MT_DYN_ODT|MT_DIS_DLL|MT_SDRAM_DRV|MT_CAS_WR_LTN|MT_NARROW|MT_CAS_LTN|MT_ROW_SIZE|MT_I_BANK|MT_E_BANK|MT_PAGE_SIZE)
		#define MT41J128MJT125_IOCTRL_VALUE		0x18B
		#define MT41J128MJT125_ZQ_CFG			0x50074BE4
		#define MT41J128MJT125_DLL_LOCK_DIFF		0x1

#ifdef DDR3_303MHZ

		#define MT41J128MJT125_EMIF_READ_LATENCY	0x6

		#define MT41J128MJT125_EMIF_TIM1		0x0668a3ab
		#define MT41J128MJT125_EMIF_TIM2		0x46337FDA
		#define MT41J128MJT125_EMIF_TIM3		0x501F830F
		#define MT41J128MJT125_EMIF_SDREF		0x0000093B
		#define MT41J128MJT125_INVERT_CLKOUT		0x0
		#define MT41J128MJT125_WR_DQS			0x46  // DATA0_WR_DQS_SLAVE_RATIO_0
		#define MT41J128MJT125_RD_DQS			0x3D  // DATA0_RD_DQS_SLAVE_RATIO_0
		#define MT41J128MJT125_PHY_FIFO_WE		0x9B  // DATA0_FIFO_WE_SLAVE_RATIO_0
		#define MT41J128MJT125_RATIO			0x80  // CMD0_CTRL_SLAVE_RATIO_0
		#define MT41J128MJT125_PHY_WR_DATA		0x81  // DATA0_WR_DATA_SLAVE_RATIO_0
#else

		/* 400 Mhz */
		#define MT41J128MJT125_EMIF_READ_LATENCY	0x100007

		#define MT41J128MJT125_EMIF_TIM1		0x0AAAD4DB
		#define MT41J128MJT125_EMIF_TIM2		0x266B7FDA
		#define MT41J128MJT125_EMIF_TIM3		0x501F867F

		#define MT41J128MJT125_EMIF_SDREF		0x00000C30	// 400Mhz
		//#define MT41J128MJT125_EMIF_SDREF		0x0000093B	// 303Mhz
		//#define MT41J128MJT125_EMIF_SDREF		0x0000024e	// 75Mhz
		//#define MT41J128MJT125_EMIF_SDREF		0x00000492	// 150 Mhz

		#define MT41J128MJT125_INVERT_CLKOUT		0x0
		#define MT41J128MJT125_WR_DQS			0x46 // DATA0_WR_DQS_SLAVE_RATIO_0
		#define MT41J128MJT125_RD_DQS			0x3C // DATA0_RD_DQS_SLAVE_RATIO_0
		#define MT41J128MJT125_PHY_FIFO_WE		0x99 // DATA0_FIFO_WE_SLAVE_RATIO_0
		#define MT41J128MJT125_RATIO			0x80 // CMD0_CTRL_SLAVE_RATIO_0
		#define MT41J128MJT125_PHY_WR_DATA		0x7c // DATA0_WR_DATA_SLAVE_RATIO_0

#endif

		#define MT41J128MJT125_WR_DQS_H			MT41J128MJT125_WR_DQS
		#define MT41J128MJT125_RD_DQS_H			MT41J128MJT125_RD_DQS
		#define MT41J128MJT125_PHY_FIFO_WE_H		MT41J128MJT125_PHY_FIFO_WE
		#define MT41J128MJT125_RATIO_H			MT41J128MJT125_RATIO
		#define MT41J128MJT125_PHY_WR_DATA_H		MT41J128MJT125_PHY_WR_DATA

		/* Micron MT41K256M16HA-125E */
		#define MT41K256M16HA125E_EMIF_READ_LATENCY	0x100007
		#define MT41K256M16HA125E_EMIF_TIM1		0x0AAAD4DB
		#define MT41K256M16HA125E_EMIF_TIM2		0x266B7FDA
		#define MT41K256M16HA125E_EMIF_TIM3		0x501F867F
		#define MT41K256M16HA125E_EMIF_SDCFG		0x61C05332
		#define MT41K256M16HA125E_EMIF_SDREF		0xC30
		#define MT41K256M16HA125E_ZQ_CFG		0x50074BE4
		#define MT41K256M16HA125E_DLL_LOCK_DIFF		0x1
		#define MT41K256M16HA125E_RATIO			0x80
		#define MT41K256M16HA125E_INVERT_CLKOUT		0x0
		#define MT41K256M16HA125E_RD_DQS		0x38
		#define MT41K256M16HA125E_WR_DQS		0x44
		#define MT41K256M16HA125E_PHY_WR_DATA		0x7D
		#define MT41K256M16HA125E_PHY_FIFO_WE		0x94
		#define MT41K256M16HA125E_IOCTRL_VALUE		0x18B

		#define	VTP_CTRL_READY			(0x1 << 5)
		#define VTP_CTRL_ENABLE			(0x1 << 6)
		#define VTP_CTRL_LOCK_EN		(0x1 << 4)
		#define VTP_CTRL_START_EN		(0x1)
		#define PHY_DLL_LOCK_DIFF			0x0			/* DATA0_DLL_LOCK_DIFF_0 */
		#define DDR_CKE_CTRL_NORMAL		0x1
		#define PHY_EN_DYN_PWRDN		(0x1 << 20)


	#else /* undef V_SDRAMTYPE_DDR3 */
		/* DDR2 memory setup */

		#define EMIF_SDMGT				0x80000000		/* EMIF0_0_SDRAM_MGMT_CTRL */

		#define	VTP_CTRL_READY			(0x1 << 5)
		#define VTP_CTRL_ENABLE			(0x1 << 6)
		#define VTP_CTRL_LOCK_EN		(0x1 << 4)
		#define VTP_CTRL_START_EN		(0x1)

		#define DDR_IOCTRL_VALUE		0x18B			/* DDR_CMD0_IOCTRL */
		#define DDR2_RATIO			0x80			/* CMDx_CTRL_SLAVE_RATIO_0 */
		#define	DDR2_INVERT_CLKOUT		0x00 			/* CMDx_PHY_INVERT_CLKOUT */

		#define CMD_FORCE			0x00			/* CMDx_CTRL_SLAVE_FORCE_0 */
		#define CMD_DELAY			0x00			/* CMDx_CTRL_SLAVE_DELAY_0 */
		#define DDR2_DLL_LOCK_DIFF		0x0			/* CMDx_DLL_LOCK_DIFF_0 */
		#define	DDR2_PHY_WRLVL			0x00			/* DATAx_WRLVL_INIT_RATIO_0 */
		#define	DDR2_PHY_GATELVL		0x00			/* DATAx_GATELVL_INIT_RATIO_0 */
		#define	DDR2_PHY_WR_DATA		0x40			/* DATAx_WR_DATA_SLAVE_RATIO_0 */
		#define	PHY_RANK0_DELAY			0x01			/* DATAx_RANK0_DELAYS_0 */
		#define PHY_DLL_LOCK_DIFF		0x0				/* DATA0_DLL_LOCK_DIFF_0 */

		/* Memory device specific registers */
		#define EMIF_SDCFG				0x438052B2		/* EMIF_SDCFG 	16 bit drr2, CL=4, 14 rows, 8 banks, 10 bit column, 1 CS */
		#define EMIF_SDCFG2				0			/* EMIF_SDCFG2 	External Bank Position = 0 (lower bits, so that rows are high) */
		#define EMIF_SDREF				2074			/* EMIF_SDREF  	tREFI(us) / (1/fDRR) (MHz) = 7.8 / (1/266) = 2074 */
		#define EMIF_READ_LATENCY			0x05			/* EMIF4_0_DDR_PHY_CTRL_1 = CL+1(+1) */

		/* Configured using AM335x DDR Calculation tool */
		#define EMIF_TIM1				0x0666B3D1		/* EMIF4_0_SDRAM_TIM_1 */
		#define EMIF_TIM2				0x142431CA		/* EMIF4_0_SDRAM_TIM_2 */
		#define	EMIF_TIM3				0x0000021F		/* EMIF4_0_SDRAM_TIM_3 */

		/* Configured using Ratio Seed Spreadsheet
		*
		* using DDR_CK_trace=0.56 DDR_DQSx_trace=0.63
		*
		* DDR2_WR_DQS				0x00	However these do not work correctly, maybe length was measured incorrectly ?
		* DDR2_RD_DQS				0x40
		* DDR2_PHY_FIFO_WE			0x50
		*
		*/
		#define	DDR2_WR_DQS				0x00			/* DATAx_WR_DQS_SLAVE_RATIO_0 */
		#define DDR2_RD_DQS				0x40			/* DATAx_RD_DQS_SLAVE_RATIO_0 */
		#define DDR2_PHY_FIFO_WE			0x4E			/* DATAx_FIFO_WE_SLAVE_RATIO_0 */

	#endif /* V_SDRAMTYPE_DDR3 */

#elif defined CONFIG_NTC_BETZY_V1_0

	/* Configuration based on http://processors.wiki.ti.com/index.php/AM335x_EMIF_Configuration_tips
	*
	* Board memory = 1 * MT46H64M16LFBF-5 IT:B CLK=200MHz
	*
	*/

	#define EMIF_SDMGT				0x80000000		/* EMIF0_0_SDRAM_MGMT_CTRL */

	#define	VTP_CTRL_READY			(0x1 << 5)
	#define VTP_CTRL_ENABLE			(0x1 << 6)
	#define VTP_CTRL_LOCK_EN		(0x1 << 4)
	#define VTP_CTRL_START_EN		(0x1)

	/* These registers have fixed values according to AM335x_EMIF_Configuration_tips */
	#define DDR_IOCTRL_VALUE		0x18B			/* DDR_CMD0_IOCTRL */
	#define DDR2_RATIO				0x80			/* CMDx_CTRL_SLAVE_RATIO_0 */
	#define	DDR2_INVERT_CLKOUT		0x00 			/* CMDx_PHY_INVERT_CLKOUT */

	/* Copied from beaglebone config, these are not described in the EMIF configuration tips */
	#define CMD_FORCE				0x00			/* CMDx_CTRL_SLAVE_FORCE_0 */
	#define CMD_DELAY				0x00			/* CMDx_CTRL_SLAVE_DELAY_0 */
	#define DDR2_DLL_LOCK_DIFF		0x0				/* CMDx_DLL_LOCK_DIFF_0 */
	#define	DDR2_PHY_WRLVL			0x00			/* DATAx_WRLVL_INIT_RATIO_0 */
	#define	DDR2_PHY_GATELVL		0x00			/* DATAx_GATELVL_INIT_RATIO_0 */
	#define	DDR2_PHY_WR_DATA		0x40			/* DATAx_WR_DATA_SLAVE_RATIO_0 */
	#define	PHY_RANK0_DELAY			0x01			/* DATAx_RANK0_DELAYS_0 */
	#define PHY_DLL_LOCK_DIFF		0x0				/* DATA0_DLL_LOCK_DIFF_0 */

	/* Memory device specific registers */
	#define EMIF_SDCFG				0x20004EA2		/* EMIF_SDCFG 	16 bit lpddr1, CL=3, 14 rows, 4 banks, 10 bit column, 1 CS */
	#define EMIF_SDCFG2				0x08000000		/* EMIF_SDCFG2 	External Bank Position = 1 */
	#define EMIF_SDREF				1560			/* EMIF_SDREF  	tREFI(us) / (1/fDRR) (MHz) = 7.8 / (1/200) = 1560 */
	#define EMIF_READ_LATENCY		0x04			/* EMIF4_0_DDR_PHY_CTRL_1 = CL+1(+1) */

	/* Configured using AM335x DDR Calculation tool */
	#define EMIF_TIM1				0x04447289		/* EMIF4_0_SDRAM_TIM_1 */
	#define EMIF_TIM2				0x10160580		/* EMIF4_0_SDRAM_TIM_2 */
	#define	EMIF_TIM3				0x000000E7		/* EMIF4_0_SDRAM_TIM_3 */

	/*#define EMIF_TIM1		0x0666B3D6
	#define EMIF_TIM2		0x143731DA
	#define	EMIF_TIM3		0x00000347 */

	/* Configured using Ratio Seed Spreadsheet
	*
	* using DDR_CK_trace=0.56 DDR_DQSx_trace=0.63
	*
	* DDR2_WR_DQS				0x00	However these do not work correctly, maybe length was measured incorrectly ?
	* DDR2_RD_DQS				0x40
	* DDR2_PHY_FIFO_WE			0x50
	*
	*/
	#define	DDR2_WR_DQS				0x00			/* DATAx_WR_DQS_SLAVE_RATIO_0 */
	#define DDR2_RD_DQS				0x12			/* DATAx_RD_DQS_SLAVE_RATIO_0 */
	#define DDR2_PHY_FIFO_WE		0x80			/* DATAx_FIFO_WE_SLAVE_RATIO_0 */


#elif defined CONFIG_AM335X

	#define EMIF_SDMGT		0x80000000
	#define EMIF_SDRAM		0x00004650
	#define EMIF_PHYCFG		0x2
	#define DDR_PHY_RESET		(0x1 << 10)
	#define DDR_FUNCTIONAL_MODE_EN	0x1
	#define DDR_PHY_READY		(0x1 << 2)
	#define	VTP_CTRL_READY		(0x1 << 5)
	#define VTP_CTRL_ENABLE		(0x1 << 6)
	#define VTP_CTRL_LOCK_EN	(0x1 << 4)
	#define VTP_CTRL_START_EN	(0x1)
	#define DDR2_RATIO		0x80	/* for mDDR */
	#define CMD_FORCE		0x00	/* common #def */
	#define CMD_DELAY		0x00
	#if	(CONFIG_AM335X_EVM_IS_13x13 == 1)
		#define EMIF_READ_LATENCY	0x05
		#define EMIF_TIM1		0x04446249
		#define EMIF_TIM2		0x101731C0
		#define EMIF_TIM3		0x137
		#define EMIF_SDCFG		0x20004EA3
		#define EMIF_SDREF		0x57c
		#define	DDR2_DLL_LOCK_DIFF	0x4
		#define DDR2_RD_DQS		0x40
		#define DDR2_PHY_FIFO_WE	0x56
	#else
		#define EMIF_READ_LATENCY	0x04
		#define EMIF_TIM1		0x0666B3D6
		#define EMIF_TIM2		0x143731DA
		#define	EMIF_TIM3		0x00000347
		#define EMIF_SDCFG		0x43805332
		#define EMIF_SDREF		0x0000081a
		#define DDR2_DLL_LOCK_DIFF	0x0
		#define DDR2_RD_DQS		0x12
		#define DDR2_PHY_FIFO_WE	0x80
	#endif
	#define	DDR2_INVERT_CLKOUT	0x00
	#define	DDR2_WR_DQS		0x00
	#define	DDR2_PHY_WRLVL		0x00
	#define	DDR2_PHY_GATELVL	0x00
	#define	DDR2_PHY_WR_DATA	0x40
	#define	PHY_RANK0_DELAY		0x01
	#define PHY_DLL_LOCK_DIFF	0x0
	#define DDR_IOCTRL_VALUE	0x18B
#endif /* CONFIG_AM335X */

#endif  /* _DDR_DEFS_H */
