/*************
 *
 * Filename:    ra_mc77xximgmgmt.c
 *
 * Purpose:     MC77xx Image Management application
 *
 * Copyright: © 2013 Sierra Wireless Inc., all rights reserved
 *
 **************/
//#define _GNU_SOURCE

#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <termios.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdbool.h>
                 
extern BOOL ra_IsGobiDevice(void);
bool ra_IsValidImageFileExist(CHAR *pImgPath);
                     
/****************************************************************
*                       DEFINES
****************************************************************/
//#define MC77xxDBG
#define ENTER_KEY          0x0A
#define OPTION_LEN         4
#define CORRUT_FW_IMG      39
#define SUCCESS            0
#define MAX_IMAGE_PATH     514
#define DEV_NODE_SZ        256
#define DEV_KEY_SZ         16
#define FALSE              0
#define TRUE               1

/****************************************************************
 *                    GLOBAL DATA
 ****************************************************************/

/* firmware download */
static BOOL fwdwlcomplete = FALSE;
static BOOL fwdwfail = FALSE;
static BYTE devicemode = DCS_DEVICE_MODE_DISCONNECTED;

/* macros*/
#define rcprint(s, u) syslog(LOG_USER, "%s: rc = 0x%lX, %s", s, u)

/****************************************************************
*                       FUNCTIONS
****************************************************************/

struct cwefwinfo{
    CHAR amss[128];
    CHAR boot[128];
    CHAR pri[128];
};



/*
 * Name:     ra_FirmwareDwldCbk
 *
 * Purpose:  Firmware download completion callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void ra_FirmwareDwldCbk(ULONG status)
{
    if (0 == status)
    {
        fprintf( stderr, "\nFirmware download complete; waiting for device...\n" );
    }
    else
    {
        fprintf( stderr, "\nFirmware download failed!\n" );
        fwdwfail = TRUE;
    }

    /* Unsubscribe from the callback */
    SetFwDldCompletionCbk(NULL);
    fwdwlcomplete = TRUE;
}

/*
 * Name:     cweFwDisplay
 *
 * Purpose:  CWE image info. display
 *
 * Return:   None
 *
 * Notes:    none
 */
static void ra_cweFwDisplay(struct cwefwinfo *pfw,CHAR *returnStr)
{
    /* QMI services are available only if the application is running */
    if ( devicemode == DCS_DEVICE_MODE_READY )
    {
        if( eQCWWAN_ERR_NONE ==
            GetFirmwareRevisions( (BYTE)sizeof(pfw->amss),
                                  pfw->amss,
                                  (BYTE)sizeof(pfw->boot),
                                  pfw->boot,
                                  (BYTE)sizeof(pfw->pri),
                                  pfw->pri ) )
        {
            sprintf( returnStr,
                     "\nCWE Image Fields\n"\
                     "----------------\n"\
                     "BOOT Version: %s\nAMSS Version: %s\nPRI Version: %s\n",
                     pfw->boot, pfw->amss, pfw->pri );
        }
        else
        {
            fprintf( stderr, "AMSS: %s\nBOOT: %s\nPRI: %s\n", "", "", "");
        }
    }
   
}   



/*
 * Name:     ra_GetDeviceImageInfo
 *
 * Purpose:  Get the information about the image running on the device.
 *
 * Return:   None
 *
 * Notes:    none
 */
void ra_GetDeviceImageInfo(CHAR *returnStr)
{
    struct cwefwinfo    cwe;
    
    SLQSGetDeviceMode ((BYTE *)&devicemode);
    ra_cweFwDisplay( &cwe ,returnStr);
}

/*
 * Name:     FirmwareDownloader
 *
 * Purpose:  Download a CWE image to the device
 *
 * Return:   None
 *
 * Notes:    none
 */
bool ra_FirmwareDownloader_MC77XX(CHAR *pImagePath)
{
    struct qmifwinfo_s  spkg;
    long                rclocal, rcfw, rc;

    /* Reset the firmware download completion and device ready flags */
    fwdwlcomplete = FALSE;
    fwdwfail = FALSE;
   
   
    if(true == ra_IsGobiDevice()|| true == IsSierraGobiDevice())
        return false;
        
   
    if (!ra_IsValidImageFileExist(pImagePath))
        return false;
    
     /* to update devicemode */
     rc = SLQSGetDeviceMode ((BYTE *)&devicemode);
    
    fprintf( stderr, "\n-- Active Device Image Before Download --\n");
    rc = SLQSGetFirmwareInfo( &spkg );
    if( eQCWWAN_ERR_NONE != rc )
    {
        /* no AMSS SPKG support */
        memset( &spkg.dev.s, 0, sizeof(spkg.dev.s) );
    }

    /* Get the information about the image located at specified path */
    rclocal = SLQSGetImageInfo( pImagePath, &spkg );

    /* Display the image information */
    if( eQCWWAN_ERR_NONE == rclocal )
    {
        fprintf( stderr,  "\n-- Preparing to Download SPKG CWE image --\n\n" );
    }
    else
    {
        fprintf( stderr,  "\n-- Preparing to Download CWE image --\n" );
    }

    rc = SetFwDldCompletionCbk( ra_FirmwareDwldCbk );
    if( SUCCESS != rc )
    {
        fprintf( stderr, "REGISTRATION FAILED - Firmware Download Completion Callback\n"\
                         "Failure Code: %ld\n", rc );
        return false;
    }

    /* Start downloading the firmware */
    rc = UpgradeFirmware2k( pImagePath );
    if( eQCWWAN_ERR_NONE != rc )
    {
        /* some firmware of MC7700 may reboot immediately after receiving
         * set firmware id command without sending out respond,
         * neglect this error to bypass issue*/
        if(eQCWWAN_ERR_NO_DEVICE !=rc )
        {
            fprintf( stderr, "Firmware Download Failed\n"\
                             "Failure Code: %ld\n", rc );
            return false;
        }
    }

    fprintf( stderr, "\n\nDownloading Firmware");
    while( !fwdwlcomplete )
    {
        /* Display "." while firmware downloads */
        fprintf( stderr, ".");
        sleep(2);
    }

    if (fwdwfail)
    {
        fprintf( stderr, "Firmware Download Failed\n");
        return false;
    }

    /* Give the firmware time to apply any additional CWE files that are
     * not firmware boot and/or application images
     */
    fprintf(stderr, "Applying SPKG updates - please wait 20 seconds...\n\n");
    sleep(20);

    rc = SLQSGetDeviceMode ((BYTE *)&devicemode);

    /* Get the information about the image loaded on the device */
    rcfw = SLQSGetFirmwareInfo( &spkg );

    /* Display the information of the newly loaded image on the device */
    fprintf( stderr, "\n-- Active Device Image After Download --\n");

    if( eQCWWAN_ERR_NONE != rcfw )
    {
        /* no AMSS SPKG support */
        return false;
    }

    fprintf( stderr, "\nExiting Firmware Downloader\n" );
    return true;
}


/*
 * Name:     ra_IsValidImageFileExist
 *
 * Purpose:  check whether the provided path is valid or not
 *
 * Return:   True if two images are the same, False otherwise.
 *
 * Notes:    none
 */
bool ra_IsValidImageFileExist(CHAR *pImgPath)
{
    struct  qmifwinfo_s spkg;
    long    rc;


    if( 0 == strnlen(pImgPath, sizeof(pImgPath)) )
    {
        /* empty string - return to main menu */
        return false;
    }

    /* Get the information about the image located at specified path */
    rc = SLQSGetImageInfoMC77xx( pImgPath, &spkg );

    if( eQCWWAN_ERR_NONE != rc )
    {
        fprintf( stderr,
                 "\nValid SPKG CWE file was not found. "\
                 "No fields are printed for non-SPKG CWE files.\n" );
        memset(&spkg.dev.s, 0, sizeof(spkg.dev.s) );
        return false;
    }
    return true;
}

