/*************
 *
 * Filename:    ra_mc73xximgmgmt.c
 *
 * Purpose:     MC73XX Image Management application
 *
 * Copyright: © 2013 Sierra Wireless Inc., all rights reserved
 *
 **************/
#define _SVID_SOURCE
#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <syslog.h>
#include <stdbool.h>

#define IMG_FW_TYPE_CWE 2
#define IMG_PRI_TYPE_NVU 3
#define TRUE 1
#define FALSE 0

static BOOL fwDwlDone = FALSE;
static BOOL fwdwfail = FALSE;

/*
 * Name:     ra_DevStateChgCbk
 *
 * Purpose:  Device State change callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void ra_DevStateChgCbk_9x15(eDevState devstatus)
{
    /* If device is ready to communicate */
    if( devstatus ==  DEVICE_STATE_READY )
    {
        fprintf( stderr, "\nDevice Ready\n" );

        /* Unsubscribe from the callback */
        SetDeviceStateChangeCbk(NULL);
        fwDwlDone = TRUE;
    }
}

/*
 * Name:     ra_FwDwldCbk_9x15
 *
 * Purpose:  Firmware download completion callback
 *
 * Return:   None
 *
 * Notes:    none
 */
void ra_FwDwldCbk_9x15(ULONG status)
{
    if ( eQCWWAN_ERR_NONE == status)
    {
        fprintf( stderr, "\nFirmware Download Completed" );
    }
    else if (eQCWWAN_ERR_SWIIM_FIRMWARE_NOT_DOWNLOADED == status)
    {
        fprintf( stderr, "\nFirmware Not Downloaded" );
        fwdwfail = TRUE;
    }
    else
    {
        fwdwfail = TRUE;
    }

    /* set firmware complete to true */
    fwDwlDone = TRUE;

    /* Unsubscribe from the callback */
    SetFwDldCompletionCbk(NULL);
}


/*
 * Name:     ra_FwDloader_9x15_NoGIM
 *
 * Purpose:  Download firmware on a 9x15 based device using non-GIM mode
 *
 * Return:   None
 *
 * Notes:    none
 */
bool ra_FwDloader_9x15_NoGIM( CHAR *pImagePath )
{
    ULONG rc = 0;
    CHAR  completeImagePath[512];
    ULONG len = 0;

    /* Reset the firmware download completion and device ready flags */
    fwDwlDone = FALSE;
    fwdwfail = FALSE;

    rc = SetFwDldCompletionCbk( ra_FwDwldCbk_9x15 );
    if( eQCWWAN_ERR_NONE != rc )
    {
        fprintf( stderr, "ERROR: Failed to register firmware Download Completion Callback\n"\
                         "Failure Code: %ld\n", rc );
        return false;
    }

    /* Concatenate image Path with '/' */
    memset( completeImagePath, 0, sizeof(completeImagePath) );
    strncpy( completeImagePath,
             pImagePath,
             strlen(pImagePath) );
    len = strlen(completeImagePath);
    CHAR *pr = &completeImagePath[len - 1];
    if( *pr != '/' )
        *(pr + 1) = '/';

    /* Start downloading the firmware */
    rc = UpgradeFirmware2k( completeImagePath );
    if( eQCWWAN_ERR_NONE != rc )
    {
        /* some firmware of MC7700 may reboot immediately after receiving
         * set firmware id command without sending out respond,
         * neglect this error to bypass issue*/
        if(eQCWWAN_ERR_NO_DEVICE !=rc )
        {
            fprintf( stderr, "Firmware Download Failed\n"\
                             "Failure Code: %ld\n", rc );
            exit(0);
        }
        return false;
    }

    fprintf( stderr, "\n\nDownloading Firmware");
    while( !fwDwlDone )
    {
        /* Display "." while firmware downloads */
        fprintf( stderr, ".");
        sleep(2);
    }

    if (fwdwfail)
    {
        fprintf( stderr, "ERROR: Firmware Download Failed\n");
        return false;
    }

    fprintf( stderr, "INFO: Firmware Download Succeeded\n");
    return true;
}

/*
 * Name:     ra_FwDloader_9x15_GIM
 *
 * Purpose:  Download firmware on a 9x15 based device using GIM mode
 *
 * Return:   None
 *
 * Notes:    none
 */
bool ra_FwDloader_9x15_GIM( CHAR *pImagePath )
{
    ULONG              resultCode = 0;
    struct qmifwinfo_s FwImgInfo, NVImgInfo;
    CurrentImgList     CurrImgList;
    CurrImageInfo      currImgInfo[5];
    BYTE               numEntries  = 5;
    CHAR               priVer[16];
    CHAR               pkgVer[16];
    CHAR               carrier[16];
    CHAR               AppVer[16];
    CHAR               completeImagePath[512];
    ULONG              len = 0;

    CurrImgList.pCurrImgInfo = currImgInfo;
    CurrImgList.numEntries   = numEntries;

    memset( completeImagePath, 0, sizeof(completeImagePath) );

    strncpy( completeImagePath,
             pImagePath,
             strlen(pImagePath) );
    len = strlen(completeImagePath);
    CHAR *pr = &completeImagePath[len - 1];
    if( *pr != '/' )
        *(pr + 1) = '/';

    /* Get Firmware image details */
    resultCode = SLQSGetImageInfo_9x15( completeImagePath,
                                        IMG_FW_TYPE_CWE,
                                        &FwImgInfo.dev.s );

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr,
                 "Error: Could not get details from the firmware image at the path: %s\n",
                 completeImagePath );
        fprintf( stderr, "Failure Code : %lu\n", resultCode );
        return false;
    }

    /* Get PRI image details */
    if( eQCWWAN_ERR_NONE == resultCode )
    {
        resultCode = SLQSGetImageInfo_9x15( completeImagePath,
                                            IMG_PRI_TYPE_NVU,
                                            &NVImgInfo.dev.s );
    }

    if ( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr,
                 "Error: Could not get details from the PRI image at the path: %s\n",
                 completeImagePath );
        fprintf( stderr, "Failure Code : %lu\n", resultCode );
        return false;
    }

    /* Save the firmware details for the image being downloaded */
    /* Initialize variables */
    memset( priVer, 0, sizeof(priVer) );
    memset( pkgVer, 0, sizeof(pkgVer) );
    memset( carrier, 0, sizeof(carrier) );
    memset( AppVer, 0, sizeof(AppVer) );

    /* Retrieve PRI version, pkg version, carrier from PRI image */
    strncpy( priVer,
             NVImgInfo.dev.s.priversion_str,
             strlen(NVImgInfo.dev.s.priversion_str) );

    strncpy( pkgVer,
             NVImgInfo.dev.s.packageid_str,
             strlen(NVImgInfo.dev.s.packageid_str) );

    strncpy( carrier,
             NVImgInfo.dev.s.carrier_str,
             strlen(NVImgInfo.dev.s.carrier_str) );

    /* Retrieve appversion( firmware version ) from fw image */
    strncpy( AppVer,
             FwImgInfo.dev.s.appversion_str,
             strlen(FwImgInfo.dev.s.appversion_str) );

    /* Reaching here means, image has been found */
    /* Subscribe to Device State Change callback */
    resultCode = SetDeviceStateChangeCbk(ra_DevStateChgCbk_9x15);
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr, "ERROR: Failed to register to device State Callback\n"\
                         "Failure Code: %ld\n", resultCode );
        return false;
    }

    /* Subscribe to Firmware Download completion callback */
    resultCode = SetFwDldCompletionCbk(ra_FwDwldCbk_9x15);
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr, "ERROR: Failed to register firmware Download Completion Callback\n"\
                         "Failure Code : %lu\n", resultCode );
        return false;
    }

    /* Start downloading the firmware */
    resultCode = UpgradeFirmware2k( completeImagePath );
    if( eQCWWAN_ERR_NONE != resultCode )
    {
        fprintf( stderr, "ERROR: Firmware Download Failed\n"\
                         "Failure Code : %lu\n", resultCode );

        /* Deregister device state change and firmware download completion
         * callback.
         */
        SetFwDldCompletionCbk( NULL );
        SetDeviceStateChangeCbk( NULL );
        return false;
    }

    /* Keep displaying "." until fimrware downloads complete */
    fprintf( stderr, "Downloading Firmware. . .");
    fwDwlDone = FALSE;
    while( !fwDwlDone )
    {
        fprintf( stderr, " .");
        sleep(1);
    }

    /* Intialize current running image structure */
    memset( (void *)&currImgInfo, 0, sizeof( currImgInfo ) );

    resultCode = SLQSSwiGetFirmwareCurr( &CurrImgList );
    if( eQCWWAN_ERR_NONE != resultCode)
    {
        fprintf( stderr, "ERROR: Failed to get firmware details after download\n"\
                         "Failure Code: %ld\n", resultCode );
        return false;
    }

    /* Check if the firmware download is success */
    /* Compare PRI, package, carrier and app version strings */
    if( 0 != strncmp( CurrImgList.priver, priVer, sizeof(priVer) ) ||
        0 != strncmp( CurrImgList.pkgver, pkgVer, sizeof(pkgVer) ) ||
        0 != strncmp( CurrImgList.carrier, carrier, sizeof(carrier) ) ||
        0 != strncmp( CurrImgList.fwvers, AppVer, sizeof(AppVer) ) )
    {
        fprintf( stderr, "ERROR: Firmware Upgrade Failed!!!\n" );
        return false;
    }
    else
    {
        fprintf( stderr, "INFO: Firmware Upgrade successful!!!\n" );

        /* Print the details of firmware downloaded */
        fprintf( stderr, "Firmware after downloaded \n" );
        fprintf( stderr, "PRI version: %s\n", CurrImgList.priver );
        fprintf( stderr, "Pkg version: %s\n", CurrImgList.pkgver );
        fprintf( stderr, "Firmware version: %s\n", CurrImgList.fwvers );
        fprintf( stderr, "Carrier : %s\n", CurrImgList.carrier );
        return true;
    }
}
/*
 * Name:     DownloadImage
 *
 * Purpose:  List all the images present on the device, prompt the user for the
 *           image path on the host, display their information, prompt again for
 *           the index of the image to be downloaded and then download that
 *           image to the device.
 *
 * Return:   none
 *
 * Notes:    none
 */
bool ra_FirmwareDownloader_MC73XX(CHAR *InputImagePath)
{
    ULONG              resultCode = 0;
    CurrentImgList     CurrImgList;
    CurrImageInfo      currImgInfo[5];
    BYTE               numEntries  = 5;
    CurrImgList.pCurrImgInfo = currImgInfo;
    CurrImgList.numEntries   = numEntries;
    bool               updateResult = false;

    /* There are 2 possible scenario's determined by calling SLQSSwiGetFwCurr
     * 1) Device does not support Gobi IM - In this case use same procedure as
     *    FwDloader_9x00. In this case, we need only one file( spkg format).
     * 2) Device supports GIM but data returned is blank. Use normal procedure. */
    resultCode = SLQSSwiGetFirmwareCurr( &CurrImgList );
    if( eQCWWAN_ERR_NONE != resultCode)
    {
        /* Assume that device is in non GIM( Gobi Image Management ) Mode.
         * Device issues can be detected on subsequent SLQS API calls */
        updateResult = ra_FwDloader_9x15_NoGIM( InputImagePath );
    }
    else
    {
        fprintf( stderr, "Firmware before download \n" );
        fprintf( stderr, "PRI version: %s\n", CurrImgList.priver );
        fprintf( stderr, "Pkg version: %s\n", CurrImgList.pkgver );
        fprintf( stderr, "Firmware version: %s\n", CurrImgList.fwvers );
        fprintf( stderr, "Carrier : %s\n", CurrImgList.carrier );

        /* Device supports GIM. Print the current firmware details. */
        updateResult = ra_FwDloader_9x15_GIM( InputImagePath );
    }

    return updateResult;
}

