/*************
 *
 * Filename:    main.c
 *
 * Purpose:     Main module of the Sierra Wireless Air Vantage Device Management Tree
 *              adapter library.  This file contains the start-up and shut-down code
 *              for the adapter.
 *
 * Copyright: © 2013 Sierra Wireless Inc., all rights reserved
 *
 **************/

#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>



/****************************************************************
*                       DEFINES
****************************************************************/
#define RA_DEBUG    /* Comment this out to turn off debug printf() calls */
#define SUCCESS            0
#define DEV_NODE_SZ        256
#define DEV_KEY_SZ         16


/****************************************************************
*                       TYPEDEFS
****************************************************************/

/* Device information structure */
typedef struct
{
  CHAR deviceNode[DEV_NODE_SZ];
  CHAR deviceKey[DEV_KEY_SZ];
}
device_info_t;


/****************************************************************
 *                    GLOBAL DATA
 ****************************************************************/

/* device connectivity */
static device_info_t DeviceList[1] = { { {'\0'}, {'\0'} } };

/* macros*/
#define rcprint(s, u) syslog(LOG_USER, "%s: rc = 0x%lX, %s", s, u)


/* ***************************************************************
*                       INTERNAL FUNCTIONS
****************************************************************/

/**
 * Fetch the current operating mode of the modem device.
 * 
 * @return The device mode (see qaGobiApiDcs.h).
 */
enum eAPI_DCS_GET_DEVICE_MODE ra_GetDeviceMode(void)
{
    /**
     * @todo    Consider optimizing by registering for callbacks from the SLQS
     *          when the device changes modes, and cache the device mode at that time.
     */

    BYTE deviceMode;
    ULONG resultCode = SLQSGetDeviceMode (&deviceMode);
    if (resultCode != eQCWWAN_ERR_NONE)
    {
        fprintf(stderr, "SLQSGetDeviceMode() failed (returned %lu).\n", resultCode);
        deviceMode = DCS_DEVICE_MODE_DISCONNECTED;
    }
    
    return deviceMode;
}



/*
 * Name:     ra_StartSDK
 *
 * Purpose:  It starts the SDK by setting the SDK path, enumerates the device
 *           and connects to the SDK.
 *
 * Return:   SUCCESS on successfully starting SDK, else error code
 *
 * Notes:    none
 */
static ULONG ra_StartSDK()
{
    ULONG rc = 0;
    BYTE  devicesSize = 1;

    /* Set SDK image path */
    if( eQCWWAN_ERR_NONE != (rc = SetSDKImagePath("bin/slqssdk")) )
    {
        return rc;
    }

    /* Establish APP<->SDK IPC */
    if( eQCWWAN_ERR_NONE != (rc = SLQSStart_AVAgent()) )
    {
        return rc;
    }

    /* Can enumerate and connect only if device is in Application mode */
    if ( ra_GetDeviceMode() == DCS_DEVICE_MODE_READY )
    {
        /* Enumerate the device */
        rc = QCWWAN2kEnumerateDevices( &devicesSize, (BYTE *)DeviceList );

        /* Return if there is an error */
        if( SUCCESS != rc )
        {
            return rc;
        }

#ifdef RA_DEBUG
    fprintf( stderr,  "#devices: %d\ndeviceNode: %s\ndeviceKey: %s\n",
            devicesSize,
            DeviceList->deviceNode,
            DeviceList->deviceKey );
#endif

        /* Connect to the SDK */
        rc = QCWWANConnect( DeviceList->deviceNode, DeviceList->deviceKey );

        if (SUCCESS != rc)
        {
            return rc;
        }
    }

    return rc;
}

void ra_init_main(void)
{
    
    static const char* deviceModeStr = "unknown";

    ra_StartSDK();
    
    switch(ra_GetDeviceMode())
    {
        case DCS_DEVICE_MODE_READY:
            deviceModeStr = "ready";
            break;
            
        case DCS_DEVICE_MODE_BOOT_READY:
            deviceModeStr = "boot and hold";
            break;

        case DCS_DEVICE_MODE_DISCONNECTED:
            deviceModeStr = "disconnected";
            break;

        default:
            break;
    };

    fprintf( stderr, "\nRunning with device in %s mode\n", deviceModeStr);    
}


