/*************
 *
 * Filename:    cellular.c
 *
 * Purpose:     Module of the Sierra Wireless Air Vantage Device Management Tree
 *              adapter library that implements the cellular sub-tree parameters
 *              (i.e., the parameters under "system.cellular".
 *
 * Copyright: © 2012 Sierra Wireless Inc., all rights reserved
 *
 **************/

#include "SWIWWANCMAPI.h"
#include "qmerrno.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include "main.h"
#include "devtree_v2.h"

#define nMaxStrLen              0xFF

/* Check the current device mode */
#define DEV_CHECK \
switch (main_GetDeviceMode()) \
{ \
    case DCS_DEVICE_MODE_DISCONNECTED: \
        return DT2_RESULT_DEVICE_NOT_FOUND; \
    case DCS_DEVICE_MODE_BOOT_READY: \
        return DT2_RESULT_WRONG_MODE; \
    case DCS_DEVICE_MODE_READY: \
        break; \
} 

/** Get APN Override.
 * 
 * This function is used to fetch the current APN override setting.  This is a writeable
 * string that can be used to override the default APN to be used by the device.
 * 
 * The APN is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined. 
 */
dt2_ResultCode_t dt2_GetApnOverride
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{

    DEV_CHECK
    
    /* Make sure the size isn't large enough to overflow a BYTE. */
    if (buffSize > 0xFF)
    {
        buffSize = 0xFF;
    }
    
    /** @todo   Figure out the real requirements for this.  For now, I'm just using
     *          the APN in the default profile for UMTS.
     */

    const ULONG profileType = 0;  /* 0 = UMTS */
    
    ULONG pdpType;
    ULONG ipAddress;
    ULONG primaryDNS;
    ULONG secondaryDNS;
    ULONG authentication;
    CHAR  name[30];  /* Don't care about this, so this is an arbitrary length. */
    BYTE  nameSize = sizeof(name) - 1;
    CHAR* apnNamePtr = stringPtr;
    BYTE  apnSize = buffSize;
    CHAR  userName[30]; /* Don't care about this, so this is an arbitrary length. */
    BYTE  userNameSize = sizeof(userName) - 1;

    ULONG rCode = GetDefaultProfile(profileType,
                                    &pdpType,
                                    &ipAddress,
                                    &primaryDNS,
                                    &secondaryDNS,
                                    &authentication,
                                    nameSize,
                                    &name[0],
                                    apnSize,
                                    &apnNamePtr[0],
                                    userNameSize,
                                    &userName[0]);

    if (rCode == eQCWWAN_ERR_NONE)
    {
        fprintf(stderr,"Received Profile Details are as follows:\n");
        fprintf(stderr," PDPType: %lx\n",pdpType);
        fprintf(stderr," IPAddress: %lx\n",ipAddress);
        fprintf(stderr," PrimaryDNS: %lx\n",primaryDNS);
        fprintf(stderr," SecondaryDNS: %lx\n",secondaryDNS);
        fprintf(stderr," Authentication: %lx\n",authentication);
        fprintf(stderr," Name: %s\n",name);
        fprintf(stderr," APNName: %s\n",apnNamePtr);
        fprintf(stderr," Username: %s\n",userName);
        
        return DT2_RESULT_OK;
    }
    else
    {
        return DT2_RESULT_ERROR;
    }
}


/** Set APN Override.
 * 
 * This function is used to set the current APN override setting.  This is a writeable
 * string that can be used to override the default APN to be used by the device.
 * 
 * The APN is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure. 
 */
dt2_ResultCode_t dt2_SetApnOverride
(
    const char* stringPtr       /**< Pointer to the new value. */
)
{

    DEV_CHECK

    /** @todo   Figure out what the real requirements are for this.
     *          For now, I'm just setting the default UMTS APN.
     */
    ULONG rCode = SetDefaultProfile(0,      // Profile type (0 = UMTS)
                                    NULL,   // PDPType
                                    NULL,   // IPAddress
                                    NULL,   // PrimaryDNS
                                    NULL,   // SecondaryDNS
                                    NULL,   // Authentication
                                    NULL,   // Name
                                    (char*)stringPtr, // APNName
                                    NULL,   // Username,
                                    NULL ); // Password

    if (rCode != eQCWWAN_ERR_NONE)
    {
        fprintf(stderr,
                "SetDefaultProfile() failed for APN name '%s' (returned %lu).\n",
                stringPtr,
                rCode);
        return DT2_RESULT_ERROR;
    }
    else
    {
        return DT2_RESULT_OK;
    }
}


/** Get APN.
 * 
 * This function is used to fetch the current APN being used by the device.  This is a
 * read-only variable.
 * 
 * The APN is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined.
 */
dt2_ResultCode_t dt2_GetApn
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK
    
    /** @todo   Figure out the real requirements.  For now, just return the
     *          APN override value.
     */

    return dt2_GetApnOverride(stringPtr, buffSize);
}


/** Get IMEI.
 * 
 * This function is used to fetch the device's IMEI.  This is a read-only variable.
 * 
 * The IMEI is a string of ANSI characters.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the string buffer contents are undefined.
 */
dt2_ResultCode_t dt2_GetImei
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    char esn[30] = "";
    char imei[30] = "";
    char meid[30] = "";

    ULONG resultCode;

    resultCode = GetSerialNumbers(  (BYTE)sizeof(esn),  /* BYTE     esnSize, */
            esn,                /* CHAR*    pESNString, */
            (BYTE)sizeof(imei), /* BYTE     imeiSize, */
            imei,               /* CHAR*    pIMEIString, */
            (BYTE)sizeof(meid), /* BYTE     meidSize, */
            meid );             /* CHAR*    pMEIDString */
    if (resultCode == eQCWWAN_ERR_NONE)
    {
        stringPtr[0] = '\0';
        strncat(stringPtr, imei, buffSize - 1); /* strncat always null terminates */
        return DT2_RESULT_OK;
    }
    else
    {
        fprintf(stderr, "GetSerialNumbers() failed (returned %lu).\n", resultCode);
        return DT2_RESULT_ERROR;
    }
}


/** Get RSSI.
 * 
 * Fetches the current cellular signal strength (RSSI).
 * 
 * The RSSI is a signed integer value from -150 to 0 dBm.
 * 
 * @return Result code.  Anything other than DT2_RESULT_OK indicates a failure, in which
 *         case the value buffer contents are undefined. 
 */
dt2_ResultCode_t dt2_GetRssi
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    int rssi = -150;    /* dBm */
    int techSpecificRssi;
    
    CDMASSInfo  cdmaSSInfo;
    HDRSSInfo   hdrSSInfo;
    BYTE        gsmSSInfo;
    CDMASSInfo  wcdmaSSInfo;
    LTESSInfo   lteSSInfo;

    nasGetSigInfoResp resp;

    resp.pCDMASSInfo  = &cdmaSSInfo;
    resp.pHDRSSInfo   = &hdrSSInfo;
    resp.pGSMSSInfo   = &gsmSSInfo;
    resp.pWCDMASSInfo = &wcdmaSSInfo;
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %lu)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    fprintf(stdout, "CDMA SS Information:\n");
    techSpecificRssi = -150;
    if (resp.pCDMASSInfo->rssi != 0xFF)
    {
        techSpecificRssi = (signed char)(resp.pCDMASSInfo->rssi);
        if (techSpecificRssi > rssi)
        {
            rssi = techSpecificRssi;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pCDMASSInfo->rssi, techSpecificRssi);
    fprintf(stdout, " ECIO  : %x\n", resp.pCDMASSInfo->ecio);


    fprintf(stdout, "HDR SS Information:\n");
    techSpecificRssi = -150;
    if (resp.pHDRSSInfo->rssi != 0xFF)
    {
        techSpecificRssi = (signed char)(resp.pHDRSSInfo->rssi);
        if (techSpecificRssi > rssi)
        {
            rssi = techSpecificRssi;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pHDRSSInfo->rssi, techSpecificRssi);
    fprintf(stdout, " ECIO  : %x\n", resp.pHDRSSInfo->ecio);
    fprintf(stdout, " SINR  : %x\n", resp.pHDRSSInfo->sinr);
    fprintf(stdout, " IO    : %lx\n", resp.pHDRSSInfo->io);

    fprintf(stdout, "GSM SS Information:\n");
    techSpecificRssi = -150;
    if (*resp.pGSMSSInfo != 0xFF)
    {
        techSpecificRssi = (signed char)(*resp.pGSMSSInfo);
        if (techSpecificRssi > rssi)
        {
            rssi = techSpecificRssi;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", *(resp.pGSMSSInfo), techSpecificRssi);

    fprintf(stdout, "WCDMA SS Information:\n");
    techSpecificRssi = -150;
    if (resp.pWCDMASSInfo->rssi != 0xFF)
    {
        techSpecificRssi = (signed char)(resp.pWCDMASSInfo->rssi);
        if (techSpecificRssi > rssi)
        {
            rssi = techSpecificRssi;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pWCDMASSInfo->rssi, techSpecificRssi);
    fprintf(stdout, " ECIO  : %x\n", resp.pWCDMASSInfo->ecio);

    fprintf(stdout, "LTE SS Information:\n");
    techSpecificRssi = -150;
    if (resp.pLTESSInfo->rssi != 0xFF)
    {
        techSpecificRssi = (signed char)(resp.pLTESSInfo->rssi);
        if (techSpecificRssi > rssi)
        {
            rssi = techSpecificRssi;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pLTESSInfo->rssi, techSpecificRssi);
    fprintf(stdout, " RSRQ  : %x\n", resp.pLTESSInfo->rsrq);
    fprintf(stdout, " RSRP  : %x\n", resp.pLTESSInfo->rsrp);
    fprintf(stdout, " SNR   : %x\n", resp.pLTESSInfo->snr);
    
    *valuePtr = rssi;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetGsmEcio
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    int ecio = -150;    /* dBm */
    int techSpecificEcio;
    
    CDMASSInfo  cdmaSSInfo;
    HDRSSInfo   hdrSSInfo;
    BYTE        gsmSSInfo;
    CDMASSInfo  wcdmaSSInfo;
    LTESSInfo   lteSSInfo;

    nasGetSigInfoResp resp;

    resp.pCDMASSInfo  = &cdmaSSInfo;
    resp.pHDRSSInfo   = &hdrSSInfo;
    resp.pGSMSSInfo   = &gsmSSInfo;
    resp.pWCDMASSInfo = &wcdmaSSInfo;
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %lu)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    fprintf(stdout, "CDMA SS Information:\n");
    techSpecificEcio = -150;
    if (resp.pCDMASSInfo->ecio != 0xFFFF)
    {
        techSpecificEcio = (signed char)(resp.pCDMASSInfo->ecio);
        if (techSpecificEcio > ecio)
        {
            ecio = techSpecificEcio;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pCDMASSInfo->rssi, techSpecificEcio);
    fprintf(stdout, " ECIO  : %x\n", resp.pCDMASSInfo->ecio);


    fprintf(stdout, "HDR SS Information:\n");
    techSpecificEcio = -150;
    if (resp.pHDRSSInfo->ecio != 0xFFFF)
    {
        techSpecificEcio = (signed char)(resp.pHDRSSInfo->ecio);
        if (techSpecificEcio > ecio)
        {
            ecio = techSpecificEcio;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pHDRSSInfo->rssi, techSpecificEcio);
    fprintf(stdout, " ECIO  : %x\n", resp.pHDRSSInfo->ecio);
    fprintf(stdout, " SINR  : %x\n", resp.pHDRSSInfo->sinr);
    fprintf(stdout, " IO    : %lx\n", resp.pHDRSSInfo->io);

    fprintf(stdout, "GSM SS Information:\n");
    techSpecificEcio = -150;
    if (*resp.pGSMSSInfo != 0xFF)
    {
        techSpecificEcio = (signed char)(*resp.pGSMSSInfo);
        if (techSpecificEcio > ecio)
        {
            ecio = techSpecificEcio;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", *(resp.pGSMSSInfo), techSpecificEcio);

    fprintf(stdout, "WCDMA SS Information:\n");
    techSpecificEcio = -150;
    if (resp.pWCDMASSInfo->ecio != 0xFFFF)
    {
        techSpecificEcio = (signed char)(resp.pWCDMASSInfo->ecio);
        if (techSpecificEcio > ecio)
        {
            ecio = techSpecificEcio;
        }
    }
    fprintf(stdout, " RSSI  : %x (%d dBm)\n", resp.pWCDMASSInfo->rssi, techSpecificEcio);
    fprintf(stdout, " ECIO  : %x\n", resp.pWCDMASSInfo->ecio);

    /*
       ECIO value representing negative 0.5 dBm increments, i.e.,
       2 means -1 dBm (14 means -7 dBm, 63 means -31.5 dBm).
       - 0xFFFF - Not Available
       */
    *valuePtr = ecio * -0.5;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetTech
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{

    DEV_CHECK

    char    *strTech = "Unknown";
    ULONG                   nRet;
    qaQmiServingSystemParam ServingSystemInfo;

    nRet = SLQSGetServingSystem( &ServingSystemInfo );

    if ( nRet == 0 && ServingSystemInfo.ServingSystem.numRadioInterfaces > 0)
    {
        fprintf(stdout, "RF %d\n", ServingSystemInfo.ServingSystem.radioInterface[0] );
        switch( ServingSystemInfo.ServingSystem.radioInterface[0] )
        {
            case 0:
                strTech = "None";
                break;
            case 1:
                strTech = "1X";
                break;
            case 2:
                strTech = "EV-DO";
                break;
            case 3:
                strTech = "AMPS";
                break;
            case 4:
                strTech = "GSM";
                break;
            case 5:
                strTech = "UMTS";
                break;
            case 8:
                strTech = "LTE";
                break;
        }
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strTech, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetPktRcvd
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double pkt_rcvd = 0;
    ULONG  TXPacketSuccesses = 0;
    ULONG  RXPacketSuccesses = 0;
    ULONG  TXPacketErrors = 0;
    ULONG  RXPacketErrors = 0;
    ULONG  TXPacketOverflows = 0;
    ULONG  RXPacketOverflows = 0;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetPacketStatus (
            &TXPacketSuccesses,
            &RXPacketSuccesses,
            &TXPacketErrors,
            &RXPacketErrors,
            &TXPacketOverflows,
            &RXPacketOverflows,
            instance );

    if ( nRet == 0 )
    {
        pkt_rcvd = RXPacketSuccesses;
    }

    *valuePtr = pkt_rcvd;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetPktSent
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double pkt_sent = 0;
    ULONG  TXPacketSuccesses = 0;
    ULONG  RXPacketSuccesses = 0;
    ULONG  TXPacketErrors = 0;
    ULONG  RXPacketErrors = 0;
    ULONG  TXPacketOverflows = 0;
    ULONG  RXPacketOverflows = 0;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetPacketStatus (
            &TXPacketSuccesses,
            &RXPacketSuccesses,
            &TXPacketErrors,
            &RXPacketErrors,
            &TXPacketOverflows,
            &RXPacketOverflows,
            instance );

    if ( nRet == 0 )
    {
        pkt_sent = TXPacketSuccesses;
    }

    *valuePtr = pkt_sent;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetOperator
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK 

    WORD  pMCC;
    WORD  pMNC;
    CHAR  pName[nMaxStrLen];
    WORD  pSID;
    WORD  pNID;

    char    *strOperator = "Unknown";
    ULONG                   nRet;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = GetHomeNetwork(&pMCC,
            &pMNC,
            nMaxStrLen,
            &pName[0],
            &pSID,
            &pNID);

    fprintf(stdout, "GetHomeNetwork rtn %lx\n", nRet);

    if ( nRet == 0 )
    {
        strOperator = pName;
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strOperator, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetIP
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    DEV_CHECK

    ULONG ipv4;
    struct WdsRunTimeSettings lRunTimeSettings;

    char    strIP[15] = "99.99.99.99";
    ULONG                   nRet;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    memset(&lRunTimeSettings, 0, sizeof(lRunTimeSettings));
    lRunTimeSettings.rts.pIPAddressV4 = &ipv4;
    nRet = SLQSGetRuntimeSettings( &lRunTimeSettings );
    fprintf(stdout, "API rtn %lx\n", nRet);

    if ( nRet == 0 )
    {
        sprintf( strIP, "IPAddressV4    : %d.%d.%d.%d\n",
                (unsigned int)(ipv4 >> 24) & 0xFF,
                (unsigned int)(ipv4 >> 16) & 0xFF,
                (unsigned int)(ipv4 >> 8) & 0xFF,
                (unsigned int)ipv4 & 0xFF );
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strIP, buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetRoamStatus
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{

    DEV_CHECK

    ULONG pRegistrationState;
    ULONG pCSDomain;
    ULONG pPSDomain;
    ULONG pRAN;
    BYTE  pRadioIfacesSize;
    BYTE  pRadioIfaces[nMaxStrLen];
    ULONG pRoaming;
    WORD  pMCC;
    WORD  pMNC;
    CHAR  pName[nMaxStrLen];

    char    *strRoaming = "Unknown";
    ULONG                   nRet;

    fprintf(stdout, "buffSize %zx\n", buffSize);

    nRet = GetServingNetwork(&pRegistrationState,
            &pCSDomain,
            &pPSDomain,
            &pRAN,
            &pRadioIfacesSize,
            &pRadioIfaces[0],
            &pRoaming,
            &pMCC,
            &pMNC,
            nMaxStrLen,
            &pName[0]);

    fprintf(stdout, "API rtn %lx\n", nRet);

    if ( nRet == 0 )
    {
        if (pRoaming == 0x00)
            strRoaming = "Roaming";
        else if (pRoaming == 0x01)
            strRoaming = "Home";
        else
            strRoaming = "Operator defined";
    }

    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, strRoaming, buffSize);

    return DT2_RESULT_OK;
}

//TODO share with rssi
dt2_ResultCode_t dt2_GetRsrp
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    int rsrp = -150;    /* dBm */
    int techSpecificRsrp;

    LTESSInfo   lteSSInfo;

    nasGetSigInfoResp resp;

    memset(&resp, 0, sizeof(resp));
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %lu)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    fprintf(stdout, "LTE SS Information:\n");
    techSpecificRsrp= -150;
    if (resp.pLTESSInfo->rsrp != 0xFF)
    {
        techSpecificRsrp = (signed char)(resp.pLTESSInfo->rsrp);
        if (techSpecificRsrp > rsrp)
        {
            rsrp = techSpecificRsrp;
        }
    }
    fprintf(stdout, " RSRQ  : %x\n", resp.pLTESSInfo->rsrq);
    fprintf(stdout, " RSRP  : %x\n", resp.pLTESSInfo->rsrp);

    *valuePtr = rsrp;

    return DT2_RESULT_OK;
}

//TODO share with rssi
dt2_ResultCode_t dt2_GetRsrq
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    int rsrq = -150;    /* dBm */
    int techSpecificRsrq;

    LTESSInfo   lteSSInfo;

    nasGetSigInfoResp resp;

    memset(&resp, 0, sizeof(resp));
    resp.pLTESSInfo   = &lteSSInfo;

    ULONG retCode = SLQSNasGetSigInfo( &resp );

    if ( eQCWWAN_ERR_NONE != retCode )
    {
        fprintf(stdout, "SLQSNasGetSigInfo Unsuccessful (error = %lu)\n", retCode);
        return DT2_RESULT_ERROR;
    }

    fprintf(stdout, "SLQSNasGetSigInfo Successful\n");

    fprintf(stdout, "LTE SS Information:\n");
    techSpecificRsrq= -150;
    if (resp.pLTESSInfo->rsrq != 0xFF)
    {
        techSpecificRsrq = (signed char)(resp.pLTESSInfo->rsrq);
        if (techSpecificRsrq > rsrq)
        {
            rsrq = techSpecificRsrq;
        }
    }
    fprintf(stdout, " RSRQ  : %x\n", resp.pLTESSInfo->rsrq);
    fprintf(stdout, " RSRP  : %x\n", resp.pLTESSInfo->rsrp);

    *valuePtr = rsrq;

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetBoardTemp
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    /* TODO measure your board temperature here */
    *valuePtr = -273;

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetRadioTemp
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    swiModemStatusResp resp;
    int temp = -273;

    memset(&resp, 0, sizeof(resp));
    ULONG nRet = SLQSNasSwiModemStatus( &resp );

    if ( eQCWWAN_ERR_NONE == nRet )
    {
        fprintf( stdout, "SLQSNasSwiModemStatus Successful \n" );
        temp = resp.commonInfo.temperature;
    }

    *valuePtr = temp;

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetResetCnt
(
    int*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    /* TODO report your system report count here */
    *valuePtr = -999;

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetBytesRcvd
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double bytes_rcvd = 0;
    ULONGLONG  TXTotalBytes;
    ULONGLONG  RXTotalBytes;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetByteTotals (
            &TXTotalBytes,
            &RXTotalBytes,
            instance );

    if ( nRet == 0 )
    {
        bytes_rcvd = TXTotalBytes;
    }

    *valuePtr = bytes_rcvd;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetBytesSent
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    DEV_CHECK

    double bytes_sent = 0;
    ULONGLONG  TXTotalBytes;
    ULONGLONG  RXTotalBytes;
    BYTE   instance = 0;

    ULONG                   nRet;

    nRet = GetByteTotals (
            &TXTotalBytes,
            &RXTotalBytes,
            instance );

    if ( nRet == 0 )
    {
        bytes_sent = TXTotalBytes;
    }

    *valuePtr = bytes_sent;
    
    return DT2_RESULT_OK;
}


/* TODO attributes has to further develop */

dt2_ResultCode_t dt2_GetCdmaEcio
(
    double*    valuePtr        /**< Pointer to the buffer into which the value will be copied. */
)
{
    *valuePtr = -150;
    
    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaPnOff
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, "Unknown", buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetCdmaSid
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, "Unknown", buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetGsmCellId
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, "Unknown", buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetLatitude
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, "49.171731", buffSize);

    return DT2_RESULT_OK;
}

dt2_ResultCode_t dt2_GetLongitude
(
    char*   stringPtr,      /**< Pointer to the buffer into which the string will be copied. */
    size_t  buffSize        /**< Size of the buffer, in bytes. */ 
)
{
    memset(stringPtr, 0, buffSize);
    strncpy(stringPtr, "-123.069347", buffSize);

    return DT2_RESULT_OK;
}
