/*
Debug support goodies for kernel development.

The macro DLEVEL controls which debug messages are output and which are not.

To hardcode a debug level into your source code, use

#define DLEVEL 2
#include <linux/debug.h>

There is a tradeoff between the amount of debug messages and the visibility
of a problem. Also, debug messages in the critical data path may well change
behaviour enough to make a bug go away. Use carefully.

The meaning of the debug levels is as follows

DLEVEL=0	Minimal debug, has no effect on runtime performance.
		Output from debug0

DLEVEL=1	Initialisation / teardown path debugging. Small runtime impact.
		Output from debug0, debug1

DLEVEL=2	Minimal data path debugging. Runtime impact.
		Output from debug0 - debug2

DLEVEL=3	Insane debug level. heavy runtime impact.
		Output from debug0 - debug3

Iwo@call-direct.com.au
*/
#ifndef __DEBUG_H
#define __DEBUG_H

#include <linux/version.h>
#include <linux/kernel.h>
#include <asm/current.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/skbuff.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	#include <linux/hardirq.h>
	#include <linux/kallsyms.h>
#endif

#include <linux/usb.h>
#include <linux/list.h>

/* Debug level. Defaults to 0. */
/* Use -DDLEVEL=1 for level 1 or -DDLEVEL=2 for level2 */

#ifndef DLEVEL
#define DLEVEL 0
#endif

#ifdef PARANOIA
#define assert(cond) do { if (!(cond)) { \
	msg("####### Assertion %s failed\n",#cond); \
	*(int*)0=0; \
	}} while(0)
#else
#define assert(cond) do { (void)(cond); } while(0)
#endif

#define DBG_LEVEL KERN_INFO

/** Helpers */

/* Check if pointer is valid kernelspace pointer. */
#define NOT_KERNEL_POINTER(ptr) (unlikely((unsigned long)(ptr) < 0xC0000000 || IS_ERR(ptr)))

/* Return macro name for error value */
static inline const char * ErrStr(int err)
{
	#define MAXL 64
	#define ADD(...) ofs+=snprintf(line+ofs,MAXL,__VA_ARGS__)
	static char line[MAXL]="";
	int ofs=0;

	ADD("%d [",err);

	if (err==0) {
		ADD("OK]");
		return line;
	}

	if (err < 0) {
		err=-err;
		ADD("-");
	}
	else
		ADD("+");

	switch (err) {
		#define E(code) case code : ADD(#code); break

		/* asm-generic/errno-base.h */
		E(EPERM); E(ENOENT); E(ESRCH); E(EINTR); E(EIO); E(ENXIO); E(E2BIG); E(ENOEXEC);
		E(EBADF); E(ECHILD); E(EAGAIN); E(ENOMEM); E(EACCES); E(EFAULT); E(ENOTBLK);
		E(EBUSY); E(EEXIST); E(EXDEV); E(ENODEV); E(ENOTDIR); E(EISDIR); E(EINVAL);
		E(ENFILE); E(EMFILE); E(ENOTTY); E(ETXTBSY); E(EFBIG); E(ENOSPC); E(ESPIPE);
		E(EROFS); E(EMLINK); E(EPIPE); E(EDOM); E(ERANGE);

		/* asm-generic/errno.h */
		E(EDEADLK); E(ENAMETOOLONG); E(ENOLCK); E(ENOSYS); E(ENOTEMPTY); E(ELOOP);
		E(ENOMSG); E(EIDRM); E(ECHRNG); E(EL2NSYNC); E(EL3HLT); E(EL3RST); E(ELNRNG);
		E(EUNATCH); E(ENOCSI); E(EL2HLT); E(EBADE); E(EBADR); E(EXFULL); E(ENOANO);
		E(EBADRQC); E(EBADSLT);

		E(EBFONT); E(ENOSTR); E(ENODATA); E(ETIME); E(ENOSR); E(ENONET); E(ENOPKG);
		E(EREMOTE); E(ENOLINK); E(EADV); E(ESRMNT); E(ECOMM); E(EPROTO); E(EMULTIHOP);
		E(EDOTDOT); E(EBADMSG); E(EOVERFLOW); E(ENOTUNIQ); E(EBADFD); E(EREMCHG);
		E(ELIBACC); E(ELIBBAD); E(ELIBSCN); E(ELIBMAX); E(ELIBEXEC); E(EILSEQ);
		E(ERESTART); E(ESTRPIPE); E(EUSERS); E(ENOTSOCK); E(EDESTADDRREQ); E(EMSGSIZE);
		E(EPROTOTYPE); E(ENOPROTOOPT); E(EPROTONOSUPPORT); E(ESOCKTNOSUPPORT);
		E(EOPNOTSUPP); E(EPFNOSUPPORT); E(EAFNOSUPPORT); E(EADDRINUSE); E(EADDRNOTAVAIL);
		E(ENETDOWN); E(ENETUNREACH); E(ENETRESET); E(ECONNABORTED); E(ECONNRESET);
		E(ENOBUFS); E(EISCONN); E(ENOTCONN); E(ESHUTDOWN); E(ETOOMANYREFS); E(ETIMEDOUT);
		E(ECONNREFUSED); E(EHOSTDOWN); E(EHOSTUNREACH); E(EALREADY); E(EINPROGRESS);
		E(ESTALE); E(EUCLEAN); E(ENOTNAM); E(ENAVAIL); E(EISNAM); E(EREMOTEIO); E(EDQUOT);

		E(ENOMEDIUM); E(EMEDIUMTYPE);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
		E(EOWNERDEAD); E(ENOTRECOVERABLE);
		E(ECANCELED); E(ENOKEY); E(EKEYEXPIRED); E(EKEYREVOKED); E(EKEYREJECTED);
#endif

		#undef E
		default : ADD("???"); break;
	}

	ADD("]");

	#undef ADD
	#undef MAXL

	return line;
}

/* Displays context info.
verbosity=0 - [ASTI]HH (A=atomic, S=softirq, T=task, I=interrupt, HH PID hash)
verbosity=1 - CTX PID=nnn commandname (CTX=(IRQ|ATM|SIQ|TSK))
verbosity=2 - CTX PID/commandname [PPID/parentcmd]
*/
static inline char * Context(const int verbosity)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	#define TNAMELEN TASK_COMM_LEN
#else
	#define TNAMELEN 16
#endif
	/* Worst case: "CTX PID__/commandname_____ [CTX PID__/commandname_____]\0" */
	static char ctx[3+1+5+1+TNAMELEN+2+3+1+5+1+TNAMELEN+1+8];
	int ofs = 0;
	char context = 0;

	if (in_irq()) context='I';
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	else if (in_atomic()) context='A';
	else if (in_softirq()) context='S';
#endif
	else context='T';

	if (verbosity==0) {
		u32 temp;
		u8 hash;
		#define CONTEXT_MAGIC 0x4afd22bc
		temp=((u32)current*CONTEXT_MAGIC);
		hash=(temp &0xFF)^((temp >> 8) & 0xFF)^((temp >> 16) & 0xFF)^((temp >> 24) & 0xFF);
		if (context != 'I')
		{
			sprintf(ctx, "%c%02x", context, hash);
		}
		else
		{
			sprintf(ctx,"%c--", context);
		}
		return ctx;
	}

	/* Verbosity >= 1 */
	switch (context) {
		case 'I': ofs+=sprintf(ctx+ofs, "IRQ "); break;
		case 'A': ofs+=sprintf(ctx+ofs, "ATM "); break;
		case 'S': ofs+=sprintf(ctx+ofs, "SIQ "); break;
		case 'T': ofs+=sprintf(ctx+ofs, "TSK "); break;
		default : ofs+=sprintf(ctx+ofs, "??? "); break;
	}
	ofs+= sprintf(ctx+ofs, "%d/", current->pid);
	ofs+= snprintf(ctx+ofs, TNAMELEN+1, "%s", current->comm);

	if (verbosity==2) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
		struct task_struct *parent = current->real_parent;
#else
		struct task_struct *parent = current->p_pptr;
#endif
		ofs+= sprintf(ctx+ofs, " [%d/", parent->pid);
		ofs+= snprintf(ctx+ofs, TNAMELEN+1, "%s]", parent->comm);
	}

	return ctx;
}

static inline char* FullContext(void)
{
	return Context(1);
}

static inline char* ContextTag(void)
{
	return Context(0);
}

static inline char* ContextWithParent(void)
{
	return Context(2);
}

/* Attempt to determine the symbol belonging to an address */
/* If you want to use this in a module, add an EXPORT_SYMBOL(kallsyms_lookup)
 in kernel/kallsyms.c. */
static inline char* Symbol(void *addr)
{
	static char sline[256]="";
	unsigned long address=(unsigned long)addr;

#ifdef CONFIG_KALLSYMS
	unsigned long offset;
	unsigned long symsize;
	const char *symname;
	char *modname;
	char namebuf[128];

	symname = kallsyms_lookup(address, &symsize, &offset, &modname, namebuf);
	if (symname) {
		if (!modname)
			modname = "kernel";

		sline[snprintf(sline,256,"[0x%p] %s | %s + 0x%lx",
			(void *)address, modname, symname,(unsigned long)offset)]='\0';
		return sline;
	}
#endif

	/* Can't decode */
	sline[snprintf(sline,256,"0x%p", (void *)address)]='\0';
	return sline;
}


/** Debug output functions */

/* copies a string, up to max characters, while converting
non-printable characters (<' ') to npc. The result is always 0-terminated. */
static inline void StrncpyPrintable(char *dest, const char *src, unsigned int max, char npc)
{
	if (src < (const char *)0x1000) {
		snprintf(dest, max, "PTR%p", src);
		return;
	}
	while (--max && *src) {
		if (*src < ' ')
			*dest = npc;
		else
			*dest = *src;
		src++;
		dest++;
	}
	*dest='\0';
}

/* Basic debug print functions */

#define debug(...)  debug1(__VA_ARGS__)
#define msg(...) debug0(__VA_ARGS__)
#define rmsg(...) printk(DBG_LEVEL __VA_ARGS__)

#if (__GNUC__ > 2)
	#define debug_no(FMT,VA...)   do {} while(0)
	#define debug_yes(FMT, VA...) printk(DBG_LEVEL "[%s]%s:%d: "FMT,ContextTag(),__FUNCTION__,__LINE__,##VA)
#else
	#define debug_no(...)  do {} while(0)
	#define debug_yes(...) printk(DBG_LEVEL __FUNCTION__": " __VA_ARGS__)
#endif

#define debug0 debug_no
#define debug1 debug_no
#define debug2 debug_no
#define debug3 debug_no

#if DLEVEL >= 3
	#undef  debug3
	#define debug3 debug_yes
#endif

#if DLEVEL >= 2
	#undef  debug2
	#define debug2 debug_yes
#endif

#if DLEVEL >= 1
	#undef  debug1
	#define debug1 debug_yes
#endif

#if DLEVEL >= 0
	#undef  debug0
	#define debug0 debug_yes
#endif

#define dump_int(varname) msg(#varname "=%d [%0x%x]\n", varname, varname)
#define dump_string(varname) if (varname) msg(#varname "='%s'\n", varname); else msg(#varname "=NULL\n")
#define dump_ptr(varname) msg(#varname "=%p\n", varname)

/* Returns n space characters */
static inline const char* indent(unsigned int n)
{
	static char line[100];
	memset(line,' ',sizeof(line));
	line[min(n,sizeof(line))]='\0';
	return line;
}

/* Simple tracer. If LOG_TRACER is set, the tracing info goes into a line-ring-buffer
which is only printed when DUMP_TRACER is called. MAX_TRACER controls how large the
buffer is (default 64 lines). FIXME: log tracer needs locking. */
#ifndef DISABLE_TRACER

#ifndef MAX_TRACER
#define MAX_TRACER 64
#endif

#ifdef LOG_TRACER
	struct dtracer_log {
		struct list_head list;
		char line[1]; /* Variable size */
	};
#endif

	struct dtracer {
		atomic_t l; /* Atomic indentation level */
		int on;     /* ON/OFF switch */
#ifdef LOG_TRACER
		int    lines;
		struct list_head buffer;
#endif
	};
#ifdef LOG_TRACER
	#define DECLARE_TRACER struct dtracer debug_tracer = {{0},0,0,LIST_HEAD_INIT(debug_tracer.buffer)}; EXPORT_SYMBOL(debug_tracer)
	#define DECLARE_TRACER_ENABLED struct dtracer debug_tracer = {{0},1,0,LIST_HEAD_INIT(debug_tracer.buffer)}; EXPORT_SYMBOL(debug_tracer)
#else
	#define DECLARE_TRACER struct dtracer debug_tracer = {{0},0}; EXPORT_SYMBOL(debug_tracer)
	#define DECLARE_TRACER_ENABLED struct dtracer debug_tracer = {{0},1}; EXPORT_SYMBOL(debug_tracer)
#endif
	extern struct dtracer debug_tracer;

#ifdef LOG_TRACER
	/*
	Add a tracer line to the log buffer. We assume a maximum size.
	*/
	#ifndef LTBMAX
	#define LTBMAX (256-sizeof(struct dtracer_log))
	#endif

static inline void LPRINT(const char *fmt,...)
{
	va_list args;
	struct dtracer_log *dl;
	dl = kmalloc(sizeof(struct dtracer_log)+LTBMAX,GFP_ATOMIC);
	BUG_ON(dl==NULL);
	va_start(args, fmt);
	vsnprintf(dl->line,LTBMAX,fmt,args);
	va_end(args);
	dl->line[LTBMAX-1]='\0';
	list_add_tail(&dl->list,&debug_tracer.buffer);
	if (debug_tracer.lines++ >= MAX_TRACER) {
		dl=list_entry(debug_tracer.buffer.next,struct dtracer_log,list);
		list_del(&dl->list);
		debug_tracer.lines--;
		kfree(dl);
	}
}

static inline void LDUMP(const char *prefix)
{
	struct dtracer_log *dl, *tmp;
	list_for_each_entry_safe(dl,tmp,&debug_tracer.buffer,list) {
		list_del(&dl->list);
		printk(DBG_LEVEL "%s%s",prefix,dl->line);
		kfree(dl);
	}
}
#else
	#define LPRINT(...) printk(DBG_LEVEL __VA_ARGS__)
	#define LDUMP(prefix) do { } while (0)
#endif

	#define E(FMT,VA...) do{\
		atomic_inc(&debug_tracer.l); \
		if (debug_tracer.on) \
			LPRINT("[%s]%s=> %s:%d: "FMT, ContextTag(), indent(atomic_read(&debug_tracer.l)),\
					__FUNCTION__,__LINE__,##VA); \
	} while (0)

	#define M(FMT,VA...) do{\
		if (debug_tracer.on) \
			LPRINT("[%s]%s | %s:%d: "FMT, ContextTag(), indent(atomic_read(&debug_tracer.l)),\
					__FUNCTION__,__LINE__,##VA); \
	} while(0)

	#define X(FMT,VA...) do {\
		if (debug_tracer.on) \
			LPRINT("[%s]%s<= %s:%d: "FMT, ContextTag(), indent(atomic_read(&debug_tracer.l)),\
					__FUNCTION__,__LINE__,##VA);\
		atomic_dec(&debug_tracer.l);\
	} while(0)

	#define T_OFF do { debug_tracer.on=0; msg("Disabling trace\n");} while(0)
	#define T_ON  do { debug_tracer.on=1; msg("Enabling trace\n"); } while(0)

#else /* DISABLE_TRACER */

	struct dtracer {
		int on;
	};
	#define DECLARE_TRACER void nothing(void)
	static const struct dtracer debug_tracer = {0};
	#define E(...) do {} while(0)
	#define M(...) do {} while(0)
	#define X(...) do {} while(0)

	#define T_OFF  do {} while(0)
	#define T_ON   do {} while(0)
	#define LDUMP(prefix) do {} while (0)
#endif

/* Canonical HexDump of data block */
#define HexDump HexDump1

#if DLEVEL >= 3
	#define HexDump3(b,s) _HexDump((b),(s),__FUNCTION__)
	#define HexDump2(b,s) _HexDump((b),(s),__FUNCTION__)
	#define HexDump1(b,s) _HexDump((b),(s),__FUNCTION__)
	#define HexDump0(b,s) _HexDump((b),(s),__FUNCTION__)
#elif DLEVEL == 2
	#define HexDump3(b,s) do {} while(0)
	#define HexDump2(b,s) _HexDump((b),(s),__FUNCTION__)
	#define HexDump1(b,s) _HexDump((b),(s),__FUNCTION__)
	#define HexDump0(b,s) _HexDump((b),(s),__FUNCTION__)
#elif DLEVEL == 1
	#define HexDump3(b,s) do {} while(0)
	#define HexDump2(b,s) do {} while(0)
	#define HexDump1(b,s) _HexDump((b),(s),__FUNCTION__)
	#define HexDump0(b,s) _HexDump((b),(s),__FUNCTION__)
#elif DLEVEL == 0
	#define HexDump3(b,s) do {} while(0)
	#define HexDump2(b,s) do {} while(0)
	#define HexDump1(b,s) do {} while(0)
	#define HexDump0(b,s) _HexDump((b),(s),__FUNCTION__)
#endif
static inline void _HexDump(const char *buffer, int size, const char* fun)
{
	#define HEXDUMP_CHAR_PER_LINE (16)
	#define HEXDUMPS (HEXDUMP_CHAR_PER_LINE * 3)
	static char str_hexDump[HEXDUMPS+1]="";
	static char str_asciiDump[HEXDUMP_CHAR_PER_LINE+1]="";
	int idx;
	int col_idx = 0;

	for (idx=0; idx < size; idx++) {
		sprintf(&str_hexDump[col_idx*3], "%02X ", (unsigned char)buffer[idx]);
		str_asciiDump[col_idx] = ((buffer[idx] >=32) && (buffer[idx] <127)) ? buffer[idx] : '.';
		col_idx++;
		if (col_idx == HEXDUMP_CHAR_PER_LINE) {
			str_asciiDump[col_idx] ='\0';
			printk(DBG_LEVEL "[%s]%s: [0x%p] %-48s : %s\n",
				ContextTag(), fun, buffer+idx-col_idx, str_hexDump, str_asciiDump);
			col_idx = 0;
		}
	}
	str_asciiDump[col_idx] ='\0';
	if (col_idx != 0) printk(DBG_LEVEL "[%s]%s: [0x%p] %-48s : %s\n",
		ContextTag(), fun, buffer+idx-col_idx, str_hexDump, str_asciiDump);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
/* Dump SKB (Network packet) contents */
#define dump_skb(skb) _dump_skb((skb),__func__,__LINE__)
static inline void _dump_skb(const struct sk_buff *skb, const char* fun, int line)
{
	static char ftag[256]="[TTT]???:??:";
	static char htag[256]="???:??";
	unsigned char *p,*start;
	unsigned long CRC;
	int flag;

	ftag[snprintf(ftag,256,"[%s]%s:%d",ContextTag(),fun,line)]='\0';
	htag[snprintf(htag,256,"%s:%d",fun,line)]='\0';

	printk(DBG_LEVEL "%s: ################ &SKB=%p #############\n",ftag,skb);

	if (NOT_KERNEL_POINTER(skb)) return;

	printk(DBG_LEVEL "%s: head=%p, data=%p, tail=%p, end=%p\n",
		ftag,skb->head,skb->data,skb->tail,skb->end);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)
	printk(DBG_LEVEL "%s: transport_header=%p, network_header=%p, mac_header=%p\n",ftag,
		skb->transport_header,skb->network_header,skb->mac_header);
	printk(DBG_LEVEL "%s: len=%d (%d), data_len=%d, truesize=%d, mac_len=%d, hdr_len=%d\n",ftag,
		skb->len,skb->tail-skb->data,
		skb->data_len,
		skb->truesize,skb->mac_len,skb->hdr_len);
#else
	printk(DBG_LEVEL "%s: len=%d (%d), data_len=%d, truesize=%d, mac_len=%d\n",ftag,
		skb->len,skb->tail-skb->data,
		skb->data_len,
		skb->truesize,skb->mac_len);
#endif
	printk(DBG_LEVEL "%s: cb =>\n",ftag);
	_HexDump(skb->cb,48,ftag);

	if (skb->data_len != 0) return; /* Can't deal with SG packets */
	if (skb->end <= skb->head) return;

	CRC=0;
	for (p=skb->head; p<skb->end; p++)
	{
		CRC=((CRC << 1) | (CRC >> 31)) ^ *p;
	}
	printk(DBG_LEVEL "%s: CRC=0x%08lX\n",ftag,CRC);

	start=skb->head; flag=0;
	for (p=skb->head; p<skb->end; p++)
	{
		#define SKBPCHECK(name) do { if (p==skb->name) flag=1; } while(0)
		#define SKBPPRINT(name) do { if (p==skb->name) \
			printk(DBG_LEVEL "%s: %p [%s] =>\n",ftag,p,#name); } while(0)

		SKBPCHECK(head); SKBPCHECK(data);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)
		SKBPCHECK(transport_header); SKBPCHECK(network_header);
		SKBPCHECK(mac_header);
#endif
		SKBPCHECK(tail);

		if (flag)
		{
			if (p>start && start>=skb->data && p<=skb->tail) _HexDump(start,p-start,htag);
			flag=0; start=p;
		}

		SKBPPRINT(head); SKBPPRINT(data);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,22)
		SKBPPRINT(transport_header); SKBPPRINT(network_header);
		SKBPPRINT(mac_header);
#endif
		SKBPPRINT(tail);
	}

	if (p>start && start>=skb->data && p<=skb->tail) _HexDump(start,p-start,htag);
	SKBPPRINT(end);
}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
/* Dump URB (USB Request Block) contents */
#define dump_urb(urb) _dump_urb((urb),__func__,__LINE__)
static inline void _dump_urb(const struct urb *u, const char* fun, int line)
{
	static char ftag[256]="[TTT]???:??:";
	static char htag[256]="???:??";

	ftag[snprintf(ftag,256,"[%s]%s:%d",ContextTag(),fun,line)]='\0';
	htag[snprintf(htag,256,"%s:%d",fun,line)]='\0';

	printk(DBG_LEVEL "%s: ################ &URB=%p #############\n",ftag,u);

	if (NOT_KERNEL_POINTER(u)) return;

	/* Private USB core fields */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,29)
	printk(DBG_LEVEL "%s: kref=%d, hcpriv=%p, use_count=%d, reject=%d, unlinked=%d\n",ftag,
		atomic_read(&u->kref.refcount),u->hcpriv,atomic_read(&u->use_count),
		atomic_read(&u->reject), u->unlinked);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,24)
	printk(DBG_LEVEL "%s: kref=%d, hcpriv=%p, use_count=%d, reject=%d, unlinked=%d\n",ftag,
		atomic_read(&u->kref.refcount),u->hcpriv,atomic_read(&u->use_count),
		u->reject, u->unlinked);
#else
	printk(DBG_LEVEL "%s: kref=%d, hcpriv=%p, use_count=%d, reject=%d\n",ftag,
		atomic_read(&u->kref.refcount),u->hcpriv,atomic_read(&u->use_count),u->reject);
#endif

	/* Public fields */
	printk(DBG_LEVEL "%s: status=%s, pipe=%08X, flags=%08X, tblength=%d, alength=%d\n",ftag,
		ErrStr(u->status), u->pipe, u->transfer_flags, u->transfer_buffer_length,u->actual_length);

	printk(DBG_LEVEL "%s: Complete(): %s\n",ftag,Symbol(u->complete));

	if (u->actual_length >0 )
	{
		_HexDump(u->transfer_buffer,u->transfer_buffer_length,htag);
	}
	else
	{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
		_HexDump(u->transfer_buffer,min(u->transfer_buffer_length,(u32)64),htag);
#else
		_HexDump(u->transfer_buffer,min(u->transfer_buffer_length,64),htag);
#endif
	}
}

static inline const char* urbTag(const struct urb *u)
{
	static char line[256]="";
	const char *type = "???";
	const char *dir  = "?";
	int of=0;

	if (NOT_KERNEL_POINTER(u)) {
		sprintf(line,"URB %p",u);
		return line;
	}

	switch(usb_pipetype(u->pipe)) {
		case PIPE_ISOCHRONOUS : type="Z"; break;
		case PIPE_INTERRUPT   : type="I"; break;
		case PIPE_CONTROL     : type="C"; break;
		case PIPE_BULK        : type="B"; break;
	}

	dir=usb_pipein(u->pipe)?"i":"o";

	of+=sprintf(line+of,"URB %p %s%s %d:%d, kref=%d, uct=%d, len=%d, alen=%d, sta=%s",
		u,type,dir,usb_pipedevice(u->pipe),usb_pipeendpoint(u->pipe),
		atomic_read(&u->kref.refcount),atomic_read(&u->use_count),
		u->transfer_buffer_length,u->actual_length,ErrStr(u->status));

	return line;
}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
/** DPIPE debugging interface **/

/* DPipe minor number for network packets */
#define DPIPE_NET 0
/* DPipe minor number for text output */
#define DPIPE_TEXT 1

#define DPIPE_MINORS 2

/* Returns a pointer to 24 bytes of libpcap file header */
static inline unsigned char* pcap_80211_header(int radiotap)
{
	static struct pcap_hdr {
		uint32_t magic;
		uint16_t vmaj;
		uint16_t vmin;
		uint32_t tz;
		uint32_t sigacc;
		uint32_t snaplen;
		uint32_t linktype;
	} __attribute__((packed)) hdr = {0xa1b2c3d4,2,4,0,0,65535,0};
	/*
	WTAP_ENCAP_IEEE_802_11_WLAN_RADIOTAP 127
	WTAP_ENCAP_IEEE_802_11 105
	*/
	hdr.linktype=(radiotap)?127:105;
	return (unsigned char*)&hdr;
}

/* Returns a pointer to 16 bytes of libpcap packet header. In the file,
it most be followed by the contents of (skb->data - skb->tail-1). */
static inline unsigned char* pcap_80211_pheader(const struct sk_buff *skb)
{
	static struct pcap_phdr {
		uint32_t ts_sec;
		uint32_t ts_usec;
		uint32_t incl_len;
		uint32_t orig_len;
	} __attribute__((packed)) phdr;
	struct timespec now;
	getnstimeofday(&now);
	phdr.ts_sec = now.tv_sec;
	phdr.ts_usec = now.tv_nsec/1000;
	phdr.incl_len = phdr.orig_len = skb->tail - skb->data;
	return (unsigned char*)&phdr;
}

/* Push a 802.11 packet into dpipe, minor 0. */
extern int dpipe_80211_packet(const struct sk_buff *skb);

/* Push a Text line into dpipe, minor 1. */
extern int dpipe_text(const char *line, int len);

/* Register open Callback */
extern int dpipe_callback(int minor, void (*cb)(void));

#endif

#endif /* __DEBUG_H */
