/*
Generic userspace GPIO driver for CDCS platforms

Iwo Mergler <Iwo@call-direct.com.au>
*/

#ifndef __CDCS_GPIO_H
#define __CDCS_GPIO_H

/* Pin definitions for platforms where it isn't a simple integer. Keep below 4096. */

/* Bovine only: */
#define BOVINE_PINA(n) (0x00U | (n))
#define BOVINE_PINB(n) (0x20U | (n))
#define BOVINE_PINC(n) (0x40U | (n))


/* Apart from GPIO, the availability of modes depends on pin
and platform. */
enum cdcs_gpio_muxmode {
	IO_GPIO,
	/* Platypus */
	IO_MUX, IO_UARTF, IO_MODEM, IO_I2S, IO_PCM, IO_REFCLK, /* Pin dependent. */
	/* Bovine. _PU have pull-up enabled. */
	IO_MUXA, IO_MUXB, 
	/* Ti AM335x */
	IO_MUXC, IO_MUXD, IO_MUXE, IO_MUXF, IO_MUXG, IO_MUXH
};

struct cdcs_gpio_data {
	unsigned gpio;               /* [input] GPIO number of pin */
	union ea { /* argument. */
		int  value;                  /* [inout] Pin value 0 | 1 */
		enum cdcs_gpio_muxmode mux;  /* [inout] Pin multiplexer state */
		int  signal;                 /* [input] Signal number for interrupt */
	} ea;
	 /* bitfield for extra features, ignored on some platforms. */
	#define IO_PULLUP   1
	#define IO_DEGLITCH 2
	unsigned extra;
};

/*
 *    IOCTL commands. The IOCTL argument can be the pin number or a pointer to
 *    struct cdcs_gpio_data. Unless otherwise noted, the return values are 0
 *    for success and <0 for error.
 */
#define CDCS_IOCTL(n) (0xcdc50000 + (n))

/* Request reservation of pin. Closing the file descriptor drops all reservations
   for this process. Argument is pin number. */
#define CDCS_GPIO_REQUEST      CDCS_IOCTL(1)
/* Free the previously requested pin. Argument is pin number. */
#define CDCS_GPIO_FREE         CDCS_IOCTL(2)

/* Set pin multiplexer. Allowable states depend on the specific platform and pin
   number. Argument is pointer to struct cdcs_gpio_data, input mux value in arg->ea.mux. */
#define CDCS_GPIO_SET_MUX      CDCS_IOCTL(3)
/* Set pin to GPIO mode, direction input. Argument is pin number. */
#define CDCS_GPIO_SET_DIR_IN   CDCS_IOCTL(4)
/* Set pin to GPIO mode, direction output. Argument is pin number. */
#define CDCS_GPIO_SET_DIR_OUT  CDCS_IOCTL(5)
/* Sets pin to GPIO mode, direction output and specified value. Argument is pointer to
   struct cdcs_gpio_data, value is arg->ea.value. */
#define CDCS_GPIO_WRITE_OUT    CDCS_IOCTL(6)

/* Read the value of pin. Returns pin value (0|1) or negative error number. */
#define CDCS_GPIO_READ         CDCS_IOCTL(7)
/* Write value to pin. Argument is pointer to struct cdcs_gpio_data, value is arg->ea.value. */
#define CDCS_GPIO_WRITE        CDCS_IOCTL(8)
/* Set GPIO pin value to '1'. Argument is pin number. */
#define CDCS_GPIO_SET          CDCS_IOCTL(9)
/* Set GPIO pin value to '0'. Argument is pin number. */
#define CDCS_GPIO_CLEAR        CDCS_IOCTL(10)

/* Enable a userspace 'interrupt' for pin. A userspace 'interrupt' is a signal, getting
   triggered by any change in the gpio line. Argument is pointer to struct cdcs_gpio_data,
   required signal is arg->ea.signal. */
#define CDCS_GPIO_ENABLE_INT  CDCS_IOCTL(11)
/* Disable a userspace 'interrupt' for pin. Argument is pin number. */
#define CDCS_GPIO_DISABLE_INT CDCS_IOCTL(12)


#endif /* __CDCS_GPIO_H */
