/*
 * Support for NetComm boards. We borrow a lot of code from the TI
 * the xevm board, including the machine ID.
 *
 * The original TI code used profiles to select available hardware. Additionally,
 * extension boards are supported, identified via I2C eeprom.
 *
 * For the NetComm architecture, we don't support extension boards as such, since
 * the Betzy board can't operate stand alone. The different supported IO boards
 * are supported via separate profiles. BOARD_NONE and IOBOARD_NONE is used for inactive
 * profiles, unused in the currently supported boards, *_ALL matches all profiles.
 *
 * The profile is a combination of BOARD_* and IOBOARD_* bit patterns, denoting
 * the CPU board and IOBOard respectively.
 *
 * Copyright (C) 2013 NetComm Wireless, Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 * Copyright (C) 2011 Texas Instruments, Inc. - http://www.ti.com/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/i2c/at24.h>
#include <linux/phy.h>
#include <linux/gpio.h>
#include <linux/spi/spi.h>
#include <linux/spi/flash.h>
#include <linux/gpio_keys.h>
#include <linux/input.h>
#include <linux/input/matrix_keypad.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/err.h>
#include <linux/export.h>
#include <linux/wl12xx.h>
#include <linux/ethtool.h>
#include <linux/mfd/tps65910.h>
#include <linux/mfd/tps65217.h>
#include <linux/regulator/userspace-consumer.h>
#include <linux/pwm_backlight.h>
#include <linux/input/ti_tsc.h>
#include <linux/platform_data/ti_adc.h>
#include <linux/mfd/ti_tscadc.h>
#include <linux/reboot.h>
#include <linux/pwm/pwm.h>
#include <linux/rtc/rtc-omap.h>
#include <linux/opp.h>

#include <mach/hardware.h>
#include <mach/board-netcomm.h>

#include <asm/mach-types.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/hardware/asp.h>

#include <plat/omap_device.h>
#include <plat/omap-pm.h>
#include <plat/irqs.h>
#include <plat/board.h>
#include <plat/common.h>
#include <plat/usb.h>
#include <plat/mmc.h>
#include <plat/emif.h>
#include <plat/nand.h>

#include <linux/mdio-gpio.h>

#include "board-flash.h"
#include "cpuidle33xx.h"
#include "mux.h"
#include "devices.h"
#include "hsmmc.h"

#include "common.h"

#define DLEVEL 2
#include <linux/debug.h>

#include <linux/cdcs_variant.h>
#include <linux/i2c-gpio.h>
/*#include <mach/mxs.h>*/
#include <linux/w1-gpio.h>

/**
 * PHY specific code
 **/

/* This is used during PHY fixup to match the Atheros AR8035 PHY,
 * to set up RGMII Tx clock delay */
#define PHY_AR8035_ID    0x004dd072
#define PHY_AR8035_MASK  0x00fffff0

/* 
	this USB ID hardware mode has USB0_ID wired to GPIO0_2
*/
#define CONFIG_nguni_hwmod_usbid

/* TSc controller */
static struct tsc_data am335x_touchscreen_data  = {
	.wires  = 0,
	.x_plate_resistance = 200,
	.steps_to_configure = 5
};


static struct adc_data am335x_adc_data = {
	.adc_channels = 8
};

static struct mfd_tscadc_board tscadc = {
	.tsc_init = &am335x_touchscreen_data,
	.adc_init = &am335x_adc_data
};

/**
 *
 * Utilities and helper functions
 *
 **/

/* Convert GPIO signal to GPIO pin number */
#define GPIO_TO_PIN(bank, gpio) (32 * (bank) + (gpio))

#ifdef CONFIG_OMAP_MUX
static struct omap_board_mux board_mux[] __initdata = {
	/*
	 * Early pinmux setup.
	 *
	 * If any modules/usecase requries it in different mode, then subsequent
	 * module init call will change the mux accordingly.
	 */

	/* Configure clock output */
	AM33XX_MUX(XDMA_EVENT_INTR0, OMAP_MUX_MODE3 | AM33XX_PIN_OUTPUT),

	{ .reg_offset = OMAP_MUX_TERMINATOR },
};
#else
#define	board_mux	NULL
#endif

/* module pin mux structure */
struct pinmux_config {
	const char *string_name; /* signal name format */
	int val; /* Options for the mux register value */
};


struct evm_dev_cfg {
	void (*device_init)(unsigned profile);
	/* Profiles bit mask, can be or-ed together.
	 * Profiles represent processor (BOARD_*) and IOBoards (IOBOARD_*). */
	u32 profile;
};

/*
* @pin_mux - single module pin-mux structure which defines pin-mux
*			details for all its pins.
*/
static void setup_pin_mux(struct pinmux_config *pin_mux)
{
	while (pin_mux->string_name != NULL) {
		omap_mux_init_signal(pin_mux->string_name, pin_mux->val);
		pin_mux++;
	}
}


/*
* @dev_cfg - single evm structure which includes
*				all module inits, pin-mux defines
* @profile - profil to configure for.
*/
static void __init _configure_device(struct evm_dev_cfg *dev_cfg, unsigned profile)
{
	int i;

	for (i = 0; dev_cfg->device_init != NULL; dev_cfg++) {
		/* We ignore empty profiles, they are placeholders. */
		if (dev_cfg->profile == 0) continue;

		if (dev_cfg->profile & profile) dev_cfg->device_init(profile);
	}
}


/**
 *
 * Pin multiplexer configuration sets. Depending on the profile
 * a subset of these sets is combined and applied.
 *
 * The pin name string must be in the pinname.modename format.
 **/

/* Pin mux for nand flash module, 8 bit wide. */
static struct pinmux_config nand_pin_mux[] __initdata = {
	{"gpmc_ad0.gpmc_ad0",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad1.gpmc_ad1",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad2.gpmc_ad2",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad3.gpmc_ad3",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad4.gpmc_ad4",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad5.gpmc_ad5",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad6.gpmc_ad6",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad7.gpmc_ad7",	  OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_wait0.gpmc_wait0", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_wpn.gpmc_wpn",	  OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_csn0.gpmc_csn0",	  OMAP_MUX_MODE0 | AM33XX_PULL_DISA},
	{"gpmc_advn_ale.gpmc_advn_ale",  OMAP_MUX_MODE0 | AM33XX_PULL_DISA},
	{"gpmc_oen_ren.gpmc_oen_ren",	 OMAP_MUX_MODE0 | AM33XX_PULL_DISA},
	{"gpmc_wen.gpmc_wen",     OMAP_MUX_MODE0 | AM33XX_PULL_DISA},
	{"gpmc_ben0_cle.gpmc_ben0_cle",	 OMAP_MUX_MODE0 | AM33XX_PULL_DISA},
	{NULL, 0},
};

/* Module pin mux for SPI0 */
static struct pinmux_config spi0_pin_mux[] __initdata = {
	{"spi0_sclk.spi0_sclk", OMAP_MUX_MODE0 | AM33XX_PULL_ENBL
							| AM33XX_INPUT_EN},
	{"spi0_d0.spi0_d0", OMAP_MUX_MODE0 | AM33XX_PULL_ENBL | AM33XX_PULL_UP
							| AM33XX_INPUT_EN},
	{"spi0_d1.spi0_d1", OMAP_MUX_MODE0 | AM33XX_PULL_ENBL
							| AM33XX_INPUT_EN},
	{"spi0_cs0.spi0_cs0", OMAP_MUX_MODE0 | AM33XX_PULL_ENBL | AM33XX_PULL_UP
							| AM33XX_INPUT_EN},
	{"spi0_cs1.spi0_cs1", OMAP_MUX_MODE0 | AM33XX_PULL_ENBL | AM33XX_PULL_UP
							| AM33XX_INPUT_EN},
	/* To match Atmel, adding GPIO1_13 & GPIO1_14 as extra CS.
	 * This requires changes to McSPI driver and platform data. */
	{"gpmc_ad13.gpio1_13", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad15.gpio1_15", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

/* SPI0 pins used as GPIO */
static struct pinmux_config spi0_gpio_pin_mux[] __initdata = {
	{"spi0_sclk.gpio0_2", OMAP_MUX_MODE7 | AM33XX_PULL_ENBL
							| AM33XX_INPUT_EN},
	{"spi0_d0.gpio0_3", OMAP_MUX_MODE7 | AM33XX_PULL_ENBL | AM33XX_PULL_UP
							| AM33XX_INPUT_EN},
	{"spi0_d1.gpio0_4", OMAP_MUX_MODE7 | AM33XX_PULL_ENBL
							| AM33XX_INPUT_EN},
	{"spi0_cs0.gpio0_5", OMAP_MUX_MODE7 | AM33XX_PULL_ENBL | AM33XX_PULL_UP
							| AM33XX_INPUT_EN},
	{"spi0_cs1.gpio0_6", OMAP_MUX_MODE7 | AM33XX_PULL_ENBL | AM33XX_PULL_UP
							| AM33XX_INPUT_EN},
	{"gpmc_ad13.gpio1_13", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad15.gpio1_15", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

/* Module pin mux for SPI1 */
static struct pinmux_config spi1_pin_mux[] __initdata = {
	{"mcasp0_aclkx.spi1_sclk", OMAP_MUX_MODE3 | AM33XX_PULL_ENBL
		| AM33XX_INPUT_EN},
	{"mcasp0_fsx.spi1_d0", OMAP_MUX_MODE3 | AM33XX_PULL_ENBL
		| AM33XX_PULL_UP | AM33XX_INPUT_EN},
	{"mcasp0_axr0.spi1_d1", OMAP_MUX_MODE3 | AM33XX_PULL_ENBL
		| AM33XX_INPUT_EN},
	{"mcasp0_ahclkr.spi1_cs0", OMAP_MUX_MODE3 | AM33XX_PULL_ENBL
		| AM33XX_PULL_UP | AM33XX_INPUT_EN},
	{NULL, 0},
};

/* Module pin mux for rgmii1 */
static struct pinmux_config rgmii1_pin_mux[] __initdata = {
	{"mii1_txen.rgmii1_tctl", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"mii1_rxdv.rgmii1_rctl", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_txd3.rgmii1_td3", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"mii1_txd2.rgmii1_td2", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"mii1_txd1.rgmii1_td1", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"mii1_txd0.rgmii1_td0", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"mii1_txclk.rgmii1_tclk", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"mii1_rxclk.rgmii1_rclk", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd3.rgmii1_rd3", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd2.rgmii1_rd2", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd1.rgmii1_rd1", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd0.rgmii1_rd0", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
#ifdef CONFIG_TI_DAVINCI_MDIO
	{"mdio_data.mdio_data", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.mdio_clk", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT_PULLUP},
#else
	{"mdio_data.gpio0_0", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.gpio0_1", OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP},
#endif
	{NULL, 0},
};

/* Module pin mux for rgmii2 (second port) */
static struct pinmux_config rgmii2_pin_mux[] __initdata = {
	{"gpmc_a0.rgmii2_tctl", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"gpmc_a1.rgmii2_rctl", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"gpmc_a2.rgmii2_td3", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"gpmc_a3.rgmii2_td2", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"gpmc_a4.rgmii2_td1", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"gpmc_a5.rgmii2_td0", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"gpmc_a6.rgmii2_tclk", OMAP_MUX_MODE2 | AM33XX_PIN_OUTPUT},
	{"gpmc_a7.rgmii2_rclk", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"gpmc_a8.rgmii2_rd3", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"gpmc_a9.rgmii2_rd2", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"gpmc_a10.rgmii2_rd1", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
	{"gpmc_a11.rgmii2_rd0", OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLDOWN},
#ifdef CONFIG_TI_DAVINCI_MDIO
	{"mdio_data.mdio_data", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.mdio_clk", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT_PULLUP},
#else
	{"mdio_data.gpio0_0", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.gpio0_1", OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP},
#endif
	{NULL, 0},
};

/* Module pin mux for mii1 */
static struct pinmux_config mii1_pin_mux[] __initdata = {
	{"mii1_rxerr.mii1_rxerr", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_txen.mii1_txen", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{"mii1_rxdv.mii1_rxdv", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_txd3.mii1_txd3", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{"mii1_txd2.mii1_txd2", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{"mii1_txd1.mii1_txd1", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{"mii1_txd0.mii1_txd0", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{"mii1_txclk.mii1_txclk", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxclk.mii1_rxclk", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd3.mii1_rxd3", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd2.mii1_rxd2", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd1.mii1_rxd1", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd0.mii1_rxd0", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
#ifdef CONFIG_TI_DAVINCI_MDIO
	{"mdio_data.mdio_data", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.mdio_clk", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT_PULLUP},
#else
	{"mdio_data.gpio0_0", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.gpio0_1", OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP},
#endif
	{NULL, 0},
};

/* Module pin mux for rmii1 */
static struct pinmux_config rmii1_pin_mux[] __initdata = {
	{"mii1_crs.rmii1_crs_dv", OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxerr.mii1_rxerr", OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_txen.mii1_txen", OMAP_MUX_MODE1 | AM33XX_PIN_OUTPUT},
	{"mii1_txd1.mii1_txd1", OMAP_MUX_MODE1 | AM33XX_PIN_OUTPUT},
	{"mii1_txd0.mii1_txd0", OMAP_MUX_MODE1 | AM33XX_PIN_OUTPUT},
	{"mii1_rxd1.mii1_rxd1", OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLDOWN},
	{"mii1_rxd0.mii1_rxd0", OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLDOWN},
	{"rmii1_refclk.rmii1_refclk", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLDOWN},
#ifdef CONFIG_TI_DAVINCI_MDIO
	{"mdio_data.mdio_data", OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.mdio_clk", OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT_PULLUP},
#else
	{"mdio_data.gpio0_0", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{"mdio_clk.gpio0_1", OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP},
#endif
	{NULL, 0},
};

/* Module pin mux for I2C port 0 */
static struct pinmux_config i2c0_pin_mux[] __initdata = {
	{"i2c0_sda.i2c0_sda", OMAP_MUX_MODE0 | AM33XX_SLEWCTRL_SLOW |
					AM33XX_PIN_INPUT_PULLUP},
	{"i2c0_scl.i2c0_scl", OMAP_MUX_MODE0 | AM33XX_SLEWCTRL_SLOW |
					AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

/* GPIO I2C0 - the I2C controller is buggy and glitches the lines, we use bitbashing */
static struct pinmux_config betzy_i2c_gpio_pin_mux[] __initdata = {
	{"i2c0_sda.gpio3_5",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP | AM33XX_SLEWCTRL_SLOW},
	{"i2c0_scl.gpio3_6",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP | AM33XX_SLEWCTRL_SLOW},
	{NULL, 0},
};

/* Module pin mux for I2C port 1 */
static struct pinmux_config i2c1_pin_mux[] __initdata = {
	{"spi0_d1.i2c1_sda",    OMAP_MUX_MODE2 | AM33XX_SLEWCTRL_SLOW |
					AM33XX_PIN_INPUT_PULLUP},
	{"spi0_cs0.i2c1_scl",   OMAP_MUX_MODE2 | AM33XX_SLEWCTRL_SLOW |
					AM33XX_PIN_INPUT_PULLUP},
};

/* Module pin mux for I2C port 2 */
static struct pinmux_config i2c2_pin_mux[] __initdata = {

	{"uart1_ctsn.i2c2_sda",    OMAP_MUX_MODE3 | AM33XX_SLEWCTRL_SLOW |
					AM33XX_PIN_INPUT_PULLUP},
	{"uart1_rtsn.i2c2_scl",   OMAP_MUX_MODE3 | AM33XX_SLEWCTRL_SLOW |
					AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

/* Module pin mux for mmc0 */
static struct pinmux_config mmc0_common_pin_mux[] __initdata = {
	{"mmc0_dat3.mmc0_dat3",	OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mmc0_dat2.mmc0_dat2",	OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mmc0_dat1.mmc0_dat1",	OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mmc0_dat0.mmc0_dat0",	OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mmc0_clk.mmc0_clk",	OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{"mmc0_cmd.mmc0_cmd",	OMAP_MUX_MODE0 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

static struct pinmux_config mmc0_wp_only_pin_mux[] __initdata = {
	{"mcasp0_aclkr.gpio3_18", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};
static struct pinmux_config mmc0_cd_only_pin_mux[] __initdata = {
	{"spi0_cs1.gpio0_6",  OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

/* Module pin mux for mmc1 */
static struct pinmux_config mmc1_common_pin_mux[] __initdata = {
	{"gpmc_ad3.mmc1_dat3",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad2.mmc1_dat2",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad1.mmc1_dat1",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad0.mmc1_dat0",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_csn1.mmc1_clk",	OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_csn2.mmc1_cmd",	OMAP_MUX_MODE2 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};
static struct pinmux_config mmc1_dat4_7_pin_mux[] __initdata = { /* For MMCplus 8-bit cards */
	{"gpmc_ad7.mmc1_dat7",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad6.mmc1_dat6",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad5.mmc1_dat5",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{"gpmc_ad4.mmc1_dat4",	OMAP_MUX_MODE1 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};
static struct pinmux_config mmc1_wp_only_pin_mux[] __initdata = {
	{"gpmc_csn0.gpio1_29",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};
static struct pinmux_config mmc1_cd_only_pin_mux[] __initdata = {
	{"gpmc_advn_ale.gpio2_2", OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},
	{NULL, 0},
};

/* Module pin mux for uart3 */
static struct pinmux_config uart3_pin_mux[] __initdata = {
	{"spi0_cs1.uart3_rxd", AM33XX_PIN_INPUT_PULLUP},
	{"ecap0_in_pwm0_out.uart3_txd", AM33XX_PULL_ENBL},
	{NULL, 0},
};

/* Module pin mux for uart2 */
static struct pinmux_config uart2_pin_mux[] __initdata = {
	{"spi0_sclk.uart2_rxd", OMAP_MUX_MODE1 | AM33XX_SLEWCTRL_SLOW |
						AM33XX_PIN_INPUT_PULLUP},
	{"spi0_d0.uart2_txd", OMAP_MUX_MODE1 | AM33XX_PULL_UP |
						AM33XX_PULL_DISA |
						AM33XX_SLEWCTRL_SLOW},
	{NULL, 0},
};

/* pinmux for usb0 drvvbus */
static struct pinmux_config usb0_pin_mux[] __initdata = {
	{"usb0_drvvbus.usb0_drvvbus",    OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{NULL, 0},
};

/* pinmux for usb1 drvvbus */
static struct pinmux_config usb1_pin_mux[] __initdata = {
	{"usb1_drvvbus.usb1_drvvbus",    OMAP_MUX_MODE0 | AM33XX_PIN_OUTPUT},
	{NULL, 0},
};


/** CPU boards GPIO **/

/* Betzy CPU board specific GPIO mux settings */
static struct pinmux_config board_betzy_gpio_pin_mux[] __initdata = {
	{"gpmc_a5.gpio1_21",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // DEBUG HEARTBEAT LED
	{NULL, 0},
};

/* Nguni CPU board specific GPIO mux settings */
static struct pinmux_config board_nguni_gpio_pin_mux[] __initdata = {
	
	{NULL, 0},
};

/* Kewel CPU board specific GPIO mux settings */
static struct pinmux_config board_kewel_gpio_pin_mux[] __initdata = {
	
	{NULL, 0},
};


/** IOBoards GPIO **/

/* Koala IOBoard specific GPIO mux settings */
static struct pinmux_config ioboard_koala_gpio_pin_mux[] __initdata = {
	{"xdma_event_intr0.xdma_event_intr0",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_19 
	{"xdma_event_intr1.clkout2",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_20
	{"gpmc_ad8.lcd_data16",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_22
	{"gpmc_ad9.lcd_data17",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_23
	{"gpmc_ad10.lcd_data18",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_26
	{"gpmc_ad11.lcd_data19",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_27
	{"gpmc_ad12.lcd_data20",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_12
	{"gpmc_ad13.lcd_data21",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_13
	{"gpmc_ad14.lcd_data22",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_14
	{"gpmc_ad15.lcd_data23",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_15
 	
 	{"mmc0_clk.mmc0_clk",           OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // BUZZER
	{"gpmc_csn3.mmc2_cmd", 		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_0 - 5V_EN

	{"lcd_vsync.lcd_vsync",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_22
	{"lcd_hsync.lcd_hsync",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_23
	{"lcd_pclk.lcd_pclk",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},   // GPIO2_24 - n3G_PSW
	{"lcd_ac_bias_en.lcd_ac_bias_en",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_25
	{"mcasp0_aclkr.mmc0_sdwp",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO3_18
	{"mcasp0_fsr.pr1_pru0_pru_r30_5",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO3_19
	{"mcasp0_axr1.mcasp1_axr0",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO3_20
	{NULL, 0},
};

/* Newman IOBoard specific GPIO mux settings */
static struct pinmux_config ioboard_newman_gpio_pin_mux[] __initdata = {
	/* TODO: Clean this up, at the moment this is a copy of Koala, probably wrong.
	 * It works sufficiently for factory programming of Betzy boards. */
	{"xdma_event_intr0.xdma_event_intr0",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_19 
	{"xdma_event_intr1.clkout2",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_20
	{"gpmc_ad8.lcd_data16",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_22
	{"gpmc_ad9.lcd_data17",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_23
	{"gpmc_ad10.lcd_data18",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_26
	{"gpmc_ad11.lcd_data19",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO0_27
	{"gpmc_ad12.lcd_data20",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_12
	{"gpmc_ad13.lcd_data21",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_13
	{"gpmc_ad14.lcd_data22",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_14
	{"gpmc_ad15.lcd_data23",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO1_15
 	
 	{"mmc0_clk.mmc0_clk",           OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // BUZZER
	{"gpmc_csn3.mmc2_cmd", 		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_0 - 5V_EN

	{"lcd_vsync.lcd_vsync",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_22
	{"lcd_hsync.lcd_hsync",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_23
	{"lcd_pclk.lcd_pclk",		OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP},   // GPIO2_24 - n3G_PSW
	{"lcd_ac_bias_en.lcd_ac_bias_en",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO2_25
	{"mcasp0_aclkr.mmc0_sdwp",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO3_18
	{"mcasp0_fsr.pr1_pru0_pru_r30_5",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO3_19
	{"mcasp0_axr1.mcasp1_axr0",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLDOWN}, // GPIO3_20
	{NULL, 0},
};

/* Elaine IOBoard specific GPIO mux settings */
static struct pinmux_config ioboard_elaine_gpio_pin_mux[] __initdata = {
	/* TODO: Fill this when needed */
	{NULL, 0},
};


/* Nguni IOBoard specific GPIO mux settings */
static struct pinmux_config ioboard_nguni_gpio_pin_mux[] __initdata = {
	/* LEDS */
	{"gpio0_20",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED1r
	{"gpio0_19",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED1g
	{"gpio1_15",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED2r
	{"gpio1_14",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED2g
	{"gpio1_13",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED3r
	{"gpio1_12",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED3g
	{"gpio0_27",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED4r
	{"gpio0_26",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED4g
	{"gpio0_23",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED5r
	{"gpio0_22",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED5g
	{"gpio3_18",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED6r
	{"gpio3_14",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED6g
	{"gpio3_16",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED7r
	{"gpio3_17",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED7g
	{"gpio3_15",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED8r
	{"gpio0_10",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED8g

	#ifdef CONFIG_nguni_hwmod_usbid
	{"gpio0_2",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP }, // USB0_ID (only on EVT4)
	#endif

	/* Ethernet PHY GPIO lines */
	{"gpio2_23",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // EARSTn
	{"gpio2_22",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // EBRSTn
	{"gpio3_1",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // EAINT
	{"gpio3_0",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // EBINT
	/* Power control */
	{"gpio2_24",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // 5V_EN
	{"gpio2_25",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // 8.2V_EN
	/* USB */
	{"gpio0_13",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // nRESET_USB Hub
	//{"gpio0_18",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // USB0_PWR_EN (TODO: Check this)
	{"gpio0_15",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // USB0_FAULTn
	/* Module */
	{"gpio3_2",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // LTE_DPR
	{"gpio0_12",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // LTE_DISABLE
	{"gpio2_1",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // LTE_POWER_SW
	{"gpio3_21",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT}, // SIM_CDn
	{"gpio3_7",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT}, // SIM_SEL
	/* WiFi */
	{"gpio3_20",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // WiFi EN
	{"gpio2_0",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // nReset WF

	/* Auxiliary GPIO */
	{"gpio1_28",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // XGPIO_ON
	{"gpio1_30",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // XGPIO_PEN
	{"gpio0_7",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // IGNn (OC transistor output)
	/* Reset button */
	{"gpio0_14",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // USR_RESETn
	/* MMC */
	{"mcasp0_fsr.pr1_pru0_pru_r30_5",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT}, // SD_CDn (GPIO3_19)
	/* End marker */
	{NULL, 0},
};

/* Kudu IOBoard specific GPIO mux settings */
static struct pinmux_config ioboard_kudu_gpio_pin_mux[] __initdata = {
	/* LEDS */
	{"gpio1_15",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED1r
	{"gpio1_14",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED1g
	{"gpio0_27",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED2r
	{"gpio0_26",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED2g
	{"gpio0_23",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED3r
	{"gpio0_22",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED3g
	{"gpio3_17",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED4r
	{"gpio3_18",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED4g
	{"gpio0_20",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED5r
	{"gpio0_19",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED5g
	{"gpio0_10",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED6r
	{"gpio3_16",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED6g
	{"gpio1_13",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED7r
	{"gpio1_12",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED7g
	{"gpio3_15",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED8r
	{"gpio3_14",	OMAP_MUX_MODE7 | AM33XX_PULL_DISA | AM33XX_SLEWCTRL_SLOW}, // LED8g

	{"gpio0_2",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP }, // USB0_ID

	/* Ethernet Switch GPIO lines */
	{"gpio2_23",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // GB_SW_RSTn

	/* Power control */
	{"gpio2_24",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // 5V_EN

	/* USB */
	{"gpio0_13",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // nRESET_USB Hub
	{"gpio0_18",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // USB0_PWR_EN  
	{"gpio0_15",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // USB0_FAULTn

	/* Module */
	{"gpio3_2",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // LTE_DPR
	{"gpio0_12",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // LTE_DISABLE
	{"gpio2_1",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // LTE_POWER_SW
	{"gpio3_21",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT}, // SIM_CDn
	{"gpio3_7",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT}, // SIM_SEL

	/* WiFi */
	{"gpio3_20",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // WiFi Sleep  
	{"gpio2_0",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT_PULLUP}, // nReset WF

	/* Auxiliary GPIO */
	{"gpio2_25",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // XGPIO_MON
	{"gpio1_30",	OMAP_MUX_MODE7 | AM33XX_PIN_OUTPUT}, // XGPIO_PEN
	{"gpio0_7",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // Power fail

	/* Reset button */
	{"gpio0_14",	OMAP_MUX_MODE7 | AM33XX_PIN_INPUT_PULLUP}, // USR_RESETn

	/* MMC */
	{"mcasp0_fsr.pr1_pru0_pru_r30_5",OMAP_MUX_MODE7 | AM33XX_PIN_INPUT}, // SD_CDn (GPIO3_19)

	/* End marker */
	{NULL, 0},
};

/* Hamster (LTE-Mice) IOBoard specific GPIO mux settings */
static struct pinmux_config ioboard_hamster_gpio_pin_mux[] __initdata = {
	/* TODO: Add this */

	/* End marker */
	{NULL, 0},
};

/**
 *
 * Harware module initialisation functions. Usually just setting up the pinmux,
 * but there are exceptions.
 *
 **/


/* Boilerplate pinmux only function declaration. NAME_init is function name,
 * NAME_pin_mux is the pinmux array used for setup. */
#define DECLARE_PINMUXF(NAME) static void  __init NAME##_init(unsigned profile) \
	{ \
		debug1("(profile=%08x)\n", profile); \
		if (NAME##_pin_mux[0].string_name) \
			setup_pin_mux(NAME##_pin_mux); \
	}

/* CPU board GPIO init functions */
DECLARE_PINMUXF(board_betzy_gpio)
DECLARE_PINMUXF(board_nguni_gpio)
DECLARE_PINMUXF(board_kewel_gpio)

/* IOBOARD GPIO init functions */
DECLARE_PINMUXF(ioboard_koala_gpio)
DECLARE_PINMUXF(ioboard_newman_gpio)
DECLARE_PINMUXF(ioboard_elaine_gpio)
DECLARE_PINMUXF(ioboard_nguni_gpio)
DECLARE_PINMUXF(ioboard_hamster_gpio)
DECLARE_PINMUXF(ioboard_kudu_gpio)

#ifndef CONFIG_TI_DAVINCI_MDIO
/* Ethernet PHY init. The MDIO implementation in the MAC has hardware
 * glitches, using bitbang-gpio-mdio. */

static struct mdio_gpio_platform_data mdio_gpio_pd = {
	.mdc = GPIO_TO_PIN(0, 1),
	.mdio = GPIO_TO_PIN(0, 0),
};

static struct platform_device mdio_gpio_pdev = {
	.name = "mdio-gpio",
	.id = 0,
	.dev = {
		.platform_data = &mdio_gpio_pd,
	},
};

static void __init PHY_bitbang(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	if (profile & IOBOARD_KUDU) {
		/* We rename this bus, so it doesn't clash with
		 * the fake / fixed PHY bus needed for the MAC */
		mdio_gpio_pdev.id = 1;
	}
	platform_device_register(&mdio_gpio_pdev);
}
#endif

/* Ethernet init */

/* Fake / fixed PHY setup for permanent Gigabit RGMII links to switch */
#include <linux/phy_fixed.h>
static struct fixed_phy_status fixed_phy_status_gbit  = {
	.link		= 1,
	.speed		= 1000,
	.duplex		= 1,
};

static void __init PHY_setup(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);

	if (profile & IOBOARD_KUDU) {
		int ret;
		/* For PHY-less boards which are using the fake / fixed phy,
		 * We register the fake PHY devices here. We register two devices,
		 * hardcoded to GBIT. WARNING: The real (bitbanged) MDIO bus
		 * should have ID=1, since 0 is hardcoded in the fixed driver. */
		ret = fixed_phy_add(PHY_POLL, 0x00, &fixed_phy_status_gbit);
		if (ret < 0) debug1("fixed_phy_add returned %d\n", ret);
		ret = fixed_phy_add(PHY_POLL, 0x01, &fixed_phy_status_gbit);
		if (ret < 0) debug1("fixed_phy_add returned %d\n", ret);
	}
}

static void __init MAC_Setup(unsigned profile)
{
	/* Default MAC addresses if nothing valid is already in the register. */
	char macs[2][ETH_ALEN] = {
		{ 0x22, 0x33, 0x22, 0x22, 0x22, 0x22 },
		{ 0x22, 0x44, 0x22, 0x22, 0x22, 0x22 }};

	/* Fillup global mac id */
	am33xx_cpsw_macidfillup(&macs[0][0],
				&macs[1][0]);
}

/* MAC1 slice in rgmii mode */
static void __init rgmii1_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	MAC_Setup(profile);
	setup_pin_mux(rgmii1_pin_mux);
}

/* MAC2 slice in rgmii mode */
static void __init rgmii2_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	MAC_Setup(profile);
	setup_pin_mux(rgmii2_pin_mux);
}

static int ar8035_phy_fixup(struct phy_device *phydev)
{
	/* We need to enable a RGMII tx clock delay in the PHY,
	 * since TI have deprecated the funtion within the processor.
	 * Using PHY debug registers, for Atheros AR8035 only. */

	/* debug register 5, enable RGMII TxClock Delay */
	phy_write(phydev, 0x1d, 0x5);
	phy_write(phydev, 0x1e, 0x2d47);

	/* Debug register 0xb, setting TxClock Delay */
	phy_write(phydev, 0x1d, 0xb);
	//phy_write(phydev, 0x1e, 0x3c00); /* 0.7ns, works badly, glitchy. */
	//phy_write(phydev, 0x1e, 0x3c20); /* 1.3ns, seems OK */
	phy_write(phydev, 0x1e, 0x3c40); /* 2.4ns, seems OK. Optimal value, furthest away from trouble. */
	//phy_write(phydev, 0x1e, 0x3c60); /* 3.4ns, works badly when freezer sprayed, OK otherwise. */

	return 0;
}

/* Common RGMII setup code, plus some fixups. Call after rgmiiX_init functions. */
static void __init rgmii_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);

	/* This sets up the MAC(s) */
	am33xx_cpsw_init(AM33XX_CPSW_MODE_RGMII, NULL, NULL);

	/* This sets up the RGMII Tx clock delay for ar8035 PHYs.
	 * Only for boards that use this particular PHY. */
	if (profile & (IOBOARD_KOALA | IOBOARD_NGUNI | IOBOARD_HAMSTER)) {
		/* Atheros Tx Clk delay Phy fixup */
		phy_register_fixup_for_uid(PHY_AR8035_ID, PHY_AR8035_MASK,
			ar8035_phy_fixup);
	}
}


/* MAC1 slice in MII mode */
static void __init mii1_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	MAC_Setup(profile);
	setup_pin_mux(mii1_pin_mux);
}

/* MAC1 slice in RMII mode, external refclk. Check errata. */
static void __init rmii1_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	MAC_Setup(profile);
	setup_pin_mux(rmii1_pin_mux);

	/* For the Newman board: we need to swap PHY identifiers, otherwise
	 * the (single) PHY gets assigned as 0:01 and assumed to service the
	 * unused MAC port. */
	am33xx_cpsw_init(AM33XX_CPSW_MODE_RMII, "0:01", "0:00");
}

/* USB0 */
DECLARE_PINMUXF(usb0)

/* USB1 */
DECLARE_PINMUXF(usb1)

/* uart3 */
DECLARE_PINMUXF(uart3)

/* uart2 */
DECLARE_PINMUXF(uart2)


/* NAND interface */

static struct mtd_partition am335x_nand_partitions[] __initdata = { }; /* Passed in from kernel command line */

/* values in ns. TODO: Use ONFI for this */
static struct gpmc_timings am335x_nand_timings __initdata = {
	.sync_clk = 0,

	.cs_on = 0,
	.cs_rd_off = 44,
	.cs_wr_off = 44,

	.adv_on = 6,
	.adv_rd_off = 34,
	.adv_wr_off = 44,
	.we_off = 40,
	.oe_off = 54,

	.access = 64,
	.rd_cycle = 82,
	.wr_cycle = 82,

	.wr_access = 40,
	.wr_data_mux_bus = 0,
};

static void __init evm_nand_init(unsigned profile)
{
	struct omap_nand_platform_data *pdata;
	struct gpmc_devices_info gpmc_device[2] = {
		{ NULL, 0 },
		{ NULL, 0 },
	};

	debug1("(profile=%08x)\n", profile);

	setup_pin_mux(nand_pin_mux);
	pdata = omap_nand_init(am335x_nand_partitions,
		ARRAY_SIZE(am335x_nand_partitions), 0, 0,
		&am335x_nand_timings);

	if (!pdata)
		return;

	pdata->ecc_opt =OMAP_ECC_BCH8_CODE_HW;
	pdata->elm_used = true;

	gpmc_device[0].pdata = pdata;
	gpmc_device[0].flag = GPMC_DEVICE_NAND;

	omap_init_gpmc(gpmc_device, sizeof(gpmc_device));
	omap_init_elm();
}


/* SPI 0 */
static struct spi_board_info am335x_spi0_slave_info[] __initdata = {
#if 0
	/* Example */
	{
		.modalias      = "m25p80",
		.platform_data = &am335x_spi_flash,
		.irq           = -1,
		.max_speed_hz  = 24000000,
		.bus_num       = 1,
		.chip_select   = 0,
	},
#endif
};

static void __init spi0_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(spi0_pin_mux);
	spi_register_board_info(am335x_spi0_slave_info,
			ARRAY_SIZE(am335x_spi0_slave_info));
}

static void __init spi0_gpio_init(unsigned profile)
{
	/* Configure SPI0 pads as GPIO, e.g. for bitbashing SPI or CPLD passthrough */
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(spi0_gpio_pin_mux);
}

/* SPI 1 */

static struct spi_board_info am335x_spi1_slave_info[] __initdata = {
};

/* setup spi1 */
static void __init spi1_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(spi1_pin_mux);
	spi_register_board_info(am335x_spi1_slave_info,
			ARRAY_SIZE(am335x_spi1_slave_info));
}

/*
 * Power supplies
 */

static struct regulator_init_data am335x_dummy = {
	.constraints.always_on	= true,
};

static struct regulator_consumer_supply am335x_vdd1_supply[] = {
	REGULATOR_SUPPLY("vdd_mpu", NULL),
};

static struct regulator_init_data am335x_vdd1 = {
	.constraints = {
		.min_uV			= 600000,
		.max_uV			= 1500000,
		.valid_modes_mask	= REGULATOR_MODE_NORMAL,
		.valid_ops_mask		= REGULATOR_CHANGE_VOLTAGE,
		.always_on		= 1,
	},
	.num_consumer_supplies	= ARRAY_SIZE(am335x_vdd1_supply),
	.consumer_supplies	= am335x_vdd1_supply,
};

static struct regulator_consumer_supply am335x_vdd3_supply[] = {
	REGULATOR_SUPPLY("v5_aux", "reg-userspace-consumer.0"),
};

static struct regulator_init_data am335x_vdd3 = {
	.constraints = {
		.min_uV				= 5000000,
		.max_uV				= 5000000,
		.valid_modes_mask	= REGULATOR_MODE_NORMAL,REGULATOR_MODE_IDLE,
		.valid_ops_mask		= REGULATOR_CHANGE_STATUS,
		.always_on			= 0,
	},
	.num_consumer_supplies	= ARRAY_SIZE(am335x_vdd3_supply),
	.consumer_supplies	= am335x_vdd3_supply,
};

/* Platform data for the I2C power supply driver. */
static struct tps65910_board am335x_tps65910_info = {
	.tps65910_pmic_init_data[TPS65910_REG_VRTC]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VIO]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VDD1]	= &am335x_vdd1,
	.tps65910_pmic_init_data[TPS65910_REG_VDD2]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VDD3]	= &am335x_vdd3,
	.tps65910_pmic_init_data[TPS65910_REG_VDIG1]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VDIG2]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VPLL]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VDAC]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VAUX1]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VAUX2]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VAUX33]	= &am335x_dummy,
	.tps65910_pmic_init_data[TPS65910_REG_VMMC]	= &am335x_dummy,
	.en_ck32k_xtal=true,
};


/*
 * Allow userspace control of usb_vbus via the "userspace consumer" interface
 * 
 * The interface can be found at /sys/devices/platform/reg-userspace-consumer.0/
 * 
 */

static struct regulator_bulk_data am335x_vusb_consumer_supply = {
       .supply         = "v5_aux",
};

static struct regulator_userspace_consumer_data am335x_vusb_consumer_data = {
        .name           = "vusb bus",
        .num_supplies   = 1,
        .supplies       = &am335x_vusb_consumer_supply
};

static struct platform_device am335x_vusb_userspace_consumer = {
        .name           = "reg-userspace-consumer",
        .id             = 0,
        .dev            = {
                .platform_data = &am335x_vusb_consumer_data,
        }
};

/* Userspace control of processor core voltage, using the virtual consumer driver.
 * This is needed to adjust core voltage to match clock speed.
 * Available as /sys/devices/platform/reg-virt-consumer.0/
 * This relies on CONFIG_REGULATOR_VIRTUAL_CONSUMER in the kernel being
 * enabled. The regulator is matched based on the string in dev.platform_data
 * matching the regulator tag in REGULATOR_SUPPLY() above, not the name
 * showing up in /sys/class/regulator/.../name.
 */

static struct platform_device am335x_vdd_mpu = {
        .name           = "reg-virt-consumer",
        .id             = 0,
	.dev            = {
		.platform_data = "vdd_mpu",
        }
};


static struct platform_device *am335x_userspace_consumers[] = {
        &am335x_vusb_userspace_consumer,
	&am335x_vdd_mpu,
};


static void __init betzy_userspace_consumers_init(void)
{
        platform_add_devices(am335x_userspace_consumers, ARRAY_SIZE(am335x_userspace_consumers));
};


/*
* I2C device mapping & init
*/
static struct i2c_board_info am335x_i2c0_boardinfo[] = {
	{ /* Power supply */
		I2C_BOARD_INFO("tps65910", TPS65910_I2C_ID1),
		.platform_data  = &am335x_tps65910_info,
	}, 
	{
		I2C_BOARD_INFO("ds3232", 0x68),
	},

};

static void __init i2c0_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(i2c0_pin_mux);
	omap_register_i2c_bus(1, 100, am335x_i2c0_boardinfo,
				ARRAY_SIZE(am335x_i2c0_boardinfo));
}


static struct i2c_board_info am335x_i2c1_boardinfo[] __initdata = {
};

static void __init i2c1_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(i2c1_pin_mux);
	omap_register_i2c_bus(2, 100, am335x_i2c1_boardinfo,
			ARRAY_SIZE(am335x_i2c1_boardinfo));
}

static struct i2c_board_info am335x_i2c2_boardinfo[] __initdata = {
};

static void __init i2c2_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(i2c2_pin_mux);
	omap_register_i2c_bus(3, 100, am335x_i2c2_boardinfo,
			ARRAY_SIZE(am335x_i2c2_boardinfo));
}


/*
 * MMC initialisation
 */

static struct omap2_hsmmc_info am335x_mmc[] __initdata = {
	{
		.mmc            = 1,
		.caps           = MMC_CAP_4_BIT_DATA,
		/* The GPIO assignment here is for Newman. */
		.gpio_cd        = -EINVAL,
		.gpio_wp        = -EINVAL,
		.ocr_mask       = MMC_VDD_32_33 | MMC_VDD_33_34, /* 3V3 */
	},
	{
		.mmc            = 0,	/* will be set at runtime */
	},
	{
		.mmc            = 0,	/* will be set at runtime */
	},
	{}      /* Terminator */
};

static struct omap2_hsmmc_info ioboard_nguni_am335x_mmc[] __initdata = {
	{
		.mmc            = 1,
		.caps           = MMC_CAP_4_BIT_DATA,
		.gpio_cd        = GPIO_TO_PIN(3,19),
		.gpio_wp        = -EINVAL,
		.ocr_mask       = MMC_VDD_32_33 | MMC_VDD_33_34, /* 3V3 */
	},
	{
		.mmc            = 0,	/* will be set at runtime */
	},
	{
		.mmc            = 0,	/* will be set at runtime */
	},
	{}      /* Terminator */
};

static void __init ioboard_nguni_mmc0_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(mmc0_common_pin_mux);
	(void)mmc0_wp_only_pin_mux; /* Not used ATM */
	(void)mmc0_cd_only_pin_mux; /* Not used ATM */

	omap2_hsmmc_init(ioboard_nguni_am335x_mmc);
}

static void __init mmc0_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(mmc0_common_pin_mux);
	(void)mmc0_wp_only_pin_mux; /* Not used ATM */
	(void)mmc0_cd_only_pin_mux; /* Not used ATM */

	omap2_hsmmc_init(am335x_mmc);
}

static void mfd_tscadc_init(unsigned profile)
{
	int err;
	err = am33xx_register_mfd_tscadc(&tscadc);
	if (err)
	  pr_err("failed to register touchscreen device\n");
}

/* mmc1_init must be called before mmc0_init. */
static void __init mmc1_init(unsigned profile)
{
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(mmc1_common_pin_mux);
	setup_pin_mux(mmc1_dat4_7_pin_mux);
	setup_pin_mux(mmc1_wp_only_pin_mux);
	setup_pin_mux(mmc1_cd_only_pin_mux);

	am335x_mmc[1].mmc = 2;
	am335x_mmc[1].caps = MMC_CAP_4_BIT_DATA;
	am335x_mmc[1].gpio_cd = GPIO_TO_PIN(2, 2);
	am335x_mmc[1].gpio_wp = GPIO_TO_PIN(1, 29);
	am335x_mmc[1].ocr_mask = MMC_VDD_32_33 | MMC_VDD_33_34; /* 3V3 */

	/* mmc will be initialized when mmc0_init is called */
}


/*
 * LEDs
 */

/* This platform data cannot be placed into __initdata, so we use
 * the preprocessor to avoid unnecessary waste. */
static struct gpio_led gpio_leds[] = {
#ifdef KV_BOARD_Betzy
	/* This LED is on the processor board */
	{
		.name			= "debug",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(1, 21),
		.active_low		= false,
	},
#endif
#ifdef KV_LEDMAP_newman_8000
	/* Not all LEDS can be used, for production only */
	{
		.name			= "led2",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 20),
		.active_low		= true,
	},
	{
		.name			= "led3",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(3, 21),
		.active_low		= true,
	},
	{
		.name			= "led4",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 23),
		.active_low		= true,
	},
	{
		.name			= "led8",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 23),
		.active_low		= true,
	},
	{
		.name			= "led9",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(1, 15),
		.active_low		= true,
	},
#endif /* KV_LEDMAP_newman_8000 */
#ifdef KV_LEDMAP_3g30ap
/* 3g30 I/O board */
	{	/* Power, red */
		.name			= "power",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(2, 28),
		.active_low		= 1,
	},
	{	/* WWAN activity, amber */
		.name			= "wwan",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 22),
		.active_low		= 1,
	},
	{	/* DCD, green */
		.name			= "dcd",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 19),
		.active_low		= 1,
	},
	{	/* service, green */
		.name			= "service",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 20),
		.active_low		= 1,
	},
	{	/* RSSI, green */
		.name			= "rssi",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(3, 21),
		.active_low		= 1,
	},
#endif
#ifdef KV_LEDMAP_3g30apo
	#error "Not implemented yet"
#endif
#ifdef KV_LEDMAP_nguni
	{
		.name			= "led1r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 20),
		.active_low		= 1,
	},
	{
		.name			= "led1g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 19),
		.active_low		= 1,
	},
	{
		.name			= "led2r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(1, 15),
		.active_low		= 1,
	},
	{
		.name			= "led2g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(1, 14),
		.active_low		= 1,
	},
	{
		.name			= "led3r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(1, 13),
		.active_low		= 1,
	},
	{
		.name			= "led3g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(1, 12),
		.active_low		= 1,
	},
	{
		.name			= "led4r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 27),
		.active_low		= 1,
	},
	{
		.name			= "led4g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 26),
		.active_low		= 1,
	},
	{
		.name			= "led5r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 23),
		.active_low		= 1,
	},
	{
		.name			= "led5g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 22),
		.active_low		= 1,
	},
	{
		.name			= "led6r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(3, 18),
		.active_low		= 1,
	},
	{
		.name			= "led6g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(3, 14),
		.active_low		= 1,
	},
	{
		.name			= "led7r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(3, 16),
		.active_low		= 1,
	},
	{
		.name			= "led7g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(3, 17),
		.active_low		= 1,
	},
	{
		.name			= "led8r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(3, 15),
		.active_low		= 1,
	},
	{
		.name			= "led8g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 10),
		.active_low		= 1,
	},
#endif
#ifdef KV_LEDMAP_hamster
	#warning TODO: implement hamster ledmap
#endif
#ifdef KV_LEDMAP_kudu
	{
		.name			= "led1r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(1, 15),
		.active_low		= 1,
	},
	{
		.name			= "led1g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(1, 14),
		.active_low		= 1,
	},
	{
		.name			= "led2r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 27),
		.active_low		= 1,
	},
	{
		.name			= "led2g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 26),
		.active_low		= 1,
	},
	{
		.name			= "led3r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 23),
		.active_low		= 1,
	},
	{
		.name			= "led3g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 22),
		.active_low		= 1,
	},
	{
		.name			= "led4r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(3, 17),
		.active_low		= 1,
	},
	{
		.name			= "led4g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(3, 18),
		.active_low		= 1,
	},
	{
		.name			= "led5r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 20),
		.active_low		= 1,
	},
	{
		.name			= "led5g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(0, 19),
		.active_low		= 1,
	},
	{
		.name			= "led6r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(0, 10),
		.active_low		= 1,
	},
	{
		.name			= "led6g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(3, 16),
		.active_low		= 1,
	},
	{
		.name			= "led7r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(1, 13),
		.active_low		= 1,
	},
	{
		.name			= "led7g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(1, 12),
		.active_low		= 1,
	},
	{
		.name			= "led8r",
		.default_trigger	= "none",
		.gpio			= GPIO_TO_PIN(3, 15),
		.active_low		= 1,
	},
	{
		.name			= "led8g",
		.default_trigger	= "heartbeat",
		.gpio			= GPIO_TO_PIN(3, 14),
		.active_low		= 1,
	},
#endif
};

static struct gpio_led_platform_data gpio_leds_pdata = {
	.leds		= gpio_leds,
	.num_leds	= ARRAY_SIZE(gpio_leds),
};

static struct platform_device gpio_leds_device = {
	.name	= "leds-gpio",
	.id	= -1,
	.dev	= {
		.platform_data	= &gpio_leds_pdata,
	},
};

static void __init led_init(unsigned profile)
{
	debug1("(profile=%08x), %d LEDs\n", profile, ARRAY_SIZE(gpio_leds));
	/* The kernel crashes if an empty array is registered */
	if (ARRAY_SIZE(gpio_leds))
		platform_device_register(&gpio_leds_device);
}

/*
*  Setup a I2C GPIO adaptor device using the SDA/SCL GPIO pins.
*  Defined in <linux/i2c-gpio.h>.
*/
static struct i2c_gpio_platform_data gpio_i2c_pdata = {
	.sda_pin	= GPIO_TO_PIN(3, 5), /* SDA/GPIO3(5) */
	.scl_pin	= GPIO_TO_PIN(3, 6), /* SCL/GPIO3(6) */ 
	.udelay		= 5,  /* Clock speed 500/5 = 100KHz. For 10KHz, this should be 50 */
	.timeout	= 1000, /* # of jiffies for access timeout */
	.sda_is_open_drain = 0, /* turn this off for TI AM335x */
	.scl_is_open_drain = 0, /* turn this off for TI AM335x */
	.scl_is_output_only = 0, /* turn this off for TI AM335x */
};

static struct platform_device gpio_i2c_device = {
	.name	= "i2c-gpio",
	.id	= 1, /* I2C Bus Number for this adaptor */ 
	.dev	= {
		.platform_data	= &gpio_i2c_pdata,
	},
};

/* Setup I2c0 gpio */
static void __init i2c0_gpio_init(unsigned profile)
{
	/* Configure I2C SDA/SCL as GPIO, e.g. for bitbashing I2C implementation */
	debug1("(profile=%08x)\n", profile);
	setup_pin_mux(betzy_i2c_gpio_pin_mux);
	platform_device_register(&gpio_i2c_device);
	/* Register the I2C board info */
	/*  I2C Bus number is 1 */
	i2c_register_board_info(1, am335x_i2c0_boardinfo,
			ARRAY_SIZE(am335x_i2c0_boardinfo));
}

static struct omap_rtc_pdata am335x_rtc_info = {
	.pm_off		= false,
	.wakeup_capable	= 1,
};

static void am335x_rtc_init(unsigned profile)
{
	void __iomem *base;
	struct clk *clk;
	struct omap_hwmod *oh;
	struct platform_device *pdev;
	char *dev_name = "am33xx-rtc";

	clk = clk_get(NULL, "rtc_fck");
	if (IS_ERR(clk)) {
		pr_err("rtc : Failed to get RTC clock\n");
		return;
	}

	if (clk_enable(clk)) {
		pr_err("rtc: Clock Enable Failed\n");
		return;
	}

	base = ioremap(AM33XX_RTC_BASE, SZ_4K);

	if (WARN_ON(!base))
		return;

	/* Unlock the rtc's registers */
	writel(0x83e70b13, base + 0x6c);
	writel(0x95a4f1e0, base + 0x70);

	/*
	 * Enable the 32K OSc
	 * TODO: Need a better way to handle this
	 * Since we want the clock to be running before mmc init
	 * we need to do it before the rtc probe happens
	 */
	writel(0x48, base + 0x54);

	iounmap(base);

	clk_disable(clk);
	clk_put(clk);

	if (omap_rev() == AM335X_REV_ES2_0)
		am335x_rtc_info.wakeup_capable = 1;

	oh = omap_hwmod_lookup("rtc");
	if (!oh) {
		pr_err("could not look up %s\n", "rtc");
		return;
	}

	pdev = omap_device_build(dev_name, -1, oh, &am335x_rtc_info,
			sizeof(struct omap_rtc_pdata), NULL, 0, 0);
	WARN(IS_ERR(pdev), "Can't build omap_device for %s:%s.\n",
			dev_name, oh->name);
}

/**
 *
 * Board initialisation. All hardware init functions are listed here,
 * with a profile bitmask. Depending on the active profiles, they are
 * getting called or not.
 *
 * PROFILE_NONE is a placeholder for currently inactive profiles,
 * PROFILE_ALL means always active.
 *
 **/

static __initdata struct evm_dev_cfg ntc_dev_cfg[] = {

	/* CPU board specific GPIO */
	{board_betzy_gpio_init, BOARD_BETZY},
	{board_nguni_gpio_init, BOARD_NGUNI},
	{board_kewel_gpio_init, BOARD_KEWEL},

	/* IOBOARD specific GPIO */
	{ioboard_koala_gpio_init, IOBOARD_KOALA},
	{ioboard_newman_gpio_init, IOBOARD_NEWMAN},
	{ioboard_elaine_gpio_init, IOBOARD_ELAINE},
	{ioboard_nguni_gpio_init, IOBOARD_NGUNI},
	{ioboard_hamster_gpio_init, IOBOARD_HAMSTER},
	{ioboard_kudu_gpio_init, IOBOARD_KUDU},

	/* NAND Flash (on CPU board) */
	{evm_nand_init,		BOARD_ALL},

	/* LEDS - the led_init functions does the config selection based on
	 * BOARD/IOBOARD, registering with the GPIO/LED driver */
	{led_init,		BOARD_ALL | IOBOARD_ALL},

	/* Ethernet, includes MAC address setup. */
#ifndef CONFIG_TI_DAVINCI_MDIO
	{PHY_bitbang,		IOBOARD_KOALA | IOBOARD_NGUNI | IOBOARD_HAMSTER | IOBOARD_KUDU},
#endif
	{PHY_setup,		IOBOARD_ALL},
	{rgmii1_init, 		IOBOARD_KOALA | IOBOARD_NGUNI | IOBOARD_HAMSTER},
	{rgmii2_init, 		IOBOARD_NGUNI | IOBOARD_KUDU },
	{rgmii_init,		IOBOARD_KOALA | IOBOARD_NGUNI | IOBOARD_HAMSTER | IOBOARD_KUDU },
	{rmii1_init,		IOBOARD_ELAINE},
	{mii1_init,		IOBOARD_NONE}, /* Placeholder, in case we ever use MII */

	/* USB */
	{usb0_init,		IOBOARD_ALL},
	{usb1_init,		IOBOARD_ALL},

	/* MMC */
	{mmc1_init,		IOBOARD_NEWMAN}, /* Must be before mmc0_init */
	{mmc0_init,		IOBOARD_NEWMAN},
	{ioboard_nguni_mmc0_init, IOBOARD_NGUNI | IOBOARD_HAMSTER | IOBOARD_KUDU}, /* Relies on CD being GPIO 3.19 */

	/* ADC */
	{mfd_tscadc_init,	IOBOARD_NGUNI | IOBOARD_HAMSTER | IOBOARD_KUDU},
	
	/* Extra UARTs */
	{uart2_init, 		IOBOARD_NONE},
	{uart3_init, 		IOBOARD_NONE},

	/* SPI. Enable OMAP2SPI driver (McSPI) in kernel config if used. 2 CS only. */
	{spi0_init, 		IOBOARD_NONE},
	{spi1_init, 		IOBOARD_NONE},

	/* SPI pads as GPIO, for bitbashing SPI or CPLD passthrough */
	{spi0_gpio_init, 	IOBOARD_ELAINE},

	/* Hardware I2C, disabled because of hardware bugs */
	{i2c0_init, 		IOBOARD_NONE},
	{i2c1_init, 		IOBOARD_NONE},
	{i2c2_init, 		IOBOARD_NONE},

	/* GPIO I2C (hardware I2C is buggy, must use GPIO bitbashing. */
	{i2c0_gpio_init,	IOBOARD_ALL},

	/* Real time clock */
	{am335x_rtc_init, 	IOBOARD_NGUNI | IOBOARD_HAMSTER | IOBOARD_KUDU},

	{NULL, 0},
};


static void __init board_init(void)
{
	unsigned profile = 0;
#if defined KV_BOARD_betzy
	pr_info("Board: NetComm Betzy\n");
	profile |= BOARD_BETZY;
#elif defined KV_BOARD_nguni
	pr_info("Board: NetComm Nguni\n");
	profile |= BOARD_NGUNI;
#elif defined KV_BOARD_kewel
	pr_info("Board: NetComm Kewel\n");
	profile |= BOARD_KEWEL;
#else
	#error Unsupported processor board
#endif

#if defined KV_IOBOARD_koala
	pr_info("IOBoard: Koala\n");
	profile |= IOBOARD_KOALA;
#elif defined KV_IOBOARD_elaine
	pr_info("IOBoard: Elaine\n");
	profile |= IOBOARD_ELAINE;
#elif defined KV_IOBOARD_newman
	pr_info("IOBoard: Newman\n");
	profile |= IOBOARD_NEWMAN;
#elif defined KV_IOBOARD_nguni
	pr_info("IOBoard: Nguni\n");
	profile |= IOBOARD_NGUNI;
#elif defined KV_IOBOARD_hamster
	pr_info("IOBoard: Hamster\n");
	profile |= IOBOARD_HAMSTER;
#elif defined KV_IOBOARD_kudu
	pr_info("IOBoard: Kudu\n");
	profile |= IOBOARD_KUDU;
#else
	#error Unsupported IOBoard
#endif

	_configure_device(ntc_dev_cfg, profile);
}

/* USB */
static struct omap_musb_board_data koala_musb_board_data = {
	.interface_type	= MUSB_INTERFACE_ULPI,
	.mode           = (MUSB_HOST << 4) | MUSB_HOST,
	.power		= 500,
	.instances	= 1,

	.gpio_usb_ids	= {-EINVAL,-EINVAL},
	.busnums	= {0,0},
};


static struct omap_musb_board_data broken_otg_musb_board_data = {
	.interface_type	= MUSB_INTERFACE_ULPI,

	/* 
		As hardware issues with VBUS, USB ID and USB D+/D- on the EVT3 of NTC-140wx, OTG cannot supported

		On the EVT4 of NTC-140wx, VBUS and USB D+/D- will be fixed and to avoid processor errata,
		USB0_ID will be wired to GPIO0_2
	*/

	.mode           = (MUSB_HOST << 4) | MUSB_HOST,
	.power		= 500,
	.instances	= 1,
	.gpio_usb_ids	= {-EINVAL,-EINVAL},
	.busnums	= {0,2},
};

static struct omap_musb_board_data otg_musb_board_data = {
	.interface_type	= MUSB_INTERFACE_ULPI,
	.mode           = (MUSB_HOST << 4) | MUSB_OTG,
	.power		= 500,
	.instances	= 1,

	/*
		* description
		gpio usb id information (struct musb_hdrc_platform_data::gpio_usb_id) is dynamically generated per-bus based on 
		struct omap_musb_board_data::gpio_usb_ids. We do not move this part directly to 
		struct musb_hdrc_platform_data::gpio_usb_id as the structure is supposed to be generic for all boards.

		* information flow
		struct omap_musb_board_data::gpio_usb_ids[] ==> struct musb_hdrc_platform_data::gpio_usb_id ==> struct musb::gpio_usb_id
		(global but board-specific)                     (per-bus but generic for all boards)            (run-time information)

	*/
	.gpio_usb_ids	= {GPIO_TO_PIN(0,2),-EINVAL}, /* Hardware ID input is buggy, using GPIO */
	.busnums	= {0,2},
};

void __iomem *am33xx_emif_base;

void __iomem * __init am33xx_get_mem_ctlr(void)
{

	am33xx_emif_base = ioremap(AM33XX_EMIF0_BASE, SZ_32K);

	if (!am33xx_emif_base)
		pr_warning("%s: Unable to map DDR2 controller",	__func__);

	return am33xx_emif_base;
}

void __iomem *am33xx_get_ram_base(void)
{
	return am33xx_emif_base;
}

void __iomem *am33xx_gpio0_base;

void __iomem *am33xx_get_gpio0_base(void)
{
	am33xx_gpio0_base = ioremap(AM33XX_GPIO0_BASE, SZ_4K);

	return am33xx_gpio0_base;
}

static struct resource am33xx_cpuidle_resources[] = {
	{
		.start		= AM33XX_EMIF0_BASE,
		.end		= AM33XX_EMIF0_BASE + SZ_32K - 1,
		.flags		= IORESOURCE_MEM,
	},
};

/* AM33XX devices support DDR2 power down */
static struct am33xx_cpuidle_config am33xx_cpuidle_pdata = {
	.ddr2_pdown	= 1,
};

static struct platform_device am33xx_cpuidle_device = {
	.name			= "cpuidle-am33xx",
	.num_resources		= ARRAY_SIZE(am33xx_cpuidle_resources),
	.resource		= am33xx_cpuidle_resources,
	.dev = {
		.platform_data	= &am33xx_cpuidle_pdata,
	},
};
static struct w1_gpio_platform_data tiomap_gpio_platform_data = {
        .pin                    = GPIO_TO_PIN(2,25),
        .is_open_drain          = 0,
};

struct platform_device tiomap_w1_gpio_device = {
        .name   = "w1-gpio",
        .id   = -1,
        .dev    = {
                .platform_data = &tiomap_gpio_platform_data
        }
};
static void __init am33xx_cpuidle_init(void)
{
	int ret;

	am33xx_cpuidle_pdata.emif_base = am33xx_get_mem_ctlr();

	ret = platform_device_register(&am33xx_cpuidle_device);

	if (ret)
		pr_warning("AM33XX cpuidle registration failed\n");

	ret = platform_device_register(&tiomap_w1_gpio_device);

	if (ret)
		pr_warning("w1-gpio registration failed\n");

}

static void __init am335x_init(void)
{
	
	am33xx_cpuidle_init();       /* Idle/powersave setup */
	am33xx_mux_init(board_mux);  /* Early mux init */
	omap_serial_init();          /* Console */

	board_init();          /* General hardware / mux init with profiles */

	omap_sdrc_init(NULL, NULL);       /* SDRAM controller? */
	betzy_userspace_consumers_init(); /* Power control */

	/* USB init based on ioboard */

	/* avoid compiler warnings for unused structures. */
	(void)koala_musb_board_data;
	(void)broken_otg_musb_board_data;
	(void)otg_musb_board_data;

#if defined KV_IOBOARD_koala
	usb_musb_init(&koala_musb_board_data);
#elif defined KV_IOBOARD_elaine
	usb_musb_init(&koala_musb_board_data);
#elif defined KV_IOBOARD_newman
	usb_musb_init(&koala_board_data);
#elif defined KV_IOBOARD_nguni
	#ifdef CONFIG_nguni_hwmod_usbid
	usb_musb_init(&otg_musb_board_data);
	#else
	usb_musb_init(&broken_otg_musb_board_data);
	#endif
#elif defined KV_IOBOARD_hamster
	usb_musb_init(&otg_musb_board_data);
#elif defined KV_IOBOARD_kudu
	usb_musb_init(&otg_musb_board_data);
#else
	#error Unsupported IOBoard
#endif

	/* TODO: Is this necessary? */
	/* Create an alias for icss clock */
	if (clk_add_alias("pruss", NULL, "pruss_uart_gclk", NULL))
		pr_warn("failed to create an alias: icss_uart_gclk --> pruss\n");
	/* Create an alias for gfx/sgx clock */
	if (clk_add_alias("sgx_ck", NULL, "gfx_fclk", NULL))
		pr_warn("failed to create an alias: gfx_fclk --> sgx_ck\n");

}

static void __init am335x_map_io(void)
{
	omap2_set_globals_am33xx();
	omapam33xx_map_common_io();
}

MACHINE_START(AM335XEVM, "am335xevm")
	/* Maintainer: Netcomm, but we pretend to be a TI am335xevm. */
	.atag_offset	= 0x100,
	.map_io		= am335x_map_io,
	.init_early	= am33xx_init_early,
	.init_irq	= ti81xx_init_irq,
	.handle_irq     = omap3_intc_handle_irq,
	.timer		= &omap3_am33xx_timer,
	.init_machine	= am335x_init,
MACHINE_END
