/*
 *	@file 	espProcs.c
 *	@brief 	Embedded Server Pages (ESP) Procedures.
 *	@overview These ESP procedures can be used in ESP pages for common tasks.
 */
/********************************* Copyright **********************************/
/*
 *	@copy	default
 *	
 *	Copyright (c) Mbedthis Software LLC, 2003-2007. All Rights Reserved.
 *	
 *	This software is distributed under commercial and open source licenses.
 *	You may use the GPL open source license described below or you may acquire 
 *	a commercial license from Mbedthis Software. You agree to be fully bound 
 *	by the terms of either license. Consult the LICENSE.TXT distributed with 
 *	this software for full details.
 *	
 *	This software is open source; you can redistribute it and/or modify it 
 *	under the terms of the GNU General Public License as published by the 
 *	Free Software Foundation; either version 2 of the License, or (at your 
 *	option) any later version. See the GNU General Public License for more 
 *	details at: http://www.mbedthis.com/downloads/gplLicense.html
 *	
 *	This program is distributed WITHOUT ANY WARRANTY; without even the 
 *	implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
 *	
 *	This GPL license does NOT permit incorporating this software into 
 *	proprietary programs. If you are unable to comply with the GPL, you must
 *	acquire a commercial license to use this software. Commercial licenses 
 *	for this software and support services are available from Mbedthis 
 *	Software at http://www.mbedthis.com 
 *	
 *	@end
 */
/********************************** Includes **********************************/

#include  "esp.h"
#include  "rdb_ops.h"

#define STRNCPY(dst,src,len) do { strncpy(dst,src,len); ((char*)(dst))[len-1]='\0'; } while(0)

/************************************ Code ************************************/
#if BLD_FEATURE_ESP_MODULE
#if BLD_FEATURE_SESSION
/*
 *	destroySession
 */

static int destroySessionProc(EspRequest *ep, int argc, char **argv)
{
	ep->esp->destroySession(ep->requestHandle);
	return 0;
}

#endif	/* BLD_FEATURE_SESSION */

/******************************************************************************/
/*
 *	include
 *
 *	This includes javascript libraries. For example:
 *
 *		<% include("file", ...); %> 
 *
 *	Don't confuse with ESP includes:
 *
 *		<% include file.esp %>
 *
 *	Filenames are relative to the base document including the file.
 *	FUTURE -- move back to EJS. Only here now because we need ep->readFile.
 */ 

static int includeProc(EspRequest *ep, int argc, char **argv)
{
	Esp		*esp;
	char	path[MPR_MAX_FNAME], dir[MPR_MAX_FNAME];
	char	*emsg, *buf;
	int		size, i;

	esp = ep->esp;
	mprAssert(argv);
	for (i = 0; i < argc; i++) {
		mprGetDirName(dir, sizeof(dir), ep->docPath);
		mprSprintf(path, sizeof(path), "%s/%s", dir, argv[i]);

		if (esp->readFile(ep->requestHandle, &buf, &size, path) < 0) {
			espError(ep, "Can't read include file: %s", path);
			return MPR_ERR_CANT_ACCESS;
		}
		buf[size] = '\0';

		if (ejsEvalScript(espGetScriptHandle(ep), buf, 0, &emsg) < 0) {
			espError(ep, "Cant evaluate script");
			mprFree(buf);
			return -1;
		}
		mprFree(buf);
	}
	return 0;
}

/******************************************************************************/
/*
 *	redirect
 *
 *	This implemements <% redirect(url, code); %> command. The redirection 
 *	code is optional.
 */ 

static int redirectProc(EspRequest *ep, int argc, char **argv)
{
	char	*url;
	int		code;

	if (argc < 1) {
		espError(ep, "Bad args");
		return MPR_ERR_BAD_ARGS;
	}
	url = argv[0];
	if (argc == 2) {
		code = atoi(argv[1]);
	} else {
		code = 302;
	}
	espRedirect(ep, code, url);
	return 0;
}

/******************************************************************************/
#if BLD_FEATURE_SESSION
/*
 *	useSession
 */

static int useSessionProc(EspRequest *ep, int argc, char **argv)
{
	int			timeout;

	if (argc > 1) {
		espError(ep, "Bad args");
		return MPR_ERR_BAD_ARGS;

	} else if (argc == 1) {
		timeout = atoi(argv[0]);
	} else {
		timeout = 0;
	}

	ep->esp->createSession(ep->requestHandle, timeout);
	espSetReturnString(ep, ep->esp->getSessionId(ep->requestHandle));
	return 0;
}

#endif /* BLD_FEATURE_SESSION */
/******************************************************************************/
/*
 *	setHeader
 *
 *	This implemements <% setHeader("key: value", allowMultiple); %> command.
 */ 

static int setHeaderProc(EspRequest *ep, int argc, char **argv)
{
	mprAssert(argv);
	if (argc != 2) {
		espError(ep, "Bad args");
		return MPR_ERR_BAD_ARGS;
	}
	ep->esp->setHeader(ep->requestHandle, argv[0], atoi(argv[1]));
	return 0;
}

/******************************************************************************/
/*
 *	write
 *
 *	This implemements <% write("text"); %> command.
 */ 

static int writeProc(EspRequest *ep, int argc, char **argv)
{
	char	*s;
	int		i, len;

	mprAssert(argv);
	for (i = 0; i < argc; i++) {
		s = argv[i];
		len = strlen(s);
		if (len > 0) {
			if (espWrite(ep, s, len) != len) {
				espError(ep, "Can't write to client");
				return -1;
			}
		}
	}
	return 0;
}

/******************************************************************************/
/*
 *	rename
 *
 *	This implemements <% rename(oldFile, newFile); %> command.
 */ 

static int renameProc(EspRequest *ep, int argc, char **argv)
{
	if (argc != 2) {
		espError(ep, "Bad args");
		return -1;
	}
	if (rename(argv[0], argv[1]) < 0) {
		espError(ep, "Can't rename uploaded file");
	}
	return 0;
}
/******************************************************************************/
#define MY_MAX_LIST_SIZE	2047

static int rdb_exists(EspRequest *ep, int argc, char **argv)
{
	struct rdb_session *s = NULL;
	int retval;
	int len = 0;

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		if (argc > 1)
			espSetReturnString(ep, argv[1] );
		else
			espSetReturnString(ep, "" );
		return 0;
	}

	retval = rdb_getinfo(s, argv[0], &len, NULL, NULL);

	rdb_close(&s);

	if (retval == -EOVERFLOW) {
		espSetReturnString(ep, "1" );
	} else {
		espSetReturnString(ep, "" );
	}

	return 0;
}

static int get_single(EspRequest *ep, int argc, char **argv)
{
	struct rdb_session *s = NULL;
	int retval;
	char value[MY_MAX_LIST_SIZE+1];
	int len = sizeof(value);

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		if (argc > 1)
			espSetReturnString(ep, argv[1] );
		else
			espSetReturnString(ep, "" );
		return 0;
	}

	retval = rdb_get(s, argv[0], value, &len);

	rdb_close(&s);

	if (retval) {
		espSetReturnString(ep, "N/A" );
	} else {
		espSetReturnString(ep, value );
	}

	return 0;
}

static int set_single_direct(EspRequest *ep, int argc, char **argv)
{
	char* perm;
	int flags;
	
	struct rdb_session *s = NULL;
	int retval;

	// check argument validation
	if (argc != 3) {
		syslog(LOG_ERR,"incorrect number of arguments used for set_single_direct() - %d",argc);
		return -1;
	}

	perm=argv[0];
	
	// build flags
	flags=0;
	if(strstr(perm,"-p"))
		flags|=PERSIST;

	retval = rdb_open(NULL, &s);
	if (retval) {
		syslog(LOG_ERR,"failed to open rdb - %s",strerror(errno));
		return -1;
	}

	retval = rdb_update_string(s, argv[1], argv[2], flags, DEFAULT_PERM);

	rdb_close(&s);

	if (retval) {
		syslog(LOG_ERR,"rdb_update_string failed to update name(%s):value(%s) - %s",argv[1], argv[2],strerror(errno));
	}

	return 0;
}

static int get_single_direct(EspRequest *ep, int argc, char **argv)
{
	struct	rdb_session *s = NULL;
	int	retval;
	char	value[MY_MAX_LIST_SIZE+1];
	int	len = sizeof(value);

	if (argc != 1) {
		syslog(LOG_ERR,"incorrect number of arguments used for get_single_direct() - %d",argc);
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		syslog(LOG_ERR,"failed to open rdb - %s",strerror(errno));
		return -1;
	}

	retval = rdb_get(s, argv[0], value, &len);

	rdb_close(&s);

	if (retval == 0) {
		espWriteString(ep, value);
	} else {
		syslog(LOG_ERR,"rdb_get failed - name(%s),retval(%d)",argv[0],retval);
	}

	return 0;
}

static int set_single(EspRequest *ep, int argc, char **argv)
{
	struct	rdb_session *s = NULL;
	char	name[MAX_NAME_LENGTH+1]; 
	char	value[MY_MAX_LIST_SIZE+1];
	char	buffer[MAX_NAME_LENGTH+MY_MAX_LIST_SIZE+1];
	char	*p_pos;
	int	retval;
	int	flags = 0;

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}
	retval = rdb_open(NULL, &s);
	if (retval) {
		syslog(LOG_ERR,"failed to open rdb - %s",strerror(errno));
		return -1;
	}
	STRNCPY( buffer, argv[0], sizeof(buffer));
	p_pos=strchr( buffer, '=' );
	if( p_pos )
	{
		*p_pos = 0;
		STRNCPY( name, buffer, sizeof(name));
		STRNCPY( value, p_pos+1, sizeof(value));
	}
	else
	{
		STRNCPY( name, buffer, sizeof(name));
	}

	if( argc > 1 )
	{
		if(strncmp( argv[1], "-p", 2 )==0)
		{
			flags |= PERSIST;
		}
	}
	retval = rdb_update_string(s, name, value, flags, DEFAULT_PERM);

	rdb_close(&s);
	if( retval )
		printf("rdb_update_string returns %i ( %s )\n",-errno, strerror(errno));

	return 0;
}

static int get_list(EspRequest *ep, int argc, char **argv)
{
	struct	rdb_session *s = NULL;
	char	value[MY_MAX_LIST_SIZE+1];
	int	len = sizeof(value);
	int	retval;

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		if (argc > 1)
			espSetReturnString(ep, argv[1] );
		else
			espSetReturnString(ep, "" );
		return 0;
	}

	*value=0;
	if(!rdb_getnames(s, argv[0], value, &len, 0) )
	{
		espSetReturnString(ep, value);
	}
	else
	{
		espSetReturnString(ep, "N/A" );
		fprintf(stderr, "rdb_getnames returns %i ( %s ) %s -- %s\n",-errno, strerror(errno),argv[0],value);
	}

	rdb_close(&s);
	return 0;
}

static int unhex( char c )
{
	return( c >= '0' && c <= '9' ? c - '0'
	    : c >= 'A' && c <= 'F' ? c - 'A' + 10
	    : c - 'a' + 10 );
}

static void unescape ( char * s)
{
	char	*p;

	for ( p = s; *s != '\0'; ++s ) {
		if ( *s == '%' ) {
			if ( *++s != '\0' ) {
				*p = unhex( *s ) << 4;
			}
			if ( *++s != '\0' ) {
				*p++ += unhex( *s );
			}
		} else {
			*p++ = *s;
		}
	}

	*p = '\0';
}

static int set_escapedlist(EspRequest *ep, int argc, char **argv)
{
	int	fd;
	int	retval;
	char	name[MAX_NAME_LENGTH+1];
	char	value[MY_MAX_LIST_SIZE+1];
	char	*str1, *token, *p_pos;


	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}
	ioctl_args ps={name, value, 0, 0, 0};
	if( (fd = open("/dev/cdcs_DD", O_RDWR ) ) ==  - 1)
	{
		printf( "can't open cdcs_DD %i ( %s )\n", -errno, strerror(errno));
		return -1;
	}

	for (str1 = argv[0]; ; str1 = NULL)
	{
		token = strtok(str1, "&");
		if (token == NULL)
			break;

		p_pos=strchr( token, '=' );
		if( p_pos )
		{
			*p_pos = 0;
			STRNCPY(name, token, sizeof(name));
			STRNCPY(value, p_pos+1, sizeof(value));
		}
		else
		{
			STRNCPY(name, token, sizeof(name));
			// do not delete a rdb variable. Otherwise, database driver causes a performance problem by flooding
			// signals to all processes that subscribed the variable. Most of time, rdb_manager gets hurt
		}

		unescape(value);

		// workaround for database driver - database driver is having data corruption if zero-length variable is created
		// so we put terminating zero together with the string to avoid this bug
		ps.len = strlen(ps.value)+1;
		ps.perm = DEFAULT_PERM;
		ps.flags = 0;
		if( argc > 1 )
		{
			if(strncmp( argv[1], "-p", 2 )==0)
			{
				ps.flags |= PERSIST;
			}
		}
		retval = ioctl(fd, SETSINGLE, (unsigned long)&ps );
		if( retval )
		{
			ps.flags |= CREATE; // set CREATE flag and try again
			errno = 0;
			retval = ioctl(fd, SETSINGLE, (unsigned long)&ps );
		}
		if( retval )
			printf("ioctl SETSINGLE in set_escapedlist returns %i ( %s )\n",-errno, strerror(errno));
	}

	close(fd);
	return 0;
}

int decodeData(char* inString, char* outStr)
{
	/* When decoding the Data we will do the following:
	   1.) Replace '+' with ' '
	   2.) Replace %xx to equivalent character
	*/
	int i = 0;
	int count = 0;
	char lbuf[3];

	while(inString[i])
	{
		switch (inString[i])
		{
				//if the character is a plus sign (+) then append a space
			case '+':
				outStr[count] = ' ';
				break;
			case '%':
				i++;
				lbuf[0] = inString[i];
				if (isupper(lbuf[0]))
					lbuf[0] = tolower(lbuf[0]);
				if ((lbuf[0] < '0' && lbuf[0] > '9') && ((lbuf[0] < 'a' && lbuf[0] > 'f')))
					return -1;
				i++;
				lbuf[1] = inString[i];
				if (isupper(lbuf[1]))
					lbuf[1] = tolower(lbuf[1]);
				if ((lbuf[1] < '0' && lbuf[1] > '9') && ((lbuf[1] < 'a' && lbuf[1] > 'f')))
					return -1;
#define atoh(c) ((c) - (((c) >= '0' && (c) <= '9') ? '0' : ('a'-10)))
				outStr[count] = (unsigned char)((atoh(lbuf[0]) << 4) + atoh(lbuf[1]));
				break;
			default:
				outStr[count] = inString[i];
				break;
		}
		i++;
		count++;
	}
	outStr[count] = 0;
	return count;
}

static int validate_name(char *str)
{
	char mychar;
	while ((mychar = *str++) != '\0') {
		if (!((mychar >= '0' && mychar <= '9') || 
			(mychar >= 'a' && mychar <= 'z') || 
			(mychar >= 'A' && mychar <= 'Z') || 
			(mychar == '-') || (mychar == '_') || 
			(mychar == '.') || (mychar == '(') || 
			(mychar == ')'))) {
			return -EINVAL;	/* names are limited to a specific character set [0-9 a-z A-Z -_.()] */
		}
	}
	return 0;
}

static int set_list(EspRequest *ep, int argc, char **argv)
{
	struct	rdb_session *s = NULL;
	int	retval;
	char	buffer[MY_MAX_LIST_SIZE+1];
	char	name[MAX_NAME_LENGTH+1];
	char	value[MY_MAX_LIST_SIZE+1];
	char	*str, *p_pos;
	int	flag;
	int	flags = 0;

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		printf( "rdb_open failed. %i ( %s )\n", -errno, strerror(errno));
		return -1;
	}

	retval = rdb_lock(s, 0);
	if (retval) {
		printf("rdb_lock failed. %i (%s)\n", -errno, strerror(errno));
	}

	if( argc > 1 )
	{
		if(strncmp( argv[1], "-p", 2 )==0)
		{
			flags |= PERSIST;
		}
	}
	for (str = argv[0]; *str != '\0';)
	{
		flag = 0;
		for (p_pos = str; *p_pos != '\0' && *p_pos != '=' && *p_pos != '&'; p_pos++) {
		}
		if (*p_pos == '=') { //value is fllowed
			flag = 1;
		} else if (*p_pos == '&') { // another varialbe followed
			flag = 2;
		}
		*p_pos = '\0';
		STRNCPY(name, str, sizeof(name));
		retval = validate_name(name);
		if (retval) {
			rdb_unlock(s);
			printf("SETLIST name(%s) not valid %i\n",name, retval);
			rdb_close(&s);
			return retval;
		}
		value[0] = '\0';
		if (flag == 1) {
			/*  when '=' found, do not delete variable */
			p_pos++;
			for (str = p_pos; *p_pos != '\0' && *p_pos != '&'; p_pos++) {
			}
			if (*p_pos == '&') { // more value variable is defined.
				flag = 2;
				*p_pos = '\0';
			} else {
				flag = 0;
			}
			STRNCPY(buffer, str, sizeof(buffer));
			decodeData(buffer, value); /* , MY_MAX_LIST_SIZE); */
		}
		str = p_pos;
		if (flag)
			str++;
		retval = rdb_update_string(s, name, value, flags, DEFAULT_PERM);
		if( retval ) {
			printf("rdb_update_string in set_list returns %i ( %s )\n",-errno, strerror(errno));
		}
	}
	rdb_unlock(s);
	rdb_close(&s);
	return 0;
}

static int rdb_lock_esp(EspRequest *ep, int argc, char **argv)
{
	struct	rdb_session *s = NULL;
	int	retval;

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		printf( "rdb_open failed. %i ( %s )\n", -errno, strerror(errno));
		return -1;
	}

	if(!rdb_lock(s, 0) )
	{
		espSetReturnString(ep, "OK" );
	}
	else
	{
		espSetReturnString(ep, "ERROR" );
	}

	rdb_close(&s);
	return 0;
}

static int rdb_unlock_esp(EspRequest *ep, int argc, char **argv)
{
	struct	rdb_session *s = NULL;
	int	retval;

	if (argc < 1) {
		espError(ep, "Bad args");
		return -1;
	}

	retval = rdb_open(NULL, &s);
	if (retval) {
		printf( "rdb_open failed. %i ( %s )\n", -errno, strerror(errno));
		return -1;
	}

	if(!rdb_unlock(s) )
	{
		espSetReturnString(ep, "OK" );
	}
	else
	{
		espSetReturnString(ep, "ERROR" );
	}

	rdb_close(&s);
	return 0;
}


static int esp_wait_for_chg(EspRequest *ep, int argc, char **argv)
{
	struct rdb_session *s;
	char* name;
	char value[MY_MAX_LIST_SIZE+1];
	const char* value2;
	int len = sizeof(value);
	int retval;

	int sleep_cnt;
	int i;


	// check invalidation of parameter
	if (argc < 3) {
		espError(ep, "Bad args");
		return -1;
	}

	// open rdb
	retval = rdb_open(NULL, &s);
	if (retval) {
		if (argc > 1)
			espSetReturnString(ep, argv[1] );
		else
			espSetReturnString(ep, "" );
		return 0;
	}

	// store parameters
	name=argv[0];
	value2=argv[1];
	sleep_cnt=atoi(argv[2]);

	// setup default return value
	strcpy(value,"N/A" );

	i=0;
	while(i++<sleep_cnt) {
		if(rdb_get(s, name, value, &len)!=0) {
			strcpy(value,"N/A" );
		}
		// bypass if value is not matching to the given value
		if(strcmp(value,value2))
			break;

		sleep(1);
	}
	rdb_close(&s);

	espSetReturnString(ep, value);

    return 0;
}

static int esp_sleep(EspRequest *ep, int argc, char **argv)
{
	unsigned long usec;
	char buf[16];
	if (argc != 1) {
		espError(ep, "Bad args");
		return -1;
	}
    	STRNCPY( buf, argv[0], sizeof(buf));
	espError(ep, buf );
	usec = atol(buf)*1000;
	usleep( usec );	
	espError(ep, "sleep OK");
	return 0;
}

static int get_pid(EspRequest *ep, int argc, char **argv)
{
  FILE* pFile;
  char respBuff[10];
  char msg[32];

  if (argc != 1) {
	espError(ep, "Bad args");
	return -1;
  }
  strcpy(msg, "pidof ");
  strncat(msg, argv[0],25);

  if( (pFile = popen(msg, "r") ) == 0)
  {
    espError(ep, "getPID failed to open pipe\n");
	espSetReturnString(ep, "0" );
    return 0;
  }
  fgets(respBuff, 10, pFile);
  pclose(pFile);
  if( atoi(respBuff)>0 )
  {
	if( *(respBuff+strlen(respBuff)-1)=='\n' )
		*(respBuff+strlen(respBuff)-1)=0;
	espSetReturnString(ep, respBuff );
  }
  else
	espSetReturnString(ep, "0" );
  return 0;
}

static int esp_syslog(EspRequest *ep, int argc, char **argv)
{
	int loglevel;
	const char* msg;

	// check validation of argument
	if(argc!=2) {
		return MPR_ERR_BAD_ARGS;
	}

	// get parameters
	loglevel=atoi(argv[0]);
	msg=argv[1];

	syslog(loglevel,msg);

	return 0;
}

static int exec_cmd(EspRequest *ep, int argc, char **argv)
{
	FILE* fp;
	const char* cmd;
	char line[1024];

	// check validation of argument
	if(argc!=1) {
		return MPR_ERR_BAD_ARGS;
	}

	// get command
	cmd=argv[0];

	// open the command
	fp=popen(cmd,"r");
	if(!fp) {
		return MPR_ERR_NOT_FOUND;
	}

	// print to the web page
	while( fgets(line,sizeof(line),fp) ) {
		espWriteString(ep,line);
	}

	fclose(fp);

	return 0;
}

static int get_rand(EspRequest *ep, int argc, char **argv)
{
char value[64];
	sprintf( value, "%u", rand());
	espSetReturnString(ep, value );
	return 0;
}
/******************************************************************************/

void espRegisterProcs()
{
	espDefineStringCFunction(0, "rename", renameProc, 0);
	espDefineStringCFunction(0, "write", writeProc, 0);
	espDefineStringCFunction(0, "setHeader", setHeaderProc, 0);
	espDefineStringCFunction(0, "redirect", redirectProc, 0);
	espDefineStringCFunction(0, "include", includeProc, 0);

	espDefineStringCFunction(0, "rdb_exists",rdb_exists, 0);
	espDefineStringCFunction(0, "get_single", get_single, 0);
	espDefineStringCFunction(0, "get_single_direct", get_single_direct, 0);
	espDefineStringCFunction(0, "set_single", set_single, 0);
	espDefineStringCFunction(0, "set_single_direct", set_single_direct, 0);
	espDefineStringCFunction(0, "get_list", get_list, 0);
	espDefineStringCFunction(0, "set_list", set_list, 0);
	espDefineStringCFunction(0, "set_escapedlist", set_escapedlist, 0);
	espDefineStringCFunction(0, "rdb_lock", rdb_lock_esp, 0);
	espDefineStringCFunction(0, "rdb_unlock", rdb_unlock_esp, 0);
	espDefineStringCFunction(0, "esp_sleep", esp_sleep, 0);
	espDefineStringCFunction(0, "get_pid", get_pid, 0);
	espDefineStringCFunction(0, "get_rand", get_rand, 0);

	espDefineStringCFunction(0, "esp_wait_for_chg", esp_wait_for_chg, 0);

	espDefineStringCFunction(0, "exec_cmd", exec_cmd, 0);

	// use this esp function instead of trace - builtin trace function is limited by appweb loglevel
	espDefineStringCFunction(0, "syslog", esp_syslog, 0);

#if BLD_FEATURE_SESSION
	/*
	 *	Create and use are synonomous
	 */
	espDefineStringCFunction(0, "useSession", useSessionProc, 0);
	espDefineStringCFunction(0, "createSession", useSessionProc, 0);
	espDefineStringCFunction(0, "destroySession", destroySessionProc, 0);
#endif
}

/******************************************************************************/

#else
void mprEspControlsDummy() {}

#endif /* BLD_FEATURE_ESP_MODULE */

/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 * vim: sw=4 ts=4 
 */

