/*
 * Driver for UART used in freescale iMX* SOCs
 *
 * The UARTs can be multiplexed to different pins. This is expected
 * to be set up early in the board support, possibly in SPL.
 *
 * Also make sure that the clock to the UARTs is enabled
 * (hw_clkctrl_xtal_clr = 0x80000000).
 *
 * Required defines:
 *	CONFIG_FREESCALE_APP_UART
 *	CONFIG_FREESCALE_UART_BASE  = Base address of UART to use
 *	CONFIG_FREESCALE_UART_CLOCK = clock frequency
 *	CONFIG_BAUDRATE             = default baud rate
 *
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com.au>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */


#include <common.h>
#include <watchdog.h>
#include <asm/io.h>

DECLARE_GLOBAL_DATA_PTR;

struct reg4 {
	volatile u32 r;   /* Set / read register value */
	volatile u32 set; /* Set individual register bits */
	volatile u32 clr; /* Clear individual register bits */
	volatile u32 tog; /* Toggle individual register bits */
};

struct reg1 {
	volatile u32 r;   /* Set / read register value */
	u32 res[3];       /* Unused */
};

struct mx28_appserial {
	struct reg4  ctrl0;
	struct reg4  ctrl1;
	struct reg4  ctrl2;
	struct reg4  linectrl;
	struct reg4  linectrl2;
	struct reg4  intr;
	struct reg1  data;
	struct reg1  stat;
	struct reg1  debug;
	struct reg1  version;
	struct reg1  autobaud;
};

static void setbaud(void *base, int baudrate)
{
	struct mx28_appserial *uartapp = base;
	u32 frac_bdiv, divint, divfrac;

#define WLEN8 (0x3 << 5) /* 8-bit word */
#define FEN   (0x1 << 4) /* FIFO enable */

	/* Fractional divider, fixed point 2^6 */
	frac_bdiv = ((u32)CONFIG_FREESCALE_UART_CLOCK * 32) / (u32)baudrate;
	divint  = ((frac_bdiv >> 6) & 0xFFFF) << 16;
	divfrac = (frac_bdiv & 0x3F) << 8;

	/* Wait for TxFIFO empty, then switch. */
	while (!(uartapp->stat.r & (1<<27)));

	/* Program the HW_UARTAPP_LINECTRL and LINECTRL2 for baud rate and word length */
	uartapp->linectrl.r  = divint | divfrac | WLEN8 | FEN;
	uartapp->linectrl2.r = divint | divfrac | WLEN8 | FEN;
}

int init_app_uart(void *base, int baudrate)
{
	struct mx28_appserial *uartapp = base;

	/* Clear the SFTRST and CLKGATE bits */
	uartapp->ctrl0.clr = 0xC0000000;

	/* Set the RX FIFO timeout and the XFER count */
	uartapp->ctrl0.r = 0x000F0001;

	/* Program the transmit XFER count */
	uartapp->ctrl1.r = 0x00000001;

	/* Program the line control 2 register to enable the RX, TX, and UART module */
	uartapp->ctrl2.r = 0x00000301;

	setbaud(base, baudrate);

	return 0;
}

void send_app_uart(void *base, unsigned char ch)
{
	struct mx28_appserial *uartapp = base;

	/* Wait for TX FIFO not full */
	while (uartapp->stat.r & (1<<25));

	/* Transmit one byte of data */
	uartapp->data.r = ch;
}

void flush_app_uart(void *base)
{
	struct mx28_appserial *uartapp = base;
	/* Wait for TX FIFO empty */
	while (!(uartapp->stat.r & (1<<27)));
}

/* Returns true if characters are available */
static int test_app_uart(void *base)
{
	struct mx28_appserial *uartapp = base;
	return !(uartapp->stat.r & (1<<24));
}

/* Returns < 0 if no char */
static int recv_app_uart(void *base)
{
	struct mx28_appserial *uartapp = base;
	int rv = -1;

	/* Any characters? */
	if (test_app_uart(base)) {
		/* Read character */
		rv = uartapp->data.r;
	}

	return rv;
}

/* U-Boot API */

int serial_init (void)
{
	int rval;
	rval = init_app_uart((void*)CONFIG_FREESCALE_UART_BASE, CONFIG_BAUDRATE);
	serial_puts("\r\nFreescale UART\r\n");
	return rval;
}

void serial_putc (const char c)
{
	if (c == '\n')
		serial_putc('\r');
	send_app_uart((void*)CONFIG_FREESCALE_UART_BASE, c);
}

void serial_puts (const char *s)
{
	while (*s) {
		serial_putc (*s++);
	}
}

int serial_getc (void)
{
	int rval=-1;
	while (rval < 0) {
		rval = recv_app_uart((void*)CONFIG_FREESCALE_UART_BASE);
		//WATCHDOG_RESET();
	}
	return rval;
}

int serial_tstc (void)
{
	int rval;
	//WATCHDOG_RESET();
	rval = test_app_uart((void*)CONFIG_FREESCALE_UART_BASE);
	return rval;
}

void serial_setbrg (void)
{
	setbaud((void*)CONFIG_FREESCALE_UART_BASE, gd->baudrate);
}
