/*
 * Netcomm Eagle board (Freescale i.MX28 processor)
 *
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * Based on m28evk.c:
 * Copyright (C) 2011 Marek Vasut <marek.vasut@gmail.com>
 * on behalf of DENX Software Engineering GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <common.h>
#include <asm/gpio.h>
#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/iomux-mx28.h>
#include <asm/arch/clock.h>
#include <asm/arch/sys_proto.h>
#include <linux/mii.h>
#include <miiphy.h>
#include <netdev.h>
#include <errno.h>
#include <linux/mtd/nand.h>

#include "gpio.h"
#include "variant.h"

DECLARE_GLOBAL_DATA_PTR;

/* Dynamic partitions */

#define G *0x40000000ULL
#define M *0x100000ULL
#define K *0x400ULL
#define FILL ( 2ULL G ) /* The OPT partition grows up to this. Assumes 2G max, increase if necessary. */

/*

Environment is kept in EEPROM. The NAND layout of boot is as follows (skipping bad blocks):

	0x00000000, (512K)	= FCB Area (marked bad), Set up with primary boot at 256 pages,

	0x00080000, (512K)	= Primary boot stream. No secondary, this skips bad blocks already.

	0x00180000, (1M)	= Spare, e.g. env, production data, etc.

Recovery / main kernels and rootfs, followed by /usr/local and /opt partitions

Partition sizes marked with the FIXED flag ignore bad blocks on their extent
*/

loff_t dynpart_size[] = { DYNPART_FIXED | 2 M,     5 M,  12 M,      5 M,   32 M,  48 M,  FILL, 0 };
char *dynpart_names[] = {              "boot", "rkern", "rfs", "kernel", "root", "usr", "opt", NULL };

/* Some pin defines */

#ifdef V_ETHPHYRST_2_0
#define ETH_RESETn PIN(2,0) /* ETH PHY reset pin */
#endif

#ifdef V_ETHPHYRST_2_4
#define ETH_RESETn PIN(2,4) /* ETH PHY reset pin */
#endif



/* Pin mux options for UARTs Signal(pins)

The numbers in the table represent Muxmodes

pin   |DRx DTx DRT DCT|0Rx 0Tx 0RT 0CT|1Rx 1Tx 1RT 1CT|2Rx 2Tx 2RT 2CT|3Rx 3Tx 3RT 3CT|4Rx 4Tx 4RT 4CT|
2/27    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1
2/26    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1   -
2/25    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -   -
2/24    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -
2/19    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -   -   -   -   -
2/18    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -   -   -   -   -   -
2/17    -   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -   -   -   -   -   -   -   -   -
2/17    -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -
3/15    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -
3/14    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -
3/13    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -
3/12    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -
3/11    -   -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -
3/10    -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -
3/09    -   -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -
3/08    -   -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -
3/07    -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -   -
3/06    -   -   -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -
3/05    -   -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -   -   -
3/04    -   -   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -
3/03    -   2   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -
3/02    2   -   -   -   -   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -   1   -   -   -
3/01    -   -   2   -   -   0   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -
3/00    -   -   -   2   0   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -
3/25    -   2   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -
3/24    2   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -   -
3/23    .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   2   .   .
3/22    .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   2   .   .   .
3/21    .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   2   .
3/20    .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   2
3/17    .   2   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .
3/16    2   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .   .

*/
struct gpio_preset {
	unsigned pin;
	struct pininfo v;
};

#define NOPIN ((unsigned)(-1))
#define pin(b,i) .pin = PIN(b,i)

static const struct gpio_preset gpio_presets[] = {

	/* Power supply 3V3_aux enable */
	{ pin(2,10), .v = { M_OUT, V33, DO(1) }},

	/* AUART0 = UART0 = Zigbee is on 3/0 - 3/3, mux 0 */
	{ pin(3,0), .v = { MM(0), V33, A12, PU }}, /* Rx */
	{ pin(3,1), .v = { MM(0), V33, A12 }},     /* Tx */
	{ pin(3,2), .v = { MM(0), V33, A12 }},     /* CTS */
	{ pin(3,3), .v = { MM(0), V33, A12 }},     /* RTS */

	/* AUART1 = UART1 = Meter is on 3/4 - 3/7, mux 0 */
	{ pin(3,4), .v = { MM(0), V33, A12, PU }}, /* Rx */
	{ pin(3,5), .v = { MM(0), V33, A12 }},     /* Tx */
	{ pin(3,6), .v = { MM(0), V33, A12 }},     /* CTS */
	{ pin(3,7), .v = { MM(0), V33, A12 }},     /* RTS */

	/* AUART2 = NC */

	/* AUART3 = Console = Debugport is on 2/18, 2/19, mux 1 */
	{ pin(2,18), .v = { MM(1), V33, A12, PU }}, /* Rx */
	{ pin(2,19), .v = { MM(1), V33, A12, PU }}, /* Tx */

	/* AUART4 = NC */

	/* I2C1 on 3/16, 3/17 */
	{ pin(3,16), .v = { MM(1), V33, A12, PU }}, /* SCL */
	{ pin(3,17), .v = { MM(1), V33, A12, PU }}, /* SDA */

	/* Ethernet PHY clock */
	{ pin(4,16), .v = { MM(0), V33, A08 }},       /* ENET_CLK */
	{ .pin = ETH_RESETn, .v = { M_OUT, V33, A12, DO(1) }}, /* ENET_Resetn, release. */

	/* Switch a LED on to indicate aliveness */
	{ pin(1,9), .v = { M_OUT, V33, A12, DO(0) }},

	{ .pin = NOPIN } /* Must be at end */
};

static void periph_board_init(void)
{
	struct gpio_preset *p;

	struct mx28_clkctrl_regs *clkctrl = (void*)MXS_CLKCTRL_BASE;
	/* make sure the UART clock is not gated in the clock control module, clear bit 31 */
	clkctrl->hw_clkctrl_xtal_clr = 0x80000000;

	/* Set up GPIO multiplexers and values */
	p = (struct gpio_preset *)gpio_presets;
	while (p->pin != NOPIN) {
		set_pininfo(p->pin, &p->v);
		p++;
	}
}

#include <asm/arch/regs-power.h>

//static struct mx28_power_regs *pw = (struct mx28_power_regs *)MXS_POWER_BASE;

#define PREG(name) printf( #name " = \t%08x\r\n", pw->name )

/*
 * Functions
 */
int board_early_init_f(void)
{
	periph_board_init();
	//serial_init(); /* For early debug, otherwise it's called after this function.
	
	init_app_uart((void*)DUART_BASE, 115200);
	printf("%s:%d:\n", __func__, __LINE__);

#if 0
	/* Dump power registers */
	PREG(hw_power_ctrl);
	PREG(hw_power_5vctrl);
	PREG(hw_power_minpwr);
	PREG(hw_power_charge);
	PREG(hw_power_vdddctrl);
	PREG(hw_power_vddactrl);
	PREG(hw_power_vddioctrl);
	PREG(hw_power_vddmemctrl);
	PREG(hw_power_dcdc4p2);
	PREG(hw_power_misc);
	PREG(hw_power_dclimits);
	PREG(hw_power_loopctrl);
	PREG(hw_power_sts);
	PREG(hw_power_speed);
	PREG(hw_power_battmonitor);
	PREG(hw_power_reset);
	PREG(hw_power_debug);
	PREG(hw_power_thermal);
	PREG(hw_power_usb1ctrl);
	PREG(hw_power_special);
	PREG(hw_power_version);
	PREG(hw_power_anaclkctrl);
	PREG(hw_power_refctrl);
#endif

	/* IO0 clock at 480MHz */
	mx28_set_ioclk(MXC_IOCLK0, 480000);

	/* IO1 clock at 480MHz */
	mx28_set_ioclk(MXC_IOCLK1, 480000);

	/* SSP0 clock at 96MHz */
	mx28_set_sspclk(MXC_SSPCLK0, 96000, 0);

	/* SSP2 clock at 96MHz */
	mx28_set_sspclk(MXC_SSPCLK2, 96000, 0);

#ifdef	CONFIG_CMD_USB
	mxs_iomux_setup_pad(MX28_PAD_SSP2_SS1__USB1_OVERCURRENT);
	mxs_iomux_setup_pad(MX28_PAD_AUART2_RX__GPIO_3_8 |
			MXS_PAD_4MA | MXS_PAD_3V3 | MXS_PAD_NOPULL);
	gpio_direction_output(MX28_PAD_AUART2_RX__GPIO_3_8, 1);
#endif

	return 0;
}

int dram_init(void)
{
	int rval=0;
	/* Memory size comes from SPL */
	rval = mx28_dram_init();
	return rval;
}

/* From ocotp_env.c: */

/* Read the fuse register shadow. word is 0 for cust0, 1 for cust1, ..., 39 for srk7 */
uint32_t ocotp_read_shadow(unsigned word);
/* Write a whole fuse word, burn fuses where bits are 1 in mask. */
/* System must be rebooted after setting fuses, otherwise clocks or voltage levels may be wrong */
int ocotp_write_word(unsigned word, uint32_t mask);


int board_init(void)
{
	/* Adress of boot parameters. */
	gd->bd->bi_boot_params = CONFIG_SYS_SDRAM_BASE + 0x100;

#if 0
	/* Dump power registers */
	PREG(hw_power_ctrl);
	PREG(hw_power_5vctrl);
	PREG(hw_power_minpwr);
	PREG(hw_power_charge);
	PREG(hw_power_vdddctrl);
	PREG(hw_power_vddactrl);
	PREG(hw_power_vddioctrl);
	PREG(hw_power_vddmemctrl);
	PREG(hw_power_dcdc4p2);
	PREG(hw_power_misc);
	PREG(hw_power_dclimits);
	PREG(hw_power_loopctrl);
	PREG(hw_power_sts);
	PREG(hw_power_speed);
	PREG(hw_power_battmonitor);
	PREG(hw_power_reset);
	PREG(hw_power_debug);
	PREG(hw_power_thermal);
	PREG(hw_power_usb1ctrl);
	PREG(hw_power_special);
	PREG(hw_power_version);
	PREG(hw_power_anaclkctrl);
	PREG(hw_power_refctrl);
#endif

	/* WARNING: Most Falcon based boards have a hardware error, where the boot
	 * mode 4 pin (voltage select) is shared with an input pin on the serial
	 * port level shifter. Since the level shifter is not powered at boot time,
	 * it drags the BM4 pin down and the rom-code reads it as low/0. This
	 * selects 3V3 as the interface voltage to the NAND which causes NAND read
	 * errors.
	 *
	 * The upshot of this is that some boards don't boot at all and the
	 * remainder might be sensitive to to temperature, supply voltage,
	 * moon phase, etc.
	 *
	 * The fix for this requires two ingredients:
	 *
	 * 1) We need to configure the OCOTP fuses for primary boot from NAND,
	 *    and we need to set the ENABLE_PIN_BOOT_CHECK fuse to respect the
	 *    external BMS pin. This is done below.
	 *
	 * 2) We need to pull the external BMS pin low, to indicate that the
	 *    rom-code shall use the OCOTP boot configuration. This requires
	 *    moving R92 into the R98 position, to pull BMS low.
	 */
#ifdef CONFIG_BOOTMODE_FUSES
	/* Burn fuses if necessary */
	{
		struct ocotp_pat {
			unsigned word;
			uint32_t mask;
		};
		const struct ocotp_pat pattern[] = {
			{ .word = 24, .mask = 0x14000000 }, /* Boot mode NAND 1V8 */
			{ .word = 31, .mask = 0x00000001 }, /* Set ENABLE_PIN_BOOT_CHECK */
			{ .word = 0,  .mask = 0 } /* Sentinel */
		};
		const struct ocotp_pat *pat = pattern;
		uint32_t v;
		int changes = 0;
		printf("Fuse check\n");
		while (pat->word | pat->mask) {
			v = ocotp_read_shadow(pat->word);
			if ((v & pat->mask) != pat->mask) {
				printf("Burning OCOTP %d %08x |=> %08x\n",
					pat->word, pat->mask, v);
				ocotp_write_word(pat->word, pat->mask);
				changes++;
			}
			pat++;
		}
		if (changes) {
			printf("OCOTP write forces reset\n");
			do_reset(NULL, 0, 0, NULL);
		}
	}
#endif

	return 0;
}

/* Runtime environment variables */
void board_extra_env(void)
{
	/* Add variant variables to environment */
	setenv("V_RECBUTTON",V_RECBUTTON);
	setenv("V_POLBUTTON",V_POLBUTTON);
}

#ifdef	CONFIG_CMD_MMC
static int mx28evk_mmc_wp(int id)
{
	int rval;
	if (id != 0) {
		printf("MXS MMC: Invalid card selected (card id = %d)\n", id);
		return 1;
	}

	rval = gpio_get_value(MX28_PAD_SSP1_SCK__GPIO_2_12);
	return rval;
}

int board_mmc_init(bd_t *bis)
{
	int rval;
	/* Configure WP as input */
	gpio_direction_input(MX28_PAD_SSP1_SCK__GPIO_2_12);
	/* Configure MMC0 Power Enable */
	gpio_direction_output(MX28_PAD_PWM3__GPIO_3_28, 0);
	rval = mxsmmc_initialize(bis, 0, mx28evk_mmc_wp);
	return rval;
}
#endif

#ifdef	CONFIG_CMD_NET

#define	MII_OPMODE_STRAP_OVERRIDE	0x16
#define	MII_PHY_CTRL1			0x1e
#define	MII_PHY_CTRL2			0x1f

void mx28_adjust_mac(int dev_id, unsigned char *mac)
{
	/* Use NetComm OID */
	mac[0] = 0x00;
	mac[1] = 0x60;

	if (dev_id == 1) /* Let MAC1 be MAC0 + 1 by default */
		mac[5] += 1;
}

int fecmxc_mii_postcall(int phy)
{
	/* Nothing at the moment, used to contain hacks for FEC1 */
	return 0;
}

int board_eth_init(bd_t *bis)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;
	struct eth_device *dev;
	int ret;

	ret = cpu_eth_init(bis);

	/* MX28EVK uses ENET_CLK PAD to drive FEC clock */
	writel(CLKCTRL_ENET_TIME_SEL_RMII_CLK | CLKCTRL_ENET_CLK_OUT_EN,
					&clkctrl_regs->hw_clkctrl_enet);

	/* Reset PHY */
	gpio_set(ETH_RESETn, 0);
	udelay(200);
	gpio_set(ETH_RESETn, 1);
	udelay(100);

	ret = fecmxc_initialize_multi(bis, 0, 0, MXS_ENET0_BASE);
	if (ret) {
		puts("FEC MXS: Unable to init FEC0\n");
		return ret;
	}

	dev = eth_get_dev_by_name("FEC0");
	if (!dev) {
		puts("FEC MXS: Unable to get FEC0 device entry\n");
		return -EINVAL;
	}

	ret = fecmxc_register_mii_postcall(dev, fecmxc_mii_postcall);
	if (ret) {
		puts("FEC MXS: Unable to register FEC0 mii postcall\n");
		return ret;
	}

	return ret;
}

#endif
