/*
 * Freescale i.MX28 RAM init
 *
 * Copyright (C) 2011 Marek Vasut <marek.vasut@gmail.com>
 * on behalf of DENX Software Engineering GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <config.h>
#include <asm/io.h>
#include <asm/arch/iomux-mx28.h>
#include <asm/arch/imx-regs.h>

#include "mx28_init.h"

/* Start memtest */
static inline unsigned hash(unsigned v)
{
	return(v ^ (v << 16) ^ (v << 8) ^ (v >> 8) ^ (v >> 16) ^ 0x2dc05c09);
}

/* a=expected, b=got */
static void bincomp_print(unsigned a, unsigned b)
{
	unsigned i;
	for (i=0; i<32; i++) {
		unsigned ma, mb;
		ma = a & 0x80000000;
		mb = b & 0x80000000;
		if (ma > mb)
			app_printf("v");
		else if (ma < mb)
			app_printf("^");
		else if (ma)
			app_printf("1");
		else
			app_printf("0");
		a<<=1; b<<=1;
	}
}

static inline unsigned mr(const unsigned addr)
{
	volatile unsigned *a = (volatile unsigned *)addr;
	return *a;
}

static inline void mw(const unsigned addr, const unsigned v)
{
	volatile unsigned *a = (volatile unsigned *)addr;
	*a = v;
}



/* Read enough data to flush data cache if enabled. */
static void flush(unsigned addr)
{
	unsigned i,d;
	for (i=0; i<64*1024/4; i++) {
		d = mr(addr+i*4);
	}
	(void)d;
}

static unsigned bitwalk(unsigned addr)
{
	unsigned e=0;
	unsigned i,v,h;

	app_printf("0x%08x: Walking 1\r\n", addr);
	for (i=0; i<32; i++) {
		v  = (1 << i);
		mw(addr,v);
		flush(addr);
		h = mr(addr);
		if (v != h) {
			app_printf("    Bit 0x%08x is stuck to 0\r\n");
			e++;
		}
	}
	app_printf("            Walking 0\r\n");
	for (i=0; i<32; i++) {
		v  = ~(1 << i);
		mw(addr,v);
		flush(addr);
		h = mr(addr);
		if (v != h) {
			app_printf("Bit 0x%08x is stuck to 1\r\n");
			e++;
		}
	}
	return e;
}

static unsigned addr_walk(unsigned base, unsigned len, unsigned pat)
{
	unsigned e=0;
	unsigned i,v;

	for (i=4; i<len; i<<=1) {
		mw(base+i, pat);
		mw(base, ~pat);
		v = mr(base+i);
		if (v != pat) {
			app_printf("Addr 0x%08x may be stuck\r\n", base+i);
			e++;
		}
	}
	return e;
}

void complicated_test(const unsigned base, const unsigned length)
{
	unsigned i,v,m,h,e;
	app_printf("Mem test, base=0x%08x, length=0x%08x (%d MB)\r\n", base, length, length / (1024*1024));

	e=0;

	e += bitwalk(base);
	e += addr_walk(base, length, 0xFFFFFFFF);
	e += addr_walk(base, length, 0x00000000);
	e += addr_walk(base, length, 0x55555555);
	e += addr_walk(base, length, 0xaaaaaaaa);

	app_printf("Fill memory with known data\r\n");
	m = 0;
	for (i=0; i<length; i+=4) {
		mw(base+i, hash(i));
		m++;
		if (m > 1024*1024/4) {
			m = 0;
			app_printf("\r[0x%08x]: ", base+i);
		}
	}
	app_printf("\r[0x%08x]: ", base+i);
	app_printf("\r\n");

	app_printf("Reading data back\r\n");
	m = 0;
	for (i=0; i<length; i+=4) {
		h = hash(i);
		v = mr(base+i);
		if (v != h) {
			app_printf("\r[0x%08x]: [0x%08x] |", base+i, base+i);
			bincomp_print(h, v);
			app_printf("|");
		}
		m++;
		if (m > 1024*1024/4) {
			m = 0;
			app_printf("\r[0x%08x]:", base+i);
		}
		
	}
	app_printf("\r[0x%08x]:", base+i);
	app_printf("\r\n");

	app_printf("Memtest end\r\n");
}
/* End mem test */


uint32_t dram_vals[] = {
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000100, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00010101, 0x01010101,
	0x000f0f01, 0x0f02020a, 0x00000000, 0x00010101,
	0x00000100, 0x00000100, 0x00000000, 0x00000002,
	0x01010000, 0x05060302, 0x06005003, 0x0a0000c8,
	0x02009c40, 0x0000030c, 0x0036a609, 0x031a0612,
	0x02030202, 0x00c8001c, 0x00000000, 0x00000000,
	0x00012100, 0xffff0303, 0x00012100, 0xffff0303,
	0x00012100, 0xffff0303, 0x00012100, 0xffff0303,
	0x00000003, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000612, 0x01000F02,
	0x06120612, 0x00000200, 0x00020007, 0xf5014b27,
	0xf5014b27, 0xf5014b27, 0xf5014b27, 0x07000300,
	0x07000300, 0x07000300, 0x07000300, 0x00000006,
	0x00000000, 0x00000000, 0x01000000, 0x01020408,
	0x08040201, 0x000f1133, 0x00000000, 0x00001f04,
	0x00001f04, 0x00001f04, 0x00001f04, 0x00001f04,
	0x00001f04, 0x00001f04, 0x00001f04, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00010000, 0x00020304,
	0x00000004, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x01010000,
	0x01000000, 0x03030000, 0x00010303, 0x01020202,
	0x00000000, 0x02040303, 0x21002103, 0x00061200,
	0x06120612, 0x04320432, 0x04320432, 0x00040004,
	0x00040004, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00010001
};

void __mx28_adjust_memory_params(uint32_t *dram_vals)
{
}
void mx28_adjust_memory_params(uint32_t *dram_vals)
	__attribute__((weak, alias("__mx28_adjust_memory_params")));

void init_m28_200mhz_ddr2(void)
{
	int i;

	mx28_adjust_memory_params(dram_vals);

	for (i = 0; i < ARRAY_SIZE(dram_vals); i++)
		writel(dram_vals[i], MXS_DRAM_BASE + (4 * i));
}

void mx28_mem_init_clock(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;

	/* Gate EMI clock */
	writeb(CLKCTRL_FRAC_CLKGATE,
		&clkctrl_regs->hw_clkctrl_frac0_set[CLKCTRL_FRAC0_EMI]);

	/* Set fractional divider for ref_emi to 480 * 18 / 21 = 411MHz */
	writeb(CLKCTRL_FRAC_CLKGATE | (21 & CLKCTRL_FRAC_FRAC_MASK),
		&clkctrl_regs->hw_clkctrl_frac0[CLKCTRL_FRAC0_EMI]);

	/* Ungate EMI clock */
	writeb(CLKCTRL_FRAC_CLKGATE,
		&clkctrl_regs->hw_clkctrl_frac0_clr[CLKCTRL_FRAC0_EMI]);

	early_delay(11000);

	/* Set EMI clock divider for EMI clock to 411 / 2 = 205MHz */
	writel((2 << CLKCTRL_EMI_DIV_EMI_OFFSET) |
		(1 << CLKCTRL_EMI_DIV_XTAL_OFFSET),
		&clkctrl_regs->hw_clkctrl_emi);

	/* Unbypass EMI */
	writel(CLKCTRL_CLKSEQ_BYPASS_EMI,
		&clkctrl_regs->hw_clkctrl_clkseq_clr);

	early_delay(10000);
}

void mx28_mem_setup_cpu_and_hbus(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;

	/* Set fractional divider for ref_cpu to 480 * 18 / 19 = 454MHz
	 * and ungate CPU clock */
	writeb(19 & CLKCTRL_FRAC_FRAC_MASK,
		(uint8_t *)&clkctrl_regs->hw_clkctrl_frac0[CLKCTRL_FRAC0_CPU]);

	/* Set CPU bypass */
	writel(CLKCTRL_CLKSEQ_BYPASS_CPU,
		&clkctrl_regs->hw_clkctrl_clkseq_set);

	/* HBUS = 151MHz */
	writel(CLKCTRL_HBUS_DIV_MASK, &clkctrl_regs->hw_clkctrl_hbus_set);
	writel(((~3) << CLKCTRL_HBUS_DIV_OFFSET) & CLKCTRL_HBUS_DIV_MASK,
		&clkctrl_regs->hw_clkctrl_hbus_clr);

	early_delay(10000);

	/* CPU clock divider = 1 */
	clrsetbits_le32(&clkctrl_regs->hw_clkctrl_cpu,
			CLKCTRL_CPU_DIV_CPU_MASK, 1);

	/* Disable CPU bypass */
	writel(CLKCTRL_CLKSEQ_BYPASS_CPU,
		&clkctrl_regs->hw_clkctrl_clkseq_clr);

	early_delay(15000);
}

void mx28_mem_setup_vdda(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	writel((0xc << POWER_VDDACTRL_TRG_OFFSET) |
		(0x7 << POWER_VDDACTRL_BO_OFFSET_OFFSET) |
		POWER_VDDACTRL_LINREG_OFFSET_1STEPS_BELOW,
		&power_regs->hw_power_vddactrl);
}

void mx28_mem_setup_vddd(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	writel((0x1c << POWER_VDDDCTRL_TRG_OFFSET) |
		(0x7 << POWER_VDDDCTRL_BO_OFFSET_OFFSET) |
		POWER_VDDDCTRL_LINREG_OFFSET_1STEPS_BELOW,
		&power_regs->hw_power_vdddctrl);
}

uint32_t mx28_mem_get_size(void)
{
	uint32_t sz, da;
	uint32_t *vt = (uint32_t *)0x20;
	/* The following is "subs pc, r14, #4", used as return from DABT. */
	const uint32_t data_abort_memdetect_handler = 0xe25ef004;

	/* Replace the DABT handler. */
	da = vt[4];
	vt[4] = data_abort_memdetect_handler;

	/* WARNING: There are two memory sizes currently in use with
	 * the Falcon boards, 64MB & 128MB. The difference is that the 128MB
	 * chip has 8 banks, the 64MB has 4 banks.
	 * 
	 * Due to this, the SDRAM controller cannot be configured to work
	 * with both, since the bank bits are mapped into the middle of the
	 * address range, between row and column addresses.
	 * 
	 * So we configure for 128MB first (see overrides in iomux.c), then
	 * run a check here for specific errors in case we have a 64MB SDRAM.
	 * We then reprogram the SDRAM controller to match and hope that there
	 * are no hardware bugs related to config changes after initialisation.
	 */
	{
		volatile uint32_t *mem = (uint32_t *)PHYS_SDRAM_1;
		volatile uint32_t *sdram_reg31 = (uint32_t *)(0x800E0000UL + 0x7c);
		mem[0x0/4] = 0;
		dmb();
		mem[0x2000/4] = 0xffffffff;
		mem[(64 MB)/4] = 0xffffffff;
		dmb();
		if (mem[0x0/4] == 0) {
			sz = 128 MB;
			D("128MB detected\r\n");
		} else {
			sz = 64 MB;
			*sdram_reg31 &= ~(1UL << 16);
			D("64MB detected\r\n");
		}
	}

	/* Restore the old DABT handler. */
	vt[4] = da;

	D("MEM %u\r\n", sz / (1024*1024));

#if 0

	if (sz < (64 MB)) {
		D("Overriding size %08x > 64 MB\r\n",sz);
		sz = 64 MB;
	}

	complicated_test(PHYS_SDRAM_1, sz);
#endif

	return sz;
}

void mx28_mem_init(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;
	struct mx28_pinctrl_regs *pinctrl_regs =
		(struct mx28_pinctrl_regs *)MXS_PINCTRL_BASE;

	/* Set DDR2 mode */
	writel(PINCTRL_EMI_DS_CTRL_DDR_MODE_DDR2,
		&pinctrl_regs->hw_pinctrl_emi_ds_ctrl_set);

	/* Power up PLL0 */
	writel(CLKCTRL_PLL0CTRL0_POWER,
		&clkctrl_regs->hw_clkctrl_pll0ctrl0_set);

	early_delay(11000);

	mx28_mem_init_clock();

	mx28_mem_setup_vdda();

	/*
	 * Configure the DRAM registers
	 */

	/* Clear START bit from DRAM_CTL16 */
	clrbits_le32(MXS_DRAM_BASE + 0x40, 1);

	init_m28_200mhz_ddr2();

	/* Clear SREFRESH bit from DRAM_CTL17 */
	clrbits_le32(MXS_DRAM_BASE + 0x44, 1);

	/* Set START bit in DRAM_CTL16 */
	setbits_le32(MXS_DRAM_BASE + 0x40, 1);

	/* Wait for bit 20 (DRAM init complete) in DRAM_CTL58 */
	while (!(readl(MXS_DRAM_BASE + 0xe8) & (1 << 20)))
		;

	mx28_mem_setup_vddd();

	early_delay(10000);

	mx28_mem_setup_cpu_and_hbus();
}
