
/*===========================================================================
FILE: 
   GobiSerial.c

DESCRIPTION:
   Linux Qualcomm Serial USB driver Implementation 

PUBLIC DRIVER FUNCTIONS:
   GobiProbe
   GobiOpen
   GobiClose
   GobiReadBulkCallback (if kernel is less than 2.6.25)
   GobiSuspend
   GobiResume (if kernel is less than 2.6.24)

Copyright (c) 2011, Code Aurora Forum. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Code Aurora Forum nor
      the names of its contributors may be used to endorse or promote
      products derived from this software without specific prior written
      permission.

Alternatively, provided that this notice is retained in full, this software
may be relicensed by the recipient under the terms of the GNU General Public
License version 2 ("GPL") and only version 2, in which case the provisions of
the GPL apply INSTEAD OF those given above.  If the recipient relicenses the
software under the GPL, then the identification text in the MODULE_LICENSE
macro must be changed to reflect "GPLv2" instead of "Dual BSD/GPL".  Once a
recipient changes the license terms to the GPL, subsequent recipients shall
not relicense under alternate licensing terms, including the BSD or dual
BSD/GPL terms.  In addition, the following license statement immediately
below and between the words START and END shall also then apply when this
software is relicensed under the GPL:

START

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 and only version 2 as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

END

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
==========================================================================*/
//---------------------------------------------------------------------------
// Include Files
//---------------------------------------------------------------------------

#include <linux/init.h>
#include <linux/module.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/usb.h> 
#include <linux/usb/serial.h>
#include <linux/version.h>

//---------------------------------------------------------------------------
// Global veriable and defination
//---------------------------------------------------------------------------

// Version Information
#define DRIVER_VERSION "1.0.20"
#define DRIVER_AUTHOR "Qualcomm Innovation Center"
#define DRIVER_DESC "GobiSerial"

#define NUM_BULK_EPS         1
#define MAX_BULK_EPS         6

//#define GPS_ENABLE

// Debug flag
static int debug;

// Global pointer to usb_serial_generic_close function
// This function is not exported, which is why we have to use a pointer
// instead of just calling it.
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
   void (* gpClose)( 
      struct usb_serial_port *,
      struct file * );
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,29 ))
   void (* gpClose)( 
      struct tty_struct *,
      struct usb_serial_port *,
      struct file * );
#else // > 2.6.29
   void (* gpClose)( struct usb_serial_port * );
#endif

// DBG macro
#define DBG( format, arg... ) \
   if (debug == 1)\
   { \
      printk( KERN_INFO "GobiSerial::%s " format, __FUNCTION__, ## arg ); \
   } \

/*=========================================================================*/
// Function Prototypes
/*=========================================================================*/

// Attach to correct interfaces
static int GobiProbe(
   struct usb_serial * pSerial, 
   const struct usb_device_id * pID );
   
// Start GPS if GPS port, run usb_serial_generic_open
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
   int GobiOpen(
      struct usb_serial_port *   pPort,
      struct file *              pFilp );
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,31 ))
   int GobiOpen(
      struct tty_struct *        pTTY,
      struct usb_serial_port *   pPort,
      struct file *              pFilp );
#else // > 2.6.31
   int GobiOpen(
      struct tty_struct *        pTTY,
      struct usb_serial_port *   pPort );
#endif

// Stop GPS if GPS port, run usb_serial_generic_close
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
   void GobiClose( 
      struct usb_serial_port *,
      struct file * );
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,29 ))
   void GobiClose( 
      struct tty_struct *,
      struct usb_serial_port *,
      struct file * );
#else // > 2.6.29
   void GobiClose( struct usb_serial_port * );
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION( 2,6,25 ))

// Read data from USB, push to TTY and user space
static void GobiReadBulkCallback( struct urb * pURB );

#endif

// Set reset_resume flag
int GobiSuspend( 
   struct usb_interface *     pIntf,
   pm_message_t               powerEvent );

#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,23 ))

// Restart URBs killed during usb_serial_suspend
int GobiResume( struct usb_interface * pIntf );

#endif

/*=========================================================================*/
// Qualcomm Gobi 3000 VID/PIDs
/*=========================================================================*/
static struct usb_device_id GobiVIDPIDTable[] = 
{
   { USB_DEVICE( 0x05c6, 0x920c ) },   // Gobi 3000 QDL device
   { USB_DEVICE( 0x05c6, 0x920d ) },   // Gobi 3000 Composite Device
   { USB_DEVICE( 0x05c6, 0x9002 ) },   // Quanta TD-LTE device
   { USB_DEVICE( 0x1199, 0x68A2 ) },   // SWI Composite Device
   { USB_DEVICE( 0x1199, 0x68C0 ) },   // SWI Composite Device
   { USB_DEVICE( 0x1199, 0x9011 ) },   // SWI MC8355 Device
   { USB_DEVICE( 0x19d2, 0x0257 ) },   // ZTE MF821
   { USB_DEVICE( 0x0408, 0xea1a ) },   // QUANTA (bulk module - model name 0)
   { USB_DEVICE( 0x0408, 0xea24 ) },   // QUANTA (bulk module - model name 0)
   { USB_DEVICE( 0x0408, 0xea25 ) },   // QUANTA (bulk module - model name 0 again!)
   { USB_DEVICE( 0x05c6, 0x9008 ) },   // QUANTA QDM-3 in download mode
   { USB_DEVICE( 0x05C6, 0x900E ) },   // QUANTA QDM-3 in download mode
   { USB_DEVICE( 0x0408, 0x900E ) },   // QUANTA QDM-3 in download mode
   { USB_DEVICE( 0x1e2d, 0x0053 ) },   // QUANTA QDM-3 in download mode
   { }                                 // Terminating entry
};
MODULE_DEVICE_TABLE( usb, GobiVIDPIDTable );

#define GOBISERIAL_ACCEPT_LIST_INTERFACES(x,y)		( (((x)<<16)&0xffff0000) | ((y)&0x0000ffff) )

struct gobiserial_accept_inf_struct {
	struct usb_device_id device_id;
	unsigned int inf_map;
};

struct gobiserial_accept_inf_struct accept_inf_list[]={

	{{ USB_DEVICE( 0x05c6, 0x920c ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // Gobi 3000 QDL device
	{{ USB_DEVICE( 0x05c6, 0x920d ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // Gobi 3000 Composite Device
	{{ USB_DEVICE( 0x05c6, 0x9002 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // Quanta TD-LTE device
	{{ USB_DEVICE( 0x1199, 0x68A2 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // SWI Composite Device
	{{ USB_DEVICE( 0x1199, 0x68C0 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // SWI Composite Device
	{{ USB_DEVICE( 0x1199, 0x9011 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // SWI MC8355 Device
	{{ USB_DEVICE( 0x19d2, 0x0257 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES( 0x00,0x07)},  // ZTE MF821

	{{ USB_DEVICE( 0x0408, 0xea1a ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES(~0x0e,0x0e)},  // QUANTA (bulk module - model name 0)
	{{ USB_DEVICE( 0x0408, 0xea24 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES( 0x00,0xF7)},  // QUANTA (bulk module - model name 0)
	{{ USB_DEVICE( 0x0408, 0xea25 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES( 0x00,0x07)},  // QUANTA (bulk module - model name 0 again!)
		       
	{{ USB_DEVICE( 0x1e2d, 0x0053 ) }, GOBISERIAL_ACCEPT_LIST_INTERFACES( 0x00,0x0f)},  // Cinterion (PHP-8)

	{} // Terminating entry
};

#if (LINUX_VERSION_CODE < KERNEL_VERSION( 3,4,0 ))
/*=========================================================================*/
// Struct usb_serial_driver
// Driver structure we register with the USB core
/*=========================================================================*/
static struct usb_driver GobiDriver = 
{
   .name       = "GobiSerial",
   .probe      = usb_serial_probe,
   .disconnect = usb_serial_disconnect,
   .id_table   = GobiVIDPIDTable,
   //.suspend    = GobiSuspend,
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,23 ))
   //.resume     = GobiResume,
#else
   .resume     = usb_serial_resume,
#endif
   .supports_autosuspend = true,
};
#endif

/*=========================================================================*/
// Struct usb_serial_driver
/*=========================================================================*/
static struct usb_serial_driver gGobiDevice = 
{
   .driver = 
   {   
      .owner     = THIS_MODULE,
      .name      = "GobiSerial driver",
   },
   .description         = "GobiSerial",
   .id_table            = GobiVIDPIDTable,
#if (LINUX_VERSION_CODE < KERNEL_VERSION( 3,4,0 ))
   .usb_driver          = &GobiDriver,
#endif   
   .num_ports           = 1,
   .probe               = GobiProbe,
   .open                = GobiOpen,
#if (LINUX_VERSION_CODE < KERNEL_VERSION( 2,6,25 ))
   .num_interrupt_in    = NUM_DONT_CARE,
   .num_bulk_in         = 1,
   .num_bulk_out        = 1,
   .read_bulk_callback  = GobiReadBulkCallback,
#endif
};

#if (LINUX_VERSION_CODE >= KERNEL_VERSION( 3,4,0 ))
static struct usb_serial_driver* const serial_drivers[]={&gGobiDevice,NULL};
#endif

//---------------------------------------------------------------------------
// USB serial core overridding Methods
//---------------------------------------------------------------------------



/*===========================================================================
METHOD:
   GobiProbe (Free Method)

DESCRIPTION:
   Attach to correct interfaces

PARAMETERS:
   pSerial    [ I ] - Serial structure 
   pID        [ I ] - VID PID table

RETURN VALUE:
   int - negative error code on failure
         zero on success
===========================================================================*/
static int GobiProbe(
   struct usb_serial * pSerial, 
   const struct usb_device_id * pID )
{
   // Assume failure
   int nRetval = -ENODEV;

   int nNumInterfaces;
   int nInterfaceNum;
   
   int numEndpoints;
   int endpointIndex;
   struct usb_host_endpoint * pEndpoint = NULL;
   struct usb_host_endpoint * pIn = NULL;
   struct usb_host_endpoint * pOut = NULL;
   struct usb_host_endpoint * pInt = NULL;
   
   struct usb_interface* pIntf;
   
   int i;
   int inf_map_nonint;
   int inf_map;
   int add;
   
   struct gobiserial_accept_inf_struct* accept_inf=0;
   int mask;
   
   DBG( "\n" );

   // Test parameters
   if ( (pSerial == NULL)
   ||   (pSerial->dev == NULL)
   ||   (pSerial->dev->actconfig == NULL)
   ||   (pSerial->interface == NULL)
   ||   (pSerial->interface->cur_altsetting == NULL)
   ||   (pSerial->type == NULL) )
   {
      DBG( "invalid parameter\n" );
      return -EINVAL;
   }

   nNumInterfaces = pSerial->dev->actconfig->desc.bNumInterfaces;
   DBG( "Num Interfaces = %d\n", nNumInterfaces );
   nInterfaceNum = pSerial->interface->cur_altsetting->desc.bInterfaceNumber;
   DBG( "This Interface = %d\n", nInterfaceNum );
   
   pIntf=pSerial->interface;
   
   // Collect In and Out endpoints
   numEndpoints = pIntf->cur_altsetting->desc.bNumEndpoints;
   for (endpointIndex = 0; endpointIndex < numEndpoints; endpointIndex++)
   {
      pEndpoint = pIntf->cur_altsetting->endpoint + endpointIndex;
      if (pEndpoint == NULL)
      {
         DBG( "invalid endpoint %u\n", endpointIndex );
         return -ENODEV;
      }

      if (usb_endpoint_dir_in( &pEndpoint->desc ) == true) {
         if( usb_endpoint_xfer_int( &pEndpoint->desc ) == true) {
            pInt = pEndpoint;
         }
         else {
            pIn = pEndpoint;
         }
      }
      else if (usb_endpoint_dir_out( &pEndpoint->desc ) == true)
      {
         pOut = pEndpoint;
      }
   }
   
   
   if (nNumInterfaces == 1)
   {
      // QDL mode?
      if (nInterfaceNum == 1 || nInterfaceNum == 0) 
      {
         DBG( "QDL port found\n" );
         nRetval = usb_set_interface( pSerial->dev, 
                                      nInterfaceNum, 
                                      0 );
         if (nRetval < 0)
         {
            DBG( "Could not set interface, error %d\n", nRetval );
         }
      }
      else
      {
         DBG( "Incorrect QDL interface number\n" );
      }
   }
   else {
	// search accept list	
	for(i=0;accept_inf_list[i].device_id.idVendor;i++) {
		if( (pID->idVendor==accept_inf_list[i].device_id.idVendor) && (pID->idProduct==accept_inf_list[i].device_id.idProduct) ) {
			accept_inf=&accept_inf_list[i];
		}
	}
		   
	if(accept_inf) {
		// get black interface map
		inf_map_nonint=(accept_inf->inf_map>>16) & 0xffff;
		inf_map=accept_inf->inf_map & 0xffff;
		
		printk( KERN_INFO "%s: accept interface map found - (id=0x%04x:0x%04x,inf=%d,nonint=0x%04x,map=0x%04x)\n", DRIVER_DESC, pID->idVendor,pID->idProduct,nInterfaceNum,inf_map_nonint,inf_map);
		
		// interface map
		mask=1<<nInterfaceNum;
		
		// if the interface exists in inf_map or if the non-interrupt-based interface exists in nonint
		add=((inf_map_nonint & mask) && !pInt) || (inf_map & mask);
	}
	else {
		printk( KERN_INFO "%s: no interface map found. accepting all interfaces - (id=0x%04x:0x%04x)\n", DRIVER_DESC, pID->idVendor,pID->idProduct);
		add=1;
	}
		
	if(add) {
		printk( KERN_INFO "%s: USB interface(%d) accepted \n", DRIVER_DESC, nInterfaceNum);
		
		nRetval = usb_set_interface( pSerial->dev, nInterfaceNum, 0 );
		
		if (nRetval < 0) {
			DBG( "Could not set interface, error %d\n", nRetval );
		}
	}
	else {
		printk( KERN_INFO "%s: USB interface(%d) rejected\n", DRIVER_DESC, nInterfaceNum);
	}
   }

   return nRetval;
}

/*===========================================================================
METHOD:
   GobiOpen (Free Method)

DESCRIPTION:
   Start GPS if GPS port, run usb_serial_generic_open

PARAMETERS:
   pTTY    [ I ] - TTY structure (only on kernels <= 2.6.26)
   pPort   [ I ] - USB serial port structure
   pFilp   [ I ] - File structure (only on kernels <= 2.6.31)

RETURN VALUE:
   int - zero for success
       - negative errno on error
===========================================================================*/
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
int GobiOpen(
   struct usb_serial_port *   pPort,
   struct file *              pFilp )
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,31 ))
int GobiOpen(
   struct tty_struct *        pTTY,
   struct usb_serial_port *   pPort,
   struct file *              pFilp )
#else // > 2.6.31
int GobiOpen(
   struct tty_struct *        pTTY,
   struct usb_serial_port *   pPort )
#endif
{
   const char startMessage[] = "$GPS_START";
   int nResult;
   int bytesWrote;
   
   DBG( "\n" );
   
   // Test parameters
   if ( (pPort == NULL)
   ||   (pPort->serial == NULL)
   ||   (pPort->serial->dev == NULL)
   ||   (pPort->serial->interface == NULL)
   ||   (pPort->serial->interface->cur_altsetting == NULL) )
   {
      DBG( "invalid parameter\n" );
      return -EINVAL;
   }
    

#ifdef GPS_ENABLE
   // Is this the GPS port?
   if (pPort->serial->interface->cur_altsetting->desc.bInterfaceNumber == 3)
   {
      // Send startMessage, 1s timeout
      nResult = usb_bulk_msg( pPort->serial->dev,
                              usb_sndbulkpipe( pPort->serial->dev,
                                               pPort->bulk_out_endpointAddress ),
                              (void *)&startMessage[0],
                              sizeof( startMessage ),
                              &bytesWrote,
                              1000 );
      if (nResult != 0)
      {
         DBG( "error %d sending startMessage\n", nResult );
         return nResult;
      }
      if (bytesWrote != sizeof( startMessage ))
      {
         DBG( "invalid write size %d, %d\n", 
              bytesWrote, 
              sizeof( startMessage ) );
         return -EIO;
      }      
   }
#endif			      
   
   // Pass to usb_serial_generic_open
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
   return usb_serial_generic_open( pPort, pFilp );
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,31 ))
   return usb_serial_generic_open( pTTY, pPort, pFilp );
#else // > 2.6.31
   return usb_serial_generic_open( pTTY, pPort );
#endif
}

/*===========================================================================
METHOD:
   GobiClose (Free Method)

DESCRIPTION:
   Stop GPS if GPS port, run usb_serial_generic_close

PARAMETERS:
   pTTY    [ I ] - TTY structure (only if kernel > 2.6.26 and <= 2.6.29)
   pPort   [ I ] - USB serial port structure
   pFilp   [ I ] - File structure (only on kernel <= 2.6.29)
===========================================================================*/
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
void GobiClose(
   struct usb_serial_port *   pPort,
   struct file *              pFilp )
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,29 ))
void GobiClose(
   struct tty_struct *        pTTY,
   struct usb_serial_port *   pPort,
   struct file *              pFilp )
#else // > 2.6.29
void GobiClose( struct usb_serial_port * pPort )
#endif
{
   const char stopMessage[] = "$GPS_STOP";
   int nResult;
   int bytesWrote;
   
   DBG( "\n" );
   
   // Test parameters
   if ( (pPort == NULL)
   ||   (pPort->serial == NULL)
   ||   (pPort->serial->dev == NULL)
   ||   (pPort->serial->interface == NULL)
   ||   (pPort->serial->interface->cur_altsetting == NULL) )
   {
      DBG( "invalid parameter\n" );
      return;
   }
   
#ifdef GPS_ENABLE
   // Is this the GPS port?
   if (pPort->serial->interface->cur_altsetting->desc.bInterfaceNumber == 3)
   {
      // Send stopMessage, 1s timeout
      nResult = usb_bulk_msg( pPort->serial->dev,
                              usb_sndbulkpipe( pPort->serial->dev,
                                               pPort->bulk_out_endpointAddress ),
                              (void *)&stopMessage[0],
                              sizeof( stopMessage ),
                              &bytesWrote,
                              1000 );
      if (nResult != 0)
      {
         DBG( "error %d sending stopMessage\n", nResult );
      }
      if (bytesWrote != sizeof( stopMessage ))
      {
         DBG( "invalid write size %d, %d\n", 
              bytesWrote, 
              sizeof( stopMessage ) );
      }      
   }
#endif   
   
   // Pass to usb_serial_generic_close
   if (gpClose == NULL)
   {
      DBG( "NULL gpClose\n" );
      return;
   }
   
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,26 ))
   gpClose( pPort, pFilp );
#elif (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,29 ))
   gpClose( pTTY, pPort, pFilp );
#else // > 2.6.29
   gpClose( pPort );
#endif
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION( 2,6,25 ))

/*===========================================================================
METHOD:
   GobiReadBulkCallback (Free Method)

DESCRIPTION:
   Read data from USB, push to TTY and user space

PARAMETERS:
   pURB  [ I ] - USB Request Block (urb) that called us 

RETURN VALUE:
===========================================================================*/
static void GobiReadBulkCallback( struct urb * pURB )
{
   struct usb_serial_port * pPort = pURB->context;
   struct tty_struct * pTTY = pPort->tty;
   int nResult;
   int nRoom = 0;
   unsigned int pipeEP;
   
   DBG( "port %d\n", pPort->number );

   if (pURB->status != 0) 
   {
      DBG( "nonzero read bulk status received: %d\n", pURB->status );

      return;
   }

   usb_serial_debug_data( debug, 
                          &pPort->dev, 
                          __FUNCTION__, 
                          pURB->actual_length, 
                          pURB->transfer_buffer );

   // We do no port throttling

   // Push data to tty layer and user space read function
   if (pTTY != 0 && pURB->actual_length) 
   {
      nRoom = tty_buffer_request_room( pTTY, pURB->actual_length );
      DBG( "room size %d %d\n", nRoom, 512 );
      if (nRoom != 0)
      {
         tty_insert_flip_string( pTTY, pURB->transfer_buffer, nRoom );
         tty_flip_buffer_push( pTTY );
      }
   }

   pipeEP = usb_rcvbulkpipe( pPort->serial->dev, 
                             pPort->bulk_in_endpointAddress );

   // For continuous reading
   usb_fill_bulk_urb( pPort->read_urb, 
                      pPort->serial->dev,
                      pipeEP,
                      pPort->read_urb->transfer_buffer,
                      pPort->read_urb->transfer_buffer_length,
                      GobiReadBulkCallback, 
                      pPort );
   
   nResult = usb_submit_urb( pPort->read_urb, GFP_ATOMIC );
   if (nResult != 0)
   {
      DBG( "failed resubmitting read urb, error %d\n", nResult );
   }
}

#endif

/*===========================================================================
METHOD:
   GobiSuspend (Public Method)

DESCRIPTION:
   Set reset_resume flag

PARAMETERS
   pIntf          [ I ] - Pointer to interface
   powerEvent     [ I ] - Power management event

RETURN VALUE:
   int - 0 for success
         negative errno for failure
===========================================================================*/
/*int GobiSuspend( 
   struct usb_interface *     pIntf,
   pm_message_t               powerEvent )
{
   struct usb_serial * pDev;
   
   if (pIntf == 0)
   {
      return -ENOMEM;
   }
   
   pDev = usb_get_intfdata( pIntf );
   if (pDev == NULL)
   {
      return -ENXIO;
   }

   // Unless this is PM_EVENT_SUSPEND, make sure device gets rescanned
   if ((powerEvent.event & PM_EVENT_SUSPEND) == 0)
   {
      pDev->dev->reset_resume = 1;
   }
   
   // Run usb_serial's suspend function
   return usb_serial_suspend( pIntf, powerEvent );
}
*/
#if (LINUX_VERSION_CODE <= KERNEL_VERSION( 2,6,23 ))

/*===========================================================================
METHOD:
   GobiResume (Free Method)

DESCRIPTION:
   Restart URBs killed during usb_serial_suspend

   Fixes 2 bugs in 2.6.23 kernel
      1. pSerial->type->resume was NULL and unchecked, caused crash.
      2. set_to_generic_if_null was not run for resume.

PARAMETERS:
   pIntf  [ I ] - Pointer to interface

RETURN VALUE:
   int - 0 for success
         negative errno for failure
===========================================================================*/
/*int GobiResume( struct usb_interface * pIntf )
{
   struct usb_serial * pSerial = usb_get_intfdata( pIntf );
   struct usb_serial_port * pPort;
   int portIndex, errors, nResult;

   if (pSerial == NULL)
   {
      DBG( "no pSerial\n" );
      return -ENOMEM;
   }
   if (pSerial->type == NULL)
   {
      DBG( "no pSerial->type\n" );
      return ENOMEM;
   }
   if (pSerial->type->resume == NULL)
   {
      // Expected behaviour in 2.6.23, in later kernels this was handled
      // by the usb-serial driver and usb_serial_generic_resume
      errors = 0;
      for (portIndex = 0; portIndex < pSerial->num_ports; portIndex++)
      {
         pPort = pSerial->port[portIndex];
         if (pPort->open_count > 0 && pPort->read_urb != NULL)
         {
            nResult = usb_submit_urb( pPort->read_urb, GFP_NOIO );
            if (nResult < 0)
            {
               // Return first error we see
               DBG( "error %d\n", nResult );
               return nResult;
            }
         }
      }

      // Success
      return 0;
   }

   // Execution would only reach this point if user has
   // patched version of usb-serial driver.
   return usb_serial_resume( pIntf );
}
*/
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION( 3,4,0 ))
/*===========================================================================
METHOD:
   GobiInit (Free Method)

DESCRIPTION:
   Register the driver and device

PARAMETERS:

RETURN VALUE:
   int - negative error code on failure
         zero on success
===========================================================================*/
static int __init GobiInit( void )
{
   int nRetval = 0;
   gpClose = NULL;

   gGobiDevice.num_ports = NUM_BULK_EPS;

   // Registering driver to USB serial core layer 
   nRetval = usb_serial_register( &gGobiDevice );
   if (nRetval != 0)
   {
      return nRetval;
   }

   // Registering driver to USB core layer
   nRetval = usb_register( &GobiDriver );
   if (nRetval != 0) 
   {
      usb_serial_deregister( &gGobiDevice );
      return nRetval;
   }

   // This will be shown whenever driver is loaded
   printk( KERN_INFO "%s: %s\n", DRIVER_DESC, DRIVER_VERSION );
   
   return nRetval;
}

/*===========================================================================
METHOD:
   GobiExit (Free Method)

DESCRIPTION:
   Deregister the driver and device

PARAMETERS:

RETURN VALUE:
===========================================================================*/
static void __exit GobiExit( void )
{
   gpClose = NULL;
   usb_deregister( &GobiDriver );
   usb_serial_deregister( &gGobiDevice );
}
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION( 3,4,0 ))
// Calling kernel module to init our driver
module_init( GobiInit );
module_exit( GobiExit );
#else
module_usb_serial_driver(serial_drivers, GobiVIDPIDTable);
#endif

MODULE_VERSION( DRIVER_VERSION );
MODULE_AUTHOR( DRIVER_AUTHOR );
MODULE_DESCRIPTION( DRIVER_DESC );
MODULE_LICENSE( "Dual BSD/GPL" );

module_param( debug, int, S_IRUGO | S_IWUSR );
MODULE_PARM_DESC( debug, "Debug enabled or not" );



