/*
 * host.c - ChipIdea USB host controller driver
 *
 * Copyright (c) 2012 Intel Corporation
 *
 * Author: Alexander Shishkin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/usb.h>
#include <linux/usb/hcd.h>
#include <linux/usb/chipidea.h>
#include <linux/regulator/consumer.h>

#define DLEVEL 2
#include <linux/debug.h>

/*
static void dump_status(u32 status, const char *prefix)
{
	msg("%s: %08x %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n", prefix, status,
		status & (1<<25)?"TI1 ":"",
		status & (1<<24)?"TI0 ":"",
		status & (1<<19)?"UPI ":"",
		status & (1<<18)?"UAI ":"",
		status & (1<<16)?"NAK ":"",
		status & (1<<15)?"AS ":"",
		status & (1<<14)?"PS ":"",
		status & (1<<13)?"RCL ":"",
		status & (1<<12)?"HCH ":"",
		status & (1<<9)?"SLI ":"",
		status & (1<<7)?"SRI ":"",
		status & (1<<6)?"URI ":"",
		status & (1<<5)?"AAI ":"",
		status & (1<<4)?"SEI ":"",
		status & (1<<3)?"FRI ":"",
		status & (1<<2)?"PCI ":"",
		status & (1<<1)?"UEI ":"",
		status & (1<<0)?"UI ":"");
}
*/

#define DEBUG

#define EHCI_STATS

/* This is the value that is written into the TXFILLTUNING register when
 * ehci->txfifo_stall is set. If it isn't set, the value defaults to 2<<16.
 * 
 * Bits 21-16 : TXFIFOTHRESH, burst length to pre-fill into TxFIFO.
 * Bits  12-8 : TXSCHEALTH, software resettable counter, increments if
 *              the HC misses the frame schedule do to pre-fill delay.
 * Bits   6-0 : TXSCHOH, pre-programmed scheduler overhead. Set this to
 *              a value that yields less than 10 scheduling errors/sec
 *              on a busy bus.
 * 
 * This is a workaround for the Freescale txfifo stall bug, where the HC
 * locks up in case of a TxFIFO stall when the buffer wasn't word aligned.
 * We need to ensure that the TxFIFO never stalls.
 */

#define TXFIFOTUNE	((16<<16) | 4)

/* Per HCD extra statistics, used by proc interface and bug workaround code. */
struct extrastats {
	/* Set by ehci driver */
	atomic_t urbs;     /* Urbs currently held by ehci */
	atomic_t urbs_out; /* Urbs scheduled for out endpoints. */

	/* Heuristics history */
	unsigned prev_ints;       /* Last interrupt count */
	unsigned prev_asyncaddr;  /* Last async addr register value */
	unsigned prev_async_hash; /* Hash of async pending QTD list heads */
	unsigned long jiffies;         /* Time delay for drastic actions */
};
#define AINC(a) atomic_inc(&(a))
#define ADEC(a) atomic_dec(&(a))
#define AGET(a) atomic_read(&(a))

#include "../host/ehci.h"

/*
 * Using heuristics to detect the factors relevant to lockup detection.
 * Factor bitmap is returned. If action is 1, print message on lockup,
 * if 2, print message and call BUG().
 * 
 * If urb is != NULL and it's an out urb, OUT_URBS needs at least 2. Call
 * after enqueing.
 */
#define NO_INT_CHANGE    1 /* There has been no interrupt since last call */
#define NO_ASYNC_CH      2 /* There was no change in async pointer */
#define OUT_URBS         4 /* There are pending out urbs. */
#define ASYNC_ACTIVE     8 /* Async schedule is on */
#define PERIODIC_ACTIVE 16 /* Periodic schedule is on */
#define ASYNC_STATIC    32 /* Async HQ/QTD structure is unchanged */
#define FAILED          64 /* A sequence of failures has occured - we are locked up. */

static const char * decode_heuristics(unsigned state)
{
	static char hs[128]="";
	int p = 0;
	if (state & NO_INT_CHANGE) p += sprintf(hs + p, "nINT ");
	if (state & NO_ASYNC_CH) p += sprintf(hs + p, "nAp ");
	if (state & OUT_URBS) p += sprintf(hs + p, "oURBs ");
	if (state & ASYNC_ACTIVE) p += sprintf(hs + p, "AsynAct ");
	if (state & PERIODIC_ACTIVE) p += sprintf(hs + p, "PerAct ");
	if (state & ASYNC_STATIC) p += sprintf(hs + p, "AsynStatic ");
	if (state & FAILED) p += sprintf(hs + p, "FAIL ");
	return hs;
}

static unsigned lockup_heuristics(struct usb_hcd *hcd, struct urb *urb, int action)
{
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	unsigned state=0;
	unsigned n;
	struct ehci_qh *qh;
	struct ehci_qh_hw *qhw;
	unsigned long flags;

	#define FAILPERIOD (HZ/2) /* Bad state must persist for half second */

	spin_lock_irqsave(&ehci->lock, flags);

	/* Number of interrupts */
	n = ehci->stats.normal
	  + ehci->stats.error
	  + ehci->stats.iaa
	  + ehci->stats.lost_iaa;
	if (n == ehci->stats.e.prev_ints) state |= NO_INT_CHANGE;
	ehci->stats.e.prev_ints = n;

	/* Async pointer register change */
	n =  ehci_readl(ehci, &ehci->regs->async_next);
	if (n == ehci->stats.e.prev_asyncaddr) state |= NO_ASYNC_CH;
	ehci->stats.e.prev_asyncaddr = n;

	/* out URBS */
	n = AGET(ehci->stats.e.urbs_out);
	if (urb && !usb_pipein(urb->pipe)) {
		if (n > 1) state |= OUT_URBS;
	} else {
		if (n) state |= OUT_URBS;
	}

	/* Async/periodic schedule is active? */
	n = ehci_readl(ehci, &ehci->regs->command);
	if (n & (1<<5)) state |= ASYNC_ACTIVE;
	if (n & (1<<4)) state |= PERIODIC_ACTIVE;

	/* Async queue hash = XOR of all first QTD HW pointers. */
	n = 0xdeadbeef;
	qh = ehci->async;
	while (qh) {
		qhw =qh->hw;
		n ^= qhw->hw_current;
		qh=qh->qh_next.qh;
	}
	if (n == ehci->stats.e.prev_async_hash) state |= ASYNC_STATIC;
	ehci->stats.e.prev_async_hash = n;

	spin_unlock_irqrestore(&ehci->lock, flags);

	/* We consider ourselves in locked up state if we have a async
	 * schedule that never changes. */
	if (state & ASYNC_ACTIVE) {
		if (state & ASYNC_STATIC &&
				state & OUT_URBS &&
				state & NO_ASYNC_CH &&
				state & NO_INT_CHANGE) {
			/* The first time we think we have a lock-up, we store the current
			 * jiffies value. After that, we wait FAILPERIOD jiffies before
			 * murdering the system. */
			if (!ehci->stats.e.jiffies) {
				ehci->stats.e.jiffies = jiffies;
			} else if (time_after(jiffies, ehci->stats.e.jiffies + FAILPERIOD)) {
				state |= FAILED;
				if (action > 0)
					msg("EHCI is locked up %08x %s\n",
						state, decode_heuristics(state));
				if (action > 1) {
					BUG();
					/* This BUG() implementation doesn't always kill the
					* system so we take the spinlock, disable ints and loop. */
					spin_lock_irqsave(&ehci->lock, flags);
					while(1);
				}
			}
		} else {
			ehci->stats.e.jiffies = 0;
		}
	}

	return state;
}

#define CHIPIDEA_EHCI
#include "ehci_proc.c"

#include "../host/ehci-hcd.c"

#include "ci.h"
#include "bits.h"
#include "host.h"


static int ci_ehci_setup(struct usb_hcd *hcd)
{
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	int ret;
	
	/* This lives in ehci_proc.c */
	/* this hotplug problem has been fixed, add this functions */
	create_ehci_debug_proc(hcd);

	hcd->has_tt = 1;

	ret = ehci_setup(hcd);
	if (ret)
		return ret;

	ehci_port_power(ehci, 0);

	return ret;
}

static const struct hc_driver ci_ehci_hc_driver = {
	.description	= "ehci_hcd",
	.product_desc	= "ChipIdea HDRC EHCI",
	.hcd_priv_size	= sizeof(struct ehci_hcd),

	/*
	 * generic hardware linkage
	 */
	.irq	= ehci_irq,
	.flags	= HCD_MEMORY | HCD_USB2,

	/*
	 * basic lifecycle operations
	 */
	.reset		= ci_ehci_setup,
	.start		= ehci_run,
	.stop		= ehci_stop,
	.shutdown	= ehci_shutdown,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue		= ehci_urb_enqueue,
	.urb_dequeue		= ehci_urb_dequeue,
	.endpoint_disable	= ehci_endpoint_disable,
	.endpoint_reset		= ehci_endpoint_reset,

	/*
	 * scheduling support
	 */
	.get_frame_number = ehci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data	= ehci_hub_status_data,
	.hub_control		= ehci_hub_control,
	.bus_suspend		= ehci_bus_suspend,
	.bus_resume		= ehci_bus_resume,
	.relinquish_port	= ehci_relinquish_port,
	.port_handed_over	= ehci_port_handed_over,

	.clear_tt_buffer_complete = ehci_clear_tt_buffer_complete,
};

static irqreturn_t host_irq(struct ci13xxx *ci)
{
	return usb_hcd_irq(ci->irq, ci->hcd);
}

static int host_start(struct ci13xxx *ci)
{
	struct usb_hcd *hcd;
	struct ehci_hcd *ehci;
	int ret;

	if (usb_disabled())
		return -ENODEV;

	hcd = usb_create_hcd(&ci_ehci_hc_driver, ci->dev, dev_name(ci->dev));
	if (!hcd)
		return -ENOMEM;

	dev_set_drvdata(ci->dev, ci);
	hcd->rsrc_start = ci->hw_bank.phys;
	hcd->rsrc_len = ci->hw_bank.size;
	hcd->regs = ci->hw_bank.abs;
	hcd->has_tt = 1;

	hcd->power_budget = ci->platdata->power_budget;
	hcd->phy = ci->transceiver;

	ehci = hcd_to_ehci(hcd);
	ehci->caps = ci->hw_bank.cap;
	ehci->has_hostpc = ci->hw_bank.lpm;
	
#ifdef TXFIFOTUNE
	ehci->txfifo_stall = 1;
#endif

	if (ci->platdata->reg_vbus) {
		dev_info(ci->dev, "enable regulator\n");
		ret = regulator_enable(ci->platdata->reg_vbus);
		if (ret) {
			dev_err(ci->dev,
				"Failed to enable vbus regulator, ret=%d\n",
				ret);
			goto put_hcd;
		}
	}

	/* use busnum preference */
	dev_info(ci->dev, "busnumber preference is %d\n",ci->platdata->busnum);
	(hcd->self).busnum=ci->platdata->busnum;
	
	ret = usb_add_hcd(hcd, 0, 0);
	if (ret)
		goto disable_reg;
	else
		ci->hcd = hcd;
		
	return ret;
	
disable_reg:
	regulator_disable(ci->platdata->reg_vbus);

put_hcd:
	usb_put_hcd(hcd);

	return ret;
	
}

static void host_stop(struct ci13xxx *ci)
{
	struct usb_hcd *hcd = ci->hcd;

	if (hcd) {
		usb_remove_hcd(hcd);
		usb_put_hcd(hcd);
	}
	if (ci->platdata->reg_vbus) {
		dev_info(ci->dev, "disable regulator\n");
		regulator_disable(ci->platdata->reg_vbus);
	}
}

int ci_hdrc_host_init(struct ci13xxx *ci)
{
	struct ci_role_driver *rdrv;

	if (!hw_read(ci, CAP_DCCPARAMS, DCCPARAMS_HC))
		return -ENXIO;

	rdrv = devm_kzalloc(ci->dev, sizeof(struct ci_role_driver), GFP_KERNEL);
	if (!rdrv)
		return -ENOMEM;

	rdrv->start	= host_start;
	rdrv->stop	= host_stop;
	rdrv->irq	= host_irq;
	rdrv->name	= "host";
	ci->roles[CI_ROLE_HOST] = rdrv;

	return 0;
}
