#include <linux/proc_fs.h>
#include <linux/seq_file.h>

/* Macros to print into proc buffer. assumes visible struct seq_file *m */
#define P(...) seq_printf(m, __VA_ARGS__)
#define PB(reg, bitno, name) P("%s", ((reg) & (1<<(bitno)))?"(" #bitno ")" name " ":"")
#define IN_MEM(pa) ((pa) >= 0x40000000 && (pa) <= (0x40000000 + 64 * 1024 * 1024))

static int _pipe_print(struct seq_file *m, unsigned pipe)
{
	switch(usb_pipetype(pipe)) {
		case PIPE_ISOCHRONOUS : P("Z"); break;
		case PIPE_INTERRUPT   : P("I"); break;
		case PIPE_CONTROL     : P("C"); break;
		case PIPE_BULK        : P("B"); break;
		default : P("?"); break;
	}

	if (usb_pipein(pipe))
		P("i");
	else
		P("o");

	P(",%d:%d", usb_pipedevice(pipe), usb_pipeendpoint(pipe));

	return 0;
}
#define PIPE_PRINT(p) _pipe_print(m, p)

//static int _qtd_print(char *buf, const char *prefix, struct ehci_qtd *qtd)
static int _qtd_print(struct seq_file *m, const char *prefix, struct ehci_qtd *qtd)
{
	int len=0;
	unsigned tok;
	P(prefix);
	if (!qtd) {
		P("<NULL>\n");
		return len;
	}
	P("%08x, n=%08x, urb(%p,", qtd->qtd_dma, qtd->hw_next, (void*)qtd->urb);
	if (qtd->urb) PIPE_PRINT(qtd->urb->pipe); else P("      ");
	tok = qtd->hw_token;
	//P("), tok=%08x\n", qtd->hw_token);
	P("), T:%04x%s:", (tok >> 16) & 0x7FFF,
		(tok & (1<<15))?":I":"");
	if (tok & (1<<7)) P("A"); else P(".");
	if (tok & (1<<6)) P("H"); else P(".");
	if (tok & (1<<5)) P("D"); else P(".");
	if (tok & (1<<4)) P("B"); else P(".");
	if (tok & (1<<3)) P("T"); else P(".");
	if (tok & (1<<2)) P("M"); else P(".");
	if (tok & (1<<1)) P("S"); else P(".");
	if (tok & (1<<0)) P("P"); else P(".");
	P("\n");
	return len;
}
//#define QTD_PRINT(pref, qtd) len+=_qtd_print(buf+len, pref, qtd)
#define QTD_PRINT(pref, qtd) _qtd_print(m, pref, qtd)

/* Analysis and dup of QTD chain belonging to qh. */
static struct ehci_qtd * _qh_analyse(struct seq_file *m, const char *prefix, struct ehci_qh *qh)
{
	struct ehci_qtd *qtd, *qtdt;
	struct ehci_qh_hw *qhw;
	struct urb *urb;
	unsigned nextp; /* Hardware next pointer */
	unsigned old_pipe = 0;
	unsigned count = 0;
	unsigned skip = 1; /* Skip entries for identical EPs */

	qhw = qh->hw;
	nextp = 0; /* Dummy, forces printing of first QTD */

	list_for_each_entry_safe(qtd, qtdt, &qh->qtd_list, qtd_list) {
		urb = qtd->urb;

		if (nextp && nextp != qtd->qtd_dma) {
			/* Broken chain */
			P("%sQTD!", prefix);
		} else if (qtd->qtd_dma == qhw->hw_current) {
			P("%sQTD*", prefix);
		} else {
			/* SkipRepeats */
			if (skip && old_pipe == urb->pipe) {
				nextp = qtd->hw_next;
				count++;
				continue;
			}
			if (count) {
				P("%sQTD %d more chain entries for EP ", prefix, count);
				PIPE_PRINT(old_pipe);
				P("\n");
				count = 0;
			}
			old_pipe = urb->pipe;
			P("%sQTD ", prefix);
		}

		QTD_PRINT("", qtd);
		nextp = qtd->hw_next;
	}

	if (count) {
		P("%sQTD %d more chain entries for EP ", prefix, count);
		PIPE_PRINT(old_pipe);
		P("\n");
	}

	P(prefix);
	QTD_PRINT("DTD ", qh->dummy);
	return NULL;
}
#define QH_ANALYSE(pref, qtd) _qh_analyse(m, pref, qtd)

static int proc_show(struct seq_file *m, void *v)
//static int ehci_proc(char *buf, char **start, off_t off,int count, int *eof, void *data)
{
	unsigned i, j;
	int found;
	unsigned long flags;

	struct usb_hcd *hcd = (struct usb_hcd *)m->private;
	struct ehci_hcd *ehci = hcd_to_ehci(hcd); /* priv field in usb_hcd */
	u32 cmd, status, intr, frindex, perbase, asyncaddr, usbmode, txfilltuning;

	struct ehci_qh *qh, *qht;
	struct ehci_qh_hw *qhw;

	spin_lock_irqsave(&ehci->lock, flags);

	cmd = ehci_readl(ehci, &ehci->regs->command);
	status = ehci_readl(ehci, &ehci->regs->status);
	intr = ehci_readl(ehci, &ehci->regs->intr_enable);
	frindex = ehci_readl(ehci, &ehci->regs->frame_index);
	perbase = ehci_readl(ehci, &ehci->regs->frame_list);
	asyncaddr = ehci_readl(ehci, &ehci->regs->async_next);
	usbmode = ehci_readl(ehci, &ehci->regs->usbmode);
	txfilltuning = ehci_readl(ehci, &ehci->regs->txfill_tuning);

	P("HCD: %p\n", (void*)hcd);

	P("      cmd: %08x ", cmd);
		PB(cmd,5,"ASE");
		PB(cmd,4,"PSE");

	P("\n");

	P("   status: %08x ", status);
		PB(status,25,"TI1");
		PB(status,24,"TI0");
		PB(status,19,"UPI");
		PB(status,18,"UAI");
		PB(status,16,"NAK");
		PB(status,15,"AS");
		PB(status,14,"PS");
		PB(status,13,"RCL");
		PB(status,12,"HCH");
		PB(status,9,"SLI");
		PB(status,7,"SRI");
		PB(status,6,"URI");
		PB(status,5,"AAI");
		PB(status,4,"SEI");
		PB(status,3,"FRI");
		PB(status,2,"PCI");
		PB(status,1,"UEI");
		PB(status,0,"UI");
	P("\n");

	P("     intr: %08x ", intr);
		PB(intr,5,"AAI");
		PB(intr,4,"SEI");
		PB(intr,3,"FRI");
		PB(intr,2,"PCI");
		PB(intr,1,"UEI");
		PB(intr,0,"UI");
	P("\n");

	P("  frindex=%08x, perbase=%08x, asyncaddr=%08x\n",
		frindex, perbase, asyncaddr);

	P("  usbmode=%08x, txfilltuning=%08x\n", usbmode, txfilltuning);

	/* Here we reset the schedule error counter in txfilltuning. The
	 * value printed out should be less than 10/sec on a busy bus. */
	ehci_writel(ehci, txfilltuning, &ehci->regs->txfill_tuning);

	P("\n");

	P("STATS: In=%lu, Ie=%lu, Ia=%lu, Ila=%lu, urb_complete=%lu, urb_unlink=%lu, urbs=%d, suspended=%u\n",
		ehci->stats.normal, ehci->stats.error,
		ehci->stats.iaa, ehci->stats.lost_iaa,
		ehci->stats.complete, ehci->stats.unlink,
		AGET(ehci->stats.e.urbs),hcd->async_suspended);

	P("\n");

	/* Periodic schedule */
	P("periodic(%u, LOG %08x, DMA %08x):\n",
		ehci->periodic_size, (unsigned)ehci->periodic, ehci->periodic_dma);
	{
		unsigned reps;
		static unsigned per_copy[1024]; /* Max periodic table size */

		/* We modify this copy in place */
		memcpy(per_copy, ehci->periodic, ehci->periodic_size * 4);

		for (i = 0; i < ehci->periodic_size; i++) {
			/* Idle entries have T-Bit set */
			if (per_copy[i] & 0x1) continue;
			/* Take out identical schedule entries, avoid cluttering. */
			reps=1;
			for (j = i + 1; j < ehci->periodic_size; j++) {
				if (per_copy[j] != per_copy[i]) continue;
				if (!(per_copy[j] & 0x1)) reps++;
				per_copy[j] = 0x1;
			}
			P("	[%3u] rep%-3u %08x ", i, reps, per_copy[i] & ~0x1F);
			/* Bits should be 0 */
			if ((per_copy[i] >> 3) & 0x3) P("!");
			/* periodic descriptor type */
			switch ((per_copy[i] >> 1) & 0x3) {
				case 0 : P("(ISO)\n"); continue;
				case 1 : /* continuing below */ break;
				case 2 : P("(SPLIT)\n"); continue;
				case 3 : P("(FSTN)\n"); continue;
			}
			/* Search the corresponding QH */
			found=0;
			list_for_each_entry_safe(qh, qht, &ehci->intr_qh_list, intr_node) {
				if (qh->qh_dma != (per_copy[i] & ~0x1F)) continue;
				qhw =qh->hw;
				P("QH %08x, n=%08x, qtd=%08x\n",
					qh->qh_dma, qhw->hw_next & ~0x1F, qhw->hw_current);
				QH_ANALYSE("		", qh);
				found=1;
				break;
			}
			if (!found) P(" No QH found!\n");
		}
	}

	P("\n");

	/* Async schedule */
	qh = ehci->async;
	P("async(%p)\n", qh);
	found = 0;
	while (qh) {
		qhw =qh->hw;
		if (qh->qh_dma == asyncaddr) {
			P("	QH*");
			found++;
		} else {
			P("	QH ");
		}
		P("%08x, n=%08x, info1=%08x, info2=%08x, qtd=%08x\n", qh->qh_dma, qhw->hw_next & ~0x1F, qhw->hw_info1, qhw->hw_info2, qhw->hw_current);
		QH_ANALYSE("		", qh);
		qh=qh->qh_next.qh;
	}
	if (found != 1) {
		P("	BAD Async pointer!\n");
	}

	P("\n");

	spin_unlock_irqrestore(&ehci->lock, flags);

	{
		unsigned h;
		h = lockup_heuristics(hcd, NULL, 1);
		P("Heuristics: %08x %s %lu\n", h, decode_heuristics(h),
			ehci->stats.e.jiffies);
	}

	return 0;
}

/*
static int proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "Hello %p\n", v);
	seq_printf(m, "World %p\n", v);
	seq_printf(m, "Ptr %p\n", m->private);
	return 0;
}
*/

static int prc_open(struct inode *inode, struct file *file)
{
	struct proc_dir_entry* pde = PDE( inode );
	if(pde->data)
	{
		return single_open(file, proc_show,pde->data);
	}
	return -1;
}




static const struct file_operations prc_fops =
	{
		.owner     = THIS_MODULE,
		.open      = prc_open,
		.read      = seq_read,
		.llseek    = seq_lseek,
		.release   = single_release,
	};
#define MAX_USB_BUS 64
#define MAX_EHCI_NAME 20

struct proc_dir_entry  *usb_proc_entry[MAX_USB_BUS]={0};



static void create_ehci_debug_proc(struct  usb_hcd  *hcd)
{
	if( hcd->self.busnum < MAX_USB_BUS && usb_proc_entry[hcd->self.busnum] ==NULL)
	{
		char name[MAX_EHCI_NAME]="";
		sprintf(name, "EHCI_%d", hcd->self.busnum);
		dev_info(hcd->self.controller, "create_ehci_proc %s for device:%s\n", name, dev_name(hcd->self.controller));

		usb_proc_entry[hcd->self.busnum] = proc_create_data(name, 0, NULL, &prc_fops, hcd);
	}
}
static void delete_ehci_debug_proc(struct usb_hcd *hcd)
{
	if( hcd->self.busnum < MAX_USB_BUS && usb_proc_entry[hcd->self.busnum])
	{
		char name[MAX_EHCI_NAME]="";
		sprintf(name, "EHCI_%d", hcd->self.busnum);

		dev_info(hcd->self.controller, "delete_ehci_proc: %s for %s\n", name, dev_name(hcd->self.controller));
		remove_proc_entry(name, NULL);
		usb_proc_entry[hcd->self.busnum] =NULL;
	}

}
