/*
 * core.c - ChipIdea USB IP core family device controller
 *
 * Copyright (C) 2008 Chipidea - MIPS Technologies, Inc. All rights reserved.
 *
 * Author: David Lopo
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/*
 * Description: ChipIdea USB IP core family device controller
 *
 * This driver is composed of several blocks:
 * - HW:     hardware interface
 * - DBG:    debug facilities (optional)
 * - UTIL:   utilities
 * - ISR:    interrupts handling
 * - ENDPT:  endpoint operations (Gadget API)
 * - GADGET: gadget operations (Gadget API)
 * - BUS:    bus glue code, bus abstraction layer
 *
 * Compile Options
 * - CONFIG_USB_GADGET_DEBUG_FILES: enable debug facilities
 * - STALL_IN:  non-empty bulk-in pipes cannot be halted
 *              if defined mass storage compliance succeeds but with warnings
 *              => case 4: Hi >  Dn
 *              => case 5: Hi >  Di
 *              => case 8: Hi <> Do
 *              if undefined usbtest 13 fails
 * - TRACE:     enable function tracing (depends on DEBUG)
 *
 * Main Features
 * - Chapter 9 & Mass Storage Compliance with Gadget File Storage
 * - Chapter 9 Compliance with Gadget Zero (STALL_IN undefined)
 * - Normal & LPM support
 *
 * USBTEST Report
 * - OK: 0-12, 13 (STALL_IN defined) & 14
 * - Not Supported: 15 & 16 (ISO)
 *
 * TODO List
 * - OTG
 * - Isochronous & Interrupt Traffic
 * - Handle requests which spawns into several TDs
 * - GET_STATUS(device) - always reports 0
 * - Gadget API (majority of optional features)
 * - Suspend & Remote Wakeup
 */
 
#define CONFIG_PRINTK
 
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/dmapool.h>
#include <linux/dma-mapping.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/idr.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/pm_runtime.h>
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/usb/otg.h>
#include <linux/usb/chipidea.h>
#include <linux/regulator/consumer.h>
#include <linux/of.h>

#include "ci.h"
#include "udc.h"
#include "bits.h"
#include "host.h"
#include "debug.h"

#include "otg.h"
#include <linux/cdcs_variant.h>

#define DEBUG_OTG_SWITCH

enum usb_dr_mode of_usb_get_dr_mode(struct device_node *np);

/* Controller register map */
static uintptr_t ci_regs_nolpm[] = {
	[CAP_CAPLENGTH]		= 0x000UL,
	[CAP_HCCPARAMS]		= 0x008UL,
	[CAP_DCCPARAMS]		= 0x024UL,
	[CAP_TESTMODE]		= 0x038UL,
	[OP_USBCMD]		= 0x000UL,
	[OP_USBSTS]		= 0x004UL,
	[OP_USBINTR]		= 0x008UL,
	[OP_DEVICEADDR]		= 0x014UL,
	[OP_ENDPTLISTADDR]	= 0x018UL,
	[OP_PORTSC]		= 0x044UL,
	[OP_DEVLC]		= 0x084UL,
	[OP_OTGSC]		= 0x064UL,
	[OP_USBMODE]		= 0x068UL,
	[OP_ENDPTSETUPSTAT]	= 0x06CUL,
	[OP_ENDPTPRIME]		= 0x070UL,
	[OP_ENDPTFLUSH]		= 0x074UL,
	[OP_ENDPTSTAT]		= 0x078UL,
	[OP_ENDPTCOMPLETE]	= 0x07CUL,
	[OP_ENDPTCTRL]		= 0x080UL,
};

static uintptr_t ci_regs_lpm[] = {
	[CAP_CAPLENGTH]		= 0x000UL,
	[CAP_HCCPARAMS]		= 0x008UL,
	[CAP_DCCPARAMS]		= 0x024UL,
	[CAP_TESTMODE]		= 0x0FCUL,
	[OP_USBCMD]		= 0x000UL,
	[OP_USBSTS]		= 0x004UL,
	[OP_USBINTR]		= 0x008UL,
	[OP_DEVICEADDR]		= 0x014UL,
	[OP_ENDPTLISTADDR]	= 0x018UL,
	[OP_PORTSC]		= 0x044UL,
	[OP_DEVLC]		= 0x084UL,
	[OP_OTGSC]		= 0x0C4UL,
	[OP_USBMODE]		= 0x0C8UL,
	[OP_ENDPTSETUPSTAT]	= 0x0D8UL,
	[OP_ENDPTPRIME]		= 0x0DCUL,
	[OP_ENDPTFLUSH]		= 0x0E0UL,
	[OP_ENDPTSTAT]		= 0x0E4UL,
	[OP_ENDPTCOMPLETE]	= 0x0E8UL,
	[OP_ENDPTCTRL]		= 0x0ECUL,
};

static int hw_alloc_regmap(struct ci13xxx *ci, bool is_lpm)
{
	int i;

	kfree(ci->hw_bank.regmap);

	ci->hw_bank.regmap = kzalloc((OP_LAST + 1) * sizeof(void *),
				     GFP_KERNEL);
	if (!ci->hw_bank.regmap)
		return -ENOMEM;

	for (i = 0; i < OP_ENDPTCTRL; i++)
		ci->hw_bank.regmap[i] =
			(i <= CAP_LAST ? ci->hw_bank.cap : ci->hw_bank.op) +
			(is_lpm ? ci_regs_lpm[i] : ci_regs_nolpm[i]);

	for (; i <= OP_LAST; i++)
		ci->hw_bank.regmap[i] = ci->hw_bank.op +
			4 * (i - OP_ENDPTCTRL) +
			(is_lpm
			 ? ci_regs_lpm[OP_ENDPTCTRL]
			 : ci_regs_nolpm[OP_ENDPTCTRL]);

	return 0;
}

/**
 * hw_port_test_set: writes port test mode (execute without interruption)
 * @mode: new value
 *
 * This function returns an error code
 */
int hw_port_test_set(struct ci13xxx *ci, u8 mode)
{
	const u8 TEST_MODE_MAX = 7;

	if (mode > TEST_MODE_MAX)
		return -EINVAL;

	hw_write(ci, OP_PORTSC, PORTSC_PTC, mode << ffs_nr(PORTSC_PTC));
	return 0;
}

/**
 * hw_port_test_get: reads port test mode value
 *
 * This function returns port test mode value
 */
u8 hw_port_test_get(struct ci13xxx *ci)
{
	return hw_read(ci, OP_PORTSC, PORTSC_PTC) >> ffs_nr(PORTSC_PTC);
}

static int hw_device_init(struct ci13xxx *ci, void __iomem *base)
{
	u32 reg;

	/* bank is a module variable */
	ci->hw_bank.abs = base;

	ci->hw_bank.cap = ci->hw_bank.abs;
	ci->hw_bank.cap += ci->platdata->capoffset;
	ci->hw_bank.op = ci->hw_bank.cap + ioread8(ci->hw_bank.cap);

	hw_alloc_regmap(ci, false);
	reg = hw_read(ci, CAP_HCCPARAMS, HCCPARAMS_LEN) >>
		ffs_nr(HCCPARAMS_LEN);
	ci->hw_bank.lpm  = reg;
	hw_alloc_regmap(ci, !!reg);
	ci->hw_bank.size = ci->hw_bank.op - ci->hw_bank.abs;
	ci->hw_bank.size += OP_LAST;
	ci->hw_bank.size /= sizeof(u32);

	reg = hw_read(ci, CAP_DCCPARAMS, DCCPARAMS_DEN) >>
		ffs_nr(DCCPARAMS_DEN);
	ci->hw_ep_max = reg * 2;   /* cache hw ENDPT_MAX */

	if (ci->hw_ep_max > ENDPT_MAX)
		return -ENODEV;

	dev_dbg(ci->dev, "ChipIdea HDRC found, lpm: %d; cap: %p op: %p\n",
		ci->hw_bank.lpm, ci->hw_bank.cap, ci->hw_bank.op);

	/* setup lock mode ? */

	/* ENDPTSETUPSTAT is '0' by default */

	/* HCSPARAMS.bf.ppc SHOULD BE zero for device */

	return 0;
}

/**
 * hw_device_reset: resets chip (execute without interruption)
 * @ci: the controller
  *
 * This function returns an error code
 */
int hw_device_reset(struct ci13xxx *ci, u32 mode)
{
	/* should flush & stop before reset */
	hw_write(ci, OP_ENDPTFLUSH, ~0, ~0);
	hw_write(ci, OP_USBCMD, USBCMD_RS, 0);

	hw_write(ci, OP_USBCMD, USBCMD_RST, USBCMD_RST);
	while (hw_read(ci, OP_USBCMD, USBCMD_RST))
		udelay(10);		/* not RTOS friendly */


	if (ci->platdata->notify_event)
		ci->platdata->notify_event(ci,
			CI13XXX_CONTROLLER_RESET_EVENT);

	if (ci->platdata->flags & CI13XXX_DISABLE_STREAMING)
		hw_write(ci, OP_USBMODE, USBMODE_CI_SDIS, USBMODE_CI_SDIS);

	/* USBMODE should be configured step by step */
	hw_write(ci, OP_USBMODE, USBMODE_CM, USBMODE_CM_IDLE);
	hw_write(ci, OP_USBMODE, USBMODE_CM, mode);
	/* HW >= 2.3 */
	hw_write(ci, OP_USBMODE, USBMODE_SLOM, USBMODE_SLOM);

	if (hw_read(ci, OP_USBMODE, USBMODE_CM) != mode) {
		pr_err("cannot enter in %s mode", ci_role(ci)->name);
		pr_err("lpm = %i", ci->hw_bank.lpm);
		return -ENODEV;
	}

	return 0;
}

/**
 * hw_wait_reg: wait the register value
 *
 * Sometimes, it needs to wait register value before going on.
 * Eg, when switch to device mode, the vbus value should be lower
 * than OTGSC_BSV before connects to host.
 *
 * @ci: the controller
 * @reg: register index
 * @mask: mast bit
 * @value: the bit value to wait
 * @timeout_ms: timeout in millisecond
 *
 * This function returns an error code if timeout
 */
int hw_wait_reg(struct ci13xxx *ci, enum ci13xxx_regs reg, u32 mask,
				u32 value, unsigned int timeout_ms)
{
	unsigned long elapse = jiffies + msecs_to_jiffies(timeout_ms);

	while (hw_read(ci, reg, mask) != value) {
		if (time_after(jiffies, elapse)) {
			dev_err(ci->dev, "timeout waiting for %08x in %d\n",
					mask, reg);
			return -ETIMEDOUT;
		}
		msleep(20);
	}

	return 0;
}

/**
 * ci_otg_role - pick role based on ID pin state
 * @ci: the controller
 */
static enum ci_role ci_otg_role(struct ci13xxx *ci)
{
	u32 sts = hw_read(ci, OP_OTGSC, ~0);
	
	enum ci_role role = sts & OTGSC_ID
		? CI_ROLE_GADGET
		: CI_ROLE_HOST;

	return role;
}

void ci_handle_vbus_change(struct ci13xxx *ci)
{
	u32 otgsc;
#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	/* Gadget mode is always interested in B Session Valid event */
	if (ci->role != CI_ROLE_GADGET) {
		return;
	}
#else
	if (!ci->is_otg)
		return;
#endif

	otgsc = hw_read(ci, OP_OTGSC, ~0);

	if (otgsc & OTGSC_BSV)
		usb_gadget_vbus_connect(&ci->gadget);
	else
		usb_gadget_vbus_disconnect(&ci->gadget);
}


#define CI_VBUS_STABLE_TIMEOUT_MS 5000
static void ci_handle_id_switch(struct ci13xxx *ci)
{
	enum ci_role role = ci_otg_role(ci);
#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	/* Switching mode is only considered in OTG mode */
	if (!ci->is_otg) {
		return;
	}
#endif
	if (role != ci->role) {
		dev_dbg(ci->dev, "switching from %s to %s\n",
			ci_role(ci)->name, ci->roles[role]->name);

		ci_role_stop(ci);
		/* wait vbus lower than OTGSC_BSV */
		hw_wait_reg(ci, OP_OTGSC, OTGSC_BSV, 0,
				CI_VBUS_STABLE_TIMEOUT_MS);
		ci_role_start(ci, role);
	}
}

/**
 * ci_role_work - perform role changing based on ID pin
 * @work: work struct
 */
static void ci_role_work(struct work_struct *work)
{
	struct ci13xxx *ci = container_of(work, struct ci13xxx, work);

	#ifdef DEBUG_OTG_SWITCH
	printk("chipidea: workqueue - id=%d,session=%d\n",ci->id_event,ci->b_sess_valid_event);
	#endif

	if (ci->id_event) {
		ci->id_event = false;
		ci_handle_id_switch(ci);
	} else if (ci->b_sess_valid_event) {
		ci->b_sess_valid_event = false;
		ci_handle_vbus_change(ci);
	} else
		dev_err(ci->dev, "unexpected event occurs at %s\n", __func__);

	/* in case, USB ID interrupt is gone missing */
	ci_handle_id_switch(ci);
	
	enable_irq(ci->irq);
}

#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
#define CI_ROLE_OTG_NAME "otg"
#define CI_ROLE_DOWN_NAME "down"
#endif

static ssize_t show_role(struct device *dev, struct device_attribute *attr,
			 char *buf)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);
	const char* name;
#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	switch (ci->role) {
		case CI_ROLE_GADGET:
		case CI_ROLE_HOST:
			if (ci->is_otg) {
				if (ci->role == CI_ROLE_GADGET) {
					name = "otg_gadget";
				}
				else {
					name = "otg_host";
				}
			}
			else {
				name=ci->roles[(ci->role==CI_ROLE_GADGET)?CI_ROLE_GADGET:CI_ROLE_HOST]->name;
			}
			break;
		case CI_ROLE_END:
			name = CI_ROLE_DOWN_NAME;
			break;
		default:
			name = "unknown";
			break;
	}
#else
	/* workaround for CI_ROLE_END BUG check */
	name=ci->roles[(ci->role==CI_ROLE_GADGET)?CI_ROLE_GADGET:CI_ROLE_HOST]->name;
#endif
	
	return sprintf(buf, "%s\n", name);
}

/*
 * Disable IDI and BSV interrupts
 */
static void disable_idi_bsv_interrupt(struct ci13xxx *ci)
{
	ci_disable_otg_interrupt(ci, OTGSC_IDIE);
	ci_disable_otg_interrupt(ci, OTGSC_BSVIE);
	ci_clear_otg_interrupt(ci, OTGSC_IDIS);
	ci_clear_otg_interrupt(ci, OTGSC_BSVIS);
}

static ssize_t store_role(struct device *dev, struct device_attribute *attr,
			  const char *buf, size_t count)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);
	enum ci_role role;
	int ret;
#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	spin_lock(&ci->switching_mode_lock);
	if (!strcmp(buf, ci->roles[CI_ROLE_HOST]->name)) {
		disable_idi_bsv_interrupt(ci);
		if (ci->role != CI_ROLE_END) {
			disable_irq(ci->irq);
		}
		/* ci_role_stop does nothing if current role == CI_ROLE_END.
		 * Otherwise it sets role = CI_ROLE_END then invoke current role's stop function.
		 * Hence no need to check whether current role is off before calling this function.
		 */
		ci_role_stop(ci);
		ci->is_otg = false;
		ci->role = CI_ROLE_HOST;
		ret = ci_role_start(ci, ci->role);
		if (ret) {
			ci->role = CI_ROLE_END;
		}
		else {
			enable_irq(ci->irq);
		}
	}
	else if (!strcmp(buf, ci->roles[CI_ROLE_GADGET]->name)) {
		disable_idi_bsv_interrupt(ci);
		if (ci->role != CI_ROLE_END) {
			disable_irq(ci->irq);
		}
		/* no need to check whether current role is off before calling this function. */
		ci_role_stop(ci);
		ci->is_otg = false;
		ci->role = CI_ROLE_GADGET;
		ret = ci_role_start(ci, ci->role);
		if (ret) {
			ci->role = CI_ROLE_END;
		}
		else {
			enable_irq(ci->irq);
			ci_enable_otg_interrupt(ci, OTGSC_BSVIE);
			ci_enable_otg_interrupt(ci, OTGSC_IDIE);
		}
	}
	else if (!strcmp(buf, CI_ROLE_OTG_NAME)) {
		disable_idi_bsv_interrupt(ci);
		if (ci->role != CI_ROLE_END) {
			disable_irq(ci->irq);
		}
		/* no need to check whether current role is off before calling this function. */
		ci_role_stop(ci);
		ci->is_otg = true;
		ci->role = ci_otg_role(ci);
		ret = ci_role_start(ci, ci->role);
		if (ret) {
			ci->role = CI_ROLE_END;
			ci->is_otg = false;
		}
		else {
			enable_irq(ci->irq);
			ci_enable_otg_interrupt(ci, OTGSC_IDIE);
		}
	}
	else if (!strcmp(buf, CI_ROLE_DOWN_NAME)) {
		disable_idi_bsv_interrupt(ci);
		if (ci->role != CI_ROLE_END) {
			disable_irq(ci->irq);
		}
		/* no need to check whether current role is off before calling this function. */
		ci_role_stop(ci);
		ci->is_otg = false;
		ci->role = CI_ROLE_END;
	}
	spin_unlock(&ci->switching_mode_lock);
	if (ret) {
		return ret;
	}
#else
	for (role = CI_ROLE_HOST; role < CI_ROLE_END; role++)
		if (ci->roles[role] && !strcmp(buf, ci->roles[role]->name))
			break;

	if (role == CI_ROLE_END || role == ci->role)
		return -EINVAL;

	/* Before ci_role_start completes, the irq may not be handled */
	disable_irq(ci->irq);
	ci_role_stop(ci);
	ret = ci_role_start(ci, role);
	enable_irq(ci->irq);

	if (ret)
		return ret;
#endif

	return count;
}

static DEVICE_ATTR(role, S_IRUSR | S_IWUSR, show_role, store_role);

/* NTC: connection state */
static ssize_t show_connection_state(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct ci13xxx *ci = dev_get_drvdata(dev);
	if (ci && ci->transceiver){
		return sprintf(buf, "%s\n", ci->transceiver->connected ? "connected" : "disconnected");
	}

	return sprintf(buf, "not applicable\n");
}
static DEVICE_ATTR(connection, S_IRUSR, show_connection_state, NULL);

static irqreturn_t ci_irq(int irq, void *data)
{
	struct ci13xxx *ci = data;
	irqreturn_t ret = IRQ_NONE;
	u32 otgsc = 0;
	bool need_checking_otgsc;

#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	need_checking_otgsc = (ci->is_otg || ci->role == CI_ROLE_GADGET);
#else
	need_checking_otgsc = ci->is_otg;
#endif

	if (need_checking_otgsc) {
		otgsc = hw_read(ci, OP_OTGSC, ~0);
	}

	/*
	 * Handle id change interrupt, it indicates device/host function
	 * switch.
	 */
	if (need_checking_otgsc && (otgsc & OTGSC_IDIE) && (otgsc & OTGSC_IDIS)) {
		#ifdef DEBUG_OTG_SWITCH
		printk("chipidea: ID interrupt - otgsc=0x%08x (idie=%d,idis=%d)\n",otgsc,(otgsc & OTGSC_IDIE)!=0,(otgsc & OTGSC_IDIS)!=0);
		#endif
		ci->id_event = true;
		ci_clear_otg_interrupt(ci, OTGSC_IDIS);
		disable_irq_nosync(ci->irq);
		queue_work(ci->wq, &ci->work);
		return IRQ_HANDLED;
	}

	/*
	 * Handle vbus change interrupt, it indicates device connection
	 * and disconnection events.
	 */
	if (need_checking_otgsc && (otgsc & OTGSC_BSVIE) && (otgsc & OTGSC_BSVIS)) {

		#ifdef DEBUG_OTG_SWITCH
		printk("chipidea: session interrupt - otgsc=0x%08x\n",otgsc);
		#endif

		ci->b_sess_valid_event = true;
		ci_clear_otg_interrupt(ci, OTGSC_BSVIS);
		disable_irq_nosync(ci->irq);
		queue_work(ci->wq, &ci->work);
		return IRQ_HANDLED;
	}

	/* Handle device/host interrupt */
	if (ci->role != CI_ROLE_END)
		ret = ci_role(ci)->irq(ci);

	return ret;
}

static int ci_get_platdata(struct device *dev,
		struct ci13xxx_platform_data *platdata)
{
	/* Get the vbus regulator */
	platdata->reg_vbus = devm_regulator_get(dev, "vbus");
	if (PTR_ERR(platdata->reg_vbus) == -EPROBE_DEFER) {
		return -EPROBE_DEFER;
	} else if (PTR_ERR(platdata->reg_vbus) == -ENODEV) {
		platdata->reg_vbus = NULL; /* no vbus regualator is needed */
	} else if (IS_ERR(platdata->reg_vbus)) {
		dev_err(dev, "Getting regulator error: %ld\n",
			PTR_ERR(platdata->reg_vbus));
		return PTR_ERR(platdata->reg_vbus);
	}

	return 0;
}

static DEFINE_IDA(ci_ida);

struct platform_device *ci13xxx_add_device(struct device *dev,
			struct resource *res, int nres,
			struct ci13xxx_platform_data *platdata)
{
	struct platform_device *pdev;
	int id, ret;

	ret = ci_get_platdata(dev, platdata);
	if (ret)
		return ERR_PTR(ret);
		
	if(platdata->reg_vbus) {
		dev_info(dev, "regulator found\n");
	}
	else {
		dev_info(dev, "regulator not found\n");
	}		

	id = ida_simple_get(&ci_ida, 0, 0, GFP_KERNEL);
	if (id < 0)
		return ERR_PTR(id);

	pdev = platform_device_alloc("ci_hdrc", id);
	if (!pdev) {
		ret = -ENOMEM;
		goto put_id;
	}

	pdev->dev.parent = dev;
	pdev->dev.dma_mask = dev->dma_mask;
	pdev->dev.dma_parms = dev->dma_parms;
	dma_set_coherent_mask(&pdev->dev, dev->coherent_dma_mask);

	ret = platform_device_add_resources(pdev, res, nres);
	if (ret)
		goto err;

	ret = platform_device_add_data(pdev, platdata, sizeof(*platdata));
	if (ret)
		goto err;

	ret = platform_device_add(pdev);
	if (ret)
		goto err;

	return pdev;

err:
	platform_device_put(pdev);
put_id:
	ida_simple_remove(&ci_ida, id);
	return ERR_PTR(ret);
}
EXPORT_SYMBOL_GPL(ci13xxx_add_device);

void ci13xxx_remove_device(struct platform_device *pdev)
{
	platform_device_unregister(pdev);
	ida_simple_remove(&ci_ida, pdev->id);
}
EXPORT_SYMBOL_GPL(ci13xxx_remove_device);

int of_usb_get_bus_num(struct device_node *np)
{
	int busnum;
	int err;

	err = of_property_read_u32(np, "busnum", &busnum);
	if (err < 0)
		return 0;

	return busnum;
}

static int __devinit ci_hdrc_probe(struct platform_device *pdev)
{
	struct device	*dev = &pdev->dev;
	struct ci13xxx	*ci;
	struct resource	*res;
	void __iomem	*base;
	int		ret;
	enum usb_dr_mode dr_mode;
	struct device_node *of_node = dev->of_node ?: dev->parent->of_node;
	
	if (!dev->platform_data) {
		dev_err(dev, "platform data missing\n");
		return -ENODEV;
	}

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(dev, "missing resource\n");
		return -ENODEV;
	}

	base = devm_request_and_ioremap(dev, res);
	if (!res) {
		dev_err(dev, "can't request and ioremap resource\n");
		return -ENOMEM;
	}

	ci = devm_kzalloc(dev, sizeof(*ci), GFP_KERNEL);
	if (!ci) {
		dev_err(dev, "can't allocate device\n");
		return -ENOMEM;
	}

	ci->dev = dev;
	ci->platdata = dev->platform_data;
	if (ci->platdata->phy)
		ci->transceiver = ci->platdata->phy;
	else
		ci->global_phy = true;

#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	spin_lock_init(&ci->switching_mode_lock);
#endif

	ret = hw_device_init(ci, base);
	if (ret < 0) {
		dev_err(dev, "can't initialize hardware\n");
		return -ENODEV;
	}

	ci->hw_bank.phys = res->start;

	ci->irq = platform_get_irq(pdev, 0);
	if (ci->irq < 0) {
		dev_err(dev, "missing IRQ\n");
		return -ENODEV;
	}
	
	INIT_WORK(&ci->work, ci_role_work);
	ci->wq = create_singlethread_workqueue("ci_otg");
	if (!ci->wq) {
		dev_err(dev, "can't create workqueue\n");
		return -ENODEV;
	}

	if (!ci->platdata->dr_mode)
		ci->platdata->dr_mode = of_usb_get_dr_mode(of_node);
		
	if (!ci->platdata->busnum)
		ci->platdata->busnum = of_usb_get_bus_num(of_node);

	if (ci->platdata->dr_mode == USB_DR_MODE_UNKNOWN)
		ci->platdata->dr_mode = USB_DR_MODE_OTG;

	dr_mode = ci->platdata->dr_mode;
	
	/* initialize role(s) before the interrupt is requested */
	if (dr_mode == USB_DR_MODE_OTG || dr_mode == USB_DR_MODE_HOST) {
		ret = ci_hdrc_host_init(ci);
		if (ret)
			dev_info(dev, "doesn't support host\n");
	}

	if (dr_mode == USB_DR_MODE_OTG || dr_mode == USB_DR_MODE_PERIPHERAL) {
		if (ci->transceiver){
			usb_add_phy(ci->transceiver, USB_PHY_TYPE_USB2);
		}
		ret = ci_hdrc_gadget_init(ci);
		if (ret)
			dev_info(dev, "doesn't support gadget\n");
	}

	if (!ci->roles[CI_ROLE_HOST] && !ci->roles[CI_ROLE_GADGET]) {
		dev_err(dev, "no supported roles\n");
		ret = -ENODEV;
		goto rm_wq;
	}

	if (ci->roles[CI_ROLE_HOST] && ci->roles[CI_ROLE_GADGET]) {
#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
		/*
		 * Initial state is DOWN and actual functions will be
		 * activated later by writing to the "role" sysfs.
		 */
		ci->is_otg = false;
		ci->role = CI_ROLE_END;
#else
		ci->is_otg = true;
		ci->role = ci_otg_role(ci);
#endif
	} else {
		ci->role = ci->roles[CI_ROLE_HOST]
			? CI_ROLE_HOST
			: CI_ROLE_GADGET;
	}

	if (ci->role != CI_ROLE_END) {
		ret = ci_role_start(ci, ci->role);
		if (ret) {
			dev_err(dev, "can't start %s role\n", ci_role(ci)->name);
			ret = -ENODEV;
			goto rm_wq;
		}
	}

	platform_set_drvdata(pdev, ci);
	ret = request_irq(ci->irq, ci_irq, IRQF_SHARED, ci->platdata->name,
			  ci);
	if (ret)
		goto stop;

#ifdef KV_USB_OTG_MANUAL_MODE_SELECTION_y
	if (ci->role == CI_ROLE_END) {
		disable_irq(ci->irq);
	}
#endif

	ret = device_create_file(dev, &dev_attr_role);
	if (ret)
		goto rm_attr;

	ret = device_create_file(dev, &dev_attr_connection);
	if (ret) {
		goto rm_attr_connection;
	}

	if (ci->is_otg) {
		hw_write(ci, OP_OTGSC, OTGSC_IDIE, OTGSC_IDIE);
	}

	return ret;

rm_attr_connection:
	device_remove_file(dev, &dev_attr_connection);
rm_attr:
	device_remove_file(dev, &dev_attr_role);
stop:
	ci_role_stop(ci);
rm_wq:
	flush_workqueue(ci->wq);
	destroy_workqueue(ci->wq);

	return ret;
}

static int __devexit ci_hdrc_remove(struct platform_device *pdev)
{
	struct ci13xxx *ci = platform_get_drvdata(pdev);

	flush_workqueue(ci->wq);
	destroy_workqueue(ci->wq);
	device_remove_file(ci->dev, &dev_attr_role);
	device_remove_file(ci->dev, &dev_attr_connection);
	free_irq(ci->irq, ci);
	if (ci->transceiver &&
			(ci->platdata->dr_mode == USB_DR_MODE_OTG || ci->platdata->dr_mode == USB_DR_MODE_PERIPHERAL)){
		usb_remove_phy(ci->transceiver);
	}
	ci_role_stop(ci);

	return 0;
}

static struct platform_driver ci_hdrc_driver = {
	.probe	= ci_hdrc_probe,
	.remove	= __devexit_p(ci_hdrc_remove),
	.driver	= {
		.name	= "ci_hdrc",
	},
};

module_platform_driver(ci_hdrc_driver);

MODULE_ALIAS("platform:ci_hdrc");
MODULE_ALIAS("platform:ci13xxx");
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("David Lopo <dlopo@chipidea.mips.com>");
MODULE_DESCRIPTION("ChipIdea HDRC Driver");
