/*
 * Freescale i.MX233/i.MX28 LRADC driver
 *
 * Copyright (c) 2012 DENX Software Engineering, GmbH.
 * Marek Vasut <marex@...>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/sysfs.h>
#include <linux/list.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/spinlock.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/stmp_device.h>
#include <linux/bitops.h>
#include <linux/completion.h>

#include <mach/mxs.h>
#include <mach/common.h>

#include <linux/iio/iio.h>
#include <linux/iio/buffer.h>
#include <linux/iio/trigger.h>
#include <linux/iio/trigger_consumer.h>
#include <linux/iio/triggered_buffer.h>

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
#include <mach/regs-lradc.h>
#include <linux/jiffies.h>
#include <linux/delay.h>
#include <linux/notifier.h>
#endif


#define DRIVER_NAME		"mxs-lradc"

#define LRADC_MAX_DELAY_CHANS	4
#define LRADC_MAX_MAPPED_CHANS	8
#define LRADC_MAX_TOTAL_CHANS	16

#define LRADC_DELAY_TIMER_HZ	2000

/*
	* MXS_LRADC_EXTRA_CONFIG - extending lradc as the following

	add extra configuration to select global sampling frequency - config/sample_frequency (IMX28 DELAY settings)
	add per-channel hardwaregain configuration read and write sysfs (IMX28 DIV_BY_TWO)
*/

#define MXS_LRADC_EXTRA_CONFIG

/*
 * Make this runtime configurable if necessary. Currently, if the buffered mode
 * is enabled, the LRADC takes LRADC_DELAY_TIMER_LOOP samples of data before
 * triggering IRQ. The sampling happens every (LRADC_DELAY_TIMER_PER / 2000)
 * seconds. The result is that the samples arrive every 500mS.
 */
#ifdef MXS_LRADC_EXTRA_CONFIG
#define LRADC_DELAY_TIMER_PER	(200*5)
#else
#define LRADC_DELAY_TIMER_PER	200
#define LRADC_DELAY_TIMER_LOOP	5
#endif



static const char * const mxs_lradc_irq_name[] = {
	"mxs-lradc-touchscreen",
	"mxs-lradc-thresh0",
	"mxs-lradc-thresh1",
	"mxs-lradc-channel0",
	"mxs-lradc-channel1",
	"mxs-lradc-channel2",
	"mxs-lradc-channel3",
	"mxs-lradc-channel4",
	"mxs-lradc-channel5",
	"mxs-lradc-channel6",
	"mxs-lradc-channel7",
	"mxs-lradc-button0",
	"mxs-lradc-button1",
};

struct mxs_lradc_chan {
	uint8_t				slot;
	uint8_t				flags;
};

struct mxs_lradc {
	struct device		*dev;
	void __iomem		*base;
	int			irq[13];

	uint32_t		*buffer;
	struct iio_trigger	*trig;

	struct mutex		lock;

	uint8_t			enable;

	struct completion	completion;

	#ifdef MXS_LRADC_EXTRA_CONFIG
	u32 dbt_glb; /* div by two global bitmap representing total 16 logical channels */
	#endif
};

#define	LRADC_CTRL0				0x00
#define LRADC_CTRL0_TOUCH_DETECT_ENABLE		(1 << 23)
#define LRADC_CTRL0_TOUCH_SCREEN_TYPE		(1 << 22)

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
/* channel 0,1,17 used by psm driver */
#define LRADC_CTRL0_SCHEDULE_MASK		(0x7C)
#endif

#define	LRADC_CTRL1				0x10
#define	LRADC_CTRL1_LRADC_IRQ(n)		(1 << (n))

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
/* channel 0,1,7 and thresh0, thresh1 used by psm driver */
#define LRADC_CTRL1_LRADC_IRQ_MASK              0x197C
#else
#define	LRADC_CTRL1_LRADC_IRQ_MASK		0x1fff
#endif

#define	LRADC_CTRL1_LRADC_IRQ_EN(n)		(1 << ((n) + 16))

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
/* channel 0,1,7 and thresh0, thresh1 used by psm driver */
#define LRADC_CTRL1_LRADC_IRQ_EN_MASK           (0x197C << 16)
#else
#define	LRADC_CTRL1_LRADC_IRQ_EN_MASK		(0x1fff << 16)
#endif

#define	LRADC_CTRL2				0x20
#define	LRADC_CTRL2_TEMPSENSE_PWD		(1 << 15)

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
/* channel 0,1,7 used by psm driver */
#define LRADC_CTRL2_DIV_BY_TWO_MASK             (0x7C << 24)
#else
#define LRADC_CTRL2_DIV_BY_TWO_MASK		(0xff << 24)
#endif

#define LRADC_CTRL2_DIV_BY_TWO_OFFSET		24

#define	LRADC_CH(n)				(0x50 + (0x10 * (n)))
#define	LRADC_CH_TOGGLE				(1 << 31)
#define	LRADC_CH_ACCUMULATE			(1 << 29)
#define	LRADC_CH_NUM_SAMPLES_MASK		(0x1f << 24)
#define	LRADC_CH_NUM_SAMPLES_OFFSET		24
#define	LRADC_CH_VALUE_MASK			0x3ffff
#define	LRADC_CH_VALUE_OFFSET			0

#define	LRADC_DELAY(n)				(0xd0 + (0x10 * (n)))
#define	LRADC_DELAY_TRIGGER_LRADCS_MASK		(0xff << 24)
#define	LRADC_DELAY_TRIGGER_LRADCS_OFFSET	24
#define	LRADC_DELAY_KICK			(1 << 20)
#define	LRADC_DELAY_TRIGGER_DELAYS_MASK		(0xf << 16)
#define	LRADC_DELAY_TRIGGER_DELAYS_OFFSET	16
#define	LRADC_DELAY_LOOP_COUNT_MASK		(0x1f << 11)
#define	LRADC_DELAY_LOOP_COUNT_OFFSET		11
#define	LRADC_DELAY_DELAY_MASK			0x7ff
#define	LRADC_DELAY_DELAY_OFFSET		0

#define	LRADC_CTRL3				0x30

#define	LRADC_CTRL4				0x140
#define	LRADC_CTRL4_LRADCSELECT_MASK(n)		(0xf << ((n) * 4))
#define	LRADC_CTRL4_LRADCSELECT_OFFSET(n)	((n) * 4)


#ifdef MXS_LRADC_EXTRA_CONFIG
/* set or clear div by two controls */
static void mxs_lradc_ctrl_dbt(struct iio_dev *iio_dev,u32 phys_dbt,int set)
{
	struct mxs_lradc *lradc = iio_priv(iio_dev);
	u32 reg_ctrl2;

	/* get reg ctrl2 */
	reg_ctrl2=(phys_dbt<<LRADC_CTRL2_DIV_BY_TWO_OFFSET)&LRADC_CTRL2_DIV_BY_TWO_MASK;
	/* change div_by_two register */
	writel(reg_ctrl2, lradc->base + LRADC_CTRL2 + (set ? STMP_OFFSET_REG_SET : STMP_OFFSET_REG_CLR));
}

static int mxs_lradc_read_hardware_gain(struct iio_dev *iio_dev,const struct iio_chan_spec *chan,int *val, int *val2, long m)
{
	struct mxs_lradc *lradc = iio_priv(iio_dev);

	/* bypass if not hardware gain */
	if (m != IIO_CHAN_INFO_HARDWAREGAIN)
		return -EINVAL;

	/* bypass if out of range */
	if (chan->channel > LRADC_MAX_TOTAL_CHANS)
		return -EINVAL;
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	if ((chan->channel == 0) || (chan->channel == 5) || (chan->channel == 7)){
		return -EINVAL;
	}
#endif

	/* read div_by_two bitmap */
	*val = ( lradc->dbt_glb & (1<<chan->channel) )?0:1;

	return IIO_VAL_INT;
}

static int mxs_lradc_write_hardware_gain(struct iio_dev *iio_dev,const struct iio_chan_spec *chan,int val, int val2, long m)
{
	struct mxs_lradc *lradc = iio_priv(iio_dev);
	int ret;

	/* bypass if not hardware gain */
	if (m != IIO_CHAN_INFO_HARDWAREGAIN)
		return -EINVAL;

	/* bypass if out of range */
	if (chan->channel > LRADC_MAX_TOTAL_CHANS)
		return -EINVAL;
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	if ((chan->channel == 0) || (chan->channel == 5) || (chan->channel == 7)){
		return -EINVAL;
	}
#endif

	ret = mutex_trylock(&lradc->lock);
	if (!ret)
		return -EBUSY;

	/* maintain div_by_two bitmap */
	if(val)
		lradc->dbt_glb&=~(1<<chan->channel);
	else
		lradc->dbt_glb|=1<<chan->channel;

	/* print log */
	printk(KERN_NOTICE "mxs-lradc: set div_by_two (ch=%d,gain=%d,dbt_glb=0x%08x)\n",chan->channel,val,lradc->dbt_glb);

	mutex_unlock(&lradc->lock);

	return 0;
}

static ssize_t show_sample_freq(struct device *dev,struct device_attribute *attr, char *buf)
{
	struct iio_dev *iio_dev = dev_to_iio_dev(dev);
	struct mxs_lradc *lradc = iio_priv(iio_dev);

	u32 reg_delay;
	int freq_div;
	int freq;

	/* get delay setting in register */
	reg_delay=readl(lradc->base + LRADC_DELAY(0));
	/* extract delay value */
	freq_div=(reg_delay & LRADC_DELAY_DELAY_MASK) >> LRADC_DELAY_DELAY_OFFSET;
	/* convert delay to frequency */
	freq=LRADC_DELAY_TIMER_HZ/freq_div;

	return sprintf(buf,"%d\n",freq);
}

static ssize_t store_sample_freq(struct device *dev, struct device_attribute *attr,const char *buf, size_t len)
{
	struct iio_dev *iio_dev = dev_to_iio_dev(dev);
	struct mxs_lradc *lradc = iio_priv(iio_dev);

	char* endp;
	int freq;
	int freq_div;

	/* get user input frequency */
	freq=simple_strtoul(buf, &endp, 0);
	if (endp == buf) {
		printk(KERN_ERR "mxs-lradc: invalid sample frequency detected - buf=%s\n",buf);
		return -EINVAL;
	}

	/* check validation */
	if(freq>LRADC_DELAY_TIMER_HZ || freq<=0) {
		printk(KERN_ERR "mxs-lradc: out of frequency range (0~%d Hz) - freq=%d\n",LRADC_DELAY_TIMER_HZ,freq);
		return -EINVAL;
	}

	/* get freq div */
	freq_div=LRADC_DELAY_TIMER_HZ/freq;

	/* clear trigger */
	writel(LRADC_DELAY_DELAY_MASK,lradc->base + LRADC_DELAY(0) + STMP_OFFSET_REG_CLR);
	/* set trigger */
	writel( (freq_div<<LRADC_DELAY_DELAY_OFFSET) & LRADC_DELAY_DELAY_MASK,lradc->base + LRADC_DELAY(0) + STMP_OFFSET_REG_SET);

	/* print log */
	printk(KERN_NOTICE "mxs-lradc: set sample_freq (freq=%d, phys_freq=%d)\n",freq,LRADC_DELAY_TIMER_HZ/freq_div);

	return len;
}

static DEVICE_ATTR(sampling_freq, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH, show_sample_freq, store_sample_freq);

static struct attribute *configs_attrs[] = {
	&dev_attr_sampling_freq.attr,
 	NULL
};

static struct attribute_group configs_group = {
	.name = "configs",
	.attrs = configs_attrs,
};

#endif

/*
 * Raw I/O operations
 */
static int mxs_lradc_read_raw(struct iio_dev *iio_dev,
			const struct iio_chan_spec *chan,
			int *val, int *val2, long m)
{
	struct mxs_lradc *lradc = iio_priv(iio_dev);
	int ret;


	#ifdef MXS_LRADC_EXTRA_CONFIG
	if(m==IIO_CHAN_INFO_HARDWAREGAIN) {
		return mxs_lradc_read_hardware_gain(iio_dev,chan,val,val2,m);
	}
	#endif

	if (m != IIO_CHAN_INFO_RAW)
		return -EINVAL;

	/* Check for invalid channel */
	if (chan->channel > LRADC_MAX_TOTAL_CHANS)
		return -EINVAL;
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	if ((chan->channel == 0) || (chan->channel == 5) || (chan->channel == 7)){
		return -EINVAL;
	}
#endif

	/*
	 * See if there is no buffered operation in progess. If there is, simply
	 * bail out. This can be improved to support both buffered and raw IO at
	 * the same time, yet the code becomes horribly complicated. Therefore I
	 * applied KISS principle here.
	 */
	ret = mutex_trylock(&lradc->lock);
	if (!ret)
		return -EBUSY;

	init_completion(&lradc->completion);

	#ifdef MXS_LRADC_EXTRA_CONFIG
	{
		u32 ch_dbt;

		ch_dbt=lradc->dbt_glb&(1<<chan->channel);

		/* clear or set div_by_reg */
		mxs_lradc_ctrl_dbt(iio_dev,1,ch_dbt);
	}
	#endif

	/*
	 * No buffered operation in progress, map the channel and trigger it.
	 * Virtual channel 0 is always used here as the others are always not
	 * used if doing raw sampling.
	 */
	writel(LRADC_CTRL1_LRADC_IRQ_EN_MASK,
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	writel(0xff & LRADC_CTRL0_SCHEDULE_MASK, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_CLR);
#else
	writel(0xff, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_CLR);
#endif
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	/* the original code mux chan->channel to channel 0, then reset channel 0 */
	/* channel 0 is used by psm driver. Replace by channel 2 */
	writel(chan->channel << LRADC_CTRL4_LRADCSELECT_OFFSET(2), lradc->base + LRADC_CTRL4);
	writel(2, lradc->base + LRADC_CH(2));
#else
	writel(chan->channel, lradc->base + LRADC_CTRL4);
	writel(0, lradc->base + LRADC_CH(0));
#endif

	/* Enable the IRQ and start sampling the channel. */
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	writel(LRADC_CTRL1_LRADC_IRQ_EN(2),
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_SET);
	writel(1 << 2, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_SET);
#else
	writel(LRADC_CTRL1_LRADC_IRQ_EN(0),
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_SET);
	writel(1 << 0, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_SET);
#endif

	/* Wait for completion on the channel, 1 second max. */
	ret = wait_for_completion_killable_timeout(&lradc->completion,
						msecs_to_jiffies(1000));
	if (!ret)
		ret = -ETIMEDOUT;
	if (ret < 0)
		goto err;

	/* Read the data. */
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	*val = readl(lradc->base + LRADC_CH(2)) & LRADC_CH_VALUE_MASK;
#else
	*val = readl(lradc->base + LRADC_CH(0)) & LRADC_CH_VALUE_MASK;
#endif
	ret = IIO_VAL_INT;

err:
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	writel(LRADC_CTRL1_LRADC_IRQ_EN(2),
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);
#else
	writel(LRADC_CTRL1_LRADC_IRQ_EN(0),
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);
#endif

	mutex_unlock(&lradc->lock);

	return ret;
}

static const struct iio_info mxs_lradc_iio_info = {
	.driver_module		= THIS_MODULE,
	.read_raw		= mxs_lradc_read_raw,
	#ifdef  MXS_LRADC_EXTRA_CONFIG
 	.write_raw		= mxs_lradc_write_hardware_gain,
	#endif
};

/*
 * IRQ Handling
 */
static irqreturn_t mxs_lradc_handle_irq(int irq, void *data)
{
	struct iio_dev *iio = data;
	struct mxs_lradc *lradc = iio_priv(iio);
	unsigned long reg = readl(lradc->base + LRADC_CTRL1);

	if (!(reg & LRADC_CTRL1_LRADC_IRQ_MASK))
		return IRQ_NONE;

	/*
	 * Touchscreen IRQ handling code shall probably have priority
	 * and therefore shall be placed here.
	 */

	if (iio_buffer_enabled(iio))
		iio_trigger_poll(iio->trig, iio_get_time_ns());
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	/* channel 0 is used by psm driver, instead use channel 2 as set in mxs_lradc_read_raw function */
	else if (reg & LRADC_CTRL1_LRADC_IRQ(2))
		complete(&lradc->completion);
#else
	else if (reg & LRADC_CTRL1_LRADC_IRQ(0))
		complete(&lradc->completion);
#endif

	writel(reg & LRADC_CTRL1_LRADC_IRQ_MASK,
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);

	return IRQ_HANDLED;
}

/*
 * Trigger handling
 */
static irqreturn_t mxs_lradc_trigger_handler(int irq, void *p)
{
	struct iio_poll_func *pf = p;
	struct iio_dev *iio = pf->indio_dev;
	struct mxs_lradc *lradc = iio_priv(iio);
	struct iio_buffer *buffer = iio->buffer;
	#ifdef MXS_LRADC_EXTRA_CONFIG
	const uint32_t chan_value = 0;
	#else
	const uint32_t chan_value = LRADC_CH_ACCUMULATE |
		((LRADC_DELAY_TIMER_LOOP - 1) << LRADC_CH_NUM_SAMPLES_OFFSET);
	#endif
	int i, j = 0;

	for_each_set_bit(i, iio->active_scan_mask, iio->masklength) {
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
		/* lradc 0,5,7 is for power management */
		if ( (i==0) || (i==5) || (i==7) ){
			continue;
		}
		/* channel 0,1,7 is used by psm driver */
		while (j<=1){
			j++;
		}
		/* maximum channel number is 7. channel 0,1,7 is used by psm driver */
		/* the original code of this driver does not check maximum channel number! */
		if (j==7){
			break;
		}
#endif
		lradc->buffer[j] = readl(lradc->base + LRADC_CH(j));
		writel(chan_value, lradc->base + LRADC_CH(j));

		lradc->buffer[j] &= LRADC_CH_VALUE_MASK;
		#ifdef MXS_LRADC_EXTRA_CONFIG
		#else
		lradc->buffer[j] /= LRADC_DELAY_TIMER_LOOP;
		#endif
		j++;
	}

	if (iio->scan_timestamp) {
		s64 *timestamp = (s64 *)((u8 *)lradc->buffer +
					ALIGN(j, sizeof(s64)));
		*timestamp = pf->timestamp;
	}

	iio_push_to_buffer(buffer, (u8 *)lradc->buffer, pf->timestamp);

	iio_trigger_notify_done(iio->trig);

	return IRQ_HANDLED;
}

static int mxs_lradc_configure_trigger(struct iio_trigger *trig, bool state)
{
	struct iio_dev *iio = trig->private_data;
	struct mxs_lradc *lradc = iio_priv(iio);
	const uint32_t st = state ? STMP_OFFSET_REG_SET : STMP_OFFSET_REG_CLR;

	writel(LRADC_DELAY_KICK, lradc->base + LRADC_DELAY(0) + st);

	return 0;
}

static const struct iio_trigger_ops mxs_lradc_trigger_ops = {
	.owner = THIS_MODULE,
	.set_trigger_state = &mxs_lradc_configure_trigger,
};

static int mxs_lradc_trigger_init(struct iio_dev *iio)
{
	int ret;
	struct iio_trigger *trig;

	trig = iio_trigger_alloc("%s-dev%i", iio->name, iio->id);
	if (trig == NULL)
		return -ENOMEM;

	trig->dev.parent = iio->dev.parent;
	trig->private_data = iio;
	trig->ops = &mxs_lradc_trigger_ops;

	ret = iio_trigger_register(trig);
	if (ret) {
		iio_trigger_free(trig);
		return ret;
	}

	iio->trig = trig;

	return 0;
}

static void mxs_lradc_trigger_remove(struct iio_dev *iio)
{
	iio_trigger_unregister(iio->trig);
	iio_trigger_free(iio->trig);
}

static int mxs_lradc_buffer_preenable(struct iio_dev *iio)
{
	struct mxs_lradc *lradc = iio_priv(iio);
	struct iio_buffer *buffer = iio->buffer;
	int ret = 0, chan, ofs = 0, enable = 0;
	uint32_t ctrl4 = 0;
	uint32_t ctrl1_irq = 0;
	#ifdef MXS_LRADC_EXTRA_CONFIG
	const uint32_t chan_value = 0;
	#else
	const uint32_t chan_value = LRADC_CH_ACCUMULATE |
		((LRADC_DELAY_TIMER_LOOP - 1) << LRADC_CH_NUM_SAMPLES_OFFSET);
	#endif
	const int len = bitmap_weight(buffer->scan_mask, LRADC_MAX_TOTAL_CHANS);

	#ifdef MXS_LRADC_EXTRA_CONFIG
	u32 phys_dbt=0;
	#endif

	if (!len)
		return -EINVAL;

	/*
	 * Lock the driver so raw access can not be done during buffered
	 * operation. This simplifies the code a lot.
	 */
	ret = mutex_trylock(&lradc->lock);
	if (!ret)
		return -EBUSY;

	lradc->buffer = kmalloc(len * sizeof(*lradc->buffer), GFP_KERNEL);
	if (!lradc->buffer) {
		ret = -ENOMEM;
		goto err_mem;
	}

	ret = iio_sw_buffer_preenable(iio);
	if (ret < 0)
		goto err_buf;

	writel(LRADC_CTRL1_LRADC_IRQ_EN_MASK,
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	writel(0xff & LRADC_CTRL0_SCHEDULE_MASK, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_CLR);
#else
	writel(0xff, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_CLR);
#endif

	for_each_set_bit(chan, buffer->scan_mask, LRADC_MAX_TOTAL_CHANS) {
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
		/* lradc 0,5,7 is for power management */
		if ( (chan==0) || (chan==5) || (chan==7) ){
			continue;
		}
		/* channel 0,1,7 is used by psm driver */
		while (ofs<=1){
			ofs++;
		}
		/* maximum channel number is 7. channel 0,1,7 is used by psm driver */
		/* FIXME: the original code of this driver does not check maximum channel number! */
		if (ofs==7){
			break;
		}
#endif
		ctrl4 |= chan << LRADC_CTRL4_LRADCSELECT_OFFSET(ofs);
		ctrl1_irq |= LRADC_CTRL1_LRADC_IRQ_EN(ofs);
		writel(chan_value, lradc->base + LRADC_CH(ofs));
		enable |= 1 << ofs;

		#ifdef MXS_LRADC_EXTRA_CONFIG
		/* collect div_by_two control based on the global bitmap */
		if(lradc->dbt_glb & (1<<chan))
			phys_dbt|=1<<ofs;
		#endif

		ofs++;
	};

	#ifdef MXS_LRADC_EXTRA_CONFIG
	/* fully update div by two */
	mxs_lradc_ctrl_dbt(iio,~0,0);
	mxs_lradc_ctrl_dbt(iio,phys_dbt,1);
	#endif

	writel(LRADC_DELAY_TRIGGER_LRADCS_MASK | LRADC_DELAY_KICK,
		lradc->base + LRADC_DELAY(0) + STMP_OFFSET_REG_CLR);

	writel(ctrl4, lradc->base + LRADC_CTRL4);
	writel(ctrl1_irq, lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_SET);

	writel(enable << LRADC_DELAY_TRIGGER_LRADCS_OFFSET,
		lradc->base + LRADC_DELAY(0) + STMP_OFFSET_REG_SET);

	return 0;

err_buf:
	kfree(lradc->buffer);
err_mem:
	mutex_unlock(&lradc->lock);
	return ret;
}

static int mxs_lradc_buffer_postdisable(struct iio_dev *iio)
{
	struct mxs_lradc *lradc = iio_priv(iio);

	writel(LRADC_DELAY_TRIGGER_LRADCS_MASK | LRADC_DELAY_KICK,
		lradc->base + LRADC_DELAY(0) + STMP_OFFSET_REG_CLR);
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	writel(0xff & LRADC_CTRL0_SCHEDULE_MASK, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_CLR);
#else
	writel(0xff, lradc->base + LRADC_CTRL0 + STMP_OFFSET_REG_CLR);
#endif
	writel(LRADC_CTRL1_LRADC_IRQ_EN_MASK,
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);

	kfree(lradc->buffer);
	mutex_unlock(&lradc->lock);

	return 0;
}

static bool mxs_lradc_validate_scan_mask(struct iio_dev *iio,
					const unsigned long *mask)
{
	const int mw = bitmap_weight(mask, iio->masklength);

	return mw <= LRADC_MAX_MAPPED_CHANS;
}

static const struct iio_buffer_setup_ops mxs_lradc_buffer_ops = {
	.preenable = &mxs_lradc_buffer_preenable,
	.postenable = &iio_triggered_buffer_postenable,
	.predisable = &iio_triggered_buffer_predisable,
	.postdisable = &mxs_lradc_buffer_postdisable,
	.validate_scan_mask = &mxs_lradc_validate_scan_mask,
};

/*
 * Driver initialization
 */

#ifdef MXS_LRADC_EXTRA_CONFIG
#define MXS_ADC_CHAN(idx, chan_type) {				\
	.type = (chan_type),					\
	.indexed = 1,						\
	.scan_index = (idx),					\
	.info_mask = IIO_CHAN_INFO_RAW_SEPARATE_BIT|IIO_CHAN_INFO_HARDWAREGAIN_SEPARATE_BIT,		\
	.channel = (idx),					\
	.scan_type = {						\
		.sign = 'u',					\
		.realbits = 18,					\
		.storagebits = 32,				\
	},							\
}
#else
#define MXS_ADC_CHAN(idx, chan_type) {				\
	.type = (chan_type),					\
	.indexed = 1,						\
	.scan_index = (idx),					\
	.info_mask = IIO_CHAN_INFO_RAW_SEPARATE_BIT,		\
	.channel = (idx),					\
	.scan_type = {						\
		.sign = 'u',					\
		.realbits = 18,					\
		.storagebits = 32,				\
	},							\
}
#endif

static const struct iio_chan_spec mxs_lradc_chan_spec[] = {
#if (!defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)) && (!defined (CONFIG_POWER_SOURCE_BATTERY_IMX28))
	MXS_ADC_CHAN(0, IIO_VOLTAGE),
#endif
	MXS_ADC_CHAN(1, IIO_VOLTAGE),
	MXS_ADC_CHAN(2, IIO_VOLTAGE),
	MXS_ADC_CHAN(3, IIO_VOLTAGE),
	MXS_ADC_CHAN(4, IIO_VOLTAGE),
#if (!defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)) && (!defined (CONFIG_POWER_SOURCE_BATTERY_IMX28))
	MXS_ADC_CHAN(5, IIO_VOLTAGE),
#endif
	MXS_ADC_CHAN(6, IIO_VOLTAGE),
#if (!defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)) && (!defined (CONFIG_POWER_SOURCE_BATTERY_IMX28))
	MXS_ADC_CHAN(7, IIO_VOLTAGE),	/* VBATT */
#endif
	MXS_ADC_CHAN(8, IIO_TEMP),	/* Temp sense 0 */
	MXS_ADC_CHAN(9, IIO_TEMP),	/* Temp sense 1 */
	MXS_ADC_CHAN(10, IIO_VOLTAGE),	/* VDDIO */
	MXS_ADC_CHAN(11, IIO_VOLTAGE),	/* VTH */
	MXS_ADC_CHAN(12, IIO_VOLTAGE),	/* VDDA */
	MXS_ADC_CHAN(13, IIO_VOLTAGE),	/* VDDD */
	MXS_ADC_CHAN(14, IIO_VOLTAGE),	/* VBG */
	MXS_ADC_CHAN(15, IIO_VOLTAGE),	/* VDD5V */
};

static void mxs_lradc_hw_init(struct mxs_lradc *lradc)
{
	int i;
	const uint32_t cfg =
		(LRADC_DELAY_TIMER_PER << LRADC_DELAY_DELAY_OFFSET);

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	for (i = 0; i < (LRADC_MAX_DELAY_CHANS-1); i++)
		writel(cfg | (1 << (LRADC_DELAY_TRIGGER_DELAYS_OFFSET + i)),
				lradc->base + LRADC_DELAY(i));

	/* Start internal temperature sensing. */
	writel(LRADC_CTRL2_TEMPSENSE_PWD, lradc->base + LRADC_CTRL2 + STMP_OFFSET_REG_CLR);
#else
	stmp_reset_block(lradc->base);

	for (i = 0; i < LRADC_MAX_DELAY_CHANS; i++)
		writel(cfg | (1 << (LRADC_DELAY_TRIGGER_DELAYS_OFFSET + i)),
			lradc->base + LRADC_DELAY(i));

	/* Start internal temperature sensing. */
	writel(0, lradc->base + LRADC_CTRL2);
#endif
}

static void mxs_lradc_hw_stop(struct mxs_lradc *lradc)
{
	int i;

	writel(LRADC_CTRL1_LRADC_IRQ_EN_MASK,
		lradc->base + LRADC_CTRL1 + STMP_OFFSET_REG_CLR);
#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	/* DELAY3 is used by battery charger driver */
	for (i = 0; i < LRADC_MAX_DELAY_CHANS-1; i++)
		writel(0, lradc->base + LRADC_DELAY(i));
#else
	for (i = 0; i < LRADC_MAX_DELAY_CHANS; i++)
		writel(0, lradc->base + LRADC_DELAY(i));
#endif
}

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)

/* number of reading ADC values to get average result */
#define NUM_ADC_READINGS_TO_AVG 3
#define TIME_TEMP_SOURCE_CURRENT_RAMP_UP_MS	1
#define TIME_WAIT_FIRST_BATT_VOLT_SENS_MS	1

#define MAX_LRADC_INPUT_VOLT_MS 1850
/* LRADC's resolution is 12-bit, so max value is 2^12-1 */
#define MAX_LRADC_RESULT 4095

typedef enum {
	DETECT_LOW = 0,
	DETECT_HIGH
} lradc_thresh_compare_settings_t;

typedef enum {
	THRESHOLD0 = 0,
	THRESHOLD1
} lradc_threshold_t;

static void __iomem *lradc_base = 0;

#define INT_LRADC_THRESH0		14
#define INT_LRADC_THRESH1		15
#define INT_LRADC_CHANNEL_7		23

/*
 * Notifier list for lradc events
 */
static ATOMIC_NOTIFIER_HEAD(lradc_notifier_list);

int register_lradc_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_register(&lradc_notifier_list, nb);
}
EXPORT_SYMBOL(register_lradc_notifier);

int unregister_lradc_notifier(struct notifier_block *nb)
{
	return atomic_notifier_chain_unregister(&lradc_notifier_list, nb);
}
EXPORT_SYMBOL(unregister_lradc_notifier);

/*
 * configure a LRADC channel
 *
 * Parameters:
 * 	channel:		LRADC channel
 * 	enable_div2:	enable DIV2
 * 	enable_acc:		enable accumulator
 * 	samples:		number of conversion cycles to sum together before reporting operation completed
 */
void imx28_psm_batt_configure_adc_channel(int channel, bool enable_div2, bool enable_acc, int samples)
{
	if (enable_div2)
		__raw_writel(BF_LRADC_CTRL2_DIVIDE_BY_TWO(1 << channel), lradc_base + HW_LRADC_CTRL2_SET);
	else
		__raw_writel(BF_LRADC_CTRL2_DIVIDE_BY_TWO(1 << channel), lradc_base + HW_LRADC_CTRL2_CLR);

	/* Clear the accumulator & NUM_SAMPLES */
	__raw_writel(0xFFFFFFFF, lradc_base + HW_LRADC_CHn_CLR(channel));

	/* Sets NUM_SAMPLES bitfield of HW_LRADC_CHn register. */
	__raw_writel(BM_LRADC_CHn_NUM_SAMPLES, lradc_base + HW_LRADC_CHn_CLR(channel));
	__raw_writel(BF_LRADC_CHn_NUM_SAMPLES(samples), lradc_base + HW_LRADC_CHn_SET(channel));

	if (enable_acc)
		__raw_writel(BM_LRADC_CHn_ACCUMULATE, lradc_base + HW_LRADC_CHn_SET(channel));
	else
		__raw_writel(BM_LRADC_CHn_ACCUMULATE, lradc_base + HW_LRADC_CHn_CLR(channel));
}
EXPORT_SYMBOL(imx28_psm_batt_configure_adc_channel);

/*
 * setup a scheduling delay (i.e HW_LRADC_DELAYn)
 *
 * Parameters:
 * 	delaySchedulerNum:	scheduling delay
 * 	channel:			LRADC channel
 *	loopCount: 			number of times this delay counter will count down and then trigger its designated targets
 * 	delay: 				time (operating on a 2KHz clock) counting down to zero then triggers either a set of LRADC channel conversions
 * 	or another delay channel, or both
 */
void imx28_psm_batt_set_lradc_delay(int delaySchedulerNum, int channel, int loopCount, int delay){
    __raw_writel(BF_LRADC_DELAYn_TRIGGER_LRADCS(1 << channel), lradc_base + HW_LRADC_DELAYn_SET(delaySchedulerNum));
    __raw_writel(BF_LRADC_DELAYn_TRIGGER_DELAYS(1 << delaySchedulerNum), lradc_base + HW_LRADC_DELAYn_SET(delaySchedulerNum));
    __raw_writel(BM_LRADC_DELAYn_LOOP_COUNT | BM_LRADC_DELAYn_DELAY, lradc_base + HW_LRADC_DELAYn_CLR(delaySchedulerNum));
    __raw_writel(BF_LRADC_DELAYn_LOOP_COUNT(loopCount), lradc_base  + HW_LRADC_DELAYn_SET(delaySchedulerNum));
    __raw_writel(BF_LRADC_DELAYn_DELAY(delay), lradc_base + HW_LRADC_DELAYn_SET(delaySchedulerNum));
}
EXPORT_SYMBOL(imx28_psm_batt_set_lradc_delay);

/*
 * Kick a scheduling delay
 *
 * Parameters:
 * 	delaySchedulerNum:	scheduling delay
 * 	kick:				kick that scheduling delay
 */
void imx28_psm_batt_kick_delay_scheduler(int delaySchedulerNum, bool kick)
{
	if (kick)
		__raw_writel(BM_LRADC_DELAYn_KICK, lradc_base + HW_LRADC_DELAYn_SET(delaySchedulerNum));
	else
		__raw_writel(BM_LRADC_DELAYn_KICK, lradc_base + HW_LRADC_DELAYn_CLR(delaySchedulerNum));
}
EXPORT_SYMBOL(imx28_psm_batt_kick_delay_scheduler);

/*
 * enable updating battery voltage from LRADC
 *
 * Parameters:
 * 	enable:	whether enable updating battery voltage from LRADC
 */
void imx28_psm_batt_enable_update_batt_volt(bool enable){
	if (enable){
		__raw_writel(BM_LRADC_CONVERSION_AUTOMATIC, lradc_base + HW_LRADC_CONVERSION_SET);
	}
	else{
		__raw_writel(BM_LRADC_CONVERSION_AUTOMATIC, lradc_base + HW_LRADC_CONVERSION_CLR);
	}
}
EXPORT_SYMBOL(imx28_psm_batt_enable_update_batt_volt);

/*
 * setup and read ADC value measured in a LRADC input
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				true	Use DIV2
 * 						false	Not use DIV2
 */
uint32_t imx28_psm_batt_measure_adc(int analog_mux_input, int channel, bool div2)
{
	uint32_t  value, lradc_irq_mask, sum = 0;
	uint32_t  out_value;
	int i;

	lradc_irq_mask = 1 << channel;

	/* mux analog input for conversion on LRADC channel . */
	__raw_writel((0xF << (4 * channel)), lradc_base + HW_LRADC_CTRL4_CLR);
	__raw_writel((analog_mux_input << (4 * channel)), lradc_base + HW_LRADC_CTRL4_SET);

	/* div2 setting */
    if (div2){
            __raw_writel(BF_LRADC_CTRL2_DIVIDE_BY_TWO(1 << channel), lradc_base + HW_LRADC_CTRL2_SET);
    }
    else{
            __raw_writel(BF_LRADC_CTRL2_DIVIDE_BY_TWO(1 << channel), lradc_base + HW_LRADC_CTRL2_CLR);
    }

    /* Clear the accumulator & NUM_SAMPLES */
    __raw_writel(0xFFFFFFFF, lradc_base + HW_LRADC_CHn_CLR(channel));


	for (i = 0; i < NUM_ADC_READINGS_TO_AVG; i++) {
		/* Clear the interrupt flag */
		__raw_writel(lradc_irq_mask, lradc_base + HW_LRADC_CTRL1_CLR);
		__raw_writel(BF_LRADC_CTRL0_SCHEDULE(1 << channel), lradc_base + HW_LRADC_CTRL0_SET);

		/* Wait for conversion complete*/
		while (!(__raw_readl(lradc_base + HW_LRADC_CTRL1) & lradc_irq_mask)){
			cpu_relax();
		}

		/* Clear the interrupt flag again */
		__raw_writel(lradc_irq_mask, lradc_base + HW_LRADC_CTRL1_CLR);

		/* read value and clr lradc */
		value = __raw_readl(lradc_base + HW_LRADC_CHn(channel)) & BM_LRADC_CHn_VALUE;

		__raw_writel(BM_LRADC_CHn_VALUE, lradc_base + HW_LRADC_CHn_CLR(channel));

		sum += value;
	}

    if (div2){
            __raw_writel(BF_LRADC_CTRL2_DIVIDE_BY_TWO(1 << channel), lradc_base + HW_LRADC_CTRL2_CLR);
    }

	/* Take the voltage average.  */
	value = sum/NUM_ADC_READINGS_TO_AVG;

	/* convert digital value to the real voltage */
	/* input voltage = (ADC value) * 1.85/(2^12-1) */
	out_value = value * MAX_LRADC_INPUT_VOLT_MS/MAX_LRADC_RESULT; /* unit: mV */
	if (div2){
		out_value *= 2;
	}

	return out_value;
}
EXPORT_SYMBOL(imx28_psm_batt_measure_adc);

/*
 * read ADC value measured in a LRADC channel
 *
 * Parameters:
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				whether using DIV2
 */
uint32_t imx28_psm_batt_read_adc(int channel, bool div2)
{
	uint32_t  value;
	uint32_t  out_value;

	/* read value and clr lradc */
	value = __raw_readl(lradc_base + HW_LRADC_CHn(channel)) & BM_LRADC_CHn_VALUE;

	__raw_writel(BM_LRADC_CHn_VALUE, lradc_base + HW_LRADC_CHn_CLR(channel));

	/* convert digital value to the real voltage */
	/* input voltage = (ADC value) * 1.85/(2^12-1) */
	out_value = value * MAX_LRADC_INPUT_VOLT_MS/MAX_LRADC_RESULT; /* unit: mV */
	if (div2){
		out_value *= 2;
	}

	return out_value;
}
EXPORT_SYMBOL(imx28_psm_batt_read_adc);

/*
 * configure threshold for triggering interrupt on value of a LRADC channel.
 * LRADC input will be sensed continuously.
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				whether using DIV2
 * 	delaySchedulerNum:	scheduling delay
 * 	delay:				delay value of the scheduling delay (operating on a 2KHz clock, so unit is 0.5ms)
 * 	thresholdNum:		THRESHOLD0	using the channel threshold comparison 0
 * 						THRESHOLD1	using the channel threshold comparison 1
 * 	polarity:			how to compare the result and threshold values
 * 						DETECT_LOW	Detect when the channel result crosses below the threshold value
 * 						DETECT_HITH	Detect when the channel result crosses above the threshold value
 */
void imx28_psm_batt_set_lradc_thresh_irq(int analog_mux_input, int channel, bool div2, int delaySchedulerNum, int delay , lradc_threshold_t thresholdNum, uint32_t thresh, lradc_thresh_compare_settings_t polarity)
{
	/* mux analog input for conversion on LRADC channel . */
	__raw_writel((0xF << (4 * channel)), lradc_base + HW_LRADC_CTRL4_CLR);
	__raw_writel((analog_mux_input << (4 * channel)), lradc_base + HW_LRADC_CTRL4_SET);

	imx28_psm_batt_configure_adc_channel(channel, div2 /* div2 */ , false /* acc */ , 0 /* num_samples */);

	/* Setup the delay trigger loop forever */
	imx28_psm_batt_set_lradc_delay(delaySchedulerNum, channel, 0, delay);

	/* Clear the accumulator & NUM_SAMPLES */
	__raw_writel(0xFFFFFFFF, lradc_base + HW_LRADC_CHn_CLR(channel));

	/* clear previous "measurement performed" status */
	__raw_writel(1 << channel, lradc_base + HW_LRADC_CTRL1_CLR);

	/* kick off the trigger */
	imx28_psm_batt_kick_delay_scheduler( delaySchedulerNum, true);

	/* set channel to apply */
	__raw_writel(BM_LRADC_THRESHOLDn_CHANNEL_SEL, lradc_base + HW_LRADC_THRESHOLDn_CLR(thresholdNum));
	__raw_writel(BF_LRADC_THRESHOLDn_CHANNEL_SEL(channel), lradc_base + HW_LRADC_THRESHOLDn_SET(thresholdNum));

	/* set detect polarity */
	__raw_writel(BM_LRADC_THRESHOLDn_SETTING, lradc_base + HW_LRADC_THRESHOLDn_CLR(thresholdNum));
	switch (polarity){
	case DETECT_HIGH:
		__raw_writel(BF_LRADC_THRESHOLDn_SETTING(BV_LRADC_THRESHOLDn_SETTING__DETECT_HIGH), lradc_base + HW_LRADC_THRESHOLDn_SET(thresholdNum));
		break;
	case DETECT_LOW:
		__raw_writel(BF_LRADC_THRESHOLDn_SETTING(BV_LRADC_THRESHOLDn_SETTING__DETECT_LOW), lradc_base + HW_LRADC_THRESHOLDn_SET(thresholdNum));
		break;
	}

	/* set threshold value */
	__raw_writel(BM_LRADC_THRESHOLDn_VALUE, lradc_base + HW_LRADC_THRESHOLDn_CLR(thresholdNum));
	/* input voltage = (ADC value) * 1.85/(2^12-1) */
	if (div2){
		thresh /= 2;
	}
	__raw_writel(BF_LRADC_THRESHOLDn_VALUE(thresh*MAX_LRADC_RESULT/MAX_LRADC_INPUT_VOLT_MS), lradc_base + HW_LRADC_THRESHOLDn_SET(thresholdNum));

	/* enable threshold  */
	__raw_writel(BM_LRADC_THRESHOLDn_ENABLE, lradc_base + HW_LRADC_THRESHOLDn_SET(thresholdNum));

	/* enable interrupt */
	switch (thresholdNum){
	case THRESHOLD0:
		__raw_writel(BM_LRADC_CTRL1_THRESHOLD0_DETECT_IRQ_EN, lradc_base + HW_LRADC_CTRL1_SET);
		break;
	case THRESHOLD1:
		__raw_writel(BM_LRADC_CTRL1_THRESHOLD1_DETECT_IRQ_EN, lradc_base + HW_LRADC_CTRL1_SET);
		break;
	}

}
EXPORT_SYMBOL(imx28_psm_batt_set_lradc_thresh_irq);

/*
 * Clear IRQ status (triggered while comparing the result and threshold values) and toggle polarity
 *
 * Parameters:
 * 	thresholdNum:		THRESHOLD0	using the channel threshold comparison 0
 * 						THRESHOLD1	using the channel threshold comparison 1
 * 	polarity:			how to compare the result and threshold values
 * 						DETECT_LOW	Detect when the channel result crosses below the threshold value
 * 						DETECT_HITH	Detect when the channel result crosses above the threshold value
 */
void imx28_psm_batt_toggle_lradc_thresh_polarity(lradc_threshold_t thresholdNum, lradc_thresh_compare_settings_t polarity)
{
	/* clear irq status */
	switch (thresholdNum){
	case THRESHOLD0:
		__raw_writel(BM_LRADC_CTRL1_THRESHOLD0_DETECT_IRQ, lradc_base + HW_LRADC_CTRL1_CLR);
		break;
	case THRESHOLD1:
		__raw_writel(BM_LRADC_CTRL1_THRESHOLD1_DETECT_IRQ, lradc_base + HW_LRADC_CTRL1_CLR);
		break;
	}

	/* set polarity */
	switch (polarity){
	case DETECT_HIGH:
		__raw_writel( (__raw_readl(lradc_base + HW_LRADC_THRESHOLDn(thresholdNum)) & (~BM_LRADC_THRESHOLDn_SETTING))
				| BF_LRADC_THRESHOLDn_SETTING(BV_LRADC_THRESHOLDn_SETTING__DETECT_HIGH),
				lradc_base + HW_LRADC_THRESHOLDn(thresholdNum));
		break;
	case DETECT_LOW:
		__raw_writel( (__raw_readl(lradc_base + HW_LRADC_THRESHOLDn(thresholdNum)) & (~BM_LRADC_THRESHOLDn_SETTING))
				| BF_LRADC_THRESHOLDn_SETTING(BV_LRADC_THRESHOLDn_SETTING__DETECT_LOW),
				lradc_base + HW_LRADC_THRESHOLDn(thresholdNum));
		break;
	}
}
EXPORT_SYMBOL(imx28_psm_batt_toggle_lradc_thresh_polarity);

/*
 * enabling automatic battery temperature input
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input
 * 	channel:			LRADC channel to mux into LRADC input
 * 	delaySchedulerNum:	scheduling delay
 * 	delay:				delay value of the scheduling delay (operating on a 2KHz clock, so unit is 0.5ms)
 * 	thresholdNum:		THRESHOLD0	using the channel threshold comparison 0
 * 						THRESHOLD1	using the channel threshold comparison 1
 * 	polarity:			how to compare the result and threshold values
 * 						DETECT_LOW	Detect when the channel result crosses below the threshold value
 * 						DETECT_HITH	Detect when the channel result crosses above the threshold value
 */
void imx28_psm_batt_setup_batt_temp_sensing(int analog_mux_input, int channel, int delaySchedulerNum, int delay , lradc_threshold_t thresholdNum, uint32_t thresh, lradc_thresh_compare_settings_t polarity)
{
	/* only LRADC0 and LRADC6 include external temperature sensor */
	if (analog_mux_input!=LRADC0 && analog_mux_input!=LRADC6){
		return;
	}

	/* setup source current 20uA */
	if (analog_mux_input == LRADC0){
		__raw_writel(BF_LRADC_CTRL2_TEMP_ISRC0(BV_LRADC_CTRL2_TEMP_ISRC0__20), lradc_base + HW_LRADC_CTRL2_SET);

		__raw_writel(BM_LRADC_CTRL2_TEMP_SENSOR_IENABLE0, lradc_base + HW_LRADC_CTRL2_SET);
	}
	else{
		__raw_writel(BF_LRADC_CTRL2_TEMP_ISRC1(BV_LRADC_CTRL2_TEMP_ISRC1__20), lradc_base + HW_LRADC_CTRL2_SET);

		__raw_writel(BM_LRADC_CTRL2_TEMP_SENSOR_IENABLE1, lradc_base + HW_LRADC_CTRL2_SET);
	}

	/* wait while the current ramps up.  */
	msleep(TIME_TEMP_SOURCE_CURRENT_RAMP_UP_MS);

	/* always div2 */
	imx28_psm_batt_set_lradc_thresh_irq(analog_mux_input, channel, true, delaySchedulerNum, delay , thresholdNum, thresh, polarity);
}
EXPORT_SYMBOL(imx28_psm_batt_setup_batt_temp_sensing);

/*
 * turn on/off temperature current source
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input, only LRADC0 and LRADC6 include external temperature sensor
 * 	on:	on/off
 */
void imx28_psm_batt_turn_temp_current_source(int analog_mux_input, bool on)
{
	/* only LRADC0 and LRADC6 include external temperature sensor */
	if (analog_mux_input!=LRADC0 && analog_mux_input!=LRADC6){
		return;
	}

	if (on){
		if (analog_mux_input == LRADC0){
			__raw_writel(BF_LRADC_CTRL2_TEMP_ISRC0(BV_LRADC_CTRL2_TEMP_ISRC0__20), lradc_base + HW_LRADC_CTRL2_SET);

			__raw_writel(BM_LRADC_CTRL2_TEMP_SENSOR_IENABLE0, lradc_base + HW_LRADC_CTRL2_SET);
		}
		else{
			__raw_writel(BF_LRADC_CTRL2_TEMP_ISRC1(BV_LRADC_CTRL2_TEMP_ISRC1__20), lradc_base + HW_LRADC_CTRL2_SET);

			__raw_writel(BM_LRADC_CTRL2_TEMP_SENSOR_IENABLE1, lradc_base + HW_LRADC_CTRL2_SET);
		}
		/* wait while the current ramps up.  */
		msleep(TIME_TEMP_SOURCE_CURRENT_RAMP_UP_MS);
	}
	else{
		/* turn off source current */
		if (analog_mux_input == LRADC0){
			__raw_writel(BM_LRADC_CTRL2_TEMP_ISRC0, lradc_base + HW_LRADC_CTRL2_CLR);

			__raw_writel(BM_LRADC_CTRL2_TEMP_SENSOR_IENABLE0, lradc_base + HW_LRADC_CTRL2_CLR);
		}
		else{
			__raw_writel(BM_LRADC_CTRL2_TEMP_ISRC1, lradc_base + HW_LRADC_CTRL2_CLR);

			__raw_writel(BM_LRADC_CTRL2_TEMP_SENSOR_IENABLE1, lradc_base + HW_LRADC_CTRL2_CLR);
		}
	}
}
EXPORT_SYMBOL(imx28_psm_batt_turn_temp_current_source);

void imx28_psm_batt_setup_batt_conversion(int channel, int delaySchedulerNum, int delay)
{
	int wait = 0;
	/*
	Select the Li-ion conversion factor and configure the automatic update in the power register:
	— Set HW_LRADC_CONVERSION[SCALE_FACTOR] = 0x2
	— Set HW_LRADC_CONVERSION[AUTOMATIC] = 0x1
	*/
	__raw_writel(BM_LRADC_CONVERSION_SCALE_FACTOR, lradc_base + HW_LRADC_CONVERSION_CLR);
	__raw_writel(BF_LRADC_CONVERSION_SCALE_FACTOR(2), lradc_base + HW_LRADC_CONVERSION_SET);

	/* mux analog input 7 for conversion on a LRADC channel */
	__raw_writel((0xF << (4 * channel)), lradc_base + HW_LRADC_CTRL4_CLR);
	__raw_writel((LRADC7 << (4 * channel)), lradc_base + HW_LRADC_CTRL4_SET);

	/* Setup the delay trigger loop forever */
	imx28_psm_batt_set_lradc_delay(delaySchedulerNum, channel, 0, delay);

	/* Clear the accumulator & NUM_SAMPLES */
	__raw_writel(0xFFFFFFFF, lradc_base + HW_LRADC_CHn_CLR(channel));

	/* clear previous "measurement performed" status */
	__raw_writel(1 << channel, lradc_base + HW_LRADC_CTRL1_CLR);

	/* kick off the trigger */
	imx28_psm_batt_kick_delay_scheduler( delaySchedulerNum, true);

	/* wait for 1st conversion to be complete before enabling automatic copy */
	while (!(__raw_readl(lradc_base + HW_LRADC_CTRL1) & (1 << channel)) && (wait < 10)) {
		wait++;
		msleep(TIME_WAIT_FIRST_BATT_VOLT_SENS_MS);
	}
}
EXPORT_SYMBOL(imx28_psm_batt_setup_batt_conversion);

#define BF_LRADC_CTRL1_LRADC_IRQ_EN(ch) (1<<(16+(ch)))
#define BP_LRADC_CTRL1_LRADC_IRQ(ch) (1<<((ch)))
/*
 * enable/disable interrupt for a channel conversions.
 *
 * Parameters:
 * 	channel:	input channel
 * 	enalbe:		enable/disable
 */
void imx28_psm_batt_enable_channel_irq(int channel, bool enable)
{
	if (channel>=LRADC_CH0 && channel<=LRADC_CH7){
		if (enable){
			__raw_writel(LRADC_CTRL1_LRADC_IRQ_EN(channel), lradc_base + HW_LRADC_CTRL1_SET);
		}
		else{
			__raw_writel(LRADC_CTRL1_LRADC_IRQ_EN(channel), lradc_base + HW_LRADC_CTRL1_CLR);
		}
	}
}
EXPORT_SYMBOL(imx28_psm_batt_enable_channel_irq);

/*
 * IRQ handler of threshold
 */
static irqreturn_t imx28_psm_batt_irq_threshold(int irq, void *cookie)
{
	/* clear irq status */
	if (irq == INT_LRADC_THRESH0){
		__raw_writel(BM_LRADC_CTRL1_THRESHOLD0_DETECT_IRQ, lradc_base + HW_LRADC_CTRL1_CLR);
	}
	else if (irq == INT_LRADC_THRESH1){
		__raw_writel(BM_LRADC_CTRL1_THRESHOLD1_DETECT_IRQ, lradc_base + HW_LRADC_CTRL1_CLR);
	}

	/* call notifier chain */
	atomic_notifier_call_chain(&lradc_notifier_list, irq, NULL);

	return IRQ_HANDLED;
}
/*
 * IRQ handler of battery voltage conversion
 */
static irqreturn_t imx28_psm_batt_irq_battery_voltage(int irq, void *cookie)
{
	/* clear irq status */
	/* TODO: this hard-coded channel should be improved when this driver is re-structured */
	__raw_writel(LRADC_CTRL1_LRADC_IRQ(LRADC_CH7), lradc_base + HW_LRADC_CTRL1_CLR);

	/* call notifier chain */
	atomic_notifier_call_chain(&lradc_notifier_list, irq, NULL);

	return IRQ_HANDLED;
}
#endif

static int __devinit mxs_lradc_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct mxs_lradc *lradc;
	struct iio_dev *iio;
	struct resource *iores;
	int ret = 0;
	int i;

	#ifdef MXS_LRADC_EXTRA_CONFIG
	printk(KERN_NOTICE "mxs-lradc: compiled with MXS LRADC extra configs (@ %s / %s)\n",__TIME__,__DATE__);
	#endif

	/* Allocate the IIO device. */
	iio = iio_device_alloc(sizeof(*lradc));
	if (!iio) {
		dev_err(dev, "Failed to allocate IIO device\n");
		return -ENOMEM;
	}

	lradc = iio_priv(iio);

	/* Grab the memory area */
	iores = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	lradc->dev = &pdev->dev;
	lradc->base = devm_request_and_ioremap(dev, iores);
	if (!lradc->base) {
		ret = -EADDRNOTAVAIL;
		goto err_addr;
	}

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
	lradc_base = lradc->base;
#endif

	/* Grab all IRQ sources */
	for (i = 0; i < 13; i++) {
		lradc->irq[i] = platform_get_irq(pdev, i);
		if (lradc->irq[i] < 0) {
			ret = -EINVAL;
			goto err_addr;
		}

#if defined (CONFIG_POWER_SOURCE_BATTERY_IMX28)	|| defined (CONFIG_POWER_SOURCE_BATTERY_IMX28_MODULE)
		if (lradc->irq[i] == INT_LRADC_THRESH0 || lradc->irq[i] == INT_LRADC_THRESH1){
			ret = devm_request_irq(dev, lradc->irq[i],
						imx28_psm_batt_irq_threshold, 0,
						mxs_lradc_irq_name[i], iio);
		}
		/* TODO: this hard-coded value should be improved when this driver is re-structured */
		else if (lradc->irq[i] == INT_LRADC_CHANNEL_7){
			ret = devm_request_irq(dev, lradc->irq[i],
						imx28_psm_batt_irq_battery_voltage, 0,
						mxs_lradc_irq_name[i], iio);
		}
		else{
			ret = devm_request_irq(dev, lradc->irq[i],
						mxs_lradc_handle_irq, 0,
						mxs_lradc_irq_name[i], iio);
		}
#else
		ret = devm_request_irq(dev, lradc->irq[i],
					mxs_lradc_handle_irq, 0,
					mxs_lradc_irq_name[i], iio);
#endif
		if (ret)
			goto err_addr;
	}

	dev_set_drvdata(&pdev->dev, iio);

	init_completion(&lradc->completion);
	mutex_init(&lradc->lock);

	iio->name = pdev->name;
	iio->dev.parent = &pdev->dev;
	iio->info = &mxs_lradc_iio_info;
	iio->modes = INDIO_DIRECT_MODE;
	iio->channels = mxs_lradc_chan_spec;
	iio->num_channels = ARRAY_SIZE(mxs_lradc_chan_spec);

	ret = iio_triggered_buffer_setup(iio, &iio_pollfunc_store_time,
				&mxs_lradc_trigger_handler,
				&mxs_lradc_buffer_ops);
	if (ret)
		goto err_addr;

#ifdef MXS_LRADC_EXTRA_CONFIG
	/* set 0 to div_by_two bitmap */
	lradc->dbt_glb=0;

	/* initiate div by two - clear all div by two */
	mxs_lradc_ctrl_dbt(iio,~0,0);
#endif


	ret = mxs_lradc_trigger_init(iio);
	if (ret)
		goto err_trig;

#ifdef MXS_LRADC_EXTRA_CONFIG
	iio->groups[iio->groupcounter++]=&configs_group;
#endif

	/* Register IIO device. */
	ret = iio_device_register(iio);
	if (ret) {
		dev_err(dev, "Failed to register IIO device\n");
		goto err_dev;
	}

	/* Configure the hardware. */
	mxs_lradc_hw_init(lradc);

	return 0;

err_dev:
	mxs_lradc_trigger_remove(iio);
err_trig:
	iio_triggered_buffer_cleanup(iio);
err_addr:
	iio_device_free(iio);
	return ret;
}

static int __devexit mxs_lradc_remove(struct platform_device *pdev)
{
	struct iio_dev *iio = dev_get_drvdata(&pdev->dev);
	struct mxs_lradc *lradc = iio_priv(iio);

	mxs_lradc_hw_stop(lradc);

	iio_device_unregister(iio);
	iio_triggered_buffer_cleanup(iio);
	mxs_lradc_trigger_remove(iio);
	iio_device_free(iio);

	return 0;
}

static const struct of_device_id mxs_lradc_dt_ids[] = {
	{ .compatible = "fsl,imx28-lradc", },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, mxs_lradc_dt_ids);

static struct platform_driver mxs_lradc_driver = {
	.driver	= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
		.of_match_table = mxs_lradc_dt_ids,
	},
	.probe	= mxs_lradc_probe,
	.remove	= __devexit_p(mxs_lradc_remove),
};

module_platform_driver(mxs_lradc_driver);

MODULE_AUTHOR("Marek Vasut <marex@...>");
MODULE_DESCRIPTION("Freescale i.MX23/i.MX28 LRADC driver");
MODULE_LICENSE("GPL v2");
