/*
 * Copyright Notice:
 * Copyright (C) 2015 NetComm Pty. Ltd.
 *
 * This file or portions thereof may not be copied or distributed in any form
 * (including but not limited to printed or electronic forms and binary or object forms)
 * without the expressed written consent of NetComm Wireless Pty. Ltd
 * Copyright laws and International Treaties protect the contents of this file.
 * Unauthorized use is prohibited.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY NETCOMM WIRELESS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * NETCOMM WIRELESS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OFF
 * SUCH DAMAGE.
 *
 */

#ifndef _MX28_PSM_BATTERY_IO_H
#define _MX28_PSM_BATTERY_IO_H

/* TODO: definition of lradc_thresh_compare_settings_t and lradc_threshold_t
 * will be in a header file of mxs-lradc driver
 */
typedef enum {
	DETECT_LOW = 0,
	DETECT_HIGH
} lradc_thresh_compare_settings_t;

typedef enum {
	THRESHOLD0 = 0,
	THRESHOLD1
} lradc_threshold_t;


/* enable VDD5V dectection by VBUSVALID */
void imx28_psm_batt_enable_5v_detection_by_VBUSVALID(void __iomem *reg_base);
/* Check VDD5V present
 *
 * Returns:
 * 	true:	VDD5V present
 * 	false:	VDD5V not present
 */
bool imx28_psm_batt_get_5v_present_flag(void __iomem *reg_base);
/*
 * Configure the VBUSVALID detection method to generate interrupt when the 5 V supply is removed
 */
void imx28_psm_batt_enable_irq_on_5v_removed(void __iomem *reg_base);
/*
 * configure the VBUSVALID detection method to generate interrupt when the 5 V supply is connected
 */
void imx28_psm_batt_enable_irq_on_5v_connected(void __iomem *reg_base);
/*
 * Get the battery voltage in mV
 *
 * Returns:
 * 	Battery voltage in mV
 */
uint32_t imx28_psm_batt_get_battery_voltage(void __iomem *reg_base);
/*
 * Convert current (in mA) to bit settings, as used in
 * HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 *
 * Parameters:
 *	current:	current value in mA
 *
 * Returns:
 * 	setting used in HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 */
unsigned int imx28_psm_batt_current_to_setting(unsigned int currentVal);
/*
 * Convert bit setting to current (in mA), as used in
 * HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 *
 * Parameters:
 *	setting used in HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 *
 * Returns:
 * 	current:	current value in mA
 */
unsigned int imx28_psm_batt_setting_to_current(unsigned int setting);
/*
 * Set charging current, in mA
 *
 * Parameters:
 *  current:	current value in mA
 */
void imx28_psm_batt_set_charge_current(void __iomem *reg_base, unsigned int currentVal);
/*
 * Set stop-charging current limit, in mA
 *
 * Parameters:
 * 	current: current value in mA
 */
void imx28_psm_batt_set_stop_current(void __iomem *reg_base, unsigned int currentVal);
/*
 * End the charging process by disabling the battery charger
 */
void imx28_psm_batt_end_charging(void __iomem *reg_base);
/*
 * Check whether charging current is below stop limit current
 *
 * Returns:
 * 	true	charging current is below stop limit current
 * 	false	charging current is not below stop limit current
 */
bool imx28_psm_batt_is_charging_current_below_stop_limit(void __iomem *reg_base);
/*
 * Check whether 4P2 LinReg is enabled
 *
 * Returns:
 * 	true	4P2 LinReg is enabled
 * 	false	4P2 LinReg is not enabled
 */
bool imx28_psm_batt_is_4p2linreg_enabled(void __iomem *reg_base);
/*
 * enable battery charger
 *
 * Returns:
 * 	true	battery charger is enabled successfully
 * 	false	failed in enabling battery charger
 */
bool imx28_psm_batt_enable_battery_charger(void __iomem *reg_base);
/*
 * check whether the battery charger is on
 *
 * Returns:
 * 	true	battery charger is on
 * 	false	battery charger is off
 */
bool imx28_psm_batt_is_battery_charger_on(void __iomem *reg_base);
/*
 * enable 4P2 LinReg
 */
void imx28_psm_batt_enable_4p2linreg(void __iomem *reg_base);
/*
 * charge 4P2 capacitor
 *
 * Parameters:
 * 	current_limit:	total current limit of 4P2 and charger
 */
void imx28_psm_batt_charge_4p2_capacitor(void __iomem *reg_base, unsigned int current_limit);

/*
 * change 4P2 and charger ilimit
 *
 * Parameters:
 * 	current_limit:	new total current limit of 4P2 and charger
 */
void imx28_psm_batt_change_4p2_ilimit(void __iomem *reg_base, unsigned int current_limit);

/*
 * enable 4P2 input for DCDC
 */
void imx28_psm_batt_enable_4p2_input_for_dcdc(void __iomem *reg_base);
/*
 * enable battery brown-out interrupt
 *
 * Parameters:
 * 	enable:	true	enable battery brown-out interrupt
 * 			false	disable battery brown-out interrupt
 */
void imx28_psm_batt_enable_battery_bo_irq(void __iomem *reg_base, bool enable);
/*
 * enable DCDC 4P2 brown-out interrupt
 *
 * Parameters:
 * 	enable:	true	enable DCDC 4P2 brown-out interrupt
 * 			false	disable DCDC 4P2 brown-out interrupt
 */
void imx28_psm_batt_enable_4p2_bo_irq(void __iomem *reg_base, bool enable);
/*
 * enable VDD5V droop interrupt
 *
 * Parameters:
 * 	enable:	true	enable VDD5V droop interrupt
 * 			false	disable VDD5V droop interrupt
 */
void imx28_psm_batt_enable_5v_droop_irq(void __iomem *reg_base, bool enable);
/*
 * enable VDDIO brown-out interrupt
 *
 * Parameters:
 * 	enable:	true	enable VDDIO brown-out interrupt
 * 			false	disable VDDIO brown-out interrupt
 */
void imx28_psm_batt_enable_vddio_bo_irq(void __iomem *reg_base, bool enable);
/*
 * Check whether a battery is present.
 * The automatic battery voltage update for the DC-DC control logic must be already enabled before calling this function.
 *
 * Returns:
 * 	true	battery is present
 * 	false	battery is not present
 */
bool imx28_psm_batt_is_battery_present(void __iomem *reg_base);
/*
 * enable updating battery voltage from LRADC
 *
 * Parameters:
 * 	enable:	whether enable updating battery voltage from LRADC
 */

/*
 * configure a LRADC channel
 *
 * Parameters:
 * 	channel:		LRADC channel
 * 	enable_div2:	enable DIV2
 * 	enable_acc:		enable accumulator
 * 	samples:		number of conversion cycles to sum together before reporting operation completed
 */
extern void imx28_psm_batt_configure_adc_channel(int channel, bool enable_div2, bool enable_acc, int samples);

/*
 * setup a scheduling delay (i.e HW_LRADC_DELAYn)
 *
 * Parameters:
 * 	delaySchedulerNum:	scheduling delay
 * 	channel:			LRADC channel
 *	loopCount: 			number of times this delay counter will count down and then trigger its designated targets
 * 	delay: 				time (operating on a 2KHz clock) counting down to zero then triggers either a set of LRADC channel conversions
 * 	or another delay channel, or both
 */
extern void imx28_psm_batt_set_lradc_delay(int delaySchedulerNum, int channel, int loopCount, int delay);

/*
 * Kick a scheduling delay
 *
 * Parameters:
 * 	delaySchedulerNum:	scheduling delay
 * 	kick:				kick that scheduling delay
 */
extern void imx28_psm_batt_kick_delay_scheduler(int delaySchedulerNum, bool kick);


extern void imx28_psm_batt_enable_update_batt_volt(bool enable);
/*
 * setup and read ADC value measured in a LRADC input
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				true	Use DIV2
 * 						false	Not use DIV2
 */
extern uint32_t imx28_psm_batt_measure_adc(int analog_mux_input, int channel, bool div2);
/*
 * read ADC value measured in a LRADC channel
 *
 * Parameters:
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				whether using DIV2
 */
extern uint32_t imx28_psm_batt_read_adc(int channel, bool div2);
/*
 * configure threshold for triggering interrupt on value of a LRADC channel.
 * LRADC input will be sensed continuously.
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				whether using DIV2
 * 	delaySchedulerNum:	scheduling delay
 * 	delay:				delay value of the scheduling delay (operating on a 2KHz clock, so unit is 0.5ms)
 * 	thresholdNum:		THRESHOLD0	using the channel threshold comparison 0
 * 						THRESHOLD1	using the channel threshold comparison 1
 * 	polarity:			how to compare the result and threshold values
 * 						DETECT_LOW	Detect when the channel result crosses below the threshold value
 * 						DETECT_HITH	Detect when the channel result crosses above the threshold value
 */
extern void imx28_psm_batt_set_lradc_thresh_irq(int analog_mux_input, int channel, bool div2, int delaySchedulerNum, int delay , lradc_threshold_t thresholdNum, uint32_t thresh, lradc_thresh_compare_settings_t polarity);
/*
 * Clear IRQ status (triggered while comparing the result and threshold values) and toggle polarity
 *
 * Parameters:
 * 	thresholdNum:		THRESHOLD0	using the channel threshold comparison 0
 * 						THRESHOLD1	using the channel threshold comparison 1
 * 	polarity:			how to compare the result and threshold values
 * 						DETECT_LOW	Detect when the channel result crosses below the threshold value
 * 						DETECT_HITH	Detect when the channel result crosses above the threshold value
 */
extern void imx28_psm_batt_toggle_lradc_thresh_polarity(lradc_threshold_t thresholdNum, lradc_thresh_compare_settings_t polarity);
/*
 * enabling automatic battery temperature input
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input
 * 	channel:			LRADC channel to mux into LRADC input
 * 	div2:				whether using DIV2
 * 	delaySchedulerNum:	scheduling delay
 * 	delay:				delay value of the scheduling delay (operating on a 2KHz clock, so unit is 0.5ms)
 * 	thresholdNum:		THRESHOLD0	using the channel threshold comparison 0
 * 						THRESHOLD1	using the channel threshold comparison 1
 * 	polarity:			how to compare the result and threshold values
 * 						DETECT_LOW	Detect when the channel result crosses below the threshold value
 * 						DETECT_HITH	Detect when the channel result crosses above the threshold value
 */
extern void imx28_psm_batt_setup_batt_temp_sensing(int analog_mux_input, int channel, int delaySchedulerNum, int delay , lradc_threshold_t thresholdNum, uint32_t thresh, lradc_thresh_compare_settings_t polarity);
/*
 * turn on/off temperature current source
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input, only LRADC0 and LRADC6 include external temperature sensor
 * 	on: on/off
 */
extern void imx28_psm_batt_turn_temp_current_source(int analog_mux_input, bool on);

extern void imx28_psm_batt_setup_batt_conversion(int channel, int delaySchedulerNum, int delay);

/*
 * enabling automatic battery voltage input
 *
 * Parameters:
 * 	channel:			LRADC channel to mux into LRADC input
 * 	delaySchedulerNum:	scheduling delay
 * 	delay:				delay value of the scheduling delay (operating on a 2KHz clock, so unit is 0.5ms)
 */
void imx28_psm_batt_setup_batt_volt_update(void __iomem *reg_base, int channel, int delaySchedulerNum, int delay);

/*
 * Check whether USB D+ and D- detected
 *
 * Returns:
 * 	true:	detected
 * 	false: 	not detected
 */
bool imx28_psm_batt_is_usb_data_signal_detected(void);

/*
 * Enable detection of USB D+ and D- plugged in
 */
void imx28_psm_batt_enable_usb_plugin_detect(void);

#endif
