/*
 * Copyright Notice:
 * Copyright (C) 2015 NetComm Pty. Ltd.
 *
 * This file or portions thereof may not be copied or distributed in any form
 * (including but not limited to printed or electronic forms and binary or object forms)
 * without the expressed written consent of NetComm Wireless Pty. Ltd
 * Copyright laws and International Treaties protect the contents of this file.
 * Unauthorized use is prohibited.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY NETCOMM WIRELESS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * NETCOMM WIRELESS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OFF
 * SUCH DAMAGE.
 *
 */

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <asm/processor.h>
#include <mach/hardware.h>
#include <mach/mxs.h>
#include <mach/mx28.h>
#include <mach/regs-common.h>
#include <mach/regs-power.h>
#include <mach/regs-lradc.h>
#include <linux/jiffies.h>
#include <linux/delay.h>

#include "imx28_psm_battery_io.h"

/* constant value for 8mV steps used in battery translation */
#define BATT_VOLTAGE_8_MV 8
/* maps bit numbers to current increments, used in the register
 * fields HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 */
#define CURRENT_INC_STEPS 5
static const uint16_t currentPerBit[] = {  10,  20,  50, 100, 200, 400 };
/* define battery checking point */
#define BATTERY_CHECKING_POINT_UPPER_THRESHOLD_MV 3520
#define BATTERY_CHECKING_POINT_LOWER_THRESHOLD_MV 3200
#define BATTERY_CHECKING_POINT_FULL_THRESHOLD_MV 4000

#define TIME_4P2_CAP_RAMP_UP_US	1
#define TIME_WAIT_BATT_VOLT_SENS_MS 300

#define TIME_WAIT_CORRECT_BATT_VOLT_SENS_MS	100
#define MAX_LOOP_WAIT_CORRECT_BATT_VOLT_SENS 20
#define INVALID_BATT_VOLT_MS	1000

/* enable VDD5V dectection by VBUSVALID */
void imx28_psm_batt_enable_5v_detection_by_VBUSVALID(void __iomem *reg_base)
{
	/* Turn ON the VBUS comparators:
	HW_POWER_5VCTRL[PWRUP_VBUS_CMPS] = 1 */
	__raw_writel( BM_POWER_5VCTRL_PWRUP_VBUS_CMPS , reg_base + HW_POWER_5VCTRL_SET);

	/* Set the VBUS threshold value to 4.0 V (register setting— 0x1 ):
	HW_POWER_5VCTRL[VBUSVALID_TRSH] = 0x1 */
	__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_VBUSVALID_TRSH) | POWER_5VCTRL_VBUSVALID_TRSH_4V0  ,
			reg_base + HW_POWER_5VCTRL);
	/* Set the threshold for the VBUSDROOP comparator: 0b00: 4.3V */
	__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_VBUSDROOP_TRSH)   ,
			reg_base + HW_POWER_5VCTRL);

	/* Change the 5 V-detection method to VBUSVALID:
	HW_POWER_5VCTRL[VBUSVALID_5VDETECT] = 1 */
	__raw_writel( BM_POWER_5VCTRL_VBUSVALID_5VDETECT , reg_base + HW_POWER_5VCTRL_SET);
}

/* Check VDD5V present
 *
 * Returns:
 * 	true:	VDD5V present
 * 	false:	VDD5V not present
 */
bool imx28_psm_batt_get_5v_present_flag(void __iomem *reg_base)
{
	return ((__raw_readl(reg_base + HW_POWER_STS) &
			BM_POWER_STS_VBUSVALID0) != 0);
}

/*
 * Configure the VBUSVALID detection method to generate interrupt when the 5 V supply is removed
 */
void imx28_psm_batt_enable_irq_on_5v_removed(void __iomem *reg_base)
{
	/*
	Set the threshold voltage for the VBUSVALID detection method to 4.0 V (register setting— 0x1 ):
	HW_POWER_5VCTRL[VBUSVALID_TRSH] = 0x1
	*/
	__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_VBUSVALID_TRSH) | POWER_5VCTRL_VBUSVALID_TRSH_4V0  ,
			reg_base + HW_POWER_5VCTRL);

	/*
	Clear the 5 V-interrupt status bit by writing 1 to the SCT clear address space of the IRQ status bit:
	HW_POWER_CTRL_CLR[VBUSVALID_IRQ] = 1
	*/
	__raw_writel(  BM_POWER_CTRL_VBUSVALID_IRQ,
			reg_base + HW_POWER_CTRL_CLR);

	/*
	Set the polarity field to detect when the 5 V supply is removed:
	HW_POWER_CTRL[POLARITY_VBUSVALID] = 0
	*/
	__raw_writel( POWER_CTRL_POLARITY_VBUSVALID,
			reg_base + HW_POWER_CTRL_CLR);

	/*
	Enable the VBUSVALID interrupt:
	HW_POWER_CTRL[ENIRQ_VBUS_VALID]
	*/
	__raw_writel( POWER_CTRL_ENIRQ_VBUS_VALID,
			reg_base + HW_POWER_CTRL_SET);

}

/*
 * configure the VBUSVALID detection method to generate interrupt when the 5 V supply is connected
 */
void imx28_psm_batt_enable_irq_on_5v_connected(void __iomem *reg_base)
{
	/*
	Set the threshold voltage for the VBUSVALID detection method:
	HW_POWER_5VCTRL[VBUSVALID_TRSH] = 0x5, 4.4V
	*/
	__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_VBUSVALID_TRSH) | POWER_5VCTRL_VBUSVALID_TRSH_4V4  ,
			reg_base + HW_POWER_5VCTRL);

	/*
	Clear the 5 V-interrupt status bit by writing 1 to the SCT clear address space of the IRQ status bit:
	HW_POWER_CTRL_CLR[VBUSVALID_IRQ] = 1
	*/
	__raw_writel(  BM_POWER_CTRL_VBUSVALID_IRQ,
			reg_base + HW_POWER_CTRL_CLR);

	/*
	Set the polarity field to detect when the 5 V supply is removed:
	HW_POWER_CTRL[POLARITY_VBUSVALID] = 0
	*/
	__raw_writel( POWER_CTRL_POLARITY_VBUSVALID,
			reg_base + HW_POWER_CTRL_SET);

	/*
	Enable the VBUSVALID interrupt:
	HW_POWER_CTRL[ENIRQ_VBUS_VALID]
	*/
	__raw_writel( POWER_CTRL_ENIRQ_VBUS_VALID,
			reg_base + HW_POWER_CTRL_SET);
}

/*
 * Get the battery voltage in mV
 *
 * Returns:
 * 	Battery voltage in mV
 */
uint32_t imx28_psm_batt_get_battery_voltage(void __iomem *reg_base)
{
	uint32_t volatile    battVolt;

	/* get the raw result of battery measurement */
	battVolt = __raw_readl(reg_base + HW_POWER_BATTMONITOR);
	battVolt &= BM_POWER_BATTMONITOR_BATT_VAL;
	battVolt >>= BP_POWER_BATTMONITOR_BATT_VAL;

	/* adjust for 8-mV resolution */
	battVolt *= BATT_VOLTAGE_8_MV;

	return battVolt;
}


/*
 * Convert current (in mA) to bit settings, as used in
 * HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 *
 * Parameters:
 *	current:	current value in mA
 *
 * Returns:
 * 	setting used in HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 */
unsigned int imx28_psm_batt_current_to_setting(unsigned int current)
{
	int       i;
	unsigned int  mask;
	unsigned int  setting = 0;

	/* Scan across the bit field, adding in current increments. */
	mask = (0x1 << CURRENT_INC_STEPS);

	for (i = CURRENT_INC_STEPS; (i >= 0) && (current > 0); i--, mask >>= 1) {
		if (current >= currentPerBit[i]) {
			current -= currentPerBit[i];
			setting |= mask;
		}
	}

	return setting;
}

/*
 * Convert bit setting to current (in mA), as used in
 * HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 *
 * Parameters:
 *	setting used in HW_POWER_CHARGE.STOP_ILIMIT and HW_POWER_CHARGE.BATTCHRG_I
 *
 * Returns:
 * 	current:	current value in mA
 */
unsigned int imx28_psm_batt_setting_to_current(unsigned int setting)
{
	int       i;
	unsigned int  mask;
	unsigned int  current = 0;

	/* Scan across the bit field, adding in current increments. */
	mask = (0x1 << CURRENT_INC_STEPS);

	for (i = CURRENT_INC_STEPS; i >= 0; i--, mask >>= 1) {
		if (setting & mask)
			current += currentPerBit[i];
	}

	return current;
}

/*
 * Set charging current, in mA
 *
 * Parameters:
 *  current:	current value in mA
 */
void imx28_psm_batt_set_charge_current(void __iomem *reg_base, unsigned int current)
{
	/*
	Write the register setting to the battery charge current field: HW_POWER_CHARGE[BATTCHRG_I
	*/
	__raw_writel( (__raw_readl(reg_base + HW_POWER_CHARGE) & (~POWER_CHARGE_BATTCHRG_I_MASK)) | imx28_psm_batt_current_to_setting(current),
			reg_base + HW_POWER_CHARGE);
}

/*
 * Set stop-charging current limit, in mA
 *
 * Parameters:
 * 	current: current value in mA
 */
void imx28_psm_batt_set_stop_current(void __iomem *reg_base, unsigned int current)
{
	/*
	Write the register setting to the battery charge current field HW_POWER_CHARGE[STOP_ILIMIT]
	*/
	__raw_writel( (__raw_readl(reg_base + HW_POWER_CHARGE) & (~POWER_CHARGE_STOP_ILIMIT_MASK))
			| (imx28_psm_batt_current_to_setting(current)<<POWER_CHARGE_STOP_ILIMIT_OFFSET),
			reg_base + HW_POWER_CHARGE);
}

/*
 * End the charging process by disabling the battery charger
 */
void imx28_psm_batt_end_charging(void __iomem *reg_base)
{
	/*
	 * End the charging process by disabling the battery charger.
	 * This is accomplished by setting the power down charger bit:
	 * HW_POWER_CHARGE[PWD_CHARGE] = 1
	 */
	__raw_writel( POWER_CHARGE_PWD_BATTCHRG , reg_base + HW_POWER_CHARGE_SET);
}

/*
 * Check whether charging current is below stop limit current
 *
 * Returns:
 * 	true	charging current is below stop limit current
 * 	false	charging current is not below stop limit current
 */
bool imx28_psm_batt_is_charging_current_below_stop_limit(void __iomem *reg_base)
{
	return (__raw_readl( reg_base + HW_POWER_STS ) & BM_POWER_STS_CHRGSTS) == 0;
}

/*
 * Check whether 4P2 LinReg is enabled
 *
 * Returns:
 * 	true	4P2 LinReg is enabled
 * 	false	4P2 LinReg is not enabled
 */
bool imx28_psm_batt_is_4p2linreg_enabled(void __iomem *reg_base){

	/* Use this condition to check 42P LinReg: HW_POWER_DCDC4P2[ENABLE_4P2] = 1 and HW_POWER_DCDC4P2[ENABLE_DCDC] = 1*/
	return (__raw_readl( reg_base + HW_POWER_DCDC4P2 ) & BM_POWER_DCDC4P2_ENABLE_4P2) !=0 &&
			(__raw_readl( reg_base + HW_POWER_DCDC4P2 ) & BM_POWER_DCDC4P2_ENABLE_DCDC) !=0 ;
}

/*
 * enable battery charger
 *
 * Returns:
 * 	true	battery charger is enabled successfully
 * 	false	failed in enabling battery charger
 */
bool imx28_psm_batt_enable_battery_charger(void __iomem *reg_base)
{
	/*	When the 4P2 LinReg is active, the charger can be enabled by toggling a register field member.
	 * 	HW_POWER_CHARGE[PWD_BATTCHRG] = 0
	 */
	if (imx28_psm_batt_is_4p2linreg_enabled(reg_base))
	{
		__raw_writel( BM_POWER_CHARGE_PWD_BATTCHRG, reg_base + HW_POWER_CHARGE_CLR);
		return true;
	}
	/* When the 4P2 LinReg is not enabled, the battery charger is enabled in a two-step process */
	else
	{
		/* In this driver, 4P2 LinReg must be enabled already */
		return false;
	}

}

/*
 * check whether the battery charger is on
 *
 * Returns:
 * 	true	battery charger is on
 * 	false	battery charger is off
 */
bool imx28_psm_batt_is_battery_charger_on(void __iomem *reg_base)
{
	return (__raw_readl( reg_base + HW_POWER_CHARGE) & BM_POWER_CHARGE_PWD_BATTCHRG) == 0;
}

/*
 * enable 4P2 LinReg
 */
void imx28_psm_batt_enable_4p2linreg(void __iomem *reg_base)
{
	/* Set the 4P2 target to 4.2 V:
	HW_POWER_DCDC4P2[TRG] = 0 */
	__raw_writel( __raw_readl(reg_base + HW_POWER_DCDC4P2) & ~BM_POWER_DCDC4P2_TRG , reg_base + HW_POWER_DCDC4P2);
	/* Set 4P2 brown out 4.0V (base 3.6V, step size 25mV --> 16*25 + 3600 = 4000mV)*/
	__raw_writel( (__raw_readl(reg_base + HW_POWER_DCDC4P2) & ~BM_POWER_DCDC4P2_BO) | BF_POWER_DCDC4P2_TRG(16) , reg_base + HW_POWER_DCDC4P2);
	/* Enable the 4P2 circuitry to control the LinReg:
	HW_POWER_DCDC4P2[ENABLE_4P2] = 1 */
	__raw_writel( __raw_readl(reg_base + HW_POWER_DCDC4P2) | BM_POWER_DCDC4P2_ENABLE_4P2, reg_base + HW_POWER_DCDC4P2);
	/* The 4P2 LinReg requires a static load to operate properly. As the DC-DC converter does not load
	the LinReg at this point, another load should be used:
	HW_POWER_CHARGE[ENABLE_LOAD] = 1 */
	__raw_writel( BM_POWER_CHARGE_ENABLE_LOAD, reg_base + HW_POWER_CHARGE_SET);
	/* Provide an initial current limit for the 4P2 LinReg with the smallest possible value:
	HW_POWER_5VCTRL[CHARGE_4P2_ILIMIT] = 1 */
	__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_CHARGE_4P2_ILIMIT) | BF_POWER_5VCTRL_CHARGE_4P2_ILIMIT(1), reg_base + HW_POWER_5VCTRL);
	/* Power ON the 4P2 LinReg:
	HW_POWER_5VCTRL[PWD_CHARGE_4P2] = 0 */
	__raw_writel( BM_POWER_5VCTRL_PWD_CHARGE_4P2, reg_base + HW_POWER_5VCTRL_CLR);
	/* Ungate the path from the 4P2 LinReg to DC-DC converter:
	HW_POWER_DCDC4P2[ENABLE_DCDC] = 1 */
	__raw_writel( __raw_readl(reg_base + HW_POWER_DCDC4P2) | BM_POWER_DCDC4P2_ENABLE_DCDC, reg_base + HW_POWER_DCDC4P2);
}

/*
 * charge 4P2 capacitor
 *
 * Parameters:
 * 	current_limit:	total current limit of 4P2 and charger
 */
void imx28_psm_batt_charge_4p2_capacitor(void __iomem *reg_base, unsigned int current_limit)
{
	unsigned current_limit_setting = imx28_psm_batt_current_to_setting(current_limit);
	unsigned int charge_4p2_ilimit = 1;

	while ( charge_4p2_ilimit < current_limit_setting ){
		/* Set HW_POWER_5VCTRL[CHARGE_4P2_ILIMIT] += 1 */
		++charge_4p2_ilimit;
		__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_CHARGE_4P2_ILIMIT) | BF_POWER_5VCTRL_CHARGE_4P2_ILIMIT(charge_4p2_ilimit), reg_base + HW_POWER_5VCTRL);
		/* Wait at least for 10 μs and repeat Step 1 until the current reaches the maximum limit of 780 mA*/
		udelay(TIME_4P2_CAP_RAMP_UP_US);
	}
}

/*
 * change 4P2 and charger ilimit
 *
 * Parameters:
 * 	current_limit:	new total current limit of 4P2 and charger
 */
void imx28_psm_batt_change_4p2_ilimit(void __iomem *reg_base, unsigned int current_limit)
{
	unsigned current_limit_setting = imx28_psm_batt_current_to_setting(current_limit);
	unsigned ilimit_setting_now = ((__raw_readl(reg_base + HW_POWER_5VCTRL) & BM_POWER_5VCTRL_CHARGE_4P2_ILIMIT) >> BP_POWER_5VCTRL_CHARGE_4P2_ILIMIT);

	if (current_limit_setting <  ilimit_setting_now){
		/* if set to lower value, just set it */
		__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_CHARGE_4P2_ILIMIT) | BF_POWER_5VCTRL_CHARGE_4P2_ILIMIT(current_limit_setting), reg_base + HW_POWER_5VCTRL);
	}
	else{
		/* setting to higher value needs ramping up */
		unsigned int charge_4p2_ilimit = ilimit_setting_now;
		while ( charge_4p2_ilimit < current_limit_setting ){
			/* Set HW_POWER_5VCTRL[CHARGE_4P2_ILIMIT] += 1 */
			++charge_4p2_ilimit;
			__raw_writel( (__raw_readl(reg_base + HW_POWER_5VCTRL) & ~BM_POWER_5VCTRL_CHARGE_4P2_ILIMIT) | BF_POWER_5VCTRL_CHARGE_4P2_ILIMIT(charge_4p2_ilimit), reg_base + HW_POWER_5VCTRL);
			/* Wait at least for 10 μs and repeat Step 1 until the current reaches the maximum limit of 780 mA*/
			udelay(TIME_4P2_CAP_RAMP_UP_US);
		}
	}
}

/*
 * enable 4P2 input for DCDC
 */
void imx28_psm_batt_enable_4p2_input_for_dcdc(void __iomem *reg_base)
{
	/* Adjust the comparison between the battery and 4P2 LinReg to 85% of the rated voltage value:
	HW_POWER_DCDC4P2[CMPTRIP] = 0x0
	Sets the trip point for the comparison between the DCDC_4P2 and BATTERY pin. When the comparator
	output is high then, the switching converter may use the DCDC_4P2 pin as the source for the switching
	converter, otherwise it will use the DCDC_BATT pin.
	0b00000 DCDC_4P2 pin >= 0.85 * BATTERY pin
	0b00001 DCDC_4P2 pin >= 0.86 * BATTERY pin
	0b11000 DCDC_4P2 pin >= BATTERY pin (default)
	0b11111 DCDC_4P2 pin >= 1.05 * BATTERY pin
	*/
	__raw_writel( __raw_readl(reg_base + HW_POWER_DCDC4P2) & ~BM_POWER_DCDC4P2_CMPTRIP, reg_base + HW_POWER_DCDC4P2);

	/* Set HW_POWER_DCDC4P2[DROPOUT_CTRL] = 0b1110
	 * 0bXX01: DcDc converter uses DCDC_4P2 always, and only enables DCDC_BATT when VDD4P2 is less than BATTERY.
	 * 0bXX1X: DcDc converter selects either VDD4P2 or BATTERY, which ever is higher.
	 */
	__raw_writel( (__raw_readl(reg_base + HW_POWER_DCDC4P2) & ~BM_POWER_DCDC4P2_DROPOUT_CTRL) | BF_POWER_DCDC4P2_DROPOUT_CTRL(0b1110), reg_base + HW_POWER_DCDC4P2);

	/* Disable the automatic DC-DC startup when the 5 V supply is powered-OFF:
	HW_POWER_5VCTRL[DCDC_XFER] = 0 */
	__raw_writel( BM_POWER_5VCTRL_DCDC_XFER, reg_base + HW_POWER_5VCTRL_CLR);

	/* Enable the DC-DC converter. This should be done by following the steps for enabling the DC-DC
	converter when the 5 V source is supplied:
	HW_POWER_5VCTRL[ENABLE_DCDC] = 1 */
	__raw_writel( BM_POWER_5VCTRL_ENABLE_DCDC, reg_base + HW_POWER_5VCTRL_SET);

	/* turn of internal load */
	__raw_writel( BM_POWER_CHARGE_ENABLE_LOAD, reg_base + HW_POWER_CHARGE_CLR);
}

/*
 * enable battery brown-out interrupt
 *
 * Parameters:
 * 	enable:	true	enable battery brown-out interrupt
 * 			false	disable battery brown-out interrupt
 */
void imx28_psm_batt_enable_battery_bo_irq(void __iomem *reg_base, bool enable)
{
	if (enable) {
		__raw_writel(BM_POWER_CTRL_BATT_BO_IRQ,
			reg_base + HW_POWER_CTRL_CLR);
		__raw_writel(BM_POWER_CTRL_ENIRQBATT_BO,
			reg_base + HW_POWER_CTRL_SET);

	} else {
		__raw_writel(BM_POWER_CTRL_ENIRQBATT_BO,
			reg_base + HW_POWER_CTRL_CLR);
	}
}

/*
 * enable DCDC 4P2 brown-out interrupt
 *
 * Parameters:
 * 	enable:	true	enable DCDC 4P2 brown-out interrupt
 * 			false	disable DCDC 4P2 brown-out interrupt
 */
void imx28_psm_batt_enable_4p2_bo_irq(void __iomem *reg_base, bool enable)
{
	if (enable) {
		__raw_writel(BM_POWER_CTRL_DCDC4P2_BO_IRQ,
			reg_base + HW_POWER_CTRL_CLR);
		__raw_writel(BM_POWER_CTRL_ENIRQ_DCDC4P2_BO,
			reg_base + HW_POWER_CTRL_SET);
	} else {
		__raw_writel(BM_POWER_CTRL_ENIRQ_DCDC4P2_BO,
			reg_base + HW_POWER_CTRL_CLR);
	}
}

/*
 * enable VDD5V droop interrupt
 *
 * Parameters:
 * 	enable:	true	enable VDD5V droop interrupt
 * 			false	disable VDD5V droop interrupt
 */
void imx28_psm_batt_enable_5v_droop_irq(void __iomem *reg_base, bool enable)
{
	if (enable) {
		__raw_writel(BM_POWER_CTRL_VDD5V_DROOP_IRQ,
			reg_base + HW_POWER_CTRL_CLR);
		__raw_writel(BM_POWER_CTRL_ENIRQ_VDD5V_DROOP,
			reg_base + HW_POWER_CTRL_SET);
	} else {
		__raw_writel(BM_POWER_CTRL_ENIRQ_VDD5V_DROOP,
			reg_base + HW_POWER_CTRL_CLR);
	}
}

/*
 * enable VDDIO brown-out interrupt
 *
 * Parameters:
 * 	enable:	true	enable VDDIO brown-out interrupt
 * 			false	disable VDDIO brown-out interrupt
 */
void imx28_psm_batt_enable_vddio_bo_irq(void __iomem *reg_base, bool enable)
{
	if (enable) {
		__raw_writel(BM_POWER_CTRL_VDDIO_BO_IRQ,
			reg_base + HW_POWER_CTRL_CLR);

		__raw_writel(BM_POWER_CTRL_ENIRQ_VDDIO_BO,
			reg_base + HW_POWER_CTRL_SET);
	} else {
		__raw_writel(BM_POWER_CTRL_ENIRQ_VDDIO_BO,
			reg_base + HW_POWER_CTRL_CLR);
	}
}

/*
 * Check whether a battery is present.
 * The automatic battery voltage update for the DC-DC control logic must be already enabled before calling this function.
 *
 * Returns:
 * 	true	battery is present
 * 	false	battery is not present
 */
bool imx28_psm_batt_is_battery_present(void __iomem *reg_base)
{
	bool volatile result;
	bool volatile restoreCharger = false;
	uint32_t batt_vol_mv = 0;

	if (imx28_psm_batt_is_battery_charger_on(reg_base)){
		/* turn off to get correct result */
		imx28_psm_batt_end_charging(reg_base);
		msleep(TIME_WAIT_BATT_VOLT_SENS_MS);
		restoreCharger = true;
	}

	/* compare battery voltage with checking-point threshold */
	batt_vol_mv = imx28_psm_batt_get_battery_voltage(reg_base);
	if (batt_vol_mv >= BATTERY_CHECKING_POINT_UPPER_THRESHOLD_MV){
		/* if battery volt is higher than the upper threshold, battery mus be connected */
		result = true;
	}
#ifdef DETECTING_BATTERY_BY_CHARGER
	else if (batt_vol_mv > BATTERY_CHECKING_POINT_LOWER_THRESHOLD_MV && batt_vol_mv < BATTERY_CHECKING_POINT_UPPER_THRESHOLD_MV){
		/* need to turn on charger for a while then check battery voltage.
		 * If battery volt raise to ~4200, there is no battery.
		 * Else there is a battery connected.
		 * */
		imx28_psm_batt_enable_battery_charger(reg_base);

		imx28_psm_batt_enable_update_batt_volt(true);
		msleep(TIME_WAIT_BATT_VOLT_SENS_MS);
		batt_vol_mv = imx28_psm_batt_get_battery_voltage(reg_base);
		imx28_psm_batt_end_charging(reg_base);
		if (batt_vol_mv > BATTERY_CHECKING_POINT_FULL_THRESHOLD_MV){
			result = false;
		}
		else{
			result = true;
		}
	}
#endif
	else {
		/* battery voltage is below BATTERY_CHECKING_POINT_LOWER_THRESHOLD_MV, we can use fast-settling bit for battery detection. */
		/*
		Disable the battery-brownout power down and enable the battery-brownout comparator:
		— Set HW_POWER_BATTMONITOR[PWD_BATTBRNOUT] = 0
		— Set HW_POWER_BATTMONITOR[BRWNOUT_PWD] = 0
		*/
		__raw_writel(  (__raw_readl(reg_base + HW_POWER_BATTMONITOR) & (~BM_POWER_BATTMONITOR_BRWNOUT_PWD)) & (~BM_POWER_BATTMONITOR_PWDN_BATTBRNOUT),
				reg_base + HW_POWER_BATTMONITOR);

		/*
		Enable the fast-settling bit for the battery detection:
		HW_POWER_REFCTRL[FASTSETTLING] = 1
		*/
		__raw_writel(  BM_POWER_REFCTRL_FASTSETTLING,
				reg_base + HW_POWER_REFCTRL_SET);

		/*
		Read the HW_POWER_STS[BATT_BO] bit:
		— If the bit is 1, battery brownout has occurred and the battery is not available
		— If the bit is 0, a battery is attached to the system
		*/
		if ( (__raw_readl(  reg_base + HW_POWER_STS) & BM_POWER_STS_BATT_BO ) != 0){
			result = false;
		}
		else{
			result = true;
		}

		/*
		Reset the fast-settling bit to complete the battery detection:
		HW_POWER_REFCTRL_CLR[FASTSETTLING] = 1
		*/
		__raw_writel(  BM_POWER_REFCTRL_FASTSETTLING,
				reg_base + HW_POWER_REFCTRL_CLR);
	}

	if (restoreCharger){
		imx28_psm_batt_enable_battery_charger(reg_base);
	}

	return result;
}

/*
 * enabling automatic battery voltage input
 *
 * Parameters:
 * 	channel:			LRADC channel to mux into LRADC input
 * 	delaySchedulerNum:	scheduling delay
 * 	delay:				delay value of the scheduling delay (operating on a 2KHz clock, so unit is 0.5ms)
 */
void imx28_psm_batt_setup_batt_volt_update(void __iomem *reg_base, int channel, int delaySchedulerNum, int delay)
{
	int wait = 0;

	imx28_psm_batt_setup_batt_conversion(channel, delaySchedulerNum, delay);

	/* lradc result may be incorrect in the first several ms */
	for (wait = 0; wait < MAX_LOOP_WAIT_CORRECT_BATT_VOLT_SENS; wait++){
		if (imx28_psm_batt_get_battery_voltage(reg_base) < INVALID_BATT_VOLT_MS) {
			msleep(TIME_WAIT_CORRECT_BATT_VOLT_SENS_MS);
		} else
			break;
	}

	imx28_psm_batt_enable_update_batt_volt(true);

	/*
	Enable the battery voltage as an input to the DC-DC switching logic:
	HW_POWER_BATTMONITOR[EN_BATADJ] = 1
	*/
	__raw_writel( __raw_readl(reg_base + HW_POWER_BATTMONITOR) | BM_POWER_BATTMONITOR_EN_BATADJ,
			reg_base + HW_POWER_BATTMONITOR);
}

/* TODO: implement these functions in USB PHY driver */
#define REGS_USBPHY0_BASE MXS_IO_ADDRESS(MX28_USBPHY0_BASE_ADDR)
#define HW_USBPHY_STATUS 0x40
#define HW_USBPHY_CTRL_SET 0x034
#define BM_USBPHY_STATUS_DEVPLUGIN_STATUS (1 << 6)
#define BM_USBPHY_CTRL_ENDEVPLUGINDETECT (1 << 4)

/*
 * Check whether USB D+ and D- detected
 *
 * Returns:
 * 	true:	detected
 * 	false: 	not detected
 */
bool imx28_psm_batt_is_usb_data_signal_detected(void)
{
	return ((__raw_readl(REGS_USBPHY0_BASE + HW_USBPHY_STATUS) & BM_USBPHY_STATUS_DEVPLUGIN_STATUS) != 0);
}

/*
 * Enable detection of USB D+ and D- plugged in
 */
void imx28_psm_batt_enable_usb_plugin_detect(void)
{
	__raw_writel(BM_USBPHY_CTRL_ENDEVPLUGINDETECT, REGS_USBPHY0_BASE + HW_USBPHY_CTRL_SET);
}
