/*
 * Copyright Notice:
 * Copyright (C) 2015 NetComm Pty. Ltd.
 *
 * This file or portions thereof may not be copied or distributed in any form
 * (including but not limited to printed or electronic forms and binary or object forms)
 * without the expressed written consent of NetComm Wireless Pty. Ltd
 * Copyright laws and International Treaties protect the contents of this file.
 * Unauthorized use is prohibited.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY NETCOMM WIRELESS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * NETCOMM WIRELESS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OFF
 * SUCH DAMAGE.
 *
 */

/*
The filter is designed with the following parameters:

    filtertype 	= 	Butterworth
    passtype 	= 	Lowpass
    ripple 	=
    order 	= 	1
    samplerate 	= 	1
    corner1 	= 	0.001
    corner2 	=
    adzero 	=
    logmin 	= 	-80

 Digital filter designed by mkfilter/mkshape/gencode   A.J. Fisher
   Command line: /www/usr/fisher/helpers/mkfilter -Bu -Lp -o 1 -a 1.0000000000e-03 0.0000000000e+00 -l

#define NZEROS 1
#define NPOLES 1
#define GAIN   3.193088390e+02

static float xv[NZEROS+1], yv[NPOLES+1];

static void filterloop()
  { for (;;)
      { xv[0] = xv[1];
        xv[1] = next input value / GAIN;
        yv[0] = yv[1];
        yv[1] =   (xv[0] + xv[1])
                     + (  0.9937364715 * yv[0]);
        next output value = yv[1];
      }
  }
*/

#define N_FILTERS 9

#define Q	30
/*#define K   (1 << (Q-1))*/
#define K	0x20000000

/*
 * multiply 2 Qx number
 * return value in Qx format
 */
static unsigned int multiply(unsigned int a, unsigned int b)
{
	unsigned int	result;
	unsigned long long int	temp;

	temp = (unsigned long long int)a * b;
	/* result is now in Q2x, convert back to Qx */
	/* rounding: mid values are rounded up */
	temp += K;
	/* convert to Qx */
	result = temp >> Q;

	return result;
}

#define Q14	14
/*#define K14	(1 << (Q14-1))*/
#define K14	0x2000
#define Q0_TO_Q14(x)	((x)<<Q14)
#define Q14_TO_Q0(x)	(((x)+K14)>>Q14)
/* convert Q14 to Q30: << (30-14) */
#define Q14_TO_Q30(x)	((x)<<16)
/* convert Q30 to Q14: (x+(1<<(30-14-1)))>>(30-14) */
#define Q30_TO_Q14(x)	(((x)+0x8000)>>16)

/* divide a number by 4096 (both in Q0), result in Q14 */
static unsigned int divide_4096_q14(unsigned int a)
{
	unsigned int result;
	unsigned long long int temp;
	/* convert a to Q28 */
	temp =(unsigned long long int) a << 28;
	/* mid values are rounded up: temp += 4096 in Q14/2 that is (1<<12<<14) >> 1 = 1<<25 */
	temp += 0x2000000;
	/* divide by 4096 in Q14 */
	result = temp>>26;
	return result;
}
/* multiply a number with 4096 in q14 */
static unsigned int multiply_4096_q14(unsigned int a)
{
	unsigned int       result;
	unsigned long long int  temp;
	/* that is a*4096, 4096=(1<<26) in Q14 */
	temp = (unsigned long long int)a << 26;
	/* Rounding; mid values are rounded up */
	temp += K14;
	/* convert back to Q14 */
	result = temp >> Q14;
	return result;
}

#define NZEROS 1
#define NPOLES 1
/*#define GAIN   3.193088390e+02*/
/* 1/GAIN = 0.003131764 = 0.003131764*(2^18) = 821 Q18
 * 			= 0.003131764*(2^30) = 3362706 Q30 */
#define INVERSE_GAIN 3362706
/* coefficient = 0.9937364715 = 0.9937364715*(2^18) = 260502 Q18
 * 				= 0.9937364715*(2^30) = 1067016412 Q30 */
#define COEFFICIENT 1067016412

static unsigned int xv[N_FILTERS][NZEROS+1], yv[N_FILTERS][NPOLES+1];
static int reset_index = 0;

/*
 * Filter function
 * Parameters:
 * 	index:	filter index
 * 	input:	intput value, must be in range 0-4500
 *
 * Return:	filtered value in Qx format, 0 if error
 */
static unsigned int filter_run(int index, unsigned int input)
{
	if (index>=N_FILTERS){
		return 0;
	}
	if (reset_index==index){
		/* starting state */
		xv[reset_index][1] = multiply(input,INVERSE_GAIN);
		yv[reset_index][1] = input;
		reset_index++;
		return input;
	}
	else if (reset_index<index){
		/* After resetting, index should run from 0 to 8.
		 * If this code branch executes, that means reseting is wrong. */
		return 0;
	}
	else{
		xv[index][0] = xv[index][1];
		xv[index][1] = multiply(input,INVERSE_GAIN);
		yv[index][0] = yv[index][1];
		yv[index][1] = (xv[index][0] + xv[index][1]) + multiply(COEFFICIENT, yv[index][0]);
		return yv[index][1];
	}
}

struct filter {
	/* 1: pending to run; 0: not pending */
	int pending;
	/* 0: matured; >0: not yet */
	int mature;
	unsigned int input;
	unsigned int output;
};
static struct filter filters[N_FILTERS];

/* reset filters */
void imx28_psm_filter_reset(void)
{
	int i;
	reset_index = 0;
	/* cancel all scheduled filters */
	for (i=1;i<N_FILTERS;i++){
		if (filters[i].pending){
			filters[i].pending = 0;
		}
	}
}

/*
 * Run filter set
 *
 * Parameters:
 * 	voltage_input:		input value, must be in range 0-4500
 * 	filtered_output:	filtered output if any. Must be pre-allocated.
 *
 * Return:
 * 	0: no output returned; 1: filtered output returned; -1: error
 */
int imx28_psm_filter(unsigned int voltage_input, unsigned int *filtered_output)
{
	int i;

	if (voltage_input<0 || voltage_input>4500 || !filtered_output){
		return -1;
	}
	/* iterate thru filter set, run matured one */
	for (i=0;i<N_FILTERS;i++){
		/* filter 0 always runs */
		if (i==0 || filters[i].pending){
			/* exact slot to run filter can be achieve by setting mature to 2^i in scheduling filters.
			 * however workload for each slot won't be balanced
			 * (i.e 1 slot run in some slots and 10 slots in some others).
			 * The tweak below (schedule: mature=(1<<(i+1))+i); run filter when mature==i)
			 * improves the workload balance: in average 2 filters run/slot, maximum 3 filters run per slot.
			 */
			if (filters[i].mature==i){
				if (i==0){
					/* divide by 4096 and convert to Q30 before set as input */
					filters[i].input = Q14_TO_Q30(divide_4096_q14(voltage_input));
				}
				/* run filter index i */
				filters[i].output = filter_run(i,filters[i].input);
				/* except filter 0, change to not pending state */
				if (i>0){
					filters[i].pending=0;
				}

				if (i==N_FILTERS-1){
					/* multiply filtered result with 4096 then return */
					*filtered_output = Q14_TO_Q0(multiply_4096_q14(Q30_TO_Q14(filters[i].output)));
					return 1;
				}
				else{ /* (i<N_FILTERS-1) */
					/* set output as input for higher rank filter */
					filters[i+1].input = filters[i].output;
					/* schedule next higher rank filter if necessary */
					if (!filters[i+1].pending){
						filters[i+1].pending = 1;
						filters[i+1].mature = (1<<(i+1))+i;
					}
				}
			}
			else{
				filters[i].mature--;
			}
		}
	}

	return 0;
}
