/*
 * Power source management and battery charger for devices based on i.MX28
 *
 * Copyright (C) 2015 NetComm Pty. Ltd.
 *
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/jiffies.h>
#include <linux/io.h>
#include <linux/sched.h>
#include <linux/clk.h>
#include <linux/suspend.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/of.h>
#include <linux/workqueue.h>
#include <linux/spinlock.h>
#include <linux/sort.h>
#include <linux/notifier.h>
#include <linux/usb/otg.h>

#include <mach/mxs.h>
#include <mach/mx28.h>

#include <mach/regs-common.h>
#include <mach/regs-power.h>
#include <mach/regs-lradc.h>
#include <mach/regs-rtc.h>
#include <mach/hardware.h>
#include <mach/irqs.h>

#include "imx28_psm_battery_io.h"

typedef enum {
	UNKNOWN,
	NOT_CHARGING,
	DISCHARGING,
	CHARGING,
	FULL,
} battery_charging_status_t;

typedef enum {
	DCDC_UNKNOWN,
	DCDC_POWERED_BY_BATTERY,
	DCDC_POWERED_BY_EXT_CHARGER,
	DCDC_POWERED_BY_USB_HOST
} power_state_t;

typedef enum {
	TEMP_UNKNOWN,
	TEMP_GOOD,
	TEMP_HOT_FOR_CHARGING,
	TEMP_COLD_FOR_CHARGING,
	TEMP_OVER_MAX,
	TEMP_UNDER_MIN,
} battery_temperature_status_t;

#define PSM_DRIVER_NAME "psm_batt"
#define BATTERY_VOLTAGE_CH	7
#define BATTERY_VOLTAGE_CH_IRQ	23
#define BATTERY_VOLT_LRADC_DELAY	3
#define BATTERY_MON_DELAY_VALUE 20

#define AC_IN_MON_LRADC_CHANNEL 1
/* use same delay as that for battery */
#define AC_IN_MON_LRADC_DELAY 3
/* unit of AC_IN_MON_LRADC_DELAY_VALUE: 0.5ms */
#define AC_IN_MON_LRADC_DELAY_VALUE 20
#define AC_IN_MON_THRESH_NUM THRESHOLD1

#define BATT_TEMP_MON_LRADC_CHANNEL 0
/* unit of BATT_MON_DELAY_VALUE: 0.5ms */
#define BATT_MON_DELAY_VALUE BATTERY_MON_DELAY_VALUE
#define BATT_TEMP_MON_THRESH_NUM THRESHOLD0

#define REPORT_VOLT_IN_MV

/* -------- module parameters --------- */

/* batt charging current in mA */
#define DEFAULT_BATT_CHARGE_CURRENT 320
#define MAX_BATT_CHARGE_CURRENT 780
#define MIN_BATT_CHARGE_CURRENT 10
static unsigned int param_batt_charging_current = DEFAULT_BATT_CHARGE_CURRENT;
module_param(param_batt_charging_current, uint, S_IRUGO);
MODULE_PARM_DESC(param_batt_charging_current, "Battery charging current in mA from "
		__MODULE_STRING(MIN_BATT_CHARGE_CURRENT) " to "
		__MODULE_STRING(MAX_BATT_CHARGE_CURRENT) ", default "
		__MODULE_STRING(DEFAULT_BATT_CHARGE_CURRENT));

/* stop charging current in mA */
#define DEFAULT_STOP_CHARGE_CURRENT 100
#define MAX_STOP_CHARGE_CURRENT 180
#define MIN_STOP_CHARGE_CURRENT 10
static unsigned int param_stop_charging_current = DEFAULT_STOP_CHARGE_CURRENT;
module_param(param_stop_charging_current, uint, S_IRUGO);
MODULE_PARM_DESC(param_stop_charging_current, "Current threshold in mA at which the battery charger signals to stop charging, from "
		__MODULE_STRING(MIN_STOP_CHARGE_CURRENT) " to "
		__MODULE_STRING(MAX_STOP_CHARGE_CURRENT) ", default "
		__MODULE_STRING(DEFAULT_STOP_CHARGE_CURRENT));

/* battery monitoring period in ms */
#define DEFAULT_BATT_MON_PERIOD 5000
#define MAX_BATT_MON_PERIOD 100000
#define MIN_BATT_MON_PERIOD 500
static unsigned int param_batt_mon_period = DEFAULT_BATT_MON_PERIOD;
module_param(param_batt_mon_period, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(param_batt_mon_period, "Battery monitoring period time in ms, from "
		__MODULE_STRING(MIN_BATT_MON_PERIOD) " to "
		__MODULE_STRING(MAX_BATT_MON_PERIOD) ", default "
		__MODULE_STRING(DEFAULT_BATT_MON_PERIOD));

/* Max battery voltage in mV */
#define DEFAULT_MAX_BATT_VOLT 4160
#define MAX_MAX_BATT_VOLT 4200
#define MIN_MAX_BATT_VOLT 3000
static unsigned int param_max_batt_volt = DEFAULT_MAX_BATT_VOLT;
module_param(param_max_batt_volt, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(param_max_batt_volt, "Max battery voltage, in mV, from "
		__MODULE_STRING(MIN_MAX_BATT_VOLT) " to "
		__MODULE_STRING(MAX_MAX_BATT_VOLT) ", default "
		__MODULE_STRING(DEFAULT_MAX_BATT_VOLT));

/* battery voltage reduction before recharging, in mV */
#define DEFAULT_BATT_REDUCTION_BEFORE_RECHARGE 30
#define MAX_BATT_REDUCTION_BEFORE_RECHARGE 1000
#define MIN_BATT_REDUCTION_BEFORE_RECHARGE 0
static unsigned int param_batt_reduction_rechrg = DEFAULT_BATT_REDUCTION_BEFORE_RECHARGE;
module_param(param_batt_reduction_rechrg, uint, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(param_batt_reduction_rechrg, "Battery voltage reduction before recharging, in mV, from "
		__MODULE_STRING(MIN_BATT_REDUCTION_BEFORE_RECHARGE) " to "
		__MODULE_STRING(MAX_BATT_REDUCTION_BEFORE_RECHARGE) ", default "
		__MODULE_STRING(DEFAULT_BATT_REDUCTION_BEFORE_RECHARGE));

#if defined(HW_EVT3)
/* threshold on battery temperature monitor voltage (in mV), with current 20uA, indicating over-heat for charging */
/* Below values refer to datasheet of Chip Type NTC Thermistor CN0603R104B4100JT */
/* 60 degree C, R=23.4988 KOhm */
#define DEFAULT_BATT_TEMP_VOLT_THRESH 470
/* 30 degree C, R=79.7955 KOhm */
#define MAX_BATT_TEMP_VOLT_THRESH 1596
/* 80 degree C, R=11.5632 KOhm */
#define MIN_BATT_TEMP_VOLT_THRESH 231

/* voltage threshold (in MV) corresponding to minimum temperature allowed for charging
 * 0 degree C, R = 346.6868 KOhm
 */
#define MIN_BATT_TEMP_CHARGING_MON_VOLT 6934
/* voltage threshold (in MV) corresponding to maximum safe temperature for battery
 * 80 degree C, R = 11.5632 KOhm
 */
#define MAX_BATT_TEMP_SAFE_MON_VOLT 231
/* voltage threshold (in MV) corresponding to minimum safe temperature for battery
 * -20 degree C, R = 1105.3301 KOhm
 */
#define MIN_BATT_TEMP_SAFE_MON_VOLT 22107
/* With the DIVIDE_BY_TWO option set, the maximum input voltage is VDDIO - 50mv */
#define MAX_INPUT_VOLT_BATT_TEMP_MON 3250
#else
/* threshold on battery temperature monitor voltage (in mV), with current 20uA, indicating over-heat for charging */
/* Below values refer to datasheet of Chip Type NTC Thermistor CN0603R104B4100JT */
/* 60 degree C, Rt=23.4988 KOhm, R_total = (Rt*100)/(Rt+100) = 19.027553304 KOhm; V=I*R */
#define DEFAULT_BATT_TEMP_VOLT_THRESH 381
/* 30 degree C, R=79.7955 KOhm, R_total =  */
#define MAX_BATT_TEMP_VOLT_THRESH 888
/* 80 degree C, R=11.5632 KOhm */
#define MIN_BATT_TEMP_VOLT_THRESH 207

/* voltage threshold (in MV) corresponding to minimum temperature allowed for charging
 * 0 degree C, R = 346.6868 KOhm
 */
#define MIN_BATT_TEMP_CHARGING_MON_VOLT 1552
/* voltage threshold (in MV) corresponding to maximum safe temperature for battery
 * 80 degree C, R = 11.5632 KOhm
 */
#define MAX_BATT_TEMP_SAFE_MON_VOLT 207
/* voltage threshold (in MV) corresponding to minimum safe temperature for battery
 * -20 degree C, R = 1105.3301 KOhm
 */
#define MIN_BATT_TEMP_SAFE_MON_VOLT 1834
/* 100KOhm is parallel to the thermistor --> Vmax = I*100KOhm */
#define MAX_INPUT_VOLT_BATT_TEMP_MON 2000
#endif

static unsigned int param_batt_temp_volt_thresh = DEFAULT_BATT_TEMP_VOLT_THRESH;
module_param(param_batt_temp_volt_thresh, uint, S_IRUGO);
MODULE_PARM_DESC(param_batt_temp_volt_thresh, "Threshold on battery temperature monitor voltage (in mV), with current 20uA, indicating over-heat, from "
		__MODULE_STRING(MIN_BATT_TEMP_VOLT_THRESH) " to "
		__MODULE_STRING(MAX_BATT_TEMP_VOLT_THRESH) ", default "
		__MODULE_STRING(DEFAULT_BATT_TEMP_VOLT_THRESH));

static bool param_debug = false;
module_param(param_debug, bool, S_IRUGO|S_IWUSR);
MODULE_PARM_DESC(param_debug, "Debug mode (1:on 0:off), default: 0(off)");

#define VDD5V_VBUS_SUSPENDED_MA	2
#define VDD5V_LOW_POWER_MA	100
#define VDD5V_HIGH_POWER_MA	500
#define VDD5V_FULL_POWER_MA	780

#define WAIT_AC_IN_MON_USB_MS 1
#define BATTERY_MONITOR_CONFIRM_STOP_PERIOD_MS 1
#define POWER_SUPPLY_WQ "imx28_power_supply_wq"

/* Battery protection against low temperature needs BATT_TEMP_MON connects properly,
 * otherwise if BATT_TEMP_MON didn't connect properly, its voltage would go to the highest value and
 * the driver would report Low temperature state.
 * Currently NTC-20 has issues related to High temperature and no requirement of protection
 * against Low temperature.
 * Hence, though protection against low temperature is implemented, the featured is commented out here,
 * unless there is requirement to enable it.
 */
#undef PROTECT_BATTERY_AGAINST_LOW_TEMP

#define INFERING_BATT_SOC
#define INIT_FILTER_BY_AVERAGE

#ifdef INFERING_BATT_SOC
/* ------------------ inferring battery state-of-charge ----------------- */
#ifdef INIT_FILTER_BY_AVERAGE
/* number of battery voltage samples for calculating median */
#define NUM_SAMPLES_BATT_VOLT 100
#define MEDIAN_OFFSET_BATT_VOLT 5
#define AVG_START_OFFSET_BATT_VOLT 60
#define AVG_NUM_SAMPLES_BATT_VOLT 20
#endif
/* period in ms of each sampling: 100Hz --> 10ms */
#define PERIOD_BATT_VOLT_SAMPLE_MS 10
/* min/max battery voltage if no external power supply connected
 * derived from battery's datasheet
 */
#define MIN_BATT_VOLT_NO_EXT_MV 3700
#define MAX_BATT_VOLT_NO_EXT_MV 4000
/* min/max battery voltage if external power supply connects but is not charging
 * derived from hardware
 */
#define MIN_BATT_VOLT_WITH_EXT_NOT_CHARGING_MV 3830
#define MAX_BATT_VOLT_WITH_EXT_NOT_CHARGING_MV 4130
/* min/max battery voltage if external power supply connects and is charging
 * derived from hardware
 */
#define MIN_BATT_VOLT_WITH_EXT_CHARGING_MV 3940
#define MAX_BATT_VOLT_WITH_EXT_CHARGING_MV 4200
#define MIN_BATT_VOLT_WITH_USB_HOST_MV 3780
#endif

/* battery safety timers' period */
#define BATT_TEMP_MON_PERIOD_MS	500
#define COOLING_OFF_CHARGER_CHECKING_PERIOD_MINS	5

#define USB_DRIVER_INFORMS_VBUS_DRAW

/* the structure contains central states and data of this driver */
struct imx28_psm_info {
	struct device *dev;
	void __iomem *base;

	struct power_supply batt;
	struct power_supply ext;

	volatile power_state_t power_state;

	volatile int checking_new_state;

	volatile battery_charging_status_t battery_charging_status;
	volatile bool power_5V_connected;
	volatile bool ac_connected;
	volatile bool battery_connected;
	volatile bool battery_overheat;
	battery_temperature_status_t battery_temperature_status;
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
	volatile bool vbus_draw_connected; /*apply for usb port with vbus draw after enumeration completed */
	uint32_t vbus_draw_current;
	struct usb_phy *usb_transceiver;
#endif
#ifdef INFERING_BATT_SOC
	/* inferred battery percents */
	int battery_percents;
	/* filtered battery voltage */
	unsigned int battery_voltage_filtered;
#ifdef INIT_FILTER_BY_AVERAGE
	/* battery voltage samples, used to calculate first value to pass to filters */
	uint32_t samples[NUM_SAMPLES_BATT_VOLT];
#endif
#endif

	uint32_t ac_in_mon_lradc;
	uint32_t ac_in_mon_volt;

	uint32_t batt_temp_mon_lradc;
	/* threshold on battery temperature monitor voltage (in mV) indicating over-heat */
	uint32_t batt_temp_mon_volt;

	int ext_source_sel_gpio;
#if defined(HW_EVT3)
	int batt_source_sel_gpio;
#endif

	uint32_t ilimit_4p2_chg;
	volatile int batt_volt_full; /* battery voltage at full, charger off, 5v connected */
	volatile int batt_volt_full_charger_on; /* battery voltage at full, charger on */

	struct workqueue_struct * power_supply_wqs;
	struct work_struct power_state_change_wq;
	struct delayed_work battery_monitor_wq;
#ifdef INFERING_BATT_SOC
	struct delayed_work battery_percents_wq;
	bool battery_sampling_active;
	bool reset_filter;
#endif
	/* timer checking battery temperature */
    struct timer_list battery_temperature_timer;
    /* timer to wait after charger is considered as being cooled off */
    struct timer_list cooling_off_charger_timer;
    /* flag "charger is cooled off", if it is true, charger can be considered to be turned on */
    bool cooled_off_charger;

	spinlock_t power_supply_changed_lock;
	spinlock_t power_state_changed_lock;
#ifdef INFERING_BATT_SOC
	spinlock_t battery_sampling_lock;
#endif
	spinlock_t battery_safety_lock;

	bool dcdc4p2_brownout;
	bool vdd5v_droop;

	volatile bool on;
};

static struct imx28_psm_info * pinfo = 0;

#define DEV_PRINT_MSG(...) if (param_debug)dev_info(info->dev, __VA_ARGS__)
#define DEV_PRINT_ERR(...) dev_err(info->dev, __VA_ARGS__)

/*
 * hand-over from battery to VDD5V
 */
static void imx28_psm_batt_battery_to_vdd5v_handover(struct imx28_psm_info *info){

	DEV_PRINT_MSG("Handling over from battery to VDD5V\n");

	/* enable the 4P2 LinReg and switch the DC-DC
	converter to use the 4P2 LinReg as the input */

	/* Enabling 4P2 LinReg */
	imx28_psm_batt_enable_4p2linreg(info->base);

	/* Charging 4P2 Capacitor*/
	imx28_psm_batt_charge_4p2_capacitor(info->base, info->ilimit_4p2_chg);

	/* Enabling 4P2 Input for DC-DC */
	imx28_psm_batt_enable_4p2_input_for_dcdc(info->base);

	/* make sure that dcdc use 4P2 as source will prevent current flowing from battery */
	/*
	 * Sets the trip point for the comparison between the DCDC_4P2 and BATTERY pin
	 * 0b00000 DCDC_4P2 pin >= 0.85 * BATTERY pin
	 */
	__raw_writel( __raw_readl(info->base + HW_POWER_DCDC4P2) & ~BM_POWER_DCDC4P2_CMPTRIP, info->base + HW_POWER_DCDC4P2);

	/* set HEADROOM_ADJ to make charging current to reach configured value. Reference: Freescale support forum. */
	__raw_writel( (__raw_readl(info->base + HW_POWER_5VCTRL) & (~BM_POWER_5VCTRL_HEADROOM_ADJ)) | BF_POWER_5VCTRL_HEADROOM_ADJ(4)  , info->base + HW_POWER_5VCTRL);
}

/*
 * hand-over from VDD5V to battery
 */
static void imx28_psm_batt_vdd5v_to_battery_handover(struct imx28_psm_info *info){
	int val;

	DEV_PRINT_MSG("Handling over from VDD5V to battery\n");
	/* turn off DCDC4P2 */
	val = __raw_readl(info->base + HW_POWER_DCDC4P2);
	val &= ~(BM_POWER_DCDC4P2_ENABLE_DCDC | BM_POWER_DCDC4P2_ENABLE_4P2);
	__raw_writel(val, info->base + HW_POWER_DCDC4P2);

	/* power down 4p2 and charger */
	__raw_writel(BM_POWER_5VCTRL_PWD_CHARGE_4P2, info->base + HW_POWER_5VCTRL_SET);

	/* Allow DCDC be to active when 5V is present. */
	__raw_writel(BM_POWER_5VCTRL_ENABLE_DCDC, info->base + HW_POWER_5VCTRL_SET);

	/*
	 * Sets the trip point for the comparison between the DCDC_4P2 and BATTERY pin
	 * 0b11111 DCDC_4P2 pin >= 1.05 * BATTERY pin
	 */
	__raw_writel( __raw_readl(info->base + HW_POWER_DCDC4P2) | BM_POWER_DCDC4P2_CMPTRIP, info->base + HW_POWER_DCDC4P2);
}

/* helper function to configure auxiliary source */
static void imx28_psm_batt_change_auxiliary_source(struct imx28_psm_info *info, power_state_t new_power_state)
{
	if (!info){
		return;
	}
	if (new_power_state == DCDC_POWERED_BY_EXT_CHARGER){
#if defined(HW_EVT3)
#define TIME_BATT_PLUS_RAMP_UP_MS 10
		/* change auxiliary source to ext */
		gpio_set_value(info->ext_source_sel_gpio,1);
		msleep(TIME_BATT_PLUS_RAMP_UP_MS);
		gpio_set_value(info->batt_source_sel_gpio,0);
#else
		gpio_direction_input(info->ext_source_sel_gpio);
#endif
	}
	else if (new_power_state == DCDC_POWERED_BY_USB_HOST || new_power_state == DCDC_POWERED_BY_BATTERY){
		/* change auxiliary source to battery */
		/* EVT3: battery switch is already turned on in irq handler */
#if defined(HW_EVT3)
		gpio_set_value(info->batt_source_sel_gpio,1);
		gpio_set_value(info->ext_source_sel_gpio,0);
#else
		gpio_direction_output(info->ext_source_sel_gpio,0);
#endif
	}
	else{
		DEV_PRINT_ERR( "Unexpected new power state %d\n",new_power_state);
	}
}

/* helper function to change to new power state */
static void imx28_psm_batt_change_new_power_state(struct imx28_psm_info *info, power_state_t new_power_state)
{
	if (!info){
		return;
	}
	if (new_power_state == DCDC_POWERED_BY_EXT_CHARGER || new_power_state == DCDC_POWERED_BY_USB_HOST){
		imx28_psm_batt_enable_5v_droop_irq(info->base, true );
		/* configure auxiliary power source */
		imx28_psm_batt_change_auxiliary_source(info, new_power_state);
		/* handover from battery to 5v */
		imx28_psm_batt_battery_to_vdd5v_handover(info);

		imx28_psm_batt_enable_battery_bo_irq(info->base, false );
		imx28_psm_batt_enable_4p2_bo_irq(info->base, true );

		info->power_state = new_power_state;

		DEV_PRINT_MSG("Power state %d --> %d.\n",DCDC_POWERED_BY_BATTERY, info->power_state);
	}
	else if (new_power_state == DCDC_POWERED_BY_BATTERY){
		/* configure auxiliary power source */
		imx28_psm_batt_change_auxiliary_source(info, new_power_state);
		/* handover from 5v to battery */
		imx28_psm_batt_vdd5v_to_battery_handover(info);

		imx28_psm_batt_enable_battery_bo_irq(info->base, true );
		imx28_psm_batt_enable_4p2_bo_irq(info->base, false );
		imx28_psm_batt_enable_5v_droop_irq(info->base, false );

		info->power_state = DCDC_POWERED_BY_BATTERY;
		DEV_PRINT_MSG( "Power state %d --> %d.\n",info->power_state, DCDC_POWERED_BY_BATTERY);
	}
	else{
		DEV_PRINT_ERR( "Unexpected new power state %d\n",new_power_state);
	}
	/* reset charger cool-off state */
	info->cooled_off_charger = true;
}

/*
 * work function of work queue power-state
 */
#define MAX_CHECKING_NEW_STATE_TIMES 20
#define USB_HOST_CURRENT_CONSUMPTION_BUFFER_MA	100
static void imx28_psm_batt_power_state_change(struct work_struct *wt){

	struct imx28_psm_info *info = container_of(wt, struct imx28_psm_info, power_state_change_wq);
	volatile bool current_5V_connected = false;
	volatile bool __ac_connected = false;
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
	volatile bool __vbus_draw_connected = false;
	volatile uint32_t __vbus_draw_current = 0;
#endif
	power_state_t new_power_state = DCDC_UNKNOWN;
	static power_state_t verifying_new_power_state = DCDC_UNKNOWN;
	unsigned long flags;
	int brownout_4p2=0, droop_5v=0, vdd5v_disconnected = 0;

	spin_lock(&info->power_state_changed_lock);

	/* waiting for updates from AC_IN_MON and USB */
	msleep(WAIT_AC_IN_MON_USB_MS);

	spin_lock_irqsave(&info->power_supply_changed_lock, flags);
	current_5V_connected = imx28_psm_batt_get_5v_present_flag(info->base);

	if (current_5V_connected){
		__ac_connected =  info->ac_connected;
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
		__vbus_draw_connected = info->vbus_draw_connected;
		__vbus_draw_current = info->vbus_draw_current;
#endif
		if (__ac_connected){
			/* AC Charger connected, use full power from VDD5V */
			info->ilimit_4p2_chg = VDD5V_FULL_POWER_MA;
			new_power_state = DCDC_POWERED_BY_EXT_CHARGER;
		}
		else{
			if (imx28_psm_batt_is_usb_data_signal_detected()){
				/* USB_DP and USB_DM connected, that means it is not a dedicated usb charger */
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
				/* we need to check usb enumeration and vbus draw current */
				if (!__vbus_draw_connected || __vbus_draw_current < USB_HOST_CURRENT_CONSUMPTION_BUFFER_MA){
					/* USB enumeration does not complete, use power from battery */
					info->ilimit_4p2_chg = 0;
					new_power_state = DCDC_POWERED_BY_BATTERY;
				}
				else{
					/* USB enumeration does complete, use power from vbus_draw_current */
					/* reserve some current buffer for other elements rather than 4P2 and charger */
					info->ilimit_4p2_chg = __vbus_draw_current-USB_HOST_CURRENT_CONSUMPTION_BUFFER_MA;
					new_power_state = DCDC_POWERED_BY_USB_HOST;
				}
#else
				info->ilimit_4p2_chg = VDD5V_HIGH_POWER_MA;
				new_power_state = DCDC_POWERED_BY_USB_HOST;
#endif
			}
			else{
				/* USB_DP and USB_DM not connected */
				/* that means either it is a dedicated usb charger or D+/D- event haven't come yet */
				/* checking will be conducted later */
				info->ilimit_4p2_chg = VDD5V_FULL_POWER_MA;
				new_power_state = DCDC_POWERED_BY_EXT_CHARGER;
			}
		}
	}
	else{
		info->ilimit_4p2_chg = 0;
		new_power_state = DCDC_POWERED_BY_BATTERY;
	}

	/* In the case of DCDC4P2 brown-out or VDD5V droop or VDD5V disconnection reported, switch to battery if possible */
	if (!info->power_5V_connected){
		vdd5v_disconnected = 1;
		info->power_5V_connected = true;
	}
	if (info->dcdc4p2_brownout){
		brownout_4p2 = 1;
		info->dcdc4p2_brownout = false;
	}
	if (info->vdd5v_droop){
		droop_5v = 1;
		info->vdd5v_droop = false;
	}
	if (brownout_4p2 || droop_5v || vdd5v_disconnected){
		DEV_PRINT_MSG("Force to battery state \n");
		new_power_state = DCDC_POWERED_BY_BATTERY;
	}
	/* check whether battery state is possible */
	if (new_power_state == DCDC_POWERED_BY_BATTERY && !info->battery_connected){
		new_power_state = DCDC_UNKNOWN;
	}

	spin_unlock_irqrestore(&info->power_supply_changed_lock, flags);
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
	DEV_PRINT_MSG("%s: new_power_state = %d, info->ilimit_4p2_chg = %d, current_5V_connected=%d, __ac_connected=%d, __vbus_draw_connected=%d, __vbus_draw_current=%d, signal=%d  \n",
			__FUNCTION__, new_power_state, info->ilimit_4p2_chg,
			current_5V_connected, __ac_connected, __vbus_draw_connected, __vbus_draw_current, imx28_psm_batt_is_usb_data_signal_detected());
#else
	DEV_PRINT_MSG("%s: new_power_state = %d, info->ilimit_4p2_chg = %d, current_5V_connected=%d, __ac_connected=%d, signal=%d  \n",
			__FUNCTION__, new_power_state, info->ilimit_4p2_chg,
			current_5V_connected, __ac_connected, imx28_psm_batt_is_usb_data_signal_detected());
#endif

	if (new_power_state != info->power_state){
		/* except DCDC_POWERED_BY_BATTERY, new state need to be verified */
		if (new_power_state != DCDC_POWERED_BY_BATTERY && new_power_state != verifying_new_power_state) {
			/* new state is changing, reset checking */
			verifying_new_power_state = new_power_state;
			info->checking_new_state = 1;
			queue_work(info->power_supply_wqs, &info->power_state_change_wq);
		}
		else if (new_power_state != DCDC_POWERED_BY_BATTERY && (new_power_state == verifying_new_power_state)  && (info->checking_new_state && info->checking_new_state < MAX_CHECKING_NEW_STATE_TIMES)){
			++info->checking_new_state;
			queue_work(info->power_supply_wqs, &info->power_state_change_wq);
		}
		else{
			info->checking_new_state = 0;
			verifying_new_power_state = DCDC_UNKNOWN;

			switch (info->power_state){
				case DCDC_UNKNOWN:
					if (new_power_state == DCDC_POWERED_BY_EXT_CHARGER || new_power_state == DCDC_POWERED_BY_USB_HOST){
						/* change to new state */
						imx28_psm_batt_change_new_power_state(info, new_power_state);
						/* schedule next call to monitor battery  */
						queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
					}
					else if (new_power_state == DCDC_POWERED_BY_BATTERY){
						/* change to new state */
						imx28_psm_batt_change_new_power_state(info, new_power_state);
						/* In the case of DCDC4P2 brown-out or VDD5V droop, schedule next power state check.
						 * If no 4p2 brown-out or 5v droop, next power state check will try to switch to ext power state
						 */
						if (brownout_4p2 || droop_5v || vdd5v_disconnected){
							queue_work(info->power_supply_wqs, &info->power_state_change_wq);
						}
						/* schedule next call to monitor battery charging */
						queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
					}
					break;

				case DCDC_POWERED_BY_BATTERY:

					if (new_power_state == DCDC_POWERED_BY_EXT_CHARGER || new_power_state == DCDC_POWERED_BY_USB_HOST){
						/* change to new state */
						imx28_psm_batt_change_new_power_state(info, new_power_state);
						/* schedule next call to monitor battery  */
						queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
					}
					break;

				case DCDC_POWERED_BY_EXT_CHARGER:
				case DCDC_POWERED_BY_USB_HOST:
					if (new_power_state == DCDC_POWERED_BY_BATTERY){
						/* change to new state */
						imx28_psm_batt_change_new_power_state(info, new_power_state);
						/* In the case of DCDC4P2 brown-out or VDD5V droop, schedule next power state check.
						 * If no 4p2 brown-out or 5v droop, next power state check will try to switch to ext power state
						 */
						if (brownout_4p2 || droop_5v || vdd5v_disconnected){
							queue_work(info->power_supply_wqs, &info->power_state_change_wq);
						}
						/* schedule next call to monitor battery charging to disable charging if any */
						queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
					}
					else if (new_power_state == DCDC_POWERED_BY_EXT_CHARGER || new_power_state == DCDC_POWERED_BY_USB_HOST){
						/* configure auxiliary power source */
						imx28_psm_batt_change_auxiliary_source(info, new_power_state);

						if (info->battery_connected){
							/* handing-over here is transited via battery */
							imx28_psm_batt_end_charging(info->base);
							imx28_psm_batt_vdd5v_to_battery_handover(info);
							msleep(10);
							imx28_psm_batt_battery_to_vdd5v_handover(info);
						}
						else{
							/* if no battery, change 4p2 ilimit */
							imx28_psm_batt_change_4p2_ilimit(info->base, info->ilimit_4p2_chg);
						}

						DEV_PRINT_MSG( "Power state %d --> %d.\n", info->power_state, new_power_state);
						info->power_state = new_power_state;
						/* reset charger cool-off state */
						info->cooled_off_charger = true;

						/* schedule next call to monitor battery charging to disable charging if any */
						queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
					}

					break;

				default:
					DEV_PRINT_ERR( "Power state should NOT be %d\n",info->power_state);
					break;
			}
			power_supply_changed(&info->ext);
		}
	}
	else{
		DEV_PRINT_MSG("Power state = %d does not change\n", info->power_state);

		if (info->checking_new_state){
			info->checking_new_state = 0;
		}
		if (verifying_new_power_state != DCDC_UNKNOWN){
			verifying_new_power_state = DCDC_UNKNOWN;
		}
	}
	spin_unlock(&info->power_state_changed_lock);
}

/*
 * IRQ Handler of DCDC4P2 brown-out
 */
static irqreturn_t imx28_psm_batt_irq_dcdc4p2_bo(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;
#ifdef HW_EVT3
	if (info->battery_connected){
		/* EVT3 HW needs batt source sel to be High immediately */
		gpio_set_value(info->batt_source_sel_gpio, 1);
	}
#endif
	__raw_writel(BM_POWER_CTRL_DCDC4P2_BO_IRQ, info->base + HW_POWER_CTRL_CLR);
	info->dcdc4p2_brownout = true;

	DEV_PRINT_MSG( "dcdc4p2 brownout interrupt occurred\n");

	imx28_psm_batt_enable_battery_bo_irq(info->base, true);
	imx28_psm_batt_enable_4p2_bo_irq(info->base, false);

	queue_work(info->power_supply_wqs, &info->power_state_change_wq);

	return IRQ_HANDLED;
}

/*
 * IRQ Handler of battery brown-out
 */
static irqreturn_t imx28_psm_batt_irq_batt_brnout(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;

	__raw_writel(BM_POWER_CTRL_BATT_BO_IRQ, info->base + HW_POWER_CTRL_CLR);
	DEV_PRINT_MSG( "battery brownout interrupt occurred\n");

	imx28_psm_batt_enable_battery_bo_irq(info->base, false);
	imx28_psm_batt_enable_4p2_bo_irq(info->base, true);

	return IRQ_HANDLED;
}

/*
 * IRQ Handler of VDDD brown-out
 */
static irqreturn_t imx28_psm_batt_irq_vddd_brnout(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;

	__raw_writel(BM_POWER_CTRL_VDDD_BO_IRQ, info->base + HW_POWER_CTRL_CLR);
	DEV_PRINT_MSG( "vddd brownout interrupt occurred\n");

	return IRQ_HANDLED;
}

/*
 * IRQ Handler of VDDA brown-out
 */
static irqreturn_t imx28_psm_batt_irq_vdda_brnout(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;

	__raw_writel(BM_POWER_CTRL_VDDA_BO_IRQ, info->base + HW_POWER_CTRL_CLR);
	DEV_PRINT_MSG( "vdda brownout interrupt occurred\n");

	return IRQ_HANDLED;
}

/*
 * IRQ Handler of VDDIO brown-out
 */
static irqreturn_t imx28_psm_batt_irq_vddio_brnout(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;

	__raw_writel(BM_POWER_CTRL_VDDIO_BO_IRQ, info->base + HW_POWER_CTRL_CLR);
	DEV_PRINT_MSG( "vddio brownout interrupt occurred\n");

	return IRQ_HANDLED;
}

/*
 * IRQ handler of VDD5V droop
 */
static irqreturn_t imx28_psm_batt_irq_vdd5v_droop(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;
#ifdef HW_EVT3
	if (info->battery_connected){
		/* EVT3 HW needs batt source sel to be High immediately */
		gpio_set_value(info->batt_source_sel_gpio, 1);
	}
#endif
	__raw_writel(BM_POWER_CTRL_VDD5V_DROOP_IRQ, info->base + HW_POWER_CTRL_CLR);
	info->vdd5v_droop = true;
	imx28_psm_batt_enable_5v_droop_irq(info->base, false );
	DEV_PRINT_MSG( "vdd5v droop interrupt occurred\n");

	queue_work(info->power_supply_wqs, &info->power_state_change_wq);

	return IRQ_HANDLED;
}

/*
 * IRQ handler of VDD5V connection/disconnection
 */
static irqreturn_t imx28_psm_batt_irq_vdd5v(int irq, void *cookie)
{
	struct imx28_psm_info *info = cookie;

	/*Clear the 5 V-interrupt status bit by writing 1 to the SCT clear address space of the IRQ status bit:
	HW_POWER_CTRL_CLR[VBUSVALID_IRQ] = 1 */
	__raw_writel(BM_POWER_CTRL_VBUSVALID_IRQ, info->base + HW_POWER_CTRL_CLR);
	__raw_writel(BM_POWER_CTRL_VDD5V_GT_VDDIO_IRQ, info->base + HW_POWER_CTRL_CLR);

	/* identify meaning of IRQ */
	if (__raw_readl( info->base + HW_POWER_CTRL) & BM_POWER_CTRL_POLARITY_VBUSVALID){
		/* detecting vdd5v connection */
		info->power_5V_connected = true;
	}
	else{
		/* detecting vdd5v disconnection */
		info->power_5V_connected = false;
	}

	/* configure polarity field based on current vdd5v connection status */
	if (imx28_psm_batt_get_5v_present_flag(info->base))
	{
		/* regarding auxiliary power source: do nothing, power state work will change auxiliary source */

		/*
		Set the polarity field to detect when the 5 V supply is removed:
		HW_POWER_CTRL[POLARITY_VBUSVALID] = 0 */
		__raw_writel( BM_POWER_CTRL_POLARITY_VBUSVALID, info->base + HW_POWER_CTRL_CLR);
	}
	else
	{
#ifdef HW_EVT3
		if (info->battery_connected){
			/* EVT3 HW needs batt source sel to be High immediately */
			gpio_set_value(info->batt_source_sel_gpio, 1);
		}
#endif
		/*
		Set the polarity field to detect when the 5 V supply is connected:
		HW_POWER_CTRL[POLARITY_VBUSVALID] = 1 */
		__raw_writel( BM_POWER_CTRL_POLARITY_VBUSVALID, info->base + HW_POWER_CTRL_SET);
	}

	DEV_PRINT_MSG( "VDD5V changed: info->power_5V_connected = %d \n",info->power_5V_connected);

	queue_work(info->power_supply_wqs, &info->power_state_change_wq);

	return IRQ_HANDLED;
}

/*
 * handler of battery temperature comparison
 */
static void imx28_psm_batt_temperature_threshold(void *cookie)
{
	struct imx28_psm_info *info = cookie;

	if (imx28_psm_batt_read_adc(BATT_TEMP_MON_LRADC_CHANNEL, true) >= info->batt_temp_mon_volt){
		/* normal temperature */
		info->battery_overheat = false;

		/* change polarity */
		imx28_psm_batt_toggle_lradc_thresh_polarity(BATT_TEMP_MON_THRESH_NUM, 0 /* detect low */);
	}
	else{
		/* overheat */
		info->battery_overheat = true;

		/* end charging */
		if (imx28_psm_batt_is_battery_charger_on(info->base)){
			imx28_psm_batt_end_charging(info->base);
		}

		/* change polarity */
		imx28_psm_batt_toggle_lradc_thresh_polarity(BATT_TEMP_MON_THRESH_NUM, 1 /* detect high */);
	}

	/* schedule battery_monitor */
	queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
}

/*
 * handler of AC_IN_MON
 */
static void imx28_psm_batt_ac_in_mon(void *cookie)
{
	struct imx28_psm_info *info = cookie;
	uint32_t ac_mon_mv;

	/* detect AC connection */
	ac_mon_mv = imx28_psm_batt_read_adc(AC_IN_MON_LRADC_CHANNEL, true);
	DEV_PRINT_MSG( "AC_IN_MON: %d mV\n",ac_mon_mv);
	if (ac_mon_mv > info->ac_in_mon_volt){
		info->ac_connected = true;
	}
	else{
		info->ac_connected = false;
	}

	if (info->ac_connected){
		/* power state work will change power source */

		/* now detect disconnection */
		imx28_psm_batt_toggle_lradc_thresh_polarity(AC_IN_MON_THRESH_NUM, DETECT_LOW);
	}
	else{

		if (imx28_psm_batt_get_5v_present_flag(info->base)){
			/* USB charger or USB host connect */

			/*
			 * EXT_SOURCE_SEL will be set to Input or output 0 in power state work later.
			 * Exception: AC removed and USB host connected. In that case EXT_SOURCE_SEL is set to 0. */
			if (imx28_psm_batt_is_usb_data_signal_detected()){
				/* not USB charger */
#if defined(HW_EVT3)
				if (info->battery_connected){
					/* EVT3 HW needs batt source sel to be High immediately */
					gpio_set_value(info->batt_source_sel_gpio, 1);
				}
				gpio_set_value(info->ext_source_sel_gpio, 0);
#else
				/* If battery is present, auxiliary will be fed by battery.
				 * Otherwise, in case of no battery, RESET pin will be Low so the device will be reseted,
				 * then the power state machine will prevent booting to kernel because of "No battery" status.
				 */
				gpio_direction_output(info->ext_source_sel_gpio,0);
#endif
			}
			else{
				/* USB charger still connects, do nothing */
			}
		}
		else{
#ifdef HW_EVT3
			if (info->battery_connected){
				/* EVT3 HW needs batt source sel to be High immediately */
				gpio_set_value(info->batt_source_sel_gpio, 1);
			}
#endif
		}

		/* now detect connection */
		imx28_psm_batt_toggle_lradc_thresh_polarity(AC_IN_MON_THRESH_NUM, DETECT_HIGH);
	}

	DEV_PRINT_MSG( "AC_IN_MON changed: info->ac_connected = %d \n",info->ac_connected);

	queue_work(info->power_supply_wqs, &info->power_state_change_wq);
}

/*
 * TODO: register_lradc_notifier and unregister_lradc_notifier will
 * be in header file of lradc driver
 */
extern int register_lradc_notifier(struct notifier_block *nb);
extern int unregister_lradc_notifier(struct notifier_block *nb);
static void imx28_psm_battery_sampling_triggered(void* data);
static int lradc_notifier_call(struct notifier_block *blk, unsigned long code, void *_param)
{
	if ( code == MX28_INT_LRADC_THRESH0 ){
		/* battery temperature */
		if (pinfo){
			imx28_psm_batt_temperature_threshold(pinfo);
		}
	}
	else if ( code == MX28_INT_LRADC_THRESH1 ){
		/* ac in mon */
		if (pinfo){
			imx28_psm_batt_ac_in_mon(pinfo);
		}
	}
	else if (code == BATTERY_VOLTAGE_CH_IRQ ){
		/* battery voltage conversion */
		if (pinfo){
			imx28_psm_battery_sampling_triggered(pinfo);
		}
	}

	return NOTIFY_OK;
}

static struct notifier_block lradc_notifier_block = {
		.notifier_call = lradc_notifier_call,
};

/*
 * 5V Power properties
 */
static enum power_supply_property power_props[] = {
	POWER_SUPPLY_PROP_ONLINE,
};

static int imx28_psm_batt_power_get_property(struct power_supply *psy,
	enum power_supply_property psp,
	union power_supply_propval *val)
{
	struct imx28_psm_info *info = container_of(psy, struct imx28_psm_info, ext);

	switch (psp) {
		case POWER_SUPPLY_PROP_ONLINE:
			val->intval = (info->power_state == DCDC_POWERED_BY_EXT_CHARGER || info->power_state == DCDC_POWERED_BY_USB_HOST);
			break;
		default:
			return -EINVAL;
	}

	return 0;
}
/*
 * Battery properties
 */
static enum power_supply_property battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_TEMP_ALERT_MIN,
	POWER_SUPPLY_PROP_TEMP_ALERT_MAX,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
#ifdef INFERING_BATT_SOC
	POWER_SUPPLY_PROP_CAPACITY,
#endif
};

static int imx28_psm_batt_battery_get_property(struct power_supply *psy,
	enum power_supply_property psp,
	union power_supply_propval *val)
{
	struct imx28_psm_info *info = container_of(psy, struct imx28_psm_info, batt);

	switch (psp) {
		case POWER_SUPPLY_PROP_STATUS:
			/* if battery percents has not been available, set status to Unknown */
			if (info->battery_percents < 0){
				val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
			}
			else{
				switch (info->battery_charging_status)
				{
					case UNKNOWN:
						val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
						break;

					case NOT_CHARGING:
						val->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
						break;

					case DISCHARGING:
						val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
						break;

					case CHARGING:
						val->intval = POWER_SUPPLY_STATUS_CHARGING;
						break;

					case FULL:
						val->intval = POWER_SUPPLY_STATUS_FULL;
						break;

					default:
						break;
				}
			}
			break;

		case POWER_SUPPLY_PROP_PRESENT:
			/* is battery present */
			val->intval = info->battery_connected?1:0;
			break;

		case POWER_SUPPLY_PROP_HEALTH:
			if (info->battery_connected){
				if (info->battery_overheat || info->battery_temperature_status==TEMP_HOT_FOR_CHARGING || info->battery_temperature_status==TEMP_OVER_MAX){
					val->intval = POWER_SUPPLY_HEALTH_OVERHEAT;
				}
				else if (info->battery_temperature_status==TEMP_COLD_FOR_CHARGING || info->battery_temperature_status==TEMP_UNDER_MIN){
					val->intval = POWER_SUPPLY_HEALTH_COLD;
				}
				else if (info->battery_temperature_status==TEMP_GOOD){
					val->intval = POWER_SUPPLY_HEALTH_GOOD;
				}
				else{
					val->intval = POWER_SUPPLY_HEALTH_UNKNOWN;
				}
			}
			else{
				val->intval = POWER_SUPPLY_HEALTH_UNKNOWN;
			}

			break;

		case POWER_SUPPLY_PROP_TEMP_ALERT_MIN:
			if (info->battery_connected){
				if (info->battery_temperature_status==TEMP_UNDER_MIN){
					val->intval = 1;
				}
				else{
					val->intval = 0;
				}
			}
			else{
				val->intval = 0;
			}

			break;

		case POWER_SUPPLY_PROP_TEMP_ALERT_MAX:
			if (info->battery_connected){
				if (info->battery_temperature_status==TEMP_OVER_MAX){
					val->intval = 1;
				}
				else{
					val->intval = 0;
				}
			}
			else{
				val->intval = 0;
			}

			break;

		case POWER_SUPPLY_PROP_VOLTAGE_NOW:
#ifdef REPORT_VOLT_IN_MV
			val->intval = imx28_psm_batt_get_battery_voltage(info->base);
#else
			/* uV */
			val->intval = imx28_psm_batt_get_battery_voltage(info->base) * 1000;
#endif
			break;
#ifdef INFERING_BATT_SOC
		case POWER_SUPPLY_PROP_CAPACITY:
			if (info->battery_connected){
				if (info->battery_percents > 0){
					val->intval = info->battery_percents;
				}
				else{
					val->intval = 0;
				}
			}
			else{
				val->intval = 0;
			}
			break;
#endif
		default:
			return -EINVAL;
	}

	return 0;
}

/* enable battery charging process */
static void imx28_psm_batt_enable_charging_process(struct imx28_psm_info *info)
{
	/*
	1. Determine the charge current and configure the charger setting:
	The maximum current is determined by the battery manufacturer and is provided in the battery data
	sheet. If the charge current is set at 600 mA, then set
	HW_POWER_CHARGE[BATTCHRG_I] = 0x30
	 */
	imx28_psm_batt_set_charge_current(info->base, param_batt_charging_current);
	/*
	2. Determine the stop current and configure the charger setting:
	This value is determined by the battery manufacturer. It is typically 10% of the maximum charge
	current and is provided in the battery data sheet. For this example, use 60 mA as the stop charge
	current, as the charge current is set at 600 mA (10% of the maximum charge current).
	Therefore, set HW_POWER_CHARGE[STOP_ILIMIT] = 0x5
	 */
	imx28_psm_batt_set_stop_current(info->base, param_stop_charging_current);
	/*
	3. Follow the step mentioned in Section 5.1.3, “Enabling Battery Charger,” to enable the battery
	charger
	 */
	if (imx28_psm_batt_enable_battery_charger(info->base))
	{
		/*
		4. Monitor the charger status flag to determine if the charger has reached the stop current threshold:
		This should be monitored for several times as the DC-DC converter can cause spurious results to
		this flag

		5. End the charging process by disabling the battery charger. This is accomplished by setting the
		power down charger bit:
		HW_POWER_CHARGE[PWD_CHARGE] = 1
		 */
		/* Step 4 & 5 is done by battery_monitor wq */

		/* schedule long charging checking timer */
		queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
	}
	else
	{
		DEV_PRINT_ERR("unable to enable battery charger \n");
	}
}

#define WAIT_POWER_STATE_VERIFIED_MS 100
#define VERIFYING_RECHARGING_VOLT_COUNTER 30

/*
 * If battery is not present, voltage at battery thermistor monitoring line is at maximum value
 * Requirement: ADC for measuring battery thermistor monitoring line must be already setup.
 */
static bool imx28_psm_batt_check_batt_present_by_temp_mon(struct imx28_psm_info *info)
{
	int batt_temp_volt;
	batt_temp_volt = imx28_psm_batt_read_adc(BATT_TEMP_MON_LRADC_CHANNEL, true);
	DEV_PRINT_MSG("Battery temperature monitoring volt = %dmV\n", batt_temp_volt);
	if (batt_temp_volt >= MAX_INPUT_VOLT_BATT_TEMP_MON){
		return false;
	}
	else{
		return true;
	}
}

/* check battery connection status */
static void imx28_psm_batt_check_battery_connected(struct imx28_psm_info *info)
{
	if (!info){
		return;
	}
	info->battery_connected =  imx28_psm_batt_check_batt_present_by_temp_mon(info);
}

/*
 * TODO: imx28_psm_batt_enable_channel_irq should be in header file of lradc driver
 */
extern void imx28_psm_batt_enable_channel_irq(int channel, bool enable);
/* work function of battery monitor work queue
 * the only place which can change battery charging status */
static void imx28_psm_batt_battery_monitor(struct work_struct *wt)
{
	struct imx28_psm_info *info = container_of(to_delayed_work(wt), struct imx28_psm_info, battery_monitor_wq);
	static int confirm_stop_limit = 0;
	bool needMonitorStopLimit = false;
	static int verifying_recharging = 0;
	static battery_temperature_status_t pre_batt_temp_status = TEMP_UNKNOWN;
	battery_temperature_status_t battery_temperature_status;
	bool cooled_off_charger;

	spin_lock(&info->power_state_changed_lock);

	/* if checking new state, do nothing */
	if (info->checking_new_state){
		spin_unlock(&info->power_state_changed_lock);
		DEV_PRINT_MSG("%s: power state is being verified. Return. %d\n",__FUNCTION__,info->power_state);
		if (info->on){
			queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, msecs_to_jiffies(WAIT_POWER_STATE_VERIFIED_MS));
		}
		return;
	}

	imx28_psm_batt_check_battery_connected(info);

	if (info->battery_connected)
	{
		DEV_PRINT_MSG("Battery is detected\n");

		spin_lock_bh(&info->battery_safety_lock);
		battery_temperature_status = info->battery_temperature_status;
		cooled_off_charger = info->cooled_off_charger;
		spin_unlock_bh(&info->battery_safety_lock);

		DEV_PRINT_MSG("Battery temperature monitor voltage = %u mV, battery_temperature_status=%d,cooled_off_charger=%d  \n",
				imx28_psm_batt_read_adc(BATT_TEMP_MON_LRADC_CHANNEL, true),battery_temperature_status,cooled_off_charger);

		if (info->power_state == DCDC_POWERED_BY_EXT_CHARGER || info->power_state == DCDC_POWERED_BY_USB_HOST){

			if (info->battery_overheat || battery_temperature_status != TEMP_GOOD){
				/* turn off charger and set status */
				/* end charging */
				if (imx28_psm_batt_is_battery_charger_on(info->base)){
					imx28_psm_batt_end_charging(info->base);
				}

				/* if temperature state just changes from GOOD, set cooled off timer
				 * to avoid continuously charging and discharging when temperature is just around the threshold */
				if (pre_batt_temp_status == TEMP_GOOD){
					spin_lock_bh(&info->battery_safety_lock);
					info->cooled_off_charger = false;
					if (info->on){
						if (!timer_pending(&info->cooling_off_charger_timer)){
							info->cooling_off_charger_timer.expires = round_jiffies(jiffies + (COOLING_OFF_CHARGER_CHECKING_PERIOD_MINS * 60 * HZ));
							add_timer(&info->cooling_off_charger_timer);
						}
						else{
							mod_timer(&info->cooling_off_charger_timer, round_jiffies(jiffies + (COOLING_OFF_CHARGER_CHECKING_PERIOD_MINS * 60 * HZ)));
						}
					}
					spin_unlock_bh(&info->battery_safety_lock);
				}

				if (info->battery_charging_status != NOT_CHARGING){
					info->battery_charging_status = NOT_CHARGING;
				}

				if (confirm_stop_limit > 0){
					confirm_stop_limit = 0;
				}
				if (needMonitorStopLimit){
					needMonitorStopLimit = false;
				}
			}
			else{
				if (imx28_psm_batt_is_battery_charger_on(info->base)){
					if (info->battery_charging_status != CHARGING){
						info->battery_charging_status = CHARGING;
					}
					/*
					4. Monitor the charger status flag to determine if the charger has reached the stop current threshold:
					This should be monitored for several times as the DC-DC converter can cause spurious results to
					this flag

					5. End the charging process by disabling the battery charger. This is accomplished by setting the
					power down charger bit:
					HW_POWER_CHARGE[PWD_CHARGE] = 1
					 */

					if (imx28_psm_batt_is_charging_current_below_stop_limit(info->base))
					{
						/* test 10 times */
						if (confirm_stop_limit < 10)
						{
							DEV_PRINT_MSG("charging current below stop limit, checking\n");
							confirm_stop_limit++;

							if (!needMonitorStopLimit){
								needMonitorStopLimit = true;
							}
						}
						else
						{
							/* first read voltage at full when charger is on, verified after detecting battery */
							info->batt_volt_full_charger_on = imx28_psm_batt_get_battery_voltage(info->base);

							/* special case, check battery presence now */
							info->battery_connected = imx28_psm_batt_check_batt_present_by_temp_mon(info);
							if (!info->battery_connected){
								/* no battery, reset batt_volt_full_charger_on */
								info->batt_volt_full_charger_on = 0;
							}

							/* stop limit confirmed, stop charging now */
							imx28_psm_batt_end_charging(info->base);

							/* sleep for awhile for ADC update */
#define WAITING_ADC_UPDATE_MS	20
							msleep(WAITING_ADC_UPDATE_MS);

							info->battery_connected = imx28_psm_batt_check_batt_present_by_temp_mon(info);

							DEV_PRINT_MSG("charging current stop-limit is confirmed, stop charging now\n");

							if (info->battery_connected){
								if (imx28_psm_batt_get_battery_voltage(info->base) >= (param_max_batt_volt - param_batt_reduction_rechrg)){

									info->battery_charging_status = FULL;

									info->batt_volt_full = imx28_psm_batt_get_battery_voltage(info->base);
									DEV_PRINT_MSG("Battery is full, batt_volt_full_charger_on=%d, batt_volt_full=%d \n",info->batt_volt_full_charger_on, info->batt_volt_full);
								}
								else{
									info->battery_charging_status = NOT_CHARGING;
								}

								/* set cooled off timer
								 * to avoid continuously charging and discharging */
								spin_lock_bh(&info->battery_safety_lock);
								info->cooled_off_charger = false;
								if (info->on){
									if (!timer_pending(&info->cooling_off_charger_timer)){
										info->cooling_off_charger_timer.expires = round_jiffies(jiffies + (COOLING_OFF_CHARGER_CHECKING_PERIOD_MINS * 60 * HZ));
										add_timer(&info->cooling_off_charger_timer);
									}
									else{
										mod_timer(&info->cooling_off_charger_timer, round_jiffies(jiffies + (COOLING_OFF_CHARGER_CHECKING_PERIOD_MINS * 60 * HZ)));
									}
								}
								spin_unlock_bh(&info->battery_safety_lock);
							}
							else{
								info->battery_charging_status = UNKNOWN;
								info->batt_volt_full_charger_on = 0;
								info->batt_volt_full = param_max_batt_volt;

								info->cooled_off_charger = true;
							}

							needMonitorStopLimit = false;
							confirm_stop_limit = 0;

							DEV_PRINT_MSG("%s %d:batt voltage=%d, chrgsts == 0 here 0x%x\n",__FUNCTION__,__LINE__,imx28_psm_batt_get_battery_voltage(info->base),__raw_readl( info->base + HW_POWER_STS ));
						}
					}
					else
					{
						DEV_PRINT_MSG("charging going on\n");
						if (confirm_stop_limit > 0){
							confirm_stop_limit = 0;
						}
						if (needMonitorStopLimit){
							needMonitorStopLimit = false;
						}
					}
				}
				else {
					DEV_PRINT_MSG("charger is off here\n");
					/* charger is off here */
					if (confirm_stop_limit > 0){
						confirm_stop_limit = 0;
					}
					if (needMonitorStopLimit){
						needMonitorStopLimit = false;
					}
					/* if battery voltage reduction > param_batt_reduction_rechrg, then turn on charger */
					if (imx28_psm_batt_get_battery_voltage(info->base)< (param_max_batt_volt - param_batt_reduction_rechrg)){
						/* re-enable charger after full state need verifying */
						if (info->battery_charging_status == FULL){
							verifying_recharging++;
							DEV_PRINT_MSG("battery volt's below recharge limit, verifying_recharging=%d\n",verifying_recharging);
							if (verifying_recharging >= VERIFYING_RECHARGING_VOLT_COUNTER){
								if (cooled_off_charger){
									DEV_PRINT_MSG("re-enable charging\n");
									imx28_psm_batt_enable_charging_process(info);
									verifying_recharging = 0;
								}
								else{
									verifying_recharging = 0;
								}
								/* battery_charging_status will be changed next run */
							}
						}
						else{
							if (cooled_off_charger){
								DEV_PRINT_MSG("re-enable charging\n");
								imx28_psm_batt_enable_charging_process(info);

								if (verifying_recharging){
									verifying_recharging = 0;
								}
							}
						}
					}
					else{

						if (verifying_recharging){
							verifying_recharging = 0;
						}

						if (info->battery_charging_status != FULL){
							info->battery_charging_status = FULL;
						}
					}
				}
			}
		}
		else{

			if (confirm_stop_limit > 0){
				confirm_stop_limit = 0;
			}
			if (needMonitorStopLimit){
				needMonitorStopLimit = false;
			}

			if (verifying_recharging){
				verifying_recharging = 0;
			}

			if (info->battery_charging_status != DISCHARGING){
				DEV_PRINT_MSG("end charging & battery_charging_status becomes DISCHARGING\n");
				imx28_psm_batt_end_charging(info->base);
				info->battery_charging_status = DISCHARGING;
			}
			spin_lock_bh(&info->battery_safety_lock);
			/* delete cooling off */
			if (timer_pending(&info->cooling_off_charger_timer)){
				del_timer(&info->cooling_off_charger_timer);
			}
			info->cooled_off_charger = true;
			spin_unlock_bh(&info->battery_safety_lock);
		}

		/* schedule battery temperature checking timer */
		if (info->on){
			spin_lock_bh(&info->battery_safety_lock);
			if (!timer_pending(&info->battery_temperature_timer)){
				info->battery_temperature_timer.expires = jiffies + msecs_to_jiffies(BATT_TEMP_MON_PERIOD_MS);
				add_timer(&info->battery_temperature_timer);
			}
			spin_unlock_bh(&info->battery_safety_lock);
		}

		/* inform user-space */
		if (pre_batt_temp_status != battery_temperature_status){
			power_supply_changed(&info->batt);
			pre_batt_temp_status = battery_temperature_status;
		}
	}
	/* if battery not present */
	else{

		DEV_PRINT_MSG("battery not present here \n");

		if (confirm_stop_limit > 0){
			confirm_stop_limit = 0;
		}
		if (needMonitorStopLimit){
			needMonitorStopLimit = false;
		}

		if (verifying_recharging){
			verifying_recharging = 0;
		}

		/* reset info->batt_volt_full */
		if (info->batt_volt_full != param_max_batt_volt){
			info->batt_volt_full = param_max_batt_volt;
		}
		if (info->batt_volt_full_charger_on){
			info->batt_volt_full_charger_on = 0;
		}

		if (info->battery_charging_status != UNKNOWN){
			DEV_PRINT_MSG("end charging & battery_charging_status becomes NOT_CHARGING\n");
			imx28_psm_batt_end_charging(info->base);
			info->battery_charging_status = UNKNOWN;
		}

		/* delete timers */
		spin_lock_bh(&info->battery_safety_lock);
		if (timer_pending(&info->battery_temperature_timer)){
			del_timer(&info->battery_temperature_timer);
		}
		if (timer_pending(&info->cooling_off_charger_timer)){
			del_timer(&info->cooling_off_charger_timer);
		}
		info->cooled_off_charger = true;
		spin_unlock_bh(&info->battery_safety_lock);

		pre_batt_temp_status = TEMP_UNKNOWN;
	}

	/* schedule next call to monitor battery */
	if (info->on){
		if (needMonitorStopLimit){
			queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, msecs_to_jiffies(BATTERY_MONITOR_CONFIRM_STOP_PERIOD_MS));
		}
		else{
			queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, msecs_to_jiffies(param_batt_mon_period));
		}

		/* if battery connected, activate battery sampling if necessary */
		if (info->battery_connected && !info->battery_sampling_active){
			info->battery_sampling_active = true;
			imx28_psm_batt_enable_channel_irq(BATTERY_VOLTAGE_CH, true);
		}
		else if ((!info->battery_connected) && info->battery_sampling_active){
			info->battery_sampling_active = false;
			imx28_psm_batt_enable_channel_irq(BATTERY_VOLTAGE_CH, false);
			/* schedule battery percents wq, which will inform user-space */
			queue_delayed_work(info->power_supply_wqs, &info->battery_percents_wq, 0);
		}
	}
	else{
		if (info->battery_charging_status == CHARGING){
			imx28_psm_batt_end_charging(info->base);
			info->battery_charging_status = NOT_CHARGING;
		}
		/* disable battery voltage channel irq any way */
		imx28_psm_batt_enable_channel_irq(BATTERY_VOLTAGE_CH, false);
	}
	spin_unlock(&info->power_state_changed_lock);

	/* power_supply_changed(&info->batt) will be called in battery percents wq */
}

#ifdef INFERING_BATT_SOC
#define MAX_DIFF_BATT_PERCENTS 3
extern void imx28_psm_filter_reset(void);
extern int imx28_psm_filter(unsigned int voltage_input, unsigned int *filtered_output);
#ifdef INIT_FILTER_BY_AVERAGE
/* compare parameters */
static int imx28_psm_batt_compare(const void *a, const void *b)
{
	return *(int *)a - *(int *)b;
}
/* calculate average value of an array */
static int imx28_psm_batt_average(int *a, int n)
{
	int i,s = 0;
	for (i = 0; i<n; i++){
		s += *(a + i);
	}
	return s/n;
}
#endif
#define Q10	10
/*#define K10	(1 << (Q10-1))*/
#define K10	0x200
#define Q0_TO_Q10(x)	((x)<<Q10)
#define Q10_TO_Q0(x)	(((x)+K10)>>Q10)
/* divide a/b in Q10 */
/* use only int to avoid division in long long int --> overflow if a >= 2^12 in Q0 */
static unsigned int divide_q10(unsigned int a, unsigned int b){
	unsigned int result;
	unsigned int temp;

	temp = a << Q10; /* overflow if a >= 2^12 in Q0 */
	/* mid values are rounded up. */
	temp += (b>>1);
	result = temp/b;
	return result;
}
/* multiply in Q10 */
static unsigned int multiply_q10(unsigned int a, unsigned int b){
	unsigned int       result;
	unsigned long long int  temp;
	temp = (unsigned long long int)a * b;
	/* mid values are rounded up */
	temp += K10;
	/* convert back to Q10 */
	result = temp >> Q10;
	return result;
}
#define MAX_VALID_BATT_VOLT_MS	4500
#define MIN_VALID_BATT_VOLT_MS	0
/* work function of battery sampling work queue
 * sampling battery voltage, filter, and infer battery percentage */
static void imx28_psm_batt_battery_percents(struct work_struct *wt)
{
	struct imx28_psm_info *info = container_of(to_delayed_work(wt), struct imx28_psm_info, battery_percents_wq);
	static int raw_percents;
	unsigned int filtered_volt;
	static power_state_t power_state = DCDC_UNKNOWN;
	static battery_charging_status_t battery_charging_status = UNKNOWN;
	static uint32_t max_volt, min_volt;
	static bool need_recalculate_max_min = false;
	static bool need_notify_user_space = false;
	static unsigned int inverse_max_min_delta = 0;

	unsigned long flags;

	spin_lock(&info->power_state_changed_lock);

	/* if checking new state, do nothing */
	if (info->checking_new_state){
		spin_unlock(&info->power_state_changed_lock);
		if (info->on){
			queue_delayed_work(info->power_supply_wqs, &info->battery_percents_wq, msecs_to_jiffies(WAIT_POWER_STATE_VERIFIED_MS));
		}

		return;
	}

	imx28_psm_batt_check_battery_connected(info);

	if (!info->battery_connected){
		/* reset internal state */
		info->battery_percents = -1;
		power_state = DCDC_UNKNOWN;
		battery_charging_status = UNKNOWN;

		/* reset filter */
		spin_lock_irqsave(&info->battery_sampling_lock, flags);
		info->reset_filter = true;
		info->battery_voltage_filtered = 0;
		spin_unlock_irqrestore(&info->battery_sampling_lock, flags);

		/* schedule battery monitor wq now */
		queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
		/* notify user-space */
		power_supply_changed(&info->batt);
		DEV_PRINT_MSG( "Battery is absent.\n");
	}
	else{
		/* reset sampling if power state or battery status changes */
		if ( power_state != info->power_state || battery_charging_status != info->battery_charging_status ){
			power_state = info->power_state;
			battery_charging_status = info->battery_charging_status;

			need_recalculate_max_min = true;

			/* reset filter */
			spin_lock_irqsave(&info->battery_sampling_lock, flags);
			info->reset_filter = true;
			spin_unlock_irqrestore(&info->battery_sampling_lock, flags);

			if ( ((power_state == DCDC_POWERED_BY_EXT_CHARGER) || (power_state == DCDC_POWERED_BY_USB_HOST)) && battery_charging_status == FULL){
				info->battery_percents = 100;
			}
			/* notify user-space */
			power_supply_changed(&info->batt);
			/* notify user-space later when battery SOC is available in next run */
			need_notify_user_space = true;

			goto exit;
		}

		if (power_state == DCDC_UNKNOWN){
			goto exit;
		}

		/* get filtered value */
		spin_lock_irqsave(&info->battery_sampling_lock, flags);
		filtered_volt = info->battery_voltage_filtered;
		spin_unlock_irqrestore(&info->battery_sampling_lock, flags);

		if (filtered_volt>0 && filtered_volt<=MAX_VALID_BATT_VOLT_MS){
			/* got valid filtered value, now infer battery percents */

			if (need_recalculate_max_min){
				int current_battery_percents = info->battery_percents;
				/* calculate new max & min voltage */
				switch (power_state){
				case DCDC_POWERED_BY_BATTERY:
					/* discharging, min voltage = 3700, calculate max volt */
					min_volt = MIN_BATT_VOLT_NO_EXT_MV;
					if (current_battery_percents > 0 && filtered_volt>min_volt){
						/* Vmax = Vmin + ((volt - Vmin)/percents)*100 */
						max_volt = min_volt + Q10_TO_Q0(multiply_q10(Q0_TO_Q10(100),
								divide_q10(Q0_TO_Q10(filtered_volt - min_volt),Q0_TO_Q10(current_battery_percents))));
					}
					else if (current_battery_percents <= 0 || filtered_volt<=min_volt){
						max_volt = MAX_BATT_VOLT_NO_EXT_MV;
					}

					break;

				case DCDC_POWERED_BY_EXT_CHARGER:
					switch (battery_charging_status){
					case CHARGING:
						/* fix max voltage, calculate min voltage */
						if (info->batt_volt_full_charger_on > 0){
							max_volt = info->batt_volt_full_charger_on;
						}
						else{
							max_volt = MAX_BATT_VOLT_WITH_EXT_CHARGING_MV;
						}

						if (filtered_volt >= max_volt){
							if (filtered_volt > MAX_BATT_VOLT_WITH_EXT_CHARGING_MV){
								DEV_PRINT_ERR( "filtered_volt = %u !\n",filtered_volt);
								filtered_volt = MAX_BATT_VOLT_WITH_EXT_CHARGING_MV;
							}
							max_volt = MAX_BATT_VOLT_WITH_EXT_CHARGING_MV;
						}

						/* Vmin = (percents*Vmax - 100*x)/(percents - 100) */
						/* Vmin = Vmax - 100*(Vmax-volt)/(100-percents) */
						if ((current_battery_percents > 0) && (current_battery_percents < 100) && (max_volt > filtered_volt)){
							min_volt = max_volt - Q10_TO_Q0(multiply_q10(Q0_TO_Q10(100),
									divide_q10(Q0_TO_Q10(max_volt-filtered_volt), Q0_TO_Q10(100-current_battery_percents)) ));
						}
						else if (filtered_volt>0 &&
								((current_battery_percents == 0) || (current_battery_percents < 0 && filtered_volt<MIN_BATT_VOLT_WITH_EXT_CHARGING_MV))){
							min_volt = filtered_volt;
						}
						else{
							min_volt = MIN_BATT_VOLT_WITH_EXT_CHARGING_MV;
						}

						DEV_PRINT_MSG( "min_volt=%d ", min_volt);

						break;
					case FULL:
					case NOT_CHARGING:
					default:
						min_volt = MIN_BATT_VOLT_WITH_EXT_NOT_CHARGING_MV;
						if (current_battery_percents > 0  && filtered_volt>min_volt){
							/* Vmax = Vmin + ((volt - Vmin)/percents)*100 */
							max_volt = min_volt + Q10_TO_Q0(multiply_q10(Q0_TO_Q10(100),
									divide_q10( Q0_TO_Q10(filtered_volt - min_volt) , Q0_TO_Q10(current_battery_percents) ) ));
						}
						else if (current_battery_percents <= 0 || filtered_volt<=min_volt){
							max_volt = MAX_BATT_VOLT_WITH_EXT_NOT_CHARGING_MV;
						}

						break;
					}

					break;

				case DCDC_POWERED_BY_USB_HOST:
					/* at this state, charging and discharging happens simultaneously */
					/* fix min volt, calculate max volt */
					min_volt = MIN_BATT_VOLT_WITH_USB_HOST_MV;
					if (current_battery_percents > 0  && filtered_volt>min_volt){
						/* Vmax = Vmin + ((volt - Vmin)/percents)*100 */
						max_volt = min_volt + Q10_TO_Q0(multiply_q10(Q0_TO_Q10(100),
									divide_q10( Q0_TO_Q10(filtered_volt - min_volt) , Q0_TO_Q10(current_battery_percents) ) ));
					}
					else if (current_battery_percents <= 0 || filtered_volt<=min_volt){
						max_volt = MAX_BATT_VOLT_WITH_EXT_CHARGING_MV;
					}

					break;
				default:
					min_volt = MIN_BATT_VOLT_NO_EXT_MV;
					max_volt = param_max_batt_volt;
					break;

				}

				inverse_max_min_delta = divide_q10(Q0_TO_Q10(100),Q0_TO_Q10(max_volt-min_volt));
				need_recalculate_max_min = false;

				DEV_PRINT_MSG( "Status changed: power_state=%d, battery_charging_status=%d, max_volt=%d, min_volt=%d, inverse_max_min_delta=%u ",
						power_state, battery_charging_status, max_volt, min_volt, inverse_max_min_delta);

			} /* calculate max min volt */

			/* calculate percents */
			if ((filtered_volt > min_volt) && (filtered_volt < max_volt) && (max_volt > min_volt)){
				/*raw_percents = 100*(filtered_volt-min_volt)/(max_volt - min_volt);*/
				raw_percents = Q10_TO_Q0(multiply_q10( Q0_TO_Q10(filtered_volt-min_volt), inverse_max_min_delta));
			}
			else if (filtered_volt >= max_volt){
				raw_percents = 100;
			}
			else{
				raw_percents = 0;
			}

			if ((raw_percents < 0) || (raw_percents > 100)){
				DEV_PRINT_ERR( "SOMETHING'S WRONG: voltage=%u, filtered_volt=%u, raw_percents=%d, info->battery_percents=%d, max_volt=%d, min_volt=%d, inverse_max_min_delta=%u  ",
						imx28_psm_batt_get_battery_voltage(info->base),filtered_volt,raw_percents, info->battery_percents, max_volt, min_volt, inverse_max_min_delta);
			}
			else {
				DEV_PRINT_MSG( "voltage=%u, filtered_volt=%u, raw_percents=%d, info->battery_percents=%d, max_volt=%d, min_volt=%d, inverse_max_min_delta=%u  ",
						imx28_psm_batt_get_battery_voltage(info->base),filtered_volt,raw_percents, info->battery_percents, max_volt, min_volt, inverse_max_min_delta);
			}

			/* policy */
			if (info->battery_percents >= 0){
				switch (power_state){
				case DCDC_POWERED_BY_BATTERY:
					/* in discharging, battery percentage must not increase */
					if (raw_percents > info->battery_percents){
						raw_percents = info->battery_percents;
					}
					else if ((info->battery_percents - raw_percents) > MAX_DIFF_BATT_PERCENTS){
						raw_percents = info->battery_percents - MAX_DIFF_BATT_PERCENTS;
					}

					break;

				case DCDC_POWERED_BY_EXT_CHARGER:
					switch (battery_charging_status){
					case CHARGING:
						/* in charging, battery percentage must not decrease */
						if (raw_percents <= info->battery_percents){
							raw_percents = info->battery_percents;
						}
						else{
							if ((raw_percents- info->battery_percents ) > MAX_DIFF_BATT_PERCENTS){
								raw_percents = info->battery_percents + MAX_DIFF_BATT_PERCENTS;
							}
							/* must be < 100, only reach 100 when it is full */
							if (raw_percents == 100){
								raw_percents = 99;
							}
						}

						break;

					case FULL:
						/* in full, battery percents must be 100 */
						raw_percents = 100;

						break;

					case NOT_CHARGING:
					default:
						/* FIXME: need to keep battery percents same as the previous one? */
						/* battery percentage must not increase */
						if (raw_percents > info->battery_percents){
							raw_percents = info->battery_percents;
						}
						else if ((info->battery_percents - raw_percents) > MAX_DIFF_BATT_PERCENTS){
							raw_percents = info->battery_percents - MAX_DIFF_BATT_PERCENTS;
						}

						break;
					}

					break;

				case DCDC_POWERED_BY_USB_HOST:
					/* battery percents with USB host can be up or down */
					if ((info->battery_percents - raw_percents) > MAX_DIFF_BATT_PERCENTS){
						raw_percents = info->battery_percents - MAX_DIFF_BATT_PERCENTS;
					}
					else if ((raw_percents - info->battery_percents ) > MAX_DIFF_BATT_PERCENTS){
						raw_percents = info->battery_percents + MAX_DIFF_BATT_PERCENTS;
					}

					break;

				default:
					break;

				}
			} /* policy */

			/* FIXME: If voltage filter is good, it's not necessary to apply a filter to raw percents. */

			info->battery_percents = raw_percents;

			DEV_PRINT_MSG( "battery_percents=%d ",info->battery_percents);
			if (need_notify_user_space){
				power_supply_changed(&info->batt);
				need_notify_user_space = false;
			}
		}/* get filtered value, infer battery percents */
	}
exit:
	spin_unlock(&info->power_state_changed_lock);
}

/* Handler function for battery sampling
 * Do sampling, filtering, and queue work when filtered result available
 */
static void imx28_psm_battery_sampling_triggered(void* data)
{
	struct imx28_psm_info *info = (struct imx28_psm_info *) data;

	unsigned int filtered_volt;
#ifdef INIT_FILTER_BY_AVERAGE
	static unsigned int sampling_counter = 0;
	static bool need_first_value = true;
#endif
	unsigned int voltage;

	if (info->reset_filter){
		/* reset filter */
		imx28_psm_filter_reset();
#ifdef INIT_FILTER_BY_AVERAGE
		need_first_value = true;
		sampling_counter = 0;
#endif
		info->battery_voltage_filtered = 0;
		/* reset done, lower the flag */
		info->reset_filter = false;
	}
	if (info->battery_sampling_active){
#ifdef INIT_FILTER_BY_AVERAGE
		/* in case filter is reseted, voltage is averaged before pass to filter */
		if (need_first_value){
			if (sampling_counter < NUM_SAMPLES_BATT_VOLT){
				info->samples[sampling_counter] = imx28_psm_batt_get_battery_voltage(info->base);
				sampling_counter++;
			}

			if (sampling_counter>=NUM_SAMPLES_BATT_VOLT){
				/* get sufficient sample, calculate average value */
				/* that is the average of 20% highest samples */
				/* start from 60 as there may be voltage peak in highest sample value */
				sort(info->samples, NUM_SAMPLES_BATT_VOLT, sizeof(unsigned int), &imx28_psm_batt_compare, NULL);
				voltage = imx28_psm_batt_average(info->samples + AVG_START_OFFSET_BATT_VOLT, AVG_NUM_SAMPLES_BATT_VOLT);

				sampling_counter = 0;
				need_first_value = false;
			}
			else{
				/* continue in next sampling */
				return;
			}
		}
		else{
			voltage = imx28_psm_batt_get_battery_voltage(info->base);
		}
#else
		voltage = imx28_psm_batt_get_battery_voltage(info->base);
#endif
		if (imx28_psm_filter(voltage, &filtered_volt)==1){
			/* check valid filtered value */
			if (filtered_volt==MIN_VALID_BATT_VOLT_MS || filtered_volt>MAX_VALID_BATT_VOLT_MS){
				DEV_PRINT_ERR( "filtered_volt=%u is out of range! Something is wrong with the filter!\n",filtered_volt);
			}
			else{
				/* got filtered value, schedule work queue to infer battery percents */
				info->battery_voltage_filtered = filtered_volt;
				queue_delayed_work(info->power_supply_wqs, &info->battery_percents_wq, 0);
			}
		}
	}
}

#endif

#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
/*
 * Callback function registered with USB driver to inform VBUS draw power
 */
static int psmbatt_inform_vbus_draw(struct notifier_block *nb, unsigned long code, void *param)
{
	if (pinfo){
		pinfo->vbus_draw_current = *((unsigned*)param);
		if (pinfo->vbus_draw_current>=VDD5V_HIGH_POWER_MA){
			pinfo->vbus_draw_connected = true;
		}
		else{
			pinfo->vbus_draw_connected = false;
		}
		queue_work(pinfo->power_supply_wqs, &pinfo->power_state_change_wq);
	}
	return NOTIFY_OK;
}

static struct notifier_block usb_notifier_block = {
		.notifier_call = psmbatt_inform_vbus_draw,
};
#endif

/* Handler function for battery_temperature_timer
 * Queue work battery monitor to check battery temperature
 */
static void imx28_psm_battery_temperature_triggered(unsigned long data)
{
	struct imx28_psm_info *info = (struct imx28_psm_info *) data;
	unsigned int batt_mon_volt;
	static battery_temperature_status_t pre_batt_temp_status = TEMP_UNKNOWN;
	static battery_temperature_status_t confirming_batt_temp_status = TEMP_UNKNOWN;

	/* check battery temperature */
	batt_mon_volt = imx28_psm_batt_read_adc(BATT_TEMP_MON_LRADC_CHANNEL, true);

	if (batt_mon_volt < MAX_BATT_TEMP_SAFE_MON_VOLT){
		if (confirming_batt_temp_status==TEMP_OVER_MAX){
			info->battery_temperature_status = TEMP_OVER_MAX;
		}
		else{
			confirming_batt_temp_status=TEMP_OVER_MAX;
		}
	}
	else if (batt_mon_volt >= MAX_BATT_TEMP_SAFE_MON_VOLT && batt_mon_volt<info->batt_temp_mon_volt){
		if (confirming_batt_temp_status==TEMP_HOT_FOR_CHARGING){
			info->battery_temperature_status = TEMP_HOT_FOR_CHARGING;
		}
		else{
			confirming_batt_temp_status=TEMP_HOT_FOR_CHARGING;
		}
	}
	/* it can't detect low temperature threshold on EVT3 */
#if defined(HW_EVT3) || !defined(PROTECT_BATTERY_AGAINST_LOW_TEMP)
	else{
		if (confirming_batt_temp_status==TEMP_GOOD){
			info->battery_temperature_status = TEMP_GOOD;
		}
		else{
			confirming_batt_temp_status=TEMP_GOOD;
		}
	}
#else
	else if (batt_mon_volt >= info->batt_temp_mon_volt && batt_mon_volt<=MIN_BATT_TEMP_CHARGING_MON_VOLT){
		if (confirming_batt_temp_status==TEMP_GOOD){
			info->battery_temperature_status = TEMP_GOOD;
		}
		else{
			confirming_batt_temp_status=TEMP_GOOD;
		}
	}
	else if (batt_mon_volt > MIN_BATT_TEMP_CHARGING_MON_VOLT && batt_mon_volt<=MIN_BATT_TEMP_SAFE_MON_VOLT){
		if (confirming_batt_temp_status==TEMP_COLD_FOR_CHARGING){
			info->battery_temperature_status = TEMP_COLD_FOR_CHARGING;
		}
		else{
			confirming_batt_temp_status=TEMP_COLD_FOR_CHARGING;
		}
	}
	else{
		if (confirming_batt_temp_status==TEMP_UNDER_MIN){
			info->battery_temperature_status = TEMP_UNDER_MIN;
		}
		else{
			confirming_batt_temp_status=TEMP_UNDER_MIN;
		}
	}
#endif

	if (info->battery_temperature_status != TEMP_GOOD){
		/* end charging */
		if (imx28_psm_batt_is_battery_charger_on(info->base)){
			imx28_psm_batt_end_charging(info->base);
		}
	}
	/* schedule battery monitor work if status changes */
	if (pre_batt_temp_status != info->battery_temperature_status){
		pre_batt_temp_status = info->battery_temperature_status;
		queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
	}
	/* schedule next run */
	if (info->on){
		info->battery_temperature_timer.expires = jiffies + msecs_to_jiffies(BATT_TEMP_MON_PERIOD_MS);
		add_timer(&info->battery_temperature_timer);
	}
}
/* Handler function for cooling_off_charger_timer
 * set flag cooled_off_charger and queue work battery monitor
 */
static void imx28_psm_battery_cooled_off_charger_triggered(unsigned long data)
{
	struct imx28_psm_info *info = (struct imx28_psm_info *) data;
	info->cooled_off_charger = true;
	queue_delayed_work(info->power_supply_wqs, &info->battery_monitor_wq, 0);
}

/* probe function */
static __devinit int imx28_psm_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct imx28_psm_info *info;
	struct device_node *node = pdev->dev.of_node;
	struct resource *iores;

	uint32_t ac_mon_mv = 0;

	/* module parameters */
	if (param_batt_charging_current < MIN_BATT_CHARGE_CURRENT || param_batt_charging_current > MAX_BATT_CHARGE_CURRENT){
		param_batt_charging_current = DEFAULT_BATT_CHARGE_CURRENT;
	}

	if (param_stop_charging_current < MIN_STOP_CHARGE_CURRENT || param_stop_charging_current > MAX_STOP_CHARGE_CURRENT){
		param_stop_charging_current = DEFAULT_STOP_CHARGE_CURRENT;
	}

	if (param_batt_mon_period < MIN_BATT_MON_PERIOD || param_batt_mon_period > MAX_BATT_MON_PERIOD){
		param_batt_mon_period = DEFAULT_BATT_MON_PERIOD;
	}

	if (param_max_batt_volt < MIN_MAX_BATT_VOLT || param_max_batt_volt > MAX_MAX_BATT_VOLT){
		param_max_batt_volt = DEFAULT_MAX_BATT_VOLT;
	}

	if (param_batt_reduction_rechrg < MIN_BATT_REDUCTION_BEFORE_RECHARGE || param_batt_reduction_rechrg > MAX_BATT_REDUCTION_BEFORE_RECHARGE){
		param_batt_reduction_rechrg = DEFAULT_BATT_REDUCTION_BEFORE_RECHARGE;
	}

	if (param_batt_temp_volt_thresh < MIN_BATT_TEMP_VOLT_THRESH || param_batt_temp_volt_thresh > MAX_BATT_TEMP_VOLT_THRESH){
		param_batt_temp_volt_thresh = DEFAULT_BATT_TEMP_VOLT_THRESH;
	}

	info = kzalloc(sizeof(*info), GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	pinfo = info;

	platform_set_drvdata(pdev, info);
	info->dev = &pdev->dev;
	/* memory area */
	iores = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    info->base = devm_request_and_ioremap(info->dev, iores);
    if (!info->base) {
            ret = -EADDRNOTAVAIL;
            goto free_info;
    }

	/* parse AC_IN_MON LRADC from device tree */
	ret=of_property_read_u32(node, "ac-in-mon-lradc", &info->ac_in_mon_lradc);
	if (ret != 0){
		DEV_PRINT_ERR( "Failed to parse AC_IN_MON LRADC from device tree\n");
		goto free_info;
	}
	else{
		DEV_PRINT_MSG( "AC_IN_MON LRADC from device tree = %d\n",info->ac_in_mon_lradc);
	}

	ret=of_property_read_u32(node, "ac-in-mon-m-volt", &info->ac_in_mon_volt);
	if (ret != 0){
		DEV_PRINT_ERR( "Failed to parse AC_IN_MON Voltage from device tree\n");
		goto free_info;
	}
	else{
		DEV_PRINT_MSG( "AC_IN_MON Voltage (mV) from device tree = %d\n",info->ac_in_mon_volt);
	}

	/* parse BATT TEMP MON LRADC from device tree */
	ret=of_property_read_u32(node, "batt-temp-mon-lradc", &info->batt_temp_mon_lradc);
	if (ret != 0){
		DEV_PRINT_ERR( "Failed to parse BATT TEMP MON LRADC from device tree\n");
		goto free_info;
	}
	else{
		DEV_PRINT_MSG( "BATT TEMP MON LRADC from device tree = %d\n",info->batt_temp_mon_lradc);
	}

	/* parse GPIO EXT_SOURCE_SEL from device tree */
	info->ext_source_sel_gpio=of_get_named_gpio(node, "ext-source-sel-gpio", 0);
	if (info->ext_source_sel_gpio < 0){
		ret = info->ext_source_sel_gpio;
		DEV_PRINT_ERR( "Failed to parse GPIO EXT_SOURCE_SEL from device tree\n");
		goto free_info;
	}
	else{
		DEV_PRINT_MSG( "GPIO EXT_SOURCE_SEL from device tree = %d\n",info->ext_source_sel_gpio);
	}
#if defined(HW_EVT3)
	/* parse GPIO BATT_SOURCE_SEL from device tree */
	info->batt_source_sel_gpio=of_get_named_gpio(node, "batt-source-sel-gpio", 0);
	if (info->batt_source_sel_gpio < 0){
		ret = info->batt_source_sel_gpio;
		DEV_PRINT_ERR( "Failed to parse GPIO BATT_SOURCE_SEL from device tree\n");
		goto free_info;
	}
	else{
		DEV_PRINT_MSG( "GPIO BATT_SOURCE_SEL from device tree = %d\n",info->batt_source_sel_gpio);
	}
#endif

	ret = gpio_request(info->ext_source_sel_gpio, PSM_DRIVER_NAME);
	if (ret){
		DEV_PRINT_ERR( "Failed to request GPIO EXT_SOURCE_SEL\n");
		goto free_info;
	}
#if defined(HW_EVT3)
	ret = gpio_request(info->batt_source_sel_gpio, PSM_DRIVER_NAME);
	if (ret){
		DEV_PRINT_ERR( "Failed to request GPIO BATT_SOURCE_SEL\n");
		goto free_ext_source_sel_gpio;
	}
#endif

	info->batt_volt_full = param_max_batt_volt;
	info->batt_volt_full_charger_on = 0;

	info->checking_new_state = 0;

	info->on = true;
#ifdef INFERING_BATT_SOC
	/* unknown battery percents = -1 */
	info->battery_percents = -1;
#endif

	/* initialize bat power_supply struct */
	info->batt.name           = "batt";
	info->batt.type           = POWER_SUPPLY_TYPE_BATTERY;
	info->batt.properties     = battery_props;
	info->batt.num_properties = ARRAY_SIZE(battery_props);
	info->batt.get_property   = imx28_psm_batt_battery_get_property;

	/* initialize external power power_supply struct */
	info->ext.name           = "ext";
	info->ext.type           = POWER_SUPPLY_TYPE_MAINS;
	info->ext.properties     = power_props;
	info->ext.num_properties = ARRAY_SIZE(power_props);
	info->ext.get_property   = imx28_psm_batt_power_get_property;

	ret = power_supply_register(&pdev->dev, &info->batt);
	if (ret) {
		DEV_PRINT_ERR( "Failed to register battery power supply\n");
		goto free_gpio;
	}

	ret = power_supply_register(&pdev->dev, &info->ext);
	if (ret) {
		DEV_PRINT_ERR( "Failed to register AC power supply\n");
		goto unregister_bat;
	}

	info->power_supply_wqs = create_singlethread_workqueue(POWER_SUPPLY_WQ);

#ifdef INFERING_BATT_SOC
    INIT_DELAYED_WORK(&info->battery_percents_wq, imx28_psm_batt_battery_percents);
    info->battery_sampling_active = false;
#endif

	INIT_WORK(&info->power_state_change_wq, imx28_psm_batt_power_state_change);
	INIT_DELAYED_WORK(&info->battery_monitor_wq, imx28_psm_batt_battery_monitor);

	/* Initilialize battery safety timers and flags */
	init_timer(&info->battery_temperature_timer);
    info->battery_temperature_timer.function = imx28_psm_battery_temperature_triggered;
    info->battery_temperature_timer.data = (unsigned long) info;

	init_timer(&info->cooling_off_charger_timer);
    info->cooling_off_charger_timer.function = imx28_psm_battery_cooled_off_charger_triggered;
    info->cooling_off_charger_timer.data = (unsigned long) info;

    info->cooled_off_charger = true;
    info->battery_temperature_status = TEMP_UNKNOWN;

	/* make sure that charger is off */
	if (imx28_psm_batt_is_battery_charger_on(info->base)){
		imx28_psm_batt_end_charging(info->base);
	}

	/* Enable the automatic battery voltage update */
	imx28_psm_batt_setup_batt_volt_update(info->base, BATTERY_VOLTAGE_CH, BATTERY_VOLT_LRADC_DELAY, BATTERY_MON_DELAY_VALUE);

	/* detect 5v connected */
	imx28_psm_batt_enable_5v_detection_by_VBUSVALID(info->base);
	info->power_5V_connected = imx28_psm_batt_get_5v_present_flag(info->base);

	imx28_psm_batt_enable_usb_plugin_detect();
	if (imx28_psm_batt_is_usb_data_signal_detected()){
		DEV_PRINT_MSG("USB data signal detected\n");
	}
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
	info->vbus_draw_connected = false;
	info->vbus_draw_current = 0;
	/* Setup USB notifier */
    info->usb_transceiver = usb_get_phy(USB_PHY_TYPE_USB2);
    if (info->usb_transceiver) {
    	info->vbus_draw_current = info->usb_transceiver->power_draw;
		if (info->vbus_draw_current>=VDD5V_HIGH_POWER_MA){
			info->vbus_draw_connected = true;
		}
		else{
			info->vbus_draw_connected = false;
		}
        usb_register_notifier(info->usb_transceiver, &usb_notifier_block);
    }
    else{
    	goto free_wq;
    }
#endif
#if defined(HW_EVT3)
	/* pin 4:16 is not mux-ed as GPIO. Both GPIOs are mux-ed here to make sure. */
	/* TODO: find which system mux 4:16 as non-GPIO, then this can be removed. */
#define PINCTRL_BASE            (MXS_IO_ADDRESS(MXS_PINCTRL_BASE_ADDR))
#define PINCTRL_MUXSEL_SET(n)       (PINCTRL_BASE + 0x0100 + (n)*0x10 + 0x4)
	__raw_writel( 0x3 << ((info->ext_source_sel_gpio%16)*2), PINCTRL_MUXSEL_SET(info->ext_source_sel_gpio/16));
	__raw_writel( 0x3 << ((info->batt_source_sel_gpio%16)*2), PINCTRL_MUXSEL_SET(info->batt_source_sel_gpio/16));

	/* BATT_SOURCE_SEL is 0 initially */
	gpio_direction_output(info->ext_source_sel_gpio,0);
	gpio_direction_output(info->batt_source_sel_gpio,0);
#endif

	/* detect AC connection */
	ac_mon_mv = imx28_psm_batt_measure_adc(info->ac_in_mon_lradc /*analog_mux_input*/, AC_IN_MON_LRADC_CHANNEL /*channel*/, true /*div2*/);
	DEV_PRINT_MSG( "AC_IN_MON: %d mV\n",ac_mon_mv);
	if (ac_mon_mv > info->ac_in_mon_volt){
		info->ac_connected = true;

		/* configure interrupt when AC disconnected */
		imx28_psm_batt_set_lradc_thresh_irq(info->ac_in_mon_lradc, AC_IN_MON_LRADC_CHANNEL, true, AC_IN_MON_LRADC_DELAY, AC_IN_MON_LRADC_DELAY_VALUE , THRESHOLD1, info->ac_in_mon_volt, DETECT_LOW);
	}
	else{
		info->ac_connected = false;

		/* configure interrupt when AC connected */
		imx28_psm_batt_set_lradc_thresh_irq(info->ac_in_mon_lradc, AC_IN_MON_LRADC_CHANNEL, true, AC_IN_MON_LRADC_DELAY, AC_IN_MON_LRADC_DELAY_VALUE , THRESHOLD1, info->ac_in_mon_volt, DETECT_HIGH);
	}

	/* setup battery temperature monitoring */
	info->batt_temp_mon_volt = param_batt_temp_volt_thresh;
	imx28_psm_batt_turn_temp_current_source(info->batt_temp_mon_lradc, true);
	/* identify current temperature to set irq polarity */
	/* if ADC value >= threshold, then it is normal temperature, otherwise, it is overheat */
	if (imx28_psm_batt_measure_adc(info->batt_temp_mon_lradc, BATT_TEMP_MON_LRADC_CHANNEL, true) >= info->batt_temp_mon_volt){
		info->battery_overheat = false;
		imx28_psm_batt_setup_batt_temp_sensing(info->batt_temp_mon_lradc,
				BATT_TEMP_MON_LRADC_CHANNEL,
				BATTERY_VOLT_LRADC_DELAY,
				BATT_MON_DELAY_VALUE ,
				THRESHOLD0 /* always use thres0 for batt temp mon */,
				info->batt_temp_mon_volt,
				DETECT_LOW /* currently normal, detect low */);
	}
	else{
		info->battery_overheat = true;
		imx28_psm_batt_setup_batt_temp_sensing(info->batt_temp_mon_lradc,
				BATT_TEMP_MON_LRADC_CHANNEL,
				BATTERY_VOLT_LRADC_DELAY,
				BATT_MON_DELAY_VALUE ,
				THRESHOLD0 /* always use thres0 for batt temp mon */,
				info->batt_temp_mon_volt,
				DETECT_HIGH /* currently overheat, detect high */);
	}

	/* set battery brownout level */
	/* 0x1E or 0x1F (max) here is only to detect battery while 5v connected */
	/* 0x1E: 3.6V; 0x1D: 3.56V; 0x1C: 3.52; 0x1B: 3.48V; 0x1A: 3.44; 0x19: 3.4; 0x18: 3.36; 0x17: 3.32; 0x16: 3.28; 0xF: 3V */
	__raw_writel(  (__raw_readl(info->base + HW_POWER_BATTMONITOR) & (~BM_POWER_BATTMONITOR_BRWNOUT_LVL)) | BF_POWER_BATTMONITOR_BRWNOUT_LVL(0x1C),
			info->base + HW_POWER_BATTMONITOR);

	/* disable automatic hardware powerdown AFTER the system is configured for 5v removal */
	__raw_writel(BM_POWER_5VCTRL_PWDN_5VBRNOUT, info->base + HW_POWER_5VCTRL_CLR);

	imx28_psm_batt_set_charge_current(info->base, param_batt_charging_current);
	imx28_psm_batt_set_stop_current(info->base, param_stop_charging_current);
	/* detect battery */
	if (info->power_5V_connected){
		info->battery_connected = imx28_psm_batt_check_batt_present_by_temp_mon(info);
	}
	else{
		info->battery_connected = true;
	}

	/* configure according to power supply status */
	if (info->power_5V_connected)
	{
		imx28_psm_batt_enable_irq_on_5v_removed(info->base);
	}
	else
	{
		imx28_psm_batt_enable_irq_on_5v_connected(info->base);
	}

	info->power_state = DCDC_UNKNOWN;
	info->dcdc4p2_brownout = false;
	info->vdd5v_droop = false;

	ret = request_irq(MX28_INT_VDD5V, imx28_psm_batt_irq_vdd5v, IRQF_DISABLED, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_VDD5V);
		goto free_usb_notifier;
	}

	ret = request_irq(MX28_INT_VDDIO_BRNOUT, imx28_psm_batt_irq_vddio_brnout, 0, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_VDDIO_BRNOUT);
		goto free_irq_vdd5v;
	}

	ret = request_irq(MX28_INT_DCDC4P2_BRNOUT, imx28_psm_batt_irq_dcdc4p2_bo, IRQF_DISABLED, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_DCDC4P2_BRNOUT);
		goto free_irq_vddio_brnout;

	}

	ret = request_irq(MX28_INT_BATT_BRNOUT, imx28_psm_batt_irq_batt_brnout, IRQF_DISABLED, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_BATT_BRNOUT);
		goto free_irq_dcdc4p2_brnout;
	}

	ret = request_irq(MX28_INT_VDDD_BRNOUT, imx28_psm_batt_irq_vddd_brnout, 0, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_VDDD_BRNOUT);
		goto free_irq_batt_brnout;
	}

	ret = request_irq(MX28_INT_VDDA_BRNOUT, imx28_psm_batt_irq_vdda_brnout, 0, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_VDDA_BRNOUT);
		goto free_irq_vddd_brnout;
	}

	ret = request_irq(MX28_INT_VDD5V_DROOP, imx28_psm_batt_irq_vdd5v_droop, IRQF_SHARED, pdev->name, info);
	if (ret) {
		DEV_PRINT_ERR( "Failed to request IRQ %d\n",MX28_INT_VDD5V_DROOP);
		goto free_irq_vdda_brnout;
	}

	register_lradc_notifier(&lradc_notifier_block);

	queue_work(info->power_supply_wqs, &info->power_state_change_wq);
	/* battery detection and charging decision will be done in battery_monitor work */

	return 0;

free_irq_vdd5v_droop:
	free_irq(MX28_INT_VDD5V_DROOP, info);

free_irq_vdda_brnout:
	free_irq(MX28_INT_VDDA_BRNOUT, info);

free_irq_vddd_brnout:
	free_irq(MX28_INT_VDDD_BRNOUT, info);

free_irq_batt_brnout:
	free_irq(MX28_INT_BATT_BRNOUT, info);

free_irq_dcdc4p2_brnout:
	free_irq(MX28_INT_DCDC4P2_BRNOUT, info);

free_irq_vddio_brnout:
	free_irq(MX28_INT_VDDIO_BRNOUT, info);

free_irq_vdd5v:
		free_irq(MX28_INT_VDD5V, info);
free_usb_notifier:
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
	usb_unregister_notifier(info->usb_transceiver, &usb_notifier_block);
#endif
free_wq:
	flush_work(&info->power_state_change_wq);
	cancel_delayed_work(&info->battery_monitor_wq);
#ifdef INFERING_BATT_SOC
	cancel_delayed_work(&info->battery_percents_wq);
#endif
	flush_workqueue( info->power_supply_wqs );
	destroy_workqueue( info->power_supply_wqs );

	power_supply_unregister(&info->ext);
unregister_bat:
	power_supply_unregister(&info->batt);
free_gpio:
#if defined(HW_EVT3)
	gpio_free(info->batt_source_sel_gpio);
#endif
free_ext_source_sel_gpio:
	gpio_free(info->ext_source_sel_gpio);
free_info:
	kfree(info);

	return ret;
}

static __devexit int imx28_psm_remove(struct platform_device *pdev)
{
	struct imx28_psm_info *info = platform_get_drvdata(pdev);

	info->on = false;

	unregister_lradc_notifier(&lradc_notifier_block);

	imx28_psm_batt_turn_temp_current_source(info->batt_temp_mon_lradc, false);

	free_irq(MX28_INT_VDDIO_BRNOUT, info);
	free_irq(MX28_INT_DCDC4P2_BRNOUT, info);
	free_irq(MX28_INT_BATT_BRNOUT, info);
	free_irq(MX28_INT_VDDD_BRNOUT, info);
	free_irq(MX28_INT_VDDA_BRNOUT, info);
	free_irq(MX28_INT_VDD5V, info);
	free_irq(MX28_INT_VDD5V_DROOP, info);
#ifdef USB_DRIVER_INFORMS_VBUS_DRAW
	usb_unregister_notifier(info->usb_transceiver, &usb_notifier_block);
#endif
	/* delete timers */
	del_timer(&info->battery_temperature_timer);
	del_timer(&info->cooling_off_charger_timer);

	flush_work(&info->power_state_change_wq);
	cancel_delayed_work(&info->battery_monitor_wq);
#ifdef INFERING_BATT_SOC
	cancel_delayed_work(&info->battery_percents_wq);
#endif
	flush_workqueue( info->power_supply_wqs );
	destroy_workqueue( info->power_supply_wqs );

	/* end charging */
	if (imx28_psm_batt_is_battery_charger_on(info->base)){
		imx28_psm_batt_end_charging(info->base);
	}

	power_supply_unregister(&info->ext);
	power_supply_unregister(&info->batt);

#if defined(HW_EVT3)
	gpio_free(info->batt_source_sel_gpio);
#endif
	gpio_free(info->ext_source_sel_gpio);

	kfree(info);
	return 0;
}

static const struct of_device_id imx28_psm_batt_dt_ids[] = {
		{ .compatible = "fsl,imx28-power", },
		{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, imx28_psm_batt_dt_ids);

static struct platform_driver imx28_psm_batt_driver = {
		.driver = {
				.name   = PSM_DRIVER_NAME,
				.owner  = THIS_MODULE,
				.of_match_table = imx28_psm_batt_dt_ids,
		},
		.probe  = imx28_psm_probe,
		.remove = __devexit_p(imx28_psm_remove)
};

module_platform_driver(imx28_psm_batt_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Ha Tran <Ha.Tran@netcommwireless.com>");
MODULE_DESCRIPTION("i.MX28 power source management and battery charger driver");
