/*
 * Copyright (C) 1999 ARM Limited
 * Copyright (C) 2000 Deep Blue Solutions Ltd
 * Copyright 2006-2007,2010 Freescale Semiconductor, Inc. All Rights Reserved.
 * Copyright 2008 Juergen Beisert, kernel@pengutronix.de
 * Copyright 2009 Ilya Yanok, Emcraft Systems Ltd, yanok@emcraft.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/module.h>

#include <asm/proc-fns.h>
#include <asm/system_misc.h>

#include <mach/mxs.h>
#include <mach/mx28.h>
#include <mach/common.h>
#include <mach/regs-common.h>
#include <mach/regs-power.h>
#include <mach/regs-rtc-m28.h>

#include <linux/cdcs_variant.h>

#define MX23_CLKCTRL_RESET_OFFSET	0x120
#define MX28_CLKCTRL_RESET_OFFSET	0x1e0
#define MXS_CLKCTRL_RESET_CHIP		(1 << 1)

#define MXS_MODULE_CLKGATE		(1 << 30)
#define MXS_MODULE_SFTRST		(1 << 31)

static void __iomem *mxs_clkctrl_reset_addr;

static void mx28_lowpwr_alarm(void)
{
#define RTC_REG_OFF_CONTROL	0x00
#define RTC_REG_OFF_STATUS	0x10
#define RTC_REG_OFF_SECONDS	0x30
#define RTC_REG_OFF_ALARM	0x40
#define RTC_REG_OFF_PERSISTENT0	0x60
#define RTC_REG_OFF_PERSISTENT1	0x70
#define RTC_REG_OFF_PERSISTENT2	0x80
#define RTC_REG_OFF_PERSISTENT3	0x90
#define RTC_REG_OFF_PERSISTENT4	0xA0
#define RTC_REG_OFF_PERSISTENT5	0xB0
void __iomem *mx28_rtc = MXS_IO_ADDRESS(MX28_RTC_BASE_ADDR);
uint32_t v, sec;

    v = __raw_readl(mx28_rtc+RTC_REG_OFF_PERSISTENT0);
	 __mxs_clrl(
		/* CLEAR */
		RTC_PERSISTENT0_CLOCKSOURCE|
		RTC_PERSISTENT0_ALARM_WAKE_EN|
		RTC_PERSISTENT0_ALARM_EN|
		RTC_PERSISTENT0_LCK_SECS|
		RTC_PERSISTENT0_XTAL24MHZ_PWRUP|
		RTC_PERSISTENT0_XTAL32KHZ_PWRUP|
		RTC_PERSISTENT0_ALARM_WAKE|
		RTC_PERSISTENT0_LOWERBIAS_MASK|
		RTC_PERSISTENT0_DISABLE_PSWITCH|
		RTC_PERSISTENT0_AUTO_RESTART|
		0, (mx28_rtc+RTC_REG_OFF_PERSISTENT0));

	while (__raw_readl(mx28_rtc+RTC_REG_OFF_STATUS) & RTC_STAT_NEW_REGS_MASK) {} ;

        sec = __raw_readl(mx28_rtc+RTC_REG_OFF_SECONDS);

	/* Mike C 2015-02-06
	 * 
	 * Do NOT set RTC_PERSISTENT0_LOWERBIAS_M50P
	 * this causes some devices to fail after they reboot
	 *
	 * I assume that the original code was setting this as a power
	 * saving measure but it's hard to say for sure.
	 * 
	 */
    v = __raw_readl(mx28_rtc+RTC_REG_OFF_PERSISTENT5);

	/* set for Low Power Mode reboot*/
	if (v) {
		__mxs_setl(sec+v, mx28_rtc+RTC_REG_OFF_ALARM);
	}
	/* set for normal reboot */
	else {
		__mxs_setl(2+v, mx28_rtc+RTC_REG_OFF_ALARM);
	}
	__mxs_setl(
		/* SET */
		RTC_PERSISTENT0_LOWERBIAS_NOMINAL|
		RTC_PERSISTENT0_AUTO_RESTART|
		RTC_PERSISTENT0_XTAL24MHZ_PWRUP|
		0, (mx28_rtc+RTC_REG_OFF_PERSISTENT0));

	while (__raw_readl(mx28_rtc+RTC_REG_OFF_STATUS) & RTC_STAT_NEW_REGS_MASK) {} ;
}

static void mx28_power_off(void)
{
void __iomem *mx28_power_reset = MXS_IO_ADDRESS(MX28_POWER_BASE_ADDR)+0x100;
	__mxs_clrl(POWER_RESET_UNLOCK_MASK|POWER_RESET_PWD_OFF, mx28_power_reset);
	__mxs_setl(POWER_RESET_UNLOCK_KEY, mx28_power_reset);
	__mxs_setl(POWER_RESET_UNLOCK_KEY | POWER_RESET_PWD, mx28_power_reset);
}

/*
 * Reset the system. It is called by machine_restart().
 */
void mxs_restart(char mode, const char *cmd)
{
	/* reset the chip */
	/*
	 * The following code sometimes causes unexpected behaviour on the NWL25 board
	 * Specifically after the board is reset using this method the kernel
	 * will experience errors when VDD5V is next turned on
	 * Thsi does not affect all boards
	 * 
	 * There is no evidence that this probelm also affects other products
	 * nor can it be conclusivly proven that it does not.
	 *
	 * It was found that the mx28_lowpwr_alarm() function was setting -50% 
	 * bias level for the 24MHz crystal oscillator, which was causing issues
	 * when the board rebooted.
	 *
	 * The mx28_lowpwr_alarm() function has been modified to set the bias
	 * level to 'nominal' and this has been found to work reliably.
	 *
	 * A support request has been sent to Freescale for their comments.
	 *
	 */

	 /* Set the RTC to wakeup the device in 2 seconds and then shutdown the 
	  * device power supply.
	  *
	  * A simple reset will not work correctly for devices that do not keep 
	  * PSWITCH high.
	  *
	  */
#ifndef KV_BATTERY_y
	/* Products supporting battery (e.g NTC-20) does not always keep PSWITCH at MID or HIGH level.
	 * After powering down, the device wouldn't start without user action.
	 * Hence devices with V_BATTERY=y will be excluded from this code.
	 */
	if (cpu_is_mx28()) {
		pr_err("mx28_power_off !\n");
		mx28_lowpwr_alarm();
		mx28_power_off();
	}
#endif
	pr_err("mxs_restart\n");
	mdelay(50);
	__mxs_setl(MXS_CLKCTRL_RESET_CHIP, mxs_clkctrl_reset_addr);

	pr_err("Failed to assert the chip reset\n");

	/* Delay to allow the serial port to show the message */
	mdelay(50);

	/* We'll take a jump through zero as a poor second */
	soft_restart(0);
}

static int __init mxs_arch_reset_init(void)
{
	struct clk *clk;

	mxs_clkctrl_reset_addr = MXS_IO_ADDRESS(MXS_CLKCTRL_BASE_ADDR) +
				(cpu_is_mx23() ? MX23_CLKCTRL_RESET_OFFSET :
						 MX28_CLKCTRL_RESET_OFFSET);
	clk = clk_get_sys("rtc", NULL);
	if (!IS_ERR(clk))
		clk_prepare_enable(clk);

	return 0;
}
core_initcall(mxs_arch_reset_init);

/*
 * Clear the bit and poll it cleared.  This is usually called with
 * a reset address and mask being either SFTRST(bit 31) or CLKGATE
 * (bit 30).
 */
static int clear_poll_bit(void __iomem *addr, u32 mask)
{
	int timeout = 0x400;

	/* clear the bit */
	__mxs_clrl(mask, addr);

	/*
	 * SFTRST needs 3 GPMI clocks to settle, the reference manual
	 * recommends to wait 1us.
	 */
	udelay(1);

	/* poll the bit becoming clear */
	while ((__raw_readl(addr) & mask) && --timeout)
		/* nothing */;

	return !timeout;
}

int mxs_reset_block(void __iomem *reset_addr)
{
	int ret;
	int timeout = 0x400;

	/* clear and poll SFTRST */
	ret = clear_poll_bit(reset_addr, MXS_MODULE_SFTRST);
	if (unlikely(ret))
		goto error;

	/* clear CLKGATE */
	__mxs_clrl(MXS_MODULE_CLKGATE, reset_addr);

	/* set SFTRST to reset the block */
	__mxs_setl(MXS_MODULE_SFTRST, reset_addr);
	udelay(1);

	/* poll CLKGATE becoming set */
	while ((!(__raw_readl(reset_addr) & MXS_MODULE_CLKGATE)) && --timeout)
		/* nothing */;
	if (unlikely(!timeout))
		goto error;

	/* clear and poll SFTRST */
	ret = clear_poll_bit(reset_addr, MXS_MODULE_SFTRST);
	if (unlikely(ret))
		goto error;

	/* clear and poll CLKGATE */
	ret = clear_poll_bit(reset_addr, MXS_MODULE_CLKGATE);
	if (unlikely(ret))
		goto error;

	return 0;

error:
	pr_err("%s(%p): module reset timeout\n", __func__, reset_addr);
	return -ETIMEDOUT;
}
EXPORT_SYMBOL(mxs_reset_block);
