/*
 * Implement MIB module for bluetooth paired devices
 *
 * Copyright Notice:
 * Copyright (C) 2015 NetComm Wireless limited.
 *
 * This file or portions thereof may not be copied or distributed in any form
 * (including but not limited to printed or electronic forms and binary or object forms)
 * without the expressed written consent of NetComm Wireless Ltd.
 * Copyright laws and International Treaties protect the contents of this file.
 * Unauthorized use is prohibited.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY NETCOMM WIRELESS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL NETCOMM
 * WIRELESS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include "bluetoothPairedDevices.h"
#include <stdio.h>
#include <string.h>
#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>

#define COLUMN_ADDRESS	1
#define COLUMN_NAME	2

/* data structure for a bluetooth paired device */
struct bt_device_entry {
	char *btAddress;
	size_t btAddress_len;
	char *btName;
	size_t btName_len;
	/* in a linked-list */
	struct bt_device_entry* next;
};
/* head of bluetooth paired device linked list */
struct bt_device_entry  *bt_device_head = NULL;

/*
 * brmgr encodes special characters (;,",%,&) in format %%%02x
 * (ie. "%" + corresponding hexa number in 2 digits)
 * This function decodes and overwrites input string
 *
 * @src: input string
 *
 * returns 0 on success, -1 on error
 */
static int decode_percent(char *src)
{
	char tmp[3] = { 0, 0, 0 };
	long char_code;
	char *endptr;
	char *dest;

	if (!src) {
		return -1;
	}
	dest = src;
	while (*src != '\0') {
		if (*src != '%') {
			*dest = *src;
		}
		else {
			if ((tmp[0] = *(++src)) && (tmp[1] = *(++src))) {
				char_code = strtol(tmp, &endptr, 16);
				if (*endptr == '\0'){
					*dest = (char)char_code;
				}
				else {
					/* invalid string */
					return -1;
				}
			}
			else {
				/* invalid string */
				return -1;
			}
		}
		src++;
		dest++;
	}
	*dest = '\0';
	return 0;
}

/*
 * create new bluetooth paired device and insert into linked-list
 */
static struct bt_device_entry *bt_device_create_entry( char* btAddress, char *btName)
{
	struct bt_device_entry *entry;

	if (!btAddress || !btName || !strlen(btAddress)) {
		return NULL;
	}

	entry = (struct bt_device_entry*)calloc(1, sizeof(struct bt_device_entry));
	if (!entry) {
		return NULL;
	}

	entry->btAddress = strdup(btAddress);
	entry->btAddress_len = strlen(btAddress);
	entry->btName = strdup(btName);
	entry->btName_len = strlen(btName);
	entry->next = bt_device_head;
	bt_device_head = entry;

	return entry;
}

/*
 * Delete all entries in linked list of bluetooth paired devices
 */
static void bt_device_remove_all_entries(void)
{
	struct bt_device_entry *ptr;

	while (bt_device_head != NULL) {
		ptr = bt_device_head;
		bt_device_head = bt_device_head->next;
		free(ptr->btAddress);
		free(ptr->btName);
		free(ptr);
	}
}

/*
 * Load bluetooth paired devices data
 */
static int bt_device_load_data(void)
{
	FILE *fp;
	/* rpc result length 4094 + newline + null = 4096 */
	char buf[4096];
	char *token, *token_i;
	char *pstring;
	char *val_str;
	char *str_address;
	char *str_name;
	int paired;

	/* execute btmgr rpc, 2 seconds time-out, buffer length 4094 */
	fp = popen("rdb invoke btmgr.rpc get_devices 2 4094", "r");
	if (fp == NULL) {
		return -1;
	}

	/* btmgr rpc will print out 1 line */
	if (fgets(buf, sizeof(buf), fp) != NULL) {
		/* remove tailing newline */
		if (buf[strlen(buf)-1]=='\n') {
			buf[strlen(buf)-1]='\0';
		}

		pstring = buf;
		while ((token = strsep(&pstring, "&")) != NULL){

			str_address = NULL;
			str_name = NULL;
			paired = 0;
			while ((token_i = strsep(&token, ";")) != NULL){
				if ((val_str = strchr(token_i, '=')) != NULL) {

					/* ignore "=" */
					*val_str = '\0';
					val_str++;
					if (!strcmp(token_i, "Address")) {
						str_address = val_str;
					} else if (!strcmp(token_i, "Name")) {
						str_name = val_str;
					} else if (!strcmp(token_i, "Paired")){
						paired = !strcmp(val_str, "true");
					} else {
						/* either invalid string or btmgr appends more information, so do nothing */
					}
				}
				else {
					/* either invalid string or btmgr appends/changes result format. Do nothing (i.e continue process next token) */
				}
			}

			/* insert pared device into link-list */
			if (paired && str_address && strlen(str_address)>0 && !decode_percent(str_name)) {
				bt_device_create_entry(str_address, str_name);
			}
		}
	}

	pclose(fp);
	return 0;
}

/*
 * Associate index and data context for next data point
 * Implement Netsnmp_Next_Data_Point *get_next_data_point of netsnmp_iterator_info
 * For more information: include/net-snmp/agent/table_iterator.h
 */
static netsnmp_variable_list * bt_device_get_next_data_point(
		void **my_loop_context,
		void **my_data_context,
		netsnmp_variable_list *put_index_data,
		netsnmp_iterator_info *mydata)
{
	struct bt_device_entry *entry = (struct bt_device_entry *)*my_loop_context;

	if ( entry ) {
		snmp_set_var_value( put_index_data, entry->btAddress, entry->btAddress_len );
		*my_data_context = (void *)entry;
		*my_loop_context = (void *)entry->next;
		return put_index_data;
	} else {
		*my_data_context = NULL;
		*my_loop_context = NULL;
		return NULL;
	}
}

/*
 * Associate index and data context for first data point
 * Implement Netsnmp_First_Data_Point *get_first_data_point of netsnmp_iterator_info
 * For more information: include/net-snmp/agent/table_iterator.h
 */
static netsnmp_variable_list * bt_device_get_first_data_point(void **my_loop_context,
		void **my_data_context,
		netsnmp_variable_list *put_index_data,
		netsnmp_iterator_info *mydata)
{
	*my_loop_context = bt_device_head;

	return bt_device_get_next_data_point(my_loop_context, my_data_context, put_index_data, mydata);
}

/*
 * handles requests for the bluetooth paired device table
 * Implement Netsnmp_Node_Handler. For more information: include/net-snmp/agent/agent_handler.h
 */
static int bt_device_handler(
		netsnmp_mib_handler               *handler,
		netsnmp_handler_registration      *reginfo,
		netsnmp_agent_request_info        *reqinfo,
		netsnmp_request_info              *requests)
{
	netsnmp_request_info       *request;
	netsnmp_table_request_info *table_info;
	struct bt_device_entry          *table_entry;
	netsnmp_variable_list *var;

	for (request=requests; request; request=request->next) {
		var = request->requestvb;

		if (request->processed != 0) {
			continue;
		}

		table_entry = (struct bt_device_entry *) netsnmp_extract_iterator_context(request);
		if (table_entry == NULL) {
			netsnmp_set_request_error(reqinfo, request, SNMP_NOSUCHINSTANCE);
			continue;
		}

		table_info = netsnmp_extract_table_info(request);

		if (table_info == NULL) {
			continue;
		}

		switch (reqinfo->mode) {
			case MODE_GET:
			case MODE_GETNEXT:
				switch (table_info->colnum) {
					case COLUMN_ADDRESS:
						snmp_set_var_typed_value( request->requestvb,
								ASN_OCTET_STR,
								table_entry->btAddress,
								table_entry->btAddress_len);
						break;

					case COLUMN_NAME:
						snmp_set_var_typed_value( request->requestvb,
								ASN_OCTET_STR,
								table_entry->btName,
								table_entry->btName_len);
						break;
					default:
						netsnmp_set_request_error(reqinfo, request, SNMP_NOSUCHOBJECT);
						break;
				}
				break;

			case MODE_SET_RESERVE1:
			case MODE_SET_RESERVE2:
			case MODE_SET_FREE:
			case MODE_SET_ACTION:
			case MODE_SET_UNDO:
			case MODE_SET_COMMIT:
				return SNMP_ERR_READONLY;

			default:
				break;
		}

	}

	return SNMP_ERR_NOERROR;
}

/*
 * Bluetooth paired device cache loading function
 * Implement NetsnmpCacheLoad
 */
static int bt_device_cache_load(netsnmp_cache *cache, void *magic) {
	bt_device_load_data();
	return SNMP_ERR_NOERROR;
}

/*
 * Bluetooth paired device cache freeing function
 * Implement NetsnmpCacheFree
 */
static void bt_device_cache_free(netsnmp_cache *cache, void *magic)
{
	/* delete old entries */
	bt_device_remove_all_entries();
}

/*
 * initialise and register table, iterator, and cache
 *
 * returns 0 on success, -1 on error
 */
static int initialize_table_bt_device(void)
{
	/* OID of Netcomm's bluetooth paired devices MIB */
	const oid bt_device_oid[] = {1,3,6,1,4,1,24063,1,1,9,5};
	const size_t bt_device_oid_len   = OID_LENGTH(bt_device_oid);
	netsnmp_handler_registration    *reg;
	netsnmp_iterator_info           *iinfo;
	netsnmp_table_registration_info *table_info;
	netsnmp_mib_handler *cache_handler;

	/* register read-only handler for the table */
	reg = netsnmp_create_handler_registration(
			"bluetoothPairedDevices",
			bt_device_handler,
			bt_device_oid, bt_device_oid_len,
			HANDLER_CAN_RONLY
	);
	if (!reg) {
		return -1;
	}
	/* allocate table registration info and add index */
	table_info = SNMP_MALLOC_TYPEDEF( netsnmp_table_registration_info );
	if (!table_info) {
		netsnmp_handler_registration_free(reg);
		return -1;
	}
	/* add btAddress as index */
	netsnmp_table_helper_add_indexes(
			table_info,
			ASN_OCTET_STR, /* btAddress */
			0);
	table_info->min_column = COLUMN_ADDRESS;
	table_info->max_column = COLUMN_NAME;
	/* allocate iterator info */
	iinfo = (netsnmp_iterator_info*)calloc(1, sizeof(netsnmp_iterator_info));
	if (!iinfo || !reg) {
		goto free_reg;
	}
	/* iterator access routines */
	iinfo->get_first_data_point = bt_device_get_first_data_point;
	iinfo->get_next_data_point  = bt_device_get_next_data_point;
	/* tie the two structures together */
	iinfo->table_reginfo = table_info;
	/* register table iterator */
	if (netsnmp_register_table_iterator2( reg, iinfo ) != MIB_REGISTERED_OK) {
		goto free_reg;
	}
	/* create cache handler */
	cache_handler = netsnmp_get_cache_handler(
			5,	/* how long in second a cache is valid for */
			bt_device_cache_load,	/* cache loading function */
			bt_device_cache_free,	/* cache freeing function */
			bt_device_oid, bt_device_oid_len);	/* the OID of the registration point */
	if (!cache_handler) {
		goto free_reg;
	}
	/* inject the cache handler into the handler chain */
	if (netsnmp_inject_handler(reg, cache_handler) != SNMPERR_SUCCESS) {
		goto free_all;
	}

	return 0;
	free_all:
	netsnmp_cache_free(cache_handler->myvoid);
	free_reg:
	netsnmp_handler_registration_free(reg);
	return -1;
}

/*
 * Initializes the bluetoothPairedDevices module
 * This will be called by net-snmp in loading
 */
void init_bluetoothPairedDevices(void)
{
	if (initialize_table_bt_device()) {
		snmp_log(LOG_ERR, "Failed to initializes the bluetoothPairedDevices module\n");
	}
}
