/*

Command line support for Ethernet test modes.

Typically, for Ethernet conformance testing, the Ethernet PHY
must be put into a test mode, generating specific wire test
signals.

The phytest.c file provides the core phytest command line, with
the phytest_PHYNAME.c files providing manufacturer or phy
specific functions.

Basic operation is very similar to the phylib, in the sense
that specific drivers are registering for pattern/mask of
PHY ID.

A phylib driver may register for phytest at the same time
Don't add it to the init list in phytest.c in that case.

*/
#include <phy.h>

/* Testmodes. Must match the table in ethtest.c as well. */
enum ethtest_mode {
	PT_MODE_NORMAL = 0,   /* Normal operation, disable test modes. */

	/* 10 MBit test modes */
	PT10_MODE_10MHZ,      /* 10MHz sinewave */
	PT10_MODE_5MHZ,       /* 5MHz sinewave */
	PT10_MODE_RANDOM,     /* Random data transmission */
	PT10_MODE_LINK,       /* Link pulses */

	/* 100MBit test modes */
	PT100_MODE_JITTER,     /* Jitter test pattern */
	PT100_MODE_OVERSHOOT,  /* Overshoot test pattern */
	PT100_MODE_DCD,        /* Data Carrier Detection test */

	/* 1GBit test modes */
	PT1G_MODE_WAVE,         /* Transmit waveform test */
	PT1G_MODE_JITTER_MASTER,/* Jitter test, master mode */
	PT1G_MODE_JITTER_SLAVE, /* Jitter test, slave mode */
	PT1G_MODE_DISTORTION,   /* Transmit distortion test */

	/* End marker */
	PT_MODE_LAST
};

/* For debugging (won't use memory if unused) */
static const char *modestr[] __attribute__((unused)) = {
	"PT_MODE_NORMAL",
	"PT10_MODE_10MHZ",
	"PT10_MODE_5MHZ",
	"PT10_MODE_RANDOM",
	"PT10_MODE_LINK",
	"PT100_MODE_JITTER",
	"PT100_MODE_OVERSHOOT",
	"PT100_MODE_DCD",
	"PT1G_MODE_WAVE",
	"PT1G_MODE_JITTER_MASTER",
	"PT1G_MODE_JITTER_SLAVE",
	"PT1G_MODE_DISTORTION",
	"???"
};

struct ethtest_driver {
	const char *name;

	/* Matching the PHY. Setting both values to 0 always matches.
	 * On match, then probe function is called and should return 0
	 * if the shoe fits.
	 */
	unsigned  uid;
	unsigned mask;

	/* Return 0 if match. */
	int (*probe)(struct ethtest_driver *ethtest);

	/* Mode switch functions. Set to NULL if unavailable.
	 * Each function is only ever called with the modes for its speed.
	 * A mode of 0 is always taken as "normal operation". */
	int (*test10)(struct ethtest_driver *ethtest, enum ethtest_mode mode);
	int (*test100)(struct ethtest_driver *ethtest, enum ethtest_mode mode);
	int (*test1G)(struct ethtest_driver *ethtest, enum ethtest_mode mode);

	/* PHY cable test, leave PHY in normal mode */
	int (*cable)(struct ethtest_driver *ethtest);

	struct phy_device *phydev;
	struct list_head list;
};

/* ethtest API - register a driver. */
int ethtest_register(struct ethtest_driver *edrv);

/* Implements IEEE Gigabit test mode - the only one that is standard on all PHYs. */
int gentest_test1G(struct ethtest_driver *ethtest, enum ethtest_mode mode);

/* Prototypes for phytest init functions, which must call ethtest_register.
 * Init functions must allow multiple calls with no bad effects. */
int ethtest_example_init(void);
