/*
 * FCB (Firmware Configuration Block) & DBBT management code for
 * NAND flash on Freescale MX28 processors.
 *
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * Some code originates from tools/mxsboot.c by Marek Vasut
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <linux/mtd/mtd.h>
#include <command.h>
#include <watchdog.h>
#include <malloc.h>
#include <asm/byteorder.h>
#include <nand.h>

#ifndef MB
	#define KB *1024
	#define MB KB*1024
#endif

static u_char *buffer; /* Enough for 1 page + OOB */

#define SPARE_SIZE 12 /* Spare area size at beginning of page in bytes */

/* Default settings */
struct settings {
	unsigned stride_pages; /* Number of pages between FCBs */
	unsigned stride_count; /* Number of FCBs */
	unsigned fw1_offset;   /* Page number of primary firmware stream */
	unsigned fw1_maxsize;  /* Maximum number of pages for primary firmware */
	unsigned fw2_offset;   /* Page number of secondary firmware stream */
	unsigned fw2_maxsize;  /* Maximum number of pages for secondary firmware */
};

static struct settings set; /* Global settings */

/* Default settings for 2K page NAND */
static const struct settings set_2048 = {
	.stride_pages = 64, /* Processor default unless changed via OCOTP fuses */
	.stride_count = 4,  /* Processor default unless changed via OCOTP fuses */

	.fw1_offset = 256,  /* pages, 512K, just beyond FCB block */
	.fw1_maxsize = 256, /* pages, 512K, limited by uenv */
	.fw2_offset = 256,  /* Points to same location as fw1. */
	.fw2_maxsize = 256,
};

/* Default settings for 4K page NAND */
static const struct settings set_4096 = {
	.stride_pages = 64, /* Processor default unless changed via OCOTP fuses */
	.stride_count = 2,  /* Processor default is 4, but we ignore that. */

	.fw1_offset = 128,  /* pages, 512K, just beyond FCB block */
	.fw1_maxsize = 128, /* pages, 512K, limited by uenv */
	.fw2_offset = 128,  /* Points to same location as fw1. */
	.fw2_maxsize = 128,
};

struct nand_timing {
	uint8_t  data_setup;
	uint8_t  data_hold;
	uint8_t  address_setup;
	uint8_t  dsample_time;
	uint8_t  nand_timing_state;
	uint8_t  rea;
	uint8_t  rloh;
	uint8_t  rhoh;
} __attribute__((packed,aligned(8)));

struct nand_fcb {
	uint32_t  checksum;
	uint32_t  fingerprint;
	uint32_t  version;
	struct nand_timing  timing;
	uint32_t  page_data_size;
	uint32_t  total_page_size;
	uint32_t  sectors_per_block;
	uint32_t  number_of_nands;		/* Ignored */
	uint32_t  total_internal_die;		/* Ignored */
	uint32_t  cell_type;			/* Ignored */
	uint32_t  ecc_block_n_ecc_type;
	uint32_t  ecc_block_0_size;
	uint32_t  ecc_block_n_size;
	uint32_t  ecc_block_0_ecc_type;
	uint32_t  metadata_bytes;
	uint32_t  num_ecc_blocks_per_page;
	uint32_t  ecc_block_n_ecc_level_sdk;	/* Ignored */
	uint32_t  ecc_block_0_size_sdk;		/* Ignored */
	uint32_t  ecc_block_n_size_sdk;		/* Ignored */
	uint32_t  ecc_block_0_ecc_level_sdk;	/* Ignored */
	uint32_t  num_ecc_blocks_per_page_sdk;	/* Ignored */
	uint32_t  metadata_bytes_sdk;		/* Ignored */
	uint32_t  erase_threshold;
	uint32_t  boot_patch;
	uint32_t  patch_sectors;
	uint32_t  firmware1_starting_sector;
	uint32_t  firmware2_starting_sector;
	uint32_t  sectors_in_firmware1;
	uint32_t  sectors_in_firmware2;
	uint32_t  dbbt_search_area_start_address;
	uint32_t  badblock_marker_byte;
	uint32_t  badblock_marker_start_bit;
	uint32_t  bb_marker_physical_offset;

	uint8_t   spare[376];
} __attribute__((packed,aligned(32)));


/* Funny checksum calculation. */
static uint32_t checksum(uint8_t *block, uint32_t size)
{
	uint32_t csum = 0;
	int i;

	for (i = 0; i < size; i++)
		csum += block[i];

	return csum ^ 0xffffffff;
}

/* Calculates Hamming(13,8) parity (5 bits) from 8 bit word. */
static uint8_t hamming_13_8(const uint8_t b)
{
	uint32_t p = 0, t;

	t = ((b >> 6) ^ (b >> 5) ^ (b >> 3) ^ (b >> 2)) & 1;
	p |= t << 0;

	t = ((b >> 7) ^ (b >> 5) ^ (b >> 4) ^ (b >> 2) ^ (b >> 1)) & 1;
	p |= t << 1;

	t = ((b >> 7) ^ (b >> 6) ^ (b >> 5) ^ (b >> 1) ^ (b >> 0)) & 1;
	p |= t << 2;

	t = ((b >> 7) ^ (b >> 4) ^ (b >> 3) ^ (b >> 0)) & 1;
	p |= t << 3;

	t = ((b >> 6) ^ (b >> 4) ^ (b >> 3) ^
		(b >> 2) ^ (b >> 1) ^ (b >> 0)) & 1;
	p |= t << 4;

	return p;
}

/* Checks parity and returns possibly corrected value.
 * Returns number of corrected bits (0-1) or -1 for uncorrectable. */
static int correct_hamming_13_8(uint8_t *v, uint8_t p)
{
	uint8_t bit;
	/* Lookup table for parity patterns. */
	#define B(n) (n) /* position of bit error */
	#define POK 100  /* no error */
	#define PPE 101  /* error in parity, ignore */
	#define PUC 102  /* uncorrectable error */
	static const uint8_t lookup[32] = {
		/* 00000 */ POK, /* 00001 */ PPE, /* 00010 */ PPE, /* 00011 */ PUC,
		/* 00100 */ PPE, /* 00101 */ PUC, /* 00110 */ PUC, /* 00111 */ B(5),
		/* 01000 */ PPE, /* 01001 */ PUC, /* 01010 */ PUC, /* 01011 */ PUC,
		/* 01100 */ PUC, /* 01101 */ PUC, /* 01110 */ B(7),/* 01111 */ PUC,
		/* 10000 */ PPE, /* 10001 */ PUC, /* 10010 */ PUC, /* 10011 */ B(2),
		/* 10100 */ PUC, /* 10101 */ B(6),/* 10110 */ B(1),/* 10111 */ PUC,
		/* 11000 */ PUC, /* 11001 */ B(3),/* 11010 */ B(4),/* 11011 */ PUC,
		/* 11100 */ B(0),/* 11101 */ PUC, /* 11110 */ PUC, /* 11111 */ PUC
	};

	p = p ^ hamming_13_8(*v); /* Any set bit is differing parity */

	bit = lookup[p];

	if (bit == POK) return 0;
	if (bit == PUC) return -1;
	if (bit == PPE) return 0;

	//printf("ECC: corrected [%p]:%d, %02x -> %02x\n", v, bit, *v, *v ^ (1 << bit));

	/* Perform correction */
	*v ^= (1 << bit);

	return 1;
}

/* generates/checks hamming(13,8) parity for a data block,
 * assuming the parity bytes (5bit+3unused) are stored
 * at &page[bytes] onwards. Returns -1 if check failed
 * uncorrectable, otherwise number of bit corrections. */
static int block_ECC(uint8_t *page, int bytes, int check)
{
	int corrections=0;
	int rval;
	uint8_t *ecc = page + bytes;
	if (check) {
		while (bytes--) {
			rval = correct_hamming_13_8(page, *ecc);
			if (rval == -1) {
				printf("ECC: uncorrectable\n");
				return -1;
			}
			corrections += rval;
			page++; ecc++;
		}
	} else {
		while (bytes--) {
			*ecc = hamming_13_8(*page);
			page++; ecc++;
		}
	}
	return corrections;
}

/* Using default settings */
static int populate_fcb(nand_info_t *nand, uint8_t *page)
{
	struct nand_fcb   *fcb = (struct nand_fcb *)&page[SPARE_SIZE];
	uint32_t bch_bits = 0;

	memset(page, 0, nand->writesize + nand->oobsize);

	fcb->fingerprint = 0x20424346; /* "FCB " */
	fcb->version = 0x01000000;

	/* Timing. TODO: fetch from ONFI */
	fcb->timing.data_setup = 80;
	fcb->timing.data_hold = 60;
	fcb->timing.address_setup = 25;
	fcb->timing.dsample_time = 6;
	fcb->timing.nand_timing_state = 0;
	fcb->timing.rea = 0;
	fcb->timing.rloh = 0;
	fcb->timing.rhoh = 0;

	/* Geometry. */
	fcb->page_data_size = nand->writesize;
	fcb->total_page_size = nand->writesize + nand->oobsize;
	fcb->sectors_per_block = nand->erasesize / nand->writesize;
	fcb->number_of_nands = 0;		/* Ignored */
	fcb->total_internal_die = 0;		/* Ignored */
	fcb->cell_type = 0;			/* Ignored */

	if (nand->writesize == 2048) {
		bch_bits = 8;
	} else if (nand->writesize == 4096) {
		if (nand->oobsize == 128) {
			bch_bits = 8;
		} else if (nand->oobsize >= 218) {
			bch_bits = 16;
		}
	}
	if (!bch_bits) {
		printf("FCB: unsupported NAND geometry (%u/%u)\n",
			nand->writesize, nand->oobsize);
		return -1;
	}

	#define ECCTYPE(bch) (bch / 2)

	/* Weird Freescale ECC layout. 12_SPARE 512_DATA 13_ECC 512_DATA 13_ECC ... */
	fcb->ecc_block_n_ecc_type = ECCTYPE(bch_bits);
	fcb->ecc_block_0_size = 512;
	fcb->ecc_block_n_size = 512;
	fcb->ecc_block_0_ecc_type = ECCTYPE(bch_bits);
	fcb->metadata_bytes = 10;
	fcb->num_ecc_blocks_per_page = nand->writesize / 512 - 1;
	fcb->ecc_block_n_ecc_level_sdk = ECCTYPE(bch_bits);	/* Ignored */
	fcb->ecc_block_0_size_sdk = 512;		/* Ignored */
	fcb->ecc_block_n_size_sdk = 512;		/* Ignored */
	fcb->ecc_block_0_ecc_level_sdk = ECCTYPE(bch_bits);	/* Ignored */
	fcb->num_ecc_blocks_per_page_sdk = nand->writesize / 512 - 1;	/* Ignored */
	fcb->metadata_bytes_sdk = 10;		/* Ignored */

	fcb->erase_threshold = 0; 
	fcb->boot_patch = 0;
	fcb->patch_sectors = 0;

	/* Firmware block - i.e. U-Boot boot stream. */
	fcb->firmware1_starting_sector = set.fw1_offset;
	fcb->firmware2_starting_sector = set.fw2_offset;
	fcb->sectors_in_firmware1 = set.fw1_maxsize;
	fcb->sectors_in_firmware2 = set.fw2_maxsize;

	/* No DBBT, rom scans as required */
	fcb->dbbt_search_area_start_address = 0;

	/* The badblock marker byte is this position in the data bytes - i.e.
	 * if one was to skip the metadata area and strip the ECCs, the BB marker
	 * at page offset nand->writesize falls into this position in the rmaining bytes.
	 * In both supported page and ECC sizes, this always falls into the
	 * last data block. Also, because we only use BCH8 or BCH16, the
	 * resulting bit offset is always 0. */
	{
		uint32_t non_data_bytes;
		/* blocks per page is stored -1 */
		non_data_bytes = fcb->metadata_bytes + (fcb->num_ecc_blocks_per_page+1-1)*((bch_bits*13+7)/8);
		fcb->badblock_marker_byte = nand->writesize - non_data_bytes;
		fcb->badblock_marker_start_bit = 0; /* BCH4 od BCH8 only */
		fcb->bb_marker_physical_offset = nand->writesize;
	}

	/* compute 32-bit checksum */
	fcb->checksum = checksum((uint8_t*)fcb + 4, sizeof(*fcb) - 4);

	/* compute ECC bytes */
	block_ECC((uint8_t*)fcb, sizeof(*fcb), 0 /* calculate & store */);

	return 0;
}

#define CHECK(cond, VA...) if (!(cond)) printf(VA)

static int is_fcb(nand_info_t *nand, u_char *page)
{
	struct nand_fcb   *fcb = (struct nand_fcb *)&page[SPARE_SIZE];
	uint32_t cs;

	if (fcb->fingerprint != 0x20424346) {
		printf("This is not a valid FCB, fingerprint=0x%08x\n", fcb->fingerprint);
		return -1;
	}
	CHECK(fcb->version == 0x01000000, "Bad FCB version: %08x\n", fcb->version);

	cs = checksum((uint8_t*)&page[SPARE_SIZE+4], 508);
	CHECK(cs == *(uint32_t*)fcb, "Checksum failed, residual %08x\n", cs);

	return 0;
}

#define FCB_FIELD(page, field) ((struct nand_fcb *)&page[SPARE_SIZE]) -> field

static int dump_fcb(nand_info_t *nand, u_char *page)
{
	struct nand_fcb   *fcb = (struct nand_fcb *)&page[SPARE_SIZE];

	if (is_fcb(nand, page)) return -1;

	printf("ECC=%08x, fingerp=%08x, ver=%08x\n",
		fcb->checksum, fcb->fingerprint, fcb->version);

	printf("NAND timing: %d, %d, %d, %d, %d, %d, %d, %d\n",
		fcb->timing.data_setup, fcb->timing.data_hold, fcb->timing.address_setup,
		fcb->timing.dsample_time, fcb->timing.nand_timing_state,
		fcb->timing.rea, fcb->timing.rloh, fcb->timing.rhoh);

	printf("NAND page=%d, total=%d, ppb=%d, num=%d, die=%d, cell=%d\n",
		fcb->page_data_size, fcb->total_page_size, fcb->sectors_per_block,
		fcb->number_of_nands, fcb->total_internal_die, fcb->cell_type);

	printf("ECC TYPn=%d, BLK0=%d, BLKn=%d, TYP0=%d, meta=%d, BPP=%d\n",
		fcb->ecc_block_n_ecc_type, fcb->ecc_block_0_size,
		fcb->ecc_block_n_size, fcb->ecc_block_0_ecc_type,
		fcb->metadata_bytes, fcb->num_ecc_blocks_per_page);

	printf("SDK TYPn==%d, BLK0=%d, BLKn=%d, TYP0=%d, meta=%d, BPP=%d\n",
		fcb->ecc_block_n_ecc_level_sdk, fcb->ecc_block_0_size_sdk,
		fcb->ecc_block_n_size_sdk, fcb->ecc_block_0_ecc_level_sdk,
		fcb->metadata_bytes_sdk, fcb->num_ecc_blocks_per_page_sdk);

	printf("BOOT ethr=%d, patch=%d, psect=%d\n",
		fcb->erase_threshold, fcb->boot_patch, fcb->patch_sectors);

	printf("FW fw1=%d, fw2=%d, fw1s=%d, fw2s=%d\n",
		fcb->firmware1_starting_sector, fcb->firmware2_starting_sector,
		fcb->sectors_in_firmware1, fcb->sectors_in_firmware2);

	printf("DBBT start=%d, bbbyte=%d, bbbit=%d, bbofs=%d\n",
		fcb->dbbt_search_area_start_address, fcb->badblock_marker_byte,
		fcb->badblock_marker_start_bit, fcb->bb_marker_physical_offset);

	return 0;
}

static int fcb_correct_ECC(u_char *page)
{
	struct nand_fcb   *fcb = (struct nand_fcb *)&page[SPARE_SIZE];
	return block_ECC((uint8_t *)fcb, sizeof(*fcb), 1 /* Check */);
}

/* NAND raw access */
static int raw_access(nand_info_t *nand, ulong addr, loff_t off, int read)
{
	int ret = 0;

	/* Raw access */
	mtd_oob_ops_t ops = {
		.datbuf = (u8 *)addr,
		.oobbuf = ((u8 *)addr) + nand->writesize,
		.len = nand->writesize,
		.ooblen = nand->oobsize,
		.mode = MTD_OOB_RAW
	};

	if (read)
		ret = nand->read_oob(nand, off, &ops);
	else
		ret = nand->write_oob(nand, off, &ops);

	if (ret) {
		printf("%s: error at offset %llx, ret %d\n",
			__func__, (long long)off, ret);
	}

	return ret;
}

int do_fcb(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	int rval = -1;
	int dev;
	nand_info_t *nand;
	char *cmd;
	loff_t offset;
	int err = 0;
	unsigned i;

	dev = nand_curr_device;
	nand = &nand_info[dev];

	buffer = malloc(nand->writesize + nand->oobsize);
	if (!buffer) {
		printf("Out of memory\n");
		goto quit;
	}

	if (argc < 2) {
		printf("No subcommand\n");
		goto quit;
	}

	/* Select defaults based on page size */
	if (nand->writesize == 2048) {
		set = set_2048;
	} else {
		set = set_4096;
	}

	cmd = argv[1];

	/* Skip command & subcommand */
	argc-=2;
	argv+=2;
	while (argc) {
		if (strcmp("-1", *argv)==0) {
			if (argc < 3) goto quit;
			set.fw1_offset = simple_strtoull(argv[1], NULL, 16) / nand->writesize;
			set.fw1_maxsize = simple_strtoull(argv[2], NULL, 16) / nand->writesize;
			argc-=2; argv+=2;
		} else if (strcmp("-2", *argv)==0) {
			if (argc < 3) goto quit;
			set.fw2_offset = simple_strtoull(argv[1], NULL, 16) / nand->writesize;
			set.fw2_maxsize = simple_strtoull(argv[2], NULL, 16) / nand->writesize;
			argc-=2; argv+=2;
		} else if (strcmp("-s", *argv)==0) {
			if (argc < 3) goto quit;
			set.stride_pages = simple_strtoul(argv[1], NULL, 0);
			set.stride_count = simple_strtoul(argv[2], NULL, 0);
			argc-=2; argv+=2;
		} else goto quit;
		argc--;
		argv++;
	}

	/* Sanity checks */
	{
		/* First and last pages of FCB, FW1 & FW2 */
		unsigned fcb_s = 0;
		unsigned fcb_e = set.stride_pages * set.stride_count - 1;
		unsigned fw1_s = set.fw1_offset;
		unsigned fw1_e = set.fw1_offset + set.fw1_maxsize - 1;
		unsigned fw2_s = set.fw2_offset;
		unsigned fw2_e = set.fw2_offset + set.fw2_maxsize - 1;

		printf("FCB(%d-%d), FW1(%d-%d), FW2(%d-%d)\n",
			fcb_s, fcb_e, fw1_s, fw1_e, fw2_s, fw2_e);

		if (fw1_s <= fcb_e || fw2_s <= fcb_e) {
			printf("Firmware area overlaps FCB\n"); goto quit;
		}
		/* Identical is OK, overlap is not */
		if ((fw1_s != fw2_s || fw1_e != fw2_e) && fw1_e >= fw2_s && fw2_e >= fw1_s) {
			printf("Firmware areas overlap\n"); goto quit;
		}
	}

	rval = 0; /* Don't print usage after this. */

	if (flag & CMD_FLAG_REPEAT) {
		goto quit; /* We don't repeat */
	}

	if (strcmp(cmd, "info")==0) {
		int good = -1;

		for (i=0; i<set.stride_count; i++) {
			offset = (loff_t)i * set.stride_pages * nand->writesize;

			printf("FCB @0x%llx: ", offset);

			err = raw_access(nand, (ulong)buffer, offset, 1 /* Read */);
			if (err) goto quit;

			err = fcb_correct_ECC(buffer);
			if (err > 0) {
				printf("ECC: %d bits corrected\n", err);
			}
			if (err < 0) break;

			if (is_fcb(nand, buffer) == 0) {
				printf("%08x\n", FCB_FIELD(buffer, checksum));
				if (good < 0) good = i;
			} else {
				err = 1;
			}
		}
		printf("\n");

		/* Re-read first good FCB and dump */
		if (good >= 0) {
			offset = (loff_t)good * set.stride_pages * nand->writesize;
			err = raw_access(nand, (ulong)buffer, offset, 1 /* Read */);
			if (err) goto quit;

			err = fcb_correct_ECC(buffer);
			if (err > 0) {
				printf("ECC: %d bits corrected\n", err);
			}
			if (err < 0) goto quit;

			printf("FCB @0x%llx: ", offset);
			err = dump_fcb(nand, buffer);

			/* Store info in environment vars */
			{
				char str[16+3]="";

				sprintf(str, "0x%llx", (loff_t)nand->writesize * FCB_FIELD(buffer, firmware1_starting_sector));
				setenv("fw1_offset", str);

				sprintf(str, "0x%llx", (loff_t)nand->writesize * FCB_FIELD(buffer, firmware2_starting_sector));
				setenv("fw2_offset", str);
			}
		}

	} else if (strcmp(cmd, "write")==0) {

		memset(buffer, 0x00, nand->writesize + nand->oobsize); /* The block should be 'bad' */

		err = populate_fcb(nand, buffer);

		/* erase and write all stride_count locations. */
		for (i=0; i<set.stride_count; i++) {
			nand_erase_options_t opts;
			offset = (loff_t)i * set.stride_pages * nand->writesize;

			opts.offset = offset;
			opts.length = nand->erasesize;
			opts.jffs2  = 0;
			opts.quiet  = 0;
			opts.spread = 0;
			err = nand_erase_opts(nand, &opts);
			if (err) break;

			printf("Writing page at 0x%llx\n", offset);
			err = raw_access(nand, (ulong)buffer, offset, 0 /* Write */);
			if (err) break;

			/* Marking the FCB blocks as bad. */
			printf("Marking block bad\n");
			nand->block_markbad(nand, offset);
		}

	} else {
		printf("Unknown subcommand\n");
		rval = -1; goto quit;
	}

	if (err) rval=err;
quit:
	if (buffer) free(buffer);
	return rval;
}

U_BOOT_CMD(fcb, CONFIG_SYS_MAXARGS, 1, do_fcb,
	"FCB control for Freescale MX28 & NAND",
	"info - Check & Dump current FCB info\n"
	"	Sets the env variables fw1_offset, fw2_offset\n"
	"write [-1 FWOFFSET1 FWSIZE1] [-2 FWOFFSET2 FWSIZE2] [-s STRIDE COUNT] - Write FCB.\n"
	"	FWOFFSETn / FWSIZEn = offset and max firmware size (multiples of page).\n"
	"	STRIDE COUNT = Stride (pages) and FCB count, default 64/4\n"
);
