/*
 * OCOTP environment command for Netcomm/Falcon Board
 *
 * Ian Cullinan <Ian.Cullinan@netcommwireless.com>
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <command.h>

#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/sys_proto.h>

#define	MXS_OCOTP_MAX_TIMEOUT	1000000

static const char * fusewords[] = {
	"cust0",
	"cust1",
	"cust2",
	"cust3",
	"crypto0",
	"crypto1",
	"crypto2",
	"crypto3",
	"hwcap0",
	"hwcap1",
	"hwcap2",
	"hwcap3",
	"hwcap4",
	"hwcap5",
	"swcap",
	"custcap",
	"lock",
	"ops0",
	"ops1",
	"ops2",
	"ops3",
	"ops4",
	"ops5",
	"ops6",
	"rom0",
	"rom1",
	"rom2",
	"rom3",
	"rom4",
	"rom5",
	"rom6",
	"rom7",
	"srk0",
	"srk1",
	"srk2",
	"srk3",
	"srk4",
	"srk5",
	"srk6",
	"srk7",
	NULL
};

static const char *binfmt(uint32_t v)
{
	static char s[32+5];
	char *p = s;
	int i,j;
	for (j=0; j<32; j+=4) {
		for (i=0; i<4; i++) {
			*p++=(v & 0x80000000)?'1':'0';
			v<<=1;
		}
		*p++=' ';
	}
	*--p = '\0';
	return s;
}

static int ocotp_wait(void)
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;

	udelay(10); /* The busy bit is unavailable for 33 HCLK cycles */
	if (mx28_wait_mask_clr(&ocotp_regs->hw_ocotp_ctrl_reg, OCOTP_CTRL_BUSY,
				MXS_OCOTP_MAX_TIMEOUT)) {
		printf("OCOTP busy timed out\n");
		return 1;
	}
	return 0;
}

static void mx28_dump_fuses(void)
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	uint32_t i;
	const char **sp;
	uint32_t v;

	writel(OCOTP_CTRL_RD_BANK_OPEN, &ocotp_regs->hw_ocotp_ctrl_set);
	if (ocotp_wait()) return;

	i = 0;
	sp = fusewords;
	while (*sp) {
		v = readl(&ocotp_regs->hw_ocotp_cust0+i*4);
		printf("ocotp%02d %8s: %08x   %s\n", i, *sp, v, binfmt(v));
		i++;
		sp++;
	}

	/* Close the fuse bank */
	writel(OCOTP_CTRL_RD_BANK_OPEN, &ocotp_regs->hw_ocotp_ctrl_clr);
}

/* from spl_power_init.c */
void mx28_power_set_vddio(uint32_t new_target, uint32_t new_brownout);

static int ocotp_blow_fuse(uint32_t addr, uint32_t data) {
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;

	printf("Burning OCOTP fuse 0x%x with data 0x%x\n", addr, data);

	/* 1. Program HCLK to 24 MHz. OTP writes do not work at frequencies above 24 MHz. */
	/* Set CPU bypass - run off the 24MHz clock */
	writel(CLKCTRL_CLKSEQ_BYPASS_CPU, &clkctrl_regs->hw_clkctrl_clkseq_set);
	/* Set the HCLK ratio to divide-by-one */
	writel(0x01, &clkctrl_regs->hw_clkctrl_hbus_set);
	writel(0x1E, &clkctrl_regs->hw_clkctrl_hbus_clr);

	/* 2. Set the VDDIO voltage to 2.8 V */
	mx28_power_set_vddio(2800, 2700);
	mdelay(1000);

	/* Check that busy is clear */
	if (ocotp_wait()) return 1;

	/* 3. Check that HW_OCOTP_CTRL_BUSY and HW_OCOTP_CTRL_ERROR are clear. */
	if (readl(&ocotp_regs->hw_ocotp_ctrl_reg) & OCOTP_CTRL_ERROR) {
		printf("OCOTP error is set, clearing\n");
		writel(OCOTP_CTRL_ERROR, &ocotp_regs->hw_ocotp_ctrl_clr);
	}

	/* 4. Write the requested address to HW_OCOTP_CTRL_ADDR  and program the
	   unlock code into HW_OCOTP_CTRL_WR_UNLOCK. */
	writel(OCOTP_CTRL_WR_UNLOCK_KEY | (OCOTP_CTRL_ADDR_MASK & addr), 
			&ocotp_regs->hw_ocotp_ctrl_reg);

	/* 5. Write the data to HW_OCOTP_DATA. */
	writel(data, &ocotp_regs->hw_ocotp_data);

	/* 6. Once complete, the controller clears BUSY. */
	if (ocotp_wait()) return 1;

	/* 2us postamble (wait 4us just to be sure) */
	udelay(4);

	/* Check for error */
	if (readl(&ocotp_regs->hw_ocotp_ctrl_reg) & OCOTP_CTRL_ERROR) {
		printf("Error writing OCOTP fuses\n");
		return 1;
	}

	return 0;
}


/* Load ethaddr, sn and snextra environment variables from OCOTP fuses */
static int do_ocotp_env(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	uint32_t data;
	uint32_t data2;
	char buf[20] = ""; /* Enough space to store a MAC *with* colons */
	uchar mac[8];

	//mx28_dump_fuses();

	printf("Reading board identity from OCOTP fuses...\n");

	writel(OCOTP_CTRL_RD_BANK_OPEN, &ocotp_regs->hw_ocotp_ctrl_set);
	if (mx28_wait_mask_clr(&ocotp_regs->hw_ocotp_ctrl_reg, OCOTP_CTRL_BUSY,
				MXS_OCOTP_MAX_TIMEOUT)) {
		printf("Can't open OCOTP fuse bank\n");
		return 1;
	}

	/* Get the MAC address */
	data = readl(&ocotp_regs->hw_ocotp_cust0);

	mac[2] = (data >> 24) & 0xff;
	mac[3] = (data >> 16) & 0xff;
	mac[4] = (data >> 8) & 0xff;
	mac[5] = data & 0xff;

	/* There are two NetComm OIDs */
	if (mac[2] == 0x64) {
		mac[0] = 0x00;
		mac[1] = 0x60;
	} else {
		mac[0] = 0x18;
		mac[1] = 0xf1;
	}

	sprintf(buf, "%pM", mac);
	printf("MAC address: %s\n", buf);
	setenv("ethaddr", buf);

	/* Get the serial number */
	data = readl(&ocotp_regs->hw_ocotp_cust1);
	data2 = readl(&ocotp_regs->hw_ocotp_cust2);
	sprintf(buf, "%07x%08x", data, data2);
	printf("Serial number: %s\n", buf);
	setenv("sn", buf);

	/* Get snextra */
	data = readl(&ocotp_regs->hw_ocotp_cust3);
	sprintf(buf, "%08x", data);
	printf("SNextra: %s\n", buf);
	setenv("snextra", buf);

	printf("%p\n", (void*)do_ocotp_env);

	/* Close the fuse bank */
	writel(OCOTP_CTRL_RD_BANK_OPEN, &ocotp_regs->hw_ocotp_ctrl_clr);

	return 0;
}

U_BOOT_CMD(ocotp_env, 1, 0, do_ocotp_env,
	"Load ethaddr, sn and snextra environment variables from OCOTP fuses",
	""
);

static int do_ocotp(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	const char **sp;
	unsigned i,bit;
	uint32_t v;
	char *endp;

	if (argc==1) {
		mx28_dump_fuses();
		return 0;
	}
	if (argc!=4) {
		printf("Bad args\n");
		return -1;
	}
	if (strcmp("write", argv[1])!=0) {
		printf("Bad cmd\n");
		return -1;
	}
	i = 0;
	sp = fusewords;
	while (*sp) {
		if (strcmp(*sp,argv[2])==0) break;
		sp++;
		i++;
	}
	if (!*sp) {
		printf("Bad word\n");
		return -1;
	}
	endp=NULL;
	bit = simple_strtoul(argv[3],&endp,10);
	if ((argv[3][0] && endp[0]) || bit>31) {
		printf("Bad bit\n");
		return -1;
	}
	v = 1 << bit;

	if (ocotp_blow_fuse(i,v)) return 1;

	printf("Forcing shadow read\n");
	writel(OCOTP_CTRL_RELOAD_SHADOWS, &ocotp_regs->hw_ocotp_ctrl_set);
	if (ocotp_wait()) return 1;
	printf("Done.\n");

	return 0;
}
U_BOOT_CMD(ocotp, 4, 0, do_ocotp,
	"Read/Write OCOTP fuses",
	"- dump all fuses\n"
	"ocotp write WORD BIT - burn specified bit\n"
);

/* External commands for use in falcon.c */

/* Read the fuse register shadow. word is 0 for cust0, 1 for cust1, ..., 39 for srk7 */
uint32_t ocotp_read_shadow(unsigned word)
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	return readl(&ocotp_regs->hw_ocotp_cust0+word*4); /* Implicit *4 => *16 */
}

/* Write a whole fuse word, burn fuses where bits are 1 in mask. */
/* System must be rebooted after setting fuses, otherwise clocks or voltage levels may be wrong */
int ocotp_write_word(unsigned word, uint32_t mask)
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	if (ocotp_blow_fuse(word,mask)) return 1;
	printf("Forcing shadow read\n");
	writel(OCOTP_CTRL_RELOAD_SHADOWS, &ocotp_regs->hw_ocotp_ctrl_set);
	if (ocotp_wait()) return 1;
	printf("Done.\n");
	return 0;
}
