/*
 * LED command line support for Netcomm/Falcon Board
 *
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <command.h>
#include <asm/byteorder.h>

#include <asm/arch/imx-regs.h>

#include "gpio.h"
#include <cdcs_variant.h>

struct led {
	unsigned green;
	unsigned red;
};

#if defined(V_LEDFUN_falcon_with_aux)

#define LEDN 8

static const struct led leds[LEDN] = {
	{ PIN(1,9), PIN(1,10) },
	{ PIN(1,16), PIN(1,17) },
	{ PIN(2,2), PIN(2,3) },
	{ PIN(1,18), PIN(1,25) },
	{ PIN(3,30), PIN(1,27) },
	{ PIN(0,21), PIN(0,17) },
	{ PIN(2,6), PIN(2,7) },   /* FIXME: 2,6 doesn't work on REV A */
	{ PIN(1,11), PIN(1,22) },
};

#elif defined(V_LEDFUN_falcon_default)

#define LEDN 7

static const struct led leds[LEDN] = {
	{ PIN(1,9), PIN(1,10) },
	{ PIN(1,16), PIN(1,17) },
	{ PIN(1,18), PIN(1,25) },
	{ PIN(3,30), PIN(1,27) },
	{ PIN(0,21), PIN(0,17) },
	{ PIN(2,6), PIN(2,7) },   /* FIXME: 2,6 doesn't work on REV A */
	{ PIN(3,2), PIN(3,3) },
};

#elif defined(V_LEDFUN_falcon_1101)

#define LEDN 3

static const struct led leds[LEDN] = {
	{ PIN(1,27), PIN(1,27) },
	{ PIN(2,6), PIN(2,6) },
	{ PIN(2,7), PIN(2,7) },
};

#elif defined(V_LEDFUN_nmc1000) || defined(V_LEDFUN_ntc_70)

#define LEDN 4

static const struct led leds[LEDN] = {
	{ PIN(3,4), PIN(3,5) },
	{ PIN(3,0), PIN(2,6) },
	{ PIN(3,3), PIN(3,2) },
	{ PIN(0,17), PIN(0,21) },
};

#elif defined(V_LEDFUN_ntc_20)

#define LEDN 1

static const struct led leds[LEDN] = {
	{ PIN(3,5), PIN(3,4) },
};


#else

#error LED function is not defined!
#endif

/*
 * A very rudimentary atoi function, only for use in below get_pin function
 */
static int atoi(char *string)
{
    char *tmp = string;
    int result = 0;
    while (*tmp != '\0') {
        result =  (result * 10) + (*tmp - '0');
        tmp++;
    }
    return result;
}


/* Expects a string of the form x:y, where x and y are both integers in the
 * GPIO range. x and y should be positive and no more than 2 digits.
 * Used to convert the V_RECBUTTON string into the same output
 * as provided by the PIN macro in gpio.h.
 */
#define MAX_STRING_SIZE 10
static unsigned get_pin(char *string)
{
	char *token;
	int x,y;
	char string_array[MAX_STRING_SIZE];

	strncpy(string_array, string, MAX_STRING_SIZE);
	token = strtok(string_array, ":");
	x = atoi(token);
	token = strtok(NULL, ":");
	y = atoi(token);
	return PIN(x,y);
}

static unsigned int reset_button;

static void setup(void)
{
	unsigned i;
	struct pininfo pr = { M_OUT, V33, A12, DO(1) }; /* All leds off */

	for (i=0; i<LEDN; i++) {
		set_pininfo(leds[i].green, &pr);
		set_pininfo(leds[i].red, &pr);
	}
}

static int randbit(void)
{
	static unsigned state=0x12345678;
	state <<=1;
	state |= (((state >> 31) & 1) ^ ((state >> 28) & 1));
	return (state & 1);
}

#define BLACK 0
#define GREEN 1
#define RED 2
#define AMBER 3
#define BLINK_TIME 50
#define COUNTDOWN_DELAY 625 /* 5000/8 milliseconds */

void ledctl(unsigned led, unsigned color)
{
	if (color & GREEN) gpio_set(leds[led].green, 0);
	else gpio_set(leds[led].green, 1);
	if (color & RED) gpio_set(leds[led].red, 0);
	else gpio_set(leds[led].red, 1);
}

static void step(void)
{
	static int led = 0;
	static unsigned color = 0;
	static int dir=-1;

	ledctl(led, color);

	led += dir;
	if (led<0) { led=0; dir=1; }
	if (led>LEDN-1) { led=LEDN-1; dir=-1; }

	color = (randbit() << 1) | randbit();

}

void LED_all (int colour)
{
    int i;
    for (i = 0; i < LEDN; i++) {
        ledctl(i, colour);
    }
}

void LED_blink (unsigned int colour, unsigned int time)
{
    LED_all(BLACK);
    while(time > 0) {
        LED_all(colour);
        mdelay(BLINK_TIME);
        LED_all(BLACK);
        mdelay(BLINK_TIME);
        time--;
    }
}

int LED_countdown (unsigned int colour, unsigned int delay)
{
    int i;

    LED_all(colour);
    for (i = 0; i < LEDN; i++) {
        mdelay(delay);
        if (gpio_read(reset_button) != 0) {
            return 1;
        }
        ledctl(i, BLACK);
    }
    mdelay(delay);
    return 0;
}
/*
 * Handle the reset button logic:
 *     1. If reset button held less than X1 seconds, normal reset.
 *     2. If reset button held between X1-X2 seconds, recovery mode.
 *     3. If reset button held less than X1 seconds, factory reset.
 *
 * returns 1 if a reset button event occurred (proceed straight to boot)
 *         0 if no events occurred, proceed as normal.
 */

#define STATE_NORMAL 0
#define STATE_RECOVERY 1
#define STATE_FACTORY 2

int handle_reset_button (void)
{
    int current_state = STATE_NORMAL;
    unsigned int button_pressed = 0; /*  has it ever been pressed? */
    char *emergency;

	reset_button = get_pin(V_RECBUTTON);
    setup();

    while(gpio_read(reset_button) == 0) {
        LED_all(BLACK);
        /* disable the watchdog so we can wait for button release */
#if 0
        /* Not currently used on Falcon */
        if (current_state == 0) {
            cmdline_wdtoff(NULL, 0, 0, NULL);
        }
#endif
        button_pressed = 1;
        if (LED_countdown(GREEN, COUNTDOWN_DELAY)) {
            printf("normal boot detected\n");
            current_state = STATE_NORMAL;
            break;
        }
        if (LED_countdown(AMBER, COUNTDOWN_DELAY * 2)) {
            printf("recovery mode detected\n");
            current_state = STATE_RECOVERY;
            break;
        }
        if (LED_countdown(RED, COUNTDOWN_DELAY)) {
            printf("factory reset mode detected\n");
            current_state = STATE_FACTORY;
            break;
        }
    }

    /* Check what state the button was released in, to determine
     * the action that should be taken.
     */
    if (button_pressed == 0) {
        /* Read emergency variable and set LED */
        emergency = getenv("emergency");
        if (strcmp(emergency, "button") == 0) {
            ledctl(0, AMBER);
        } else {
            ledctl(0, GREEN);
        }
    } else {
        LED_all(BLACK);
        switch (current_state % 3) {
        case 0:
            // Normal Boot - flash green
            LED_blink(GREEN, 3);

            /* Make sure the power LED is left on */
            ledctl(0, GREEN);
            break;
        case 1:
            // Recovery mode - flash yellow
            LED_blink(AMBER, 3);
            setenv("emergency", "button");
            /* if "recboot" is not empty here, "bootcheck" will erase "emergency"
             * and the unit will boot to main system
             */
            setenv("recboot", "");
            saveenv();

            /* Make sure the power LED is left on */
            ledctl(0, AMBER);
            break;
        case 2:
            // Factory reset - flash red
            LED_blink(RED, 3);
            setenv("emergency", "reset");
            /* same reason as above */
            setenv("recboot", "");
            saveenv();

            /* Make sure the power LED is left on */
            ledctl(0, RED);
            break;
        default:
            // This should never happen

            /* Make sure the power LED is left on */
            ledctl(0, GREEN);
            break;
        }
    }

    return current_state;
}


/* Set gpio muxmode. Args: bank bit mux0|mux1|mux2|in|out */
static int fun_leds(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	printf("Hit SPACEBAR to stop demo\n");
	setup();
	while (1) {
		step();
		mdelay(100); /* 100 ms */
		if (tstc() && getc()==' ') {
			break;
		}
	}

	return 0;
}
U_BOOT_CMD(leds, CONFIG_SYS_MAXARGS, 1, fun_leds,
	"LED demo for Falcon",
	"Hit any key to stop.\n"
);
