/*
 * Freescale MX28EVK IOMUX setup
 *
 * Copyright (C) 2011 Marek Vasut <marek.vasut@gmail.com>
 * on behalf of DENX Software Engineering GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <common.h>
#include <config.h>
#include <asm/io.h>
#include <asm/arch/iomux-mx28.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/sys_proto.h>

#include "gpio.h"
#include <cdcs_variant.h>

#define	MUX_CONFIG_SSP0	(MXS_PAD_3V3 | MXS_PAD_8MA | MXS_PAD_PULLUP)
#define	MUX_CONFIG_GPMI	(MXS_PAD_1V8 | MXS_PAD_4MA | MXS_PAD_NOPULL)
#define	MUX_CONFIG_ENET	(MXS_PAD_3V3 | MXS_PAD_8MA | MXS_PAD_PULLUP)
#define	MUX_CONFIG_EMI	(MXS_PAD_1V8 | MXS_PAD_12MA | MXS_PAD_NOPULL)
#define	MUX_CONFIG_SSP2	(MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_PULLUP)

const iomux_cfg_t iomux_setup[] = {

#ifdef CONFIG_NAND_MXS
	/* GPMI NAND */
	MX28_PAD_GPMI_D00__GPMI_D0 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D01__GPMI_D1 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D02__GPMI_D2 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D03__GPMI_D3 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D04__GPMI_D4 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D05__GPMI_D5 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D06__GPMI_D6 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_D07__GPMI_D7 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_CE0N__GPMI_CE0N | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_RDY0__GPMI_READY0 | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_RDN__GPMI_RDN | (MXS_PAD_1V8 | MXS_PAD_4MA | MXS_PAD_PULLUP),
	MX28_PAD_GPMI_WRN__GPMI_WRN | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_ALE__GPMI_ALE | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_CLE__GPMI_CLE | MUX_CONFIG_GPMI,
	MX28_PAD_GPMI_RESETN__GPMI_RESETN | MUX_CONFIG_GPMI,
#endif

	/* FEC0 */
	MX28_PAD_ENET0_MDC__ENET0_MDC | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_MDIO__ENET0_MDIO | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_RX_EN__ENET0_RX_EN | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_TX_EN__ENET0_TX_EN | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_RXD0__ENET0_RXD0 | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_RXD1__ENET0_RXD1 | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_TXD0__ENET0_TXD0 | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_TXD1__ENET0_TXD1 | MUX_CONFIG_ENET,
	MX28_PAD_ENET_CLK__CLKCTRL_ENET | MUX_CONFIG_ENET,
	/* FEC0 Enable */
	MX28_PAD_SSP1_DATA3__GPIO_2_15 |
		(MXS_PAD_12MA | MXS_PAD_3V3),
	/* FEC0 Reset */
	MX28_PAD_ENET0_RX_CLK__GPIO_4_13 |
		(MXS_PAD_12MA | MXS_PAD_3V3 | MXS_PAD_PULLUP),

	/* FEC1 */
	MX28_PAD_ENET0_COL__ENET1_TX_EN | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_CRS__ENET1_RX_EN | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_RXD2__ENET1_RXD0 | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_RXD3__ENET1_RXD1 | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_TXD2__ENET1_TXD0 | MUX_CONFIG_ENET,
	MX28_PAD_ENET0_TXD3__ENET1_TXD1 | MUX_CONFIG_ENET,

	/* EMI */
	MX28_PAD_EMI_D00__EMI_DATA0 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D01__EMI_DATA1 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D02__EMI_DATA2 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D03__EMI_DATA3 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D04__EMI_DATA4 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D05__EMI_DATA5 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D06__EMI_DATA6 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D07__EMI_DATA7 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D08__EMI_DATA8 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D09__EMI_DATA9 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D10__EMI_DATA10 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D11__EMI_DATA11 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D12__EMI_DATA12 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D13__EMI_DATA13 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D14__EMI_DATA14 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_D15__EMI_DATA15 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_ODT0__EMI_ODT0 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_DQM0__EMI_DQM0 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_ODT1__EMI_ODT1 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_DQM1__EMI_DQM1 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_DDR_OPEN_FB__EMI_DDR_OPEN_FEEDBACK | MUX_CONFIG_EMI,
	MX28_PAD_EMI_CLK__EMI_CLK | MUX_CONFIG_EMI,
	MX28_PAD_EMI_DQS0__EMI_DQS0 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_DQS1__EMI_DQS1 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_DDR_OPEN__EMI_DDR_OPEN | MUX_CONFIG_EMI,

	MX28_PAD_EMI_A00__EMI_ADDR0 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A01__EMI_ADDR1 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A02__EMI_ADDR2 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A03__EMI_ADDR3 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A04__EMI_ADDR4 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A05__EMI_ADDR5 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A06__EMI_ADDR6 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A07__EMI_ADDR7 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A08__EMI_ADDR8 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A09__EMI_ADDR9 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A10__EMI_ADDR10 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A11__EMI_ADDR11 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A12__EMI_ADDR12 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A13__EMI_ADDR13 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_A14__EMI_ADDR14 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_BA0__EMI_BA0 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_BA1__EMI_BA1 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_BA2__EMI_BA2 | MUX_CONFIG_EMI,
	MX28_PAD_EMI_CASN__EMI_CASN | MUX_CONFIG_EMI,
	MX28_PAD_EMI_RASN__EMI_RASN | MUX_CONFIG_EMI,
	MX28_PAD_EMI_WEN__EMI_WEN | MUX_CONFIG_EMI,
	MX28_PAD_EMI_CE0N__EMI_CE0N | MUX_CONFIG_EMI,
	MX28_PAD_EMI_CE1N__EMI_CE1N | MUX_CONFIG_EMI,
	MX28_PAD_EMI_CKE__EMI_CKE | MUX_CONFIG_EMI,
};

static struct gpio_preset gpio_presets[] = {
	/* AUART0 is on 3/0 - 3/3, mux 0 */
	{ pin(3,0), .v = { MM(0), V33, A12, PU }}, /* Rx */
	{ pin(3,1), .v = { MM(0), V33, A12, PU }},     /* Tx */
	{ pin(3,2), .v = { MM(0), V33, A12 }},     /* CTS */
	{ pin(3,3), .v = { MM(0), V33, A12 }},     /* RTS */
#ifdef V_BOARD_ntc_20
	/* 3/4: red light; 3/5: green light, mux gpio output 0 */
	{ pin(3,5), .v = { M_OUT, DO(0) }},

	/* SPI for display. Early display uses bitbashing SPI */
	{ pin(2,10),.v = { M_OUT, V33, A04, DO(0) }}, /* SSP0_SCK, SCL_DSP */
	{ pin(2,8), .v = { M_OUT, V33, A04, DO(0) }}, /* SSP0_CMD, SDIN_DSP */
	{ pin(2,3), .v = { M_OUT, V33, A04, DO(1) }}, /* SSP0_DATA3, CS_DSPN */
	{ pin(2,20),.v = { M_OUT, V33, A04, DO(0) }}, /* SSP2_SS1, D/C_DSP */
	{ pin(0,17),.v = { M_OUT, V33, A04, DO(1) }}, /* GPMI_CE1, SYS_DISPLAY_RESETN */
#else
	/* AUART1 is on 3/4 - 3/7, mux 0 */
	{ pin(3,4), .v = { MM(0), V33, A12, PU }}, /* Rx */
	{ pin(3,5), .v = { MM(0), V33, A12, PU }},     /* Tx */
	{ pin(3,6), .v = { MM(0), V33, A12 }},     /* CTS */
	{ pin(3,7), .v = { MM(0), V33, A12 }},     /* RTS */
#endif
	/* AUART2 = NC */

	/* AUART3 is on 2/18, 2/19, mux 1 */
	{ pin(2,18), .v = { MM(1), V33, A12, PU }}, /* Rx */
	{ pin(2,19), .v = { MM(1), V33, A12, PU }}, /* Tx */

	/* AUART4 = NC */

	/* I2C1 on 3/16, 3/17 */
	{ pin(3,16), .v = { MM(1), V33, A12, PU }}, /* SCL */
	{ pin(3,17), .v = { MM(1), V33, A12, PU }}, /* SDA */

	{ .pin = NOPIN } /* Must be at end */
};

#ifndef V_GPIOBUTTONS_none
extern void gpio_buttons_init(struct gpio_preset *gpio_presets, const char *instring);
#endif

static void uart_board_init(void)
{
	struct gpio_preset *p;
	struct mx28_clkctrl_regs *clkctrl = (void*)MXS_CLKCTRL_BASE;
	/* make sure the UART clock is not gated in the clock control module, clear bit 31 */
	clkctrl->hw_clkctrl_xtal_clr = 0x80000000;

#ifndef V_GPIOBUTTONS_none
	gpio_buttons_init(gpio_presets, V_GPIOBUTTONS);
#endif

	/* Set up GPIO multiplexers and values */
	p = (struct gpio_preset *)gpio_presets;
	while (p->pin != NOPIN) {
		set_pininfo(p->pin, &p->v);
		p++;
	}
}


#define HW_DRAM_CTL29	(0x74 >> 2)
#define CS_MAP		0xf
#define COLUMN_SIZE	0x2
#define ADDR_PINS	0x2
#define BANKS		8
#define APREBIT		0xa

struct sdram_override {
	int regno;
	u32 val;
};

const struct sdram_override sdram_evk[] = {
	{ 29,  CS_MAP << 24 | COLUMN_SIZE << 16 | 2/*ADDR_PINS*/ << 8 | APREBIT},
	{ -1, 0}
};

/* We set SDRAM up for 512MB, then attempt to autodetect (see spl_mem_init.c) */
const struct sdram_override sdram_eagle[] = {
	{ 29, CS_MAP << 24 | COLUMN_SIZE << 16 | ADDR_PINS << 8 | APREBIT},
	{ 31, 0x00000101 | (!!(BANKS==8)) << 16 }, /* 4 -bank mode */
	{ 37, 0x07080403 },
	{ 41, 0x0002030c },
	{ 71, 0xf4004a27 },
	{ 72, 0xf4004a27 },
	{ 73, 0xf4004a27 },
	{ 74, 0xf4004a27 },
	{ 77, 0x07400300 },
	{ 78, 0x07400300 },
	{ 79, 0x00000005 },
	{ 163, 0x00030404 }, /* DDR2 */
	{ 164, 0x00000003 },
	{ 181, 0x04420442 },
	{ 182, 0x04420442},
	{ 189, 0xffffffff},
	{ -1, 0}
};

void mx28_adjust_memory_params(uint32_t *dram_vals)
{
	const struct sdram_override *sov = sdram_eagle;

	while (sov->regno >= 0) {
#if 0
		D("SDRAM: reg %d: [%08x] => [%08x] %s\r\n",
			sov->regno, dram_vals[sov->regno], sov->val,
			(dram_vals[sov->regno] == sov->val)?"(red)":"");
#endif
		dram_vals[sov->regno] = sov->val;
		sov++;
	}
}

static unsigned auarts[] = {
	MXS_UARTAPP0_BASE,
	MXS_UARTAPP1_BASE,
	MXS_UARTAPP2_BASE,
	MXS_UARTAPP3_BASE,
	MXS_UARTAPP4_BASE
};

#include <asm/arch/regs-power.h>

struct mx28_power_regs *pw = (struct mx28_power_regs *)MXS_POWER_BASE;

#define PREG(name) D( #name " = \t%08x\r\n", pw->name )

#if defined(CONFIG_HW_WATCHDOG) && defined(CONFIG_MXS_WATCHDOG)
static void inline mxs_reg_setl(uint32_t value,struct mx28_register_32 * reg)
{
	writel(value,&reg->reg_set);
}

static void wdt_enable(u32 value)
{
	struct mx28_rtc_regs *rtc_regs = (struct mx28_rtc_regs *)MXS_RTC_BASE;

	writel(value, &rtc_regs->hw_rtc_watchdog);

	mxs_reg_setl(RTC_CTRL_WATCHDOGEN,&rtc_regs->hw_rtc_ctrl_reg);
}
#endif

#ifdef V_BOARD_ntc_20

#include "display.h"
#include "netcomm128x64.h"
#include "cbm_decoder.h"

/* Draw pixel callback for bitmap decoder */
static int xoffs = 0;
static int yoffs = 0;
static int transparent = 0;
static void pixel(unsigned x, unsigned y, int c)
{
	int rx = x + xoffs;
	int ry = y + yoffs;
	if (rx < 0 || rx > DSP_RESX - 1) return;
	if (ry < 0 || ry > DSP_RESY - 1) return;
	if (c)
		DSP_SETPIXEL(rx, ry);
	else if (!transparent)
		DSP_CLRPIXEL(rx, ry);
}

/* Show screen. Compressed bitmap is generated
 * by cbmcompress (svn://svn/main/tools/bitmap) */
static void dsp_show_screen(const uint8_t *screen, int size)
{
	xoffs = 0;
	yoffs = 0;
	transparent = 0;
	CBM_decompress(screen, size, pixel);
	dsp_flush();
}

/* Show logo */
static void dsp_show_logo(void)
{
	dsp_show_screen(startlogo, sizeof(startlogo));
}

#endif /* V_BOARD_ntc_20 */

#ifdef CONFIG_POWER_STATE_MACHINE
#ifdef V_BOARD_ntc_20
const uint8_t charging[] = { /* CBM compressed bitmap: 128 x 64 pixels */
	165,127, 63,255,  0,106,  0,  3, 31, 40,  0,  2, 31, 40,  0,  2, 31, 40,  0,  2,  3, 28,  3, 40,
	  0,  2,  3, 28,  3, 40,  0,  2,  3, 28,  3, 40,  0,  2,  3, 28,  3, 40,  0,  2,  3, 28,  3, 39,
	  0,  2,  7, 28,  3, 39,132, 63,128, 16,  0, 16,  0,  1,  3, 39,132, 63,128, 24,  0, 16,  0,  1,
	  3, 39,133, 56,  0, 28, 16,  0,  9,  3, 39,136, 56,  0, 31, 28,  0,  0,112,  0, 36,  0,  2,135,
	224,  0,126,120,  0,  1,192, 38,135,224,  0,127,252,  0,  1,192, 38,135,224,  0,127,255,  0,  1,
	192, 38,135,224,  0,127,255,128,  1,192, 38,135,224,  0, 31,231,192,  1,192, 38,135,224,  0,  7,
	224,240,  1,192, 38,135,254,  0,  0,224, 24,  1,192, 38,  7, 28,  3, 39,  0,  2,  7, 28,  3, 40,
	  0,  2,  3, 28,  3, 40,  0,  2,  3, 28,  3, 40,  0,  2,  3, 28,  3, 40,  0,  2,  3, 28,  3, 40,
	  0,  2,  3, 28,  3, 40,  0,  2, 31, 40,  0,  2, 31, 40,  0,  2, 31,};
const uint8_t batt_too_low[] = { /* CBM compressed bitmap: 128 x 64 pixels */
	165,127, 63,255,  0,106,  0,  3, 31, 40,  0,  2, 31, 40,  0,  2, 31, 40,  0,  2,  3, 28,  3, 40,
	  0,  2,  3, 28,  3, 40,  0,  2,  3, 25,130,243,128, 38,  0,  3,  3, 25,130,243,128, 38,  0,  3,
	  3, 25,130,243,128, 37,  0,  3,  7, 25,130,243,128, 37,  0,  3,  7, 25,130,243,128, 37,  0,  3,
	  7, 25,130,243,128, 37,  0,  3,  3, 27,130,243,128, 37,  0,  3,  3, 27,130,243,128, 37,  0,  3,
	  3, 27,130,243,128, 37,  0,  3,  3, 27,130,243,128, 37,  0,  3,  3, 27,130,243,128, 37,  0,  3,
	  3, 27,130,243,128, 37,  0,  3,  3, 27,130,243,128, 37,  0,  3,  3, 27,130,243,128, 37,  0,  3,
	  7, 25,130,243,128, 37,  0,  3,  7, 25,130,243,128, 37,  0,  3,  7, 25,130,243,128, 38,  0,  3,
	  3, 25,130,243,128, 38,  0,  3,  3, 25,130,243,128, 38,  0,  3,  3, 25,130,243,128, 38,  0,  3,
	  3, 28,  3, 40,  0,  2,  3, 28,  3, 40,  0,  2, 31, 40,  0,  2, 31, 40,  0,  2, 31,};
const uint8_t no_batt[] = { /* CBM compressed bitmap: 128 x 64 pixels */
	165,127, 63,244,  0,  4,  1, 42,  0,  2,  1, 25,  0,  1,  2, 42,  0,  2,  2, 24,  0,  1,  3, 42,
	  0,  2,  3, 23,  0,  1,  3, 43,  3, 22,  0,  1,  3, 42, 31, 40,  0,  2, 31, 40,  0,  2, 31, 40,
	  0,  2,134,224,112,  0,  0,112, 28, 40,134,224, 56,  0,  0,224, 28, 40,134,224, 28,  0,  1,192,
	 28, 40,134,224, 14,  0,  3,128, 28, 40,134,224,  7,  0,  7,  0, 28, 39,135,254,  0, 56,  0,224,
	  1,192, 38,135,254,  0, 28,  1,192,  1,192, 38,135,254,  0, 14,  3,128,  1,192, 38,135,224,  0,
	  7,  7,  0,  1,192, 38,135,224,  0,  3,142,  0,  1,192, 38,135,224,  0,  1,252,  0,  1,192, 38,
	135,224,  0,  0,240,  0,  1,192, 38,135,224,  0,  0,240,  0,  1,192, 38,135,224,  0,  1,248,  0,
	  1,192, 38,135,224,  0,  3,158,  0,  1,192, 38,135,224,  0,  7,  7,  0,  1,192, 38,135,254,  0,
	 14,  3,128,  1,192, 38,135,254,  0, 28,  1,192,  1,192, 38,135,254,  0, 56,  0,224,  1,192, 39,
	134,224,  7,  0,  7,  0, 28, 40,134,224, 14,  0,  3,128, 28, 40,134,224, 28,  0,  1,192, 28, 40,
	134,224, 56,  0,  0,224, 28, 40,134,224,112,  0,  0,112, 28, 40, 31, 40,  0,  2, 31, 40,  0,  2,
	 31, 41,  0,  3,  3, 22,  0,  1,  3, 43,  3, 23,  0,  1,  3, 42,  0,  3,  2, 24,  0,  1,  3, 42,
	  0,  2,  1, 25,  0,  1,  2, 51,  0,  3,  1,};
#endif
/*
 * Phase 2 power state machine
 * - Check power state:
 * 		+ if battery only:
 * 			* battery is good to boot: BOOT
 * 			* battery is not good to boot: show "low battery" screen, shutdown
 * 		+ if battery and charger: BOOT
 * 		+ if charger only: BOOT
 * 		+ if battery and USB host:
 * 			* battery is good to boot: BOOT
 * 			* battery is not good to boot: charge (show "charging" screen)
 * 			until good level, then boot
 * 		+ if USB host only: show "no battery" screen and keep PSWITCH state
 */
static void mx28_spl_phase_2_power_state_machine(void)
{
#define DELAY_FOR_STATE_MACHINE_CYCLE_US 1000000
#define DELAY_FOR_SHOWING_LOW_BATT_SCREEN_US 3000000
#define WATCHDOG_TIMER_STATE_MACHINE_MS 5000
	int pswitch_mid = 0;
	struct mx28_rtc_regs *rtc_regs = (struct mx28_rtc_regs *)MXS_RTC_BASE;
	uint32_t watchdog_timer_backup;
	struct power_status pw_state;
#ifdef V_BOARD_ntc_20
#define SCREEN_LOW_BATTERY	1
#define SCREEN_CHARGING	2
#define SCREEN_NO_BATTERY	3
	int screen_type = 0;
#endif

	/* backup watchdog time */
	watchdog_timer_backup = readl(&rtc_regs->hw_rtc_watchdog);
	/* loop until BOOTING or OFF */
	while (state_machine != BOOTING && state_machine != OFF)
	{
		/* kick watch dog */
		writel(WATCHDOG_TIMER_STATE_MACHINE_MS, &rtc_regs->hw_rtc_watchdog);/*ms */
		/* update power state */
		mx28_check_power_state(&pw_state);
		/* check pswitch level change */
		pswitch_mid = mx28_pswitch_event();
		D("Phase 2 State machine:: state_machine=%d, vdd5v_detected=%d, vdd5v_charger=%d, pswitch_mid=%d, batt_detected=%d, batt_ok_to_boot=%d \r\n",
				state_machine, pw_state.vdd5v_detected, pw_state.vdd5v_charger, pswitch_mid, pw_state.batt_detected, pw_state.batt_ok_to_boot);
		switch (state_machine){
			case PSWITCH_MID:
				if (pw_state.vdd5v_charger || pw_state.batt_ok_to_boot){
					if (!pw_state.vdd5v_charger){
						/* If no VDD5V charger, it is battery only or USB host with battery.
						 * In both case it should use power from battery because
						 * USB enumeration has't been completed and it may draw more than 100mA
						 * in further booting. So hand over to battery.
						 */
						mx28_hand_over_battery();
					}
					state_machine = BOOTING;
				}
				else if ((!pw_state.vdd5v_detected) && pw_state.batt_detected && !pw_state.batt_ok_to_boot){
					/* only battery and it is not good to boot: show "low battery" screen, shutdown */
#ifdef V_BOARD_ntc_20
					if (screen_type != SCREEN_LOW_BATTERY){
						screen_type = SCREEN_LOW_BATTERY;
						/* show screen "low battery" */
						dsp_show_screen(batt_too_low, sizeof(batt_too_low));
					}
					udelay(DELAY_FOR_SHOWING_LOW_BATT_SCREEN_US);
#endif
					state_machine = OFF;
					mx28_powerdown_the_chip();
				}
				else if ( pw_state.vdd5v_detected && (!pw_state.vdd5v_charger) &&  pw_state.batt_detected && (!pw_state.batt_ok_to_boot)){
					/* USB host and battery is not good to boot: charge (show "charging" screen) */
					/* charging if necessary */
					if (mx28_is_batt_temp_ok_for_charging()){
						/* if charger is on, check ilimit stop */
						if (mx28_is_charger_on()){
							mx28_check_charge_ilimit_stop();
						}
						/* else: check battery voltage threshold to re-enable charger */
						else {
							mx28_turn_on_charger(POWER_CHARGE_STOP_ILIMIT_10MA,
									POWER_CHARGE_BATTCHRG_I_50MA | POWER_CHARGE_BATTCHRG_I_20MA | POWER_CHARGE_BATTCHRG_I_10MA);
						}
					}
					else{
						/* turn off charger */
						mx28_turn_off_charger();
					}
#ifdef V_BOARD_ntc_20
					if (mx28_is_charger_on()){
						if (screen_type != SCREEN_CHARGING){
							screen_type = SCREEN_CHARGING;
							/* show screen "charging" */
							dsp_show_screen(charging,sizeof(charging));
						}
					}
#endif
				}
				else if (pw_state.vdd5v_detected && (!pw_state.vdd5v_charger) && (!pw_state.batt_detected)){
					/* USB host only: show "no battery" screen and keep current state */
#ifdef V_BOARD_ntc_20
					if (screen_type != SCREEN_NO_BATTERY){
						screen_type = SCREEN_NO_BATTERY;
						/* show screen "no battery" */
						dsp_show_screen(no_batt,sizeof(no_batt));
					}
#endif
				}
				break;

			default:
				break;
		}

		if (state_machine != BOOTING){
			udelay(DELAY_FOR_STATE_MACHINE_CYCLE_US);
		}
	}
	D("State machine 2:: state_machine=%d, continue... \r\n", state_machine);
	/* disable pswitch irq */
	mx28_disable_pswitch_irq();
	mx28_disable_usb_plugin_detect();
	/* restore watchdog time */
	writel(watchdog_timer_backup, &rtc_regs->hw_rtc_watchdog);
}
#endif

void board_init_ll(void)
{
	int i;

	#if defined(CONFIG_HW_WATCHDOG) && defined(CONFIG_MXS_WATCHDOG)
	wdt_enable(19 * 1000);
	#endif

	uart_board_init();
	for (i=0; i<5; i++) {
		init_app_uart((void*)auarts[i], 115200);
		//app_uartbase(auarts[i]);
		//app_printf("\r\n\nSPL " __DATE__ " " __TIME__ ", UART %d\r\n", i);
	}
	init_app_uart((void*)CONFIG_FREESCALE_UART_BASE, 115200);

	D("\r\n\r\n");
	D("SPL Start\r\n");

#ifdef CONFIG_SPL_FUSEDATA_ADDR
	mx28_blow_fuses();
#endif
	mx28_common_spl_init(iomux_setup, ARRAY_SIZE(iomux_setup));

	uart_board_init(); /* In case the above messes things up. FIXME: merge the settings. */

	/* Power supply needs some more handholding. */
	pw->hw_power_vddioctrl &= ~POWER_VDDIOCTRL_PWDN_BRNOUT;
	mx28_dump_fuses();

#ifdef V_BOARD_ntc_20
	dsp_init();
#endif
#ifdef CONFIG_POWER_STATE_MACHINE
	mx28_spl_phase_2_power_state_machine();
#endif
#ifdef V_BOARD_ntc_20
	dsp_show_logo();
#endif

	D("SPL ends\r\n");
}
